/**
 * @file IxOsCacheMMU.h
 *
 * @brief this file contains the API of the @ref IxCacheMMU component
 * 
 * <hr>
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#ifndef IxOsCacheMMU_H

#ifndef __doxygen_hide
#define IxOsCacheMMU_H
#endif /* __doxygen_hide */

#ifdef __doxygen_HIDE
#define IX_OS_CACHE_DOXYGEN
#endif /* __doxygen_HIDE */

#include "IxTypes.h"

#ifdef __vxworks

#include <vxWorks.h>
#include <cacheLib.h>
#include <memLib.h>

#endif

/**
 * @defgroup IxOsCacheMMU IXP425 OS Cache MMU (IxOsCacheMMU) API
 *
 * @brief This service provides services to the access components and codelets to
 * abstract out any cache coherency issues and mmu mappings.
 *
 * @{
 */

/** 
 * 
 * @def IX_ACC_CACHE_ENABLED
 *
 * @brief This macro enable cached memory in Access layers.  
 *
 * When defined, this macro enable the use of cached memory in the access
 * layers.
 *
 * To disable cache on mbufs, #undef the macro
 *
 * <hr>
 *
 */
#define IX_ACC_CACHE_ENABLED



/** 
 *
 * @brief Allocate memory for driver use, that will be shared between XScale and NPE's.
 *
 * Allocate memory for driver use, that will be shared between XScale and NPE's. 
 * @note The buffer allocated with have the system defined atributes, 
 * and as such the Invalidate and 
 * flush macros functionality must be updated.
 *
 * @note The buffer allocated is aligned on a cache line boundary.
 *
 * @param UINT32 size -  number of bytes of memory requested.
 *
 * @return  void * Pointer to memory that can be used 
 * between XScale and NPE's. 
 * 
 * @sa ixOsServCacheDmaFree
 *
 * <hr>
 */
void *ixOsServCacheDmaAlloc(UINT32 size);

/** 
 *
 * @brief Free memory allocated from ixOsServCacheDmaAlloc.
 *
 * This function frees the memory allocated from @a ixOsServCacheDmaAlloc.
 *
 * @param void *ptr - pointer to the memory area to be freed.
 * @param UINT32 size -  number of bytes of memory allocated.
 *
 * @return  void
 *
 * @sa ixOsServCacheDmaMalloc
 *
 * <hr>
 */
void ixOsServCacheDmaFree(void *ptr, UINT32 size);

/** 
 *
 * @def  IX_ACC_DRV_DMA_MALLOC
 *
 * @brief Allocate memory for driver use, that will be shared between XScale and NPE's.
 *
 * This macro is used allocate memory for driver use, 
 * that will be shared between XScale and NPE's. 
 * @note The buffer allocated with have the system defined atributes, 
 * and as such the Invalidate and 
 * flush macros functionality must be updated.
 *
 * @note The buffer allocated is aligned on a cache line boundary.
 *
 * @param UINT32 size -  number of bytes of memory requested.
 *
 * @return  void * Pointer to memory that can be used 
 * between XScale and NPE's. 
 * 
 * @sa IX_ACC_DRV_DMA_FREE
 *
 * <hr>
 */
#define IX_ACC_DRV_DMA_MALLOC(size) ixOsServCacheDmaAlloc(size)

/** 
 *
 * @def  IX_ACC_DRV_DMA_FREE
 *
 * @brief Free memory allocated from IX_ACC_DRV_DMA_MALLOC.
 *
 * This function frees the memory allocated from @a IX_ACC_DRV_DMA_MALLOC.
 *
 * @param void * ptr - pointer to the memory area to be freed.
 * @param UINT32 size -  number of bytes of memory allocated.
 *
 * @return  void
 *
 * @sa IX_ACC_DRV_DMA_MALLOC
 *
 * <hr>
 */
#define IX_ACC_DRV_DMA_FREE(ptr,size) ixOsServCacheDmaFree((ptr),(size))

#if (defined(__vxworks)||defined(IX_OS_CACHE_DOXYGEN))

/**
 * @def IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr)
 *
 * @brief Return a virtual address for the provided physical address.
 *
 * This macro return a virtual address for the provided physical address.
 *
 * @attention There are different implementations for this macro 
 * 
 * - Linux OS implementation:   
 *  - \#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) ((addr) ? virt_to_phys((void*)(addr)) : 0)
 *
 * - VxWorks OS implementation: 
 *  - \#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) (addr)
 * 
 * - default implementation: 
 *  - \#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) (addr)
 *
 * <hr>
 */
#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) (addr)

/**
 * @def IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr)
 * 
 * @brief Return a physical address for the provided virtual address.
 * 
 * This macro return a physical address for the provided virtual address
 *
 * @attention There are different implementations for this macro 
 * 
 * - Linux OS implementation:   
 *  - \#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) ((addr) ? virt_to_phys((void*)(addr)) : 0)
 *
 * - VxWorks OS implementation: 
 *  - \#define IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr) (addr)
 * 
 * - default implementation: 
 *  - \#define IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr) (addr)
 * 
 * <hr>
 *
 */

#define IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr) (addr)

/**
 * @def IX_XSCALE_CACHE_LINE_SIZE
 *
 * @brief IX_XSCALE_CACHE_LINE_SIZE = size of cache line for both flush and invalidate.
 *
 *
 * <hr>
 */
#define IX_XSCALE_CACHE_LINE_SIZE (32)

/**
 * @def IX_ACC_DRAM_PHYS_OFFSET
 *
 * @brief PHYS_OFFSET = Physical DRAM offset..
 *
 * @attention There are different implementations for this macro 
 * 
 * - Linux OS implementation:   
 *  - \#define IX_ACC_DRAM_PHYS_OFFSET     (PHYS_OFFSET) 
 *
 * - VxWorks OS implementation: 
 *  - \#define IX_ACC_DRAM_PHYS_OFFSET     (0x00000000UL) 
 *
 *
 * <hr>
 */
#define IX_ACC_DRAM_PHYS_OFFSET     (0x00000000UL) 


#ifdef IX_ACC_CACHE_ENABLED
/**
 * @def  IX_ACC_DATA_CACHE_INVALIDATE(addr,size) 
 *
 * @brief Invalidate a cache range.  
 *
 * Invalidate a cache range.  
 * @note
 * This is typically done prior to submitting a buffer to the NPE's which you
 * expect the NPE to populate with data.
 * @attention The size argument must be a multiple of cacheline size, i.e. a multiple
 * if 32bytes for the XSCALE. The argument shall be rounded up to the next 32byte boundry.
 * Extreem care must be taken when invalidating cache lines due. 
 *
 * - If memory space used is non cached, then this function may be null.
 * - Functionality required: 
 *   -# Non-Cached space         : No functionality required. 
 *   -# Write Through Enabled    : Invalidate area specified
 *   -# Copy Back Enabled        : Invalidate area specified
 * 
 * @attention There are different implementations for this macro 
 * 
 * - Linux OS implementation:   
 *  - \#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size) invalidate_dcache_range((__u32)addr, (__u32)addr + size ) 
 *
 * - VxWorks OS implementation: 
 *  - \#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size) cacheInvalidate(DATA_CACHE, addr, size)
 * 
 * - default implementation: 
 *  - \#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size)
 *
 * <hr>
 *
 */

#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size) cacheInvalidate(DATA_CACHE, addr, size)

/** 
 * 
 * @def IX_ACC_DATA_CACHE_FLUSH(addr,size)
 *
 * @brief Flush a cache range to physical memory.  
 *
 * Flush a cache range to physical memory.
 * @note
 * This is typically done prior to submitting a buffer to the NPE's which you
 * expect the NPE to read from. Entire Cache lines will be flushed.
 *
 * - If memory space used is non cached, then this function does may be null.
 * - Functionality required: 
 *   -# Non-Cached space      	: Flush CPU WB, No cache Flush.
 *   -# Write Through Enabled  	: Flush CPU WB, No cache Flush. 
 *   -# Copy Back Enabled     	: Flush CPU WB, Invalidate area specified
 *
 * @attention There are different implementations for this macro   
 *   
 * - Linux OS implementation:   
 *   - \#define IX_ACC_DATA_CACHE_FLUSH(addr,size) clean_dcache_range((__u32)addr, (__u32)addr + size )  
 *  
 * - VxWorks OS implementation:  
 *  - \#define IX_ACC_DATA_CACHE_FLUSH(addr,size) do { cacheFlush(DATA_CACHE, addr, size); cachePipeFlush(); } while(0)  
 *   
 * - VxWorks OS implementation (write-through mode):   
 *  - \#define IX_ACC_DATA_CACHE_FLUSH(addr,size) cachePipeFlush();  
 *    
 * - default implementation (uncached memory):   
 *  - \#define IX_ACC_DATA_CACHE_FLUSH(addr,size)   
 *  
 * <hr>  
 *   
 */
#define IX_ACC_DATA_CACHE_FLUSH(addr,size) \
    do { cacheFlush(DATA_CACHE, addr, size); cachePipeFlush(); } while(0)

#else /* IX_ACC_CACHE_ENABLED */

#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size) {}
#define IX_ACC_DATA_CACHE_FLUSH(addr,size) {}

#endif /* IX_ACC_CACHE_ENABLED */

#endif /* (defined(__vxworks)||defined(IX_OS_CACHE_DOXYGEN))*/

#if (defined(__linux)||defined(IX_OS_CACHE_DOXYGEN))

#ifndef IX_ACC_CACHE_ENABLED
#error "Uncached memory not supported in linux environment"
#endif

#include <linux/cache.h>
#include <linux/mm.h>
#include <linux/config.h>
#include <asm/pgalloc.h>

/**
 * Linux implementations of above macros.
 */

#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) ((addr) ? virt_to_phys((void*)(addr)) : 0)
#define IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr) ((addr) ? phys_to_virt((unsigned int)(addr)) : 0)
#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size) invalidate_dcache_range((__u32)addr, (__u32)addr + size )
#define IX_ACC_DATA_CACHE_FLUSH(addr,size) clean_dcache_range((__u32)addr, (__u32)addr + size )
#define IX_ACC_DRAM_PHYS_OFFSET     (PHYS_OFFSET) 

#define IX_XSCALE_CACHE_LINE_SIZE 32

#endif /* (defined(__linux)||defined(IX_OS_CACHE_DOXYGEN))*/

#ifndef __vxworks
#ifndef __linux

/* default implementatins of above macros here.
 */

#define IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(addr) (addr)
#define IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(addr) (addr)
#define IX_ACC_DATA_CACHE_INVALIDATE(addr,size)
#define IX_ACC_DATA_CACHE_FLUSH(addr,size) 

#define IX_XSCALE_CACHE_LINE_SIZE 32

#endif /* __vxworks */
#endif /* __linux */

/**
 * @} IxOsCacheMMU
 */

#endif /* IxOsCacheMMU_H */

