/**
 * @file IxEthDBPortDefs.h
 *
 * @brief Private MAC learning API
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

/**
 * @defgroup IxEthDBPortDefs IXP425 Ethernet Database Port Definitions (IxEthDBPortDefs)
 *
 * @brief IXP425 Ethernet Port Definitions for private MAC learning API
 *
 * @{
 */

#ifndef IxEthDBPortDefs_H
#define IxEthDBPortDefs_H

/** 
 * @brief Port types - currently only Ethernet NPEs are recognized as specific types 
 */
typedef enum
{
    ETH_GENERIC = 0, /**< generic ethernet port - not supported */
    ETH_NPE          /**< specific Ethernet NPE */
} IxEthDBPortType;

/** 
 * @brief Port capabilities - used by ixEthAccDatabaseMaintenance to decide whether it
 * should manually age entries or not depending on the port capabilities.
 *
 * Ethernet NPEs have aging capabilities, meaning that they will age the entries
 * automatically (by themselves).*/
typedef enum
{
    NO_CAPABILITIES   = 0,   /**< no aging capabilities */
    ENTRY_AGING       = 0x1  /**< aging capabilities present */
} IxEthDBPortCapability;

/** 
 * @brief Port Definition - a structure contains the Port type and capabilities 
 */
typedef struct
{
    IxEthDBPortType type;
    IxEthDBPortCapability capabilities;
} IxEthDBPortDefinition;

/** 
 * @brief Port definitions structure, indexed on the port ID
 * @warning Ports 0 and 1 are used by the Ethernet access component therefore
 * it is essential to be left untouched. Port 2 here (WAN) is given as 
 * an example port. The NPE firmware also assumes the NPE B to be
 * the port 0 and NPE C to be the port 1.
 */
static IxEthDBPortDefinition ixEthDBPortDefinitions[] = 
{
    /*    id       type           capabilities */
    {   /* 0 */    ETH_NPE,       ENTRY_AGING },    /* Ethernet NPE B */
    {   /* 1 */    ETH_NPE,       ENTRY_AGING },    /* Ethernet NPE C */
    {   /* 2 */    ETH_GENERIC,   NO_CAPABILITIES } /* WAN port */
};

/** 
 * @def IX_ETH_DB_NUMBER_OF_PORTS
 * @def IX_ETH_DB_NUMBER_OF_PORTS 
 * @brief number of supported ports 
 */
#define IX_ETH_DB_NUMBER_OF_PORTS (sizeof (ixEthDBPortDefinitions) / sizeof (ixEthDBPortDefinitions[0]))

/**
 * @def IX_ETH_DB_PORTS_ASSERTION
 * @brief catch invalid port definitions (<2) with a 
 * compile-time assertion resulting in a duplicate case error. 
 */
#define IX_ETH_DB_PORTS_ASSERTION { switch(0) { case 0 : ; case 1 : ; case IX_ETH_DB_NUMBER_OF_PORTS : ; }}

/** 
 * @def COMPLETE_ETH_PORT_MAP 
 * @brief complete set of ports in use 
 *
 * only ports 0, 1 and 2 are in use - sets bit[n] to 1 if port[n] exists
 */
#define COMPLETE_ETH_PORT_MAP ((1 << IX_ETH_DB_NUMBER_OF_PORTS) - 1)

/** 
 * @def IX_ETH_DB_CHECK_PORT(portID)
 * @brief safety checks to verify whether the port is invalid or uninitialized
 */
#define IX_ETH_DB_CHECK_PORT(portID) \
{ \
    if ((portID) >= IX_ETH_DB_NUMBER_OF_PORTS) \
    { \
        return IX_ETH_DB_INVALID_PORT; \
    } \
    \
    if (!ixEthDBPortInfo[(portID)].enabled) \
    { \
        return IX_ETH_DB_PORT_UNINITIALIZED; \
    } \
}

#endif /* IxEthDBPortDefs_H */
/**
 *@}
 */
