/*
 * smartctl.c
 *
 * Copyright (C) 2000 Michael Cornwell <cornwell@acm.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * You should have received a copy of the GNU General Public License
 * (for example COPYING); if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/hdreg.h>
#include <string.h>
#include "extern.h"
#include "smart_all.h"

unsigned char driveinfo               = FALSE;
unsigned char checksmart              = FALSE;
unsigned char smartvendorattrib       = FALSE;
unsigned char generalsmartvalues      = FALSE;
unsigned char smartselftestlog        = FALSE;
unsigned char smarterrorlog           = FALSE;
unsigned char smartdisable            = FALSE;
unsigned char smartenable             = FALSE;
unsigned char smartstatus             = FALSE;
unsigned char smartexeoffimmediate    = FALSE;
unsigned char smartshortselftest      = FALSE;
unsigned char smartextendselftest     = FALSE;
unsigned char smartshortcapselftest   = FALSE;
unsigned char smartextendcapselftest  = FALSE;
unsigned char smartselftestabort      = FALSE;
unsigned char smartautoofflineenable  = FALSE;
unsigned char smartautoofflinedisable = FALSE;

void ataPrintDriveInfo (struct hd_driveid,char opt[40]);
void ataPrintGeneralSmartValues (struct ata_smart_values);
void ataPrintSmartThresholds (struct ata_smart_thresholds);
void ataPrintSmartErrorlog ( struct ata_smart_errorlog);
// void PrintSmartAttributes ( struct ata_smart_values data);
void PrintSmartAttribWithThres ( struct ata_smart_values data, 
				struct ata_smart_thresholds thresholds);
void ataPrintSmartSelfTestlog (struct ata_smart_selftestlog data);
int ataPsuedoCheckSmart ( struct ata_smart_values , struct ata_smart_thresholds );
void ataPrintSmartAttribName (unsigned char id,FILE *fp);
int ataPrintMain ( int fd,char opt[40]);

//int ataReadHDIdentity (int device, struct hd_driveid *buf);
//int ataReadSmartValues (int device,struct ata_smart_values *);
//int ataReadSmartThresholds (int device, struct ata_smart_thresholds *);
int ataReadErrorLog ( int device, struct ata_smart_errorlog *);
int ataReadSelfTestLog (int device, struct ata_smart_selftestlog *);
int ataSmartStatus ( int device);
int ataSetSmartThresholds ( int device, struct ata_smart_thresholds *);

/* Enable/Disable SMART on device */
//int ataEnableSmart ( int device );
int ataDisableSmart (int device );

/* Automatic Offline Testing */
int ataEnableAutoOffline ( int device );
int ataDisableAutoOffline (int device );

/* S.M.A.R.T. test commands */
int ataSmartOfflineTest (int device);
int ataSmartExtendSelfTest (int device);
int ataSmartShortSelfTest (int device);
int ataSmartShortCapSelfTest (int device);
int ataSmartExtendCapSelfTest (int device);
int ataSmartSelfTestAbort (int device);

/*Check Parameters of Smart Data */

/* int ataVersionInfo ( struct hd_driveid drive) 
*  Returns the latest compability of ATA/ATAPI Version  
*  the device supports								
* Returns -1 if Version command is not supported
*/
//int ataVersionInfo ( struct hd_driveid drive);

/*  int ataSmartSupport ( int device, struct hd_driveid drive)
*   Check if S.M.A.R.T. is supported and enabled in drive 
*	returns -1:if S.M.A.R.T. cabability can not be checked
*	returns	0: if drive does not support S.M.A.R.T.
*			1: if drive supports S.M.A.R.T. but not enabled 
*			2: if drive supports S.M.A.R.T. and enabled 
*		  255: if drive supports S.M.A.R.T. but does not   
*			   support ATA-4. 
*	ATA 3 and lower do not support S.M.A.R.T. enabled bit
*   Attempt a Read S.M.A.R.T. attributes to check if enabled
*/ 

//int ataSmartSupport ( struct hd_driveid drive);
//int ataCheckSmart ( struct ata_smart_values data, struct ata_smart_thresholds thresholds);
int isOfflineTestTime ( struct ata_smart_values data);
int isShortSelfTestTime ( struct ata_smart_values data);
int isExtendedSelfTestTime ( struct ata_smart_values data);
int isSmartErrorLogCapable ( struct ata_smart_values data);
int isSupportExecuteOfflineImmediate ( struct ata_smart_values data);
int isSupportAutomaticTimer ( struct ata_smart_values data);
int isSupportOfflineAbort ( struct ata_smart_values data);
int isSupportOfflineSurfaceScan ( struct ata_smart_values data);
//int isSupportSelfTest (struct ata_smart_values data);


int ParseOpts ( int opts)
{
	
	switch (opts)
	{

	case DRIVEINFO ://'i'
		driveinfo  = TRUE;
		break;		

	case CHECKSMART ://'c'
		driveinfo  = TRUE;
		checksmart = TRUE;		
		break;
/*
	case SMARTVERBOSEALL ://'a'
		driveinfo              = TRUE;
		checksmart	       = TRUE;
		generalsmartvalues     = TRUE;
		smartvendorattrib      = TRUE;
		smarterrorlog          = TRUE;
//		smartselftestlog       = TRUE;
		break;
*/
	case SMARTVENDORATTRIB ://'v'
		smartvendorattrib      = TRUE;
		break;

	case GENERALSMARTVALUES ://'g'
		generalsmartvalues     = TRUE;
		break;

	case SMARTERRORLOG ://'l'
		smarterrorlog          = TRUE;
		break;
/*
	case SMARTSELFTESTLOG ://'L'
		smartselftestlog       = TRUE;
		break;
*/
	case SMARTDISABLE ://'d'
		smartdisable           = TRUE;
		break;

	case SMARTENABLE ://'e'
		smartenable	       = TRUE;
		break;

	case SMARTAUTOOFFLINEENABLE: //'t'
		smartautoofflineenable = TRUE;
		break;

	case SMARTAUTOOFFLINEDISABLE://'T'
		smartautoofflinedisable= TRUE;
		break;

	case SMARTEXEOFFIMMEDIATE://'O'
		smartexeoffimmediate   = TRUE;
		break;

	case SMARTSHORTSELFTEST ://'S'
		smartshortselftest     = TRUE;
		break;

	case SMARTEXTENDSELFTEST ://'X'
		smartextendselftest    = TRUE;
		break;

	case  SMARTSHORTCAPSELFTEST://'s'
		smartshortcapselftest  = TRUE;
		break;

	case SMARTEXTENDCAPSELFTEST://'x'
		smartextendcapselftest = TRUE;
		break;

	case SMARTSELFTESTABORT://'A'
		smartselftestabort     = TRUE;
		break;
	
        default:
		return -1;	
	}
	return 1;
}


// Main Program 

//int main (int argv, char **argc)
int SmartCommand(int commandoption,char *harddisktype,char opt[40])
{
	
	int fd;
	char device[10];	
	strcpy(device,harddisktype);
	
	if(ParseOpts (commandoption)==-1)
			return -2;
		
	fd = open ( device, O_RDWR );

	if ( fd < 0) {
		return -1; //no this device
	}
	
	if ( device[5] == 'h') //only do with ATA device
		return ataPrintMain (fd,opt);
//	else if (device[5] == 's')
//		scsiPrintMain (fd);
	else 
		return -3;
}


void ataPrintDriveInfo (struct hd_driveid drive,char opt[40])
{

	sprintf(opt,"%s",drive.model);
//	printf ("Device: %s  Supports ATA Version %i\n", drive.model, ataVersionInfo ( drive) );
 
}

// void PrintSmartOfflineStatus ( struct ata_smart_values data) 
//   prints verbose value Off-line data collection status byte 

void PrintSmartOfflineStatus ( struct ata_smart_values data,FILE *fp)
{
	fprintf(fp,"Off-line data collection status: ");	
	switch (data.offline_data_collection_status)
	{
	case  0x0: case 0x80:
		fprintf(fp,"(0x%02x)\tOffline data collection activity was\n\t\t\t\t\t",
		data.offline_data_collection_status);
		fprintf(fp,"never started\n");
		break;
	
	case  0x01: case 0x81:
		fprintf(fp,"(0x%02x)\tReserved\n",
		data.offline_data_collection_status);
		break;

	case  0x02: case 0x82:
		fprintf(fp,"(0x%02x)\tOffline data collection activity \n\t\t\t\t\t",
		data.offline_data_collection_status);
		fprintf(fp,"completed without error\n");
		break;
	
	case  0x03: case 0x83:
		fprintf(fp,"(0x%02x)\tReserved\n",
		data.offline_data_collection_status);
		break;
		
	case  0x04: case 0x84:
		fprintf(fp,"(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
		data.offline_data_collection_status);
		fprintf(fp,"suspended by an interrupting command\n");
		break;
	
	case  0x05: case 0x85:
		fprintf(fp,"(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
		 	data.offline_data_collection_status);
		fprintf(fp,"aborted by an interrupting command\n");
		break;
	
	case  0x06: case 0x86:
		fprintf(fp,"(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
		data.offline_data_collection_status);
		fprintf(fp,"aborted by the device with fatal error\n");
		break;
	
	default:
		if ( ((data.offline_data_collection_status >= 0x07) &&
              (data.offline_data_collection_status <= 0x3f)) ||
			 ((data.offline_data_collection_status >= 0xc0) 
                         /* && (data.offline_data_collection_status <= 0xff) */) )
		{
			fprintf(fp,"(0x%02x)\tVendor Specific\n",
			        data.offline_data_collection_status);
		} else {
			fprintf(fp,"(0x%02x)\tReserved\n",
					data.offline_data_collection_status);
		}
	}
}

void PrintSmartSelfExecStatus ( struct ata_smart_values data,FILE *fp)
{
	fprintf(fp,"Self-test execution status:      ");
	switch (data.self_test_exec_status >> 4)
	{
	case  0:
		fprintf(fp,"(%4d)\tThe previous self-test routine completed\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"without error or no self-test has ever \n\t\t\t\t\tbeen run\n");
		break;

	case  1:
		fprintf(fp,"(%4d)\tThe self-test routine was aborted by\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"the host\n");
		break;
	case  2:
		fprintf(fp,"(%4d)\tThe self-test routine was interrupted\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"by the host with a hard or soft reset\n");
		break;
	
	case  3:
		fprintf(fp,"(%4d)\tA fatal error or unknown test error\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"occurred while the device was executing\n\t\t\t\t\t");
		fprintf(fp,"its self-test routine and the device \n\t\t\t\t\t");
		fprintf(fp,"was unable to complete the self-test \n\t\t\t\t\t");
		fprintf(fp,"routine\n");
		break;
	
	case  4:
		fprintf(fp,"(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"a test element that failed and the test\n\t\t\t\t\t");
		fprintf(fp,"element that failed is not known\n");
		break;

	case  5:
		fprintf(fp,"(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"the electrical element of the test\n\t\t\t\t\t");
		fprintf(fp,"failed\n");
		break;
	case  6:
		fprintf(fp,"(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"the servo (and/or seek) element of the \n\t\t\t\t\t");
		fprintf(fp,"test failed\n");
		break;
	case  7:
		fprintf(fp,"(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"the read element of the test failed\n");
		break;
	case  15:
		fprintf(fp,"(%4d)\tSelf-test routine in progess\n\t\t\t\t\t",
			data.self_test_exec_status);
		fprintf(fp,"%1d0%% of test remaining\n", data.self_test_exec_status & 0x0f);
		break;
	default:
			fprintf(fp,"(%4d)\tReserved\n",
			data.self_test_exec_status);
		break;
	}
}

void PrintSmartTotalTimeCompleteOffline ( struct ata_smart_values data,FILE *fp)
{
	fprintf(fp,"Total time to complete off-line \n");
	fprintf(fp,"data collection: \t\t (%4d) Seconds\n", 
			data.total_time_to_complete_off_line);

}

void PrintSmartOfflineCollectCap ( struct ata_smart_values data,FILE *fp)
{
	fprintf(fp,"Offline data collection \n");
	fprintf(fp,"Capabilities: \t\t\t (0x%02x)",
		data.offline_data_collection_capability);

    if (data.offline_data_collection_capability == 0x00)
	{
		fprintf(fp,"\tOff-line data collection not supported\n");
	} else {
	
		if ( isSupportExecuteOfflineImmediate(data) )
		{
			fprintf(fp," SMART EXECUTE OFF-LINE IMMEDIATE\n");
		}
		else
		{
			fprintf(fp,"NO SMART EXECUTE OFF-LINE IMMEDIATE\n");
		}
		
		if ( isSupportAutomaticTimer (data) )
		{
			fprintf(fp,"\t\t\t\t\tAutomatic timer ON/OFF support\n");
		}
		else
		{
			fprintf(fp,"\t\t\t\t\tNO Automatic timer ON/OFF support\n");
		}
		
		if ( isSupportOfflineAbort (data) )
		{
			fprintf(fp,"\t\t\t\t\tAbort Offline Collection upon new\n");
			fprintf(fp,"\t\t\t\t\tcommand\n");
		}
		else
		{
			fprintf(fp,"\t\t\t\t\tSuspend Offline Collection upon new\n");
			fprintf(fp,"\t\t\t\t\tcommand\n");
		}

		if ( isSupportOfflineSurfaceScan (data) )
		{
			fprintf(fp,"\t\t\t\t\tOffline surface scan supported\n");
		}
		else
		{
			fprintf(fp,"\t\t\t\t\tNO Offline surface scan supported\n");
		}

		if ( isSupportSelfTest(data) )
		{
			fprintf(fp,"\t\t\t\t\tSelf-test supported\n");
		}
		else
		{
			fprintf(fp,"\t\t\t\t\tNO Self-test supported\n");
		}
	}
}

void PrintSmartCapability ( struct ata_smart_values data,FILE *fp)
{
	fprintf(fp,"Smart Capablilities:           ");
	fprintf(fp,"(0x%04x)\t", data.smart_capability);
	if (data.smart_capability == 0x00)
	{
		fprintf(fp,"automatic saving of SMART data"); 
		fprintf(fp,"\t\t\t\t\tis not implemented\n");
	} else {
	
		if ( data.smart_capability & 0x01 )
		{
			fprintf(fp,"Saves SMART data before entering \n");
			fprintf(fp,"\t\t\t\t\tpower-saving mode\n");
		}
		else
		{
			fprintf(fp,"does not save SMART data before  \n");
			fprintf(fp,"\t\t\t\t\tentering power-saving mode\n");
		}
		
		if ( data.smart_capability & 0x02 )
		{
			fprintf(fp,"\t\t\t\t\tSupports SMART auto save timer\n");
		}
	}
}

void PrintSmartErrorLogCapability ( struct ata_smart_values data,FILE *fp)
{

	fprintf(fp,"Error logging capability:       ");
    
	if ( isSmartErrorLogCapable(data) )
	{
		fprintf(fp," (0x%02x)\tError logging supported\n",
			data.errorlog_capability);
	} else 
	{
		fprintf(fp," (0x%02x)\tError logging NOT supported\n",
			data.errorlog_capability);
	}
}

void PrintSmartShortSelfTestPollingTime ( struct ata_smart_values data,FILE *fp)
{
	if ( isSupportSelfTest(data) )
	{
		fprintf(fp,"Short self-test routine \n");
		fprintf(fp,"recommended polling time: \t (%4d) Minutes\n", 
			data.short_test_completion_time);

	}
	else
	{
		fprintf(fp,"Short self-test routine \n");
		fprintf(fp,"recommended polling time: \t        Not Supported\n");
	}
}


void PrintSmartExtendedSelfTestPollingTime ( struct ata_smart_values data,FILE *fp)
{
	if ( isSupportSelfTest(data) )
	{
		fprintf(fp,"Extended self-test routine \n");
		fprintf(fp,"recommended polling time: \t (%4d) Minutes\n", 
			data.extend_test_completion_time);
	}
	else
	{
		fprintf(fp,"Extended self-test routine \n");
		fprintf(fp,"recommended polling time: \t        Not Supported\n");
	}
}

void PrintSmartAttribWithThres ( struct ata_smart_values data, struct ata_smart_thresholds thresholds)
{
	int i,j;
	FILE *fp;
	
	fp=fopen("/tmp/attribwithThres","w");
	
	fprintf(fp,"Vendor Specific SMART Attributes with Thresholds:\n");
	fprintf(fp, "Revision Number: %i\n", data.revnumber);
	fprintf(fp,"Attribute                    Flag     Value Worst Threshold Raw Value\n");
	
	for ( i = 0 ; i < 30 ; i++ )
	{
	   
	 	if ( (data.vendor_attributes[i].id !=0) &&
			 (thresholds.thres_entries[i].id != 0))
		{
		   
                     ataPrintSmartAttribName(data.vendor_attributes[i].id,fp);
                     fprintf(fp," 0x%04x   %.3i   %.3i   %.3i       ", 
				data.vendor_attributes[i].status_flag,
				data.vendor_attributes[i].normalized,
				data.vendor_attributes[i].worse_normal,
				thresholds.thres_entries[i].normalized_threshold);
		   for (j = 5 ; j >= 0 ; j--) 
		   {
			   fprintf(fp,"%02x", data.vendor_attributes[i].raw[j]);
		   }
		   fprintf(fp,"\n");
	   }
	}
	fclose(fp);
}

void ataPrintGeneralSmartValues  ( struct ata_smart_values data)
{
	FILE *fp;
	
	fp=fopen("/tmp/generalvalues","w");
	
	fprintf (fp,"\nGeneral Smart Values: \n");

	PrintSmartOfflineStatus (data,fp); 
	fprintf(fp,"\n");
	
	if (isSupportSelfTest(data))
	{
		PrintSmartSelfExecStatus (data,fp);
		fprintf(fp,"\n");
	}
	
	PrintSmartTotalTimeCompleteOffline (data,fp);
	fprintf(fp,"\n");
	
	PrintSmartOfflineCollectCap (data,fp);
	fprintf(fp,"\n");
	
	PrintSmartCapability ( data,fp);
	fprintf(fp,"\n");

	PrintSmartErrorLogCapability (data,fp);
	fprintf(fp,"\n");
	
	if (isSupportSelfTest(data))
	{
		PrintSmartShortSelfTestPollingTime (data,fp);
		fprintf(fp,"\n");

		PrintSmartExtendedSelfTestPollingTime (data,fp);
		fprintf(fp,"\n");
	}
	fclose(fp);
}

void ataPrintSmartThresholds (struct ata_smart_thresholds data)
{
	int i;

	printf ("Smart Thresholds\n");
	printf ("Smart Threshold Revision Number: %i\n", data.revnumber);
	
	for ( i = 0 ; i < 30 ; i++)
	{
		if (data.thres_entries[i].id != 0)	
		printf ("Atrribute %3i threshold: %02x (%2i)\n", data.thres_entries[i].id, 
				data.thres_entries[i].normalized_threshold, 
				data.thres_entries[i].normalized_threshold);	
	}
}

void ataPrintSmartErrorlog (struct ata_smart_errorlog data)
{
	int i,j;
	FILE * fp;
	
	fp=fopen("/tmp/errorlog","w");
	
	fprintf (fp,"SMART Error Log:\n");
	fprintf (fp, "SMART Error Logging Version: %i\n", data.revnumber);

	if ( ! data.error_log_pointer)
	{
		fprintf (fp,"No Errors Logged\n");
		return;
	}

	fprintf (fp, "Error Log Data Structure Pointer: %02x\n", data.error_log_pointer);
	fprintf (fp, "ATA Error Count: %u\n", data.ata_error_count);
	fprintf (fp, "Non-Fatal Count: %u\n", data.non_fatal_count);
	
	
	for (i = 0; i < 5; i++ )
	{			
		if ( ( data.errorlog_struct[i].commands[0].devicecontrolreg ||
				   data.errorlog_struct[i].commands[0].featuresreg ||
                   data.errorlog_struct[i].commands[0].sector_count ||
                   data.errorlog_struct[i].commands[0].sector_number ||
                   data.errorlog_struct[i].commands[0].cylinder_low ||
                   data.errorlog_struct[i].commands[0].cylinder_high ||
                   data.errorlog_struct[i].commands[0].drive_head ||
                   data.errorlog_struct[i].commands[0].commandreg )
				   != 0)
		{
			fprintf(fp,"\nError Log Structure %i:\n", i+1);
			fprintf(fp,"DCR   FR   SC   SN   CL   SH   D/H   CR   Timestamp\n");
		
			for ( j = 0; j < 6; j++)
			{						
		 				
				fprintf (fp, " %02x   %02x   %02x   %02x   %02x   %02x    %02x   %02x     %u\n", 
					data.errorlog_struct[i].commands[j].devicecontrolreg,
					data.errorlog_struct[i].commands[j].featuresreg,
					data.errorlog_struct[i].commands[j].sector_count,
					data.errorlog_struct[i].commands[j].sector_number,
					data.errorlog_struct[i].commands[j].cylinder_low,
					data.errorlog_struct[i].commands[j].cylinder_high,
					data.errorlog_struct[i].commands[j].drive_head,
					data.errorlog_struct[i].commands[j].commandreg,
					( unsigned int) data.errorlog_struct[i].commands[j].timestamp / 1000); 
			}
		}

		if ( (data.errorlog_struct[i].error_struct.error_condition ||
			  data.errorlog_struct[i].error_struct.state) != 0)
		{
			fprintf (fp, "Error condition: %3i\t",
			data.errorlog_struct[i].error_struct.error_condition);
			fprintf (fp, "Error State:     %3i\n", 
			data.errorlog_struct[i].error_struct.state);
			fprintf (fp, "Number of Hours in Drive Life: %u (life of the drive in hours)\n",  
				data.errorlog_struct[i].error_struct.timestamp );	
		} 
	}
	fclose(fp);
}
	
void ataPrintSmartSelfTestlog (struct ata_smart_selftestlog data)
{
	/* excluded from first release */
}

int ataPsuedoCheckSmart ( struct ata_smart_values data, struct ata_smart_thresholds thresholds)
{
	int i;
	int failed = 0;
	for ( i = 0 ; i < 30 ; i++ )
	{
	   
	 if ( (data.vendor_attributes[i].id !=0) &&   
		  (thresholds.thres_entries[i].id != 0) &&
		  ( data.vendor_attributes[i].status_flag & 0x1) &&
		  ( data.vendor_attributes[i].normalized <
		    thresholds.thres_entries[i].normalized_threshold) &&
		  (thresholds.thres_entries[i].normalized_threshold != 0xFE) )
		{
//		   printf("Attribute ID %i Failed\n", data.vendor_attributes[i].id);
			
		   failed = 1;
	    }
	}

	if ( failed )
	{
//		printf ("Please save all data and call drive manufacture immediately. \n");
		return 3;
	}
	else
	{
//		printf ("Check S.M.A.R.T. Passed\n");
		return 4;
	}
}

void ataPrintSmartAttribName ( unsigned char id,FILE *fp)
{
	switch (id)
 	{
	
        case 1:
	   fprintf(fp,"(  1)Raw Read Error Rate    ");
	   break;
	case 2:
	   fprintf(fp,"(  2)Throughput Performance ");
	   break;
        case 3:
           fprintf(fp,"(  3)Spin Up Time           ");
           break;
        case 4:
	   fprintf(fp,"(  4)Start Stop Count       ");
	   break;
        case 5:
	   fprintf(fp,"(  5)Reallocated Sector Ct  ");
	   break;
	case 7:
	   fprintf(fp,"(  7)Seek Error Rate        ");
  	   break;
	case 8:
	   fprintf(fp,"(  8)Seek Time Preformance  ");
           break;
	case 9:
           fprintf(fp,"(  9)Power On Hours         ");
 	   break;
	case 10:
 	   fprintf(fp,"( 10)Spin Retry Count       ");
	   break;
  	case 12:
	   fprintf(fp,"( 12)Power Cycle Count      ");
	   break;
 	case 196:
	   fprintf(fp,"(196)Reallocated Event Count");
	   break;
	case 197:
	   fprintf(fp,"(197)Current Pending Sector ");
      	   break;
	case 198:
 	   fprintf(fp,"(198)Offline Uncorrectable  ");
	   break;
	case 199:
	   fprintf(fp,"(199)UDMA CRC Error Count   ");
	   break;
	default:
	   fprintf(fp,"(%3d)Unknown Attribute      ", id);
           break;
	}
}	

int ataPrintMain ( int fd,char opt[40])
{
	struct hd_driveid drive;
	struct ata_smart_values smartval;
	struct ata_smart_thresholds smartthres;
	struct ata_smart_errorlog smarterror;
	
	int return_value=0;
	
	if ( driveinfo )//'i' or 'c' or 'a'
	{

		if (  ataReadHDIdentity ( fd, &drive) != 0 )
		{
			return -4;
		}
		
		ataPrintDriveInfo(drive,opt); 

		switch (ataSmartSupport(drive))
		{
			case 0:
				return -7;
			case 1:
				return -8;
			case 2: 
				return_value=0;
				break;
			case 255:
				if ( ataSmartStatus(fd) != 0) 
					return -8;
				else
					return_value=0;
				break;
			default :
				break;
		}
	}

	if ( smartdisable )//'d'
	{
		if ( ataDisableSmart(fd) != 0) 
		{
			return -6;
		}
			return 0;
	}

	if ( smartenable )//'e'
	{
		if ( ataEnableSmart(fd) != 0) 
		{
			return -6;
		}
		return 0;
	}
	
	/* for everything else read values and thresholds 
	   are needed */	

	if ( ataReadSmartValues ( fd, &smartval) != 0 )
	{
		return -4;
	}

	if ( ataReadSmartThresholds ( fd, &smartthres) != 0 )
	{
		return -5;
	}
	
	if ( checksmart )//'c' or 'a'
	{
		/* pseudo is used because linux does not support access to
		   Task Fiule registers */
		if(ataPsuedoCheckSmart ( smartval , smartthres)==3)
		{
			return_value=-6;
		}
		else
		{
			return_value=0;
		}
		
	}

	if (  generalsmartvalues )//'g'
	{
		ataPrintGeneralSmartValues( smartval ); 
		return 0;
	}

	if ( smartvendorattrib )//'v'
	{
		PrintSmartAttribWithThres( smartval, smartthres);
		return 0;
	}

	if ( smarterrorlog )//'l'
	{
		if ( isSmartErrorLogCapable(smartval) == 0)
		{
			return 0;
		} 
		else
		{
			if ( ataReadErrorLog ( fd, &smarterror) != 0 )
			{
				return -6;
			}
			else
			{
				ataPrintSmartErrorlog ( smarterror);
				return 0;
			}
		}
	}

	if (  smartautoofflineenable  )//'t'
	{
		if ( !isSupportAutomaticTimer (smartval))
		{
			return -9;
		}

		if ( ataEnableAutoOffline (fd) != 0) 
		{

			return -6;
		}
		return 0;
	}

	if (  smartautoofflinedisable  )//'T'
	{
		if ( !isSupportAutomaticTimer (smartval))
		{
			return -9;
		}
			
		if ( ataDisableAutoOffline (fd) != 0) 
		{
			return -6;
		}
			return 0;
	}

	if ( smartexeoffimmediate )//'O'
	{
		if ( ataSmartOfflineTest (fd) != 0) 
		{
			return -6;
		}
		return 0;
	}

	if ( smartshortcapselftest )//'s'
	{
		if ( ! isSupportSelfTest(smartval) )
		{
			return -10;
		}
		
		if ( ataSmartShortCapSelfTest (fd) != 0) 
		{
			return -6;
		}
		return 0;
	}

	if ( smartshortselftest )//'S'
	{
		if ( ! isSupportSelfTest(smartval) )
		{
			return -10;
		}
		
		if ( ataSmartShortSelfTest (fd) != 0) 
		{
			return -6;
		}

		return 0;
	}
	
	if ( smartextendselftest )//'X'
	{
		
		if ( ! isSupportSelfTest(smartval) )
		{

			return -10;
		}

		if ( ataSmartExtendSelfTest (fd) != 0) 
		{

			return -11;
		}
		
		return 0;
	}

	
	if ( smartextendcapselftest )//'x'
	{
		
		if ( ! isSupportSelfTest(smartval) )
		{
			return -10;
		}


		if ( ataSmartExtendCapSelfTest (fd) != 0) 
		{
			return -11;
		}
		
		return 0;
	}

	if ( smartselftestabort )//'A'
	{
		
		if ( ! isSupportSelfTest(smartval) )
		{
			return -10;
		}

		if ( ataSmartSelfTestAbort (fd) != 0) 
		{
			return -6;
		}
		
		return 0;
     }
     return 	return_value;
	
}

int ataReadSelfTestLog (int device, struct ata_smart_selftestlog *data)
{	

	char buf[512];

	unsigned char parms[4] = { WIN_SMART, 0x06, SMART_READ_LOG_SECTOR, 1};
	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms, &buf ) != 0)
	{
//		perror ("Smart Error Log Read failed");
		return -1;
	}
	
	memcpy( data, buf, 512);
	
	return 0;
}

int ataReadErrorLog (int device, struct ata_smart_errorlog *data)
{	

	char buf[512];
	unsigned char parms[4] = { WIN_SMART, 0x01, SMART_READ_LOG_SECTOR, 1};
         	
         	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms, &buf ) != 0)
	{
//		perror ("Smart Error Log Read failed");
		return -1;
	}
	memcpy( data, buf, 512);

	return 0;
}

int ataSetSmartThresholds ( int device, struct ata_smart_thresholds *data)
{	
	char buf[512];
	unsigned char parms[4] = { WIN_SMART, 1, 0xD7, 1};
	
	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms, &buf ) != 0)
	{
//		perror ("Smart Thresholds Read failed");
		return -1;
	}
	memcpy( data, &buf, sizeof(buf));

	return 0;
}


int ataDisableSmart (int device )
{	

	
	unsigned char parms[4] = { WIN_SMART, 1, SMART_DISABLE, 0};
	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms ) != 0)
	{
//		perror ("Smart Disable failed");
		return -1;
	}

	return 0;
}


int ataEnableAutoOffline (int device )
{	

	/* timer hard coded to 4 hours */
	unsigned char parms[4] = { WIN_SMART, 248, SMART_AUTO_OFFLINE, 0};
	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms ) != 0)
	{
//		perror ("Smart Enable Automatic Offline failed");
		return -1;
	}

	return 0;
}


int ataDisableAutoOffline (int device )
{	
	
	unsigned char parms[4] = { WIN_SMART, 0, SMART_AUTO_OFFLINE, 0};
	
	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms ) != 0)
	{
//		perror ("Smart Disable Automatic Offline failed");
		return -1;
	}

	return 0;
}


int ataSmartStatus (int device )
{	
	
	unsigned char parms[4] = { WIN_SMART, 1, SMART_STATUS, 0};
	

	if (ioctl ( device , HDIO_DRIVE_CMD,  &parms, NULL) != 0)
	{
#ifdef DEBUG
		perror ("Smart Disable failed");
#endif
		return -1;
	}

	return 0;
}

int ataSmartTest (int device, int testtype)
{	

	unsigned char parms[4] = { WIN_SMART, testtype, SMART_IMMEDIATE_OFFLINE	, 0};
	
	if (ioctl ( device , HDIO_DRIVE_CMD, &parms) != 0)
	{
//		perror ("Smart Offline failed");
		return -1;
	
	}
//	printf("Completed Off-line command\n");
	
	return 0;
}


int ataSmartOfflineTest (int device)
{	
	return ataSmartTest( device, OFFLINE_FULL_SCAN  );
}


int ataSmartShortSelfTest (int device)
{	
	return ataSmartTest( device, SHORT_SELF_TEST  );
}

int ataSmartExtendSelfTest (int device)
{	
	return ataSmartTest( device, EXTEND_SELF_TEST  );
}

int ataSmartShortCapSelfTest (int device)
{	
	return ataSmartTest( device, SHORT_CAPTIVE_SELF_TEST );
}

int ataSmartExtendCapSelfTest (int device)
{
	return ataSmartTest( device, EXTEND_CAPTIVE_SELF_TEST );
}

int ataSmartSelfTestAbort (int device)
{
	return ataSmartTest( device, 127 );
}

/* Test Time Functions */


int isOfflineTestTime ( struct ata_smart_values data)
{
	return (int) data.total_time_to_complete_off_line;
}

int isShortSelfTestTime ( struct ata_smart_values data)
{
	return (int) data.short_test_completion_time;
}

int isExtendedSelfTestTime ( struct ata_smart_values data)
{
	return (int) data.extend_test_completion_time;
}

int isSmartErrorLogCapable ( struct ata_smart_values data)
{
	return data.errorlog_capability & 0x01;
}

int isSupportExecuteOfflineImmediate ( struct ata_smart_values data)
{
	return data.offline_data_collection_capability & 0x01;
}

int isSupportAutomaticTimer ( struct ata_smart_values data)
{
	return data.offline_data_collection_capability & 0x02;
}

int isSupportOfflineAbort ( struct ata_smart_values data)
{
	return data.offline_data_collection_capability & 0x04;
}

int isSupportOfflineSurfaceScan ( struct ata_smart_values data)
{
	return data.offline_data_collection_capability & 0x08;
}
