/**
 * @file IxEthAccCodelet.c
 *
 * @date 22 April 2002
 *
 * @brief This file contains the implementation of the Ethernet Access Codelet.
 *
 * Descriptions of the functions used in this codelet is contained in
 * IxEthAccCodelet_p.h
 *
 * IxEthAccCodelet API Functions:
 *       ixEthAccCodeletInit()
 *       ixEthAccCodeletUninit()
 *       ixEthAccCodeletRxSink()
 *       ixEthAccCodeletSwLoopback()
 *       ixEthAccCodeletTxGenRxSinkLoopback()
 *       ixEthAccCodeletPhyLoopback()
 *       ixEthAccCodeletBridge()
 *       ixEthAccCodeletDBLearning()
 *       ixEthAccCodeletShow()
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/

/*
 * Put the system defined include files required.
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef __vxworks
#include <end.h>    /* END drivers */
#include <endLib.h> /* END drivers */
#endif

#include <vxWorks.h>
#include <taskLib.h>
#include <sysLib.h>

/*
 * Put the user defined include files required.
 */

#include "IxTypes.h"
#include "IxQMgr.h"
#include "IxNpeDl.h"
#include "IxNpeMh.h"
#include "IxFeatureCtrl.h"
#include "IxOsServices.h"
#include "IxOsCacheMMU.h"
#include "ix_ossl.h"
#include "IxEthDB.h"
#include "IxEthAccCodelet.h"
#include "IxNpeMicrocode.h"

#include "ixp425.h"	/* Chip level definitions required */
#include "IxEthAccCodelet_p.h"

#ifdef IX_ETHACC_CODELET_USE_NVRAM_MAC 
#include "config.h"
#endif

#ifdef __linux
/* use interrupts for performances */
#define IX_ETH_CODELET_QMGR_DISPATCH_MODE TRUE
#else
/* use polled mode for performances */
#define IX_ETH_CODELET_QMGR_DISPATCH_MODE FALSE
#endif

/*
 * defines and macros used in this file.
 */
/**
 * @def ETH_NPEB_IMAGEID
 * @brief NPE B firmware image to load for the required features
 */
#define ETH_NPEB_IMAGEID IX_NPEDL_NPEIMAGE_NPEB_ETH

/**
 * @def ETH_NPEC_IMAGEID
 * @brief NPE C firmware image to load for the required features
 */
#define ETH_NPEC_IMAGEID IX_NPEDL_NPEIMAGE_NPEC_ETH

/*
 * Variable declarations global to this file only.
 */

PRIVATE IX_MBUF *ixEthAccCodeletBufPool;

PRIVATE UINT8 *ixEthAccCodeletBufData;

PRIVATE UINT8 compData[IX_ETHACC_CODELET_PCK_LEN];

PRIVATE IX_MBUF *ixEthAccCodeletFreeBufQ;

PRIVATE volatile UINT32 memPoolFreeCnt;

PRIVATE UINT32 phyAddresses[IX_ETHACC_CODELET_MAX_PORT];
PRIVATE UINT32 maxPhyNo;

PRIVATE IxQMgrDispatcherFuncPtr ixEthAccCodeletDispatcherFunc ;

PRIVATE IxEthAccCodeletPhyConf phyConf = 
{
    TRUE,	/* 100 Mbits */
    TRUE,	/* Full duplex */
    TRUE	/* Autonegotiate */
};

PRIVATE struct _IxEthAccCodeletStats
{
    UINT32 rxCount;
    UINT32 txCount;
} IxEthAccCodeletStats[IX_ETHACC_CODELET_MAX_PORT];
  
PRIVATE BOOL ixEthAccCodeletInitialised = FALSE;
PRIVATE BOOL ixEthAccCodeletNpeInitialised = FALSE;

PRIVATE volatile BOOL ixEthAccCodeletPollEnabled = FALSE;

PRIVATE IxMutex ixEthAccCodeletStatsPollTaskRunning;
PRIVATE IxMutex ixEthAccCodeletDBMaintenanceTaskRunning;
PRIVATE IxMutex ixEthAccCodeletDispatcherPollRunning;

PRIVATE volatile BOOL ixEthAccCodeletStatsPollTaskStop = FALSE;
PRIVATE volatile BOOL ixEthAccCodeletDBMaintenanceTaskStop = FALSE;
PRIVATE volatile BOOL ixEthAccCodeletDispatcherPollStop = FALSE;

/*
 * Static function prototypes.
 */

PRIVATE IX_STATUS ixEthAccCodeletThreadsStop(void);
PRIVATE IX_STATUS ixEthAccCodeletDispatcherStart(BOOL useInterrupt);
PRIVATE IX_STATUS ixEthAccCodeletDispatcherStop(BOOL useInterrupt);
PRIVATE void ixEthAccCodeletDispatcherPoll(void* arg, void** ptrRetObj);
PRIVATE void ixEthAccCodeletDBMaintenanceTask(void* arg, void** ptrRetObj);
PRIVATE IX_STATUS ixEthAccCodeletPhyInit(BOOL  phyLoopback);
PRIVATE IX_STATUS ixEthAccCodeletMemPoolInit(void);
PRIVATE IX_STATUS ixEthAccCodeletMacSet(void);
PRIVATE void ixEthAccCodeletMbufChainSizeSet(IX_MBUF *mBufPtr);
PRIVATE IX_STATUS ixEthAccCodeletRecoverBuffers(void);
PRIVATE IX_STATUS ixEthAccCodeletReplenishBuffers(IxEthAccPortId portNo, UINT32 num); 
PRIVATE IX_STATUS ixEthAccCodeletLoop(void);
PRIVATE void ixEthAccCodeletStatsPollTask(void* arg, void** ptrRetObj);
PRIVATE IX_STATUS ixEthAccCodeletLinkUpCheck(unsigned int phyNo);
PRIVATE IX_STATUS ixEthAccCodeletStart(IxEthAccPortRxCallback port1RxCB,
			               IxEthAccPortRxCallback port2RxCB,
			               IxEthAccPortTxDoneCallback port1TxDoneCB,
				       IxEthAccPortTxDoneCallback port2TxDoneCB);
PRIVATE IX_STATUS ixEthAccCodeletStop(IxEthAccPortRxCallback port1RxCB,
				      IxEthAccPortRxCallback port2RxCB,
				      IxEthAccPortTxDoneCallback port1TxDoneCB,
				      IxEthAccPortTxDoneCallback port2TxDoneCB);



/* Ethernet callbacks */

PRIVATE void ixEthAccCodeletRxSinkCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletTxGenRxSinkLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletTxGenRxSinkLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr);
PRIVATE void ixEthAccCodeletPhyLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletPhyLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr);
PRIVATE void ixEthAccCodeletSwLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletSwLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr);
PRIVATE void ixEthAccCodeletBridgeRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletBridgeTxCB(UINT32 cbTag, IX_MBUF* mBufPtr);
PRIVATE void ixEthAccCodeletMemPoolFreeRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId);
PRIVATE void ixEthAccCodeletMemPoolFreeTxCB(UINT32 cbTag, IX_MBUF* mBufPtr);

/*
 * Function definition: ixEthAccCodeletMain()
 *
 * See header file for documentation.
 */
PUBLIC IX_STATUS
ixEthAccCodeletMain(IxEthAccCodeletOperation operationType)
{
 
  if (IX_ETHACC_CODELET_PHY_LOOPBACK == operationType) 
  {
      if (IX_SUCCESS != ixEthAccCodeletInit(TRUE))
      {
           printf("ixEthAccCodeletInit() fails ! Exit \n") ;
           return IX_FAIL ;
      } 
  }
  else
  {
      if (IX_SUCCESS != ixEthAccCodeletInit(FALSE))
      {
          printf("ixEthAccCodeletInit() fails ! Exit \n") ;
          return IX_FAIL ;
      } 
  }

  switch (operationType)
  {
    case IX_ETHACC_CODELET_RX_SINK :
         if (IX_SUCCESS != ixEthAccCodeletRxSink())
         {
	     printf("ixEthAccCodeletRxSink() fails ! Exit \n") ;
	     return IX_FAIL ;
         }
	 break ;

    case IX_ETHACC_CODELET_SW_LOOPBACK :
         if (IX_SUCCESS != ixEthAccCodeletSwLoopback())
         {
	     printf("ixEthAccCodeletSwLoopback() fails ! Exit \n") ;
	     return IX_FAIL ;
         }
	 break ;
  
    case IX_ETHACC_CODELET_TXGEN_RXSINK : 
         if (IX_SUCCESS != ixEthAccCodeletTxGenRxSinkLoopback())
         {
	     printf("ixEthAccCodeletTxGenRxSinkLoopback() fails ! Exit \n") ;
	     return IX_FAIL ;
         }
	 break ;

    case IX_ETHACC_CODELET_PHY_LOOPBACK :
         if(IX_SUCCESS != ixEthAccCodeletPhyLoopback())
           {
	     printf("ixEthAccCodeletPhyLoopback() fails ! Exit \n") ;
	     return IX_FAIL ;
           }
	 break;

    case IX_ETHACC_CODELET_BRIDGE :
         if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_B0)
         {
             if ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) &&
                 (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED))
             {
	         /* For B0 Silicon, Bridge operation only works when both Eth
                  * NPEs are available
                  */
                 if (IX_SUCCESS != ixEthAccCodeletBridge())
                 {
	             printf("ixEthAccCodeletBridge() fails ! Exit \n") ;
	             return IX_FAIL ;
                 }
             }
             else
	     {
  	         printf ("Bridge operation needs two Eth coprocessors.\n");
                 printf ("Underlying silicon only has one Eth coprocessor!\n");
                 return IX_FAIL ; 
             }
         }
         else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                   IX_FEATURE_CTRL_SILICON_TYPE_A0) 
	 {
	     /* A0 silicon has two Eth coprocessors */           
             if (IX_SUCCESS != ixEthAccCodeletBridge())
             {
	         printf("ixEthAccCodeletBridge() fails ! Exit \n") ;
                 return IX_FAIL ;
             }
         }
         else
	 {
	   printf("Undefined conditions for silicon stepping. \n");
           return IX_FAIL ;
         } 
 
      break;

    case IX_ETHACC_CODELET_ETH_LEARNING :
         if (IX_SUCCESS != ixEthAccCodeletDBLearning())
         {
	     printf("ixEthAccCodeletDBLearning() fails ! Exit \n") ;
	     return IX_FAIL ;
         }
	 break ;

    default :
         printf("ERROR: Invalid input. Please choose again...\n ");
 	 printf(">ixEthAccCodeletMain(operationType)\n");
	 printf("Where operationType : %d = Rx Sink\n", IX_ETHACC_CODELET_RX_SINK);
	 printf("                      %d = Sw Loopback\n", IX_ETHACC_CODELET_SW_LOOPBACK);
	 printf("                      %d = Tx Gen/Rx Sink Loopback\n", IX_ETHACC_CODELET_TXGEN_RXSINK);
	 printf("                      %d = Perform PHY loopback on the same port\n",IX_ETHACC_CODELET_PHY_LOOPBACK);
	 printf("                      %d = Bridge\n",IX_ETHACC_CODELET_BRIDGE);
         printf("                      %d = Eth DB Learning\n", IX_ETHACC_CODELET_ETH_LEARNING);               
         return IX_FAIL ;
    }
    return IX_SUCCESS ;
}

/*
 * Function definition: ixEthAccCodeletInit()
 *
 * See header file for documentation.
 */
IX_STATUS ixEthAccCodeletInit(BOOL phyLoopback)
{  
    unsigned int statstid;

    if(ixEthAccCodeletInitialised) 
    {
	printf("Ethernet codelet already initialised\n");
	return(IX_SUCCESS);
    }

#ifdef __vxworks
    /* When the ixe drivers are running, the codelets
    * cannot run.
    */
    if (endFindByName ("ixe", 0) != NULL)
    {
      printf("FAIL : Driver ixe0 detected\n");
      return IX_FAIL;
    }
    if (endFindByName ("ixe", 1) != NULL)
    {
      printf("FAIL : Driver ixe1 detected\n");
      return IX_FAIL;
    }
#endif

    /* Create mutexes for thread control */
    ixEthAccCodeletStatsPollTaskStop = TRUE;
    ixEthAccCodeletDBMaintenanceTaskStop = TRUE;
    ixEthAccCodeletDispatcherPollStop = TRUE;

    ixOsServMutexInit (&ixEthAccCodeletStatsPollTaskRunning);
    ixOsServMutexInit (&ixEthAccCodeletDBMaintenanceTaskRunning);
    ixOsServMutexInit (&ixEthAccCodeletDispatcherPollRunning);

    /* Initialise Queue Manager */
    printf("Initialising Queue Manager...\n");
    if (ixQMgrInit() != IX_SUCCESS)
    {
	printf("Error initialising queue manager!\n");
	return (IX_FAIL);
    }

    /* Start the Queue Manager dispatcher */   
    if(ixEthAccCodeletDispatcherStart(IX_ETH_CODELET_QMGR_DISPATCH_MODE) != IX_SUCCESS)
    {
	printf("Error starting queue manager dispatch loop!\n");
	return (IX_FAIL);
    }

    /* Initialise and Start NPEs */
    printf ("Initialising and Start NPEs...\n");
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
      /*
       * If it is B0 Silicon, we download NPE image only when Eth Coprocessor is 
       * available.
       */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEB_IMAGEID))
            {
	        printf ("Error initialising and starting NPE B!\n");
	        return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEC_IMAGEID))
            {
	        printf ("Error initialising and starting NPE C!\n");
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        /* 
         * If it is A0 silicon, all Ethernet Coprocessors are available. 
         * So, we no need to check availability of Eth Coprocessor
         */
        if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEB_IMAGEID))
        {
	    printf ("Error initialising and starting NPE B!\n");
   	    return (IX_FAIL);
        }

        if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEC_IMAGEID))
        {
	    printf ("Error initialising and starting NPE C!\n");
     	    return (IX_FAIL);
        }
    }   
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    /* Initialise NPE Message handler */
    printf("\nStarting NPE message handler...\n");
    if(ixNpeMhInitialize(IX_NPEMH_NPEINTERRUPTS_YES) != IX_SUCCESS)
    {
	printf("Error initialising NPE Message handler!\n");
	return (IX_FAIL);
    }

    ixEthAccCodeletNpeInitialised = TRUE;

    /***********************************************************************
     *
     * System initialisation done. Now initialise Ethernet Access components. 
     *
     ***********************************************************************/

    if (ixEthAccInit() != IX_ETH_ACC_SUCCESS)
    {
	printf("Error initialising Ethernet access driver!\n");
	return (IX_FAIL);
    }

    /* Initialise Ethernet ports */
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
      /*
       * If it is B0 Silicon, we only enable port when its corresponding  
       * Eth Coprocessor is available.
       */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (ixEthAccPortInit(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error initialising Ethernet port 1!\n");	
	        return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (ixEthAccPortInit(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error initialising Ethernet port 2!\n");	
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        /*
         * If it is A0 Silicon, we enable both as both Eth Coprocessors 
         * are available. 
         */ 
        if(ixEthAccPortInit(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error initialising Ethernet port 1!\n");	
	    return (IX_FAIL);
        }

        if(ixEthAccPortInit(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error initialising Ethernet port 2!\n");	
	    return (IX_FAIL);
        }
    }
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    /* Find and initialise all available PHYs */
    if(ixEthAccCodeletPhyInit(phyLoopback) != IX_SUCCESS)
    {
	printf("Error initialising Ethernet phy(s)!\n");
	return (IX_FAIL);
    }

    /* Program MAC addresses for available PHYs */
    if(ixEthAccCodeletMacSet() != IX_SUCCESS)
    {
	printf("Error programming MAC address for Ethernet phy(s)!\n");
	return (IX_FAIL);
    }

    /* Set ports to promiscuous mode  */
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
      /*
       * If it is B0 Silicon, we need to check for Eth Coprocessor's availability
       * before we enable promiscuous mode.     
       */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthAccPortPromiscuousModeSet(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
            {
   	        printf("Error setting promiscuous mode for port 1\n");
	        return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthAccPortPromiscuousModeSet(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error setting promiscuous mode for port 2\n");
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        /* 
         * If it is A0 silicon, all Ethernet Coprocessors are available. 
         */
        if(ixEthAccPortPromiscuousModeSet(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
        {
   	    printf("Error setting promiscuous mode for port 1\n");
	    return (IX_FAIL);
        }

        if(ixEthAccPortPromiscuousModeSet(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error setting promiscuous mode for port 2\n");
	    return (IX_FAIL);
        }
    }   
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    /* Initialise MBUF pool */
    if(ixEthAccCodeletMemPoolInit() != IX_SUCCESS)
    {
	printf("Error initialising mBuf pool\n");
	return (IX_FAIL);
    }

    /* Set scheduling discipline */
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
      /*
       * If it is B0 Silicon, we set Tx Scheduling Discipline when its corresponding  
       * Eth Coprocessor is available.
       */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            ixEthAccTxSchedulingDisciplineSet(IX_ETH_PORT_1, FIFO_NO_PRIORITY);
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            ixEthAccTxSchedulingDisciplineSet(IX_ETH_PORT_2, FIFO_NO_PRIORITY);
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        /*
         * If it is A0 Silicon, we set Tx Scheduling Discipline for both ports. 
         */ 
        ixEthAccTxSchedulingDisciplineSet(IX_ETH_PORT_1, FIFO_NO_PRIORITY);
        ixEthAccTxSchedulingDisciplineSet(IX_ETH_PORT_2, FIFO_NO_PRIORITY);
    }
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }


    ixEthAccCodeletPollEnabled = FALSE;
    if (ix_ossl_thread_create((ix_ossl_thread_entry_point_t)
			      ixEthAccCodeletStatsPollTask,
			      NULL,
			      &statstid) != IX_OSSL_ERROR_SUCCESS)
    {
	printf("Error spawning stats task\n");
	    return (IX_FAIL);
    }
    
    ixEthAccCodeletInitialised = TRUE;
    return (IX_SUCCESS);
}

/*
 * Function definition: ixEthAccCodeletUninit()
 *
 * Stop all threads and interrupts
 */

IX_STATUS ixEthAccCodeletUninit(void)
{  
    if(ixEthAccCodeletInitialised) 
    {
	ixEthAccCodeletThreadsStop();
	ixEthAccCodeletDispatcherStop(IX_ETH_CODELET_QMGR_DISPATCH_MODE);
	ixEthAccCodeletInitialised = FALSE;
    }
    return(IX_SUCCESS);
}

/*
 * Function definition: ixEthAccCodeletLinkUpCheck()
 *
 * Check the phys are ready
 */

IX_STATUS ixEthAccCodeletLinkUpCheck(unsigned int phyNo)
{
   IX_STATUS returnStatus = IX_SUCCESS;
   BOOL fullDuplex;
   BOOL linkUp;
   BOOL speed;
   BOOL autoneg;

   /* get the status */
   ixEthAccMiiLinkStatus(phyAddresses[phyNo], &linkUp, &speed, &fullDuplex, &autoneg);

   if (linkUp == FALSE)
   {
          unsigned int retry = 20; /* 20 retries */
          printf("Wait for PHY %u to be ready ...\n", phyNo);
          while (linkUp == FALSE && retry-- > 0)
          {
            ixOsServTaskSleep(100);  /* 100 milliseconds */

	    /* get the status again */
            ixEthAccMiiLinkStatus(phyAddresses[phyNo], 
				  &linkUp, 
				  &speed, 
				  &fullDuplex, 
				  &autoneg);
          }
          if (linkUp == FALSE) 
          {
            returnStatus = IX_FAIL;
          }
   }
   /* return fail if one of the links is not up */
   return returnStatus;
}

/*
 * Function definition: ixEthAccCodeletStart()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletStart(IxEthAccPortRxCallback port1RxCB,
			       IxEthAccPortRxCallback port2RxCB,
			       IxEthAccPortTxDoneCallback port1TxDoneCB,
			       IxEthAccPortTxDoneCallback port2TxDoneCB)
{
    unsigned int phyNo;

    IX_ETHACC_IS_CODELET_INITIALISED();

    /* Clear stats */
    memset(&IxEthAccCodeletStats[0], 0, sizeof(IxEthAccCodeletStats[0]));
    memset(&IxEthAccCodeletStats[1], 0, sizeof(IxEthAccCodeletStats[1]));

    /* check links are up and continue, even on link failure */
   for(phyNo=0; phyNo<maxPhyNo; phyNo++)
   {
       if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
            IX_FEATURE_CTRL_SILICON_TYPE_B0)
       {
           /*Only when Ethernet is available, then add dynaic entries */
           if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                 IX_FEATURE_CTRL_COMPONENT_ENABLED) && (0 == phyNo)) ||
               ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                 IX_FEATURE_CTRL_COMPONENT_ENABLED) && (1 == phyNo))) 
	   {
               (void)ixEthAccCodeletLinkUpCheck(phyNo);
           }
       }
       else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                 IX_FEATURE_CTRL_SILICON_TYPE_A0)
       {
           (void)ixEthAccCodeletLinkUpCheck(phyNo);
       } 
       else
       {
           printf("Error. Operation for other silicon stepping is undefined!.\n");
           return (IX_FAIL);        
       }   
   }

    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
      /*
       * If it is B0 Silicon, we need to check Eth Coprocessor's availability
       */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (port1RxCB != NULL)
            {
	         if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_1,
		    			           port1RxCB,
					           IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	         {
	             printf("Failed to register Rx callback for port 1\n");
	             return (IX_FAIL);
	         }
            }

            if(port1TxDoneCB != NULL)
            {
	         if (ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_1,
					                port1TxDoneCB,
		  			                IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	         {
	              printf("Failed to register Tx done callback for port 1\n");
	              return (IX_FAIL);
	         }
            } 
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if (port2RxCB != NULL)
            {
	         if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_2,
		  			           port2RxCB,
					           IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	         {
	              printf("Failed to register Rx callback for port 2\n");
	              return (IX_FAIL);
	         }
            }

            if(port2TxDoneCB != NULL)
            {
	         if(ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_2,
					               port2TxDoneCB,
					               IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	         {
	             printf("Failed to register Tx done callback for port 2\n");
	             return (IX_FAIL);
	         }
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        /*
         * If it is A0 Silicon, both Ethernet Coprocessors are available. 
         */ 
        if (port1RxCB != NULL)
        {
	     if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_1,
		   			       port1RxCB,
					       IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	     {
	         printf("Failed to register Rx callback for port 1\n");
	         return (IX_FAIL);
	     }
        }

        if (port2RxCB != NULL)
        {
	    if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_2,
					      port2RxCB,
					      IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	    {
	        printf("Failed to register Rx callback for port 2\n");
	        return (IX_FAIL);
	    }
        }

        if(port1TxDoneCB != NULL)
        {
	    if (ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_1,
					           port1TxDoneCB,
		  			           IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	    {
	         printf("Failed to register Tx done callback for port 1\n");
	         return (IX_FAIL);
	    }
        }

        if(port2TxDoneCB != NULL)
        {
	     if(ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_2,
					           port2TxDoneCB,
					           IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	     {
	         printf("Failed to register Tx done callback for port 2\n");
	         return (IX_FAIL);
	     }
        }

    }
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    if (ixEthAccCodeletNpeInitialised == FALSE)
    {
        printf ("Initialising and Starting NPEs...\n");
	if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
	    IX_FEATURE_CTRL_SILICON_TYPE_B0)
	{
	   /*
            * If it is B0 Silicon, we download NPE image only when Eth Coprocessor is 
            * available.
            */
            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEB_IMAGEID))
                {
	            printf ("Error initialising and starting NPE B!\n");
	            return (IX_FAIL);
                }
            }

            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEC_IMAGEID))
                {
	            printf ("Error initialising and starting NPE C!\n");
	            return (IX_FAIL);
                }
            }
       }
       else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                IX_FEATURE_CTRL_SILICON_TYPE_A0) 
       {
            /* 
             * If it is A0 silicon, all Ethernet Coprocessors are available. 
             * So, we no need to check availability of Eth Coprocessor
             */
            if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEB_IMAGEID))
            {
	        printf ("Error initialising and starting NPE B!\n");
   	        return (IX_FAIL);
            }

            if (IX_SUCCESS != ixNpeDlNpeInitAndStart(ETH_NPEC_IMAGEID))
            {
	        printf ("Error initialising and starting NPE C!\n");
     	        return (IX_FAIL);
            }
       }   
       else
       {
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
       }
	
	ixEthAccCodeletNpeInitialised = TRUE;
    }
 
	if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
	    IX_FEATURE_CTRL_SILICON_TYPE_B0)
	{
            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if(ixEthAccCodeletReplenishBuffers(IX_ETH_PORT_1, 
				                   IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE) 
                   != IX_SUCCESS)
                {
	             printf("Error replenish port 1\n");
	             return (IX_FAIL);
                }

#ifdef IX_ETHACC_CODELET_RX_FCS_STRIP
                if(ixEthAccPortRxFrameAppendFCSDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
                {
	             printf("Error disabling FCS for port 1\n");
	             return (IX_FAIL);
                }
#else
                if(ixEthAccPortRxFrameAppendFCSEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
                {
	             printf("Error enabling FCS for port 1\n");
	             return (IX_FAIL);
                }
#endif

#ifdef IX_ETHACC_CODELET_TX_FCS_APPEND
                if(ixEthAccPortTxFrameAppendFCSEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
                {
	             printf("Error enabling FCS for port 1\n");
	             return (IX_FAIL);
                } 
#else
                if(ixEthAccPortTxFrameAppendFCSDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
                {
	             printf("Error disabling FCS for port 1\n");
 	             return (IX_FAIL);
                }
#endif

		if (ixEthDBFilteringPortMaximumFrameSizeSet(IX_ETH_PORT_1,
							    IX_ETHACC_CODELET_FRAME_SIZE) 
		    != IX_SUCCESS)
		{
	             printf("Error set the frame size port 1\n");
	             return (IX_FAIL);
                }

                /* Enable ports */
                if(ixEthAccPortEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
                {
	            printf("Error enabling port 1\n");
	            return (IX_FAIL);
                }

            } /* End of if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
               *             IX_FEATURE_CTRL_COMPONENT_ENABLED)  
               */

            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if(ixEthAccCodeletReplenishBuffers(IX_ETH_PORT_2,
				                   IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE)
                   != IX_SUCCESS)
                {
	             printf("Error replenish port 2\n");
	             return (IX_FAIL);
                }
	    
#ifdef IX_ETHACC_CODELET_RX_FCS_STRIP
                if(ixEthAccPortRxFrameAppendFCSDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
                {
   	             printf("Error disabling FCS for port 2\n");
	             return (IX_FAIL);
                }
#else
                if(ixEthAccPortRxFrameAppendFCSEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
                {
	             printf("Error enabling FCS for port 2\n");
	             return (IX_FAIL);
                }
#endif

#ifdef IX_ETHACC_CODELET_TX_FCS_APPEND
                if(ixEthAccPortTxFrameAppendFCSEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
                {
        	     printf("Error enabling FCS for port 2\n");
	             return (IX_FAIL);
                } 
#else
                if(ixEthAccPortTxFrameAppendFCSDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
                {
          	     printf("Error disabling FCS for port 2\n");
	             return (IX_FAIL);
                }
#endif

		if (ixEthDBFilteringPortMaximumFrameSizeSet(IX_ETH_PORT_2, 
							    IX_ETHACC_CODELET_FRAME_SIZE) 
		    != IX_SUCCESS)
		{
		    printf("Error set the frame size port 2\n");
		    return (IX_FAIL);
		}

                if(ixEthAccPortEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
                {
	            printf("Error enabling port 2\n");
	            return (IX_FAIL);
                }

            } /* End of if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
               *             IX_FEATURE_CTRL_COMPONENT_ENABLED)  
               */

       }
       else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                 IX_FEATURE_CTRL_SILICON_TYPE_A0) 
       {
            /* 
             * If it is A0 silicon, all Ethernet Coprocessors are available. 
             * So, we no need to check availability of Eth Coprocessor
             */
             if(ixEthAccCodeletReplenishBuffers(IX_ETH_PORT_1, 
				                IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE) 
                != IX_SUCCESS)
             {
	         printf("Error replenish port 1\n");
	         return (IX_FAIL);
             }

             if(ixEthAccCodeletReplenishBuffers(IX_ETH_PORT_2,
				                IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE)
                != IX_SUCCESS)
             {
	         printf("Error replenish port 2\n");
	         return (IX_FAIL);
             }

#ifdef IX_ETHACC_CODELET_RX_FCS_STRIP
             if(ixEthAccPortRxFrameAppendFCSDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
             {
	         printf("Error disabling FCS for port 1\n");
	         return (IX_FAIL);
             }

             if(ixEthAccPortRxFrameAppendFCSDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
             {
   	         printf("Error disabling FCS for port 2\n");
	         return (IX_FAIL);
             }
#else
             if(ixEthAccPortRxFrameAppendFCSEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
             {
	         printf("Error enabling FCS for port 1\n");
	         return (IX_FAIL);
             }

             if(ixEthAccPortRxFrameAppendFCSEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
             {
	         printf("Error enabling FCS for port 2\n");
	         return (IX_FAIL);
             }
#endif

#ifdef IX_ETHACC_CODELET_TX_FCS_APPEND
             if(ixEthAccPortTxFrameAppendFCSEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
             {
	         printf("Error enabling FCS for port 1\n");
	         return (IX_FAIL);
             }

             if(ixEthAccPortTxFrameAppendFCSEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
             {
        	 printf("Error enabling FCS for port 2\n");
	         return (IX_FAIL);
             } 
#else
             if(ixEthAccPortTxFrameAppendFCSDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
             {
	         printf("Error disabling FCS for port 1\n");
 	         return (IX_FAIL);
             }

             if(ixEthAccPortTxFrameAppendFCSDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
             {
          	 printf("Error disabling FCS for port 2\n");
	         return (IX_FAIL);
             }
#endif

	    if (ixEthDBFilteringPortMaximumFrameSizeSet(IX_ETH_PORT_1, 
							 IX_ETHACC_CODELET_FRAME_SIZE) 
		 != IX_SUCCESS)
	     {
		 printf("Error set the frame size port 1\n");
		 return (IX_FAIL);
	     }

	     if (ixEthDBFilteringPortMaximumFrameSizeSet(IX_ETH_PORT_2, 
							 IX_ETHACC_CODELET_FRAME_SIZE) 
		 != IX_SUCCESS)
	     {
		 printf("Error set the frame size port 2\n");
		 return (IX_FAIL);
	     }

             /* Enable ports */
            if(ixEthAccPortEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error enabling port 1\n");
	        return (IX_FAIL);
            }

            if(ixEthAccPortEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error enabling port 2\n");
	        return (IX_FAIL);
            }

       }   
       else
       {
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
       }

    return IX_SUCCESS;
}

PRIVATE IX_STATUS ixEthAccCodeletStop(IxEthAccPortRxCallback port1RxCB,
				      IxEthAccPortRxCallback port2RxCB,
				      IxEthAccPortTxDoneCallback port1TxDoneCB,
				      IxEthAccPortTxDoneCallback port2TxDoneCB)
{

	if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
	    IX_FEATURE_CTRL_SILICON_TYPE_B0)
	{
	   /*
            * If it is B0 Silicon, we download NPE image only when Eth Coprocessor is 
            * available.
            */
            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if (port1RxCB != NULL)
                {
	             if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_1,
					               port1RxCB,
					               IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	             {
	                 printf("Failed to register Rx callback for port 1\n");
	                 return (IX_FAIL);
	             }
                }

                if(port1TxDoneCB != NULL)
                {
      	             if (ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_1,
					                    port1TxDoneCB,
					                    IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	             {
	                printf("Failed to register Tx done callback for port 1\n");
	                return (IX_FAIL);
	             }
                }
            }

            if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                IX_FEATURE_CTRL_COMPONENT_ENABLED)
            {
                if (port2RxCB != NULL)
                {
	             if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_2,
					               port2RxCB,
					               IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	             {
	                 printf("Failed to register Rx callback for port 2\n");
	                 return (IX_FAIL);
	             }
                } 

                if(port2TxDoneCB != NULL)
                {
	             if(ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_2,
					                   port2TxDoneCB,
					                   IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	            {
	                 printf("Failed to register Tx done callback for port 2\n");
	                 return (IX_FAIL);
	            }
                }
            }

            if (ixEthAccCodeletNpeInitialised == TRUE)
            {
                if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                    IX_FEATURE_CTRL_COMPONENT_ENABLED)
                {
	            /* Disable ports - unused MBufs will be returned */
	             if(ixEthAccPortDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	             { 
	                 printf("Error disabling port 1\n");
	                 return (IX_FAIL);
	             }
	        }

                if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                    IX_FEATURE_CTRL_COMPONENT_ENABLED)
                {
 	             if(ixEthAccPortDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	             {
	                 printf("Error disabling port 2\n");
	                 return (IX_FAIL);
	             }
                }
    
	        ixEthAccCodeletNpeInitialised = FALSE;
            }

       }
       else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                IX_FEATURE_CTRL_SILICON_TYPE_A0) 
       {
            /* 
             * If it is A0 silicon, all Ethernet Coprocessors are available. 
             * So, we no need to check availability of Eth Coprocessor
             */
            if (port1RxCB != NULL)
            {
	        if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_1,
					          port1RxCB,
					          IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Failed to register Rx callback for port 1\n");
	            return (IX_FAIL);
	        }
            }

            if (port2RxCB != NULL)
            {
	        if(ixEthAccPortRxCallbackRegister(IX_ETH_PORT_2,
					          port2RxCB,
					          IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Failed to register Rx callback for port 2\n");
	            return (IX_FAIL);
	        }
            }

            if(port1TxDoneCB != NULL)
            {
      	        if (ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_1,
					               port1TxDoneCB,
					               IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Failed to register Tx done callback for port 1\n");
	            return (IX_FAIL);
	        }
            }

            if(port2TxDoneCB != NULL)
            {
	         if(ixEthAccPortTxDoneCallbackRegister(IX_ETH_PORT_2,
					               port2TxDoneCB,
					               IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	        {
	             printf("Failed to register Tx done callback for port 2\n");
	             return (IX_FAIL);
	        }
            }

            if (ixEthAccCodeletNpeInitialised == TRUE)
            {
	         /* Disable ports - unused MBufs will be returned */
	         if(ixEthAccPortDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
	         { 
	             printf("Error disabling port 2\n");
	             return (IX_FAIL);
	         }
	
	         if(ixEthAccPortDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
	         {
	             printf("Error disabling port 1\n");
	             return (IX_FAIL);
	         }
    
	         ixEthAccCodeletNpeInitialised = FALSE;
            }
       }   
       else
       {
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
       }

    return IX_SUCCESS;
}

PRIVATE IX_STATUS
ixEthAccCodeletThreadsStop(void)
{
    if (!ixEthAccCodeletStatsPollTaskStop)
    {
	ixEthAccCodeletStatsPollTaskStop = TRUE;
	if (ixOsServMutexLock (&ixEthAccCodeletStatsPollTaskRunning) != IX_SUCCESS)
	{
	    printf("Error stopping Statistics Polling thread!\n");
	    return (IX_FAIL);
	}
	ixOsServMutexUnlock (&ixEthAccCodeletStatsPollTaskRunning);
    }

    if (!ixEthAccCodeletDBMaintenanceTaskStop)
    {
	ixEthAccCodeletDBMaintenanceTaskStop = TRUE;
	if (ixOsServMutexLock (&ixEthAccCodeletDBMaintenanceTaskRunning) != IX_SUCCESS)
	{
	    printf("Error stopping Database Maintenance thread!\n");
	    return (IX_FAIL);
	}
	ixOsServMutexUnlock (&ixEthAccCodeletDBMaintenanceTaskRunning);
    }

    if (!ixEthAccCodeletDispatcherPollStop)
    {
	ixEthAccCodeletDispatcherPollStop = TRUE;
	if (ixOsServMutexLock (&ixEthAccCodeletDispatcherPollRunning) != IX_SUCCESS)
	{
	    printf("Error stopping QMgr Dispatcher thread!\n");
	    return (IX_FAIL);
	}
	ixOsServMutexUnlock (&ixEthAccCodeletDispatcherPollRunning);
    }

    return (IX_SUCCESS);
}

/*
 * Function definition: ixEthAccCodeletMbufSizeSet()
 *
 * Reset the size of each mbuf of the chain to
 * the allocated size
 */

void ixEthAccCodeletMbufChainSizeSet(IX_MBUF *mBufPtr)
{
    while (mBufPtr)
    {
	/* reset the mBuf length to the allocated value */
	IX_MBUF_MLEN(mBufPtr) = IX_ETHACC_CODELET_PCK_SIZE;
	/* move to the next element of the chain */
	mBufPtr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr); 
    }
}

/*
 * Function definition: ixEthAccCodeletRxSink()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletRxSink(void)
{
    IX_ETHACC_IS_CODELET_INITIALISED();

    if (ixEthAccCodeletStart(ixEthAccCodeletRxSinkCB, 
			     ixEthAccCodeletRxSinkCB,
			     ixEthAccCodeletMemPoolFreeTxCB,
                             ixEthAccCodeletMemPoolFreeTxCB
			     ) != IX_ETH_ACC_SUCCESS)
    {
	printf("Failed to start the RX Sink Operation\n");
	return IX_FAIL;
    }

    printf("Rx Sink Operation started. Begin sending packets in from external source\n");

    if(ixEthAccCodeletLoop() != IX_SUCCESS)
    {
	return (IX_FAIL);
    }
    
    /* Recover buffers */
    if(ixEthAccCodeletRecoverBuffers() != IX_SUCCESS)
    {
	printf("Error recovering buffers for Rx Sink Operation\n");
	return (IX_FAIL);
    }

    return (IX_SUCCESS);
}


/*
 * Function definition: ixEthAccCodeletSwLoopback()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletSwLoopback(void)
{
    IX_ETHACC_IS_CODELET_INITIALISED();

    if ( ixEthAccCodeletStart(ixEthAccCodeletSwLoopbackRxCB, 
			      ixEthAccCodeletSwLoopbackRxCB, 
			      ixEthAccCodeletSwLoopbackTxCB,
			      ixEthAccCodeletSwLoopbackTxCB) != IX_SUCCESS)
    {
	printf("Failed to start the Sw Loopback Operation\n");
	return (IX_FAIL);
    }


    printf("Software loopback successfully started. Begin sending packets from external source\n");  

    if(ixEthAccCodeletLoop() != IX_SUCCESS)
    {
	return (IX_FAIL);
    }

    /* Recover buffers */
    if(ixEthAccCodeletRecoverBuffers() != IX_SUCCESS)
    {
	printf("Error recovering buffers for Sw Loopback Operation\n");
	return (IX_FAIL);
    }    

    return (IX_SUCCESS);
}


/*
 * Function definition: ixEthAccCodeletTxGenRxSinkLoopback()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletTxGenRxSinkLoopback(void)
{
    IX_MBUF *mBufPtr;
    UINT32 numBufs;

    IX_ETHACC_IS_CODELET_INITIALISED();
    
#if defined(__linux) && (IX_ETH_CODELET_QMGR_DISPATCH_MODE==TRUE)
    printf("\n*********************************************************************\n");
    printf("WARNING: This operation uses too much CPU time in interrupt mode to display\n");
    printf("         any counters. Please recompile the codelet for polled mode when\n");
    printf("         using this operation\n\n");
    printf("Polled mode: set IX_ETH_CODELET_QMGR_DISPATCH_MODE to FALSE in IxEthAccCodelet.c\n");
    printf("*********************************************************************\n\n");
#endif

    if (ixEthAccCodeletStart(ixEthAccCodeletMemPoolFreeRxCB, 
			     ixEthAccCodeletTxGenRxSinkLoopbackRxCB,
			     ixEthAccCodeletTxGenRxSinkLoopbackTxCB,
                             ixEthAccCodeletMemPoolFreeTxCB
			     ) != IX_ETH_ACC_SUCCESS)
    {
	printf("Failed to start the Tx Gen RX Sink Operation\n");
	return IX_FAIL;
    }

    /* Generate our random data for the payload */
    IX_ETHACC_CODELET_DATAGEN(compData);

    /* Now start the loopback by transmitting the first few frames */
    for (numBufs=0; numBufs<IX_ETHACC_CODELET_TXGEN_PCKS; numBufs++)
    {
	IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	
	if (mBufPtr == NULL)
	{
	    printf("Buffer queue empty. Not enough free buffers to transmit in TxGen-RxSink Loopback!\n");
	    return (IX_FAIL);
	}

	IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_TXGEN_PCK_LEN;
	memcpy(IX_MBUF_MDATA(mBufPtr), &compData[0], IX_ETHACC_CODELET_TXGEN_PCK_LEN);
	IX_ACC_DATA_CACHE_FLUSH(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));

	if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_1, mBufPtr,
				     IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	{
	    printf("Error starting TxGen-RxSink Loopback\n");
	    return (IX_FAIL);
	}
    }

    printf("Port %d Tx pool has %d buffers\n", IX_ETH_PORT_1 + 1, numBufs);
    printf("TxGen-RxSink Operation successfully started\n");

    if(ixEthAccCodeletLoop() != IX_SUCCESS)
    {
	return (IX_FAIL);
    }
    
    /* Recover buffers */
    if(ixEthAccCodeletRecoverBuffers() != IX_SUCCESS)
    {
	printf("Error recovering buffers for TxGen-RxSink Loopback Operation\n");
	return (IX_FAIL);
    } 

    return (IX_SUCCESS);
}

/*
 * Function definition: ixEthAccCodeletPhyLoopback()
 *
 * See header file for documentation.
 */
IX_STATUS
ixEthAccCodeletPhyLoopback(void)
{
    IX_MBUF *mBufPtr;
    UINT32 numBufs;
    UINT32 numPktTx ; 

    IX_ETHACC_IS_CODELET_INITIALISED();
    
#if defined(__linux) && (IX_ETH_CODELET_QMGR_DISPATCH_MODE==TRUE)
    printf("\n*********************************************************************\n");
    printf("WARNING: This operation uses too much CPU time in interrupt mode to display\n");
    printf("         any counters. Please recompile the codelet for polled mode when\n");
    printf("         using this operation.\n\n");
    printf("Polled mode: set IX_ETH_CODELET_QMGR_DISPATCH_MODE to FALSE in IxEthAccCodelet.c\n");
    printf("*********************************************************************\n\n");
#endif

    if (ixEthAccCodeletStart(ixEthAccCodeletPhyLoopbackRxCB,
			     ixEthAccCodeletPhyLoopbackRxCB,
			     ixEthAccCodeletPhyLoopbackTxCB,
                             ixEthAccCodeletPhyLoopbackTxCB
                            ) != IX_ETH_ACC_SUCCESS)
    {
	printf("Failed to start the PHY Loopback operation\n");
	return IX_FAIL;
    }

    /* Generate our random data for the payload */
    IX_ETHACC_CODELET_DATAGEN(compData);

#if IX_ETHACC_CODELET_TXGEN_PCKS <=64
    numPktTx = IX_ETHACC_CODELET_TXGEN_PCKS ;
#else
    numPktTx = 64 ;
#endif 

    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
	 IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
        /*
         * If it is B0 Silicon, we download NPE image only when Eth Coprocessor is 
         * available.
         */
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
            IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {

            /* Now start the loopback by transmitting the first few frames to Port 1 */
            for (numBufs=0; numBufs<numPktTx; numBufs++)
            {
	        IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	
                if (mBufPtr == NULL)
	        {
	            printf("Buffer queue empty. Not enough free buffers to transmit in PHY Loopback!\n");
        	    return (IX_FAIL);
	        }

	        IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_TXGEN_PCK_LEN;
	        memcpy(IX_MBUF_MDATA(mBufPtr), &compData[0], IX_ETHACC_CODELET_TXGEN_PCK_LEN);
	        IX_ACC_DATA_CACHE_FLUSH(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));

     	        if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_1, mBufPtr,
		  		             IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Error starting PHY Loopback\n");
	            return (IX_FAIL);
	        }
            }

            printf("Port %d Tx pool has %d buffers\n", IX_ETH_PORT_1 + 1, numBufs);

	} /* End of if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           *            IX_FEATURE_CTRL_COMPONENT_ENABLED) 
           */

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
            IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {

            /* Now start the loopback by transmitting the first few frames to Port 2 */
            for (numBufs=0; numBufs<numPktTx; numBufs++)
            {
	        IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	
	        if (mBufPtr == NULL)
	        {
	            printf("Buffer queue empty. Not enough free buffers to transmit in PHY Loopback!\n");
	            return (IX_FAIL);
	        }

	        IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_TXGEN_PCK_LEN;
	        memcpy(IX_MBUF_MDATA(mBufPtr), &compData[0], IX_ETHACC_CODELET_TXGEN_PCK_LEN);
	        IX_ACC_DATA_CACHE_FLUSH(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));

	        if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_2, mBufPtr,
		  		             IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Error starting PHY Loopback\n");
	            return (IX_FAIL);
	        }
            }

            printf("Port %d Tx pool has %d buffers\n", IX_ETH_PORT_2 + 1, numBufs);

	}/* End of if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
          *            IX_FEATURE_CTRL_COMPONENT_ENABLED) 
          */
    }/* End of if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
      *  	    IX_FEATURE_CTRL_SILICON_TYPE_B0)  
      */
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
	 IX_FEATURE_CTRL_SILICON_TYPE_A0)
    {

        /* Now start the loopback by transmitting the first few frames to Port 1 */
        for (numBufs=0; numBufs<IX_ETHACC_CODELET_TXGEN_PCKS; numBufs++)
        {
	    IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	
            if (mBufPtr == NULL)
	    {
	        printf("Buffer queue empty. Not enough free buffers to transmit in PHY Loopback!\n");
                return (IX_FAIL);
	    }

            IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_TXGEN_PCK_LEN;
            memcpy(IX_MBUF_MDATA(mBufPtr), &compData[0], IX_ETHACC_CODELET_TXGEN_PCK_LEN);
            IX_ACC_DATA_CACHE_FLUSH(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));

 	    if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_1, mBufPtr,
		  		         IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	    {
	        printf("Error starting PHY Loopback\n");
                return (IX_FAIL);
            }
        }

        /* Now start the loopback by transmitting the first few frames to Port 2 */
        for (numBufs=0; numBufs<IX_ETHACC_CODELET_TXGEN_PCKS; numBufs++)
        {
            IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	
            if (mBufPtr == NULL)
            {
                printf("Buffer queue empty. Not enough free buffers to transmit in PHY Loopback!\n");
                return (IX_FAIL);
            }

            IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_TXGEN_PCK_LEN;
            memcpy(IX_MBUF_MDATA(mBufPtr), &compData[0], IX_ETHACC_CODELET_TXGEN_PCK_LEN);
            IX_ACC_DATA_CACHE_FLUSH(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));

            if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_2, mBufPtr,
      	  		                 IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	    {
	        printf("Error starting PHY Loopback\n");
                return (IX_FAIL);
            }
        }

        printf("Port %d Tx pool has %d buffers\n", IX_ETH_PORT_1 + 1, numBufs);
        printf("Port %d Tx pool has %d buffers\n", IX_ETH_PORT_2 + 1, numBufs);

    }
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    } 
     
    printf("PHY Loopback operation successfully started\n");

    if(ixEthAccCodeletLoop() != IX_SUCCESS)
    {
	return (IX_FAIL);
    }
    
    /* Recover buffers */
    if(ixEthAccCodeletRecoverBuffers() != IX_SUCCESS)
    {
	printf("Error recovering buffers for PHY Loopback operation\n");
	return (IX_FAIL);
    } 

    return (IX_SUCCESS);
}

/*
 * Function definition: ixEthAccCodeletBridge()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletBridge(void)
{
    IX_ETHACC_IS_CODELET_INITIALISED();

    if ( ixEthAccCodeletStart(ixEthAccCodeletBridgeRxCB, 
			      ixEthAccCodeletBridgeRxCB, 
			      ixEthAccCodeletBridgeTxCB,
			      ixEthAccCodeletBridgeTxCB) != IX_SUCCESS)
    {
	printf("Failed to start the Bridge Operation\n");
	return (IX_FAIL);
    }

    printf("Bridge Operation started. Begin sending packets from external source\n");

    if(ixEthAccCodeletLoop() != IX_SUCCESS)
    {
	return (IX_FAIL);
    }

    /* Recover buffers */
    if(ixEthAccCodeletRecoverBuffers() != IX_SUCCESS)
    {
	printf("Error recovering buffers for Bridge Operation\n");
	return (IX_FAIL);
    } 

    return (IX_SUCCESS);
}


/*
 * Function definition: ixEthAccCodeletDBLearning()
 *
 * See header file for documentation.
 */

IX_STATUS ixEthAccCodeletDBLearning(void)
{
    IxEthAccPortId portNo, portPtr;
    ix_ossl_thread_t dbMaintenanceid;

    IxEthDBMacAddr staticMacAddress[IX_ETHACC_CODELET_MAX_PORT] = 
	{{{0,1,2,3,4,5}}, {{6,7,8,9,0,1}}};

    IxEthDBMacAddr dynamMacAddress[IX_ETHACC_CODELET_MAX_PORT] = 
	{{{0xa,0xb,0xc,0xd,0xe,0xf}}, {{0x0,0xb,0x3,0xc,0x4,0xd}}};

    IX_ETHACC_IS_CODELET_INITIALISED();

    if (ix_ossl_thread_create((ix_ossl_thread_entry_point_t) 
			      ixEthAccCodeletDBMaintenanceTask,
			      NULL,
			      &dbMaintenanceid) != IX_OSSL_ERROR_SUCCESS)
    {
	printf("Error spawning DB maintenance task\n");
	return (IX_FAIL);
    }

    ix_ossl_thread_set_priority(dbMaintenanceid, IX_ETHACC_CODELET_DB_PRIORITY);

    /* Enable ports */
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthDBPortEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error enabling port 1\n");
	        return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthDBPortEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error enabling port 2\n");
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {

        if(ixEthDBPortEnable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error enabling port 1\n");
	    return (IX_FAIL);
        }

        if(ixEthDBPortEnable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error enabling port 2\n");
	    return (IX_FAIL);
        }
    }
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    /* Add some static entries */
    printf("Adding static entries...\n");

    for (portNo=IX_ETH_PORT_1; portNo<IX_ETHACC_CODELET_MAX_PORT; portNo++)
    {

        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
	    /*Only when Ethernet is available, then add dynaic entries */
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_1 == portNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_2 == portNo))) 
	    {
	        if (ixEthDBFilteringStaticEntryProvision(portNo,
		 				         &staticMacAddress[portNo])
	            != IX_ETH_DB_SUCCESS)
	        {
	            printf("Failed to add static MAC to port %d\n", portNo+1);
	            return (IX_FAIL);
	        }

    	        /* Verify that the address has successfully been added */
	        if (ixEthDBFilteringDatabaseSearch(&portPtr, 
					           &staticMacAddress[portNo]) 
	            == IX_ETH_DB_NO_SUCH_ADDR)
  	        {
	            printf("Static MAC address not found in Database\n");
	            return (IX_FAIL);
	        }
            } 
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                  IX_FEATURE_CTRL_SILICON_TYPE_A0) 
        {
	    if (ixEthDBFilteringStaticEntryProvision(portNo,
						     &staticMacAddress[portNo])
	        != IX_ETH_DB_SUCCESS)
	    {
	        printf("Failed to add static MAC to port %d\n", portNo+1);
	        return (IX_FAIL);
	    }

  	    /* Verify that the address has successfully been added */
	    if (ixEthDBFilteringDatabaseSearch(&portPtr, 
					       &staticMacAddress[portNo]) 
	        == IX_ETH_DB_NO_SUCH_ADDR)
  	    {
	        printf("Static MAC address not found in Database\n");
	        return (IX_FAIL);
	    }
        }
        else
        {
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
        }
    }

    /* Add some dynamic entries */
    printf("Adding dynamic entries...\n");

    for (portNo=IX_ETH_PORT_1; portNo<IX_ETHACC_CODELET_MAX_PORT; portNo++)
    {
        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_1 == portNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_2 == portNo))) 
	    {
  	        if (ixEthDBFilteringDynamicEntryProvision(portNo,
						          &dynamMacAddress[portNo])
	            != IX_ETH_DB_SUCCESS)
	        {
	             printf("Failed to add dynamic MAC to port %d\n", portNo+1);
	        }

	        /* Verify that the address has successfully been added */
	        if (ixEthDBFilteringDatabaseSearch(&portPtr, 
					           &dynamMacAddress[portNo]) 
	            == IX_ETH_DB_NO_SUCH_ADDR)
	        {
	            printf("Dynamic MAC address not found in Database\n");
	            return (IX_FAIL);
	        }
            }
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_A0)  
	{
  	    if (ixEthDBFilteringDynamicEntryProvision(portNo,
						      &dynamMacAddress[portNo])
	        != IX_ETH_DB_SUCCESS)
	    {
	        printf("Failed to add dynamic MAC to port %d\n", portNo+1);
	    }

	    /* Verify that the address has successfully been added */
	    if (ixEthDBFilteringDatabaseSearch(&portPtr, 
					       &dynamMacAddress[portNo]) 
	        == IX_ETH_DB_NO_SUCH_ADDR)
	    {
	        printf("Dynamic MAC address not found in Database\n");
	        return (IX_FAIL);
	    }
        }
        else
	{
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
        }    
    }

    ixEthDBFilteringDatabaseShowAll();

    /* Now age the dynamic entries */
    for (portNo=IX_ETH_PORT_1; portNo<IX_ETHACC_CODELET_MAX_PORT; portNo++)
    {
        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_1 == portNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_2 == portNo))) 
	    {
	        printf("Enabling aging for port %d...\n", portNo+1);

   	        if (ixEthDBPortAgingEnable(portNo) != IX_ETH_DB_SUCCESS)
	        {
	            printf("Failed to enable aging for port %d\n", portNo+1);
	            return (IX_FAIL);
	        }
            }
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_A0)  
	{
  	    printf("Enabling aging for port %d...\n", portNo+1);

  	    if (ixEthDBPortAgingEnable(portNo) != IX_ETH_DB_SUCCESS)
	    {
	        printf("Failed to enable aging for port %d\n", portNo+1);
	        return (IX_FAIL);
	    }
        }
        else
	{
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
        }    
    }

    /* Wait 3 minutes over aging time (for safety) and verify that the dynamic entries have been removed */
    printf("Aging entries.\nWaiting for %d minutes...", IX_ETH_DB_LEARNING_ENTRY_AGE_TIME / 60 + 3);
    
    ix_ossl_sleep((IX_ETH_DB_LEARNING_ENTRY_AGE_TIME + 180) * 1000);

    ixEthDBFilteringDatabaseShowAll();

    /* Remove static entries so they can be used for another operation */
    for (portNo=IX_ETH_PORT_1; portNo<IX_ETHACC_CODELET_MAX_PORT; portNo++)
    {
        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_1 == portNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_2 == portNo))) 
	    {
	        if(ixEthDBFilteringEntryDelete(&staticMacAddress[portNo])
	           != IX_ETH_DB_SUCCESS)
	        {
	            printf("Failed to remove static MAC on port %d\n", portNo+1);
	        }
            }
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_A0)  
	{

	     if(ixEthDBFilteringEntryDelete(&staticMacAddress[portNo])
	        != IX_ETH_DB_SUCCESS)
	     {
	         printf("Failed to remove static MAC on port %d\n", portNo+1);
	     }
        }
        else
	{
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
        }    
    }
    
    /* Disable ports */
    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthDBPortDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error disabling port 1\n");
	        return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthDBPortDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
            {
	        printf("Error disabling port 2\n");
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        if(ixEthDBPortDisable(IX_ETH_PORT_1) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error disabling port 1\n");
	    return (IX_FAIL);
        }

        if(ixEthDBPortDisable(IX_ETH_PORT_2) != IX_ETH_ACC_SUCCESS)
        {
	    printf("Error disabling port 2\n");
	    return (IX_FAIL);
        }
    }   
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    /* Stop the codelet threads cleanly */
    if(ixEthAccCodeletThreadsStop() != IX_SUCCESS)
    {
        printf("Error stopping codelet threads!\n");
        return (IX_FAIL);
    }	

    return (IX_SUCCESS);
}


/*
 * Function definition: ixEthAccCodeletShow()
 *
 * See header file for documentation.
 */

void ixEthAccCodeletShow(void)
{
    static IxEthEthObjStats *portStats = NULL;
    UINT32 portNo;

    if(portStats == NULL)
    {
	/*
	 * The statistics are gathered by the NPE therefore we use DMA_MALLOC
	 * to make it cache safe for us to read.
	 */
	if((portStats = IX_ACC_DRV_DMA_MALLOC(sizeof(IxEthEthObjStats))) == NULL)
	{
	    printf("Unable to create port stats!\n");
	    return;
	}
    }

    for(portNo=0; portNo<IX_ETHACC_CODELET_MAX_PORT; portNo++)
    {
        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_1 == portNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (IX_ETH_PORT_2 == portNo))) 
	    {
	        memset(portStats, 0, sizeof(IxEthEthObjStats));

     	        printf("\nStatistics for port %d:\n", portNo);
	        if(ixEthAccMibIIStatsGetClear(portNo, portStats) != IX_ETH_ACC_SUCCESS)
	        {
	            printf("Unable to retrieve statistics for port %d!\n", portNo);
	        }
	        else
	        {
	            printf("dot3StatsAlignmentErrors: \t\t%u\n",
		           portStats->dot3StatsAlignmentErrors);
  	            printf("dot3StatsFCSErrors: \t\t\t%u\n",
		           portStats->dot3StatsFCSErrors);
   	            printf("dot3StatsFrameTooLongs: \t\t%u\n",
		           portStats->dot3StatsFrameTooLongs);
 	            printf("dot3StatsInternalMacReceiveErrors: \t%u\n",
		           portStats->dot3StatsInternalMacReceiveErrors);
	            printf("dot3StatsSingleCollisionFrames: \t%u\n",
		           portStats->dot3StatsSingleCollisionFrames);
	            printf("dot3StatsMultipleCollisionFrames: \t%u\n",
		           portStats->dot3StatsMultipleCollisionFrames);
	            printf("dot3StatsDeferredTransmissions: \t%u\n",
		           portStats->dot3StatsDeferredTransmissions);
	            printf("dot3StatsLateCollisions: \t\t%u\n",
		           portStats->dot3StatsLateCollisions);
	            printf("dot3StatsExcessiveCollisions: \t\t%u\n",
		           portStats->dot3StatsExcessiveCollsions);
	            printf("dot3StatsInternalMacTransmitErrors: \t%u\n",
		           portStats->dot3StatsInternalMacTransmitErrors);
	            printf("dot3StatsCarrierSenseErrors: \t\t%u\n",
		           portStats->dot3StatsCarrierSenseErrors);
	            printf("LearnedEntryDiscards: \t\t\t%u\n",
		           portStats->LearnedEntryDiscards);
   	            printf("\n");
 	        }
            }
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_A0)  
	{
	    memset(portStats, 0, sizeof(IxEthEthObjStats));

   	    printf("\nStatistics for port %d:\n", portNo);
	    if(ixEthAccMibIIStatsGetClear(portNo, portStats) != IX_ETH_ACC_SUCCESS)
	    {
	        printf("Unable to retrieve statistics for port %d!\n", portNo);
	    }
	    else
	    {
	        printf("dot3StatsAlignmentErrors: \t\t%u\n",
		       portStats->dot3StatsAlignmentErrors);
	        printf("dot3StatsFCSErrors: \t\t\t%u\n",
		       portStats->dot3StatsFCSErrors);
   	        printf("dot3StatsFrameTooLongs: \t\t%u\n",
		       portStats->dot3StatsFrameTooLongs);
 	        printf("dot3StatsInternalMacReceiveErrors: \t%u\n",
		       portStats->dot3StatsInternalMacReceiveErrors);
	        printf("dot3StatsSingleCollisionFrames: \t%u\n",
		       portStats->dot3StatsSingleCollisionFrames);
	        printf("dot3StatsMultipleCollisionFrames: \t%u\n",
		       portStats->dot3StatsMultipleCollisionFrames);
	        printf("dot3StatsDeferredTransmissions: \t%u\n",
		       portStats->dot3StatsDeferredTransmissions);
	        printf("dot3StatsLateCollisions: \t\t%u\n",
		       portStats->dot3StatsLateCollisions);
	        printf("dot3StatsExcessiveCollisions: \t\t%u\n",
		       portStats->dot3StatsExcessiveCollsions);
	        printf("dot3StatsInternalMacTransmitErrors: \t%u\n",
		       portStats->dot3StatsInternalMacTransmitErrors);
	        printf("dot3StatsCarrierSenseErrors: \t\t%u\n",
		       portStats->dot3StatsCarrierSenseErrors);
	        printf("LearnedEntryDiscards: \t\t\t%u\n",
		       portStats->LearnedEntryDiscards);
   	        printf("\n");
 	    }
        }
        else
	{
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return ;
        }    
    }
}


/******************************************************************************
 *
 * Ethernet Codelet support functions
 *
 ******************************************************************************/

/**
 * @fn void ixEthAccCodeletDBMaintenanceTask
 *
 * This function calls the ixEthAccDatabaseMaintenance every 
 * IX_ETH_DB_MAINTENANCE_TIME seconds.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletDBMaintenanceTask (void* arg, void** ptrRetObj)
{
    UINT32 count;

    ixEthAccCodeletDBMaintenanceTaskStop = FALSE;

    if (ixOsServMutexLock (&ixEthAccCodeletDBMaintenanceTaskRunning) != IX_SUCCESS)
    {
        printf("Error starting Database Maintenance thread! Failed to lock mutex.\n");
        return;
    }

    while (1)
    {
	/* 
	 * The Database maintenance function must be called at a period of
	 * approximately IX_ETH_ACC_MAINTENANCE_TIME seconds regardless of 
	 * whether learning is enabled or not.
         * 10 milliseconds are removed from the 60000 millisecond of delay
         * to take care about
         * the maintenance time itself.
	 */
        for (count = 0; count < IX_ETH_DB_MAINTENANCE_TIME; count++)
        { 
           if (ixEthAccCodeletDBMaintenanceTaskStop)
           {
               break;  /* Exit the thread */
           }
	   ix_ossl_sleep((count == 0 ? 990 : 1000));
        }

        if (ixEthAccCodeletDBMaintenanceTaskStop)
        {
            break;  /* Exit the thread */
        }
	ixEthDBDatabaseMaintenance();
    }

    ixOsServMutexUnlock (&ixEthAccCodeletDBMaintenanceTaskRunning);
}


/**
 * @fn void ixEthAccCodeletDispatcherPoll
 *
 * This function polls the Queue manager loop.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletDispatcherPoll (void* arg, void** ptrRetObj)
{
    ixEthAccCodeletDispatcherPollStop = FALSE;

    if (ixOsServMutexLock (&ixEthAccCodeletDispatcherPollRunning) != IX_SUCCESS)
    {
        printf("Error starting QMgr Dispatcher thread! Failed to lock mutex.\n");
        return;
    }

    while (1)
    {
        if (ixEthAccCodeletDispatcherPollStop)
        {
            break;  /* Exit the thread */
        }

	ix_ossl_sleep(1);
	(*ixEthAccCodeletDispatcherFunc) (IX_QMGR_QUELOW_GROUP);
    }

    ixOsServMutexUnlock (&ixEthAccCodeletDispatcherPollRunning);
}


/**
 * @fn IX_STATUS ixEthAccCodeletDispatcherStart()
 *
 * @param BOOL useInterrupt - start in interrupt or polled mode
 *
 * This function starts the Queue manager dispatch timer.
 * 
 * @return IX_SUCCESS - Dispatch timer successfully started
 * @return IX_FAIL - Error starting dispatch timer
 */

PRIVATE IX_STATUS ixEthAccCodeletDispatcherStart(BOOL useInterrupt)
{
    unsigned int dispatchtid;

    ixQMgrDispatcherLoopGet(&ixEthAccCodeletDispatcherFunc);

    if(useInterrupt)	/* Interrupt mode */
    {
	/* 
	 * Hook the QM QLOW dispatcher to the interrupt controller. 
	 */
	if (ixOsServIntBind(IXP425_INT_LVL_QM1,
			    (IxVoidFnVoidPtr)(ixEthAccCodeletDispatcherFunc),
			    (void *)IX_QMGR_QUELOW_GROUP) != IX_SUCCESS)
	{
	    printf("Failed to bind to QM1 interrupt\n");
	    return (IX_FAIL);
	}
    }
    else	/* Polled mode */
    {
 	if (ix_ossl_thread_create((ix_ossl_thread_entry_point_t)
				  ixEthAccCodeletDispatcherPoll,
				  NULL,
				  &dispatchtid) != IX_OSSL_ERROR_SUCCESS)
	{
	    printf("Error spawning dispatch task\n");
	    return (IX_FAIL);
	}

	ix_ossl_thread_set_priority(dispatchtid, IX_ETHACC_CODELET_QMR_PRIORITY);
    }

    return (IX_SUCCESS);
}


PRIVATE IX_STATUS ixEthAccCodeletDispatcherStop(BOOL useInterrupt)
{
    if(useInterrupt)	/* Interrupt mode */
    {
	/* 
	 * Unhook the QM QLOW dispatcher to the interrupt controller. 
	 */
	if (ixOsServIntUnbind(IXP425_INT_LVL_QM1) != IX_SUCCESS)
	{
	    printf("Failed to unbind to QM1 interrupt\n");
	    return (IX_FAIL);
	}
    }
    return (IX_SUCCESS);
}


/**
 * @fn IX_STATUS ixEthAccCodeletPhyInit()
 *
 * This function scans for and then initialises any available PHYs on
 * the board. It uses the EthAcc MII library routines to do so.
 * 
 * @return IX_SUCCESS - PHY(s) detected and initialised
 * @return IX_FAIL - Error initialising PHY(s)
 */

PRIVATE IX_STATUS ixEthAccCodeletPhyInit(BOOL phyLoopback)
{
   BOOL phyPresent[IXP425_ETH_ACC_MII_MAX_ADDR];

   UINT32 phyNo;
   UINT32 phyNoAddr;

   /* Scan for available Ethernet PHYs on the board */
   if(ixEthAccMiiPhyScan(phyPresent) == IX_FAIL)
   {	
       return (IX_FAIL);
   }
   else
   {
       maxPhyNo = 0;
       for(phyNoAddr=0, phyNo=0; phyNoAddr<IXP425_ETH_ACC_MII_MAX_ADDR; phyNoAddr++)
       {
	   if(phyPresent[phyNoAddr] == TRUE)
	   {
	       phyAddresses[phyNo++] = phyNoAddr;
               maxPhyNo = phyNo;

	       if(phyNo == IX_ETHACC_CODELET_MAX_PORT)
	       {
		   break;
	       }
	   }
       }
   }
    
   for(phyNo=0; phyNo<maxPhyNo; phyNo++)
   {
        /* Reset each phy */
        if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
             IX_FEATURE_CTRL_SILICON_TYPE_B0)
        {
	    /*Only when Ethernet is available, then add dynaic entries */
            if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (0 == phyNo)) ||
                ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                  IX_FEATURE_CTRL_COMPONENT_ENABLED) && (1 == phyNo))) 
	    {
                ixEthAccMiiPhyReset(phyAddresses[phyNo]);

                if (!phyLoopback)
                {
                    /* Set each phy properties */
                   ixEthAccMiiPhyConfig(phyAddresses[phyNo],
	  	     	                phyConf.speed100, 
 			                phyConf.fullDuplex, 
			                phyConf.autonegotiate);
                }
                else
                {
                     /* Enable PHY Loopback */
                     ixEthMiiPhyLoopbackEnable(phyAddresses[phyNo]);
                } 
            } 
        }
        else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                  IX_FEATURE_CTRL_SILICON_TYPE_A0) 
        {
            ixEthAccMiiPhyReset(phyAddresses[phyNo]);

            if (!phyLoopback)
            {
                /* Set each phy properties */
               ixEthAccMiiPhyConfig(phyAddresses[phyNo],
	  	  	            phyConf.speed100, 
 			            phyConf.fullDuplex, 
			            phyConf.autonegotiate);
            }
            else
            {
                 /* Enable PHY Loopback */
                 ixEthMiiPhyLoopbackEnable(phyAddresses[phyNo]);
            } 
        }
        else
        {
            printf("Error. Operation for other silicon stepping is undefined!.\n");
            return (IX_FAIL);
        }
   }

   if(!phyLoopback)
   {
       for(phyNo=0; phyNo<maxPhyNo; phyNo++)
       {

           BOOL speed, linkUp, fullDuplex, autoneg;

           if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                IX_FEATURE_CTRL_SILICON_TYPE_B0)
           {
	        /*Only when Ethernet is available, then add dynaic entries */
                if (((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
                      IX_FEATURE_CTRL_COMPONENT_ENABLED) && (0 == phyNo)) ||
                    ((ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                      IX_FEATURE_CTRL_COMPONENT_ENABLED) && (1 == phyNo))) 
	       {
                   /* wait until the link is up before setting the MAC duplex
                    * mode, the PHY duplex mode may change after autonegotiation 
                    */
                   (void)ixEthAccCodeletLinkUpCheck(phyNo);

                   /* get the link status */
                   ixEthAccMiiLinkStatus(phyAddresses[phyNo], 
                                         &linkUp,
                                         &speed,
                                         &fullDuplex,
                                         &autoneg);

                   /* Set the MAC duplex mode */
                   if(fullDuplex)
                   {
	                ixEthAccPortDuplexModeSet (phyNo, IX_ETH_ACC_FULL_DUPLEX);
                   }
                   else
                   {
	                ixEthAccPortDuplexModeSet (phyNo, IX_ETH_ACC_HALF_DUPLEX);
                   }
                   printf("\nPHY %d configuration:\n", phyNo);
                   ixEthAccMiiShow(phyAddresses[phyNo]);
               }  
           }
           else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
                     IX_FEATURE_CTRL_SILICON_TYPE_A0) 
           {
               /* wait until the link is up before setting the MAC duplex
                * mode, the PHY duplex mode may change after autonegotiation 
                */
               (void)ixEthAccCodeletLinkUpCheck(phyNo);

               /* get the link status */
               ixEthAccMiiLinkStatus(phyAddresses[phyNo], 
                                     &linkUp,
                                     &speed,
                                     &fullDuplex,
                                     &autoneg);

               /* Set the MAC duplex mode */
               if(fullDuplex)
               {
	            ixEthAccPortDuplexModeSet (phyNo, IX_ETH_ACC_FULL_DUPLEX);
               }
               else
               {
	            ixEthAccPortDuplexModeSet (phyNo, IX_ETH_ACC_HALF_DUPLEX);
               }
               printf("\nPHY %d configuration:\n", phyNo);
               ixEthAccMiiShow(phyAddresses[phyNo]);
           }
           else
           {
               printf("Error. Operation for other silicon stepping is undefined!.\n");
               return (IX_FAIL);
           }
       }
   }

   return (IX_SUCCESS);
}


/**
 * @fn IX_STATUS ixEthAccCodeletMemPoolInit()
 *
 * Initialise Ethernet Access Codelet MBUF pool
 *
 * @return IX_SUCCESS - MBuf pool successfully initialised
 * @return IX_FAIL - Error initialising MBuf pool
 */

PRIVATE IX_STATUS ixEthAccCodeletMemPoolInit(void)
{
    UINT32 bufNo;
    IX_MBUF *mBufPtr;
    
#define CACHE_LINE_BOUNDARY (32)
    
    UINT32 mBlkSize = CACHE_LINE_BOUNDARY * (sizeof (IX_MBUF) / CACHE_LINE_BOUNDARY + sizeof (IX_MBUF) % CACHE_LINE_BOUNDARY ? 1 : 0);

    ixEthAccCodeletBufPool = IX_ACC_DRV_DMA_MALLOC(IX_ETHACC_CODELET_MBUF_POOL_SIZE * mBlkSize);
        
    if (ixEthAccCodeletBufPool == NULL)
    {
      printf("Error allocating mBuf pool for Ethernet Codelet!\n");
      return (IX_FAIL);
    }
    
    mBufPtr = ixEthAccCodeletBufPool;

    ixEthAccCodeletBufData = IX_ACC_DRV_DMA_MALLOC(IX_ETHACC_CODELET_MBUF_DATA_POOL_SIZE);
    
    if (ixEthAccCodeletBufData == NULL)
    {
       printf("Error allocating mBuf data pool for Ethernet Codelet!\n");
       return (IX_FAIL);
    }

    /* Format our Receive mBufs */
    for(bufNo = 0;
      	bufNo < IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE * IX_ETHACC_CODELET_MAX_PORT;
      	bufNo++)
    {
       /* Initialise payload */ 
       IX_MBUF_MDATA(mBufPtr) = (char *) &ixEthAccCodeletBufData[bufNo * IX_ETHACC_CODELET_PCK_SIZE];

       /* Initialise length */
       IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_PCK_SIZE;

       /* No chains */
       IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr) = NULL;
       IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(mBufPtr) = NULL;
       mBufPtr->m_flags = 0x0;

       /* Add newly formated buffer to our Receive free queue */
       IX_ETHACC_CODELET_ADD_MBUF_TO_Q_TAIL(ixEthAccCodeletFreeBufQ, mBufPtr);
       
       mBufPtr = (IX_MBUF *)((UINT8 *)mBufPtr + mBlkSize);
    }

    /* Format our Transmit mBufs */
    for(bufNo = IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE * IX_ETHACC_CODELET_MAX_PORT; 
        bufNo < (IX_ETHACC_CODELET_MBUF_POOL_SIZE); 
        bufNo++)
    {
       /* Initialise payload */ 
       IX_MBUF_MDATA(mBufPtr) = (char *) &ixEthAccCodeletBufData[bufNo * IX_ETHACC_CODELET_PCK_SIZE];

       /* Initialise length */
       IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_PCK_SIZE;

       /* No chains */
       IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr) = NULL;
       IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(mBufPtr) = NULL;
       mBufPtr->m_flags = 0x0;

       /* Add newly formated buffer to our free queue */
       IX_ETHACC_CODELET_ADD_MBUF_TO_Q_TAIL(ixEthAccCodeletFreeBufQ, mBufPtr);
       
       mBufPtr = (IX_MBUF *)((UINT8 *)mBufPtr + mBlkSize);
    }

    return (IX_SUCCESS);
}


/**
 * @fn IX_STATUS ixEthAccCodeletMacSet()
 *
 * Set the MAC address for available Ports on the board.
 *
 * @return IX_SUCCESS - MAC address successfuly set
 * @return IX_FAIL - Error setting MAC address
 */

PRIVATE IX_STATUS ixEthAccCodeletMacSet(void)
{
    IxEthAccMacAddr npeMacAddr1 = IX_ETHACC_CODELET_NPEB_MAC;
    IxEthAccMacAddr npeMacAddr2 = IX_ETHACC_CODELET_NPEC_MAC;

#ifdef IX_ETHACC_CODELET_USE_NVRAM_MAC
    printf("\nReading MAC address information from non-volatile storage...\n");

    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(sysNvRamGet((UINT8 *)&npeMacAddr1.macAddress, 
	 	           IX_IEEE803_MAC_ADDRESS_SIZE, 
		           NV_MAC_ADRS_NPE1) == ERROR)
            {
   	         printf("Unable to read NPEB MAC address from non-volatile storage!\n");
	         return (IX_FAIL);
            }
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(sysNvRamGet((UINT8 *)&npeMacAddr2.macAddress, 
		           IX_IEEE803_MAC_ADDRESS_SIZE, 
		           NV_MAC_ADRS_NPE2) == ERROR)
            {
	        printf("Unable to read NPEC MAC address from non-volatile storage!\n");
	        return (IX_FAIL);
            }
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        if(sysNvRamGet((UINT8 *)&npeMacAddr1.macAddress, 
	 	       IX_IEEE803_MAC_ADDRESS_SIZE, 
		       NV_MAC_ADRS_NPE1) == ERROR)
        {
   	    printf("Unable to read NPEB MAC address from non-volatile storage!\n");
	    return (IX_FAIL);
        }

        if(sysNvRamGet((UINT8 *)&npeMacAddr2.macAddress, 
		       IX_IEEE803_MAC_ADDRESS_SIZE, 
		       NV_MAC_ADRS_NPE2) == ERROR)
        {
	    printf("Unable to read NPEC MAC address from non-volatile storage!\n");
	    return (IX_FAIL);
        }
    }   
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }
#endif

    if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
       IX_FEATURE_CTRL_SILICON_TYPE_B0)
    {
        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthAccPortUnicastMacAddressSet(IX_ETH_PORT_1, &npeMacAddr1) != IX_ETH_ACC_SUCCESS)
            {
	        return (IX_FAIL);
            }

            printf("PHY %d MAC address is:\t", IX_ETH_PORT_1);
            ixEthAccPortUnicastAddressShow(IX_ETH_PORT_1);
        }

        if (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
           IX_FEATURE_CTRL_COMPONENT_ENABLED)
        {
            if(ixEthAccPortUnicastMacAddressSet(IX_ETH_PORT_2, &npeMacAddr2) != IX_ETH_ACC_SUCCESS)
            {
	        return (IX_FAIL);
            }

            printf("PHY %d MAC address is:\t", IX_ETH_PORT_2);
            ixEthAccPortUnicastAddressShow(IX_ETH_PORT_2);
        }
    }
    else if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
              IX_FEATURE_CTRL_SILICON_TYPE_A0) 
    {
        if(ixEthAccPortUnicastMacAddressSet(IX_ETH_PORT_1, &npeMacAddr1) != IX_ETH_ACC_SUCCESS)
        {
	    return (IX_FAIL);
        }

        printf("PHY %d MAC address is:\t", IX_ETH_PORT_1);
        ixEthAccPortUnicastAddressShow(IX_ETH_PORT_1);

        if(ixEthAccPortUnicastMacAddressSet(IX_ETH_PORT_2, &npeMacAddr2) != IX_ETH_ACC_SUCCESS)
        {
	    return (IX_FAIL);
        }

        printf("PHY %d MAC address is:\t", IX_ETH_PORT_2);
        ixEthAccPortUnicastAddressShow(IX_ETH_PORT_2);
    }   
    else
    {
        printf("Error. Operation for other silicon stepping is undefined!.\n");
        return (IX_FAIL);
    }

    return (IX_SUCCESS);
}

/**
 * @fn IX_STATUS ixEthAccCodeletRecoverBuffers()
 *
 * Recover buffers used by a operation into the free queue.
 *
 * @return IX_SUCCESS - Recovery succeeded
 * @return IX_FAIL - Error recovering buffers
 */

IX_STATUS ixEthAccCodeletRecoverBuffers(void)
{
    unsigned int count = 0;

    memPoolFreeCnt = 0;

    printf("Recovering buffers...\n");

    /* Register callbacks to recover the buffers used */
    if(ixEthAccCodeletStop(ixEthAccCodeletMemPoolFreeRxCB,
			   ixEthAccCodeletMemPoolFreeRxCB,
			   ixEthAccCodeletMemPoolFreeTxCB,
			   ixEthAccCodeletMemPoolFreeTxCB) != IX_SUCCESS)
    {
	printf("Failed to stop the codelet\n");
	return (IX_FAIL);
    }

    /* wait 1 second for mbufs to be received */
    while ((count++ < 10) &&
	   ( memPoolFreeCnt != IX_ETHACC_CODELET_RX_MBUF_POOL_SIZE 
                               + IX_ETHACC_CODELET_TX_MBUF_POOL_SIZE))
    {
	ixOsServTaskSleep(100); /* 100 millisecs */
    }
    
    printf("%d buffers recovered\n", memPoolFreeCnt);

    return (IX_SUCCESS);
}


/**
 * @fn IX_STATUS ixEthAccCodeletReplenishBuffers()
 *
 * @param portNo - port number to replenish buffers into.
 * @param num - number of buffers to replenish.
 *
 * Function to replenish buffers into the Rx free queue.
 *
 * @return IX_SUCCESS - Replenish succeeded
 * @return IX_FAIL - Error replenishing buffers in Rx queue for port
 */

PRIVATE IX_STATUS ixEthAccCodeletReplenishBuffers(IxEthAccPortId portNo,
						 UINT32 num)
{
    IX_MBUF *mBufPtr;
    UINT32 numBufs;

    for(numBufs=0; numBufs<num; numBufs++)
    {
	if(IX_ETHACC_CODELET_IS_Q_EMPTY(ixEthAccCodeletFreeBufQ))
	{
	    printf("Buffer queue empty."
		   "Not enough free buffers for port %d!\n", portNo+1);
	    return (IX_FAIL);
	}
	
	IX_ETHACC_CODELET_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccCodeletFreeBufQ, mBufPtr);
	IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mBufPtr) = NULL;
	IX_MBUF_MLEN(mBufPtr) = IX_MBUF_PKT_LEN(mBufPtr) = IX_ETHACC_CODELET_PCK_SIZE;
	IX_ACC_DATA_CACHE_INVALIDATE(IX_MBUF_MDATA(mBufPtr), IX_MBUF_MLEN(mBufPtr));
	
	if(ixEthAccPortRxFreeReplenish(portNo, mBufPtr) != IX_SUCCESS)
	{
	    printf("Error replenishing free queue on port %d\n", portNo+1);
	    return (IX_FAIL);
	}
    }

    printf("Port %d Rx Free pool has %d buffers\n", portNo+1, numBufs);
    return (IX_SUCCESS);
}

/**
 * @fn void ixEthAccCodeletStatsPollTask
 *
 * This task polls the Codelet Stats and displays the rate of Rx and 
 * Tx packets per second.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletStatsPollTask (void* arg, void** ptrRetObj)
{
    static char stillRunning[] = "|/-\\";
    static int  stillRunningIndex = 0;

    ixEthAccCodeletStatsPollTaskStop = FALSE;

    if (ixOsServMutexLock (&ixEthAccCodeletStatsPollTaskRunning) != IX_SUCCESS)
    {
        printf("Error starting QMgr Dispatcher thread! Failed to lock mutex.\n");
        return;
    }

    while (1)
    {
	while (ixEthAccCodeletPollEnabled == FALSE)
	{
	    /* Sleep 1 sec */
	    ix_ossl_sleep(1000);
	    if (ixEthAccCodeletStatsPollTaskStop)
	    {
		break;  /* Exit the thread */
	    }
	}

	if (ixEthAccCodeletStatsPollTaskStop)
	{
	    break;  /* Exit the thread */
	}

	printf("\nPort1 Rates:              | Port2 Rates:\n"
	       "=====================================================\n");
	while (ixEthAccCodeletPollEnabled == TRUE)
	{
	    UINT32 timeStart = ixOsServTimestampGet();
	    /* Clear stats */
	    IxEthAccCodeletStats[0].rxCount=0;
	    IxEthAccCodeletStats[0].txCount=0;
	    IxEthAccCodeletStats[1].rxCount=0;
	    IxEthAccCodeletStats[1].txCount=0;

	    /* Sleep approximatively 1 sec */
	    ix_ossl_sleep(1000);
       
            /* check if the task should stop */ 
            if (ixEthAccCodeletStatsPollTaskStop)
            {
                break;  /* Exit the thread */
            }

	    if  (ixEthAccCodeletPollEnabled == TRUE)
	    {
		/* get a snapshot */
		UINT32 timeEnd = ixOsServTimestampGet();
		UINT32 rx0Count = IxEthAccCodeletStats[0].rxCount;
		UINT32 tx0Count = IxEthAccCodeletStats[0].txCount;
		UINT32 rx1Count = IxEthAccCodeletStats[1].rxCount;
		UINT32 tx1Count = IxEthAccCodeletStats[1].txCount;
		if (rx0Count < 420000 
		    && tx0Count < 420000 
		    && rx1Count < 420000 
		    && tx1Count < 420000)
		{
		    /* convert bus clocks to 10th of milliseconds 
		     * (with 420000 pckts/sec, there will not be overflows)
		     * and rate accuracy is 0.01%
		     */
		    UINT32 time10thMillisec = ((timeEnd - timeStart) * 15) / 100000;
                    if (time10thMillisec == 0)
                    {
                      /* this task didn't get enough cpu to run properly and timers may have overrun  */
                      continue;
                    }
                    else
                    {
		    /* correct the snaphot with the exact timing */
		    rx0Count = (rx0Count * 10000) / time10thMillisec;
		    tx0Count = (tx0Count * 10000) / time10thMillisec;
		    rx1Count = (rx1Count * 10000) / time10thMillisec;
		    tx1Count = (tx1Count * 10000) / time10thMillisec;
                    }
		}
		else
		{
		    /* convert the bus clocks to 100th seconds 
		     * (up to 42000000 packets per second without overflow)
		     * and rate accuracy is 1%
		     */
		    UINT32 time100thSec = ((timeEnd - timeStart) * 15) / 10000000;
                    if (time100thSec == 0)
                    {
                      /* this task didn't get enough cpu to run properly and timers may have overrun  */
                      continue;
                    }
                    else
                    {
		    /* correct the snaphot with the exact timing */
		    rx0Count = (rx0Count * 100) / time100thSec;
		    tx0Count = (tx0Count * 100) / time100thSec;
		    rx1Count = (rx1Count * 100) / time100thSec;
		    tx1Count = (tx1Count * 100) / time100thSec;
                    }
		}
		/* Display stats */
		printf("\rRx: %6u     Tx: %6u | Rx: %6u     Tx: %6u %c",
		       rx0Count, tx0Count, rx1Count, tx1Count,
		       stillRunning[stillRunningIndex++ & 3]);
	    }
	}
	printf("\n");
    }

    ixOsServMutexUnlock (&ixEthAccCodeletStatsPollTaskRunning);
}


/**
 * @fn void ixEthAccCodeletLoop()
 *
 * This function is called once a operation has started. It spawns the
 * stats poll task and waits for use interrupt to end the codelet.
 * 
 * @return IX_SUCCESS - operation successfully run
 * @return IX_FAIl - error spawning stats poll task
 */

PRIVATE IX_STATUS ixEthAccCodeletLoop(void)
{
    int ch = 0;

    printf(IX_ETHACC_CODELET_USER_INSTRUCTION);

    ixEthAccCodeletPollEnabled = TRUE;

    do
    {
	ch = getchar();
    }while(ch != 27);
    
    ixEthAccCodeletPollEnabled = FALSE;

    /* Dump Stats */
    ixEthAccCodeletShow();

    return (IX_SUCCESS);
}

/******************************************************************************
 *
 * Ethernet Codelet callback functions
 *
 ******************************************************************************/

/**
 * @fn void ixEthAccCodeletRxSinkCB()
 *
 * Callback for the Rx Sink Operation. Simply replenish the buffer back into the rx queue.
 * 
 * @return void
 */ 

PRIVATE void ixEthAccCodeletRxSinkCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
    /* Put the mbuf back on the rx free pool */
    if(cbTag == IX_ETH_PORT_1 || cbTag == IX_ETH_PORT_2)
    {
	IxEthAccCodeletStats[cbTag].rxCount++;

	ixEthAccCodeletMbufChainSizeSet(mBufPtr);

	if(ixEthAccPortRxFreeReplenish(cbTag, mBufPtr) != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Error replenishing RX free queue on port %d\n", 
			cbTag+1, 0, 0, 0, 0, 0);
	}    
    }
    else
    {
	ixOsServLog(LOG_ERROR, "Invalid Rx callback tag %d in RxSink Operation\n", 
		    cbTag, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletSwLoopbackTxCB()
 *
 * Transmit callback for Sw Loopback Operation. Transmitted frames are put on
 * the receive free pool.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletSwLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr)
{
    /* Put the mbuf back on the rx free pool */
    if(cbTag == IX_ETH_PORT_1 || cbTag == IX_ETH_PORT_2)
    {
	IxEthAccCodeletStats[cbTag].txCount++;

	ixEthAccCodeletMbufChainSizeSet(mBufPtr);

	if(ixEthAccPortRxFreeReplenish(cbTag, mBufPtr) != IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Error replenishing RX free queue on port %d\n", 
			cbTag+1, 0, 0, 0, 0, 0);
	}    
    }
    else
    {
	ixOsServLog(LOG_ERROR, "Invalid Tx callback tag %d in RxSink Operation\n", 
		    cbTag, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletSwLoopbackRxCB()
 *
 * Receive callback for the Sw Loopback Operation. Any frames received are transmited
 * back out on the same port.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletSwLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
    /* Transmit the buffer back */
    if(cbTag == IX_ETH_PORT_1 || cbTag == IX_ETH_PORT_2)
    {
	IxEthAccCodeletStats[cbTag].rxCount++;

	if(ixEthAccPortTxFrameSubmit(cbTag, mBufPtr,
				     IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Tx Buffer submission failure port %d\n", 
			cbTag+1, 0, 0, 0, 0, 0);
	}
    }
    else
    {
	ixOsServLog(LOG_ERROR, "Invalid Rx callback tag %d in Sw Loopback Operation\n", 
		    cbTag, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletBridgeTxCB()
 *
 * Transmit callback for Bridge Operation. Transmitted frames are put on
 * the receive queue of the other port.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletBridgeTxCB(UINT32 cbTag, IX_MBUF* mBufPtr)
{
    /* 
     * Put the mbuf on the rx free q of the other port since it was received
     * there initially.
     */
    if(cbTag == IX_ETH_PORT_1)
    {
	IxEthAccCodeletStats[cbTag].txCount++;

	ixEthAccCodeletMbufChainSizeSet(mBufPtr);

	if(ixEthAccPortRxFreeReplenish(IX_ETH_PORT_2, mBufPtr)!=IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Error replenishing RX free q for port 2\n", 
			0, 0, 0, 0, 0, 0);
	}    
    }
    else if(cbTag == IX_ETH_PORT_2)
    {
	IxEthAccCodeletStats[cbTag].txCount++;

	ixEthAccCodeletMbufChainSizeSet(mBufPtr);

	if(ixEthAccPortRxFreeReplenish(IX_ETH_PORT_1, mBufPtr)!=IX_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Error replenishing RX free q for port 1\n", 
			0, 0, 0, 0, 0, 0);
	}    
    }
    else
    {
	ixOsServLog(LOG_ERROR, "Invalid Tx callback tag %d in Bridge Operation\n", 
		    cbTag, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletBridgeRxCB()
 *
 * Receive callback for Bridge Operation. Received frames are retransmited on
 * the other port.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletBridgeRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
    /* Transmit the buffer on the other port */
    if(cbTag == IX_ETH_PORT_1)
    {
	IxEthAccCodeletStats[cbTag].rxCount++;

	if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_2, mBufPtr,
				     IX_ETH_ACC_TX_DEFAULT_PRIORITY)!=IX_ETH_ACC_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Tx Buffer submission failed on port 2\n", 
			0, 0, 0, 0, 0, 0);
	}
    }
    else if(cbTag == IX_ETH_PORT_2)
    {
	IxEthAccCodeletStats[cbTag].rxCount++;

	if(ixEthAccPortTxFrameSubmit(IX_ETH_PORT_1, mBufPtr,
				     IX_ETH_ACC_TX_DEFAULT_PRIORITY)!=IX_ETH_ACC_SUCCESS)
	{
	    ixOsServLog(LOG_ERROR, "Tx Buffer submission failed on port 1\n", 
			0, 0, 0, 0, 0, 0);
	}
    }
    else
    {
	ixOsServLog(LOG_ERROR, "Invalid Rx callback tag %d in Bridge Operation\n", 
		    cbTag, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletTxGenRxSinkLoopbackRxCB()
 *
 * Receive callback for TxGen-RxSink Loopback Operation. Received frames are put back on
 * the free queue.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletTxGenRxSinkLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
    /* Return mBuf to free queue on same port */
    IxEthAccCodeletStats[cbTag].rxCount++;

#ifdef IX_ETHACC_CODELET_TXGENRXSINK_VERIFY 
    IX_ETHACC_CODELET_DATA_VERIFY(IX_MBUF_MDATA(mBufPtr), compData);
#endif
    ixEthAccCodeletMbufChainSizeSet(mBufPtr);

    if(ixEthAccPortRxFreeReplenish(cbTag, mBufPtr) != IX_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Error replenishing RX free q in TxGen-RxSink\n", 
		    0, 0, 0, 0, 0, 0);
    }
}


/**
 * @fn void ixEthAccCodeletTxGenRxSinkLoopbackTxCB()
 *
 * Transmit callback for TxGenRxSink Loopback Operation. Transmitted frames are re-
 * transmitted.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletTxGenRxSinkLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr)
{
    IxEthAccCodeletStats[cbTag].txCount++;

    /* Re-transmit the frame on same port */
    if(ixEthAccPortTxFrameSubmit(cbTag, mBufPtr,
				 IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Tx Buffer submission failure in TxGen-RxSink\n", 
		    0, 0, 0, 0, 0, 0);
    }
}

/**
 * @fn void ixEthAccCodeletPhyLoopbackRxCB()
 *
 * Receive callback for Phy loopback operation. Received frames are replenished
 * back to Rx Free queue to further packet reception.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletPhyLoopbackRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
  IxEthAccCodeletStats[cbTag].rxCount++;

#ifdef IX_ETHACC_CODELET_TXGENRXSINK_VERIFY 
  IX_ETHACC_CODELET_DATA_VERIFY(IX_MBUF_MDATA(mBufPtr), compData);
#endif

  ixEthAccCodeletMbufChainSizeSet(mBufPtr);

  if(ixEthAccPortRxFreeReplenish(cbTag, mBufPtr) != IX_SUCCESS)
  {
	ixOsServLog(LOG_ERROR, "Error replenishing RX free q in Phy Loopback\n", 
		    0, 0, 0, 0, 0, 0);
  }
}


/**
 * @fn void ixEthAccCodeletPhyLoopbackTxCB()
 *
 * Transmit callback for Phy Loopback operation. Transmitted frames are retransmitted
 * back to ensure traffics are continuous.
 * 
 * @return void
 */

PRIVATE void ixEthAccCodeletPhyLoopbackTxCB(UINT32 cbTag, IX_MBUF* mBufPtr)
{
    /* Return mBuf to free queue on same port */
    IxEthAccCodeletStats[cbTag].txCount++;

    /* Re-transmit the frame on same port */
    if(ixEthAccPortTxFrameSubmit(cbTag, mBufPtr,
				 IX_ETH_ACC_TX_DEFAULT_PRIORITY) != IX_ETH_ACC_SUCCESS)
    {
	ixOsServLog(LOG_ERROR, "Tx Buffer submission failure in PHY Loopback\n", 
		    0, 0, 0, 0, 0, 0);
    }
}

/*
 * @fn ixEthAccCodeletMemPoolFreeRxCb
 *
 * This callback is used at the end of each ethernet operation.
 * It recovers the buffers used and places them back in the free queue.
 * This way the MBUF pool is free to be used by another ethernet operation.
 *
 * @return void
 */

PRIVATE void ixEthAccCodeletMemPoolFreeRxCB(UINT32 cbTag, IX_MBUF* mBufPtr, IxEthAccPortId portId)
{
    IX_ETHACC_CODELET_ADD_MBUF_TO_Q_TAIL(ixEthAccCodeletFreeBufQ, mBufPtr);
    memPoolFreeCnt++;
}

/*
 * @fn ixEthAccCodeletMemPoolFreeTxCb
 *
 * This callback is used at the end of each ethernet operation.
 * It recovers the buffers used and places them back in the free queue.
 * This way the MBUF pool is free to be used by another ethernet operation.
 *
 * @return void
 */

PRIVATE void ixEthAccCodeletMemPoolFreeTxCB(UINT32 cbTag, IX_MBUF* mBufPtr)
{
    IX_ETHACC_CODELET_ADD_MBUF_TO_Q_TAIL(ixEthAccCodeletFreeBufQ, mBufPtr);
    memPoolFreeCnt++;
}



