/*
 * Alsa Driver for AIC26 codec on MPC5200B platform board
 *
 * Copyright (C) 2006 AppSpec Computer Technologies Corp.
 *                    Jeff Gibbons <jeff.gibbons@appspec.com>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 *
 * Support up to 4 devices simultaneously by
 * using PSC1, PSC2, PSC3, PSC6 as I2S port
 *
 * Assumption:
 * Fsref = 44.1kHz, MClk generated by external devices (11.2896 MHz)
 * I2S run as slave
 * SPI_SS# gpio pin is defined in spi driver
 *
 */

#include <linux/config.h>
#include <sound/driver.h>
#include <sound/info.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/spi/mpc52xx-spi.h>

#include <asm/mpc52xx.h>
#include <asm/mpc52xx_psc.h>
#include <asm/io.h>

#include <platforms/52xx/bestcomm.h>
#include <platforms/52xx/gen_bd.h>

#ifdef CONFIG_PM
#include <linux/pm.h>
#endif

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/memalloc.h>

#include "mpc52xx-aic26.h"
#include "mpc52xx-aic26-mixer.h"

#define SPI_use_PSC3_6_9

#define DRIVER_NAME "mpc52xx-aic26"

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("MPC52XX AIC26 driver for ALSA");

static int index[SNDRV_CARDS] = SNDRV_DEFAULT_IDX;	/* Index 0-MAX */
static char *id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int enable[SNDRV_CARDS] = {	/* Enable this card */
	[0 ... (SNDRV_CARDS - 1)] = 0,
};
static int psc_idx[SNDRV_CARDS] = {	/* Valid PSC is 0,1,2,5 */
	[0 ... (SNDRV_CARDS - 1)] = -1
};

static int spi_ss[SNDRV_CARDS] = {
	[0 ... (SNDRV_CARDS - 1)] = -1,
};

module_param_array(index, int, NULL, 0444);
MODULE_PARM_DESC(index, "Index value for MPC52XX AIC26.");
module_param_array(id, charp, NULL, 0444);
MODULE_PARM_DESC(id, "ID string for MPC52XX AIC26.");
module_param_array(enable, bool, NULL, 0444);
MODULE_PARM_DESC(enable, "Enable MPC52XX AIC26 ALSA.");
module_param_array(psc_idx, int, NULL, 0444);
MODULE_PARM_DESC(psc_idx, "PSC port for MPC52XX AIC26 ALSA.");
module_param_array(spi_ss, int, NULL, 0444);
MODULE_PARM_DESC(spi_ss, "SPI SS for MPC52XX AIC26 ALSA.");

static snd_card_t *snd_mpc52xx_aic26[SNDRV_CARDS] = SNDRV_DEFAULT_PTR;

#define PSC(chip) ((struct mpc52xx_psc __iomem *)((chip)->membase))

static int sdma_ipr_psc_tx[] = {
	SDMA_IPR_PSC1_TX,
	SDMA_IPR_PSC2_TX,
	SDMA_IPR_PSC3_TX,
	SDMA_IPR_PSC4_TX,
	SDMA_IPR_PSC5_TX,
	SDMA_IPR_PSC6_TX,
};

static int sdma_ipr_psc_rx[] = {
	SDMA_IPR_PSC1_RX,
	SDMA_IPR_PSC2_RX,
	SDMA_IPR_PSC3_RX,
	SDMA_IPR_PSC4_RX,
	SDMA_IPR_PSC5_RX,
	SDMA_IPR_PSC6_RX,
};

static int sdma_initiator_psc_rx[] = {
	SDMA_INITIATOR_PSC1_RX,
	SDMA_INITIATOR_PSC2_RX,
	SDMA_INITIATOR_PSC3_RX,
	SDMA_INITIATOR_PSC4_RX,
	SDMA_INITIATOR_PSC5_RX,
	SDMA_INITIATOR_PSC6_RX,
};

static int sdma_initiator_psc_tx[] = {
	SDMA_INITIATOR_PSC1_TX,
	SDMA_INITIATOR_PSC2_TX,
	SDMA_INITIATOR_PSC3_TX,
	SDMA_INITIATOR_PSC4_TX,
	SDMA_INITIATOR_PSC5_TX,
	SDMA_INITIATOR_PSC6_TX,
};

static int psc_irq[] = {
	MPC52xx_PSC1_IRQ,
	MPC52xx_PSC2_IRQ,
	MPC52xx_PSC3_IRQ,
	MPC52xx_PSC4_IRQ,
	MPC52xx_PSC5_IRQ,
	MPC52xx_PSC6_IRQ,
};

static unsigned int mpc52xx_aic26_fixed_rates[] = {
	8000, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000,
};

static snd_pcm_hw_constraint_list_t hw_constraints_rates = {
	.count = ARRAY_SIZE(mpc52xx_aic26_fixed_rates),
	.list = mpc52xx_aic26_fixed_rates,
	.mask = 0,
};

void audio_aic26_write(int spi_ss, u16 address, u16 data)
{
	uint8_t in[4], out[4];
	in[0] = AIC26_WRITE_COMMAND_WORD(address) >> 8;
	in[1] = AIC26_WRITE_COMMAND_WORD(address) & (0xff);
	in[2] = data >> 8;
	in[3] = data & (0xff);
	mpc52xx_spi_transfer(spi_ss, in, out, 4, 0);
}

u16 audio_aic26_read(int spi_ss, u16 address)
{
	uint8_t in[4], out[4];
	in[0] = AIC26_READ_COMMAND_WORD(address) >> 8;
	in[1] = AIC26_READ_COMMAND_WORD(address) & (0xff);
	in[2] = 0;
	in[3] = 0;
	mpc52xx_spi_transfer(spi_ss, in, out, 4, 0);
	return (out[2] << 8 | out[3]);
}

static int mpc52xx_aic26_set_fsref(int spi_ss, long rate)
{
	union aic26_data prog1, prog2, ctrl3;

	snd_printd("%s(spi_ss=%i, rate=%li)\n", __FUNCTION__, spi_ss, rate);

	prog1.value = audio_aic26_read(spi_ss, AIC26_PLL_PROG1_ADDR);
	prog2.value = audio_aic26_read(spi_ss, AIC26_PLL_PROG2_ADDR);
	ctrl3.value = audio_aic26_read(spi_ss, AIC26_AUDIO_CTRL3_ADDR);

	prog1.pll_prog1.pllsel = 1;
	prog1.pll_prog1.pval = 1;
	ctrl3.audio_ctrl3.slvms = 1;	/* Make sure Master Mode is set */

	switch (rate) {
	case 44100:
		prog1.pll_prog1.jval = 7;
		prog2.pll_prog2.dval = 5264;
		ctrl3.audio_ctrl3.reffs = 1;
		break;

	case 48000:
		prog1.pll_prog1.jval = 8;
		prog2.pll_prog2.dval = 1920;
		ctrl3.audio_ctrl3.reffs = 0;
		break;

	default:
		return -EINVAL;
	}
	audio_aic26_write(spi_ss, AIC26_PLL_PROG1_ADDR, prog1.value);
	audio_aic26_write(spi_ss, AIC26_PLL_PROG2_ADDR, prog2.value);
	audio_aic26_write(spi_ss, AIC26_AUDIO_CTRL3_ADDR, ctrl3.value);
	return 0;
}

static int mpc52xx_aic26_set_samplerate(int spi_ss, long rate)
{
	union aic26_data data;
	int dividorid;
	long int fsref;
	int rc = -EINVAL;

	snd_printd("%s(spi_ss=%i, rate=%li)\n", __FUNCTION__, spi_ss, rate);
	switch (rate) {
	case 8000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_6; break;
	case 11025: fsref = 44100; dividorid = AIC26_SAMPLE_RATE_DIV_4; break;
	case 12000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_4; break;
	case 16000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_3; break;
	case 22050: fsref = 44100; dividorid = AIC26_SAMPLE_RATE_DIV_2; break;
	case 24000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_2; break;
	case 32000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_1_5; break;
	case 44100: fsref = 44100; dividorid = AIC26_SAMPLE_RATE_DIV_1; break;
	case 48000: fsref = 48000; dividorid = AIC26_SAMPLE_RATE_DIV_1; break;
	default:
		return -EINVAL;
	}

	if ((rc = mpc52xx_aic26_set_fsref(spi_ss, fsref)) < 0)
		return rc;

	data.value = audio_aic26_read(spi_ss, AIC26_AUDIO_CTRL1_ADDR);
	data.audio_ctrl1.dacfs = dividorid;
	data.audio_ctrl1.adcfs = dividorid;
	audio_aic26_write(spi_ss, AIC26_AUDIO_CTRL1_ADDR, data.value);

	return 0;
}

static int mpc52xx_aic26_set_format(struct snd_card_mpc52xx_aic26 *chip,
				    int spi_ss, long format)
{
	struct mpc52xx_psc __iomem *psc;
	union aic26_data data;
	int formatid;
	int mode;
	u32 value;

	snd_printd("%s(chip=%p, spi_ss=%i, format=%li)\n",
	           __FUNCTION__, chip, spi_ss, format);

	switch (format) {
	case SNDRV_PCM_FORMAT_S8: formatid = 0; mode = 0x1; break;
	case SNDRV_PCM_FORMAT_S16_BE: formatid = 0; mode = 0x2; break;
	case SNDRV_PCM_FORMAT_S24_BE: formatid = 2; mode = 0x7; break;
	case SNDRV_PCM_FORMAT_S32_BE: formatid = 3; mode = 0xF; break;
	default:
		return -EINVAL;
	}

	data.value = audio_aic26_read(spi_ss, AIC26_AUDIO_CTRL1_ADDR);
	data.audio_ctrl1.wlen = formatid;
	audio_aic26_write(spi_ss, AIC26_AUDIO_CTRL1_ADDR, data.value);

	psc = PSC(chip);
	value = (in_be32(&psc->sicr) & ~(0xF << 24)) | (mode << 24);
	out_be32(&psc->sicr, value);

	return 0;
}

static int mpc52xx_aic26_set_channel(int spi_ss, long channel)
{
	snd_printd("%s(spi_ss=%i, channel=%li)\n",
	           __FUNCTION__, spi_ss, channel);
	switch (channel) {
	case 2:
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

/* Do some hardware initialization */
static int mpc52xx_aic26_audio_init(struct snd_card_mpc52xx_aic26 *chip)
{
	struct mpc52xx_psc __iomem *psc;
	union aic26_data ctrl1;

	snd_printd("%s(chip=%p)\n", __FUNCTION__, chip);

	/* Config the PSC Port of MPC52XX */
	chip->mapbase = MPC52xx_PA(MPC52xx_PSCx_OFFSET(chip->psc_idx + 1));
	/* We ioremap ourself */
	chip->membase = ioremap(chip->mapbase, MPC52xx_PSC_SIZE);
	if (chip->membase == NULL)
		return -EINVAL;
	if (request_mem_region(chip->mapbase, MPC52xx_PSC_SIZE,
			       "mpc52xx_psc_i2s") == NULL) {
		iounmap(chip->membase);
		return -EBUSY;
	}

	psc = PSC(chip);

	/* Make sure all PSC interrupts disabled. */
	/* Interrupt Event comes from Bestcomm task */
	out_be16(&psc->mpc52xx_psc_imr, 0);

	out_8(&psc->command, 3 << 4); /* reset transmitter */
	out_8(&psc->command, 2 << 4); /* reset receiver */
	out_8(&psc->command, 1 << 4); /* reset mode */
	out_8(&psc->command, 4 << 4); /* reset error */

	/* Select codec I2S slave mode, 16 bits, msb first,  DTS1=1 */
	/* We assume MClk is not generated by MPC52XX */
	out_be32(&psc->sicr, 0x22600000);

	/* First write: RxRdy (FIFO Alarm) generates receive FIFO interrupt */
	/* Second write to mode: register Normal mode for non loopback */
	out_8(&psc->mode, 0);
	out_8(&psc->mode, 0);

	out_be16(&psc->rfalarm, 0x100);		/* set RFALARM level */
	out_8(&psc->rfcntl, 0x4);		/* set RFGRAN level (bytes) */
	out_be16(&psc->tfalarm, 0x100);		/* set TFALARM level */
	out_8(&psc->tfcntl, 0x7);		/* set TFGRAN level (bytes*4) */

	/* Config AIC26 Codec */
	/* Reset AIC26 codec to powerup defaults */
	audio_aic26_write(chip->spi_ss, AIC26_RESET_ADDR, 0xBB00);

	/* Config I2S Mode */
	ctrl1.value = audio_aic26_read(chip->spi_ss, AIC26_AUDIO_CTRL1_ADDR);
	ctrl1.audio_ctrl1.datfm = 0;
	audio_aic26_write(chip->spi_ss, AIC26_AUDIO_CTRL1_ADDR, ctrl1.value);

	/* Start with a sane default */
	mpc52xx_aic26_set_samplerate(chip->spi_ss, 44100);

	return 0;
}

/*
 * this stops the dma and clears the dma ptrs
 */
static void pcm_trigger_stop(snd_pcm_substream_t * substream)
{
	struct mpc52xx_psc __iomem *psc;
	unsigned long flags;
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	int stream_id = substream->pstr->stream;
	struct audio_stream *s = &chip->s[stream_id];

	spin_lock_irqsave(&s->dma_lock, flags);
	s->active = 0;

	s->sdma->flags = SDMA_FLAGS_NONE;
	sdma_disable(s->sdma);	/* disable task */
	sdma_clear_irq(s->sdma);	/* clean any pending irq */

	while (!sdma_queue_empty(s->sdma)) {
		sdma_retrieve_buffer(s->sdma, NULL);
	}

	psc = PSC(chip);

	if (stream_id == SNDRV_PCM_STREAM_CAPTURE) {
		if (chip->s[SNDRV_PCM_STREAM_PLAYBACK].active)
			out_be16(&psc->mpc52xx_psc_imr, 1 << 27);
		else {
			out_be16(&psc->mpc52xx_psc_imr, 0);
			out_8(&psc->command, 2 << 4);	/* reset receiver */
			out_8(&psc->command, 3 << 4);	/* reset transmitter */
			out_8(&psc->command, 4 << 4);	/* reset error */
		}
	} else {
		out_8(&psc->command, 3 << 4);	/* reset transmitter */
		out_8(&psc->command, 4 << 4);	/* reset error */
		if (chip->s[SNDRV_PCM_STREAM_CAPTURE].active)
			out_be16(&psc->mpc52xx_psc_imr, 1 << 28);
		else {
			out_be16(&psc->mpc52xx_psc_imr, 0);
			out_8(&psc->command, 2 << 4);	/* reset receiver */
		}
	}

	spin_unlock_irqrestore(&s->dma_lock, flags);
}

#if defined(CONFIG_SND_MPC52xx_CODEC3_SPI_BUGFIX)
static void mpc52xx_fixup_codec3_spi(void* data)
{
	struct mpc52xx_psc __iomem *psc;
	struct mpc52xx_gpio __iomem *gpio;
	u32 val;

	gpio = ioremap(MPC52xx_PA(MPC52xx_GPIO_OFFSET), MPC52xx_GPIO_SIZE);
	psc = ioremap(MPC52xx_PA(MPC52xx_PSCx_OFFSET(3)), MPC52xx_PSC_SIZE);

	/* SPI+CODEC3 do not play nice; disable SPI until synced w/ AIC26 */
	while(mpc52xx_spi_pause());

	/* Check if the fixup condition exists */
	val = in_be32(&gpio->port_config);
	if ((val & 0xF00) != 0xF00)
		goto exit;

	/* disable SPI pin routing */
	out_be32(&gpio->port_config, (val & ~0xF00) | 0x600);

	msleep(1);

	/* Wait for sync with codec */
	while ((in_8(&psc->ipcr_acr.ipcr) & 0x80) == 0) {
		printk("9\n");
		schedule();
	}

	/* reenable SPI pins */
	out_be32(&gpio->port_config, in_be32(&gpio->port_config) | 0xF00);

exit:
	iounmap(gpio);
	iounmap(psc);
	mpc52xx_spi_resume();
}

static void mpc52xx_fixup_codec3_spi(void* data);
DECLARE_WORK(mpc52xx_fixup_codec3_spi_work, mpc52xx_fixup_codec3_spi, NULL);
#endif

/*
 *  Main dma routine, requests dma according where you are in main alsa buffer
 */
static void pcm_trigger_start(snd_pcm_substream_t * substream)
{
	struct mpc52xx_psc __iomem *psc;
	snd_pcm_runtime_t *runtime;
	unsigned int period_byte_size;
	void *dma_data_ptr;
	phys_addr_t rxtx_fifo;

	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	int stream_id = substream->pstr->stream;
	struct audio_stream *s = &chip->s[stream_id];

	runtime = substream->runtime;
	period_byte_size = frames_to_bytes(runtime, runtime->period_size);
	dma_data_ptr = (void *)virt_to_phys(runtime->dma_area);

	/* The DMA channel will repeat itself once initialized */

	s->period_byte_size = period_byte_size;
	s->period_start = dma_data_ptr;
	s->period_end = dma_data_ptr + period_byte_size * runtime->periods;
	s->period_next_pt = dma_data_ptr;

	psc = PSC(chip);
	rxtx_fifo =
	    MPC52xx_PA(MPC52xx_PSCx_OFFSET(chip->psc_idx + 1) +
		       FIELD_OFFSET(mpc52xx_psc, mpc52xx_psc_buffer_32));

	if (stream_id == SNDRV_PCM_STREAM_CAPTURE) {
		out_8(&psc->command, 2 << 4);	/* reset receiver */
		out_8(&psc->command, 4 << 4);	/* reset error */
		out_8(&psc->command, 1 << 0);	/* enable receiver */

		out_be16(&psc->mpc52xx_psc_imr, 1 << 28);	/* enable error int. */
		sdma_gen_bd_rx_init(chip->psc_idx, s->sdma, rxtx_fifo,
				    sdma_initiator_psc_rx[chip->psc_idx],
				    sdma_ipr_psc_rx[chip->psc_idx],
				    period_byte_size);
		sdma_set_task_auto_start(s->sdma->tasknum, s->sdma->tasknum);
	} else {
		out_8(&psc->command, 3 << 4);	/* reset transmitter */
		out_8(&psc->command, 4 << 4);	/* reset error */
		/* enable transmitter (need to enable receiver also) */
		/* see 15.3.2.3 of MPC5200B User's Guide */
		out_8(&psc->command, (1 << 2) | (1 << 0));

		out_be16(&psc->mpc52xx_psc_imr, 1 << 27);	/* enable error int. */
		sdma_gen_bd_tx_init(chip->psc_idx, s->sdma, rxtx_fifo,
				    sdma_initiator_psc_tx[chip->psc_idx],
				    sdma_ipr_psc_tx[chip->psc_idx]);
		sdma_set_task_auto_start(s->sdma->tasknum, s->sdma->tasknum);
	}

	while (!sdma_queue_full(s->sdma)) {
		sdma_submit_buffer(s->sdma, NULL, s->period_next_pt,
				   period_byte_size);
		s->period_next_pt += period_byte_size;
		if (s->period_next_pt >= s->period_end) {
			s->period_next_pt = s->period_start;
		}
	}
	s->sdma->flags = SDMA_FLAGS_ENABLE_TASK;

	sdma_enable(s->sdma);

#if defined(CONFIG_SND_MPC52xx_CODEC3_SPI_BUGFIX)
	/* SPI+CODEC3 do not play nice; call workaround function */
	if (chip->psc_idx == 2)
		schedule_work(&mpc52xx_fixup_codec3_spi_work);
#endif
}

/* 
 * Alsa section
 * PCM settings and callbacks
 */

static int snd_mpc52xx_aic26_trigger(snd_pcm_substream_t * substream, int cmd)
{
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	int stream_id = substream->pstr->stream;
	struct audio_stream *s = &chip->s[stream_id];
	int err = 0;

	/* note local interrupts are already disabled in the midlevel code */
	spin_lock(&s->dma_lock);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		/* requested stream startup */
		s->active = 1;
		pcm_trigger_start(substream);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		/* requested stream shutdown */
		pcm_trigger_stop(substream);
		break;
	default:
		err = -EINVAL;
		break;
	}
	spin_unlock(&s->dma_lock);

	return err;
}

static int snd_mpc52xx_aic26_prepare(snd_pcm_substream_t * substream)
{
	return 0;
}

/*
 *  This function should calculate the current position of the dma in the
 *  buffer. It will help alsa middle layer to continue update the buffer.
 *  Its correctness is crucial for good functioning.
 */
static snd_pcm_uframes_t snd_mpc52xx_aic26_pointer(snd_pcm_substream_t *
						   substream)
{
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	int stream_id = substream->pstr->stream;
	struct audio_stream *s = &chip->s[stream_id];
	snd_pcm_runtime_t *runtime = substream->runtime;
	unsigned long flags;
	dma_addr_t count;

	spin_lock_irqsave(&s->dma_lock, flags);

	count = s->sdma->bd[s->sdma->outdex].data - s->period_start;

	/* FIXME better get the current DMA process pointer */
	count += 0;

	spin_unlock_irqrestore(&s->dma_lock, flags);

	return bytes_to_frames(runtime, count);
}

/* Hardware capabilities */
static snd_pcm_hardware_t snd_mpc52xx_aic26_capture = {
	.info = (SNDRV_PCM_INFO_INTERLEAVED | SNDRV_PCM_INFO_BLOCK_TRANSFER |
		 SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats = (SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_BE |
		    SNDRV_PCM_FMTBIT_S24_BE | SNDRV_PCM_FMTBIT_S32_BE),
	.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 |
		  SNDRV_PCM_RATE_22050 | SNDRV_PCM_RATE_44100 |
		  SNDRV_PCM_RATE_KNOT),
	.rate_min = 8000,
	.rate_max = 48000,
	.channels_min = 2,
	.channels_max = 2,
	.buffer_bytes_max = 2 * 1024 * 1024,
	.period_bytes_min = 32,
	.period_bytes_max = 1024 * 1024,
	.periods_min = 2,
	.periods_max = 256,
	.fifo_size = 0,
};

static snd_pcm_hardware_t snd_mpc52xx_aic26_playback = {
	.info = (SNDRV_PCM_INFO_INTERLEAVED | SNDRV_PCM_INFO_BLOCK_TRANSFER |
		 SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats = (SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_BE |
		    SNDRV_PCM_FMTBIT_S24_BE | SNDRV_PCM_FMTBIT_S32_BE),
	.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 |
		  SNDRV_PCM_RATE_22050 | SNDRV_PCM_RATE_44100 |
		  SNDRV_PCM_RATE_KNOT),
	.rate_min = 8000,
	.rate_max = 48000,
	.channels_min = 2,
	.channels_max = 2,
	.buffer_bytes_max = 2 * 1024 * 1024,
	.period_bytes_min = 32,
	.period_bytes_max = 1024 * 1024,
	.periods_min = 2,
	.periods_max = 256,
	.fifo_size = 0,
};

int mpc52xx_aic26_clock_on(struct snd_card_mpc52xx_aic26 *chip)
{
	union aic26_data aic26;

	snd_printd("%s(chip=%p)\n", __FUNCTION__, chip);

	/* send command to codec to stop playback or capture */
	aic26.value =
	    audio_aic26_read(chip->spi_ss, AIC26_CODEC_POWER_CTRL_ADDR);
	aic26.codec_power_ctrl.adpwdn = 0;	/* capture */
	aic26.codec_power_ctrl.dapwdn = 0;	/* playback */
	aic26.codec_power_ctrl.pwdnc = 0;	/* codec */
	audio_aic26_write(chip->spi_ss, AIC26_CODEC_POWER_CTRL_ADDR,
			  aic26.value);

	/* without the following delay, the first second sample will be missed */
	msleep_interruptible(1000);

	return 0;
}

int mpc52xx_aic26_clock_off(struct snd_card_mpc52xx_aic26 *chip)
{
	union aic26_data aic26;

	snd_printd("%s(chip=%p)\n", __FUNCTION__, chip);

	/* send command to codec to stop playback or capture */
	aic26.value =
	    audio_aic26_read(chip->spi_ss, AIC26_CODEC_POWER_CTRL_ADDR);
	aic26.codec_power_ctrl.adpwdn = 1;	/* capture */
	aic26.codec_power_ctrl.dapwdn = 1;	/* playback */
	aic26.codec_power_ctrl.pwdnc = 1;	/* codec */
	audio_aic26_write(chip->spi_ss, AIC26_CODEC_POWER_CTRL_ADDR,
			  aic26.value);
	return 0;
}

static int snd_card_mpc52xx_aic26_open(snd_pcm_substream_t * substream)
{
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int stream_id = substream->pstr->stream;
	int err;

	snd_printd("%s(substream=%p)\n", __FUNCTION__, substream);

	chip->s[stream_id].stream = substream;

	mpc52xx_aic26_clock_on(chip);

	if (stream_id == SNDRV_PCM_STREAM_PLAYBACK)
		runtime->hw = snd_mpc52xx_aic26_playback;
	else
		runtime->hw = snd_mpc52xx_aic26_capture;
	if ((err =
	     snd_pcm_hw_constraint_integer(runtime,
					   SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
	if ((err =
	     snd_pcm_hw_constraint_list(runtime, 0,
					SNDRV_PCM_HW_PARAM_RATE,
					&hw_constraints_rates)) < 0)
		return err;

	return 0;
}

static int snd_card_mpc52xx_aic26_close(snd_pcm_substream_t * substream)
{
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);

	snd_printd("%s(substream=%p)\n", __FUNCTION__, substream);

	mpc52xx_aic26_clock_off(chip);
	chip->s[substream->pstr->stream].stream = NULL;

	return 0;
}

/* HW params & free */

static int snd_mpc52xx_aic26_hw_params(snd_pcm_substream_t * substream,
				       snd_pcm_hw_params_t * hw_params)
{
	struct snd_card_mpc52xx_aic26 *chip = snd_pcm_substream_chip(substream);
	int err;

	snd_printd("%s(substream=%p) p_size=%i p_bytes=%i periods=%i)\n"
	           "buffer_size=%i, buffer_bytes=%i tick_time=%i\n",
	           __FUNCTION__, substream,
		   params_period_size(hw_params),
		   params_period_bytes(hw_params),
		   params_periods(hw_params),
		   params_buffer_size(hw_params),
		   params_buffer_bytes(hw_params),
		   params_tick_time(hw_params));

	/* set requested samplerate */
	if ((err =
	     mpc52xx_aic26_set_samplerate(chip->spi_ss,
					  params_rate(hw_params))) < 0) {
		return err;
	}

	/* set requested sampling format */
	if ((err =
	     mpc52xx_aic26_set_format(chip, chip->spi_ss,
				      params_format(hw_params))) < 0) {
		return err;
	}

	/* set requested channel */
	if ((err =
	     mpc52xx_aic26_set_channel(chip->spi_ss,
				       params_channels(hw_params))) < 0) {
		return err;
	}

	err = snd_pcm_lib_malloc_pages(substream,
				       params_buffer_bytes(hw_params));

	return err;
}

static int snd_mpc52xx_aic26_hw_free(snd_pcm_substream_t * substream)
{
	return snd_pcm_lib_free_pages(substream);
}

/* pcm operations */

static snd_pcm_ops_t snd_card_mpc52xx_aic26_playback_ops = {
	.open = snd_card_mpc52xx_aic26_open,
	.close = snd_card_mpc52xx_aic26_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = snd_mpc52xx_aic26_hw_params,
	.hw_free = snd_mpc52xx_aic26_hw_free,
	.prepare = snd_mpc52xx_aic26_prepare,
	.trigger = snd_mpc52xx_aic26_trigger,
	.pointer = snd_mpc52xx_aic26_pointer,
};

static snd_pcm_ops_t snd_card_mpc52xx_aic26_capture_ops = {
	.open = snd_card_mpc52xx_aic26_open,
	.close = snd_card_mpc52xx_aic26_close,
	.ioctl = snd_pcm_lib_ioctl,
	.hw_params = snd_mpc52xx_aic26_hw_params,
	.hw_free = snd_mpc52xx_aic26_hw_free,
	.prepare = snd_mpc52xx_aic26_prepare,
	.trigger = snd_mpc52xx_aic26_trigger,
	.pointer = snd_mpc52xx_aic26_pointer,
};

/* 
 *  This is called when dma IRQ occurs at the end of each transmited block
 */
static irqreturn_t dma_tx_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_mpc52xx_aic26 *chip = dev_id;
	struct audio_stream *s = &chip->s[SNDRV_PCM_STREAM_PLAYBACK];

	for (;;) {
		sdma_clear_irq(s->sdma);
		if (!sdma_buffer_done(s->sdma)) {
			break;
		}
		sdma_retrieve_buffer(s->sdma, NULL);
		/*
		 * If we are getting a callback for an active stream then we inform
		 * the PCM middle layer we've finished a period
		 */
		sdma_submit_buffer(s->sdma, NULL, s->period_next_pt,
				   s->period_byte_size);
		s->period_next_pt += s->period_byte_size;
		if (s->period_next_pt >= s->period_end) {
			s->period_next_pt = s->period_start;
		}
	}
	if (s->active)
		snd_pcm_period_elapsed(s->stream);

	return IRQ_HANDLED;
}

static irqreturn_t status_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_mpc52xx_aic26 *chip = dev_id;
	struct mpc52xx_psc __iomem *psc;
	psc = PSC(chip);
	printk("ALSA overrun/underrun: %08X\n", in_be16(&psc->mpc52xx_psc_isr));
	out_8(&psc->command, 4 << 4);	/* reset error */

	return IRQ_HANDLED;
}

/* 
 *  This is called when dma IRQ occurs at the end of each received block
 */
static irqreturn_t dma_rx_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_mpc52xx_aic26 *chip = dev_id;
	struct audio_stream *s = &chip->s[SNDRV_PCM_STREAM_CAPTURE];

	for (;;) {
		sdma_clear_irq(s->sdma);
		if (!sdma_buffer_done(s->sdma)) {
			break;
		}
		sdma_retrieve_buffer(s->sdma, NULL);
		/*
		 * If we are getting a callback for an active stream then we inform
		 * the PCM middle layer we've finished a period
		 */
		sdma_submit_buffer(s->sdma, NULL, s->period_next_pt,
				   s->period_byte_size);
		s->period_next_pt += s->period_byte_size;
		if (s->period_next_pt >= s->period_end) {
			s->period_next_pt = s->period_start;
		}
	}
	if (s->active)
		snd_pcm_period_elapsed(s->stream);

	return IRQ_HANDLED;
}

/*
 *  Alsa init and exit section
 *  
 *  Inits pcm alsa structures, allocate the alsa buffer, suspend, resume
 */
static int __devinit snd_card_mpc52xx_aic26_pcm(snd_card_t * card, int device)
{
	struct snd_card_mpc52xx_aic26 *chip =
	    (struct snd_card_mpc52xx_aic26 *)card->private_data;
	snd_pcm_t *pcm;
	int err;
	phys_addr_t rxtx_fifo;

	snd_printd("%s(card=%p, device=%i)\n", __FUNCTION__, card, device);

	/* Init the hardware  */
	if ((err = mpc52xx_aic26_audio_init(chip)) < 0)
		return err;

	/* setup Bestcomm DMA controller */
	/* 2 Buffer Descriptors */
	/* sdma_alloc last one will not use, for queue full detect */
	chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma = sdma_alloc(2 + 1);
	chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma = sdma_alloc(2 + 1);

	if (!chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma
	    || !chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma) {
		err = -ENOMEM;
		goto pcm_error;
	}

	rxtx_fifo = MPC52xx_PA(MPC52xx_PSCx_OFFSET(chip->psc_idx + 1) +
			       FIELD_OFFSET(mpc52xx_psc,
					    mpc52xx_psc_buffer_32));

	err = sdma_gen_bd_rx_init(chip->psc_idx,
				  chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma,
				  rxtx_fifo,
				  sdma_initiator_psc_rx[chip->psc_idx],
				  sdma_ipr_psc_rx[chip->psc_idx],
				  0 /* will set later */ );
	if (err < 0)
		goto pcm_error;
	err = sdma_gen_bd_tx_init(chip->psc_idx,
				  chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma,
				  rxtx_fifo,
				  sdma_initiator_psc_tx[chip->psc_idx],
				  sdma_ipr_psc_tx[chip->psc_idx]);
	if (err < 0)
		goto pcm_error;

	/* Get the IRQ we need one by one */
	if (request_irq(psc_irq[chip->psc_idx], &status_interrupt, SA_INTERRUPT,
			DRIVER_NAME "_status", (void *)chip)) {
		printk(KERN_ERR DRIVER_NAME
		       ": status interrupt request failed\n");
		err = -EBUSY;
		chip->irq = -1;	/* Don't try to free it */
		goto pcm_error;
	}

	chip->s[SNDRV_PCM_STREAM_CAPTURE].irq =
	    sdma_irq(chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma);
	if (request_irq
	    (chip->s[SNDRV_PCM_STREAM_CAPTURE].irq, &dma_rx_interrupt,
	     SA_INTERRUPT, DRIVER_NAME "_rx", (void *)chip)) {
		printk(KERN_ERR DRIVER_NAME ": rx interrupt request failed\n");
		err = -EBUSY;
		/* Don't try to free it */
		chip->s[SNDRV_PCM_STREAM_CAPTURE].irq = -1;
		goto pcm_error;
	}

	chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq =
	    sdma_irq(chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma);
	if (request_irq
	    (chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq, &dma_tx_interrupt,
	     SA_INTERRUPT, DRIVER_NAME "_tx", (void *)chip)) {
		printk(KERN_ERR DRIVER_NAME ": tx interrupt request failed\n");
		err = -EBUSY;
		/* Don't try to free it */
		chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq = -1;
		goto pcm_error;
	}

	/* Create ALSA PCM instance */
	if ((err = snd_pcm_new(card, "AIC26", device, 1, 1, &pcm)) < 0)
		goto pcm_error;

	chip->pcm = pcm;

	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK,
			&snd_card_mpc52xx_aic26_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE,
			&snd_card_mpc52xx_aic26_capture_ops);

	/* sets up initial buffer with continuous allocation */
	snd_pcm_lib_preallocate_pages_for_all(pcm,
					      SNDRV_DMA_TYPE_CONTINUOUS,
					      snd_dma_continuous_data
					      (GFP_KERNEL),
					      128 * 1024, 128 * 1024);

	pcm->private_data = chip;
	pcm->info_flags = 0;
	strcpy(pcm->name, "MPC52XX AIC26 PCM");

	return 0;

      pcm_error:
	if (chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma)
		sdma_free(chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma);
	if (chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma)
		sdma_free(chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma);

	if (chip->irq >= 0)
		free_irq(chip->irq, chip);
	if (chip->s[SNDRV_PCM_STREAM_CAPTURE].irq >= 0)
		free_irq(chip->s[SNDRV_PCM_STREAM_CAPTURE].irq, chip);
	if (chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq >= 0)
		free_irq(chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq, chip);

	return err;
}

#ifdef CONFIG_PM
static int snd_mpc52xx_aic26_suspend(snd_card_t * card, pm_message_t state)
{
	struct snd_card_mpc52xx_aic26 *chip = card->pm_private_data;

	if (chip->card->power_state != SNDRV_CTL_POWER_D3hot) {
		snd_power_change_state(chip->card, SNDRV_CTL_POWER_D3hot);
		snd_pcm_suspend_all(chip->pcm);
		/* Mutes and turn clock off */
		mpc52xx_aic26_clock_off(chip);
	}

	return 0;
}

/*
 *  Prepare hardware for resume
 */
static int snd_mpc52xx_aic26_resume(snd_card_t * card)
{
	struct snd_card_mpc52xx_aic26 *chip = card->pm_private_data;

	if (chip->card->power_state != SNDRV_CTL_POWER_D0) {
		snd_power_change_state(chip->card, SNDRV_CTL_POWER_D0);
		mpc52xx_aic26_clock_on(chip);
	}

	return 0;
}
#endif				/* CONFIG_PM */

void snd_mpc52xx_aic26_free(snd_card_t * card)
{
	struct snd_card_mpc52xx_aic26 *chip = card->private_data;

	if (chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma)
		sdma_free(chip->s[SNDRV_PCM_STREAM_CAPTURE].sdma);
	if (chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma)
		sdma_free(chip->s[SNDRV_PCM_STREAM_PLAYBACK].sdma);

	if (chip->irq >= 0)
		free_irq(chip->irq, chip);
	if (chip->s[SNDRV_PCM_STREAM_CAPTURE].irq >= 0)
		free_irq(chip->s[SNDRV_PCM_STREAM_CAPTURE].irq, chip);
	if (chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq >= 0)
		free_irq(chip->s[SNDRV_PCM_STREAM_PLAYBACK].irq, chip);
}

/*
 * proc interface
 */
static void snd_mpc52xx_aic26_proc_read(struct snd_info_entry *entry,
					struct snd_info_buffer *buffer)
{
	u16 value;
	int page, addr;
	struct snd_card_mpc52xx_aic26 *chip =
	    (struct snd_card_mpc52xx_aic26 *)entry->private_data;

	snd_iprintf(buffer, "Page\tAddr\tRegisters:\n\n");

	for (page = 0; page < 3; page++) {
		for (addr = 0; addr < 0x20; addr++) {
			value = audio_aic26_read(chip->spi_ss,
			                         AIC26_PAGE_ADDR(page, addr));
			snd_iprintf(buffer, "%d\t%02X\t%04X\n",
			            page, addr, value);
		}
	}
}

static void snd_mpc52xx_aic26_proc_write(snd_info_entry_t * entry,
					 snd_info_buffer_t * buffer)
{
	unsigned int page, addr, val;
	char line[64];
	struct snd_card_mpc52xx_aic26 *chip =
	    (struct snd_card_mpc52xx_aic26 *)entry->private_data;

	while (!snd_info_get_line(buffer, line, sizeof(line))) {
		if (sscanf(line, "%x %x %x", &page, &addr, &val) != 3)
			continue;

		if ((page < 3) && (addr < 0x1f) && (val < 0xffff)) {
			audio_aic26_write(chip->spi_ss,
			                  AIC26_PAGE_ADDR(page, addr), val);
		}
	}
}

static void __devinit snd_mpc52xx_aic26_proc_init(snd_card_t * card)
{
	struct snd_info_entry *entry;
	if (!snd_card_proc_new(card, "aic26_regs", &entry)) {
		snd_info_set_text_ops(entry, card->private_data, 1024,
		                      snd_mpc52xx_aic26_proc_read);
		/* add write support for debugging */
		entry->mode |= S_IWUSR;
		entry->c.text.write_size = 64;
		entry->c.text.write = snd_mpc52xx_aic26_proc_write;
	}
}

/* module init & exit */

/* 
 *  Inits alsa soudcard structure
 */
static int snd_mpc52xx_aic26_probe(int dev)
{
	int err = 0;
	snd_card_t *card;
	struct snd_card_mpc52xx_aic26 *chip;

	/* register the soundcard */
	card = snd_card_new(index[dev], id[dev], THIS_MODULE,
	                    sizeof(struct snd_card_mpc52xx_aic26));
	if (card == NULL)
		return -ENOMEM;

	chip = (struct snd_card_mpc52xx_aic26 *)card->private_data;
	chip->psc_idx = psc_idx[dev];
	chip->spi_ss = spi_ss[dev];
	card->private_free = snd_mpc52xx_aic26_free;

	spin_lock_init(&chip->s[0].dma_lock);
	spin_lock_init(&chip->s[1].dma_lock);

	/* Power Management */
#ifdef CONFIG_PM
	snd_card_set_pm_callback(card, snd_mpc52xx_aic26_suspend,
				 snd_mpc52xx_aic26_resume, card);
#endif

	/* mixer */
	if ((err = snd_mpc52xx_aic26_mixer(card)) < 0)
		goto _err;

	/* PCM */
	if ((err = snd_card_mpc52xx_aic26_pcm(card, 0)) < 0)
		goto _err;

	snd_mpc52xx_aic26_proc_init(card);

	snd_mpc52xx_aic26_init_mixer(chip);

	strcpy(card->driver, "AIC26");
	strcpy(card->shortname, "MPC52XX AIC26");
	sprintf(card->longname, "MPC52XX AIC26 using PSC%d", psc_idx[dev] + 1);

	if ((err = snd_card_register(card)) < 0)
		goto _err;

	snd_mpc52xx_aic26[dev] = card;

	return 0;

      _err:
	snd_card_free(card);
	return err;
}

static int mpc52xx_aic26_init(void)
{
	int dev, cards = 0;
	int i;
	static int valid_psc_idxs[] = { 0, 1, 2, 5 };

	for (dev = 0; dev < SNDRV_CARDS; dev++) {
		if (!enable[dev])
			continue;
		/* TODO check for valid spi ss port */
		for (i = 0; i < ARRAY_SIZE(valid_psc_idxs); i++) {
			if (psc_idx[dev] == valid_psc_idxs[i])
				break;
		}
		if (i == ARRAY_SIZE(valid_psc_idxs))
			continue;
		if (!snd_mpc52xx_aic26_probe(dev)) {
			cards++;
			continue;
		}
	}
	if (!cards) {
		snd_printk(KERN_ERR "AIC26 chipset not found\n");
		return -ENODEV;
	}
	return 0;
}

static void __exit mpc52xx_aic26_exit(void)
{
	int dev;
	struct snd_card_mpc52xx_aic26 *chip;

	for (dev = 0; dev < SNDRV_CARDS; dev++) {
		if (snd_mpc52xx_aic26[dev] != NULL) {
			chip = (struct snd_card_mpc52xx_aic26 *)
			    snd_mpc52xx_aic26[dev]->private_data;
			iounmap(chip->membase);
			release_mem_region(chip->mapbase, MPC52xx_PSC_SIZE);
			snd_omap_exit_mixer(chip);
			snd_card_free(snd_mpc52xx_aic26[dev]);
		}
	}
}

module_init(mpc52xx_aic26_init);
module_exit(mpc52xx_aic26_exit);
