/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _DWAPBSSI_H_
#define _DWAPBSSI_H_

#include <linux/spi/spi.h> /* for struct spi_board_info */

/*-------------------------------------------------------------------------*/

/* General */

/* 
 * AMBA component ID - This value is arbitrarily set,
 * we leave the low order byte for device ID,
 * and the high order byte for designer ID.
 */
#define DWAPBSSI_ID				0x00198600
#define DWAPBSSI_ID_MASK			0x00ffff00

#define DWAPBSSI_MAX_FREQUENCY_NODMA		10000
#define DWAPBSSI_MAX_FREQUENCY_NODMA_SPIN	1000000
#define DWAPBSSI_MAX_FREQUENCY_1DMA		20000

struct dwapbssi_bus_data {
	const char 		*clock_id;
	char 			use_tx_dma, use_rx_dma;
	struct spi_board_info 	*devices;
	uint16_t 			num_devices;
	uint16_t 			num_chipselect;
	
#define DWAPBSSI_SLAVE_TIMEOUT_INFINITE 	~0
#define DWAPBSSI_DEFAULT_SLAVE_TIMEOUT_MSECS 	3000
	/* Zero to set default value */
	unsigned long 		slave_timeout;
};

/*-------------------------------------------------------------------------*/

/* Driver API */

/**
 * Use this function to modify bus configuration at runtime.
 * Set force_speed to 0 not to force speed and leave per-device
 * configuration / per-spi-transfer configuration if any.
 * Set transfer_timeout to 0 not to modify its current value.
 * Set use_tx_dma/use_rx_dma to -1 not to modify current settings.
 * @return -ENODEV if bus_id did not match any AMBA bus device name.
 * @return -EINVAL if bad input is provided.
 * @return 0 upon success.
 */
int dwapbssi_configure(const char *bus_id, unsigned long force_speed,
	unsigned long slave_timeout);

/**
 * Use this function to modify SPI device configuration at runtime.
 * chip_select is the chipselect for the device you want to configure.
 * Set speed_hz to 0 not to change speed.
 * Set modalias to NULL not to modify the modalias.
 * Set mode/use_tx_dma/use_rx_dma to -1 not to modify current settings.
 * @return -ENODEV if bus_id did not match any AMBA bus device name.
 * @return -EINVAL if bad input is provided.
 * @return 0 upon success.
 */
int dwapbssi_configure_device(const char *bus_id, u8 chip_select, 
	unsigned long speed_hz, const char *modalias, int mode, 
	int use_tx_dma, int use_rx_dma);

/**
 * This aborts current transfer, if any, and disable the driver 
 * on request, meaning that it will not accept any new SPI message.
 * @return -ENODEV if bus_id did not match any AMBA bus device name.
 * @return -EINVAL if bad input is provided.
 * @return 0 upon success.
 */
int dwapbssi_abort_transfer(const char *bus_id, int disable);

/**
 * Reenables the bus / clears error state.
 * @return -ENODEV if bus_id did not match any AMBA bus device name.
 * @return -EINVAL if bad input is provided.
 * @return 0 upon success.
 */
int dwapbssi_reenable(const char *bus_id);

/**
 * Helper function to find a spi_device given the bus_id and the chipselect.
 * @return found spi_device or NULL if not found or if error occurred.
 */
struct spi_device *dwapbssi_find_device(const char *bus_id, u8 chip_select);

/*-------------------------------------------------------------------------*/

/* Component Registers */

#define DWAPBSSI_IC_CTRLR0			0x00
#define DWAPBSSI_IC_CTRLR1			0x04
#define DWAPBSSI_IC_SSIENR			0x08
#define DWAPBSSI_IC_MWCR			0x0c
#define DWAPBSSI_IC_SER				0x10
#define DWAPBSSI_IC_BAUDR			0x14
#define DWAPBSSI_IC_TXFTLR			0x18
#define DWAPBSSI_IC_RXFTLR			0x1c
#define DWAPBSSI_IC_TXFLR			0x20
#define DWAPBSSI_IC_RXFLR			0x24
#define DWAPBSSI_IC_SR				0x28
#define DWAPBSSI_IC_IMR				0x2c
#define DWAPBSSI_IC_ISR				0x30
#define DWAPBSSI_IC_RISR			0x34
#define DWAPBSSI_IC_TXOICR			0x38
#define DWAPBSSI_IC_RXOICR			0x3c
#define DWAPBSSI_IC_RXUICR			0x40
#define DWAPBSSI_IC_MSTICR			0x44
#define DWAPBSSI_IC_ICR				0x48
#define DWAPBSSI_IC_DMACR			0x4c
#define DWAPBSSI_IC_DMATDLR			0x50
#define DWAPBSSI_IC_DMARDLR			0x54
#define DWAPBSSI_IC_IDR				0x58
#define DWAPBSSI_IC_SSI_COMP_VERSION		0x5c
#define DWAPBSSI_IC_DR				0x60

/* Registers values & masks */

#define DWAPBSSI_SR_BUSY 			0x01
#define DWAPBSSI_SR_TXFIFO_NOTFULL 		0x02
#define DWAPBSSI_SR_RXFIFO_NOTEMPTY		0x08
#define DWAPBSSI_SR_RXFIFO_FULL			0x10

#define DWAPBSSI_IR_RXFIFO_FULL 		0x10
#define DWAPBSSI_IR_RXFIFO_OVERFLOW		0x08
#define DWAPBSSI_IR_RXFIFO_UNDERFLOW		0x04
#define DWAPBSSI_IR_TXFIFO_OVERFLOW		0x02
#define DWAPBSSI_IR_TXFIFO_EMPTY 		0x01

#define DWAPBSSI_BAUDRATE_MIN 			2
#define DWAPBSSI_BAUDRATE_MAX 			65534

#define DWAPBSSI_DMACR_TDMAE			0x2
#define DWAPBSSI_DMACR_RDMAE 			0x1

#define DWAPBSSI_TXFIFO_DEPTH			8

/*-------------------------------------------------------------------------*/

#endif /* #ifndef _DWAPBSSI_H_ */
