/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _DWAPBI2C_H_
#define _DWAPBI2C_H_

/*-------------------------------------------------------------------------*/

/* General */

/* 
 * AMBA component ID - This value is arbitrarily set,
 * we leave the low order byte for device ID,
 * and the high order byte for designer ID.
 */
#define DWAPBI2C_ID				0x00198500
#define DWAPBI2C_ID_MASK			0x00ffff00

/*
 * This is used to identify the adapter in the I2C subsystem.
 * XXX: Move to i2c-id.h
 */
#define I2C_HW_DWAPBI2C 			DWAPBI2C_ID

struct dwapbi2c_bus_data {
	const char 	*clock_id;
	
	char 		use_dma;
	
	/* one of the DWAPBI2C_CON_SPEED_... define, see below. */
	char		i2c_speed;
	
	char		is_slave;
	
#define DWAPBI2C_DEFAULT_SLAVE_ADDR_BASE	0xaa
	/*
	 * Can be zero, in this case DWAPBI2C_DEFAULT_SLAVE_ADDR_BASE
	 * will be used to generate a slave address.
	 */
	u16		slave_addr;

#define DWAPBI2C_TRANSFER_TIMEOUT_INFINITE 	~0
#define DWAPBI2C_DEFAULT_TRANSFER_TIMEOUT_MSECS 3000
	/* Zero to set default value */
	unsigned long 	transfer_timeout;
};

/*-------------------------------------------------------------------------*/

/* Driver API */

/**
 * Use this function to configure the driver at runtime for a particular bus
 * (1 bus == 1 adapter).
 * NOTE: If i2c_speed or transfer_timeout or sda_setup_delay equal 0, 
 * their respective current value will not be modified.
 * If slave_addr is an i2c reserved address (e.g. '1'), 
 * the current slave address will not be modified. If slave_addr   
 * equals 0, the driver will pick a slave address for you.
 * If use_dma < 0, setting will not be changed.
 * @return 0 upon success.
 * @return -ENODEV if bus_id did not match any AMBA bus device name.
 * @return -EINVAL if bad input is provided.
 */

int dwapbi2c_configure(const char *bus_id, int i2c_speed, int is_slave,
	u16 slave_addr, unsigned long transfer_timeout, int use_dma,
	u8 sda_setup_delay);

/*-------------------------------------------------------------------------*/

/* Component registers */

#define DWAPBI2C_IC_CON				0x00
#define DWAPBI2C_IC_TAR				0x04
#define DWAPBI2C_IC_SAR				0x08
#define DWAPBI2C_IC_HS_MADDR			0x0c
#define DWAPBI2C_IC_DATA_CMD			0x10
#define DWAPBI2C_IC_SS_SCL_HCNT			0x14
#define DWAPBI2C_IC_SS_SCL_LCNT			0x18
#define DWAPBI2C_IC_FS_SCL_HCNT			0x1c
#define DWAPBI2C_IC_FS_SCL_LCNT			0x20
#define DWAPBI2C_IC_HS_SCL_HCNT			0x24
#define DWAPBI2C_IC_HS_SCL_LCNT			0x28
#define DWAPBI2C_IC_INTR_STAT			0x2c
#define DWAPBI2C_IC_INTR_MASK			0x30
#define DWAPBI2C_IC_RAW_INTR_STAT		0x34
#define DWAPBI2C_IC_RX_TL			0x38
#define DWAPBI2C_IC_TX_TL			0x3c
#define DWAPBI2C_IC_CLR_INTR			0x40
#define DWAPBI2C_IC_CLR_RX_UNDER		0x44
#define DWAPBI2C_IC_CLR_RX_OVER			0x48
#define DWAPBI2C_IC_CLR_TX_OVER			0x4c
#define DWAPBI2C_IC_CLR_RD_REQ			0x50
#define DWAPBI2C_IC_CLR_TX_ABRT			0x54
#define DWAPBI2C_IC_CLR_RX_DONE			0x58
#define DWAPBI2C_IC_CLR_ACTIVITY		0x5c
#define DWAPBI2C_IC_CLR_STOP_DET		0x60
#define DWAPBI2C_IC_CLR_START_DET		0x64
#define DWAPBI2C_IC_CLR_GEN_CALL		0x68
#define DWAPBI2C_IC_ENABLE			0x6c
#define DWAPBI2C_IC_STATUS			0x70
#define DWAPBI2C_IC_TXFLR			0x74
#define DWAPBI2C_IC_RXFLR			0x78
#define DWAPBI2C_IC_TX_ABRT_SOURCE		0x80
#define DWAPBI2C_IC_DMA_CR			0x88
#define DWAPBI2C_IC_DMA_TDLR			0x8c
#define DWAPBI2C_IC_DMA_RDLR			0x90
#define DWAPBI2C_IC_SDA_SETUP			0x94
#define DWAPBI2C_IC_COMP_PARAM_1		0xf4
#define DWAPBI2C_IC_COMP_VERSION		0xf8
#define DWAPBI2C_IC_COMP_TYPE			0xfc

/*-------------------------------------------------------------------------*/

/* Registers values and masks */

	/* DWAPBI2C_IC_CON register */

#define DWAPBI2C_CON_MASTER_MODE_EN_MASK	0x01
#define DWAPBI2C_CON_MASTER_MODE_EN_LSB		0
#define DWAPBI2C_CON_MASTER_MODE_ENABLED	0x01

#define DWAPBI2C_CON_SPEED_MASK			0x06
#define DWAPBI2C_CON_SPEED_LSB			1
#define DWAPBI2C_CON_SPEED_LOW			0x01
#define DWAPBI2C_CON_SPEED_STANDARD		0x02
#define DWAPBI2C_CON_SPEED_FAST			0x04
#define DWAPBI2C_CON_SPEED_HIGH			0x06

#define DWAPBI2C_CON_SADDRESSING_MASK		0x08
#define DWAPBI2C_CON_SADDRESSING_LSB		3
#define DWAPBI2C_CON_SADDRESSING_7BIT		0x00
#define DWAPBI2C_CON_SADDRESSING_10BIT		0x08

#define DWAPBI2C_CON_MADDRESSING_MASK		0x10
#define DWAPBI2C_CON_MADDRESSING_LSB		4
#define DWAPBI2C_CON_MADDRESSING_7BIT		0x00
#define DWAPBI2C_CON_MADDRESSING_10BIT		0x10

#define DWAPBI2C_CON_RESTART_EN_MASK		0x20
#define DWAPBI2C_CON_RESTART_EN_LSB		5
#define DWAPBI2C_CON_RESTART_ENABLED		0x20

#define DWAPBI2C_CON_SLAVE_MODE_DIS_MASK	0x40
#define DWAPBI2C_CON_SLAVE_MODE_DIS_LSB		6
#define DWAPBI2C_CON_SLAVE_MODE_DISABLED	0x40

	/* DWAPBI2C_IC_TAR register */
	
#define DWAPBI2C_TAR_ADDRESS_MASK		0x03ff
#define DWAPBI2C_TAR_ADDRESS_LSB		0

#define DWAPBI2C_TAR_GCORTART_MASK		0x0400
#define DWAPBI2C_TAR_GCORTART_LSB		10
#define DWAPBI2C_TAR_GCORTART_GCALLADDR	0x0000
#define DWAPBI2C_TAR_GCORTART_STARTBYTE	0x0400

#define DWAPBI2C_TAR_SPECIAL_MASK		0x0800
#define DWAPBI2C_TAR_SPECIAL_LSB		11
#define DWAPBI2C_TAR_SPECIAL_ENABLED		0x0800

#define DWAPBI2C_TAR_MADDRESSING_MASK		0x1000
#define DWAPBI2C_TAR_MADDRESSING_LSB		12
#define DWAPBI2C_TAR_MADDRESSING_7BIT		0x0000
#define DWAPBI2C_TAR_MADDRESSING_10BIT		0x1000
	
	/* DWAPBI2C_IC_SAR registers */

#define DWAPBI2C_SAR_ADDRESS_MASK		0x03ff
#define DWAPBI2C_SAR_ADDRESS_LSB		0

	/* DWAPBI2C_IC_HS_MADDR registers */

#define DWAPBI2C_HS_MASTERCODE_MASK		0x7
#define DWAPBI2C_HS_MASTERCODE_LSB		0

	/* DWAPBI2C_IC_DATA_CMD register */

#define DWAPBI2C_DATA_CMD_DATA_MASK		0x0ff
#define DWAPBI2C_DATA_CMD_DATA_LSB		0

#define DWAPBI2C_DATA_CMD_CMD_MASK		0x100
#define DWAPBI2C_DATA_CMD_CMD_LSB		8
#define DWAPBI2C_DATA_CMD_CMD_READ		0x100
#define DWAPBI2C_DATA_CMD_CMD_WRITE		0x000

	/* DWAPBI2C_IC_STATUS register */

#define DWAPBI2C_STATUS_ACTIVITY		0x01
#define DWAPBI2C_STATUS_TFNF			0x02
#define DWAPBI2C_STATUS_TFE			0x04
#define DWAPBI2C_STATUS_RFNE			0x08
#define DWAPBI2C_STATUS_RFF			0x10
#define DWAPBI2C_STATUS_MST_ACTIVITY		0x20
#define DWAPBI2C_STATUS_SLV_ACTIVITY		0x40

	/* DWAPBI2C_IC_COMP_PARAM_1 register */

#define DWAPBI2C_COMP_PARAM_1_TXDEPTH_MASK	0xff0000
#define DWAPBI2C_COMP_PARAM_1_TXDEPTH_LSB	16
#define DWAPBI2C_COMP_PARAM_1_RXDEPTH_MASK	0x00ff00
#define DWAPBI2C_COMP_PARAM_1_RXDEPTH_LSB	8

	/* DWAPBI2C_IC_xS_SCL_xCNT registers */

#define DWAPBI2C_IC_SS_SCL_HCNT_MIN		6
#define DWAPBI2C_IC_SS_SCL_LCNT_MIN		8
#define DWAPBI2C_IC_FS_SCL_HCNT_MIN		6
#define DWAPBI2C_IC_FS_SCL_LCNT_MIN		8
#define DWAPBI2C_IC_HS_SCL_HCNT_MIN		6
#define DWAPBI2C_IC_HS_SCL_LCNT_MIN		8

#define DWAPBI2C_SS_SCL_HIGH_MIN_NS		4000
#define DWAPBI2C_SS_SCL_LOW_MIN_NS		4700
#define DWAPBI2C_FS_SCL_HIGH_MIN_NS		600
#define DWAPBI2C_FS_SCL_LOW_MIN_NS		1300
#define DWAPBI2C_HS_SCL_HIGH_MIN_NS		60
#define DWAPBI2C_HS_SCL_LOW_MIN_NS		160

	/* DWAPBI2C_IC_INTR_MASK, DWAPBI2C_IC_(RAW_)INTR_STAT register */

#define DWAPBI2C_INTR_GENCALL			0x800
#define DWAPBI2C_INTR_START_DET			0x400
#define DWAPBI2C_INTR_STOP_DET			0x200
#define DWAPBI2C_INTR_ACTIVITY			0x100
#define DWAPBI2C_INTR_RX_DONE			0x080
#define DWAPBI2C_INTR_TX_ABRT			0x040
#define DWAPBI2C_INTR_RD_REQ			0x020
#define DWAPBI2C_INTR_TX_EMPTY			0x010
#define DWAPBI2C_INTR_TX_OVER			0x008
#define DWAPBI2C_INTR_RX_FULL			0x004
#define DWAPBI2C_INTR_RX_OVER			0x002
#define DWAPBI2C_INTR_RX_UNDER			0x001
	
	/* Misc. */
	
#define DWAPBI2C_DMA_CR_TDMAE			0x2
#define DWAPBI2C_DMA_CR_RDMAE 			0x1

#define DWAPBI2C_TXABRT_SLVFLUSH		0x2000

/*-------------------------------------------------------------------------*/

#endif /* _DWAPBI2C_H_ */
