/*
 *  drivers/char/watchdog/mobi_wdt.c
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * \file
 *  Driver for Synopsys DW Watchdog
 *
 * 	This driver actually implements two watchdogs, hardware and
 * software.  The hardware watchdog is used to monitor the system
 * will reset the system if it stops responding or there is a problem
 * with the softwar watchdog.
 * 	The software watchdog is something that userspace applications
 * can use to handle application hangs.  It works like this
 * 	- open /dev/watchdog
 * 	- use the ioctl interface to set the timeout
 * 		- both second and millisecond resolution available
 * 	- set the options
 * 		- start/enable watchdog with WDIOS_ENABLECARD
 * 		- define timeout behaviour
 * 			- WDIOS_SWWDT_KILLPROC will kill the process
 * 			on timeout via SIGTERM note, the death of the
 * 			process is not verified
 * 			- WDIOS_SWWDT_REBOOT will reboot the system on timeout.
 * 		- the default is WDIOS_SWWDT_KILLPROC
 * 	- do some work
 * 	- "pat the dog" via the ioctl interface.  The application
 * 	   MUST do this to keep the watchdog from triggering.
 * 	- close /dev/watchdog
 *
 */
#ifndef DOXYGEN_SKIP

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/watchdog.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/mutex.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/amba/bus.h>

#include <linux/bitops.h>
#include <linux/uaccess.h>
#include <linux/io.h>

#include <linux/dw_watchdog_device.h>
#include <mach/platform.h>
#include <mach/dw_wdt_regs.h>
#include <mach/mobi_reset.h>
#include <mach/mobi_qcc.h>

#define PFX "dw_wdt "
/* in seconds */
static int hw_default_heartbeat;
/* value for TOP in TORR register */
static int hw_heartbeat_top = 14;

module_param(hw_default_heartbeat, int, 0);
MODULE_PARM_DESC(hw_default_heartbeat,
		"HW Watchdog time in seconds. (default="
		__MODULE_STRING(WATCHDOG_DEFAULT_HEARTBEAT) ")");

uint32_t wdt_timeouts[16];
void __iomem *watchdog_base;

struct wdt_timeval {
	int msec;
	int sec;
};

struct softdog {
	struct list_head 	node;
	pid_t pid;
	int timeout_cnt;
	struct wdt_timeval 	tv;
	struct timer_list	timer;
	uint32_t options;
};

#undef DW_WDT_DEBUG
#ifdef DW_WDT_DEBUG

/* on watchdog timeout reset, can verify reset by looking at
 *  this from the mips:
 *  echo "R S 68" > /proc/drivers/sbus/cs1
 *  cat /proc/drivers/sbus/cs1
 *  should see 0x3
 *  echo "R S 68" > /proc/drivers/sbus/cs1
 *  cat /proc/drivers/sbus/cs1
 *  should see 0x400
 */
static int8_t no_intr_reply;
static int8_t no_intr_reply_count = 3;

#define dprintk(x...)	printk(x)
#else
#define dprintk(x...)	do {} while (0)
#endif

#define WRITEL(data, offset) \
	writel((uint32_t) data, watchdog_base+(uint32_t)offset)

#define READL(offset) \
	readl(watchdog_base+(uint32_t)offset)

static LIST_HEAD(softdogs);
static DEFINE_MUTEX(softdogs_mutex);

static struct wdt_timeval hw_heartbeat;
static int softdog_reset;

#define WATCHDOG_PROC_DIR "driver/watchdog"
static struct proc_dir_entry *watchdog_proc_dir;
static int wdt_expires_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data);
static int wdt_expires_proc_wr(struct file *file,
		const char *buffer, unsigned long count, void *data);
static int wdt_status_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data);

static int mobi_hw_wdt_pat_the_dog(void);

#endif /* DOXYGEN_SKIP */

/**
 *  \brief mobi_hw_wdt_start
 *  	Start the hardware watchdog.
 *
 *  \param void - none
 *
 *  \retval Zero upon success.
 *
 *  \remark
 *    This involves setting the enable in the watchdog hw
 *  block but also enabling the hw reset in the reset controller.
 *  This is an internal fuction.
 */
static int mobi_hw_wdt_start(void)
{
	uint32_t reg_shadow;
	int32_t ret;

#ifdef CONFIG_ARCH_MERLIN
	ret = mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_WDOG_RESET_CONTROL, 
			0x1, 4);
#endif
#ifdef CONFIG_ARCH_FALCON
	ret = mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_WATCHDOG_OFFSET, 
			0x1, 4);
#endif
	if (ret >= 0) {
		reg_shadow = READL(MOBI_WDT_CR_OFFSET);
		reg_shadow |= (MOBI_WDT_CR_RMOD_MASK|MOBI_WDT_CR_WDT_EN_MASK);
		WRITEL(reg_shadow, MOBI_WDT_CR_OFFSET);
		dprintk("control_reg = 0x%x\n", reg_shadow);
		mobi_hw_wdt_pat_the_dog();
	}

	return ret;
}

/**
 * \brief mobi_hw_wdt_stop
 *    Disable the hw watchdog in the reset controller.
 *
 * \param void - none
 *
 * \reval Zero upon success.
 *
 * \remark
 *    According to the synopsys doc the wdt CANNOT be disabled in the
 *  watchdog block once it has been started except by system
 *  reset.  But disabling the watchdog in the reset controller can
 *  prevent a system reset.
 */
static int mobi_hw_wdt_stop(void)
{

	/* Note.  Not going to stop the software watchdogs even if the
	 *  hardware one is disabled.  The sw watchdogs will continue to
	 *  work but if an app hangs and can't be killed the system won't
	 *  be reset.  It seems that it would be very rare to ever be running
	 *  like this.
	 */
	dprintk("%s: disable system reset\n", __func__);
	printk(KERN_NOTICE "Watchdog timer disabled\n");
	WRITEL(0, MOBI_WDT_CR_OFFSET); /* turn off interrupts */
#ifdef CONFIG_ARCH_MERLIN
	return mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_WDOG_RESET_CONTROL, 
			0x0, 4);
#endif
#ifdef CONFIG_ARCH_FALCON
	return mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_WATCHDOG_OFFSET, 
			0x0, 4);
#endif
}

static int mobi_hw_wdt_pat_the_dog(void)
{
	/*
	 *  a write to this register restarts the WDT counter.  As
	 *  as safety measure, the value 0x76 must be written.  A
	 *  restart also clears the WDT interrupt.  This will only be
	 *  used to get things started/restarted; otherwise, reading
	 *  the interrupt status register will reset the timer
	 */
	dprintk("patting the dog\n");
	WRITEL(0x76, MOBI_WDT_CCR_OFFSET);

	return 0;
}

/**
 *  \brief mobi_sw_wdt_stop:
 *  	Set the timeout to zero for a sw watchdog.
 *
 *  \param void - none
 *
 *  \retval Zero - upon success.
 *
 *  \remark
 *    Set the timeout to zero for the pid in the softdogs list
 *  won't delete the timer until the app closes /dev/watchdog
 *
 */
static int mobi_sw_wdt_stop(void)
{
	struct softdog *sd = NULL;

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == current->pid)
			break;
	}
	if (sd != NULL)
		mod_timer(&sd->timer, 0);

	mutex_unlock(&softdogs_mutex);
	return 0;
}

/**
 * \brief mobi_sw_wdt_set_heartbeat
 * 	Initialize a wdt_timeval structure for SW watchdog
 *
 * \param t - struct wdt_timeval which contains user set timeout
 *
 * \retval -EINVAL  - if timeout is to large
 * \retval Zero     - upon success
 *
 * \remark
 * 	User passes in timeout via ioctl interface, this function
 * takes care of initializing internal structures.
 *
 */
static int mobi_sw_wdt_set_heartbeat(struct wdt_timeval t)
{
	struct softdog *sd = NULL;

	if ((t.sec < 0x0001) || (t.sec > 0xFFFF))
		return -EINVAL;

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd != NULL && sd->pid == current->pid) {
			dprintk("added new timeout for pid %d of %dms, %ds\n",
					sd->pid, t.msec, t.sec);
			sd->tv.msec = t.msec;
			sd->tv.sec = t.sec;
			break;

		} else {
			printk(KERN_ERR PFX
					"Unable to locate software "
					"watchdog for pid %d\n",
					current->pid);
			mutex_unlock(&softdogs_mutex);
			return -ENODEV;
		}
	}

	mutex_unlock(&softdogs_mutex);

	return 0;
}

/**
 * \brief mobi_hw_wdt_set_heartbeat:
 * 	Sets the timeout for the HW watchdog timer
 *
 * \param t - wdt_timeval with timeout value set
 *
 * \retval -EINVAL - if requested timout is too large.
 *
 * \remark
 * 	The hardware watchdog is a bit convoluted.  You can't
 * define a numeric timeout, you program a register with a value
 * that tells the watchdog what to count down from.  This function
 * is called 2 way.  During bootup where the timeout is a module
 * param that can be changed or via the /proc filesystem.  Note
 * that a timeval of 0 will disable the hardware watchdog.
 *
 */
static int mobi_hw_wdt_set_heartbeat(struct wdt_timeval t)
{
	uint32_t shadow_reg = 0;
	int requested_time;
	int i;

	dprintk("%s: msec = %d, sec = %d\n", __func__, t.msec, t.sec);
	if (t.msec != 0) {
		if ((t.msec < 2) || (t.msec >= 1000))
			return -EINVAL;
		requested_time = t.msec;

	} else if (t.sec != 0) {
		if (t.sec < 1)
			return -EINVAL;
		/* timeouts will be stored in array as ms, so
		 *  turn seconds into mseconds */
		requested_time = t.sec*1000;

	} else { /* the are both 0, disable the hw watchdog */
		mobi_hw_wdt_stop();
		return 0;
	}

	/* the timeout period for wdt is a 4 bit register
	 *  this the 16 here
	 */
	dprintk("requested time = %d\n", requested_time);
	for (i = 0; i < 16; i++) {
		if (wdt_timeouts[i] == requested_time)
			break;
		else if (wdt_timeouts[i] > requested_time) {
			i--;
			break;
		}
	}

	dprintk("timeout = %d\n", i);
	/* value for top 0 will load 2^16, 15 will load 2^31 */
	if (t.msec != 0) {
		hw_heartbeat_top = i;
		hw_heartbeat.msec = wdt_timeouts[i];
		hw_heartbeat.sec = 0;
		dprintk("%s: timeout will be %dms\n",
				__func__, hw_heartbeat.msec);

	} else {
		/* secs start at '9' in top register */
		hw_heartbeat_top = i;
		hw_heartbeat.msec = 0;
		hw_heartbeat.sec = wdt_timeouts[i];
		dprintk("%s: timeout will be %ds\n",
				__func__, hw_heartbeat.sec);
	}

	dprintk("hw_heartbeat_top = %d\n", hw_heartbeat_top);
	shadow_reg = (MOBI_WDT_TORR_TOP_INIT_W(hw_heartbeat_top)
			| MOBI_WDT_TORR_TOP_W(hw_heartbeat_top));
	WRITEL(shadow_reg, MOBI_WDT_TORR_OFFSET);

	return 0;
}

/**
 * \brief dw_wdt_interrupt:
 * 	Handles watchdog interrupt
 *
 * \remark
 * 	An interrupt is received when the watchdog timer expires.
 * This handler will reset the watchdog timer except in the case where
 * a software watchdog timer has expired and we are unable to kill the
 * process.  In this case the interrupt is ignored and a system reset
 * will occur.
 *
 */
static irqreturn_t dw_wdt_interrupt(int irq, void *dev_id)
{
	uint32_t status = READL(MOBI_WDT_STAT_OFFSET);

#ifdef DW_WDT_DEBUG
	if (no_intr_reply) {
		if (no_intr_reply == no_intr_reply_count) {
			printk("not acking intr, expect reset\n");
			return IRQ_HANDLED;

		} else {
			no_intr_reply++;
		}
	}
#endif

	if (status & 0x1) {
		/*
		 * if none of the soft watchdogs have timed out and
		 *  been unable to kill the app, then ack the interrupt
		 *  which will reload the timer, otherwise, don't ack
		 */
		/* dprintk("wdt interrupt... "); */
		if (softdog_reset != 1) {
			/* dprintk("cleared\n");*/
			READL(MOBI_WDT_EOI_OFFSET);

		} else {
			/* dprintk("softreset enabled, prepare for reset\n");*/
		}
	}
	return IRQ_HANDLED;
}

/**
 * \brief softdog_fire
 * 	Called when software watchdog timer expires.
 *
 * \param data - data will contain the pid of the process which
 * 		 has timed out.
 *
 * \remark
 * 	Function will search list of software watchdogs looking
 * for the associated pid.  It will then attempt to kill the
 * process.  If there is an error killing the procces a flag
 * will be set which will result in the system being reset.
 *
 */
static void softdog_fire(unsigned long data)
{
	struct softdog *sd = NULL;
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == data)
			break;
	}
	if (sd == NULL) {
		dprintk("No entry found for pid %lu\n", data);
		return;
	}

	if (sd->options & WDIOS_SWWDT_REBOOT) {
		dprintk("%s: pid %d did not pat the dog, will reset system\n",
				__func__, sd->pid);
		softdog_reset = 1;

	} else {
		dprintk("%s: attempting to kill pid %d\n", __func__, sd->pid);
		/* return < 0 if proc is already dead */
		kill_pid(find_vpid(sd->pid), SIGTERM, 1);
	}
}

/**
 * \brief mobi_sw_wdt_keepalive:
 *	Pat the dog, for a software watchdog.
 *
 * \params void - none
 *
 * \retval Zero - upon success
 *
 * \remark
 * 	Find the entry for the calling pid in the softdogs list and
 *  modify the timer(bump time to future time).  This function is
 *  called by the application via the ioctl interface.  It MUST be
 *  called by an application to let us know the app is alive,
 *  otherwise the watchdog timer for the process will expired and
 *  the process will be killed
 */
static int mobi_sw_wdt_keepalive(void)
{
	struct softdog *sd = NULL;
	int timeout;

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == current->pid)
			break;
	}
	mutex_unlock(&softdogs_mutex);
	/* not sure what the value of HZ is and if it's not big enough
	 *  will keep getting a timeout of 0.  I think jiffies are supposed
	 *  to be roughly 1ms. try using this function to convert ms to jiffies,
	 *  hopfully this will take care of any weirdness for me
	 */
	if (sd != NULL) {
		if (sd->tv.msec != 0)
			timeout = msecs_to_jiffies(sd->tv.msec);
		else
			timeout = (sd->tv.sec*HZ);
		mod_timer(&sd->timer, jiffies+timeout);
	}
	return 0;
}

/**
 * \brief mobi_sw_wdt_gettime:
 * 	Return the current timeout for a watchdog
 *
 * \params void - none
 *
 * \retval time - in seconds or milliseconds
 * \retval EINVAL - no software watchdog found
 *
 * \remark
 * 	Called from usrspace via ioctl to query the
 * current timeout setting.
 *
 */
static int mobi_sw_wdt_gettime(void)
{
	struct softdog *sd = NULL;

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == current->pid)
			break;
	}
	mutex_unlock(&softdogs_mutex);
	if (sd == NULL)
		return -EINVAL;
	else
		return (sd->tv.msec == 0 ? sd->tv.sec : sd->tv.msec);
}

/**
 * \brief mobi_sw_wdt_open
 * 	Opens /dev/watchdog and initialize SW watchdog
 *
 * \param inode - pointer to inode struct
 * \param file  - pointer to file struct
 *
 * \retval int
 *
 * \remark
 * 	This is the open fop for /dev/watchdog.  It does not start the
 * timer, just initializes it.  Also adds an entry in the softdogs list
 * for the calling pid
 */
static int mobi_sw_wdt_open(struct inode *inode, struct file *file)
{
	struct softdog *newsd = NULL;

	newsd = (struct softdog *) kmalloc(sizeof(struct softdog), GFP_KERNEL);
	if (!newsd)
		return -ENOMEM;

	newsd->pid = current->pid;
	newsd->tv.msec = 0;
	newsd->tv.sec = 0;
	newsd->timeout_cnt = 0;

	setup_timer(&newsd->timer, softdog_fire, current->pid);
	newsd->timer.expires = 0;

	mutex_lock(&softdogs_mutex);
	list_add(&newsd->node, &softdogs);
	mutex_unlock(&softdogs_mutex);

	return nonseekable_open(inode, file);
}

/**
 * \brief mobi_sw_wdt_close
 * 	Close /dev/watchdog.
 *
 * \param inode - pointer to inode struct
 * \param file  - pointer to file struct
 *
 * \retval int
 *
 * \remark
 * 	This is the close fop for /dev/watchdog.  It stops the
 * timer, deletes it and then removed the entry from the software
 * watchdog list.
 */
static int mobi_sw_wdt_close(struct inode *inode, struct file *file)
{
	/* remove the pid from the list */
	struct softdog *sd = NULL;

	mobi_sw_wdt_stop();

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == current->pid)
			break;
	}
	if (sd != NULL) {
		del_timer(&sd->timer);
		list_del(&sd->node);
		kfree(sd);
	}
	mutex_unlock(&softdogs_mutex);

	return 0;
}

static struct watchdog_info dw_wdt_info = {
	.options		= WDIOF_KEEPALIVEPING | WDIOF_SETTIMEOUT,
	.firmware_version	= 1,
	.identity		= "DW WDT",
};

static void sw_wdt_setoptions(uint32_t options)
{
	struct softdog *sd = NULL;

	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->pid == current->pid)
			break;
	}
	if (sd != NULL)
		sd->options = options;
	mutex_unlock(&softdogs_mutex);

	return;
}


/*
 * Query basic information from the device or ping it, as outlined by the
 * watchdog API.
 */
static int mobi_sw_wdt_ioctl(struct inode *inode, struct file *file,
		unsigned int cmd, unsigned long arg)
{
	int new_heartbeat;
	struct wdt_timeval new_soft_heartbeat;
	int options, retval = -EINVAL;

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		return copy_to_user((struct watchdog_info *)arg,
				&dw_wdt_info,
				sizeof(dw_wdt_info)) ? -EFAULT : 0;
	case WDIOC_GETSTATUS:
	case WDIOC_GETBOOTSTATUS:
		return put_user(0, (int *)arg);
	case WDIOC_KEEPALIVE:
		dprintk("ioctl: keepalive\n");
		mobi_sw_wdt_keepalive();
		return 0;
	case WDIOC_SETTIMEOUT:
		if (get_user(new_heartbeat, (int *)arg))
			return -EFAULT;
		new_soft_heartbeat.msec = 0;
		new_soft_heartbeat.sec = new_heartbeat;

		dprintk("settimout ioctl will set timeout to %d\n",
				new_heartbeat);
		/* save heartbeat for this pid */
		if (mobi_sw_wdt_set_heartbeat(new_soft_heartbeat))
			return -EINVAL;

		return put_user(new_soft_heartbeat.sec, (int *)arg);

	case WDIOC_GETTIMEOUT:
		return put_user(mobi_sw_wdt_gettime(), (int *)arg);

	case WDIOC_SETTIMEOUTMS:
		if (get_user(new_heartbeat, (int *)arg))
			return -EFAULT;

		new_soft_heartbeat.msec = new_heartbeat;
		new_soft_heartbeat.sec = 0;
		if (mobi_sw_wdt_set_heartbeat(new_soft_heartbeat))
			return -EINVAL;

		return put_user(new_soft_heartbeat.msec, (int *)arg);

	case WDIOC_SETOPTIONS:
		if (get_user(options, (int *)arg))
			return -EFAULT;

		sw_wdt_setoptions(options);
		if (options & WDIOS_DISABLECARD) {
			dprintk("stopping sw watchdog\n");
			mobi_sw_wdt_stop();
			retval = 0;
		}

		if (options & WDIOS_ENABLECARD) {
			dprintk("starting sw watchdog\n");
			mobi_sw_wdt_keepalive();
			retval = 0;
		}
		return retval;
	default:
		return -ENOIOCTLCMD;
	}

	return 0;
}

static ssize_t mobi_sw_wdt_write(struct file *file,
		const char *data, size_t len, loff_t *ppos)
{
	mobi_sw_wdt_keepalive();
	return len;
}

static int mobi_wdt_notify_sys(struct notifier_block *this,
		unsigned long code, void *unused)
{
	if (code == SYS_DOWN || code == SYS_HALT) {
		/* Turn the hw watchdog off */
		mobi_hw_wdt_stop();
		/* should we also kill any software timers ?? */
	}
	return NOTIFY_DONE;
}

static const struct file_operations mobi_sw_wdt_fops = {
	.owner	= THIS_MODULE,
	.llseek = no_llseek,
	.write	= mobi_sw_wdt_write,
	.ioctl	= mobi_sw_wdt_ioctl,
	.open	= mobi_sw_wdt_open,
	.release	= mobi_sw_wdt_close,
};

static struct miscdevice mobi_wdt_miscdev = {
	.minor	= WATCHDOG_MINOR,
	.name	= "watchdog",
	.fops	= &mobi_sw_wdt_fops,
};

/*
 *	The WDT card needs to learn about soft shutdowns in order to
 *	turn the timebomb registers off.
 */
static struct notifier_block mobi_wdt_notifier = {
	.notifier_call = mobi_wdt_notify_sys,
};

#ifdef CONFIG_PROC_FS

/* a little helper but beware, does not return error
 *  if invalid characters are found
 */
static int wdt_atoi(const char *s)
{
	int k = 0;

	k = 0;
	while (*s != '\0' && *s >= '0' && *s <= '9') {
		k = 10 * k + (*s - '0');
		s++;
	}
	return k;
}

static int wdt_expires_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int len = 0;

	if (hw_heartbeat.msec == 0  && hw_heartbeat.sec == 0) {
		len += sprintf(buf+len, "%s\n",  "disabled");

	} else {
		len += sprintf(buf+len, "%d%s\n",
				(hw_heartbeat.msec == 0 ?
				 (hw_heartbeat.sec/1000) : hw_heartbeat.msec),
				(hw_heartbeat.msec == 0 ? "s" : "ms"));
	}
	*eof = 1;

	return len;
}

static int wdt_expires_proc_wr(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	int len = strlen(buffer);
	char *tmp_buffer = NULL;
	struct wdt_timeval new_kernbeat;
	new_kernbeat.msec = 0;
	new_kernbeat.sec = 0;

	tmp_buffer = (char *) kmalloc(len, GFP_KERNEL);
	if (tmp_buffer == NULL)
		return -ENOMEM;

	strcpy(tmp_buffer, buffer);

	if (*tmp_buffer == '.') {
		tmp_buffer++;
		while (*tmp_buffer == '0')
			tmp_buffer++;

		new_kernbeat.msec = wdt_atoi(tmp_buffer);

	} else if (*(tmp_buffer + (len-2)) == 'm') {
		/* end in ms, chop it off */
		*(tmp_buffer+(len-2)) = '\0';
		new_kernbeat.msec = wdt_atoi(tmp_buffer);

	} else if (*(tmp_buffer + (len-1)) == 's') {
		/* end is s, chop it off */
		*(tmp_buffer+(len-1)) = '\0';
		new_kernbeat.sec = wdt_atoi(tmp_buffer);

	} else {
		new_kernbeat.sec = wdt_atoi(tmp_buffer);
	}

	mobi_hw_wdt_set_heartbeat(new_kernbeat);

	mobi_hw_wdt_pat_the_dog();

	kfree(tmp_buffer);
	return count;
}

static int wdt_status_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int len = 0;
	struct softdog *sd;

	if (hw_heartbeat.msec == 0  && hw_heartbeat.sec == 0) {
		len += (sprintf(buf+len,
					"\n%s\n\n",
					"Hardware Watchdog Timeout:  disabled"));

	} else {
		len += (sprintf(buf+len, "\n%s\t%d%s\n\n",
					"Hardware Watchdog Timeout:",
					(hw_heartbeat.msec == 0 ?
					 (hw_heartbeat.sec/1000) : hw_heartbeat.msec),
					(hw_heartbeat.msec == 0 ? "s" : "ms")));
	}

	len += (sprintf(buf+len, "\n%s\n",
				"Software Watchdog Status listed by pid"));
	len += (sprintf(buf+len, "%s\n",
				"--------------------------------------"));
	list_for_each_entry(sd, &softdogs, node) {
		len += (sprintf(buf+len, "%d\t\t%d%s\n", sd->pid,
					(sd->tv.msec == 0 ? sd->tv.sec : sd->tv.msec),
					(sd->tv.msec == 0 ? "s" : "ms")));
	}
	len += (sprintf(buf+len, "\n"));

	*eof = 1;

	return len;
}

#ifdef DW_WDT_DEBUG

#define DEVICE_DBG_READ_REGISTER 	0x1
#define DEVICE_DBG_WRITE_REGISTER 	0x2
#define DEVICE_DBG_INTR_REPLY	 	0x3

static void proc_advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	while (((**ptr == ' ') || (**ptr == '\t') ||
				(**ptr == '\r') || (**ptr == '\n'))
			&& (*ptr-buf < count)) {
		(*ptr)++;
	}

}

static int proc_get_next_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		return -EINVAL;
	}
	return 0;
}

/* use this function after getting fixed number of input */
static int proc_check_for_arg(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	proc_advance_ptr(ptr, buffer, count);
	if (*ptr-buf != count) {
		if (*ptr-buf >= count) {
			return -EINVAL;
		}
		return 1;
	}
	return 0;
}

static int device_proc_wr_debug(struct file *file,
		const char *buffer, unsigned long count, void *data)
{
	char *ptr = (char *)buffer;
	int cmd = 0, arg1 = -1, arg2 = -1;
	int ret = 0;
	uint32_t value;

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		cmd = simple_strtoul(ptr, &ptr, 0);
		if (cmd < 0) {
			printk(KERN_ERR PFX "Invalid debug command\n");
			goto err_help;
		}
	} else {
		printk(KERN_ERR "Invalid cmd string\n");
		goto err_help;
	}

	if (proc_get_next_arg(&ptr, (uint32_t)buffer, count) == 0) {
		arg1 = simple_strtoul(ptr, &ptr, 0);
		if (arg1 < 0) {
			printk(KERN_ERR PFX "invalid argument\n");
			goto err_help;
		}
	} else {
		printk(KERN_ERR "Invalid argument string\n");
		goto err_help;
	}

	ret = proc_check_for_arg(&ptr, (uint32_t)buffer, count);
	if (ret == 1) {
		arg2 = simple_strtoul(ptr, &ptr, 0);
		if (arg2 < 0) {
			printk(KERN_ERR PFX "invalid argument\n");
			goto err_help;
		}
	} else if (ret < 0) {
		printk(KERN_ERR "Invalid argument string\n");
		goto err_help;
	}

	/* read everything we don't care about */
	while (ptr-buffer < count)
		ptr++;

	switch (cmd) {
	case DEVICE_DBG_READ_REGISTER:
		value = readl(watchdog_base + arg1);
		printk(KERN_ERR "Read at offset 0x%x is 0x%x(%d)\n",
				arg1, value, value);
		break;
	case DEVICE_DBG_WRITE_REGISTER:
		printk(KERN_ERR "Write 0x%x(%d) to offset 0x%x\n",
				arg2, arg2, arg1);
		writel(arg2, watchdog_base + arg1);
		break;
	case DEVICE_DBG_INTR_REPLY:
		if (arg1 == 1)
			no_intr_reply = 1;
		else
			no_intr_reply = 0;
		printk(KERN_ERR "Interrupt response %sabled\n",
				(arg1 == 1) ? "dis" : "en");
		break;
	default:
		break;
	}

	return ptr - buffer;

err_help:

#define PRINT_CMD(x)	#x
	printk(KERN_ERR "\nAvailable debug commands are(see code for details):\n");
	/* read everything we don't care about */
	while (ptr-buffer < count)
		ptr++;

	return -EINVAL;
}
#endif
#endif

static int device_powerup(void)
{
	return mobi_reset_disable(RESET_ID_WDT);
}

static void device_powerdown(void)
{
	mobi_reset_enable(RESET_ID_WDT);
}

static int dw_wdt_probe(struct amba_device *pdev, struct amba_id *id)
{
	int rc = 0;
	int16_t i;
	struct dw_wdt_driver_data_t *wdt_pdata =
		(struct dw_wdt_driver_data_t *)pdev->dev.platform_data;
	unsigned long clock_rate = 0;
	struct clk *wdt_clk;

	if (mobi_wdt_miscdev.parent)
		return -EBUSY;
	mobi_wdt_miscdev.parent = &pdev->dev;

	rc = register_reboot_notifier(&mobi_wdt_notifier);
	if (rc) {
		printk(KERN_ERR PFX
				"Can't register reboot notifier (err=%d)\n", rc);
		return rc;
	}

	rc = device_powerup();
	if (rc) {
		printk(KERN_ERR PFX "Could not take device out of reset\n");
		goto err_reboot;
	}

	rc = misc_register(&mobi_wdt_miscdev);
	if (rc) {
		printk(KERN_ERR PFX
				"Can't register miscdev on minor=%d (err=%d)\n",
				mobi_wdt_miscdev.minor, rc);
		goto err_reboot;
	}

	rc = amba_request_regions(pdev, PFX);
	if (rc) {
		printk(KERN_ERR PFX "No IO memory region found, probe failed.");
		rc = -ENODEV;
		goto err_regions;
	}

	watchdog_base = ioremap(pdev->res.start, pdev->res.end-pdev->res.start+1);
	if (!watchdog_base) {
		printk(KERN_ERR PFX
				"Watchdog failed to ioremap register space\n");
		rc = -EIO;
		goto err_ioremap;
	}

	/* make sure interrupts are off before requesting irq */
	WRITEL(0, MOBI_WDT_CR_OFFSET);

	rc = request_irq(pdev->irq[0], dw_wdt_interrupt,
			IRQF_DISABLED, PFX, NULL);
	if (rc) {
		printk(KERN_ERR PFX "Can't get IRQ for watchdog\n");
		goto err_irq;
	}

#ifdef CONFIG_PROC_FS
	{
		struct proc_dir_entry *pentry;
		char string[20];

		/* create /proc/driver/watchdog */
		if (watchdog_proc_dir == NULL)
			watchdog_proc_dir = proc_mkdir(WATCHDOG_PROC_DIR, NULL);

		sprintf(string, "%s", "expires");
		pentry = create_proc_entry(string,
				S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH,
				watchdog_proc_dir);
		if (pentry) {
			pentry->read_proc  = wdt_expires_proc_rd;
			pentry->write_proc = wdt_expires_proc_wr;
		}

		sprintf(string, "%s", "status");
		pentry = create_proc_entry(string,
				S_IRUSR | S_IRGRP | S_IROTH,
				watchdog_proc_dir);
		if (pentry) {
			pentry->read_proc  = wdt_status_proc_rd;
		}

#ifdef DW_WDT_DEBUG
		sprintf(string, "%s", "debug");
		pentry = create_proc_entry(string,
				S_IRUSR | S_IRGRP | S_IROTH,
				watchdog_proc_dir);
		if (pentry) {
			pentry->write_proc  = device_proc_wr_debug;
		}
#endif
	}
#endif

	/* Check that the heartbeat value is within range;
	 *  if not reset to the default for now, only time in
	 *  seconds will be allow for initialization default
	 */
	/* modparam overrides platform data */
	if (hw_default_heartbeat != 0) {
		hw_heartbeat.sec = hw_default_heartbeat;
		hw_heartbeat.msec = 0;

	} else {
		hw_heartbeat.sec  = wdt_pdata->timeout_sec;
		hw_heartbeat.msec = wdt_pdata->timeout_msec;
	}

	/*
	 *  calculate what timeouts we can program based on
	 *  the current clock
	 */
	wdt_clk = clk_get(&pdev->dev, wdt_pdata->clk_name);
	clock_rate = clk_get_rate(wdt_clk);

	dprintk("%s clock rate: %lu\n", wdt_pdata->clk_name, clock_rate);
	if (clock_rate != 0) {
		for (i = 0; i < 16; i++) {
			wdt_timeouts[i] = (0x1 << (16+i))/(clock_rate/1000);
			dprintk("wdt_timeouts[%d] = %u\n", i, wdt_timeouts[i]);
		}
	} else {
		printk(KERN_ERR "Watchdog clock is not running,"
				" watchdog timer not started\n");
		rc = -EIO;
		goto err_clock;
	}

	if (mobi_hw_wdt_set_heartbeat(hw_heartbeat)) {
		printk(KERN_ERR "Invalid watchdog timeout request,"
				" watchdog timer not started\n");
		rc = -EINVAL;
		goto err_clock;
	}

	/* start the hardware watchdog upon driver init */
	return mobi_hw_wdt_start();

err_clock:
	free_irq(pdev->irq[0], NULL);

err_irq:
	iounmap(watchdog_base);

err_ioremap:
	amba_release_regions(pdev);

err_regions:
	misc_deregister(&mobi_wdt_miscdev);

err_reboot:
	unregister_reboot_notifier(&mobi_wdt_notifier);

	device_powerdown();
	return rc;
}

static int dw_wdt_remove(struct amba_device *pdev)
{
	int res;

#ifdef CONFIG_PROC_FS
	char string[20];
	sprintf(string, "%s", "expires");
	remove_proc_entry(string, watchdog_proc_dir);
	sprintf(string, "%s", "status");
	remove_proc_entry(string, watchdog_proc_dir);
#ifdef DW_WDT_DEBUG
	sprintf(string, "%s", "debug");
	remove_proc_entry(string, watchdog_proc_dir);
#endif
	remove_proc_entry(WATCHDOG_PROC_DIR, NULL);
#endif

	mobi_hw_wdt_stop();
	unregister_reboot_notifier(&mobi_wdt_notifier);

	free_irq(pdev->irq[0], NULL);
	iounmap(watchdog_base);
	amba_release_regions(pdev);

	res = misc_deregister(&mobi_wdt_miscdev);
	if (!res)
		mobi_wdt_miscdev.parent = NULL;

	device_powerdown();
	return res;
}

static void dw_wdt_shutdown(struct amba_device *pdev)
{
	mobi_hw_wdt_stop();
}

static int dw_wdt_suspend(struct amba_device *pdev, pm_message_t message)
{
	struct softdog *sd;

	/* stop the hw watchdog */
	mobi_hw_wdt_stop();

	/* cycle thu list of software timers and set expires to 0 */
	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		mod_timer(&sd->timer, 0);
	}
	mutex_unlock(&softdogs_mutex);

	return 0;
}

static int dw_wdt_resume(struct amba_device *pdev)
{
	struct softdog *sd;
	int timeout = 0;

	/* get things moving again */
	mobi_hw_wdt_start();

	/* cycle thu list of software timers and get the timers running again */
	mutex_lock(&softdogs_mutex);
	list_for_each_entry(sd, &softdogs, node) {
		if (sd->tv.msec != 0)
			timeout = msecs_to_jiffies(sd->tv.msec);
		else
			timeout = (sd->tv.sec*HZ);

		mod_timer(&sd->timer, jiffies+timeout);
	}
	mutex_unlock(&softdogs_mutex);

	return 0;
}

static struct amba_id dw_wdt_ids[] = {
	{
		.id     = DW_WDT_AMBA_DEVID,
		.mask   = 0xffffffff,
	},
	{ 0, 0 },
};

static struct amba_driver dw_wdt_driver = {
	.probe    = dw_wdt_probe,
	.remove	  = __exit_p(dw_wdt_remove),
	.shutdown = dw_wdt_shutdown,
#ifdef CONFIG_PM
	.suspend  = dw_wdt_suspend,
	.resume	  = dw_wdt_resume,
#else
	.suspend    = NULL,
	.resume     = NULL,
#endif
	.id_table   = dw_wdt_ids,
	.drv    = {
		.name   = DW_WDT_AMBA_NAME,
	},
};

static int __init dw_wdt_init(void)
{
	return amba_driver_register(&dw_wdt_driver);
}

static void __exit dw_wdt_exit(void)
{
	amba_driver_unregister(&dw_wdt_driver);
}

module_init(dw_wdt_init);
module_exit(dw_wdt_exit);

MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Mobilygen driver for Synopsys DW Watchdog");
MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);
