/************************************************************
 * ISP1761 Hardware Abstraction Layer code file
 *
 * (c) 2007 Vineyard Technologis, Inc.
 *
 * File Name: hal_x86.c
 *
 * History:
 *
 * Date                Author                  Comments
 * ---------------------------------------------------------------------
 * Aug 17 2007         Steve Lee               Initial Creation     
 *
 **********************************************************************
 */
#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>  
#include <linux/usb.h>
#include <linux/pci.h>
#include <linux/poll.h>
#include <linux/vmalloc.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/unaligned.h>


/*--------------------------------------------------------------*
 *               linux system include files
 *--------------------------------------------------------------*/
#include "hal_x86.h"
#include "hal_intf.h"
#include "isp1761.h"
#include "asm/arch/mg3500_devices.h"


/*--------------------------------------------------------------*
 *               Local variable Definitions
 *--------------------------------------------------------------*/
struct isp1761_dev              isp1761_loc_dev[ISP1761_LAST_DEV];
static __u32			mem_phy = ISP1761_PHYADDR;
static __u32			mem_len = ISP1761_PHYSIZE;
static int 			isp1761hw_init = 0;

/*--------------------------------------------------------------*
 *               Local # Definitions
 *--------------------------------------------------------------*/
#define isp1761_driver_name     "isp1761-arm-merlin"

#define	WORD	unsigned short
#define	LONG	unsigned long

static int  isp1761_probe(void);
static void isp1761_remove(void);

static int __init isp1761_hw_module_init (void);

/* 
   ISP1760_4.pdf $7.3 page 19
   When accessing the host controller registers in 16-bit mode,
   the register access must always be completed using two subsequent accesses.
*/
//==============================================================================
// Write a 32 bit Register of isp1761
//==============================================================================
void isp1761_reg_write32(struct isp1761_dev *dev,__u16 reg,__u32 data)
{ 
    /* Write the 32bit to the register address given to us*/
	writel(data,dev->baseaddress+reg);
}

// Read a 32 bit Register of isp1761
//==============================================================================
__u32 isp1761_reg_read32(struct isp1761_dev *dev,__u16 reg,__u32 data)
{ 
	data = readl(dev->baseaddress + reg);
    return data;
}

//==============================================================================
//   
// Module dtatils: isp1761_mem_read
//
// Memory read using PIO method.
//
//  Input: struct isp1761_driver *drv  -->  Driver structure.
//              __u32 start_add    ---> Starting address of memory 
//              __u32 end_add      ---> End address 
//              __u32 *buffer      ---> Buffer pointer.
//              __u32 length       ---> Length 
//              __u16 dir          ---> Direction ( Inc or Dec)
//                      
//  Output     int Length  ----> Number of bytes read 
//
//  Called by: system function 
// 
//==============================================================================
int isp1761_mem_dread( struct isp1761_dev *dev, __u32 start_add, __u32 end_add, __u32 * buffer, __u32 length, __u16 dir)
{
    int a = (int)length;
    volatile u32 fake = 0;

    if(buffer == NULL)
    {
	printk("@@@@ In isp1761_mem_read The buffer is pointing to null\n");
	return 0;
    }

    /*initialize the Register 0x33C-used to manage Multiple threads */
    writel( start_add-4, dev->baseaddress+0x33c);

    /* fake read, because in some moments first 2 bytes are corrupted */
    fake = readl(dev->baseaddress + start_add-4);

    while(a>0){
	*(buffer++) = readl(dev->baseaddress + start_add);
	start_add +=4;
	a -= 4;
    }

    return ((a < 0) || (a == 0))?0:(-1);
}

int isp1761_mem_read( struct isp1761_dev *dev, __u32 start_add, __u32 end_add, __u32 * buffer, __u32 length, __u16 dir)
{
    int a = (int)length;
    u32 w;
    volatile u32 fake = 0;

    if(buffer == NULL)
    {
	printk("@@@@ In isp1761_mem_read The buffer is pointing to null\n");
	return 0;
    }

    /*initialize the Register 0x33C-used to manage Multiple threads */
    writel( start_add-4, dev->baseaddress+0x33c);

    /* fake read, because in some moments first 2 bytes are corrupted */
    fake = readl(dev->baseaddress + start_add-4);

    while(a>0){
	w = readl(dev->baseaddress + start_add);
        *(buffer++) = le32_to_cpu(w);

        start_add +=4;
        a -= 4;
    }

    return ((a < 0) || (a == 0))?0:(-1);
}

//==============================================================================
//
// Module dtatils: isp1761_mem_write
//
// Memory write using PIO method.
//
//  Input: struct isp1761_driver *drv  -->  Driver structure.
//              __u32 start_add   ---> Starting address of memory 
//              __u32 end_add     ---> End address 
//              __u32 * buffer    ---> Buffer pointer.
//              __u32 length      ---> Length 
//              __u16 dir         ---> Direction ( Inc or Dec)
//                      
//  Output     int Length  ----> Number of bytes read 
//
//  Called by: system function 
// 
//==============================================================================
int isp1761_mem_dwrite(struct isp1761_dev *dev, __u32 start_add, __u32 end_add, __u32 * buffer, __u32 length, __u16 dir)
{
    int a = length;

    while(a>0){
	writel( *(buffer++), dev->baseaddress+start_add);
        start_add +=4;
        a -=4;
    }

    return ((a < 0) || (a == 0))?0:(-1);
}

int isp1761_mem_write(struct isp1761_dev *dev, __u32 start_add, __u32 end_add, __u32 * buffer, __u32 length, __u16 dir)
{
    int a = length;
    u32 w;

    while(a>0){
	w = le32_to_cpu(*(buffer++));
	writel( w, dev->baseaddress+start_add);
        start_add +=4;
        a -=4;
    }

    return ((a < 0) || (a == 0))?0:(-1);
}

//==============================================================================
//  
// Module dtatils: isp1761_register_driver
//
// This function is used by top driver (OTG, HCD, DCD) to register
// their communication functions (probe, remove, suspend, resume) using
// the drv data structure.
// This function will call the probe function of the driver if the ISP1761
// corresponding to the driver is enabled
//
//  Input: struct isp1761_driver *drv  --> Driver structure.
//  Output result 
//         0= complete 
//         1= error.
//
//  Called by: system function module_init 
// 
//==============================================================================

struct device_driver fake;

int isp1761_register_driver(struct isp1761_driver *drv) 
{
    struct isp1761_dev  *dev;
    int result;
    isp1761_id *id;

    hal_entry("%s: Entered\n",__FUNCTION__);
//    info("isp1761_register_driver(drv=%p)\n",drv);

    if(isp1761hw_init==0) 

    if(!drv) return -EINVAL;

    dev = &isp1761_loc_dev[drv->index];

#if 1 /* LHM : registering device driver as Kobject not PCI type*/
    device_initialize(&dev->gdev);
    kobject_set_name(&dev->gdev.kobj, "isp1761_usb");
    kobject_add(&dev->gdev.kobj);
    kobject_uevent(&dev->gdev.kobj, KOBJ_ADD);
#endif /* LHM */

    if(drv->index == ISP1761_DC)
    { /*FIX for device*/
        result = drv->probe(dev,drv->id);
    }
    else
    {              
        id = drv->id;
	fake.name = drv->name;
	dev->gdev.driver = &fake;
	
	if(dev->active)
		result = drv->probe(dev,id);
	else
		result = -ENODEV;
    }

    if(result >= 0 )
	dev->driver = drv;

    hal_entry("%s: Exit\n",__FUNCTION__);
    return result;
}


//==============================================================================
// 
// Module dtatils: isp1761_unregister_driver
//
// This function is used by top driver (OTG, HCD, DCD) to de-register
// their communication functions (probe, remove, suspend, resume) using
// the drv data structure.
// This function will check whether the driver is registered or not and
// call the remove function of the driver if registered
//
//  Input: struct isp1761_driver *drv  --> Driver structure.
//  Output result 
//         0= complete 
//         1= error.
//
//  Called by: system function module_init 
// 
//==============================================================================
void isp1761_unregister_driver(struct isp1761_driver *drv)
{
    struct isp1761_dev  *dev;
    
    hal_entry("%s: Entered\n",__FUNCTION__);

    info("isp1761_unregister_driver(drv=%p)\n",drv);
    dev = &isp1761_loc_dev[drv->index];
    if(dev->driver == drv) {
        /* driver registered is same as the requestig driver */
        drv->remove(dev);
        dev->driver = NULL;
        info("De-registered Driver %s\n", drv->name);
        return;
    }
    hal_entry("%s: Exit\n",__FUNCTION__);
}


//==============================================================================
//
//  Module dtatils: isp1761_hw_module_init
//
//  This function initializes the most low level data structures and 
//  interface to access isp1761 device from sparc.
//	
//
//  Input: void 
//  Output result 
//         0= complete 
//         1= error.
//
//  Called by: system function module_init 
// 
// 
// 
//==============================================================================
static int __init isp1761_hw_module_init (void) 
{
    int result = 0;

    if(isp1761hw_init==1) return 0;

    hal_entry("%s: Entered\n",__FUNCTION__);
    
    memset(isp1761_loc_dev,0,sizeof(isp1761_loc_dev));
    result = isp1761_probe();
    if( !result) isp1761_int_init();

    printk("%s Initialization %s\n", isp1761_driver_name, (result==0)? "Success" : "Fail" );
    
    isp1761hw_init = 1;
    
    hal_entry("%s: Exit\n",__FUNCTION__);
    
    return result;
}

//==============================================================================
//
//  Module dtatils: isp1761_hw_module_cleanup
//
//  This function clear the global variables related to isp1761 at low level
//  and deregister every things that were registered in module init function.
//
//  Input: void 
//  Output void
//
//  Called by: system function module_cleanup 
// 
// 
//==============================================================================
static void __exit isp1761_hw_module_cleanup (void) 
{
	if(isp1761hw_init==0) return;
    hal_entry("%s: Entered\n",__FUNCTION__);

    isp1761_remove();
    memset(isp1761_loc_dev,0,sizeof(isp1761_loc_dev));
    
    isp1761hw_init = 0;
    isp1761_int_exit();
    hal_entry("%s: Exit\n",__FUNCTION__);
}


EXPORT_SYMBOL(isp1761_reg_read32);
EXPORT_SYMBOL(isp1761_reg_write32);
EXPORT_SYMBOL(isp1761_mem_read);
EXPORT_SYMBOL(isp1761_mem_write);
EXPORT_SYMBOL(isp1761_mem_dread);
EXPORT_SYMBOL(isp1761_mem_dwrite);
EXPORT_SYMBOL(isp1761_register_driver);
EXPORT_SYMBOL(isp1761_unregister_driver);

MODULE_AUTHOR (DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE ("GPL");

module_init (isp1761_hw_module_init);
module_exit (isp1761_hw_module_cleanup);

#define	ISP1761_PLAT_DEVID	0x00011761

static int isp1761_checkhw( void)
{
    struct isp1761_dev	*loc_dev = &(isp1761_loc_dev[ISP1761_HC]);
    __u32		reg_data = 0;
    int			ret = 0;

// for 16 bit MHIF access, all ISP1761 accesses must be 32 bits read/write
    isp1761_reg_write32( loc_dev, HC_HW_MODE_REG, 0x0000);
    reg_data = isp1761_reg_read32( loc_dev, HC_CHIP_ID_REG, reg_data);
    printk("%s chip ID=0x%08x\n", isp1761_driver_name, reg_data);

    if(reg_data != ISP1761_PLAT_DEVID) {
	printk("========= %s isp1761 chip ID is wrong 0x%04x =========\n", isp1761_driver_name, reg_data);
        ret = -ENODEV;
    }

    return ret;
}

//==============================================================================
//
//  Module dtatils: isp1761_probe
//
//  sparc bus probe function of ISP1761
//
//  Input: void (Constant) 
//  Output void
//
//  Called by: system function isp1761_hw_module_init 
// 
//==============================================================================
static int isp1761_probe(void)
{
	
//	#define ACCESS_RETRY_COUNT	100
    struct isp1761_dev  *loc_dev;
    void *address = 0;
    //int length = 0;
    int status = 1;
    
    hal_entry("%s: Entered\n",__FUNCTION__);

    mem_phy = ISP1761_PHYADDR;
    hal_init("isp1761 base address = 0x%x\n", mem_phy);

    /* Get the Host Controller IO and INT resources
     */
    loc_dev = &(isp1761_loc_dev[ISP1761_HC]);

    loc_dev->irq = ISP1761_IRQ_NUM;

    loc_dev->io_base = mem_phy;
    loc_dev->io_len  = mem_len; /*64K*/

    loc_dev->start   = mem_phy;
    loc_dev->length  = mem_len;

    loc_dev->index = ISP1761_HC;/*zero*/

    if(check_mem_region(loc_dev->io_base,loc_dev->io_len)<0){
        err("ISP1761 controller is already in use\n");
        release_mem_region(loc_dev->io_base, loc_dev->io_len);
        return -EBUSY;
    }

    if(!request_mem_region(loc_dev->io_base, loc_dev->io_len, isp1761_driver_name)){
        err("ISP1761 controller is already in use(2)\n");
        release_mem_region(loc_dev->io_base, loc_dev->io_len);
        return -EBUSY;
    }

    /*map available memory*/
    address = ioremap_nocache(loc_dev->io_base,loc_dev->io_len);
    if(address == NULL){
        err("ISP1761 memory map problem\n");
        release_mem_region(loc_dev->io_base, loc_dev->io_len);
        return -ENOMEM;
    } 

    loc_dev->baseaddress = (u8*)address;
    loc_dev->dmabase = (u8*)0;

    hal_init("isp1761 HC MEM Base= %p irq = %d\n", loc_dev->baseaddress, loc_dev->irq);

    mobi_hmux_request();
    status = isp1761_checkhw();
    mobi_hmux_release();

#ifdef ISP1761_DEVICE   

    /*initialize device controller framework*/  
    loc_dev = &(isp1761_loc_dev[ISP1761_DC]);
    loc_dev->irq = ISP1761_IRQ_NUM;
    loc_dev->io_base = mem_phy;
    loc_dev->start   = mem_phy;
    loc_dev->length  = mem_len;
    loc_dev->io_len  = mem_len;
    loc_dev->index   = ISP1761_DC;
    loc_dev->baseaddress = address;
    loc_dev->active = 1;
    memcpy(loc_dev->name,"isp1761_dev",11);
    loc_dev->name[12] = '\0';


    {
        u32 chipid = 0;
        chipid = readl(address + 0x270);
        info("pid %04x, vid %04x\n", (chipid & 0xffff), (chipid >> 16));
    }   
    hal_init(("isp1761 DC MEM Base= %lx irq = %d\n", 
                loc_dev->io_base,loc_dev->irq));
    /* Get the OTG Controller IO and INT resources
     * OTG controller resources are same as Host Controller resources
     */
    loc_dev = &(isp1761_loc_dev[ISP1761_OTG]);
    loc_dev->irq = ISP1761_IRQ_NUM; /*same irq also*/
    loc_dev->io_base = mem_phy;
    loc_dev->start   = mem_phy;
    loc_dev->length  = mem_len;     
    loc_dev->io_len  = mem_len;
    loc_dev->index   = ISP1761_OTG; 
    loc_dev->baseaddress = address; /*having the same address as of host*/
    loc_dev->active = 1;
    memcpy(loc_dev->name,"isp1761_otg",11);
    loc_dev->name[12] = '\0';

    hal_init(("isp1761 OTG MEM Base= %lx irq = %x\n", loc_dev->io_base,loc_dev->irq));

#endif

    memcpy(loc_dev->name, isp1761_driver_name, sizeof(isp1761_driver_name));
    loc_dev->name[sizeof(isp1761_driver_name)] = 0;
    loc_dev->active = 1;

    hal_entry("%s: Exit\n",__FUNCTION__);
    return status;
}


//==============================================================================
//
//  Module dtatils: isp1761_remove
//
//  Input: void 
//  Output void
//
//  Called by: system function module_cleanup 
// 
//==============================================================================
static void isp1761_remove(void)
{
	struct isp1761_dev  *loc_dev;
	
    hal_entry("%s: Entered\n",__FUNCTION__);

    hal_init(("isp1761_remove\n"));
    
    loc_dev  = &isp1761_loc_dev[ISP1761_HC]; /*Lets handle the host first*/
    release_mem_region(loc_dev->io_base, loc_dev->io_len); /*free the memory occupied by host*/      	
    iounmap(loc_dev->baseaddress); /*unmap the occupied memory resources*/

    hal_entry("%s: Exit\n",__FUNCTION__);
    return;
}

