/******************************************************************************
 * uvc_mg3500.c -- uvc gadget driver implementation
 *
 *
 ******************************************************************************/

#ifndef CONFIG_USB_GADGET_DUALSPEED
#define CONFIG_USB_GADGET_DUALSPEED
#endif

#include <linux/version.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#include <linux/config.h>
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/utsname.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/unaligned.h>

#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#else
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#endif

#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <linux/delay.h> 				// for mdelay() 
#include <linux/string.h>
#include <linux/err.h>
#include <linux/proc_fs.h>
#include <linux/highmem.h> 				// kmap()
#include <linux/dma-mapping.h>

#include <linux/pagemap.h>

#include "uvc_cap_interface.h"
#include "uac_cap_interface.h"
#include "uvcdescript.h"
#include "uacdescript.h"
#include "platform_caps.h"
#include "uvc_codec_if.h"
#include "uvc_callback.h"
#include "dbgout.h"
#include "drvrev.h"


#define EN_GEN_EOS_ON_STOP			// Enable this if ZLP is not used for EoS detection
#define EN_PRE_QUEUE_EP_BUFFERS		// Enable this prequeue buffers to OTG EP

#define GADGETFS_MAGIC		0xaee71ee7

uint32_t dbg_mask = 0x00;
module_param(dbg_mask, int, 0644);
MODULE_PARM_DESC(dbg_mask, "Debug Mask");


/**
 * id numbers - use normal USB-IF procedures
 */
// TODO : Confirm
#define DRIVER_VENDOR_NUM	 	0x0B6A		/* MAXIM */

#if (BOARD_ID == BOARD_ID_MOBICAM3)

#define DRIVER_PRODUCT_NUM 		0x4D51

#elif (BOARD_ID == BOARD_ID_EVP)

#define DRIVER_PRODUCT_NUM 		0x4556

#endif

#define UVC_STRM_FID	0x01
#define UVC_STRM_EOF	0x02
#define UVC_STRM_EOH	0x80

#define REQ_BUFF_SIZE	4096

// if nonzero, how many seconds to wait before trying to wake up the host after suspend
static unsigned autoresume = 0;
module_param(autoresume, uint, 0);

static char * serial = "12345";
module_param(serial, charp, S_IRUGO);

#include "../usbstring.c"

int usb_dispatch_non_standard_setup_request(struct usb_request	*req, const struct usb_ctrlrequest *ctrl);
void ep_out_complete(struct usb_ep *ep, struct usb_request *req);
int get_string(int id, u8 *buf);

struct device;
struct class_device;

#define POOL_ENTRIES 3
/* other /dev/gadget/$ENDPOINT files represent endpoints */
enum ep_state {
	STATE_EP_DISABLED = 0,
	STATE_EP_READY,
	STATE_EP_DEFER_ENABLE,
	STATE_EP_ENABLED,
	STATE_EP_UNBOUND,
};

enum file_pos {
	FILE_STREAMING = 0,
	FILE_EOS,
};

struct req_pool {
	struct usb_request *req;			// allocated request
	int inuse;
	struct list_head list;
};

struct properties_set {
	int					number;
	const char			*ep_name;			// PCD name for ep
	struct usb_ep		*ep;				// PCD's EP Object
	wait_queue_head_t	ep_waitq;			// interrupt waiting and notification
	int					ep_waitq_flag;      // interrupt wait queue flag
	struct usb_endpoint_descriptor *ep_desc;
											// end point descriptor initialized by interface
	int					pos;				// Position FILE_STREAMING(Streaming), FILE_EOS(EOS)
	struct usb_request	*ep_req;			// The ep request i.e. currently used to transfer data to user
	struct req_pool		*reqpool;			// array of requests, used for IN EP
	struct list_head 	queue;				// list of requests used for OUT EP
	struct dentry		*dentry;
	struct inode		*inode;
	char				*file_name;			// name exposed through usbav_fs
	int					state;
	int					proc_uvc_hdr;
	int					first_uvc_pkt;
	unsigned char		uvc_fid;
	int					total_bytes;
};

struct usb_ep *usb_get_pcd_ep(struct usb_gadget *gadget,
										const char *find_name);

static const char driver_name[] 	= "uvc_mg3500";
static const char usb_str_product[] = "MG3500";

extern char	uvcmg3500_str_serial_num[40];

// big enough to hold our biggest descriptor
#define USB_BUFSIZ	8096

/**
 *  Device context
 */
struct uvc_mg3500 {
	spinlock_t          lock;
	struct usb_gadget	*gadget;
	struct usb_request	*req;		// for control responses
	u8			config; 	// 0, CONFIG_VALUE_MAIN, CONFIG_VALUE_DFU
	u8			alt_setting[USB_INTERFACE_INDEX_LAST]; 	// 0, CONFIG_VALUE_MAIN, CONFIG_VALUE_DFU
	struct timer_list	resume;		// autoresume timer

	struct dentry			*dentry;
	struct super_block		*sb;
	
	unsigned				vid_cap_cfg_changed;
	unsigned				vid_enc_cfg_changed;
	unsigned				vid_dec_cfg_changed;
	unsigned				aud_codec_cfg_changed;
	unsigned				pu_cfg_changed;
	unsigned				camera_cfg_changed;
	
	wait_queue_head_t		codec_cfg_waitq;
	
	codec_config_t			vid_cap_cfg_data;
	codec_config_t			vid_enc_cfg_data;
	codec_config_t			vid_dec_cfg_data;

	codec_config_t			aud_codec_cfg_data;
	codec_config_t			pu_cfg_data;
	codec_config_t			camera_cfg_data;

	int						is_defered_ctrl;
	struct usb_ctrlrequest	defered_ctrl;
};


struct properties_set ep1_in_set = {
	.number = 1,
	.pos = 0,
	.ep_req = 0,
	.reqpool = 0,
	.ep_desc = 0,
	.ep = 0,
	.file_name = FILE_VID_CAP_STREAM_0,
#if HAS_VIDCAP_ISOCH == 1
	.proc_uvc_hdr = 0
#else
	.proc_uvc_hdr = 1
#endif
};


struct properties_set ep2_in_set = {
	.number = 2,
	.pos = 0,
	.ep_req = 0,
	.reqpool = 0,
	.ep_desc = 0,
	.ep = 0,
	.file_name = FILE_VID_TRANS_ENC,
	.proc_uvc_hdr = 1
};

struct properties_set ep2_out_set = {
	.number = 2,
	.pos = 0,
	.ep_req = 0,
	.reqpool = 0,
	.ep_desc = 0,
	.ep = 0,
	.file_name = FILE_ID_TRANS_DEC,
	.proc_uvc_hdr = 1
};

struct properties_set ep3_in_set = {
	.number = 3,
	.pos = 0,
	.ep_req = 0,
	.reqpool = 0,
	.ep_desc = 0,
	.ep = 0,
	.file_name = FILE_AUD_CAP_STREAM,
	.proc_uvc_hdr = 0
};

// * TODO : Reconfigure in PCD to make it in type
struct properties_set ep4_in_set = {
	.number = 4,
	.pos = 0,
	.ep_req = 0,
	.reqpool = 0,
	.ep_desc = 0,
	.ep = 0,
};

// TODO : remove this hack after testing
struct uvc_mg3500 *the_device = 0;

#define CONFIG_VALUE_MAIN 		1 	

static struct usb_otg_descriptor otg_descriptor = {
	.bLength 			= sizeof(otg_descriptor),		//
	.bDescriptorType 	= USB_DT_OTG,					// 9
	//------------------
	.bmAttributes 		= USB_OTG_SRP,					// 
};

/**
 * DEVICE descriptor 
 */
struct usb_device_descriptor device_desc = 
{
	.bLength 			= sizeof(device_desc),
	.bDescriptorType 	= USB_DT_DEVICE,
	//------------------
	.bcdUSB		= __constant_cpu_to_le16(0x0200),
	.bDeviceClass	= 0xEF, 	/* Multi interface function code device  */
	.bDeviceSubClass = 0x02, 	// Common Class
	.bDeviceProtocol = 0x01, 	// IAD
	.bMaxPacketSize0 = 0x40,
	//------------------
	.idVendor	= __constant_cpu_to_le16(DRIVER_VENDOR_NUM),
	.idProduct	= __constant_cpu_to_le16(DRIVER_PRODUCT_NUM),
	.bcdDevice	= 0, //ph
	//------------------
	.iManufacturer	= STN_MANUFACTURER,
	.iProduct	= STN_PRODUCT,
	.iSerialNumber	= STN_SERIAL,
	.bNumConfigurations = 1
};

/**
 * CONFIGURATION descriptor
 */
struct usb_config_descriptor config_desc_main = {
	.bLength 			= sizeof(config_desc_main),
	.bDescriptorType 	= USB_DT_CONFIG,
	//------------------
	/* compute wTotalLength on the fly */
	.bNumInterfaces 	= USB_INTERFACE_INDEX_LAST, 
	.bConfigurationValue=  CONFIG_VALUE_MAIN,
	.iConfiguration 	= 0,
	// TODO : remove USB_CONFIG_ATT_SELFPOWER attrib for bus powered
	.bmAttributes 		= USB_CONFIG_ATT_ONE | USB_CONFIG_ATT_SELFPOWER,	
										//Configuration characteristics
										// D7: Reserved (set to one)
										// D6: Self-powered
										// D5: Remote Wakeup
										// D4...0: Reserved (reset to zero)
	.bMaxPower 			= 500>>1, /* 500mA */			//was 1, /* self-powered */
};

/*
 * used in USB 2.0 only:
 */
static struct usb_qualifier_descriptor dev_qualifier = {
	.bLength 			= sizeof(dev_qualifier),
	.bDescriptorType 	= USB_DT_DEVICE_QUALIFIER,
	//-------------------
	.bcdUSB 	= __constant_cpu_to_le16(0x0200),
	.bDeviceClass 	= 0xEF,
	.bDeviceSubClass = 0x02, 	// Common Class
	.bDeviceProtocol = 0x01, 	// IAD
	.bMaxPacketSize0 = 0x40,

	.bNumConfigurations = 1, 
};


/**
 * Helper function to fill interface descriptor
 */
int
usb_descriptor_fill_intf_buf(void *buf, unsigned buflen,
		const struct usb_descriptor_header **src)
{
	u8	*dest = buf;

	if (!src)
		return -EINVAL;

	/* fill buffer from src[] until null descriptor ptr */
	for (; 0 != *src; src++) {
		unsigned		len = (*src)->bLength;

		if (len > buflen)
			return -EINVAL;
		memcpy(dest, *src, len);
		buflen -= len;
		dest += len;
	}
	return dest - (u8 *)buf;
}


/**
 * Helper function to fill configuration descriptor.
 * Calls usb_descriptor_fill_intf_buf to fill constituent interface
 * descriptors.
 */

int uav_gadget_fill_config_buf(
	const struct usb_config_descriptor	*config,
	void					*buf,
	unsigned				length
)
{
	struct usb_config_descriptor		*cp = buf;
	int					len;
	int					used_len = 0;

	UAV_DBG_CFG("uav_gadget_fill_config_buf\n");
	/* config descriptor first */
	if (length < USB_DT_CONFIG_SIZE)
		return -EINVAL;
	*cp = *config; 

	buf = USB_DT_CONFIG_SIZE + (u8*)buf;
	used_len = USB_DT_CONFIG_SIZE;
#if HAS_VID_CAP == 1
	len = usb_descriptor_fill_intf_buf(buf,	length - used_len, uvc_interface_descriptors);
	if (len < 0){
		UAV_DBG_CFG("uav_gadget_fill_config_buf len < 0\n");
		return len;
	}
	buf += len;
	used_len += len;
#endif
#if HAS_VID_TRANS == 1
	len = usb_descriptor_fill_intf_buf(buf,	length - used_len, transcode_interface_descriptors);
	if (len < 0){
		UAV_DBG_CFG("uav_gadget_fill_config_buf len < 0\n");
		return len;
	}
	buf += len;
	used_len += len;
#endif

#if HAS_AUD_CAP == 1
	len = usb_descriptor_fill_intf_buf(buf,	length - used_len, uac_interface_descriptors);
	if (len < 0){
		UAV_DBG_CFG("uav_gadget_fill_config_buf len < 0\n");
		return len;
	}
	buf += len;
	used_len += len;
#endif

	UAV_DBG_CFG("uav_gadget_fill_config_buf%d\n", used_len);
	/* patch up the config descriptor */
	cp->bLength = USB_DT_CONFIG_SIZE;
	cp->bDescriptorType = USB_DT_CONFIG;
	UAV_DBG_CFG("config desc len %d\n",len);
	cp->wTotalLength = cpu_to_le16(used_len);
	cp->bmAttributes |= USB_CONFIG_ATT_ONE;
	return used_len;
}



/**
 * Handles Config Descriptor request
 */
static int config_buf(struct usb_gadget *gadget, u8 *buf, u8 type, unsigned index)
{
	int	len;

	UAV_DBG_MSG("\n");

	// 'index' input param tells us which configuration: 0=src_sink, 1=loopback
	// current we support only one (index=0)
	if (index > 0 ) {
		UAV_DBG_ERR("Error: index too big\n");  
		return -EINVAL;
	}

	len = uav_gadget_fill_config_buf(&config_desc_main, buf, USB_BUFSIZ);
	UAV_DBG_MSG("config_buff:w_length=%d\n",len);
	return len;
}

/**
 * allocate a request and also allocate buffer
 * For future use.
 */
#if 0
static struct usb_request * alloc_ep_req(struct usb_ep *ep, unsigned length)
{
	struct usb_request *req;

	req = usb_ep_alloc_request(ep, GFP_ATOMIC);
	if (req) {
		req->length = length;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
		//req->buf = kmalloc(length, GFP_ATOMIC  );
		req->buf = dma_alloc_coherent (NULL, length, &req->dma, GFP_ATOMIC);
#else
		req->buf = usb_ep_alloc_buffer(ep, length, &req->dma, GFP_ATOMIC);
#endif

		if (!req->buf) 	{
			usb_ep_free_request(ep, req);
			req = NULL;
		}
	}
	return req;
}
#endif
/**
 * free buffer associated with the request and the request itself
 */
static void free_ep_req(struct usb_ep *ep, struct usb_request *req)
{
	if (req->buf)
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
		//kfree(req->buf);
		dma_free_coherent (NULL, req->length, req->buf, req->dma);
#else
		usb_ep_free_buffer(ep, req->buf, req->dma, req->length); // usb_gadget.h inline
#endif	
	usb_ep_free_request(ep, req);
}


/**
 * Init ep_set queue
*/
int req_list_init(struct properties_set *ep_set, struct usb_ep *ep)
{
	INIT_LIST_HEAD(&ep_set->queue);
	return 0;
}

/**
 * get request from the list. This is empty requst for IN EPs and processed request for OUT EPs
 */

struct usb_request * req_list_get(struct properties_set *ep_set)
{
	struct uvc_mg3500 *gm = ep_set->ep->driver_data;	
	struct req_pool *v = 0;
	struct usb_request *req = 0;
	spin_lock(&gm->lock);

	if(!list_empty(&ep_set->queue)) {
		v = list_first_entry(&ep_set->queue, typeof(*v), list);
		if(v) {
			list_del_init(&v->list);
			req =  v->req;
		}
	} else {
		ep_set->ep_waitq_flag = 0;
	}
	spin_unlock(&gm->lock);
	UAV_DBG_STRM("req=%p v=%p\n", req, v);
	return req;
}

/**
 * remove pending requests from OTG
*/

void req_list_deque_all(struct properties_set *ep_set)
{
}

/**
 * que the request to OTG
 */
int req_list_put(struct properties_set *ep_set, struct usb_request *_req)
{
	struct uvc_mg3500 *gm = ep_set->ep->driver_data;	
	struct req_pool *reqpool_entry = 0;
	int i;
	UAV_DBG_STRM("req=%p\n", _req);
	spin_lock(&gm->lock);

	for (i=0; i<POOL_ENTRIES; i++) {
		if (ep_set->reqpool[i].req == _req) {
			reqpool_entry = &ep_set->reqpool[i];
			break;
		}
	}
	if(reqpool_entry) {
		UAV_DBG_STRM("req_pool=%p\n", reqpool_entry);
		list_add_tail(&reqpool_entry->list, &ep_set->queue);	
		ep_set->ep_waitq_flag = 1;
	} else {
		UAV_DBG_ERR("Corrupted req=%p\n", _req);
	}
	spin_unlock(&gm->lock);
	return 0;
}


/**
 * setup a pool of requests 
 */
static int pool_init(struct properties_set *ep_set, struct usb_ep *ep)
{
	int i;
	int	status;
	struct usb_request *req;
	struct req_pool *reqpool;

	status = 0;
	reqpool = kzalloc(POOL_ENTRIES * sizeof(*reqpool), GFP_KERNEL);
	if (!reqpool)
		return -ENOMEM;

	ep_set->reqpool = reqpool;		// so that we can look it up later 
	
	for (i=0; i<POOL_ENTRIES; i++) {
		req = usb_ep_alloc_request(ep, GFP_ATOMIC);
		if (!req) {
			UAV_DBG_ERR("pool_init can't alloc req\n");
			status = -EIO;
			break;
		}
		
		req->context = (struct properties_set *)ep_set;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
		req->buf = dma_alloc_coherent (NULL, REQ_BUFF_SIZE, &req->dma, GFP_ATOMIC);
#else
		req->buf = usb_ep_alloc_buffer(ep, REQ_BUFF_SIZE, &req->dma, GFP_ATOMIC);
#endif
		if (!req->buf) {
			UAV_DBG_ERR("Error allocating buffer in write\n");
		}
		reqpool[i].inuse = 0;
		reqpool[i].req = req;
	}
	return status;
}

/**
 * shut down pool of requests 
 */
static int pool_exit(struct properties_set *ep_set, struct usb_ep *ep)
{
	int i;
	int	status;
	struct usb_request *req;
	struct req_pool *reqpool;
	status = 0;
	if(ep_set == 0){
		UAV_DBG_MSG("ep_set == 0\n");
		return status;
	}

	if(ep_set->reqpool == 0){
		UAV_DBG_MSG("ep_set->reqpool == 0\n");
		return status;
	}

	reqpool = ep_set->reqpool;
	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].inuse) {
			UAV_DBG_MSG("pool request to unqueue\n");
		}
		req = reqpool[i].req;
		if (req && ep_set->ep) {
			free_ep_req(ep_set->ep, req);
		}
	}
	kfree(reqpool);
	ep_set->reqpool = NULL;
	return status;
}

/**
 * check out an entry
 */
struct usb_request * pool_get_req(struct properties_set *ep_set)
{
	int i;
	struct req_pool *reqpool;
	struct uvc_mg3500 *gm = ep_set->ep->driver_data;	
	struct usb_request *req = 0;	

	reqpool = ep_set->reqpool;
	spin_lock(&gm->lock);
	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].inuse == 0) {
			reqpool[i].inuse = 1;
			UAV_DBG_STRM("pool_get_req #%d\n", i);
			req = reqpool[i].req;
			break;
		}
	}
	if(!req) {
		UAV_DBG_MSG("pool_get_req no entry\n");
		ep_set->ep_waitq_flag = 0;
	}
	spin_unlock(&gm->lock);
	return req; //none available at this time
}

/**
 * check out an entry
 */
void pool_deque_all_reqests(struct properties_set *ep_set)
{
	int i;
	struct req_pool *reqpool;
	struct usb_request *req = 0;	

	UAV_DBG_CFG(" ep=%s\n", ep_set->ep_name);

	reqpool = ep_set->reqpool;

	for (i=0; i<POOL_ENTRIES; i++) {
		if (reqpool[i].inuse == 1) {
			UAV_DBG_MSG(" req#%d\n", i);
			req = reqpool[i].req;
			if(ep_set->ep) {
			usb_ep_dequeue(ep_set->ep, req);
			} else {
				UAV_DBG_ERR(" Not initialized ep=%p req=%p\n", ep_set->ep, req);
			}
			reqpool[i].inuse = 0;
		}
	}
}

/**
 * Helper function to queue all requests, used by bulk OUT end point 
 */

void pool_que_all_reqests(struct properties_set *ep_set)
{
	int i;
	struct req_pool *reqpool;
	struct usb_request	*req = 0;	
	struct usb_ep		*ep;
	int in = (ep_set->ep_desc->bEndpointAddress	& USB_DIR_IN);
	//int wAlignedpktLen = ((ep_set->ep_desc->wMaxPacketSize + 3) >> 2) << 2;
	int wPtLen = ep_set->ep_desc->wMaxPacketSize;
	int req_buf_len = REQ_BUFF_SIZE;

	UAV_DBG_CFG(" ep=%s\n", ep_set->ep_name);
	
	/* Make the read buffer len integer multiple of ep max pkt length and 4 byte aligned*/
	req_buf_len = (REQ_BUFF_SIZE / wPtLen) * wPtLen;

	if(in ) {
		UAV_DBG_CFG("Not OUT Ep #%dn",ep_set->number);
		return;
	}

	reqpool = ep_set->reqpool;
	ep = ep_set->ep;
	req_list_init(ep_set, ep);
	for (i=0; i<POOL_ENTRIES; i++) {
		int result = 0;
		UAV_DBG_CFG(" req#%d\n", i);
		INIT_LIST_HEAD(&reqpool[i].list);
		req = reqpool[i].req;
		req->complete = ep_out_complete;
		req->length = req_buf_len;
		if(ep) {
		result = usb_ep_queue(ep, req, GFP_ATOMIC);
		if (result){
				UAV_DBG_ERR("can't queue req%d for %s\n", i, ep->name);
			return;
			}
		} else {
			UAV_DBG_ERR(" Not initialized ep=%p req=%p\n", ep, req);
		}
		reqpool[i].inuse = 1;
	}
}


/**
 * check back in an entry
 */
int pool_put_req(struct properties_set *ep_set, struct usb_request *req)
{
	int i;
	int done = 0;
	struct req_pool *reqpool;
	
	if(ep_set && ep_set->ep) {
		struct uvc_mg3500 *gm = ep_set->ep->driver_data;	
		reqpool = ep_set->reqpool;
		spin_lock(&gm->lock);
		for (i=0; i<POOL_ENTRIES; i++) {
			if (reqpool[i].req == req) {
				reqpool[i].inuse = 0;
				done = 1;
				ep_set->ep_waitq_flag = 1;
				break;
			}
		}
		spin_unlock(&gm->lock);
		if(!done) {
			UAV_DBG_ERR("Error pool_put_req()\n");
		}
	}
	return 0;
}



/**
 * completion handler for IN Eps
 */
static void ep_in_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct uvc_mg3500 *gm;
	int	status;
	int ep_nb;
	struct properties_set *ep_set;
	
	gm = ep->driver_data;
	status = req->status;
	ep_set = (struct properties_set *)req->context;
	ep_nb = ((struct properties_set *)req->context)->number;
	
	UAV_DBG_STRM("%s: status=%d\n", __FUNCTION__, status);

	pool_put_req(ep_set, req);

	if (status==0) {
	   	wake_up_interruptible((wait_queue_head_t*)&(ep_set->ep_waitq));
	} else {
		switch (status) 
		{
		/* this endpoint is normally active while we're configured */
		case -ECONNABORTED:		/* hardware forced ep reset */
			UAV_DBG_ERR("IN%i complete CONNABORT\n", ep_nb);
			break;
		case -ECONNRESET:		/* request dequeued */
			UAV_DBG_ERR("IN%i complete CONNRESET\n", ep_nb);
			break;
		case -ESHUTDOWN:		/* disconnect from host */
			ep_set->state = STATE_EP_DISABLED;
			wake_up_interruptible((wait_queue_head_t*)&(ep_set->ep_waitq));
			UAV_DBG_ERR("IN%i complete SHUTDOWN \n", ep_nb);
			UAV_DBG_ERR("%s gone (%d), %d/%d\n", ep->name, status, req->actual, req->length);
			//free_ep_req(ep, req);	
			return;

		case -EOVERFLOW: // buffer overrun on read means we didn't provide a big enough buffer
			UAV_DBG_ERR("IN%i complete OVERFLOW\n", ep_nb);
			break;
		default:
			UAV_DBG_ERR("IN%i complete DEFAULT\n", ep_nb);
			UAV_DBG_ERR("%s complete --> %d, %d/%d\n", ep->name, status, req->actual, req->length);
			break;
		case -EREMOTEIO:		/* short read */
			UAV_DBG_ERR("IN%i complete REMOTEIO\n", ep_nb);
			break;
		}
	}
}


/**
 * completion handler for OUT Eps
 */
void ep_out_complete(struct usb_ep *ep, struct usb_request *req)
{
	int	status = req->status;
	int ep_nb;
	struct properties_set *ep_set;
	ep_set = (struct properties_set *)req->context;
	ep_nb = ((struct properties_set *)req->context)->number;


	// we get here when we have received enough packets to make one 4096(buflen) byte chunk 
	// resync with set_interface or set_config
	UAV_DBG_STRM("ep%iout req->actual=%d\n", ep_nb, req->actual);
	if (status==0) {
		// req->buf has capacity req->length, and contains req->actual bytes 
		// update the current req struct address for use by read
		// ep_set->ep_req = req;
		req_list_put(ep_set, req);
		wake_up_interruptible((wait_queue_head_t*)&(ep_set->ep_waitq));
	}
	else {
		switch (status) 
		{
		case -EOVERFLOW: // buffer overrun on read means we didn't provide a big enough buffer
			UAV_DBG_MSG("OUT%i complete OVERFLOW\n", ep_nb);
		default:
			UAV_DBG_MSG("OUT%i complete DEFAULT\n", ep_nb);
			UAV_DBG_MSG("%s complete --> %d, %d/%d\n", ep->name, status, req->actual, req->length);
		case -EREMOTEIO:		/* short read */
			UAV_DBG_MSG("OUT%i complete REMOTEIO\n", ep_nb);
			break;

		/* this endpoint is normally active while we're configured */
		case -ECONNABORTED:		/* hardware forced ep reset */
			UAV_DBG_MSG("OUT%i complete CONNABORT\n", ep_nb);
		case -ECONNRESET:		/* request dequeued */
			UAV_DBG_MSG("OUT%i complete CONNRESET\n", ep_nb);
		case -ESHUTDOWN:		/* disconnect from host */
			UAV_DBG_MSG("OUT%i complete SHUTDOWN \n", ep_nb);
			UAV_DBG_MSG("%s gone (%d), %d/%d\n", ep->name, status, req->actual, req->length);
			//free_ep_req(ep, req);	
			return;
		}
		
		status = usb_ep_queue(ep, req, GFP_ATOMIC);
		if (status) 
		{
			UAV_DBG_ERR( "kill %s:  resubmit %d bytes --> %d\n", ep->name, req->length, status);
			usb_ep_set_halt(ep);
		}		
	}
}


/**
 * part of the uvc_mg3500_set_config()
 * This is called for ISOCH end points when alternate setting is noz-zero
 */
static int config_endpoint(struct uvc_mg3500 *gm, gfp_t gfp_flags, struct  properties_set  *ep_set)
{
	int	result = 0;
	struct usb_ep	  *ep = NULL;
	struct usb_gadget *gadget = gm->gadget;
	int in = 0;
	
	UAV_DBG_CFG(": %s\n", ep_set->ep_name);

	ep = usb_get_pcd_ep(gadget, ep_set->ep_name);
	if(ep != NULL)	{
		in = (ep_set->ep_desc->bEndpointAddress	& USB_DIR_IN);
	 	ep->driver_data = gm;
		ep_set->ep = ep;

		if (in) {
			UAV_DBG_MSG("%d\n",__LINE__);
			if(ep_set->state !=  STATE_EP_READY) {
			result = usb_ep_enable(ep, ep_set->ep_desc);
			if (result != 0){
					UAV_DBG_ERR("can't enable EP name=%s \n", ep->name);
				result = -EIO;
			}
		} else  {
				UAV_DBG_ERR("Skipping redundant enabling of %s \n", ep->name);
			}
		} else  {
			UAV_DBG_MSG("%d\n",__LINE__);
			if(ep_set->state !=  STATE_EP_READY) {
			result = usb_ep_enable(ep, ep_set->ep_desc);
			if (result != 0){
					UAV_DBG_ERR("can't enable EP name=%s \n", ep->name);
				result = -EIO;
			} else {
					// work-around for OTG PING NAK issue.
					// Queue buffers ready for next session.
#ifdef EN_PRE_QUEUE_EP_BUFFERS
					pool_que_all_reqests(ep_set);
#endif
				}
			} else {
				UAV_DBG_ERR("Skipping redundant enabling of %s \n", ep->name);
			}
		}
		ep_set->state = STATE_EP_READY;
	} else {
		UAV_DBG_MSG("Can not find %s\n",ep_set->ep_name);
		result = -EIO;
	}

	return result;
}
#if 0
/**
 * helper function flush the data) 
 */
static void flush_ep_data(struct properties_set  *ep_set)
{
	UAV_DBG_CFG(":%s\n",ep_set->ep_name);
	if(ep_set->ep && ep_set->ep_desc) {
		usb_ep_disable(ep_set->ep);
		usb_ep_enable(ep_set->ep, ep_set->ep_desc);
	}
}
#endif
/**
 * helper function used by uvc_mg3500_setup() 
 */
static int reset_endpoint(struct uvc_mg3500 *gm,  struct  properties_set  *ep_set)
{
	UAV_DBG_CFG(":%s\n",ep_set->ep_name);
	if (gm->config == 0)
		return 0;

	UAV_DBG_MSG("%d\n",__LINE__);

	ep_set->state = STATE_EP_DISABLED;
	if (ep_set->ep) {
		usb_ep_disable(ep_set->ep);
		ep_set->ep = NULL;
	}
#if 0
	gm->config = 0;
	del_timer(&gm->resume);
#endif
	return 0;
}

/**
 * Handles Set Interface 
 */
int configure_interface(struct usb_gadget *gadget, int intf, int alt_setting)
{
	int value = 0;
	struct uvc_mg3500 *gm = get_gadget_data(gadget);
	UAV_DBG_CFG(":%d intf=%d alt_setting=%d\n",__LINE__,intf, alt_setting);
	switch(intf){
#if HAS_VID_CAP ==  1
		case UVC_CTRL_INTERFACE_INDEX:
			{
#ifdef EN_EP4_IN
				struct usb_endpoint_descriptor *pEp = uvc_cap_get_int_decsc();
				// TODO :obtain through mapping
				ep4_in_set.ep_desc = pEp;
				return config_endpoint(gm, GFP_ATOMIC, &ep4_in_set);
#endif
			}
			break;
		case UVC_CAP_STRM_IN_INDEX:
			{
				struct usb_endpoint_descriptor *pEp = NULL;
				UAV_DBG_MSG(" UVC_CAP_STRM_IN_INDEX\n");
				pEp = get_endpoint_descriptor(intf, alt_setting);
				if(pEp) {
					if((pEp->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
						if(alt_setting == 0){
							return reset_endpoint(gm, &ep1_in_set);
						} else {
							struct usb_endpoint_descriptor *pEp = get_endpoint_descriptor(intf, alt_setting);
							if(pEp) {
								ep1_in_set.ep_desc = pEp;
								return config_endpoint(gm, GFP_ATOMIC, &ep1_in_set);
							}
						}
					} else {
						UAV_DBG_MSG("Configuring end point %d in for bulk transfer\n", ep1_in_set.number);
						/* Handle BULK end point */
						ep1_in_set.ep_desc = pEp;
						return config_endpoint(gm, GFP_ATOMIC, &ep1_in_set);
					}
				} else {
					UAV_DBG_ERR("No cap endpoint descriptor\n");
					value = -1;
				}

			}
			break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_CTRL_IN_INDEX:
			// TODO
			break;
		case UVC_ENC_STRM_IN_INDEX:
		{
			struct usb_endpoint_descriptor *pEp = NULL;
			UAV_DBG_MSG("UVC_ENC_STRM_IN_INDEX\n");
			pEp = get_tr_endpoint_descriptor(intf, alt_setting);
			if(pEp) {
				if((pEp->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
					UAV_DBG_MSG("Configuring end point %d in for ISOCH transfer alsetting=%d\n", ep2_in_set.number,alt_setting);
					/* Handle ISOCH end point */
					if(alt_setting == 0){
						return reset_endpoint(gm, &ep2_in_set);
					} else {
						ep2_in_set.ep_desc = pEp;
						return config_endpoint(gm, GFP_ATOMIC, &ep2_in_set);

					}
				}else {
					UAV_DBG_MSG("Configuring end point %d in for bulk transfer\n", ep2_in_set.number);
					/* Handle BULK end point */
					ep2_in_set.ep_desc = pEp;
					return config_endpoint(gm, GFP_ATOMIC, &ep2_in_set);
				}
			} else {
				UAV_DBG_ERR("No trans enc endpoint descriptor\n");
				value = -1;
			}
		}
		break;
		case UVC_TRANS_CTRL_OUT_INDEX:
			UAV_DBG_ERR("UVC_TRANS_CTRL_OUT_INDEX not implemented\n");
			break;
		case UVC_TRANS_STRM_OUT_INDEX:
		{
			struct usb_endpoint_descriptor *pEp = get_tr_endpoint_descriptor(intf, alt_setting);
			UAV_DBG_MSG("UVC_TRANS_STRM_OUT_INDEX\n");
			if(pEp) {
				if((pEp->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
					/* Handle ISOCH end point */
					UAV_DBG_MSG("Configuring end point %d out for ISOCH transfer alsetting=%d\n", ep2_in_set.number,alt_setting);
					if(alt_setting == 0){
						return reset_endpoint(gm, &ep2_out_set);
					} else {
						ep2_out_set.ep_desc = pEp;
						return config_endpoint(gm, GFP_ATOMIC, &ep2_out_set);
					}
				}else {
					/* Handle BULK end point */
					UAV_DBG_MSG("Configuring end point %d out for bulk transfer\n", ep2_in_set.number);
					ep2_out_set.ep_desc = pEp;
					return config_endpoint(gm, GFP_ATOMIC, &ep2_out_set);
				}
			} else {
				UAV_DBG_ERR("No tran dec endpoint descriptor\n");
				value = -1;
			}
		}
		break;
			break;
#endif

#if HAS_AUD_CAP ==  1
		case UAC_CTRL_INTERFACE_INDEX:
			gm->alt_setting[UAC_CTRL_INTERFACE_INDEX] = 0;
			value = 0;
		break;

		case UAC_STRM_INTERFACE_INDEX:
		{
			struct usb_endpoint_descriptor *pEp = NULL;
			gm->alt_setting[UAC_STRM_INTERFACE_INDEX] = alt_setting;
			/* Reset Current Interface and Endpoint*/
            if (alt_setting == 0)  {
				reset_endpoint(gm, &ep3_in_set);
				value = 0;
            } else if ( (pEp = uac_get_endp_desc( alt_setting )) != NULL) {
				ep3_in_set.ep_desc = pEp;
				config_endpoint(gm, GFP_ATOMIC, &ep3_in_set);
				value = 0;
			} else {
				value = -1;
			}
		}
		break;
#endif
		default:
			UAV_DBG_ERR("unsupported interface %d\n", intf);
			value = -1;//ignore unknown request
			break;

	}
	return value;
}


/**
 * part of the uvc_mg3500_set_config()
 */
static int set_config_desc_main(struct uvc_mg3500 *gm, gfp_t gfp_flags)
{
	int	result = 0;
	/* Do nothing for now */
	UAV_DBG_MSG("%d\n",__LINE__);
	return result;
}

/**
 * helper function used by uvc_mg3500_setup() 
 */
static void uvc_mg3500_reset_config(struct uvc_mg3500 *gm)
{
	UAV_DBG_CFG("%d\n",__LINE__);
	UAV_DBG_MSG("%d\n",__LINE__);

	/* just disable endpoints, forcing completion of pending i/o.
	 * all our completion handlers free their requests in this case.
	 */
	reset_endpoint(gm, &ep1_in_set);
	reset_endpoint(gm, &ep2_in_set);
	reset_endpoint(gm, &ep2_out_set);
	reset_endpoint(gm, &ep2_out_set);
	reset_endpoint(gm, &ep3_in_set);
#ifdef EN_EP4_IN
	reset_endpoint(gm, &ep4_in_set);
#endif
	gm->config = 0;
	gm->vid_cap_cfg_changed = 0;
	gm->aud_codec_cfg_changed = 0;
	gm->pu_cfg_changed = 0;
	gm->camera_cfg_changed = 0;
	del_timer(&gm->resume);
}


/**
 * helper function used by uvc_mg3500_setup() to setup the configuration
 *
 * change our operational config.  this code must agree with the code
 * that returns config descriptors, and altsetting code.
 *
 * it's also responsible for power management interactions. some
 * configurations might not work with our current power sources.
 *
 * note that some device controller hardware will constrain what this
 * code can do, perhaps by disallowing more than one configuration or
 * by limiting configuration choices (like the pxa2xx).
 */
static int uvc_mg3500_set_config(struct uvc_mg3500 *gm, unsigned number, gfp_t gfp_flags)
{
	int result = 0; 
	struct usb_gadget *gadget = gm->gadget;

	UAV_DBG_CFG("%d\n",__LINE__);

	if (number == gm->config){
		UAV_DBG_MSG("set config is already set...\n");
		return 0;
	}

	uvc_mg3500_reset_config(gm);

	switch(number) 
	{
		case CONFIG_VALUE_MAIN:
			result = set_config_desc_main(gm, gfp_flags);
			break;
		default:
			result = -EINVAL;
			/* FALL THROUGH */
		case 0:
			return result;
	}
		
	if (result) {
		uvc_mg3500_reset_config(gm);
	} else 	{
		char *speed;

		switch (gadget->speed) 
		{
		case USB_SPEED_LOW:	 speed = "low";  break;
		case USB_SPEED_FULL: speed = "full"; break;
		case USB_SPEED_HIGH: speed = "high"; break;
		default: 			 speed = "?";    break;
		}

		gm->config = number;
		UAV_DBG_CFG("%s speed config #%d\n", speed, number);
	}

	/* Config Default interface */
	UAV_DBG_MSG("*** Configuring Default Settings ***\n");
#if HAS_VID_CAP ==  1
	configure_interface(gadget, UVC_CAP_STRM_IN_INDEX, 0);
#endif
#if HAS_VID_TRANS ==  1
	configure_interface(gadget, UVC_ENC_STRM_IN_INDEX, 0);
	configure_interface(gadget, UVC_TRANS_STRM_OUT_INDEX, 0);
#endif
#if HAS_AUD_CAP ==  1
	configure_interface(gadget, UAC_STRM_INTERFACE_INDEX, 0);
#endif

	return result;
}

/**
 * installed by uvc_mg3500_bind(), and invoked by uvc_mg3500_setup
 */
static void uvc_mg3500_setup_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct uvc_mg3500 *gm = ep->driver_data;	
	if(gm->is_defered_ctrl) {
		int value = usb_dispatch_non_standard_setup_request(req, &gm->defered_ctrl);
		UAV_DBG_CFG("processed defered ctrl_request result=%d\n\n", value);
	} else {
		// Do nothing
		UAV_DBG_CFG("setup complete --> %d, %d/%d\n\n", req->status, req->actual, req->length);
	}
}

/**
 * This function will handle class requests
 */

int usb_dispatch_class_request(
	char *req_buf, 
	const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u16	w_index = __le16_to_cpu(ctrl->wIndex);
	int intf_num = (w_index & 0x00FF);
	UAV_DBG_CFG("intf_num=%d\n", intf_num);
	switch(intf_num){
#if HAS_VID_CAP ==  1
		case UVC_CTRL_INTERFACE_INDEX:
			return handle_uvc_cap_control(req_buf, ctrl);
			break;

		case UVC_CAP_STRM_IN_INDEX:
			return handle_uvc_stream_if_req(req_buf, ctrl);
			break;
#endif
#if HAS_VID_TRANS == 1
		case UVC_ENC_CTRL_IN_INDEX:
			return handle_uvc_enc_control(req_buf, ctrl);
			break;
		case UVC_ENC_STRM_IN_INDEX:
			return handle_uvc_stream_if_req(req_buf, ctrl);
			break;
		case UVC_TRANS_CTRL_OUT_INDEX:
			return handle_uvc_dec_control(req_buf, ctrl);
			break;

		case UVC_TRANS_STRM_OUT_INDEX:
			return handle_uvc_stream_if_req(req_buf, ctrl);
			break;
#endif

#if HAS_AUD_CAP ==  1
		case UAC_CTRL_INTERFACE_INDEX:
			UAV_DBG_CFG("Class request Audio Control Interface type 0x%02X\n", ctrl->bRequestType );
			value = handle_uac_cap_control(req_buf,ctrl);
			break;

		case UAC_STRM_INTERFACE_INDEX:
			UAV_DBG_CFG("Class request Audio Stream Interface type 0x%02X\n", ctrl->bRequestType );
			value = 0;//handle_uac_audio_control(req_buf,ctrl);
			break;
#endif
		default:
			UAV_DBG_ERR("unknown interface %d\n",intf_num);
			value = 0;//ignore unknown request
			break;
	}
	return value;
}


int usb_dispatch_non_standard_setup_request(struct usb_request	*req, const struct usb_ctrlrequest *ctrl)
{
	int	value = -1;
	__u8 bRequestType = ctrl->bRequestType & USB_TYPE_MASK ;
	switch(bRequestType)
	{
		case USB_TYPE_CLASS:
			UAV_DBG_CFG("Class Type Request \n");
			value = usb_dispatch_class_request(req->buf, ctrl);
			UAV_DBG_CFG("Class Type Request %d \n", value);
			break;
		case USB_TYPE_VENDOR:
			UAV_DBG_CFG("VENDOR  request not implemented");
			break;
		default:
			UAV_DBG_CFG("Unknown non-standard request");
			break;
	}
	return value;
}

/**
 *  Handles standard setup requests
 *  returns length of data written to the buf, -1 on failure.
 */

int usb_dispatch_standard_setup_request (__u8 *buf, const struct usb_ctrlrequest *setup)
{
	int		len = 0;	// TODO : should be -1

	UAV_DBG_CFG( "SETUP bRequestType=%02x setup->bRequest=%02x wValue=%04x wIndex=%04x %d\n", setup->bRequestType, 
			setup->bRequest, setup->wValue, setup->wIndex,	setup->wLength);

	switch (setup->bRequest) {	/* usb 2.0 spec ch9 requests */
		case USB_REQ_SET_INTERFACE:
			if (setup->bRequestType == USB_RECIP_INTERFACE){
				switch(setup->wIndex){
#if HAS_VID_CAP == 1
					case UVC_CTRL_INTERFACE_INDEX: 
					{
						UAV_DBG_ERR("Ignoring USB_REQ_SET_INTERFACE on UVC_CTRL_INTERFACE_INDEX\n");
						len = 0;
					}
					break;

					case UVC_CAP_STRM_IN_INDEX:
					{
#if HAS_VIDCAP_ISOCH == 1
						len = uvc_cap_set_interface(setup);
#endif
					}
					break;
#endif	// HAS_VID_CAP
#if HAS_VID_TRANS == 1
					case UVC_ENC_CTRL_IN_INDEX:
					{
						UAV_DBG_ERR("Ignoring USB_REQ_SET_INTERFACE on UVC_ENC_CTRL_IN_INDEX\n");
						len = 0;
					}
					break;
					case UVC_ENC_STRM_IN_INDEX:
					{
#if HAS_ENC_ISOCH == 1
						len = uvc_cap_set_interface(setup);
#endif
					}
					break;

					case UVC_TRANS_CTRL_OUT_INDEX:
					{
						UAV_DBG_ERR("Ignoring USB_REQ_SET_INTERFACE on UVC_TRANS_CTRL_OUT_INDEX\n");
						len = 0;
					}
					break;
					case UVC_TRANS_STRM_OUT_INDEX:
					{
#if HAS_DEC_ISOCH == 1
						len = uvc_dec_set_interface(setup);
#endif
					}
					break;

#endif	// HAS_VID_TRANS

#if HAS_AUD_CAP ==  1
					case UAC_CTRL_INTERFACE_INDEX:
						len = 0;
						break;

					case UAC_STRM_INTERFACE_INDEX:
						len = uac_cap_set_interface(setup);
				//		DbgOut(""); // to see ..
					//	setup->wValue = 0 for stop
					//	setup->wValue = non zero for start
					len = 0;
					break;

#endif
					default:
					break;
				}
			}
			break;
	}
	return len;
}


/**
 * The setup() callback implements all the ep0 functionality that's
 * not handled lower down, in hardware or the hardware driver (like
 * device and endpoint feature flags, and their status).  It's all
 * housekeeping for the gadget function we're implementing.  Most of
 * the work is in config-specific setup.
 */
static int uvc_mg3500_setup(struct usb_gadget *gadget, const struct usb_ctrlrequest *ctrl)
{
	struct uvc_mg3500 *gm = get_gadget_data(gadget);
	struct usb_request	*req = gm->req;
	int	value = -EOPNOTSUPP;
	u16	w_index = le16_to_cpu(ctrl->wIndex);
	u16	w_value = le16_to_cpu(ctrl->wValue);
	u16	w_length = le16_to_cpu(ctrl->wLength);

	UAV_DBG_CFG("===Start====\n");
	UAV_DBG_CFG("bRequestType=0x%x bRequest=0x%02X w_value=0x%x w_index=0x%x\n",ctrl->bRequestType, ctrl->bRequest, w_value, w_index);

	gm->is_defered_ctrl = 0;
	// usually this stores reply data in the pre-allocated ep0 buffer,
	// but config change events will reconfigure hardware.

	req->zero = 0;
	switch (ctrl->bRequest) 
	{
	case USB_REQ_GET_DESCRIPTOR:
		if (ctrl->bRequestType != USB_DIR_IN)
			goto unknown;
		switch (w_value >> 8) 
		{
		
		case USB_DT_DEVICE:
			value = min(w_length, (u16)sizeof(device_desc));
			memcpy(req->buf, &device_desc, value);
			break;

		case USB_DT_DEVICE_QUALIFIER:	// this is for DUALSPEED 
			if (!gadget->is_dualspeed){
				UAV_DBG_CFG("USB_DT_DEVICE_QUALIFIER:!gadget->is_dualspeed\n");
				break;
			}
			value = min(w_length, (u16)sizeof(dev_qualifier));
			memcpy(req->buf, &dev_qualifier, value);
			break;

		case USB_DT_OTHER_SPEED_CONFIG:	// this is for DUALSPEED 
			if (!gadget->is_dualspeed) {
				UAV_DBG_CFG("USB_DT_OTHER_SPEED_CONFIG:!gadget->is_dualspeed\n");
				break;
			}
			// FALLTHROUGH

		case USB_DT_CONFIG:
			value = config_buf(gadget, req->buf, w_value >> 8, w_value & 0xff);
			if (value >= 0)
				value = min(w_length, (u16)value);
			break;

		case USB_DT_STRING:
			/* wIndex == language code.
			 * this driver only handles one language, you can
			 * add string tables for other languages, using
			 * any UTF-8 characters
			 */
			value = get_string(w_value & 0xff, req->buf);
			if (value >= 0){
				value = min(w_length, (u16)value);
			} else {

				UAV_DBG_CFG("return w_value:USB_DT_STRING%d\n", value);
			}
			break;
		}
		break;

	/* currently two configs, two speeds */
	case USB_REQ_SET_CONFIGURATION:
		if (ctrl->bRequestType != 0)
			goto unknown;
		spin_lock(&gm->lock);
		value = uvc_mg3500_set_config(gm, w_value, GFP_ATOMIC);
		spin_unlock(&gm->lock);
		break;

	case USB_REQ_GET_CONFIGURATION:
		if (ctrl->bRequestType != USB_DIR_IN)
			goto unknown;
		*(u8*)req->buf = gm->config;
		value = min(w_length, (u16)1);
		break;

	case USB_REQ_SET_INTERFACE:
		if (ctrl->bRequestType != USB_RECIP_INTERFACE)
			goto unknown;
		spin_lock(&gm->lock);
		gm->alt_setting[w_index] = w_value;
		configure_interface(gadget, w_index, w_value);
		value = usb_dispatch_standard_setup_request(req->buf, ctrl);
		spin_unlock(&gm->lock);
		break;

	case USB_REQ_GET_INTERFACE:
		if (ctrl->bRequestType != (USB_DIR_IN|USB_RECIP_INTERFACE))
			goto unknown;
		if (!gm->config)
			break;
		if (w_index >= USB_INTERFACE_INDEX_LAST ) 	{
			UAV_DBG_MSG("\nInterface out of range %d",w_index);
			value = -EDOM;
			break;
		}
		*(u8*)req->buf = gm->alt_setting[w_index];
		value = min(w_length, (u16)1);
		break;

	default:
		/* 
		** defer processing requests that involve OUT transfer until uvc_mg3500_setup_complete
		** This may cause problem for probe/commit rejection.
		*/
		if(!(ctrl->bRequestType & USB_DIR_IN) && w_length){
			gm->is_defered_ctrl = 1;
			memcpy(&gm->defered_ctrl, ctrl, sizeof(struct usb_ctrlrequest));
			value = 0;
		} else {
			value = usb_dispatch_non_standard_setup_request(req, ctrl);
		}
		break;

	}
unknown:
	if (value == -1) {
		UAV_DBG_CFG("unknown control req (bRequestType=0x%02X bRequest=0x%02X\n"
			"value=0x%04X index=0x%04X length=%d)\n", ctrl->bRequestType, ctrl->bRequest,
				w_value, w_index, w_length);
	}
	/* respond with data transfer before status phase? */
	if (value >= 0) {
		req->length = value;
		req->zero = value < w_length;
		value = usb_ep_queue(gadget->ep0, req, GFP_ATOMIC);
		if (value < 0) {
			UAV_DBG_CFG("ep_queue --> %d\n", value);
			req->status = 0;
			uvc_mg3500_setup_complete(gadget->ep0, req);
		}
	}
	else {
		UAV_DBG_CFG("at end of uvc_mg3500_setup(), value is negative --> stalls\n");
	}

	/* device either stalls (value < 0) or reports success */
	return value;
}
/**
 * Handles disconnect callback from PCD
 */
static void uvc_mg3500_disconnect(struct usb_gadget *gadget)
{
	struct uvc_mg3500 *gm = get_gadget_data(gadget);
	unsigned long flags;

	UAV_DBG_CFG("%d\n", __LINE__);

	spin_lock_irqsave(&gm->lock, flags);
	uvc_mg3500_reset_config(gm);

	// a more significant application might have some non-usb
	// activities to quiesce here, saving resources like power
	// or pushing the notification up a network stack.
	spin_unlock_irqrestore(&gm->lock, flags);

	// next we may get setup() calls to enumerate new connections;
	// or an unbind() during shutdown (including removing module).
}

#if 0
static void uvc_mg3500_autoresume(unsigned long _gm)
{
	struct uvc_mg3500 *gm = (struct uvc_mg3500 *) _gm;
	int status;

	// normally the host would be woken up for something
	// more significant than just a timer firing...
	if (gm->gadget->speed != USB_SPEED_UNKNOWN) {
		status = usb_gadget_wakeup(gm->gadget);
		UAV_DBG_CFG("wakeup --> %d\n", status);
	}
}
#endif

void destroy_ep_file (struct uvc_mg3500 *dev, struct properties_set *ep_set);

void destroy_usbav_files (struct uvc_mg3500 *gm)
{
	UAV_DBG_MSG("\n");
	if(gm) {
	destroy_ep_file(gm, &ep1_in_set);
	destroy_ep_file(gm, &ep2_in_set);
	destroy_ep_file(gm, &ep2_out_set);
	destroy_ep_file(gm, &ep3_in_set);
	}
}

/**
 * Unbind EPs
 */
static void uvc_mg3500_unbind(struct usb_gadget *gadget)
{
	struct uvc_mg3500 *gm = get_gadget_data(gadget);

	/* we've already been disconnected ... no i/o is active */
	if (gm->req) {
		gm->req->length = USB_BUFSIZ;
		free_ep_req(gadget->ep0, gm->req);
	}

	if (0 != pool_exit(&ep1_in_set, ep1_in_set.ep)) {
		UAV_DBG_ERR("pool_exit error on %s\n", ep1_in_set.ep->name);
	}

	if (0 != pool_exit(&ep2_in_set, ep2_in_set.ep)) {
		UAV_DBG_ERR("pool_exit error on %s\n", ep2_in_set.ep->name);
	}
	if (0 != pool_exit(&ep3_in_set, ep3_in_set.ep)) {
		UAV_DBG_CFG("pool_exit error on %s\n", ep2_in_set.ep->name);
	}
#ifdef EN_EP4_IN
	if (0 != pool_exit(&ep4_in_set, ep4_in_set.ep)) {
		UAV_DBG_CFG("pool_exit error on %s\n", ep2_in_set.ep->name);
	}
#endif	
	del_timer_sync(&gm->resume);
	kfree(gm);
	set_gadget_data(gadget, NULL);
}

			
struct usb_ep * usb_get_pcd_ep(struct usb_gadget *gadget,
										const char *find_name)
{
	struct usb_ep	*ep;
	/* look at endpoints until an unclaimed one looks usable */
	list_for_each_entry(ep, &gadget->ep_list, ep_list) 
	{
		if (0 == strcmp(ep->name, find_name)) {
				return ep;
		}
	}

	/* Fail */
	return NULL;  // no match found
}
static int __init bind_endpoint(struct usb_gadget *gadget, struct properties_set *ep_set, char *ep_name, int in)
{
	struct usb_ep	*ep;
	UAV_DBG_CFG("configure ep=%d name=%s\n", ep_set->number, ep_name);
	ep = usb_get_pcd_ep(gadget,  ep_name);
	if (!ep) {
		UAV_DBG_CFG("Failed to get ep=%d name=%s \n", ep_set->number,ep_name);
		return -1;
	}
	ep_set->ep_name = ep->name;		// assign a str name for this EP

	ep_set->state = STATE_EP_DISABLED;
	ep->driver_data = &ep_set;				// TODO Verify claim
	//if(in) 
	{
		if (0 != pool_init(ep_set, ep)) {
			UAV_DBG_ERR("pool_init error on %s\n", ep->name);
			return -1;
		}
	}
	ep_set->ep = ep;
	init_waitqueue_head (&ep_set->ep_waitq);
	return 0;
}

/**
 * Bind EPs
 */
static int __init uvc_mg3500_bind(struct usb_gadget *gadget)
{
	struct uvc_mg3500 *gm;
	struct usb_ep  *ep;
	{
		//usb_ep_autoconfig_reset
		list_for_each_entry (ep, &gadget->ep_list, ep_list) {
			ep->driver_data = NULL;
		}
	}

	//=========================================================================
	// Configure endpoint 1
	//=========================================================================

	UAV_DBG_CFG("configure ep1 in\n");
	if(bind_endpoint(gadget,  &ep1_in_set, "ep1in", 1) == -1)
		goto epconf_fail;
	//=========================================================================
	// Configure endpoint 2 in
	//=========================================================================
	if(bind_endpoint(gadget,  &ep2_in_set, "ep2in", 1) == -1)
		goto epconf_fail;
	//=========================================================================
	// Configure endpoint 2 out
	//=========================================================================
	if(bind_endpoint(gadget,  &ep2_out_set, "ep2out", 0) == -1)
		goto epconf_fail;

	//=========================================================================
	// Configure endpoint 3
	//=========================================================================
	if(bind_endpoint(gadget,  &ep3_in_set, "ep3in", 1) == -1)
		goto epconf_fail;

	//=========================================================================
	// Configure endpoint 4
	//=========================================================================
#ifdef EN_EP4_IN
	if(bind_endpoint(gadget,  &ep4_in_set, "ep4in", 1) == -1)
		goto epconf_fail;
#endif

	/* ok, we made sense of the hardware ... */
	gm = kzalloc(sizeof(*gm), GFP_KERNEL);
	//same as: kmalloc(sizeof(*gm), SLAB_KERNEL);  if (gm) memset(gm, 0, sizeof(*gm));

	if (!gm)
		return -ENOMEM;

	spin_lock_init(&gm->lock);
	init_waitqueue_head (&gm->codec_cfg_waitq);
	gm->vid_cap_cfg_changed = 0;
	gm->aud_codec_cfg_changed = 0;
	gm->gadget = gadget;
	set_gadget_data(gadget, gm);

	/* preallocate control response and buffer */
	gm->req = usb_ep_alloc_request(gadget->ep0, GFP_KERNEL);
	if (!gm->req)
		goto enomem;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 20)
	//gm->req->buf = kmalloc(USB_BUFSIZ, GFP_KERNEL);
	gm->req->buf = dma_alloc_coherent (NULL, USB_BUFSIZ, &gm->req->dma, GFP_ATOMIC);
#else
	gm->req->buf = usb_ep_alloc_buffer(gadget->ep0, USB_BUFSIZ, &gm->req->dma, GFP_KERNEL);
#endif


	if (!gm->req->buf)
		goto enomem;

	gm->req->complete = uvc_mg3500_setup_complete;

	device_desc.bMaxPacketSize0 = gadget->ep0->maxpacket;	// 64

	/* assume ep0 uses the same value for both speeds ... */
	dev_qualifier.bMaxPacketSize0 = device_desc.bMaxPacketSize0;

// TODO: RAM
#if 0
	if (gadget->is_otg) {
		otg_descriptor.bmAttributes |= USB_OTG_HNP,
		config_desc_main.bmAttributes |= USB_CONFIG_ATT_WAKEUP;
	}
	usb_gadget_set_selfpowered(gadget);

	init_timer(&gm->resume);
	gm->resume.function = uvc_mg3500_autoresume;
	gm->resume.data = (unsigned long)gm;
	if (autoresume) {
		config_desc_main.bmAttributes |= USB_CONFIG_ATT_WAKEUP;
	}
#endif
	gadget->ep0->driver_data = gm;
	the_device = gm;
	//snprintf(usb_str_manufacturer, sizeof(usb_str_manufacturer), "Mobilygen");
	
	return 0;

epconf_fail:
		UAV_DBG_ERR("can't configure EP\n");
		return -ENODEV;

enomem:
	uvc_mg3500_unbind(gadget);
	return -ENOMEM;
}

/**
 * Handles suspend callback from PCD
 */
static void uvc_mg3500_suspend(struct usb_gadget *gadget)
{
	struct uvc_mg3500 *gm = get_gadget_data(gadget);

	if (gadget->speed == USB_SPEED_UNKNOWN)
		return;

	if (autoresume) {
		mod_timer(&gm->resume, jiffies + (HZ * autoresume));
		UAV_DBG_CFG("suspend, wakeup in %d seconds\n", autoresume);
	} 
	else
		UAV_DBG_CFG("suspend\n");
}

/**
 * Handles resume callback from PCD
 */
static void uvc_mg3500_resume(struct usb_gadget *gadget)
{
	struct uvc_mg3500 *gm = get_gadget_data(gadget);

	UAV_DBG_CFG("resume\n");
	del_timer(&gm->resume);
}


/**
 * Gadget driver initilization/callbacks to PCD
 */
static struct usb_gadget_driver uvc_mg3500_driver = {
#ifdef CONFIG_USB_GADGET_DUALSPEED
	.speed		= USB_SPEED_HIGH,	// dual speed (full and high)
#else
	.speed		= USB_SPEED_FULL,	// full speed only
#endif
	.function	= (char*)usb_str_product,
	
	.bind		= uvc_mg3500_bind,
	.unbind		= __exit_p(uvc_mg3500_unbind),

	.setup		= uvc_mg3500_setup,
	.disconnect	= uvc_mg3500_disconnect,

	.suspend	= uvc_mg3500_suspend,
	.resume		= uvc_mg3500_resume,

	.driver 	= {
		.name		= (char*)driver_name,
		.owner		= THIS_MODULE,
	},
};

/**
 * Destroy USBAV File nodes
 */
void destroy_ep_file (struct uvc_mg3500 *dev, struct properties_set *ep_set)
{
	struct inode	*parent;
	struct dentry	*dentry;

	spin_lock_irq (&dev->lock);
	dentry = ep_set->dentry;
	ep_set->dentry = NULL;
	parent = dentry->d_parent->d_inode;

	/* break link to dcache */
	mutex_lock (&parent->i_mutex);
	d_delete (dentry);
	dput (dentry);
	mutex_unlock (&parent->i_mutex);

	spin_unlock_irq (&dev->lock);
}

const struct file_operations ep_operations;
static struct inode *
usbav_create_file (struct super_block *sb, char const *name,
		void *data, const struct file_operations *fops,
		struct dentry **dentry_p);

static int activate_ep_files(struct uvc_mg3500 *dev)
{
	struct properties_set *ep_set;
	ep_set = &ep1_in_set;
	ep_set->inode = usbav_create_file (dev->sb, ep_set->file_name,
			ep_set, &ep_operations, &ep_set->dentry);
	if (!ep_set->inode)
		goto enomem0;

	ep_set = &ep2_in_set;
	ep_set->inode = usbav_create_file (dev->sb, ep_set->file_name,
			ep_set, &ep_operations, &ep_set->dentry);
	if (!ep_set->inode)
		goto enomem0;

	ep_set = &ep2_out_set;
	ep_set->inode = usbav_create_file (dev->sb, ep_set->file_name,
			ep_set, &ep_operations, &ep_set->dentry);
	if (!ep_set->inode)
		goto enomem0;

	ep_set = &ep3_in_set;
	ep_set->inode = usbav_create_file (dev->sb, ep_set->file_name,
			ep_set, &ep_operations, &ep_set->dentry);
	if (!ep_set->inode)
		goto enomem0;

	return 0;

enomem0:
	if(ep_set) {
		UAV_DBG_ERR("Filed to create %d\n", ep_set->number);
	}
	return -ENOMEM;
}

/**
 * This is common routine for all IN end points.
 * The user buffer is split in to OTG transfer buffers(4K max). Each OTG transfer buffer
 * contains ineger number of ep transfer lengths, else a short packet indicator is set.
 * If ep transfer length is not 4-byte aligned, total padding is calculated corresponding
 * to number of packets in each OTG transfer buffer and extra data is copied from user buffer.
 * The application is expected to include these additional padding bytes in the stream.
 */
static ssize_t
ep_write_isoc (struct properties_set *ep_set, const char __user *userbuf, unsigned len)
{
	int value;
	int retval=0, status=0;
	int xferlen; 
	int MaxPktCount;
	struct usb_request *req;

	struct usb_endpoint_descriptor *ep_desc = ep_set->ep_desc;

	UAV_DBG_STRM("ep=%d\n", ep_set->number);
	retval = len;

	if(ep_set->state !=  STATE_EP_READY || !ep_desc || ep_set->pos == FILE_EOS){
		UAV_DBG_ERR("state =%d ep_desc=%p pos=%d\n", ep_set->state, ep_desc, ep_set->pos);
		return -EFAULT;
	}
	while ((len > 0)&&(ep_set->state ==  STATE_EP_READY )) {
		/* Due to OTG issue, we always expect 4 byte aligned data */
		/* If it is not aligned, it causes data corruption */
		int wAlignedpktLen = ((ep_desc->wMaxPacketSize + 3) >> 2) << 2;
		int wPaddingPerPkt = wAlignedpktLen - ep_desc->wMaxPacketSize;
		int wTotalPadding = 0;
		MaxPktCount = REQ_BUFF_SIZE / wAlignedpktLen;
		xferlen = MaxPktCount * wAlignedpktLen;//integer number of packets to avoid short packets when data is available.
		do {
			req = pool_get_req(ep_set);
			if (!req){
				/* Buffer not available. Reset the ep_waitq_flag and wait for buffer availability.*/
				UAV_DBG_STRM("wait for buffer ep=%d\n", ep_set->number);
				value = wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
				if(value != 0 || ep_set->state !=  STATE_EP_READY || ep_set->pos == FILE_EOS) {
					UAV_DBG_ERR("returning due to a signal or ep shutdown\n");
					status = -EFAULT;				// return with error code
					goto exit;
				}
				if (ep_set->ep_waitq_flag < 0) {	// if error reported by handler
					UAV_DBG_ERR("write waitq error\n");
					status = -EFAULT;				// return with error code
					goto exit;
				}
			}
		} while (!req); 

		if(ep_set->state !=  STATE_EP_READY) {
			UAV_DBG_ERR("Aborting transfer\n");
			pool_put_req(ep_set, req);
			goto exit;
		}

		req->context = (struct properties_set *)ep_set;
		req->zero = 0;		
		if(len < xferlen ){
			xferlen = len;
			MaxPktCount = xferlen / wAlignedpktLen;
			if(xferlen % wAlignedpktLen)
				req->zero = 1;
		}
		wTotalPadding = wPaddingPerPkt * MaxPktCount;
		req->length = xferlen - wTotalPadding;
		UAV_DBG_STRM("%s: transfer len(remain)=%d length=%d wPaddingPerPkt=%d wTotalPadding=%d shortpacket=%d\n",__FUNCTION__, len,req->length,wPaddingPerPkt, wTotalPadding, req->zero); 
		if (copy_from_user(req->buf, userbuf, xferlen)) {
			UAV_DBG_ERR("copy_from_user failed\n");
			pool_put_req(ep_set, req);
			retval = -EFAULT;
			goto exit;
		}

		req->complete = ep_in_complete;
		if(ep_set->ep != NULL) {
			status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC); // enqueue I/O request to this EP
			if (status) {
				UAV_DBG_ERR("kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, req->length, status);
				usb_ep_set_halt(ep_set->ep);
			}
		} else {
			UAV_DBG_ERR("error: EP is shutdown\n");
			pool_put_req(ep_set, req);
			retval = -EFAULT;
			goto exit;
		}
		userbuf += xferlen;
		len -= xferlen;
	}
exit:
	UAV_DBG_STRM("ep=%d end\n", ep_set->number);
	return retval;
}

int enc_uvc_hdr(struct properties_set *ep_set, char *buf, int len)
{
	if(ep_set->proc_uvc_hdr && ep_set->first_uvc_pkt && len >= 2){
		ep_set->first_uvc_pkt = 0;
		buf[0] = 0x02;
		ep_set->uvc_fid ^= UVC_STRM_FID;
		buf[1] = UVC_STRM_EOH | UVC_STRM_EOF | ep_set->uvc_fid; // todo
		return 2;
	}
	return 0;
}

/**
 * Add uvc header on payload sie boundaries
 */
static ssize_t
ep_write_bulk_with_uvc_enc (struct properties_set *ep_set, const char __user *userbuf, unsigned len)
{
	int value;
	int retval=0, status=0;
	int xferlen; 
	int MaxPktCount;
	int hdr_len = 0;
	struct usb_request *req;

	struct usb_endpoint_descriptor *ep_desc = ep_set->ep_desc;

	UAV_DBG_STRM("ep=%d\n", ep_set->number);
	ep_set->first_uvc_pkt = 1;
	retval = len;

	if(ep_set->state !=  STATE_EP_READY || !ep_desc || ep_set->pos == FILE_EOS){
		UAV_DBG_ERR("%s():ep=%d ep_set->state !=  STATE_EP_READY || !ep_desc || ep_set->pos==FILE_EOS\n", __FUNCTION__, ep_set->number);
		return -EFAULT;
	}
	while ((len > 0)&&(ep_set->state ==  STATE_EP_READY )) {
		MaxPktCount = REQ_BUFF_SIZE / ep_desc->wMaxPacketSize;
		xferlen = MaxPktCount * ep_desc->wMaxPacketSize;
		do {
			req = pool_get_req(ep_set);
			if (!req){
				/* Buffer not available. Reset the ep_waitq_flag and wait for buffer availability.*/
				UAV_DBG_STRM("wait for buffer ep=%d\n", ep_set->number);
				value = wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
				if(value != 0 || ep_set->state !=  STATE_EP_READY || ep_set->pos == FILE_EOS) {
					UAV_DBG_ERR("%s: returning due to a signal\n",__FUNCTION__);
					status = -EFAULT;				// return with error code
					goto exit;
				}
				if (ep_set->ep_waitq_flag < 0) {	// if error reported by handler
					UAV_DBG_ERR("write waitq error\n");
					status = -EFAULT;				// return with error code
					goto exit;
				}
			}
		} while (!req); 

		if(ep_set->state !=  STATE_EP_READY) {
			UAV_DBG_ERR("Aborting transfer\n");
			pool_put_req(ep_set, req);
			goto exit;
		}

		req->context = (struct properties_set *)ep_set;
		req->zero = 0;		
		hdr_len = enc_uvc_hdr(ep_set, req->buf, len);
		if((len + hdr_len) <= xferlen ){
			xferlen = len + hdr_len;
			req->zero = 1;
		}
		req->length = xferlen;

		UAV_DBG_STRM("%s: transfer len(remain)=%d length=%d  shortpacket=%d\n",__FUNCTION__, len,req->length,req->zero); 
		if (copy_from_user(req->buf + hdr_len, userbuf, xferlen - hdr_len)) {
			UAV_DBG_ERR("copy_from_user failed\n");
			pool_put_req(ep_set, req);
			retval = -EFAULT;
			goto exit;
		}

		req->complete = ep_in_complete;
		if(ep_set->ep != NULL) {
			status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC); // enqueue I/O request to this EP
			if (status) {
				UAV_DBG_ERR("kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, req->length, status);
				usb_ep_set_halt(ep_set->ep);
			}
		} else {
			UAV_DBG_ERR("error: EP is shutdown\n");
			pool_put_req(ep_set, req);
			retval = -EFAULT;
			goto exit;
		}
		userbuf += (xferlen - hdr_len);
		len -= (xferlen - hdr_len);
		ep_set->total_bytes += (xferlen - hdr_len);
	}
exit:
	UAV_DBG_STRM("ep=%d end\n", ep_set->number);
	return retval;
}

/**
 * Calclualtes 4 byte aligned length
 */
int padded_len(int pl_len, int pkt_size)
{
	int aligned_pkt_size = ((pkt_size + 3) >> 2) << 2;
	return ((pl_len / pkt_size) * aligned_pkt_size);
}

int dec_uvc_hdr(struct properties_set *ep_set, char *buf, int len)
{
	if(ep_set->proc_uvc_hdr && ep_set->first_uvc_pkt && len >= 2){
		ep_set->first_uvc_pkt = 0;
		UAV_DBG_STRM("uvc hdr: %d %d\n", buf[0], buf[1]);
		return 2;
	}
	return 0;
}

/** 
 * handle a synchronous OUT bulk/intr/iso transfer
 */
 static ssize_t
ep_read (struct file *fd, char __user *buf, size_t len, loff_t *ptr)
{
	struct properties_set			*ep_set = 0;
	struct usb_endpoint_descriptor	*ep_desc = 0;
	ssize_t							bytesRead = 0;
	struct usb_request				*req = 0;

	int status=0;
	int xferlen; 
	int zlp = 0;
	int hdr_len = 0;
	if(fd && fd->private_data) {
		ep_set = fd->private_data;
		UAV_DBG_STRM("ep=%d\n", ep_set->number);
	} else {
		UAV_DBG_CFG ( "error : Unknown\n");
		return -EFAULT;
	}
	
	ep_desc = ep_set->ep_desc;
	
	if(ep_set->state !=  STATE_EP_READY || !ep_desc || ep_set->pos == FILE_EOS){
		UAV_DBG_ERR("ep=%d state(%d) !=  STATE_EP_READY || !ep_desc pos=%d\n", ep_set->number, ep_set->state, ep_set->pos);
		return -EFAULT;
	}

	UAV_DBG_STRM ( "%s read from %d status= %d\n",ep_set->file_name, len, (int) bytesRead);

	while ((bytesRead < len)&&(ep_set->state ==  STATE_EP_READY )) {
		/* get the partially processed request */
		req = ep_set->ep_req;
		while(!req) {
			/* Buffer not available. wait for buffer availability.*/
			UAV_DBG_STRM("wait for buffer ep=%d\n", ep_set->number);
			status = wait_event_interruptible(ep_set->ep_waitq, ep_set->ep_waitq_flag != 0);
			req = req_list_get(ep_set);
			if(status != 0 || ep_set->state !=  STATE_EP_READY || ep_set->pos == FILE_EOS) {
				UAV_DBG_ERR("ep=%d statis =%d state(%d) !=  STATE_EP_READY || !ep_desc pos=%d\n", ep_set->number, status, ep_set->state, ep_set->pos);
				bytesRead = -EFAULT;				// return with error code
				goto exit;
			}
			if (ep_set->ep_waitq_flag < 0) {	// if error reported by handler
				UAV_DBG_ERR("write waitq error\n");
				bytesRead = -EFAULT;				// return with error code
				goto exit;
			}
			/* 
			 * Change the length to padded length. Workaround for OTG 4byte alignment requirement.
			 * This will be used only in case of EP wMaxPacketSize is not 4 byte aligned
			 */
			if(req && ep_desc->wMaxPacketSize % 4) {
				req->actual = padded_len(req->actual, ep_desc->wMaxPacketSize);
				UAV_DBG_STRM("Len after padding=%d\n",req->actual);
			}
		}

		if(ep_set->state !=  STATE_EP_READY) {
			UAV_DBG_ERR("Aborting transfer\n");
			goto exit;
		}

		xferlen = req->actual;
		/* 
		 * Ignore the zero length packets sent by the host in the beginning of the session using 
		 * the variable ep_set->total_bytes
		 */
		if (req->actual)
		{
			if (req->zero &&  ep_set->total_bytes){
				UAV_DBG_STRM("zlp@%d\n", bytesRead);
				zlp = 1;
				req->zero = 0;
			}
		}
		else
		{
			UAV_DBG_WARN("zlp received with zero length\n");
		}

		if(xferlen > (len - bytesRead)) {
			/* Not enough user buffer */
			UAV_DBG_ERR("!!! Not enough user buffer !!!\n");
			xferlen = len - bytesRead;
		}
		hdr_len = dec_uvc_hdr(ep_set, req->buf, xferlen);
		UAV_DBG_STRM("avail buff space=%d req->length=%d req->actual=%d req=%p\n",(len-bytesRead),req->length,req->actual, req); 
		if (copy_to_user(buf, req->buf + hdr_len, xferlen - hdr_len)) {
			UAV_DBG_ERR("copy_to_user failed\n");
			bytesRead = -EFAULT;
			goto exit;
		}
		/* Enque the request again */
		if(ep_set->ep != NULL) {

			if (xferlen == req->actual) {
				status = usb_ep_queue(ep_set->ep, req, GFP_ATOMIC); // enqueue I/O request to this EP
				if (status) {
					UAV_DBG_ERR("kill %s:  resubmit %d bytes --> %d\n", ep_set->ep->name, req->length, status);
					usb_ep_set_halt(ep_set->ep);
				}
				ep_set->ep_req = 0;
				req = 0;
			} else {
				/* User buffer is full and data still remains in req */
				int rem_len = req->actual - xferlen;
				memmove(req->buf, req->buf + xferlen, rem_len);
				req->actual = rem_len;
				ep_set->ep_req = req;
				bytesRead += (xferlen - hdr_len);
				ep_set->total_bytes += (xferlen - hdr_len);
				UAV_DBG_STRM("rem_len=%d\n",rem_len);
				goto exit;
			}
		} else {
			UAV_DBG_ERR("error: EP is shutdown\n");
			bytesRead = -EFAULT;
			goto exit;
		}
		buf += (xferlen - hdr_len);
		bytesRead += (xferlen - hdr_len);
		ep_set->total_bytes += (xferlen - hdr_len);

		if(zlp){
			ep_set->first_uvc_pkt = 1;
			break;
		}
	}
exit:
	UAV_DBG_STRM("bytesRead=%d total_bytesd=%d\n", bytesRead, ep_set->total_bytes);
	return bytesRead;
}

/* handle a synchronous IN bulk/intr/iso transfer */
static ssize_t
ep_write (struct file *fd, const char __user *buf, size_t len, loff_t *ptr)
{
	
	if(fd && fd->private_data) {
		struct properties_set		*ep_set = fd->private_data;
		ssize_t			value;
		UAV_DBG_STRM("ep=%d\n", ep_set->number);
		value = 0;
		if(ep_set->proc_uvc_hdr)
			value = ep_write_bulk_with_uvc_enc(ep_set, buf, len);
		else
			value = ep_write_isoc (ep_set, buf, len);
		UAV_DBG_STRM ( "%s write from %d status= %d\n",ep_set->file_name, len, (int) value);
		return value;
	} else {
		UAV_DBG_ERR ( "error : Unknown\n");
		return -EFAULT;
	}
}


/**
 * For bulk endpoints set the encoder stream State to STOP 
 */

void set_eos(struct properties_set *ep_set, int val)
{
	vid_codec_config_t Param = {0};
	UAV_DBG_CFG ( ": Processing EoS for %s\n",  ep_set->file_name);

	if (0 == strcmp(ep_set->file_name, FILE_VID_TRANS_ENC)){
		UAV_DBG_CFG ( "Internally Generating Stop event for Enc\n");
		vid_enc_get_state(&Param);
		Param.dwEoS = val;
#ifdef EN_GEN_STOP_ON_FILE_CLOSE
		if(val) {
			Param.dwStreamState = CODEC_STATE_STOP;
			vid_enc_set_state(&Param, 1);
		}
#else
		vid_enc_set_state(&Param, 0);
#endif
		//flush_ep_data(ep_set);
	} else if (0 == strcmp(ep_set->file_name, FILE_VID_CAP_STREAM_0)){
		vid_cap_get_state(&Param);
		Param.dwEoS = val;
#ifdef EN_GEN_STOP_ON_FILE_CLOSE
		if(val) {
			Param.dwStreamState = CODEC_STATE_STOP;
			vid_cap_set_state(&Param, 1);
		}
#else
		vid_cap_set_state(&Param, 0);
#endif
	}
}

static int
ep_release (struct inode *inode, struct file *fd)
{
	struct properties_set	*ep_set = inode->i_private;
	UAV_DBG_CFG ( ": %s total=%d\n",  ep_set->file_name, ep_set->total_bytes);

	if(ep_set && ep_set->ep_desc && ep_set->ep) {
		int in = (ep_set->ep_desc->bEndpointAddress	& USB_DIR_IN);		
		if(in) {
			if((ep_set->ep_desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
				// Do nothing. 
			} else /* Bulk EP */ {
				set_eos(ep_set, 0);				
			}
		} else /* out EP */ {
			if((ep_set->ep_desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
				// Do nothing. 
			} else /* Bulk EP */ {
				// work-around for OTG PING NAK issue.
				// Queue buffers ready for next session.
				pool_deque_all_reqests(ep_set);
#ifdef EN_PRE_QUEUE_EP_BUFFERS
				pool_que_all_reqests(ep_set);
#endif
			}
		}
		ep_set->pos = FILE_EOS;
	} else {
		UAV_DBG_CFG ("%s : not used\n", ep_set->file_name);
	}

	if(fd && fd->private_data) {
		//struct properties_set	*ep_set = fd->private_data;
		fd->private_data = NULL;
	}
	return 0;
}

static int ep_ioctl (struct inode *inode, struct file *fd,
		unsigned code, unsigned long value)
{
	return 0;
}
static int
ep_open (struct inode *inode, struct file *fd)
{

	struct properties_set		*ep_set = inode->i_private;
	int			value = -EBUSY;
	UAV_DBG_CFG(": %s\n", ep_set->file_name);
	value = 0;
	if(ep_set && ep_set->ep_desc && ep_set->ep) {
		int in = (ep_set->ep_desc->bEndpointAddress	& USB_DIR_IN);
		fd->private_data = ep_set;
		UAV_DBG_CFG ("%s : ready\n", ep_set->file_name);
		
		if(in) {
			if((ep_set->ep_desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
				// Do nothing. 
			} else /* Bulk EP */ {
				/* Trigger parameter change event*/
			}
			pool_deque_all_reqests(ep_set);
			set_eos(ep_set, 0);
		} else /* out EP */ {
			if((ep_set->ep_desc->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_ISOC){
				// Do nothing. 
			} else /* Bulk EP */ {
#ifndef EN_PRE_QUEUE_EP_BUFFERS
				pool_que_all_reqests(ep_set);
#endif

			}
		}
		ep_set->first_uvc_pkt = 1;
		ep_set->uvc_fid = 0;
		ep_set->total_bytes = 0;
		ep_set->pos = FILE_STREAMING;
	} else {
		UAV_DBG_CFG ("%s : not ready\n", ep_set->file_name);
		return -EFAULT;
	}
	return value;
}


/* used before endpoint configuration */
const struct file_operations ep_operations = {
	.owner =	THIS_MODULE,
	.llseek =	no_llseek,

	.open =		ep_open,
	.read =		ep_read,
	.write =	ep_write,
	.ioctl =	ep_ioctl,
	.release =	ep_release,
};



/////////////////////////////////////////////////////////////////////////////////////////////////



static ssize_t
codec_config_write (struct file *fd, const char __user *buf, size_t len, loff_t *ptr)
{
	return -EINVAL;
}
#define DEV_STATUS_CAHNGED	(dev->vid_cap_cfg_changed || dev->vid_enc_cfg_changed || dev->vid_dec_cfg_changed || \
							dev->aud_codec_cfg_changed || dev->pu_cfg_changed || dev->camera_cfg_changed )

static ssize_t
codec_config_read (struct file *fd, char __user *buf, size_t len, loff_t *ptr)
{
	struct uvc_mg3500 *dev = fd->private_data;
	ssize_t			retval;

	if ((fd->f_flags & O_NONBLOCK) != 0	&& !(DEV_STATUS_CAHNGED)){
		retval = -EAGAIN;
		goto done;
	}
	retval = wait_event_interruptible (dev->codec_cfg_waitq, (DEV_STATUS_CAHNGED) != 0);
	if (retval)
			goto done;

	if(dev->vid_cap_cfg_changed ) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->vid_cap_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->vid_cap_cfg_changed = 0;
	} else if(dev->vid_enc_cfg_changed ) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->vid_enc_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->vid_enc_cfg_changed = 0;
	} else if(dev->vid_dec_cfg_changed ) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->vid_dec_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->vid_dec_cfg_changed = 0;

	} else if (dev->aud_codec_cfg_changed) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->aud_codec_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->aud_codec_cfg_changed = 0;

	} else if (dev->pu_cfg_changed) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->pu_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->pu_cfg_changed = 0;
	} else if (dev->camera_cfg_changed) {
		len = min (len, sizeof(codec_config_t));
		if (copy_to_user (buf, &dev->camera_cfg_data, len))
			retval = -EFAULT;
		else
			retval = len;
		dev->camera_cfg_changed = 0;
	}
done:
	return retval;
}

static int
codec_config_open (struct inode *inode, struct file *fd)
{
	int			value = -EBUSY;
	struct uvc_mg3500 *dev  = inode->i_private;
	UAV_DBG_MSG("%d\n", __LINE__);
	value = 0;
	fd->private_data = dev;
	return value;
}
static int
ep0_fasync (int f, struct file *fd, int on)
{
	int			value = -EBUSY;
	return value;
}
static int dev_ioctl (struct inode *inode, struct file *fd,
		unsigned code, unsigned long value)
{
	return -ENOTTY;
}
static int
dev_release (struct inode *inode, struct file *fd)
{
	return 0;
}

static const struct file_operations codec_config_operations = {
	.owner =	THIS_MODULE,
	.llseek =	no_llseek,

	.open =		codec_config_open,
	.write =	codec_config_write,
	.read	=	codec_config_read,
	.fasync =	ep0_fasync,
	.ioctl =	dev_ioctl,
	.release =	dev_release,
};



/**
 * FILESYSTEM AND SUPERBLOCK OPERATIONS
 *
 * Mounting the filesystem creates a controller file, used first for
 * device configuration then later for event monitoring.
 */

static unsigned default_uid;
static unsigned default_gid;
static unsigned default_perm = S_IRUSR | S_IWUSR;

module_param (default_uid, uint, 0644);
module_param (default_gid, uint, 0644);
module_param (default_perm, uint, 0644);


static struct inode *
usbav_make_inode (struct super_block *sb,
		void *data, const struct file_operations *fops,
		int mode)
{
	struct inode *inode = new_inode (sb);
	if (inode) {
		inode->i_mode = mode;
		inode->i_uid = default_uid;
		inode->i_gid = default_gid;
		inode->i_blocks = 0;
		inode->i_atime = inode->i_mtime = inode->i_ctime
				= CURRENT_TIME;
		inode->i_private = data;
		inode->i_fop = fops;
	}
	return inode;
}

/** 
 * creates in fs root directory, so non-renamable and non-linkable.
 * so inode and dentry are paired, until device reconfig.
 */
static struct inode *
usbav_create_file (struct super_block *sb, char const *name,
		void *data, const struct file_operations *fops,
		struct dentry **dentry_p)
{
	struct dentry	*dentry;
	struct inode	*inode;

	dentry = d_alloc_name(sb->s_root, name);
	if (!dentry)
		return NULL;

	inode = usbav_make_inode (sb, data, fops,
			S_IFREG | (default_perm & S_IRWXUGO));
	if (!inode) {
		dput(dentry);
		return NULL;
	}
	d_add (dentry, inode);
	*dentry_p = dentry;
	return inode;
}

static struct super_operations gadget_fs_operations = {
	.statfs =	simple_statfs,
	.drop_inode =	generic_delete_inode,
};

static int
usbav_fill_super (struct super_block *sb, void *opts, int silent)
{
	struct inode	*inode;
	struct dentry	*d;
	struct uvc_mg3500 *dev;

	if(!the_device){
		UAV_DBG_MSG("Device does not exist\n");
	}
	/* superblock */
	sb->s_blocksize = (64 * 1024/*PAGE_CACHE_SIZE*/);
	sb->s_blocksize_bits = (16/*PAGE_CACHE_SHIFT */);
	sb->s_magic = GADGETFS_MAGIC;
	sb->s_op = &gadget_fs_operations;
	sb->s_time_gran = 1;

	/* root inode */
	inode = usbav_make_inode (sb,
			NULL, &simple_dir_operations,
			S_IFDIR | S_IRUGO | S_IXUGO);
	if (!inode)
		goto enomem0;
	inode->i_op = &simple_dir_inode_operations;
	if (!(d = d_alloc_root (inode)))
		goto enomem1;
	sb->s_root = d;
	dev = the_device;

	dev->sb = sb;
	if (!usbav_create_file (sb, FILE_CFG_CTRL_FILE_NAME,
				dev, &codec_config_operations,
				&dev->dentry))
		goto enomem3;

	activate_ep_files(dev);


	return 0;

enomem3:
	//put_dev (dev);
enomem1:
	iput (inode);
enomem0:
	return -ENOMEM;
}

/* "mount -t usbav path /dev/usbav" ends up here */
static int
usbav_get_sb (struct file_system_type *t, int flags,
		const char *path, void *opts, struct vfsmount *mnt)
{
	return get_sb_single (t, flags, opts, usbav_fill_super, mnt);
}

static void
usbav_kill_sb (struct super_block *sb)
{
	struct uvc_mg3500 *gm = the_device;
	destroy_usbav_files (gm);
	kill_litter_super (sb);
}


/**
 *  Application Notifications for UVC configuration changes
 */

/**
 * Sets the current state of video capture interface
 * It wakes up any thread waiting for the status update.
 */
int vid_cap_set_state(vid_codec_config_t *pParam, int notify)
{
	if(the_device) {
		if(pParam)
			memcpy(&the_device->vid_cap_cfg_data,pParam, sizeof(vid_codec_config_t));
		if(notify) {
			the_device->vid_cap_cfg_changed = 1;
			wake_up_interruptible(&the_device->codec_cfg_waitq);
//#if HAS_VIDCAP_ISOCH ==1

//#else
			/* Work around for not able obtain EoS through EP/ZLP */
			if (pParam->dwStreamState == CODEC_STATE_STOP) {
				UAV_DBG_MSG("Internally Generating EoS\n");
				ep1_in_set.pos = FILE_EOS;
				ep1_in_set.ep_waitq_flag = 1;
				wake_up_interruptible((wait_queue_head_t*)&(ep1_in_set.ep_waitq));
			}
//#endif

		}
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Obtians the current state of video capture interface
 */
int vid_cap_get_state(vid_codec_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->vid_cap_cfg_data, sizeof(vid_codec_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}

/**
 * Sets the current state of video encoder interface
 * It wakes up any thread waiting for the status update.
 */
int vid_enc_set_state(vid_codec_config_t *pParam, int notify)
{
	if(the_device) {
		if(pParam)
			memcpy(&the_device->vid_enc_cfg_data,pParam, sizeof(vid_codec_config_t));
		if(notify) {
			the_device->vid_enc_cfg_changed = 1;
			wake_up_interruptible(&the_device->codec_cfg_waitq);
		}
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Obtians the current state of video encoder interface
 */
int vid_enc_get_state(vid_codec_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->vid_enc_cfg_data, sizeof(vid_codec_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}

/**
 * Sets the current state of video decoder interface
 * It wakes up any thread waiting for the status update.
 */
int vid_dec_set_state(vid_codec_config_t *pParam, int notify)
{
	if(the_device) {
		if(pParam) 
			memcpy(&the_device->vid_dec_cfg_data,pParam, sizeof(vid_codec_config_t));
		if(notify) {
			the_device->vid_dec_cfg_changed = 1;
			wake_up_interruptible(&the_device->codec_cfg_waitq);
#ifdef EN_GEN_EOS_ON_STOP
			/* Work around for not able obtain EoS through EP/ZLP */
			if (pParam->dwStreamState == CODEC_STATE_STOP) {
				UAV_DBG_MSG("Internally Generating EoS\n");
				ep2_out_set.pos = FILE_EOS;
				ep2_out_set.ep_waitq_flag = 1;
				wake_up_interruptible((wait_queue_head_t*)&(ep2_out_set.ep_waitq));
			}
#endif

		}
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Obtians the current state of video decoder interface
 */
int vid_dec_get_state(vid_codec_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->vid_dec_cfg_data, sizeof(vid_codec_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}

/**
 * Obtians the current state of capture processing unit
 */
int pu_get_state(pu_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->pu_cfg_data, sizeof(pu_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}

/**
 * Sets the current state of capture processing unit
 * It wakes up any thread waiting for the status update.
 */
int pu_set_state(pu_config_t *pParam)
{
	if(the_device) {
		memcpy(&the_device->pu_cfg_data,pParam, sizeof(pu_config_t));
		the_device->pu_cfg_changed = 1;
		wake_up_interruptible(&the_device->codec_cfg_waitq);
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Obtians the current state of camera unit
 */
int camera_get_state(camera_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->camera_cfg_data, sizeof(camera_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}

/**
 * Sets the current state of capture processing unit
 * It wakes up any thread waiting for the status update.
 */
int camera_set_state(camera_config_t *pParam)
{
	if(the_device) {
		if(pParam)
			memcpy(&the_device->camera_cfg_data,pParam, sizeof(camera_config_t));
		the_device->camera_cfg_changed = 1;
		wake_up_interruptible(&the_device->codec_cfg_waitq);
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Sets the current state of audio capture
 * It wakes up any thread waiting for the status update.
 */
int aud_codec_set_state(aud_codec_config_t *pParam)
{
	if(the_device) {
		if(pParam)
			memcpy(&the_device->aud_codec_cfg_data,pParam, sizeof(aud_codec_config_t));
		the_device->aud_codec_cfg_changed = 1;
		wake_up_interruptible(&the_device->codec_cfg_waitq);
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return 0;
}

/**
 * Obtians the current state of audio capture
 */
int aud_codec_get_state(aud_codec_config_t *pParam)
{
	if(the_device) {
		memcpy(pParam, &the_device->aud_codec_cfg_data, sizeof(aud_codec_config_t));
		return 0;
	} else {
		UAV_DBG_MSG("%d : the_device == 0", __LINE__);
	}
	return -1;
}


static const char shortname [] = "usbav";
static struct file_system_type usbav_type = {
	.owner		= THIS_MODULE,
	.name		= shortname,
	.get_sb		= usbav_get_sb,
	.kill_sb	= usbav_kill_sb,
};



/**
 * module initialization 
 */
static int __init uvc_mg3500_init(void)
{
	int err;
	printk("uvc_mg3500 Driver Revision %s\n", UVC_DRIVER_REVISION);
	strlcpy(uvcmg3500_str_serial_num, serial,	sizeof(uvcmg3500_str_serial_num));
	err =  usb_gadget_register_driver(&uvc_mg3500_driver);
	if (err == 0) {
		err = register_filesystem (&usbav_type);
		if (err == 0) {
			UAV_DBG_CFG("%s:  ",shortname);
		} else {
			UAV_DBG_ERR("register_filesystem %s failed(%x)\n",shortname, err);
		}
	} else {
		UAV_DBG_ERR("usb_gadget_register_driver failed(%x)\n", err);
	}
	return err;
}

/**
 * module exit 
 */
static void __exit uvc_mg3500_exit(void)
{
	UAV_DBG_CFG("%d\n",__LINE__);
	unregister_filesystem (&usbav_type);
	usb_gadget_unregister_driver(&uvc_mg3500_driver);
}		


module_init(uvc_mg3500_init);
module_exit(uvc_mg3500_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Maxim");
MODULE_DESCRIPTION("MG3500 UVC Gadget Driver");

