/* ============================================================================
 *
 * drivers/net/stmmac/stmmac_main.c
 *
 * This is the driver for the MAC 10/100/1000 on-chip Ethernet controllers.
 *
 * Author: Giuseppe Cavallaro <peppe.cavallaro@st.com>
 *
 * Copyright (C) 2007 by STMicroelectronics
 *
 * ----------------------------------------------------------------------------
 *
 * Changelog:
 *
 * Oct 2007:
 *	- The driver completely merges the new GMAC code and the previous 
 *	  stmmac Ethernet driver (tested on the 7109/7200 STM platforms).
 *	  The GMAC core remains not tested yet.
 * ===========================================================================*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/platform_device.h>
#include <linux/amba/bus.h>
#include <linux/skbuff.h>
#include <linux/ethtool.h>
#include <linux/if_ether.h>
#include <linux/crc32.h>
#include <linux/mii.h>
#include <linux/phy.h>
#include <linux/if_vlan.h>
#include <linux/dma-mapping.h>
#include <linux/dw_gmac_device.h>

#include <mach/mobi_reset.h>
#include <mach/mobi_clock.h>
#include <mach/mobi_codec_reset.h>
#include <mach/mobi_qcc.h>

#include "stmmac.h"
#include "gmac.h"

/* maximum value in according to the TBS1/2 RBS1/2 bits */
#define DMA_MAX_BUFFER_SIZE 0x7ff
#define DMA_BUFFER_SIZE DMA_MAX_BUFFER_SIZE
#define TDES1_MAX_BUF1_SIZE ((DMA_BUFFER_SIZE << DES1_RBS1_SIZE_SHIFT) & \
		DES1_RBS1_SIZE_MASK);
#define TDES1_MAX_BUF2_SIZE ((DMA_BUFFER_SIZE << DES1_RBS2_SIZE_SHIFT) & \
		DES1_RBS2_SIZE_MASK);
#define MIN_MTU 46
#define MAX_MTU ETH_DATA_LEN

#undef STMMAC_DEBUG
/*#define STMMAC_DEBUG*/
#ifdef STMMAC_DEBUG
/*#define DBG(nlevel, klevel, fmt, args...) \
  (void)(netif_msg_##nlevel(lp) && \
  printk(KERN_##klevel fmt, ## args))*/
#define DBG(nlevel, klevel, fmt, args...) printk(KERN_ERR fmt, ## args)
#else
#define DBG(nlevel, klevel, fmt, args...)  do { } while(0)
#endif

#undef STMMAC_TX_DEBUG
/*#define STMMAC_TX_DEBUG*/
#ifdef STMMAC_TX_DEBUG
/*#define TX_DBG(mss, klevel, fmt, args...) \
  if (mss!=0)     \
  printk(KERN_##klevel fmt, ## args)*/
#define TX_DBG(mss, klevel, fmt, args...) printk(KERN_ERR "[TX] mss=%d ", mss); printk(fmt, ## args)
#else
#define TX_DBG(mss, klevel, fmt, args...)  do { } while(0)
#endif

#undef STMMAC_RX_DEBUG
/*#define STMMAC_RX_DEBUG*/
#ifdef STMMAC_RX_DEBUG
//#define RX_DBG(fmt,args...)  printk(fmt, ## args)
//#define RX_DBG(mss, klevel, fmt, args...) if (mss!=0) { printk(KERN_ERR "[RX] mss=%d ", mss); printk(fmt, ## args); }
#define RX_DBG(fmt, args...)  printk(KERN_ERR "[RX]  "); printk(fmt, ## args)
#else
#define RX_DBG(fmt, args...) do { } while(0)
#endif

/* Module Arguments */
#define TX_TIMEO (5*HZ)
static int watchdog = TX_TIMEO;
module_param(watchdog, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(watchdog, "Transmit Timeout (in milliseconds)");

static int debug = -1;		/* -1: default, 0: no output, 16:  all */
module_param(debug, int, S_IRUGO);
MODULE_PARM_DESC(debug, "Message Level (0: no output, 16: all)");

static int rx_copybreak = 0;
module_param(rx_copybreak, int, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(rx_copybreak, "Copy only tiny-frames");

static int phy_n = -1;
module_param(phy_n, int, S_IRUGO);
MODULE_PARM_DESC(phy_n, "Physical device address");

int dma_control = DMA_CONTROL_INIT;
module_param(dma_control, int, S_IRUGO);
MODULE_PARM_DESC(dma_control, "DMA control register");

static int dma_buffer_size = DMA_BUFFER_SIZE;
module_param(dma_buffer_size, int, S_IRUGO);
MODULE_PARM_DESC(dma_buffer_size, "DMA buffer size");

#define DMA_TX_SIZE 32
static int dma_tx_size_param = DMA_TX_SIZE;
module_param(dma_tx_size_param, int, S_IRUGO);
MODULE_PARM_DESC(dma_tx_size_param, "Number of descriptors in the TX list");

#define DMA_RX_SIZE 32
static int dma_rx_size_param = DMA_RX_SIZE;
module_param(dma_rx_size_param, int, S_IRUGO);
MODULE_PARM_DESC(dma_rx_size_param, "Number of descriptors in the RX list");

static int flow_ctrl = FLOW_OFF;
module_param(flow_ctrl, int, S_IRUGO);
MODULE_PARM_DESC(flow_ctrl, "Flow control ability [on/off]");

static int pause = PAUSE_TIME;
module_param(pause, int, S_IRUGO);
MODULE_PARM_DESC(pause, "Flow Control Pause Time");

#define TX_BUFFS_AVAIL(lp) \
	(lp->dirty_tx + lp->dma_tx_size - lp->cur_tx - 1)

static const char version[] = "STMMAC - (C) STMicroelectronics\n";

static const u32 default_msg_level = (NETIF_MSG_DRV 
		| NETIF_MSG_PROBE 
		| NETIF_MSG_LINK 
		| NETIF_MSG_IFUP 
		| NETIF_MSG_IFDOWN 
		| NETIF_MSG_TIMER);

extern int stmmac_mdio_unregister(struct net_device *ndev);
extern int stmmac_mdio_register(struct net_device *ndev);
extern struct ethtool_ops stmmac_ethtool_ops;
static irqreturn_t stmmac_interrupt(int irq, void *dev_id);
static int stmmac_poll(struct napi_struct *napi, int budget);
extern struct device_info_t *gmac_setup(unsigned long addr);

// ST ARCH SPECIFICS
struct plat_stmmacphy_data {
        int bus_id;
        int phy_addr;
        unsigned int phy_mask;
        int interface;
        int (*phy_reset)(void *priv);
        void *priv;
};

/* Private data for the STM on-board ethernet driver */
struct plat_stmmacenet_data {
        int bus_id;
        int pbl;
        int fb;
        void *bsp_priv;
};
// END ST ARCH SPECIFICS
static struct codec_reset_device reset_dev;

static inline void print_mac_addr(u8 addr[6])
{
	int i;
	for (i = 0; i < 5; i++)
		printk("%2.2x:", addr[i]);
	printk("%2.2x\n", addr[5]);
	return;
}

static __inline__ void stmmac_verify_args(void)
{
	/* Wrong parameters are forced with the default values */
	if (watchdog < 0)
		watchdog = TX_TIMEO;
	if (dma_buffer_size > DMA_MAX_BUFFER_SIZE)
		dma_buffer_size = DMA_MAX_BUFFER_SIZE;
	if (rx_copybreak < 0)
		rx_copybreak = ETH_FRAME_LEN;
	if (dma_rx_size_param < 0)
		dma_rx_size_param = DMA_RX_SIZE;
	if (dma_tx_size_param < 0)
		dma_tx_size_param = DMA_TX_SIZE;
	if (flow_ctrl > 1) {
		flow_ctrl = FLOW_AUTO;
	} else if (flow_ctrl < 0) {
		flow_ctrl = FLOW_OFF;
	}
	if ((pause < 0) || (pause > 0xffff))
		pause = PAUSE_TIME;
	return;
}

#ifdef STMMAC_DEBUG
static __inline__ void print_pkt(unsigned char *buf, int len)
{
	int j;
	printk("len = %d byte, buf addr: 0x%p", len, buf);
	for (j = 0; j < len; j++) {
		if ((j % 16) == 0)
			printk("\n %03x:", j);
		printk(" %02x", buf[j]);
	}
	printk("\n");
	return;
}
#endif

/**
 * stmmac_adjust_link
 * @dev: net device structure
 * Description: it adjusts the link parameters.
 */
static void stmmac_adjust_link(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	struct phy_device *phydev = lp->phydev;
	struct dw_gmac_driver_data_t *gmac_pdata = lp->gmac_pdata;
	unsigned long ioaddr = dev->base_addr;
	unsigned long flags;
	int new_state = 0;
	unsigned int fc = lp->flow_ctrl, pause_time = lp->pause;

	spin_lock_irqsave(&lp->lock, flags);
	if (phydev->link) {
		unsigned int ctrl =
			(unsigned int)readl(ioaddr + lp->mac->hw.control);

		if (gmac_pdata->options & DWGMAC_O_FORCE_SPEED_MASK) {
			phydev->autoneg = AUTONEG_DISABLE;
			if (gmac_pdata->options & DWGMAC_O_FORCE_SPEED_10)
				phydev->speed = SPEED_10;
			else if (gmac_pdata->options & DWGMAC_O_FORCE_SPEED_100)
				phydev->speed = SPEED_100;
			else 
				phydev->speed = SPEED_1000;
		}

		if (gmac_pdata->options & DWGMAC_O_FORCE_DUPLEX_MASK) {
			phydev->autoneg = AUTONEG_DISABLE;
			if (gmac_pdata->options & DWGMAC_O_FORCE_DUPLEX_FULL)
				phydev->duplex = DUPLEX_FULL;
			else
				phydev->duplex = DUPLEX_HALF;
		}

		/* Now we make sure that we can be in full duplex mode.
		* If not, we operate in half-duplex mode. */
		if (phydev->duplex != lp->oldduplex) {
			new_state = 1;
			if (!(phydev->duplex)) {
				ctrl &= ~lp->mac->hw.link.duplex;
			} else {
				ctrl |= lp->mac->hw.link.duplex;
			}
			lp->oldduplex = phydev->duplex;
		}
		/* Flow Control operation */
		if (phydev->pause)
			lp->mac->ops->flow_ctrl(ioaddr, phydev->duplex,
					fc, pause_time);

		if (phydev->speed != lp->speed) {
			new_state = 1;
			switch (phydev->speed) {
				case 1000:
					ctrl &= ~lp->mac->hw.link.port;	/* GMII */
					break;
				case 100:
				case 10:
					ctrl |= lp->mac->hw.link.port;	/* MII */

					if (phydev->speed == SPEED_100) {
						ctrl |= lp->mac->hw.link.speed;
					} else {
						ctrl &= ~(lp->mac->hw.link.speed);
					}
					break;
				default:
					if (netif_msg_link(lp))
						printk(KERN_WARNING
								"%s: Ack!  Speed (%d) is not 10, 100 or 1000!\n",
								dev->name, phydev->speed);
					break;
			}

			lp->speed = phydev->speed;
		}

		writel(ctrl, ioaddr + lp->mac->hw.control);

		if (!lp->oldlink) {
			new_state = 1;
			lp->oldlink = 1;
		}
	} else if (lp->oldlink) {
		new_state = 1;
		lp->oldlink = 0;
		lp->speed = 0;
		lp->oldduplex = -1;
	}

	if (new_state && netif_msg_link(lp))
		phy_print_status(phydev);

	spin_unlock_irqrestore(&lp->lock, flags);
}

/**
 * stmmac_init_phy - PHY initialization
 * @dev: net device structure
 * Description: it initializes driver's PHY state, and attaches to the PHY.
 *  Return value:
 *  0 on success
 */
static int stmmac_init_phy(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	struct mii_bus *mii = lp->mii;
	struct phy_device *phydev;
	char phy_id[MII_BUS_ID_SIZE];
	int phy_addr;

	if(phy_n != -1)
		phy_addr = phy_n;
	else
		for( phy_addr = 0 ; phy_addr < PHY_MAX_ADDR; phy_addr++ )
			if( mii->phy_map[phy_addr] )
				break;

	if( phy_addr == PHY_MAX_ADDR ) {
		printk(KERN_ERR "%s: Could not find any PHY\n", dev->name);
		return PTR_ERR((void*)-ENOSYS);
	}

	snprintf(phy_id, MII_BUS_ID_SIZE, PHY_ID_FMT, mii->id, phy_addr);
	DBG(probe, DEBUG, "stmmac_init_phy:  trying to attach to %s\n", phy_id);

	phydev = phy_connect(dev, phy_id, &stmmac_adjust_link, 0, lp->phy_interface);

	if (IS_ERR(phydev)) {
		printk(KERN_ERR "%s: Could not attach to PHY %s\n", dev->name, phy_id);
		return PTR_ERR(phydev);
	}

	lp->oldlink = 0;
	lp->speed = 0;
	lp->oldduplex = -1;
	lp->phy_addr = phy_addr;
	lp->phydev = phydev;

	DBG(probe, DEBUG,
			"stmmac_init_phy:  %s: attached to PHY [%s]. Link = %d\n",
			dev->name, phy_id, phydev->link);

	return 0;
}

/**
 * set_mac_addr
 * @ioaddr: device I/O address
 * @Addr: new MAC address
 * @high: High register offset
 * @low: low register offset
 * Description: the function sets the hardware MAC address
 */
static void set_mac_addr(unsigned long ioaddr, u8 Addr[6],
			 unsigned int high, unsigned int low)
{
	unsigned long data;

	data = (Addr[5] << 8) | Addr[4];
	writel(data, ioaddr + high);
	data = (Addr[3] << 24) | (Addr[2] << 16) | (Addr[1] << 8) | Addr[0];
	writel(data, ioaddr + low);

	return;
}

/**
 * get_mac_addr
 * @ioaddr: device I/O address
 * @addr: mac address
 * @high: High register offset
 * @low: low register offset
 * Description: the function gets the hardware MAC address
 */
static void get_mac_address(unsigned long ioaddr, unsigned char *addr,
			    unsigned int high, unsigned int low)
{
	unsigned int hi_addr, lo_addr;

	/* Read the MAC address from the hardware */
	hi_addr = (unsigned int)readl(ioaddr + high);
	lo_addr = (unsigned int)readl(ioaddr + low);

	/* Extract the MAC address from the high and low words */
	addr[0] = lo_addr & 0xff;
	addr[1] = (lo_addr >> 8) & 0xff;
	addr[2] = (lo_addr >> 16) & 0xff;
	addr[3] = (lo_addr >> 24) & 0xff;
	addr[4] = hi_addr & 0xff;
	addr[5] = (hi_addr >> 8) & 0xff;

	return;
}

/**
 * stmmac_mac_enable_rx
 * @dev: net device structure
 * Description: the function enables the RX MAC process
 */
static void stmmac_mac_enable_rx(struct net_device *dev)
{
	unsigned long ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int value = (unsigned int)readl(ioaddr + lp->mac->hw.control);

	if ((value & lp->mac->hw.enable_rx) != 1)
		napi_enable(&lp->napi);
	/* set the RE (receive enable, bit 2) */
	value |= lp->mac->hw.enable_rx;
	writel(value, ioaddr + lp->mac->hw.control);
	return;
}

/**
 * stmmac_mac_enable_tx
 * @dev: net device structure
 * Description: the function enables the TX MAC process
 */
static void stmmac_mac_enable_tx(struct net_device *dev)
{
	unsigned long ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int value = (unsigned int)readl(ioaddr + lp->mac->hw.control);

	/* set: TE (transmitter enable, bit 3) */
	value |= lp->mac->hw.enable_tx;
	writel(value, ioaddr + lp->mac->hw.control);
	return;
}

/**
 * stmmac_mac_disable_rx
 * @dev: net device structure
 * Description: the function disables the RX MAC process
 */
static void stmmac_mac_disable_rx(struct net_device *dev)
{
	unsigned long ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int value = (unsigned int)readl(ioaddr + lp->mac->hw.control);

	/* this function is called by release and remove, rmmod the driver
	 * calls both functions and calling napi_disable twice hangs so only 
	 * call it once 
	 */
	if (value & lp->mac->hw.enable_rx)
		napi_disable(&lp->napi);

	value &= ~(lp->mac->hw.enable_rx);
	writel(value, ioaddr + lp->mac->hw.control);
	return;
}

/**
 * stmmac_mac_disable_tx
 * @dev: net device structure
 * Description: the function disables the TX MAC process
 */
static void stmmac_mac_disable_tx(struct net_device *dev)
{
	unsigned long ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int value = (unsigned int)readl(ioaddr + lp->mac->hw.control);

	value &= ~(lp->mac->hw.enable_tx);
	writel(value, ioaddr + lp->mac->hw.control);
	return;
}

static void display_dma_desc_ring(dma_desc *p, int size)
{
	int i;
	for (i = 0; i < size; i++) {
		printk("\t%2d virt[0x%x] phys[0x%x]: "
				"desc0=0x%x desc1=0x%x buffer1=0x%x", i,
				(uint32_t)&p[i].des0,
				(unsigned int)virt_to_phys(&p[i].des0), p[i].des0,
				p[i].des1, (unsigned int)p[i].des2);
		if (p[i].des3 != 0)
			printk(" buffer2=0x%x", (unsigned int)p[i].des3);
		printk("\n");
	}
}

/**
 * clear_dma_descs - reset the DMA descriptors
 * @p: it starts pointing to the first element in the ring.
 * @ring_size: it is the size of the ring.
 * @own_bit: it is the owner bit (RX: OWN_BIT - TX: 0).
 * Description: this function clears both RX and TX descriptors.
 * Note that the driver uses the 'implicit' scheme for implementing
 * the TX/RX DMA linked lists. So the second buffer doesn't point
 * to the next descriptor.
 */
static void clear_dma_descs(dma_desc * p, unsigned int ring_size,
			    unsigned int own_bit)
{
	int i;
	for (i = 0; i < ring_size; i++) {
		p->des0 = own_bit;
		if (!(own_bit))
			p->des1 = 0;
		else
			p->des1 = (dma_buffer_size << DES1_RBS1_SIZE_SHIFT);
		if (i == ring_size - 1) {
			p->des1 |= DES1_CONTROL_TER;
		}
		p->des3 = 0;
		p++;
	}
	return;
}

/**
 * init_dma_desc_rings - init the RX/TX descriptor rings
 * @dev: net device structure
 * Description:  this function initializes the DMA RX/TX descriptors
 */
static void init_dma_desc_rings(struct net_device *dev)
{
	int i;
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int txsize = lp->dma_tx_size;
	unsigned int rxsize = lp->dma_rx_size;
	lp->dma_buf_sz = dma_buffer_size;

	DBG(probe, DEBUG, "%s: allocate and init the DMA RX/TX\n",
			ETH_RESOURCE_NAME);

	lp->rx_skbuff_dma =
		(dma_addr_t *) kmalloc(lp->dma_rx_size * sizeof(dma_addr_t),
							   GFP_KERNEL);
	lp->rx_skbuff =
		(struct sk_buff **)kmalloc(sizeof(struct sk_buff *) * rxsize,
								   GFP_KERNEL);
	lp->dma_rx = (dma_desc *) dma_alloc_coherent(lp->device,
			rxsize *
			sizeof(struct dma_desc_t),
			&lp->dma_rx_phy,
			GFP_KERNEL);
	lp->tx_skbuff =
		(struct sk_buff **)kmalloc(sizeof(struct sk_buff *) * txsize,
								   GFP_KERNEL);
	lp->dma_tx = (dma_desc *) dma_alloc_coherent(lp->device,
			txsize *
			sizeof(struct dma_desc_t),
			&lp->dma_tx_phy,
			GFP_KERNEL);

	if ((lp->dma_rx == NULL) || (lp->dma_tx == NULL)) {
		printk(KERN_ERR "%s: ERROR allocating the DMA Tx/Rx desc\n",
				__FUNCTION__);
		return;
	}
	DBG(probe, DEBUG, "%s: DMA desc rings: virt addr (Rx 0x%08x, "
			"Tx 0x%08x) DMA phy addr (Rx 0x%08x,Tx 0x%08x)\n",
			dev->name, (unsigned int)lp->dma_rx, (unsigned int)lp->dma_tx,
			(unsigned int)lp->dma_rx_phy, (unsigned int)lp->dma_tx_phy);

	/* ---- RX INITIALIZATION */
	DBG(probe, DEBUG, "[RX skb data]   [DMA RX skb data] "
			"(buff size: %d)\n", lp->dma_buf_sz);
	for (i = 0; i < rxsize; i++) {
		dma_desc *p = lp->dma_rx + i;
		struct sk_buff *skb = netdev_alloc_skb(dev, lp->dma_buf_sz + NET_IP_ALIGN);
		if (unlikely(skb == NULL)) {
			printk(KERN_ERR "%s: Rx init fails; skb is NULL\n",
					__FUNCTION__);
			break;
		}
		skb_reserve(skb, NET_IP_ALIGN);
		lp->rx_skbuff[i] = skb;
		lp->rx_skbuff_dma[i] = dma_map_single(lp->device, skb->data,
				lp->dma_buf_sz,
				DMA_FROM_DEVICE);
		p->des2 = lp->rx_skbuff_dma[i];
		DBG(probe, DEBUG, "[0x%08x]\t[0x%08x]\n",
				(unsigned int)lp->rx_skbuff[i]->data,
				(unsigned int)lp->rx_skbuff_dma[i]);
	}
	lp->cur_rx = 0;
	lp->dirty_rx = (unsigned int)(i - rxsize);

	/* ---- TX INITIALIZATION */
	for (i = 0; i < txsize; i++) {
		lp->tx_skbuff[i] = NULL;
		lp->dma_tx[i].des2 = 0;
		lp->dma_tx[i].des3 = 0;
	}
	lp->dirty_tx = lp->cur_tx = 0;

	/* Clear the Rx/Tx descriptors */
	clear_dma_descs(lp->dma_rx, rxsize, OWN_BIT);
	clear_dma_descs(lp->dma_tx, txsize, 0);

	if (netif_msg_hw(lp)) {
		printk("RX descriptor ring:\n");
		display_dma_desc_ring(lp->dma_rx, rxsize);
		printk("TX descriptor ring:\n");
		display_dma_desc_ring(lp->dma_tx, txsize);
	}
	return;
}

/**
 * dma_free_rx_skbufs
 * @dev: net device structure
 * Description:  this function frees all the skbuffs in the Rx queue
 */
static void dma_free_rx_skbufs(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	int i;

	for (i = 0; i < lp->dma_rx_size; i++) {
		if (lp->rx_skbuff[i]) {
			dma_unmap_single(lp->device, lp->rx_skbuff_dma[i],
					lp->dma_buf_sz, DMA_FROM_DEVICE);
			dev_kfree_skb(lp->rx_skbuff[i]);
		}
		lp->rx_skbuff[i] = NULL;
	}
	return;
}

/**
 * dma_free_tx_skbufs
 * @dev: net device structure
 * Description:  this function frees all the skbuffs in the Tx queue
 */
static void dma_free_tx_skbufs(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	int i;

	for (i = 0; i < lp->dma_tx_size; i++) {
		if (lp->tx_skbuff[i] != NULL) {
			dma_desc *p = lp->dma_tx + i;
			if (p->des2) {
				dma_unmap_single(lp->device, p->des2,
						(p->
						 des1 & DES1_RBS1_SIZE_MASK) >>
						DES1_RBS1_SIZE_SHIFT,
						DMA_TO_DEVICE);
			}
			if (p->des3) {
				dma_unmap_single(lp->device, p->des3,
						(p->
						 des1 & DES1_RBS2_SIZE_MASK) >>
						DES1_RBS2_SIZE_SHIFT,
						DMA_TO_DEVICE);
			}
			dev_kfree_skb_any(lp->tx_skbuff[i]);
			lp->tx_skbuff[i] = NULL;
		}
	}
	return;
}

/**
 * free_dma_desc_resources
 * @dev: net device structure
 * Description:  this function releases and free ALL the DMA resources
 */
static void free_dma_desc_resources(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	/* Release the DMA TX/RX socket buffers */
	dma_free_rx_skbufs(dev);
	dma_free_tx_skbufs(dev);

	/* Free the region of consistent memory previously allocated for 
	 * the DMA */
	dma_free_coherent(lp->device,
			  lp->dma_tx_size * sizeof(struct dma_desc_t),
			  lp->dma_tx, lp->dma_tx_phy);
	dma_free_coherent(lp->device,
			  lp->dma_rx_size * sizeof(struct dma_desc_t),
			  lp->dma_rx, lp->dma_rx_phy);
	kfree(lp->rx_skbuff_dma);
	kfree(lp->rx_skbuff);
	kfree(lp->tx_skbuff);

	return;
}

/**
 * stmmac_dma_reset - STMAC DMA SW reset
 * @ioaddr: device I/O address
 * Description:  this function performs the DMA SW reset.
 *  NOTE1: the MII_TxClk and the MII_RxClk must be active before this
 *	   SW reset otherwise the MAC core won't exit the reset state.
 *  NOTE2: after a SW reset all interrupts are disabled
 */
static void stmmac_dma_reset(unsigned long ioaddr)
{
	unsigned int value;

	value = (unsigned int)readl(ioaddr + DMA_BUS_MODE);
	value |= DMA_BUS_MODE_SFT_RESET;
	writel(value, ioaddr + DMA_BUS_MODE);
	while ((readl(ioaddr + DMA_BUS_MODE) & DMA_BUS_MODE_SFT_RESET)) { }
	return;
}

/**
 * stmmac_dma_start_tx
 * @ioaddr: device I/O address
 * Description:  this function starts the DMA tx process
 */
static void stmmac_dma_start_tx(unsigned long ioaddr)
{
	unsigned int value;
	value = (unsigned int)readl(ioaddr + DMA_CONTROL);
	value |= DMA_CONTROL_ST;
	writel(value, ioaddr + DMA_CONTROL);
	return;
}

static void stmmac_dma_stop_tx(unsigned long ioaddr)
{
	unsigned int value;
	value = (unsigned int)readl(ioaddr + DMA_CONTROL);
	value &= ~DMA_CONTROL_ST;
	writel(value, ioaddr + DMA_CONTROL);
	return;
}

/**
 * stmmac_dma_start_rx
 * @ioaddr: device I/O address
 * Description:  this function starts the DMA rx process
 */
static void stmmac_dma_start_rx(unsigned long ioaddr)
{
	unsigned int value;
	value = (unsigned int)readl(ioaddr + DMA_CONTROL);
	value |= DMA_CONTROL_SR;
	writel(value, ioaddr + DMA_CONTROL);

	return;
}

static void stmmac_dma_stop_rx(unsigned long ioaddr)
{
	unsigned int value;

	value = (unsigned int)readl(ioaddr + DMA_CONTROL);
	value &= ~DMA_CONTROL_SR;
	writel(value, ioaddr + DMA_CONTROL);

	return;
}

static __inline__ void stmmac_dma_enable_irq_rx(unsigned long ioaddr)
{
	writel(DMA_INTR_DEFAULT_MASK, ioaddr + DMA_INTR_ENA);
	return;
}

static __inline__ void stmmac_dma_disable_irq_rx(unsigned long ioaddr)
{
	writel(DMA_INTR_NO_RX, ioaddr + DMA_INTR_ENA);
	return;
}

/**
 * stmmac_dma_init - DMA init function
 * @dev: net device structure
 * Description: the DMA init function performs:
 * - the DMA RX/TX SW descriptors initialization
 * - the DMA HW controller initialization
 * NOTE: the DMA TX/RX processes will be started in the 'open' method.
 */
static int stmmac_dma_init(struct net_device *dev)
{
	unsigned long ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);

	DBG(probe, DEBUG, "STMMAC: DMA Core setup\n");

	/* DMA SW reset */
	stmmac_dma_reset(ioaddr);

	/* Enable Application Access by writing to DMA CSR0 */
	DBG(probe, DEBUG, "\t(PBL: %d)\n", lp->pbl);
	writel(DMA_BUS_MODE_DEFAULT 
			| ((lp->pbl) << DMA_BUS_MODE_PBL_SHIFT)
			| ((lp->fb)  << DMA_BUS_MODE_FB_SHIFT),
			ioaddr+DMA_BUS_MODE);

	/* Mask interrupts by writing to CSR7 */
	writel(DMA_INTR_DEFAULT_MASK, ioaddr + DMA_INTR_ENA);
	/* The base address of the RX/TX descriptor lists must be written into
	* DMA CSR3 and CSR4, respectively. */
	writel((unsigned long)lp->dma_tx_phy, ioaddr + DMA_TX_BASE_ADDR);
	writel((unsigned long)lp->dma_rx_phy, ioaddr + DMA_RCV_BASE_ADDR);
	if (netif_msg_hw(lp))
		lp->mac->ops->dma_registers(ioaddr);

	return 0;
}

#ifdef STMMAC_DEBUG
/**
 * show_tx_process_state
 * @status: tx descriptor status field
 * Description: it shows the Transmit Process State for CSR5[22:20]
 */
static void show_tx_process_state(unsigned int status)
{
	unsigned int state;
	state = (status & DMA_STATUS_TS_MASK) >> DMA_STATUS_TS_SHIFT;

	switch (state) {
		case 0:
			printk("- TX (Stopped): Reset or Stop command\n");
			break;
		case 1:
			printk("- TX (Running):Fetching the Tx desc\n");
			break;
		case 2:
			printk("- TX (Running): Waiting for end of tx\n");
			break;
		case 3:
			printk("- TX (Running): Reading the data "
					"and queuing the data into the Tx buf\n");
			break;
		case 6:
			printk("- TX (Suspended): Tx Buff Underflow "
					"or an unavailable Transmit descriptor\n");
			break;
		case 7:
			printk("- TX (Running): Closing Tx descriptor\n");
			break;
		default:
			break;
	}
	return;
}

/**
 * show_rx_process_state
 * @status: rx descriptor status field
 * Description: it shows the  Receive Process State for CSR5[19:17]
 */
static void show_rx_process_state(unsigned int status)
{
	unsigned int state;
	state = (status & DMA_STATUS_RS_MASK) >> DMA_STATUS_RS_SHIFT;

	switch (state) {
		case 0:
			printk("- RX (Stopped): Reset or Stop command\n");
			break;
		case 1:
			printk("- RX (Running): Fetching the Rx desc\n");
			break;
		case 2:
			printk("- RX (Running):Checking for end of pkt\n");
			break;
		case 3:
			printk("- RX (Running): Waiting for Rx pkt\n");
			break;
		case 4:
			printk("- RX (Suspended): Unavailable Rx buf\n");
			break;
		case 5:
			printk("- RX (Running): Closing Rx descriptor\n");
			break;
		case 6:
			printk("- RX(Running): Flushing the current frame"
					" from the Rx buf\n");
			break;
		case 7:
			printk("- RX (Running): Queuing the Rx frame"
					" from the Rx buf into memory\n");
			break;
		default:
			break;
	}
	return;
}
#endif

/**
 * stmmac_tx
 * @dev: net device structure
 * Description: it is used for freeing the TX resources.  
 */
static __inline__ void stmmac_tx(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int txsize = lp->dma_tx_size;
	int entry = lp->dirty_tx % txsize;

	spin_lock(&lp->tx_lock);

	while (lp->dirty_tx != lp->cur_tx) {
		dma_desc *p = lp->dma_tx + entry;
		int status = p->des0;

		if (status & OWN_BIT)
			break;

		/* When the transmission is completed the frame status
		 * is written into TDESC0 of the descriptor having the 
		 * LS bit set. */
		if (likely(p->des1 & TDES1_CONTROL_LS)) {
			if (unlikely(lp->mac->ops->check_tx_summary(&lp->stats, status) < 0)) {
				lp->stats.tx_errors++;
			} else {
				lp->stats.tx_packets++;
			}
		}
		if (p->des2) {
			dma_unmap_single(lp->device, p->des2,
					(p->
					 des1 & DES1_RBS1_SIZE_MASK) >>
					DES1_RBS1_SIZE_SHIFT, DMA_TO_DEVICE);
			p->des2 = 0;
		}
		if (unlikely(p->des3)) {
			dma_unmap_single(lp->device, p->des3,
					(p->
					 des1 & DES1_RBS2_SIZE_MASK) >>
					DES1_RBS2_SIZE_SHIFT, DMA_TO_DEVICE);
			p->des3 = 0;
		}
		if (lp->tx_skbuff[entry] != NULL) {
			dev_kfree_skb_irq(lp->tx_skbuff[entry]);
			lp->tx_skbuff[entry] = NULL;
		}
		entry = (++lp->dirty_tx) % txsize;
	}
	if (netif_queue_stopped(dev) && TX_BUFFS_AVAIL(lp) > (MAX_SKB_FRAGS + 1))
		netif_wake_queue(dev);

	spin_unlock(&lp->tx_lock);
	return;
}

/**
 * stmmac_dma_interrupt - Interrupt handler for the STMMAC DMA
 * @dev: net device structure
 * Description: It determines if we have to call either the Rx or the Tx
 * interrupt handler.
 */
static void stmmac_dma_interrupt(struct net_device *dev)
{
	unsigned int status;
	unsigned int ioaddr = dev->base_addr;
	struct eth_driver_local *lp = netdev_priv(dev);

	/* read the status register (CSR5) */
	status = (unsigned int)readl(ioaddr + DMA_STATUS);

again:	/* Clear the interrupt by writing a logic 1 to the CSR5[15-0] */
	writel(status, ioaddr + DMA_STATUS);

	DBG(intr, INFO, "%s: [CSR5: 0x%08x]\n", __FUNCTION__, status);

#ifdef STMMAC_DEBUG
	/* It displays the DMA transmit process state (CSR5 register) */
	if (netif_msg_tx_done(lp))
		show_tx_process_state(status);
	if (netif_msg_rx_status(lp))
		show_rx_process_state(status);
#endif
	/* Process the NORMAL interrupts */
	if (status & DMA_STATUS_NIS) {
		DBG(intr, INFO, " CSR5[16]: DMA NORMAL IRQ: 0x%08x | ", status);
		if (status & DMA_STATUS_RI) {
			lp->rx_buff = readl(ioaddr + DMA_CUR_RX_BUF_ADDR);

			DBG(intr, INFO, "Receive irq [buf: 0x%08x]\n", lp->rx_buff);
			/*display_dma_desc_ring(lp->dma_rx, lp->dma_rx_size); */
			stmmac_dma_disable_irq_rx(ioaddr);
			if (likely(napi_schedule_prep(&lp->napi))) {
				__napi_schedule(&lp->napi);
			} else {
				RX_DBG("IRQ: bug!interrupt while in poll\n");
			}
		}
		if (status & DMA_STATUS_TI) {
			DBG(intr, INFO, " Transmit irq [buf: 0x%08x]\n",
					readl(ioaddr + DMA_CUR_TX_BUF_ADDR));
			stmmac_tx(dev);
		}
	}
	/* ABNORMAL interrupts */
	if (unlikely(status & (DMA_STATUS_AIS | DMA_STATUS_FBI))) {
		DBG(intr, INFO, "CSR5[15] DMA ABNORMAL IRQ: ");
		if (status & DMA_STATUS_TPS) {
			DBG(intr, INFO, "Transmit Process Stopped \n");
		}
		if (status & DMA_STATUS_TJT) {
			DBG(intr, INFO, "Transmit Jabber Timeout\n");
		}
		if (status & DMA_STATUS_OVF) {

			unsigned int rxsize = lp->dma_rx_size;
			int entry = lp->cur_rx % rxsize;
			dma_desc *drx = lp->dma_rx + entry;
			DBG(intr, INFO, "Receive Overflow\n");
			printk("DESC %x: %x %x\n", (unsigned int)drx, drx->des0, drx->des1);
		}
		if (status & DMA_STATUS_UNF) {
			DBG(intr, INFO, "Transmit Underflow\n");
		}
		if (status & DMA_STATUS_RU) {
			DBG(intr, INFO, "Rx Buffer Unavailable\n");
		}
		if (status & DMA_STATUS_RPS) {
			DBG(intr, INFO, "Receive Process Stopped\n");
		}
		if (status & DMA_STATUS_RWT) {
			DBG(intr, INFO, "Rx Watchdog Timeout\n");
		}
		if (status & DMA_STATUS_ETI) {
			DBG(intr, INFO, "Early Tx Interrupt\n");
		}
		if (status & DMA_STATUS_FBI) {
			DBG(intr, INFO, "Fatal Bus Error Interrupt\n");
		}
	}
	DBG(intr, INFO, "\n\n");

	/* read the status register (CSR5) */
	status = (unsigned int)readl(ioaddr + DMA_STATUS);
	if (status & (DMA_STATUS_AIS | DMA_STATUS_NIS))
		goto again;

	return;
}

/**
 *  stmmac_open - open entry point of the driver
 *  @dev : pointer to the device structure.
 *  Description:
 *  This function is the open entry point of the driver.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *  file on failure.
 */

int stmmac_open(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned long ioaddr = dev->base_addr;
	int ret;
	/* Check that the MAC address is valid.  If its not, refuse
	 * to bring the device up. The user must specify an
	 * address using the following linux command:
	 *      ifconfig eth0 hw ether xx:xx:xx:xx:xx:xx  */
	if (!is_valid_ether_addr(dev->dev_addr)) {
		DBG(probe, ERR, "%s: no valid eth hw addr\n", __FUNCTION__);
		return -EINVAL;
	}
	/* Request the IRQ lines */
	ret = request_irq(dev->irq, &stmmac_interrupt, IRQF_DISABLED, dev->name, dev);
	if (ret < 0) {
		printk(KERN_ERR
				"%s: ERROR: allocating the IRQ %d (error: %d)\n",
				__FUNCTION__, dev->irq, ret);
		return ret;
	}
	/* Attach the PHY */
	ret = stmmac_init_phy(dev);
	if (ret) {
		printk(KERN_ERR "%s: Cannot attach to PHY (error: %d)\n",
				__FUNCTION__, ret);
		return -ENODEV;
	}
	/* Create and initialize the TX/RX descriptors rings */
	init_dma_desc_rings(dev);
	/* Intialize the DMA controller and send the SW reset
	 * This must be done after we have successfully initialised the PHY
	 * (see comment in stmmac_dma_reset). */
	if (stmmac_dma_init(dev) < 0) {
		DBG(probe, ERR, "%s: DMA initialization failed\n",
				__FUNCTION__);
		return -1;
	}
	/* Copy the MAC addr into the HW in case we have set it with nwhw */
	print_mac_addr(dev->dev_addr);
	set_mac_addr(ioaddr, dev->dev_addr, lp->mac->hw.addr_high,
			lp->mac->hw.addr_low);
	/* Initialize the MAC100 or GMAC Core */
	lp->mac->ops->core_init(ioaddr);
	/* Enable the MAC/DMA */
	stmmac_mac_enable_rx(dev);
	stmmac_mac_enable_tx(dev);
	/* Dump MAC registers */
	if (netif_msg_hw(lp))
		lp->mac->ops->mac_registers((unsigned int)ioaddr);
	phy_start(lp->phydev);
	/* Start the ball rolling... */
	DBG(probe, DEBUG, "%s: DMA RX/TX processes started...\n",
			ETH_RESOURCE_NAME);
	stmmac_dma_start_rx(ioaddr);
	stmmac_dma_start_tx(ioaddr);
	netif_start_queue(dev);
	return 0;
}

/**
 *  stmmac_release - close entry point of the driver
 *  @dev : device pointer.
 *  Description:
 *  This is the stop entry point of the driver.
 *  Return value:
 *  0 on success and an appropriate (-)ve integer as defined in errno.h
 *  file on failure.
 */
int stmmac_release(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	/* Stop the PHY */
	phy_stop(lp->phydev);
	phy_disconnect(lp->phydev);
	lp->phydev = NULL;

	/* Free the IRQ lines */
	free_irq(dev->irq, dev);

	/* Stop TX/RX DMA and clear the descriptors */
	stmmac_dma_stop_tx(dev->base_addr);
	stmmac_dma_stop_rx(dev->base_addr);

	free_dma_desc_resources(dev);

	/* Disable the MAC core */
	stmmac_mac_disable_tx(dev);
	stmmac_mac_disable_rx(dev);

	/* Change the link status */
	netif_carrier_off(dev);

	return 0;
}

/**
 *  stmmac_fill_tx_buffer
 *  @data : data buffer
 *  @size : fragment size
 *  @mss : Maximum  Segment Size
 *  @lp : driver local structure
 *  @first : first element in the ring
 *  Description: it is used for filling the DMA tx ring with the frame to be
 *  transmitted.
 *  Note that the algorithm works both for the non-paged data and for the paged
 *  fragment (SG).
 *  Return value:
 *    current entry point in the tx ring
 */
static int stmmac_fill_tx_buffer(void *data, unsigned int size,
				 unsigned int mss,
				 struct eth_driver_local *lp, int first)
{
	int new_des = 0;
	void *addr = data;
	dma_desc *p = lp->dma_tx;
	unsigned int entry;
	int bsize = lp->dma_buf_sz;
	unsigned int txsize = lp->dma_tx_size;

	TX_DBG(mss, INFO, "  %s (size=%d, addr=0x%x)\n", __FUNCTION__,
			size, (unsigned int)addr);
	do {
		if (new_des) {
			lp->cur_tx++;
			new_des = 0;
		}
		entry = lp->cur_tx % txsize;
		/* Reset the descriptor number 1 */
		p[entry].des1 = (p[entry].des1 & DES1_CONTROL_TER);
		if (first)
			p[entry].des1 |= TDES1_CONTROL_FS;
		else
			lp->tx_skbuff[entry] = NULL;

		TX_DBG(mss, INFO, "\t[entry =%d] buf1 len=%d\n",
				entry, min((int)size, bsize));
		/* If the data size is too big we need to use buffer 2
		* (in the same descriptor) or, if necessary, another descriptor
		* in the ring. */
		if (likely(size < bsize)) {
			p[entry].des1 |= ((size << DES1_RBS1_SIZE_SHIFT) &
					DES1_RBS1_SIZE_MASK);
			p[entry].des2 = dma_map_single(lp->device, addr,
					size, DMA_TO_DEVICE);
		} else {
			int b2_size = (size - bsize);

			p[entry].des1 |= TDES1_MAX_BUF1_SIZE;
			p[entry].des2 = dma_map_single(lp->device, addr, bsize,
					DMA_TO_DEVICE);

			/* Check if we need to use the buffer 2 */
			if (b2_size > 0) {
				void *buffer2 = addr;

				TX_DBG(mss, INFO, "\t[entry=%d] buf2 len=%d\n",
						entry, min(b2_size, bsize));

				/* Check if we need another descriptor. */
				if (b2_size > bsize) {
					b2_size = bsize;
					size -= (2 * bsize);
					addr += ((2 * bsize) + 1);
					new_des = 1;
					TX_DBG(mss, INFO,
							"\tnew descriptor - "
							"%s (len = %d)\n",
							(first) ? "skb->data" :
							"Frag", size);
				}
				p[entry].des3 = dma_map_single(lp->device,
						(buffer2 +
						 bsize + 1),
						b2_size,
						DMA_TO_DEVICE);
				if (b2_size == bsize) {
					p[entry].des1 |= TDES1_MAX_BUF2_SIZE;
				} else {
					p[entry].des1 |=
						((b2_size << DES1_RBS2_SIZE_SHIFT)
						 & DES1_RBS2_SIZE_MASK);
				}
			}
		}
	} while (new_des);
	return entry;
}

/**
 *  stmmac_xmit - Tx entry point of the driver
 *  @skb : the socket buffer
 *  @dev : device pointer
 *  Description :
 *  This function is the Tx entry point of the driver.
 */
int stmmac_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	dma_desc *p = lp->dma_tx;
	unsigned int txsize = lp->dma_tx_size;
	unsigned int nfrags = skb_shinfo(skb)->nr_frags, start = lp->cur_tx % txsize,
				 entry = lp->cur_tx % txsize, i, mss = 0, nopaged_len;
	unsigned long flags;

	local_irq_save(flags);
	if (!spin_trylock(&lp->tx_lock)) {
		/* Tell upper layer to requeue */
		local_irq_restore(flags);
		return NETDEV_TX_LOCKED;
	}

	/* This is a hard error log it. */
	if (unlikely(TX_BUFFS_AVAIL(lp) < nfrags + 1)) {
		netif_stop_queue(dev);
		spin_unlock_irqrestore(&lp->tx_lock, flags);
		printk(KERN_ERR "%s: bug! Tx Ring full when queue awake!\n",
				dev->name);
		return NETDEV_TX_BUSY;
	}
	if (dev->features & NETIF_F_GSO)
		mss = skb_shinfo(skb)->gso_size;

	/* Verify the checksum */
	lp->mac->ops->tx_checksum(skb);
	/* Get the amount of non-paged data (skb->data). */
	nopaged_len = skb_headlen(skb);

	if (unlikely((lp->tx_skbuff[entry] != NULL))) {
		/* This should never happen! */
		printk(KERN_ERR "%s: bug! Inconsistent Tx skb utilization!\n",
				dev->name);
		dev_kfree_skb(skb);
		return -1;
	}
	lp->tx_skbuff[entry] = skb;
	TX_DBG(mss, INFO, "%s:(skb->len=%d, nfrags=%d, "
			"nopaged_len=%d, mss=%d)\n", __FUNCTION__, skb->len,
			nfrags, nopaged_len, mss);
	/* Handle the non-paged data (skb->data) */
	stmmac_fill_tx_buffer(skb->data, nopaged_len, mss, lp, 1);
	/* Handle the paged fragments */
	for (i = 0; i < nfrags; i++) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
		void *addr =
			((void *)page_address(frag->page) + frag->page_offset);
		int len = frag->size;

		lp->cur_tx++;
		entry = stmmac_fill_tx_buffer(addr, len, mss, lp, 0);
	}
	/* If there are more than one fragment, we set the interrupt
	* on completition in the latest fragment (where we also set 
	* the LS bit. */
	p[entry].des1 |= TDES1_CONTROL_LS | TDES1_CONTROL_IC;
	/* Finally, set the owner field */
	do {
		if(start == txsize)
			start = 0;
		p[start].des0 = OWN_BIT;
	} while(start++ != entry);
	lp->cur_tx++;
	lp->stats.tx_bytes += skb->len;
#ifdef STMMAC_DEBUG
	if (netif_msg_pktdata(lp)) {
		printk(">>> (current=%d, dirty=%d; entry=%d)\n",
				(lp->cur_tx % txsize), (lp->dirty_tx % txsize), entry);
		display_dma_desc_ring(lp->dma_tx, txsize);
		printk(">>> frame to be transmitted: ");
		print_pkt(skb->data, skb->len);
	}
#endif
	if (TX_BUFFS_AVAIL(lp) <= (MAX_SKB_FRAGS + 1)) {
		netif_stop_queue(dev);
	}

	/* CSR1 enables the transmit DMA to check for new descriptor */
	writel(1, dev->base_addr + DMA_XMT_POLL_DEMAND);
	spin_unlock_irqrestore(&lp->tx_lock, flags);
	dev->trans_start = jiffies;
	return NETDEV_TX_OK;
}

static __inline__ void stmmac_rx_refill(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned int rxsize = lp->dma_rx_size;

	for (; lp->cur_rx - lp->dirty_rx > 0; lp->dirty_rx++) {
		struct sk_buff *skb;
		int entry = lp->dirty_rx % rxsize;
		if (lp->rx_skbuff[entry] == NULL) {
			skb = netdev_alloc_skb(dev, lp->dma_buf_sz + NET_IP_ALIGN);
			if (unlikely(skb == NULL)) {
				printk(KERN_ERR "%s: skb is NULL\n",
						__FUNCTION__);
				break;
			}
			skb_reserve(skb, NET_IP_ALIGN);
			lp->rx_skbuff[entry] = skb;
			lp->rx_skbuff_dma[entry] = dma_map_single(lp->device,
					skb->data,
					lp->dma_buf_sz,
					DMA_FROM_DEVICE);
			(lp->dma_rx + entry)->des2 = lp->rx_skbuff_dma[entry];
			RX_DBG(rx_status, INFO, "\trefill entry #%d\n", entry);
		}
		(lp->dma_rx + entry)->des0 = OWN_BIT;
	}
	return;
}

static int stmmac_poll(struct napi_struct *napi, int budget)
{
	struct eth_driver_local *lp = container_of(napi, struct eth_driver_local, napi);
	struct net_device* dev = lp->dev;
	unsigned int rxsize = lp->dma_rx_size;
	int frame_len = 0, entry = lp->cur_rx % rxsize, work_done = 0;
	unsigned int ioaddr = dev->base_addr;
	dma_desc *drx = lp->dma_rx + entry;

#ifdef STMMAC_RX_DEBUG
	printk(">>> stmmac_poll: RX descriptor ring:\n");
	display_dma_desc_ring(lp->dma_rx, rxsize);
#endif

	while (!(drx->des0 & OWN_BIT)) {
		unsigned int status = drx->des0;

		if (unlikely
				(lp->mac->ops->check_rx_summary(&lp->stats, status) < 0)) {
			lp->stats.rx_errors++;
		} else {
			struct sk_buff *skb;

			/* frame_len is the length in bytes (omitting the FCS) */
			frame_len = (((status & RDES0_STATUS_FL_MASK) >>
						RDES0_STATUS_FL_SHIFT) - 4);

			RX_DBG
				("\tquota %d, desc addr: 0x%0x [entry: %d] buff=0x%x\n",
				 rx_work_limit, (unsigned int)drx, entry,
				 drx->des2);

			/* Check if the packet is long enough to accept without
			   copying to a minimally-sized skbuff. */
// FIXME We don't support copying at this point
#if 0
			if ((frame_len < rx_copybreak) &&
					(skb =
					 netdev_alloc_skb(dev, frame_len + 2)) != NULL) {
				skb_reserve(skb, NET_IP_ALIGN);
				dma_sync_single_for_cpu(lp->device,
						lp->
						rx_skbuff_dma[entry],
						frame_len,
						DMA_FROM_DEVICE);
				skb_copy_to_linear_data(skb,
						lp->rx_skbuff[entry]->
						data, frame_len);

				skb_put(skb, frame_len);
				dma_sync_single_for_device(lp->device,
						lp->
						rx_skbuff_dma[entry],
						frame_len,
						DMA_FROM_DEVICE);
			} else	/* zero-copy */
#endif
			{
				skb = lp->rx_skbuff[entry];
				if (unlikely(!skb)) {
					printk(KERN_ERR "%s: Inconsistent Rx "
							"descriptor chain.\n",
							dev->name);
					lp->stats.rx_dropped++;
					break;
				}
				lp->rx_skbuff[entry] = NULL;
				skb_put(skb, frame_len);
				dma_unmap_single(lp->device,
						lp->rx_skbuff_dma[entry],
						frame_len, DMA_FROM_DEVICE);
			}
#ifdef STMMAC_DEBUG
			if (netif_msg_pktdata(lp)) {
				printk(KERN_DEBUG " - frame received: ");
				print_pkt(skb->data, frame_len);
			}
#endif
			skb->protocol = eth_type_trans(skb, dev);
			lp->mac->ops->rx_checksum(skb, status);

			netif_receive_skb(skb);

			lp->stats.rx_packets++;
			lp->stats.rx_bytes += frame_len;
			dev->last_rx = jiffies;
		}
		entry = (++lp->cur_rx) % rxsize;
		drx = lp->dma_rx + entry;
		if( ++work_done == budget )
			break;
	}

	stmmac_rx_refill(dev);
	if( work_done < budget )
	{
		napi_complete(napi);
		RX_DBG("<<< stmmmac_poll: poll stopped and exits...\n");
		stmmac_dma_enable_irq_rx(ioaddr);
	}
	else
	{
		RX_DBG("<<< stmmmac_poll: not done... \n");
	}

	return work_done;
}

/**
 *  stmmac_tx_timeout
 *  @dev : Pointer to net device structure
 *  Description: this function is called when a packet transmission fails to
 *   complete within a reasonable period. The driver will mark the error in the
 *   netdev structure and arrange for the device to be reset to a sane state
 *   in order to transmit a new packet.
 */
void stmmac_tx_timeout(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	printk(KERN_WARNING "%s: Tx timeout at %ld, latency %ld\n",
			dev->name, jiffies, (jiffies - dev->trans_start));

	return;

// FIXME
{
	unsigned int status;
	unsigned int ioaddr = dev->base_addr;
	printk("(current=%d, dirty=%d)\n", (lp->cur_tx % lp->dma_tx_size),
	       (lp->dirty_tx % lp->dma_tx_size));

	status = (unsigned int)readl(ioaddr + DMA_STATUS);
	printk("[CSR5: 0x%08x]\n", status);
}

#ifdef STMMAC_DEBUG
	printk("(current=%d, dirty=%d)\n", (lp->cur_tx % lp->dma_tx_size),
			(lp->dirty_tx % lp->dma_tx_size));
	printk("DMA tx ring status: \n");
	display_dma_desc_ring(lp->dma_tx, lp->dma_tx_size);
#endif
	netif_stop_queue(dev);
	spin_lock(&lp->tx_lock);
	stmmac_dma_stop_tx(dev->base_addr);
	clear_dma_descs(lp->dma_tx, lp->dma_tx_size, 0);
	stmmac_dma_start_tx(dev->base_addr);
	spin_unlock(&lp->tx_lock);

	lp->stats.tx_errors++;
	dev->trans_start = jiffies;
	netif_wake_queue(dev);

	return;
}

/**
 *  stmmac_stats
 *  @dev : Pointer to net device structure
 *  Description: this function returns statistics to the caller application
 */
struct net_device_stats *stmmac_stats(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	return &lp->stats;
}

/* Configuration changes (passed on by ifconfig) */
int stmmac_config(struct net_device *dev, struct ifmap *map)
{
	if (dev->flags & IFF_UP)	/* can't act on a running interface */
		return -EBUSY;

	/* Don't allow changing the I/O address */
	if (map->base_addr != dev->base_addr) {
		printk(KERN_WARNING "%s: can't change I/O address\n",
				dev->name);
		return -EOPNOTSUPP;
	}

	/* Don't allow changing the IRQ */
	if (map->irq != dev->irq) {
		printk(KERN_WARNING "%s: can't change IRQ number %d\n",
				dev->name, dev->irq);
		return -EOPNOTSUPP;
	}

	/* ignore other fields */
	return 0;
}

/**
 *  stmmac_multicast_list - entry point for multicast addressing
 *  @dev : pointer to the device structure
 *  Description:
 *  This function is a driver entry point which gets called by the kernel
 *  whenever multicast addresses must be enabled/disabled.
 *  Return value:
 *  void.
 */
static void stmmac_multicast_list(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	/* Calling the hw function. */
	lp->mac->ops->set_filter(dev);

	return;
}

/**
 *  stmmac_change_mtu - entry point to change MTU size for the device.
 *   @dev : device pointer.
 *   @new_mtu : the new MTU size for the device.
 *   Description: the Maximum Transfer Unit (MTU) is used by the network layer
 *     to drive packet transmission. Ethernet has an MTU of 1500 octets 
 *     (ETH_DATA_LEN). This value can be changed with ifconfig.
 *  Return value:
 *   0 on success and an appropriate (-)ve integer as defined in errno.h
 *   file on failure.
 */
static int stmmac_change_mtu(struct net_device *dev, int new_mtu)
{
	if (netif_running(dev)) {
		printk(KERN_ERR "%s: must be stopped to change its MTU\n",
				dev->name);
		return -EBUSY;
	}

	if ((new_mtu < MIN_MTU) || (new_mtu > MAX_MTU))
		return -EINVAL;

	dev->mtu = new_mtu;

	return 0;
}

static irqreturn_t stmmac_interrupt(int irq, void *dev_id)
{
	struct net_device *dev = (struct net_device *)dev_id;

	if (unlikely(!dev)) {
		printk(KERN_ERR "%s: invalid dev pointer\n", __FUNCTION__);
		return IRQ_NONE;
	}

	stmmac_dma_interrupt(dev);

	return IRQ_HANDLED;
}

#ifdef CONFIG_NET_POLL_CONTROLLER
/* Polling receive - used by NETCONSOLE and other diagnostic tools
 * to allow network I/O with interrupts disabled. */
static void stmmac_poll_controller(struct net_device *dev)
{
	disable_irq(dev->irq);
	stmmac_interrupt(dev->irq, dev);
	enable_irq(dev->irq);
}
#endif

/**
 *  stmmac_ioctl - Entry point for the Ioctl
 *  @dev :  Device pointer.
 *  @rq :  An IOCTL specefic structure, that can contain a pointer to
 *  a proprietary structure used to pass information to the driver.
 *  @cmd :  IOCTL command
 *  Description:
 *  Currently there are no special functionality supported in IOCTL, just the
 *  phy_mii_ioctl(...) can be invoked.
 */
static int stmmac_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	int ret = -EOPNOTSUPP;

	if (!netif_running(dev))
		return -EINVAL;

	switch (cmd) {
		case SIOCGMIIPHY:
		case SIOCGMIIREG:
		case SIOCSMIIREG:
			if (!lp->phydev)
				return -EINVAL;

			spin_lock(&lp->lock);
			ret = phy_mii_ioctl(lp->phydev, if_mii(rq), cmd);
			spin_unlock(&lp->lock);
		default:
			break;
	}
	return ret;
}

#ifdef STMMAC_VLAN_TAG_USED
static void stmmac_vlan_rx_register(struct net_device *dev,
				    struct vlan_group *grp)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	spin_lock(&lp->lock);
	/* VLAN Tag identifier register already contains the VLAN tag ID. 
	   (see hw mac initialization). */
	lp->vlgrp = grp;
	spin_unlock(&lp->lock);
}
#endif

static const struct net_device_ops stmmac_netdev_ops = {
	.ndo_open               = stmmac_open,
	.ndo_stop               = stmmac_release,
	.ndo_start_xmit         = stmmac_xmit,
	.ndo_set_multicast_list = stmmac_multicast_list,
	.ndo_do_ioctl           = stmmac_ioctl,
	.ndo_tx_timeout         = stmmac_tx_timeout,
	.ndo_change_mtu         = stmmac_change_mtu,
	.ndo_set_config         = stmmac_config,
	.ndo_get_stats          = stmmac_stats,
	.ndo_set_mac_address    = eth_mac_addr,
	.ndo_validate_addr      = eth_validate_addr,
#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller    = stmmac_poll_controller,
#endif
#ifdef STMMAC_VLAN_TAG_USED
	.ndo_vlan_rx_register    = stmmac_vlan_rx_register,
#endif
};
/**
 *  stmmac_probe - Initialization of the adapter .
 *  @dev : device pointer
 *  Description: The function initializes the network device structure for
 *		the STMMAC driver. It also calls the low level routines 
 *		 in order to init the HW (i.e. the DMA engine)
 */
static int stmmac_probe(struct net_device *dev)
{
	int ret = 0;
	struct eth_driver_local *lp = netdev_priv(dev);
	lp->dev = dev;

	stmmac_verify_args();

	ether_setup(dev);

	dev->netdev_ops = &stmmac_netdev_ops;

	dev->features |= (NETIF_F_SG /*| NETIF_F_TSO */| NETIF_F_HW_CSUM |
			  NETIF_F_HIGHDMA | NETIF_F_LLTX |
			  NETIF_F_FRAGLIST);
#ifdef STMMAC_VLAN_TAG_USED
	/*Supports IEEE 802.1Q VLAN tag detection for reception frames */
	dev->features |= NETIF_F_HW_VLAN_RX;
#endif

	dev->watchdog_timeo = msecs_to_jiffies(watchdog);
	SET_ETHTOOL_OPS(dev, &stmmac_ethtool_ops);

	lp->msg_enable = netif_msg_init(debug, default_msg_level);

	lp->rx_csum = 0;

	lp->dma_tx_size = dma_tx_size_param;
	lp->dma_rx_size = dma_rx_size_param;

// FIXME
//	if (flow_ctrl)
//		lp->flow_ctrl = FLOW_AUTO;	/* RX/TX pause on */
	lp->flow_ctrl = FLOW_AUTO;
	lp->pause = pause;

	netif_napi_add(dev, &lp->napi, stmmac_poll, lp->dma_rx_size);

	/* Get the MAC address */
	get_mac_address(dev->base_addr, dev->dev_addr,
			lp->mac->hw.addr_high, lp->mac->hw.addr_low);

	/* 
	 * ifcofig is run after probe so this is false in our case 
	if (!is_valid_ether_addr(dev->dev_addr)) {
		printk(KERN_WARNING "stmmac: please set MAC address\n");
	}
	*/

	if ((ret = register_netdev(dev))) {
		printk(KERN_ERR "%s: ERROR %i registering the device\n",
				__FUNCTION__, ret);
		return -ENODEV;
	}

	spin_lock_init(&lp->lock);
	spin_lock_init(&lp->tx_lock);

#ifdef CONFIG_ARCH_MERLIN
	{
		/* 
		 * We must tell the Ethernet in which mode the PHY is going to be for next time 
		 * NOTE: this must be done before the MAC is getting out of reset in open
		 */
		mobi_qccsisc_sys_config_t sysconfig;

		sysconfig.r = mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);

		/* note RMII and RGMII are untested */
		if (lp->phy_interface == PHY_INTERFACE_MODE_RMII) {
			sysconfig.b.phy_intf_sel = 0x4;
		} else if (lp->phy_interface == PHY_INTERFACE_MODE_RGMII) {
			sysconfig.b.phy_intf_sel = 0x1;
		} else if (lp->phy_interface == PHY_INTERFACE_MODE_MII) {
			sysconfig.b.phy_intf_sel = 0x0;
		} else if (lp->phy_interface == PHY_INTERFACE_MODE_GMII) {
			sysconfig.b.phy_intf_sel = 0x0;
		} else {
			printk(KERN_ERR "%s: ERROR Unsupported either phy interface type %d\n",
				__FUNCTION__, lp->phy_interface);
		}

		mobi_qcc_writereg(sysconfig.r, 
				MOBI_QCCSISC_SYS_CONFIG_OFFSET);
	}
#endif

#ifdef CONFIG_ARCH_FALCON
	{
		/* 
		 * We must tell the Ethernet in which mode the PHY is going to be for next time 
		 * NOTE: this must be done before the MAC is getting out of reset in open
		 */
		qcc_chipctl_sysconfig_t sysconfig;

		mobi_qcc_read(QCC_BID_CHIPCTL, 
				QCC_CHIPCTL_SYSCONFIG_OFFSET,
				(unsigned long *) &sysconfig.r, 4);

		if (lp->phy_interface == PHY_INTERFACE_MODE_RMII) {
			sysconfig.b.phy_intf_sel_i = 0x4;
		} else if (lp->phy_interface == PHY_INTERFACE_MODE_RGMII) {
			sysconfig.b.phy_intf_sel_i = 0x1;
		} else {
			printk(KERN_ERR "%s: ERROR Unsupported either phy interface type %d\n",
				__FUNCTION__, lp->phy_interface);
		}

		mobi_qcc_write(QCC_BID_CHIPCTL, 
				QCC_CHIPCTL_SYSCONFIG_OFFSET,
				sysconfig.r, 4);
	}
#endif

	return ret;
}

/**
 * stmmac_mac_device_setup
 * @dev : device pointer
 * Description: it detects and inits the hw mac device.
 */
static inline void stmmac_mac_device_setup(struct net_device *dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	unsigned long ioaddr = dev->base_addr;

	struct device_info_t *device = gmac_setup(ioaddr);

	if (device == NULL) {
		printk(KERN_ERR "%s: Unrecognized device detected\n", __func__);
	}
	lp->mac = device;
}

/**
 * stmmacphy_dvr_probe
 * @pdev: platform device pointer
 * Description: The driver is initialized through the platform_device
 * 		structures which define the configuration needed by the SoC.
 */
static int stmmacphy_dvr_probe(struct platform_device *pdev)
{
	struct plat_stmmacphy_data *plat_dat;
	plat_dat = (struct plat_stmmacphy_data *)((pdev->dev).platform_data);

	printk(KERN_DEBUG "stmmacphy_dvr_probe: added phy for bus %d\n",
			plat_dat->bus_id);

	return 0;
}

static int stmmacphy_dvr_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver stmmacphy_driver = {
	.driver = {
		.name = PHY_RESOURCE_NAME,
	},
	.probe = stmmacphy_dvr_probe,
	.remove = stmmacphy_dvr_remove,
};

/**
 * stmmac_associate_phy
 * @dev: pointer to device structure
 * @data: points to the private structure.
 * Description: Scans through all the PHYs we have registered and checks if
 *		any are associated with our MAC.  If so, then just fill in
 *		the blanks in our local context structure
 */
static int stmmac_associate_phy(struct device *dev, void *data)
{
	struct eth_driver_local *lp = (struct eth_driver_local *)data;
	struct plat_stmmacphy_data *plat_dat;

	plat_dat = (struct plat_stmmacphy_data *)(dev->platform_data);

	DBG(probe, DEBUG,
			"stmmac_associate_phy: checking phy for bus %d\n", plat_dat->bus_id);

	/* Check that this phy is for the MAC being initialised */
	if (lp->bus_id != plat_dat->bus_id)
		return 0;

	/* OK, this PHY is connected to the MAC.  Go ahead and get the parameters */
	DBG(probe, DEBUG, "stmmac_associate_phy: OK. Found PHY config\n");
	lp->phy_irq =
		platform_get_irq_byname(to_platform_device(dev), "phyirq");
	DBG(probe, DEBUG,
			"stmmac_associate_phy: PHY irq on bus %d is %d\n",
			plat_dat->bus_id, lp->phy_irq);

	lp->phy_addr = plat_dat->phy_addr;
	lp->phy_mask = plat_dat->phy_mask;
	/* pull in the phy interface as defined in the board module */
	plat_dat->interface = lp->phy_interface;
	lp->phy_reset = plat_dat->phy_reset;

	DBG(probe, DEBUG, "stmmacphy_dvr_probe: exiting\n");
	return 1;	/* forces exit of driver_for_each_device() */
}

/**
 * stmmac_dvr_probe
 * @pdev: amba device pointer
 * Description: The driver is initialized through amba_device.  
 */
static int stmmac_dvr_probe(struct amba_device *pdev, struct amba_id *id)
{
	int ret = 0;
	struct resource *res = &pdev->res;
	unsigned int *addr = NULL;
	struct net_device *ndev = NULL;
	struct plat_stmmacenet_data *plat_dat = ((struct amba_id*)id)->data;
	struct eth_driver_local *lp;

	mobi_reset_disable(RESET_ID_GMAC);

	ret = amba_request_regions(pdev, ETH_RESOURCE_NAME);
	if (ret < 0) {
		printk(KERN_ERR "%s: ERROR: memory allocation failed"
				" cannot get the I/O addr 0x%x\n",
				__FUNCTION__, (unsigned int)res->start);
		goto out;
	}

	addr = ioremap(res->start, res->end - res->start + 1);
	if (!addr) {
		printk(KERN_ERR "%s: ERROR: memory mapping failed \n",
				__FUNCTION__);
		ret = -ENOMEM;
		goto out;
	}

	reset_dev.data = pdev;
	ret = codec_reset_device_register(&reset_dev);
	if(ret < 0) {
		printk(KERN_ERR "%s: ERROR: codec reset registration failed \n",
				__FUNCTION__);
		goto out;
	}

	ndev = alloc_etherdev(sizeof(struct eth_driver_local));
	if (!ndev) {
		printk(KERN_ERR "%s: ERROR: allocating the device\n",
				__FUNCTION__);
		ret = -ENOMEM;
		goto out;
	}

	/* Get the MAC information */
	if ((ndev->irq = pdev->irq[0]) < 0) {
		printk(KERN_ERR "%s: ERROR: MAC IRQ configuration "
				"information not found\n", __FUNCTION__);
		ret = -ENODEV;
		goto out;
	}

	lp = netdev_priv(ndev);
	lp->device = &(pdev->dev);
	lp->bus_id = plat_dat->bus_id;
	lp->pbl = plat_dat->pbl;
	lp->fb = plat_dat->fb;
	lp->gmac_pdata = (struct dw_gmac_driver_data_t*)pdev->dev.platform_data;

	lp->phy_interface = (lp->gmac_pdata)->phy_interface;

	amba_set_drvdata(pdev, ndev);

	/* Set the I/O base addr */
	ndev->base_addr = (unsigned long)addr;

	/* MAC HW device detection */
	stmmac_mac_device_setup(ndev);

	/* Network Device Registration */
	ret = stmmac_probe(ndev);
	if (ret < 0) {
		goto out;
	}

	/* associate a PHY - it is provided by another platform bus */
	if (!driver_for_each_device(&(stmmacphy_driver.driver), 
				NULL, (void *)lp, stmmac_associate_phy)) {
		printk(KERN_ERR "No PHY device is associated with this MAC!\n");
		ret = -ENODEV;
		goto out;
	}

	lp->bsp_priv = plat_dat->bsp_priv;

	/* MDIO bus Registration */
	ret = stmmac_mdio_register(ndev);

out:
	if (ret < 0) {
		mobi_reset_enable(RESET_ID_GMAC);
		amba_set_drvdata(pdev, NULL);
		amba_release_regions(pdev);
		if (addr != NULL)
			iounmap(addr);
		codec_reset_device_unregister(&reset_dev);
	}

	return ret;
}

/**
 * stmmac_dvr_remove
 * @pdev: amba device pointer
 * Description: This function resets the TX/RX processes, disables the MAC RX/TX
 *   changes the link status, releases the DMA descriptor rings,
 *   unregisters the MDIO busm and unmaps allocated memory.
 */
static int stmmac_dvr_remove(struct amba_device *pdev)
{
	struct net_device *ndev = amba_get_drvdata(pdev);

	if(!ndev)
		return 0;

	stmmac_dma_stop_rx(ndev->base_addr);
	stmmac_dma_stop_tx(ndev->base_addr);

	stmmac_mac_disable_rx(ndev);
	stmmac_mac_disable_tx(ndev);

	netif_carrier_off(ndev);
	stmmac_mdio_unregister(ndev);

	codec_reset_device_unregister(&reset_dev);

	amba_set_drvdata(pdev, NULL);
	unregister_netdev(ndev);

	iounmap((void *)ndev->base_addr);
	amba_release_regions(pdev);

	free_netdev(ndev);

	mobi_reset_enable(RESET_ID_GMAC);

	return 0;
}

#ifdef CONFIG_PM
static int stmmac_suspend(struct amba_device *pdev, pm_message_t state)
{
	struct net_device *dev = amba_get_drvdata(pdev);
	unsigned long flags;
	struct eth_driver_local *lp = netdev_priv(dev);

	if (!dev || !netif_running(dev))
		return 0;

	netif_device_detach(dev);
	netif_stop_queue(dev);

	spin_lock_irqsave(&lp->lock, flags);

	/* Disable Rx and Tx */
	stmmac_dma_stop_tx(dev->base_addr);
	stmmac_dma_stop_rx(dev->base_addr);

	clear_dma_descs(lp->dma_tx, lp->dma_tx_size, 0);
	clear_dma_descs(lp->dma_rx, lp->dma_rx_size, OWN_BIT);

	/* Disable the MAC core */
	stmmac_mac_disable_tx(dev);
	stmmac_mac_disable_rx(dev);

	spin_unlock_irqrestore(&lp->lock, flags);

	return 0;
}

static int stmmac_resume(struct amba_device *pdev)
{
	struct net_device *dev = amba_get_drvdata(pdev);
	unsigned long flags;
	struct eth_driver_local *lp = netdev_priv(dev);

	if (!netif_running(dev))
		return 0;

	netif_device_attach(dev);

	spin_lock_irqsave(&lp->lock, flags);

	/* Enable the MAC/DMA */
	stmmac_mac_enable_rx(dev);
	stmmac_mac_enable_tx(dev);

	stmmac_dma_start_rx(dev->base_addr);
	stmmac_dma_start_tx(dev->base_addr);

	netif_start_queue(dev);

	spin_unlock_irqrestore(&lp->lock, flags);

	return 0;
}
#endif

static struct plat_stmmacenet_data stmmaceth_private_data = {
	.pbl      = 8,
	.fb       = 1,
	.bsp_priv = (void*)0,
};

static struct amba_id gmac_ids[] = {
	{
		.id     = DW_GMAC_AMBA_DEVID,
		.mask   = DW_GMAC_AMBA_DEVID_MASK,
		.data   = &stmmaceth_private_data,
	},
	{ 0, 0 },
};

static struct amba_driver stmmac_driver = {
	.drv    = {
		.name   = DW_GMAC_AMBA_NAME,
	},
	.probe      = stmmac_dvr_probe,
	.remove     = stmmac_dvr_remove,
#ifdef CONFIG_PM
	.suspend    = stmmac_suspend,
	.resume     = stmmac_resume,
#endif
	.id_table   = gmac_ids,
};

typedef int (*callback_t)(void*);
static struct codec_reset_device reset_dev = {
	.name       = DW_GMAC_AMBA_NAME,
	.suspend    = (callback_t)stmmac_suspend,
	.resume     = (callback_t)stmmac_resume,
};

static struct plat_stmmacphy_data phy_private_data = {
	.bus_id = 0,
	.phy_addr = 0,
	.phy_mask = 0,
	.phy_reset = NULL,
};

static void empty_platform_release(struct device *dev) { }

static struct platform_device myphy_device = {
	.name           = PHY_RESOURCE_NAME,
	.id             = 0,
	.num_resources  = 1,
	.resource       = (struct resource[]) {
		{
			.name   = "phyirq",
			.start  = -1,
			.end    = -1,
			.flags  = IORESOURCE_IRQ,
		},
	},
	.dev = {
		.platform_data = &phy_private_data,
		.release = empty_platform_release,
	}
};


/**
 * stmmac_init_module - Entry point for the driver
 * Description: This function is the entry point for the driver.
 */
static int __init stmmac_init_module(void)
{
	platform_device_register(&myphy_device);

	if (platform_driver_register(&stmmacphy_driver)) {
		printk(KERN_ERR "No PHY devices registered!\n");
		return -ENODEV;
	}
	
	return amba_driver_register(&stmmac_driver);
}

/**
 * stmmac_cleanup_module - Cleanup routine for the driver
 * Description: This function is the cleanup routine for the driver.
 */
static void __exit stmmac_cleanup_module(void)
{
	amba_driver_unregister(&stmmac_driver);
	platform_driver_unregister(&stmmacphy_driver);
	platform_device_unregister(&myphy_device);
}

#ifndef MODULE

static int __init stmmac_cmdline_opt(char *str)
{
	char *opt;

	if (!str || !*str)
		return -EINVAL;

	while ((opt = strsep(&str, ",")) != NULL) {
		if (!strncmp(opt, "msglvl:", 7)) {
			debug = simple_strtoul(opt + 7, NULL, 0);
		} else if (!strncmp(opt, "phyaddr:", 8)) {
			phy_n = simple_strtoul(opt + 8, NULL, 0);
		} else if (!strncmp(opt, "watchdog:", 9)) {
			watchdog = simple_strtoul(opt + 9, NULL, 0);
		} else if (!strncmp(opt, "minrx:", 6)) {
			rx_copybreak = simple_strtoul(opt + 6, NULL, 0);
		} else if (!strncmp(opt, "bfsize:", 7)) {
			dma_buffer_size = simple_strtoul(opt + 7, NULL, 0);
		} else if (!strncmp(opt, "txsize:", 7)) {
			dma_tx_size_param = simple_strtoul(opt + 7, NULL, 0);
		} else if (!strncmp(opt, "rxsize:", 7)) {
			dma_rx_size_param = simple_strtoul(opt + 7, NULL, 0);
		} else if (!strncmp(opt, "flow_ctrl:", 10)) {
			flow_ctrl = simple_strtoul(opt + 10, NULL, 0);
		} else if (!strncmp(opt, "pause:", 6)) {
			pause = simple_strtoul(opt + 6, NULL, 0);
		}
	}
	return 0;
}
__setup("stmmaceth=", stmmac_cmdline_opt);

#endif

module_init(stmmac_init_module);
module_exit(stmmac_cleanup_module);

MODULE_DESCRIPTION("STMMAC 10/100/1000 Ethernet driver");
MODULE_AUTHOR("Giuseppe Cavallaro, Gregoire Pean <gpean@mobilygen.com>");
MODULE_LICENSE("GPL");
