/*
 * drivers/net/stmmac/stmmac_ethtool.c
 *
 * STMMAC Ethernet Driver
 * Ethtool support for STMMAC Ethernet Driver
 *
 * Author: Giuseppe Cavallaro
 *
 * Copyright (c) 2006-2007 STMicroelectronics
 *
 */
#include <linux/kernel.h>
#include <linux/etherdevice.h>
#include <linux/mm.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/phy.h>
#include <asm/io.h>

#include "stmmac.h"

#define REG_SPACE_SIZE	0x1058

void stmmac_ethtool_getdrvinfo(struct net_device *dev,
			       struct ethtool_drvinfo *info)
{
	strcpy(info->driver, ETH_RESOURCE_NAME);
	strcpy(info->version, DRV_MODULE_VERSION);
	info->fw_version[0] = '\0';
	return;
}

int stmmac_ethtool_getsettings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	struct phy_device *phy = lp->phydev;
	int rc;
	if (phy == NULL) {
		printk(KERN_ERR "%s: %s: PHY is not registered\n",
		       __FUNCTION__, dev->name);
		return -ENODEV;
	}

	if (!netif_running(dev)) {
		printk(KERN_ERR "%s: interface is disabled: we cannot track "
		       "link speed / duplex setting\n", dev->name);
		return -EBUSY;
	}

	cmd->transceiver = XCVR_INTERNAL;
	spin_lock_irq(&lp->lock);
	rc = phy_ethtool_gset(phy, cmd);
	spin_unlock_irq(&lp->lock);
	return rc;
}

int stmmac_ethtool_setsettings(struct net_device *dev, struct ethtool_cmd *cmd)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	struct phy_device *phy = lp->phydev;
	int rc;

	spin_lock(&lp->lock);
	rc = phy_ethtool_sset(phy, cmd);
	spin_unlock(&lp->lock);

	return rc;
}

u32 stmmac_ethtool_getmsglevel(struct net_device * dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	return lp->msg_enable;
}

void stmmac_ethtool_setmsglevel(struct net_device *dev, u32 level)
{
	struct eth_driver_local *lp = netdev_priv(dev);
	lp->msg_enable = level;

}

int stmmac_check_if_running(struct net_device *dev)
{
	if (!netif_running(dev))
		return -EBUSY;
	return (0);
}

int stmmac_ethtool_get_regs_len(struct net_device *dev)
{
	return (REG_SPACE_SIZE);
}

void stmmac_ethtool_gregs(struct net_device *dev,
			  struct ethtool_regs *regs, void *space)
{
	int i;
	u32 reg;
	u32 *reg_space = (u32 *) space;

	memset(reg_space, 0x0, REG_SPACE_SIZE);

	/* MAC registers */
	for (i = 0; i < 11; i++) {
		reg = readl(dev->base_addr + i * 4);
		reg_space[i] = reg;
	}

	/* DMA registers */
	for (i = 0; i < 9; i++) {
		reg = readl(dev->base_addr + (DMA_BUS_MODE + i * 4));
		reg_space[DMA_BUS_MODE/4 + i] = reg;
	}
	reg = readl(dev->base_addr + DMA_CUR_TX_BUF_ADDR);
	reg_space[DMA_CUR_TX_BUF_ADDR/4] = reg;
	reg = readl(dev->base_addr + DMA_CUR_RX_BUF_ADDR);
	reg_space[DMA_CUR_RX_BUF_ADDR/4] = reg;

	return;
}

int stmmac_ethtool_set_tx_csum(struct net_device *dev, u32 data)
{
	if (data)
		dev->features |= NETIF_F_HW_CSUM;
	else
		dev->features &= ~NETIF_F_HW_CSUM;

	return 0;
}

u32 stmmac_ethtool_get_rx_csum(struct net_device * dev)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	return (lp->rx_csum);
}

int stmmac_ethtool_set_rx_csum(struct net_device *dev, u32 data)
{
	struct eth_driver_local *lp = netdev_priv(dev);

	if (data)
		lp->rx_csum = 1;
	else
		lp->rx_csum = 0;

	return 0;
}

static void
stmmac_get_pauseparam(struct net_device *netdev,
		      struct ethtool_pauseparam *pause)
{
	struct eth_driver_local *lp = netdev_priv(netdev);

	spin_lock(&lp->lock);

	pause->rx_pause = pause->tx_pause = 0;
	pause->autoneg = lp->phydev->autoneg;

	if (lp->flow_ctrl & FLOW_RX)
		pause->rx_pause = 1;
	if (lp->flow_ctrl & FLOW_TX)
		pause->tx_pause = 1;

	spin_unlock(&lp->lock);
	return;
}

static int
stmmac_set_pauseparam(struct net_device *netdev,
		      struct ethtool_pauseparam *pause)
{
	struct eth_driver_local *lp = netdev_priv(netdev);
	struct phy_device *phy = lp->phydev;
	int new_pause = FLOW_OFF;
	int ret = 0;

	spin_lock(&lp->lock);

	if (pause->rx_pause)
		new_pause |= FLOW_RX;
	if (pause->tx_pause)
		new_pause |= FLOW_TX;

	lp->flow_ctrl = new_pause;

	if (phy->autoneg) {
		if (netif_running(netdev)) {
			struct ethtool_cmd cmd;
			/* auto-negotiation automatically restarted */
			cmd.cmd = ETHTOOL_NWAY_RST;
			cmd.supported = phy->supported;
			cmd.advertising = phy->advertising;
			cmd.autoneg = phy->autoneg;
			cmd.speed = phy->speed;
			cmd.duplex = phy->duplex;
			cmd.phy_address = phy->addr;
			ret = phy_ethtool_sset(phy, &cmd);
		}
	} else {
		unsigned long ioaddr = netdev->base_addr;
		lp->mac->ops->flow_ctrl(ioaddr, phy->duplex,
					lp->flow_ctrl, lp->pause);
	}
	spin_unlock(&lp->lock);
	return ret;
}


/* statistics */

static const struct {
	const char name[ETH_GSTRING_LEN];
} stmmac_stat_names[] = {
	{ "mmc_cntrl" }, 
	{ "mmc_intr_rx" }, 
	{ "mmc_intr_tx" }, 
	{ "mmc_intr_mask_rx" }, 
	{ "mmc_intr_mask_tx" }, 
	{ "txoctetcount_gb" }, 
	{ "txframecount_gb" }, 
	{ "txbroadcastframes_g" }, 
	{ "txmulticastframes_g" }, 
	{ "tx64octets_gb" }, 
	{ "tx65to127octets_gb" }, 
	{ "tx128to255octets_gb" }, 
	{ "tx256to511octets_gb" }, 
	{ "tx512to1023octets_gb" }, 
	{ "tx1024tomaxoctets_gb" }, 
	{ "txunicastframes_gb" }, 
	{ "txmulticastframes_gb" }, 
	{ "txbroadcastframes_gb" }, 
	{ "txunderflowerror" }, 
	{ "txsinglecol_g" }, 
	{ "txmulticol_g" }, 
	{ "txdeferred" }, 
	{ "txlatecol" }, 
	{ "txexesscol" }, 
	{ "txcarriererror" }, 
	{ "txoctetcount_g" }, 
	{ "txframecount_g" }, 
	{ "txexcessdef" }, 
	{ "txpauseframes" }, 
	{ "txvlanframes_g" }, 
	{ "rxframecount_gb" }, 
	{ "rxoctetcount_gb" }, 
	{ "rxoctetcount_g" }, 
	{ "rxbroadcastframes_g" }, 
	{ "rxmulticastframes_g" }, 
	{ "rxcrcerror" }, 
	{ "rxalignmenterror" }, 
	{ "rxrunterror" }, 
	{ "rxjabbererror" }, 
	{ "rxundersize_g" }, 
	{ "rxoversize_g" }, 
	{ "rx64octets_gb" }, 
	{ "rx65to127octets_gb" }, 
	{ "rx128to255octets_gb" }, 
	{ "rx256to511octets_gb" }, 
	{ "rx512to1023octets_gb" }, 
	{ "rx1024tomaxoctets_gb" }, 
	{ "rxunicastframes_g" }, 
	{ "rxlengtherror" }, 
	{ "rxoutofrangetype" }, 
	{ "rxpauseframes" }, 
	{ "rxfifooverflow" }, 
	{ "rxvlanframes_gb" }, 
	{ "rxwatchdogerror" }, 
	{ "mmc_ipc_intr_mask_rx" }, 
	{ "mmc_ipc_intr_rx" }, 
	{ "rxipv4_gd_frms" }, 
	{ "rxipv4_hdrerr_frms" }, 
	{ "rxipv4_nopay_frms" }, 
	{ "rxipv4_frag_frms" }, 
	{ "rxipv4_udsbl_frms" }, 
	{ "rxipv6_gd_frms" }, 
	{ "rxipv6_hdrerr_frms" }, 
	{ "rxipv6_nopay_frms" }, 
	{ "rxudp_gd_frms" }, 
	{ "rxudp_err_frms" }, 
	{ "rxtcp_gd_frms" }, 
	{ "rxtcp_err_frms" }, 
	{ "rxicmp_gd_frms" }, 
	{ "rxicmp_err_frms" }, 
	{ "rxipv4_gd_octets" }, 
	{ "rxipv4_hdrerr_octets" }, 
	{ "rxipv4_nopay_octets" }, 
	{ "rxipv4_frag_octets" }, 
	{ "rxipv4_udsbl_octets" }, 
	{ "rxipv6_gd_octets" }, 
	{ "rxipv6_hdrerr_octets" }, 
	{ "rxipv6_nopay_octets" }, 
	{ "rxudp_gd_octets" }, 
	{ "rxudp_err_octets" }, 
	{ "rxtcp_gd_octets" }, 
	{ "rxtcp_err_octets" }, 
	{ "rxicmp_gd_octets" }, 
	{ "rxicmp_err_octets" }, 
};

static const u32 stmmac_stat_regs[] = {
	0x0100, 
	0x0104, 
	0x0108, 
	0x010C, 
	0x0110, 
	0x0114, 
	0x0118, 
	0x011C, 
	0x0120, 
	0x0124, 
	0x0128, 
	0x012C, 
	0x0130, 
	0x0134, 
	0x0138, 
	0x013C, 
	0x0140, 
	0x0144, 
	0x0148, 
	0x014C, 
	0x0150, 
	0x0154, 
	0x0158, 
	0x015C, 
	0x0160, 
	0x0164, 
	0x0168, 
	0x016C, 
	0x0170, 
	0x0174, 
	0x0180, 
	0x0184, 
	0x0188, 
	0x018C, 
	0x0190, 
	0x0194, 
	0x0198, 
	0x019C, 
	0x01A0, 
	0x01A4, 
	0x01A8, 
	0x01AC, 
	0x01B0, 
	0x01B4, 
	0x01B8, 
	0x01BC, 
	0x01C0, 
	0x01C4, 
	0x01C8, 
	0x01CC, 
	0x01D0, 
	0x01D4, 
	0x01D8, 
	0x01DC, 
	0x0200, 
	0x0208, 
	0x0210, 
	0x0214, 
	0x0218, 
	0x021C, 
	0x0220, 
	0x0224, 
	0x0228, 
	0x022C, 
	0x0230, 
	0x0234, 
	0x0238, 
	0x023C, 
	0x0240, 
	0x0244, 
	0x0250, 
	0x0254, 
	0x0258, 
	0x025C, 
	0x0260, 
	0x0264, 
	0x0268, 
	0x026C, 
	0x0270, 
	0x0274, 
	0x0278, 
	0x027C, 
	0x0280, 
	0x0284,
};

static int stmmac_get_stats_count(struct net_device *dev)
{
	return sizeof(stmmac_stat_names) / sizeof(*stmmac_stat_names);
}

static void stmmac_get_strings(struct net_device *dev, u32 stringset, u8 *buffer)
{
	switch (stringset) {
	case ETH_SS_STATS:
		memcpy(buffer, &stmmac_stat_names, sizeof(stmmac_stat_names));
		break;
	case ETH_SS_TEST:
		/* not implemented */
		break;
	}
}

static void stmmac_get_ethtool_stats(struct net_device *dev, struct ethtool_stats *estats, u64 *buffer)
{
	int i;
	unsigned long ioaddr = dev->base_addr;
	for (i = 0; i < sizeof(stmmac_stat_names) / sizeof(*stmmac_stat_names); i++)
		buffer[i] = readl(ioaddr + stmmac_stat_regs[i]);
}

struct ethtool_ops stmmac_ethtool_ops = {
	.begin = stmmac_check_if_running,
	.get_drvinfo = stmmac_ethtool_getdrvinfo,
	.get_settings = stmmac_ethtool_getsettings,
	.set_settings = stmmac_ethtool_setsettings,
	.get_msglevel = stmmac_ethtool_getmsglevel,
	.set_msglevel = stmmac_ethtool_setmsglevel,
	.get_regs = stmmac_ethtool_gregs,
	.get_regs_len = stmmac_ethtool_get_regs_len,
	.get_link = ethtool_op_get_link,
	.get_rx_csum = stmmac_ethtool_get_rx_csum,
	.set_rx_csum = stmmac_ethtool_set_rx_csum,
	.get_tx_csum = ethtool_op_get_tx_csum,
	.set_tx_csum = stmmac_ethtool_set_tx_csum,
	.get_sg = ethtool_op_get_sg,
	.set_sg = ethtool_op_set_sg,
#ifdef NETIF_F_TSO
	.get_tso = ethtool_op_get_tso,
	.set_tso = ethtool_op_set_tso,
#endif
	.get_ufo = ethtool_op_get_ufo,
	.set_ufo = ethtool_op_set_ufo,
	.get_pauseparam = stmmac_get_pauseparam,
	.set_pauseparam = stmmac_set_pauseparam,
	.get_strings = stmmac_get_strings,
        .get_stats_count = stmmac_get_stats_count,
	.get_ethtool_stats = stmmac_get_ethtool_stats,
};
