/* 
 *  linux/drivers/mtd/nand/aumb3000_nand.h
 *  
 *  Copyright (C) 2006 Mobilygen Corp
 *  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#ifndef AUMB3000_NAND_H
#define AUMB3000_NAND_H

#define AUMB3000_NAND_SIZE_64M	    0
#define AUMB3000_NAND_SIZE_128M	    1
#define AUMB3000_NAND_SIZE_256M	    2
#define AUMB3000_NAND_SIZE_512M	    3
#define AUMB3000_NAND_SIZE_1G	    4
#define AUMB3000_NAND_SIZE_2G	    5
#define AUMB3000_NAND_SIZE_4G	    6
#define AUMB3000_NAND_SIZE_8G	    7

#define AUMB3000_NANDNUM_EXTERNAL_BANKS1	0
#define AUMB3000_NANDNUM_EXTERNAL_BANKS2	1
#define AUMB3000_NANDNUM_EXTERNAL_BANKS4	2

/* 
 *  this is width of bus from the controller to the chips, it is
 *  not the same as the width of the actual chip(8 or 16)
 */
#define AUMB3000_NANDCTRL_DATABUS_WIDTH8	0
#define AUMB3000_NANDCTRL_DATABUS_WIDTH16	1
#define AUMB3000_NANDCTRL_DATABUS_WIDTH32	2
#define AUMB3000_NANDCTRL_DATABUS_WIDTH64	3

/* 
 *  we can enable/disable ecc checking and generation 
 *  for the main and spare areas
 */
#define ECC_TYPE_NONE		0x0
#define ECC_TYPE_SPARE_ONLY	0x1
#define ECC_TYPE_MAIN_ONLY	0x2
#define ECC_TYPE_MAIN_AND_SPARE	0x3

struct timing_intervals {
	uint32_t interval0;
	uint32_t interval1;
	uint32_t interval2;
	uint32_t interval3;
	uint32_t interval4;
	uint32_t interval5;
};

struct flash_cmd {
	uint32_t nand_cmd;
	uint32_t au3000_cmd;
	uint32_t read_data;
	uint32_t page_addr;
	uint32_t erase_addr;
	uint32_t readoob_pos;
	uint32_t seq_addr;
	uint8_t  last_cmd_byte_read;
	uint32_t errno;
};


struct aumb3000_nand_info {

	uint8_t mfg_id;
	uint8_t dev_id;
	uint8_t chip_desc;

	uint8_t  select_chip;
	uint8_t  use_dma;
	uint8_t  chips_per_bank;
	uint8_t  banks_per_system;
	uint32_t ns_per_cycle;

	struct flash_cmd cmd;
	struct timing_intervals timing;

	uint32_t ctrl_shadow;
	uint32_t *ctrl_reg_mem;
	/* fields in the control register */
	int8_t sz;
	int8_t xs;
	int8_t eb;
	int8_t wd;
	int8_t gen_ecc;
	int8_t check_ecc;

	/* dma stuff */
	mobi_dma_handle dma_handle;
	uint32_t buf_dma_addr;
	uint32_t dma_length;
	uint8_t dma_mode;
	uint8_t prev_dma_mode;
	int32_t xfer_status;

	struct nand_hw_control controller;
	void __iomem *reg_base;
	void __iomem *data_base;

	/* a copy of our platform data to carry around */
	struct aumb3000_nand_device_data_t *pdata;
};

static uint8_t scan_ff_pattern[] = { 0xff, 0xff };

/*   samsung 8 bit small page bbt position */
static struct nand_bbt_descr samsungx8_smallpage_bbt = {
	.options = 0,
	.offs = 5,
	.len = 1,
	.pattern = scan_ff_pattern
};

/* samsung 16 bit small page bbt position */
/* even though 16 bit device, code in nand_bbt does byte
 *  access so offset are in bytes not shorts
 */
static struct nand_bbt_descr samsungx16_smallpage_bbt = {
	.options = 0,
	.offs = 11,
	.len = 1,
	.pattern = scan_ff_pattern
};

/*   samsung 8 bit large page bbt position */
static struct nand_bbt_descr samsungx8_largepage_bbt = {
	.options = 0,
	.offs = 0,
	.len = 1,
	.pattern = scan_ff_pattern
};

/* samsung 16 bit large page bbt position */
/* even though 16 bit device, code in nand_bbt does byte
 *  access so offset are in bytes not shorts
 */
static struct nand_bbt_descr samsungx16_largepage_bbt = {
	.options = 0,
	.offs = 0,
	.len = 2,
	.pattern = scan_ff_pattern
};

/* 
 *  small page 528B & x8 org. 
 *  Device 64Mb, 128Mb, 256Mb, 512Mb, 1Gb
 *  8 bit device, spare area 16 bytes wide
 *
 *  a 16-bit device will use the same layout since spare
 *  area is accessed via bytes
 */
static struct nand_ecclayout samsungx8_layout_oob_16 = {
	.eccbytes = 3,
	.eccpos = {6, 7, 8},
	.oobfree = {
		{.offset = 0, .length = 5},
		{.offset = 9, .length = 7}
	},
	.oobavail = 12,
};

/* 
 *  large page, 2KB & x8 org.
 *  Device: 1Gb, 2Gb, 4Gb
 *  8 bit device, spare area is 4 16bit areas = 64
 *
 *  a 16-bit device will use the same layout since spare
 *  area is accessed via bytes
 *
 *  jffs2 need 8 consecutive bytes for the clean marker,  the
 *  bb indicator is only found in the first byte of the spare 
 *  area, so byte 0 is available after that which means we can
 *  let the free area wrap.  the cleanmarker code only looks 
 *  at the first oobfree entry so change the order so that an
 *  entry >= to 8 is first!
 */
static struct nand_ecclayout samsungx8_layout_oob_64 = {
	.eccbytes = 12,
	.eccpos = {
		8,  9,  10,
		24, 25, 26,
		40, 41, 42,
		56, 57, 58,
	},
	.oobfree = {
		{.offset = 1,  .length = 7},
		{.offset = 11, .length = 13},
		{.offset = 27, .length = 13},
		{.offset = 43, .length = 13},
		{.offset = 59, .length = 5},
	},
	.oobavail = 51,
};

/* module params */
static char use_dma  = -1;
static char databus_16bit     = -1;
static char banks_per_system  = -1;
static char chips_per_bank    = -1;

/* default timing charervals taken from aurora nand simulations */
static  char tds  = -1;
static  char tcs  = -1;
static  char tals = -1;
static  char tcls = -1;
static  char tdh  = -1;
static  char tch  = -1;
static  char talh = -1;
static  char tclh = -1;
static  char trr  = -1;
static  char twb  = -1;
static  char twh  = -1;
static  char twp  = -1;
static  char tceh = -1;
static  char trb  = -1;
static  char treh = -1;
static  char trp  = -1;
static  char tir  = -1;
static  char twhr = -1;
static  char tclr = -1;
static  char trdelay = -1;
static  char tar  = -1;
static  char trhz = -1;
static  char tww  = -1;

#define AREA_MODE_SINGLE 	0x1
#define AREA_MODE_MULTIPLE	0x2

#define DMA_MODE_NONE    0
#define DMA_MODE_READ    1
#define DMA_MODE_WRITE   2

/* we access the nand chip thru a shared bus so if we are reading data or 
 *  executing a cmd that will access the chip we need to grab the hmux lock
 *  to avoid conflicts, defines used to indicate if need lock or not 
 */
#define NAND_LOCK_NO		0x0
#define NAND_LOCK_YES		0x5	/* grab and release in single call*/
#define NAND_LOCK_HOLD		0x1	/* grab but do not release */
#define NAND_LOCK_RELEASE	0x4	/* release held hmux lock */
#define NAND_LOCK_MASK		0x1
#define NAND_UNLOCK_MASK	0x4

#ifdef CONFIG_ARCH_MERLIN
#define NAND_LOCK	mobi_hmux_request()
#define NAND_UNLOCK	mobi_hmux_release()
#else
#define NAND_LOCK	do {} while(0)
#define NAND_UNLOCK	do {} while(0)
#endif

/* prototypes */
static uint32_t nand_reg_read(uint32_t offset, uint8_t lock);
static void nand_reg_write(uint32_t data, uint32_t addr, uint8_t lock);
static void program_timing_intervals(struct aumb3000_nand_info *info, 
		uint16_t timer_mask);

#define REG_READ(offset) \
	nand_reg_read(offset, NAND_LOCK_NO)

#define REG_READ_LOCKED(offset) \
	nand_reg_read(offset, NAND_LOCK_YES)

#define REG_WRITE(data, offset) \
	nand_reg_write(data, offset, NAND_LOCK_NO)

#define REG_WRITE_LOCKED(data, offset) \
	nand_reg_write(data, offset, NAND_LOCK_YES)

/* control reg access when already holding the lock */
#define CONTROL_WRITE(data) \
	REG_WRITE(data, MOBI_FLASH_NAND_CONTROL_OFFSET)

#define CONTROL_READ() \
	REG_READ(MOBI_FLASH_NAND_CONTROL_OFFSET)

#endif  /* AUMB3000_NAND_H */
