/*
 * drivers/mobimbox/mobi_mbox.c
 *
 * Copyright 2007 (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* enable doxygen */
/** \file
* Driver for Mobilygen Mailbox Interface
*/

#ifndef DOXYGEN_SKIP

#include <linux/init.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <linux/types.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/version.h>

#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/setup.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/map.h>
#include <asm/mach/time.h>

#include <asm/arch/platform.h>
#include <asm/arch/mobi_qccsisc.h>

#endif

#define MOBILYGEN_DEBUG_MBOX 1

#if MOBILYGEN_DEBUG_MBOX
#define dprintk(x...)   printk(x)
#else
#define dprintk(x...)   do {} while(0)
#endif

/* boy, these are some stupid, confusing names */
/* use to choose between  CPU0 or CPU1 interupt registers */
#define MBOX_INT_CPU0_MASK	0x0000
#define MBOX_INT_CPU1_MASK	0x0100

/* from each CPU, we can chose which CPU's interrupts we want
*  to look at.  in other words, a cpu can look at interrupts on 
*  both CPUs but can only write/clear it's own interrupts */
#define MBOX_CPU0_INT_REG	0x10
#define MBOX_CPU1_INT_REG	0x14

/* use to choose between  CPU0 or CPU1 data registers */
#define MBOX_DATA_CPU0_MASK	0x0200
#define MBOX_DATA_CPU1_MASK	0x0300

/* for a given CPU, use this mask to select the direction that
*  you're interested in looking at
*/
#define MBOX_DATA_CPU0_TO_CPU1_MASK 	0x0
#define MBOX_DATA_CPU1_TO_CPU0_MASK 	0x4

/* bit positions for exported mbox interrupts in the qcc register */
#define QCC_PER_INTPEND_MBOX0_MASK	0x2
#define QCC_PER_INTPEND_MBOX1_MASK	0x4
#define QCC_PER_INTPEND_MBOX4_MASK	0x8
#define QCC_PER_INTPEND_MBOX5_MASK	0x10

/* shift these base on mbox number to get interrupt */
#define MBOX_READ_INT_MASK	0x1
#define MBOX_RDY_INT_MASK	0x2

/* CPU1->CPU1 data mask */
#define MBOX0_DATA_MASK		0x04
#define MBOX1_DATA_MASK		0x14
#define MBOX2_DATA_MASK		0x24
#define MBOX3_DATA_MASK		0x34
#define MBOX4_DATA_MASK		0x44

// to create an offset for CPU0 data, mbx 3, dir 1 to 0 
// MBOX_DATA_CPU0_MASK | MBOX3_DATA_MASK | MBOX_DATA_CPU1_TO_CPU0_MASK
// final offset is 0x234

static irqreturn_t mobi_mbox_interrupt(int irq, void *dev_id)
{
	unsigned long data = 0;
	uint32_t perint_pend = 0;
	short mbox;


	perint_pend = mobi_qcc_readreg(MOBI_QCCSISC_PERI_INTPEND_OFFSET) & 
				MOBI_QCCSISC_PERI_INTPEND_MBOX_INT_MASK;


	if (perint_pend & QCC_PER_INTPEND_MBOX0_MASK) {
		dprintk("peri_intpend mbox 0 interrupt: 0x%x\n", perint_pend);
		mbox = 0;
	}
	else if (perint_pend & QCC_PER_INTPEND_MBOX1_MASK) {
		dprintk("peri_intpend mbox 1 interrupt: 0x%x\n", perint_pend);
		mbox = 1;
	}
	else if (perint_pend & QCC_PER_INTPEND_MBOX4_MASK) {
		dprintk("peri_intpend mbox 4 interrupt: 0x%x\n", perint_pend);
		mbox = 4;
	}
	else if (perint_pend & QCC_PER_INTPEND_MBOX5_MASK) {
		dprintk("peri_intpend mbox 5 interrupt: 0x%x\n", perint_pend);
		mbox = 5;
	}
	else {
		dprintk("peri_intpend is unknown 0x%x\n", perint_pend);
		mbox = -1;
	}

	/* for mbox 0 & 1, the ARM(or QMM) is CPU 0, so read and clear interrupts 
	*  for that CPU and CPU1 is Mips(external)
	*/
	if ((mbox >= 0) && ((perint_pend & QCC_PER_INTPEND_MBOX0_MASK) || 
			(perint_pend & QCC_PER_INTPEND_MBOX1_MASK))) {

		/* read CPU0Int reg for CPU 0 */
		mobi_qcc_read(QCC_BID_MBOX,
				MBOX_INT_CPU0_MASK|MBOX_CPU0_INT_REG,
				&data,
				4);
		if (data) {
			dprintk("Read CPU0's CPU0_INT(0x%x) = 0x%lx\n", 
					MBOX_INT_CPU0_MASK|MBOX_CPU0_INT_REG, data);
			dprintk("mbox%d read %ld, rdy %ld\n",
					mbox, (data>>(2*mbox)) & 0x1, ((data>>((2*mbox)+1)) & 0x1));

#if 0
			/* if even mbox rdy bit is set, read the data */
			if ((data>>((2*EVEN_TEST_MBOX)+1)) & 0x1) {
				mobi_qcc_read(QCC_BID_MBOX,
						(MBOX_DATA_CPU0_MASK|MBOX0_DATA_MASK),
						&data2,
						4);
				dprintk("read CPU0 data: 0x%x\n", (uint32_t)data2);

				/* read cpu1 intr reg from cpu0 */
				dprintk("Reading the CPU1_INT register\n");
				mobi_qcc_read(QCC_BID_MBOX,
						QCC_BID2_MBOX_CPU1_INT|MBOX_INT_CPU0_MASK,
						&data2,
						4);

			}
#endif
			/* write back to clear */
			mobi_qcc_write(QCC_BID_MBOX,
				MBOX_INT_CPU0_MASK|MBOX_CPU0_INT_REG,
				data,
				4);
			data = 0;
		}
	}
    /* for mbox 4 & 5, the ARM is CPU 1 and the mips(extenal host)
	*  is CPU0, so here we clear interrupt for CPU1
	*/
	if ((mbox >= 0) && ((perint_pend & QCC_PER_INTPEND_MBOX4_MASK) || 
			(perint_pend & QCC_PER_INTPEND_MBOX5_MASK))) {


		/* read CPU1Int reg for CPU 1 */
		mobi_qcc_read(QCC_BID_MBOX,
				MBOX_INT_CPU1_MASK|MBOX_CPU1_INT_REG,
				&data,
				4);
		if (data) {
			dprintk("Read CPU1's CPU1_INT(0x%x) = 0x%lx\n", 
					MBOX_INT_CPU1_MASK|MBOX_CPU1_INT_REG, data);
			dprintk("mbox%d read %ld, rdy %ld\n",
					mbox, (data>>(2*mbox)) & 0x1, ((data>>((2*mbox)+1)) & 0x1));

			mobi_qcc_write(QCC_BID_MBOX,
				MBOX_INT_CPU1_MASK|MBOX_CPU1_INT_REG,
				data,
				4);
			data = 0;
		}
	}
	/* clear perint_pending */
	mobi_qcc_writereg(perint_pend, MOBI_QCCSISC_PERI_INTPEND_OFFSET);

	return IRQ_HANDLED;
}

static void __exit mobi_mbox_exit(void)
{
	mobi_qcc_writereg(MOBI_QCCSISC_PERI_INTENCLR_MBOX_MASK,
			MOBI_QCCSISC_PERI_INTENCLR_OFFSET);

	free_irq(MOBI_IRQ_MBOX, NULL);
}

static int __init mobi_mbox_init(void)
{
	int ret = 0;

	dprintk("mobi_mbox module loaded\n");
	request_irq(MOBI_IRQ_MBOX, mobi_mbox_interrupt, 
		IRQF_DISABLED, "mobimbox", NULL);

	mobi_qcc_writereg(MOBI_QCCSISC_PERI_INTENSET_MBOX_MASK,
			MOBI_QCCSISC_PERI_INTENSET_OFFSET);

	return ret;
}

module_init(mobi_mbox_init);
module_exit(mobi_mbox_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeff Hane <jhane@mobilygen.com>");
MODULE_DESCRIPTION("Mobilygen Mailbox Driver");

