/*
 *  drivers/mmc/dwc_msh.c - DesignWare Core Mobile Storage Host driver
 *
 *  Copyright (C) 2007 Mobilygen Corp., All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/* Note about this driver:
 * This driver is a generic driver for the DWC IP from Synopsys. It includes
 * few non-standard calls to set clocks and/or reset but those must be under
 * 'ifdef's'. The user must register the DesignWare Core device
 * specifying where and how the core is configured. 
 */

#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/highmem.h>
#include <linux/delay.h>
#include <linux/mmc/host.h>
#include <linux/mmc/sd.h>
#include <linux/mmc/mmc.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <asm/uaccess.h>
#include <asm/scatterlist.h>
#include <asm/kmap_types.h>
#include <linux/amba/bus.h>
#include <linux/clk.h>
#include <linux/vmalloc.h>

#include <linux/dw_sdmmc_device.h>
#include <mach/mobi_dma.h>
#include <mach/mobi_hmux.h>
#include <mach/mobi_reset.h>

#include "dwc_msh.h"

#define DRIVER_NAME "DW-SDMMC "

#undef LOCAL_DW_SDMMC_DEBUG_ENABLE

static int __attribute__((unused)) loglevel = -1;
static int __attribute__((unused)) use_memlog = 0;

#if defined(LOCAL_DW_SDMMC_DEBUG_ENABLE) || defined(CONFIG_DWCMSH_DEBUG)
#define DW_SDMMC_DEBUG 1
#else
#define DW_SDMMC_DEBUG 0
#endif

#if DW_SDMMC_DEBUG
static int __attribute__((unused)) one_bit_enable = 0;
module_param(one_bit_enable, int, 0644);
MODULE_PARM_DESC(one_bit_enable, "Use 1 bit data instead of 4");

static int __attribute__((unused)) fmax_clock_rate = 0;
module_param(fmax_clock_rate, int, 0644);
MODULE_PARM_DESC(fmax_clock_rate, "Set max requested clock rate");

module_param(loglevel, int, 0644);
MODULE_PARM_DESC(loglevel, "Set module debug logging level(0-5)");

#define DBGPFX "SD:"

/* 
 * define to compile in support for memory logging, use modparam
 *  use_memlog to enable(which disables all dprintk* calls except dprintk)
 */
#define SD_MEM_LOG_ENABLE

#ifdef SD_MEM_LOG_ENABLE
module_param(use_memlog, int, 0644);
MODULE_PARM_DESC(use_memlog, "Send debug logs to memory instead of the console");

#define USE_CODEC_ADDR_FOR_MEMLOG
#ifdef USE_CODEC_ADDR_FOR_MEMLOG
/* offset 34M on second bridge */
#define MEMLOG_CODEC_OFFSET (0xc0000000 | 0x02200000)
#define MEMLOG_BUFFER_SIZE 33554432
#else
#define MEMLOG_BUFFER_SIZE 1048576
#endif
char *memlog_buf = NULL;
uint32_t memlog_buf_len;
#endif

#define dwcmsh_debug(n, fmt, args...)    \
	do {                        \
		if (loglevel >= n) {    \
			if (!use_memlog) { \
				printk(DBGPFX"%s| " fmt, __func__, ##args); \
			} \
			else { \
				if (memlog_buf != NULL) { \
					if (memlog_buf_len+2048 >= MEMLOG_BUFFER_SIZE) { \
						printk("\n****** memlog_buffer full, clearing! ******\n\n"); \
						memset(memlog_buf, 0x0, MEMLOG_BUFFER_SIZE); \
						memlog_buf_len = 0; \
					} \
					memlog_buf_len += \
					sprintf(memlog_buf+memlog_buf_len, \
							"%s: " fmt, __func__, ##args); \
				} \
			} \
		} \
	} while(0)

#else
#define dwcmsh_debug(n, fmt, args...)	do {} while(0)
#endif

/* just a way to mark extra, tmp printk */
#define dprintk(x...)	printk(x)
/* level 0 always prints when debug is enabled */
#define dprintk0(fmt, x...)	dwcmsh_debug(0, fmt, ##x)
#define dprintk1(fmt, x...)	dwcmsh_debug(1, fmt, ##x)
#define dprintk2(fmt, x...)	dwcmsh_debug(2, fmt, ##x)
#define dprintk3(fmt, x...)	dwcmsh_debug(3, fmt, ##x)
#define dprintk4(fmt, x...)	dwcmsh_debug(4, fmt, ##x)
#define dprintk5(fmt, x...)	dwcmsh_debug(5, fmt, ##x)

/* module params */
static int use_dma = -1;
static int disable_highspeed = 1;
static int dedicated_dma_channel = 0;
static int override_cclock_rate = 0;

static int dma_waitq_flag = 0;
static int dto_waitq_flag = 0;
static DECLARE_WAIT_QUEUE_HEAD(dma_waitq);
static DECLARE_WAIT_QUEUE_HEAD(dto_waitq);

static void set_interrupt_mask(dwc_msh_data_t *p_drv_data);
static void dwcmsh_enable_int(dwc_msh_data_t *p_drv_data);
static void dwcmsh_disable_int(dwc_msh_data_t *p_drv_data);
static void dwcmsh_write_block_pio(dwc_msh_data_t *p_drv_data);
static void dwcmsh_read_block_pio(dwc_msh_data_t *p_drv_data);
static void dwcmsh_transfer_pio(dwc_msh_data_t *p_drv_data);
static void dwcmsh_finish_data(dwc_msh_data_t *p_drv_data);

/**sysfs attributes
 * Those APIs allows user space programs to configure the driver.
 * Some of those parameters are RW which means they can be changed
 * after the driver has been loaded. Other are read-only and cannot be changed.
 */

/* hcon attribute is Read-Only and allows a user to read HCON register
 */
static ssize_t dwcmsh_hcon_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	return sprintf (buf, "0x%08x\n", 
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_HCON));
}
DEVICE_ATTR(hcon, 0444, dwcmsh_hcon_show, NULL);

/* regs attribute is Read-Only and allows a user to read all hardware registers
 */
static ssize_t dwcmsh_regs_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	int len=0;
	len+=sprintf (buf+len, "DesignWare Core Mobile Storage Host registers "
			"at 0x%p (phys=0x%08x):\n",pdev->reg_addr, pdev->phys_reg_addr);
	len+=sprintf (buf+len, "CTRL\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CTRL));
	len+=sprintf (buf+len, "PWREN\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_PWREN));
	len+=sprintf (buf+len, "CLKDIV\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKDIV));
	len+=sprintf (buf+len, "CLKSRC\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKSRC));
	len+=sprintf (buf+len, "CLKENA\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKENA));
	len+=sprintf (buf+len, "TMOUT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TMOUT));
	len+=sprintf (buf+len, "CTYPE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CTYPE));
	len+=sprintf (buf+len, "BLKSIZE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BLKSIZE));
	len+=sprintf (buf+len, "BYTCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BYTCNT));
	len+=sprintf (buf+len, "INTMASK\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_INTMASK));
	len+=sprintf (buf+len, "CMDARG\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CMDARG));
	len+=sprintf (buf+len, "CMD\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CMD));
	len+=sprintf (buf+len, "RESP0\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP0));
	len+=sprintf (buf+len, "RESP1\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP1));
	len+=sprintf (buf+len, "RESP2\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP2));
	len+=sprintf (buf+len, "RESP3\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP3));
	len+=sprintf (buf+len, "MINTSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_MINTSTS));
	len+=sprintf (buf+len, "RINTSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RINTSTS));
	len+=sprintf (buf+len, "STATUS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_STATUS));
	len+=sprintf (buf+len, "FIFOTH\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_FIFOTH));
	len+=sprintf (buf+len, "CDETECT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CDETECT));
	len+=sprintf (buf+len, "WRTPRT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_WRTPRT));
	len+=sprintf (buf+len, "GPIO\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_GPIO));
	len+=sprintf (buf+len, "TCBCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TCBCNT));
	len+=sprintf (buf+len, "TBBCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TBBCNT));
	len+=sprintf (buf+len, "DEBNCE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DEBNCE));
	len+=sprintf (buf+len, "USRID\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_USRID));
	len+=sprintf (buf+len, "VERID\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_VERID));
	len+=sprintf (buf+len, "HCON\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_HCON));
	len+=sprintf (buf+len, "BMOD\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BMOD));
	len+=sprintf (buf+len, "PLDMND\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_PLDMND));
	len+=sprintf (buf+len, "DBADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DBADDR));
	len+=sprintf (buf+len, "IDSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_IDSTS));
	len+=sprintf (buf+len, "IDINTEN\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_IDINTEN));
	len+=sprintf (buf+len, "DSCADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DSCADDR));
	len+=sprintf (buf+len, "BUFADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BUFADDR));
	len+=sprintf (buf+len, "\n");
	return len;
}
DEVICE_ATTR(regs, 0444, dwcmsh_regs_show, NULL);

/* info attribute is Read-Only and allows user to read the state of the device
 */
static ssize_t dwcmsh_info_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	int len=0;
	dwcmsh_reg_hcon_t cfg;
	uint32_t clkdiv_reg;

	cfg.reg=DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_HCON);
	len+=sprintf (buf+len, "DesignWare Core Mobile Storage Host driver:\n");
	len+=sprintf (buf+len, "\tCore ID: 0x%08x-0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_VERID),
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_USRID));
	len+=sprintf (buf+len, "\tRegisters address: 0x%08x remapped at 0x%p\n",
			pdev->phys_reg_addr, pdev->reg_addr);
	if(pdev->dma_enable)
		len+=sprintf (buf+len, "\tDMA is supported and enabled.\n");
	else if(cfg.fields.dma_interface) 
		len+=sprintf (buf+len, "\tDMA is supported but disabled.\n");
	else 
		len+=sprintf (buf+len, "\tDMA is not supported.\n");
	if(cfg.fields.card_type)
		len+=sprintf (buf+len, "\tCore supports SD and MMC devices.\n");
	else len+=sprintf (buf+len, "\tCore supports only MMC cards.");	
	len+=sprintf(buf+len, "\tCard clock input is running at %uMHz\n",
			pdev->cclock/1000000);
	clkdiv_reg = 
		DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKDIV) 
		& DWCMSH_CLKDIV_0;
	len+=sprintf(buf+len, "\tCard clock output is running at %uHz\n",
			(pdev->cclock/(clkdiv_reg ? (clkdiv_reg*2) : 1)));
	len+=sprintf(buf+len, "\n");
	return len;
}
DEVICE_ATTR(info, 0444, dwcmsh_info_show, NULL);

/* dma_enable attribute is Read-Write and tell the driver if it should use DMA
 */
static ssize_t dwcmsh_dma_enable_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);

	return sprintf (buf, "%d\n", pdev->dma_enable);
}

static ssize_t dwcmsh_dma_enable_store(struct device *dev, 
		struct device_attribute* attr, const char *buf, size_t count)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	uint32_t val = simple_strtoul(buf, NULL, 0);
	dwcmsh_reg_hcon_t cfg;

	cfg.reg=DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_HCON);
	if(val && cfg.fields.ge_dma_data_width)
		pdev->dma_enable=cfg.fields.ge_dma_data_width;
	else pdev->dma_enable=0;
	return count;
}
DEVICE_ATTR(dma_enable, 0644, dwcmsh_dma_enable_show, dwcmsh_dma_enable_store);

#if DW_SDMMC_DEBUG
static void dump_regs(dwc_msh_data_t *pdev)
{	
	printk("DesignWare Core Mobile Storage Host registers "
			"at 0x%p (phys=0x%08x):\n",pdev->reg_addr, pdev->phys_reg_addr);
	printk("CTRL\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CTRL));
	printk("PWREN\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_PWREN));
	printk("CLKDIV\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKDIV));
	printk("CLKSRC\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKSRC));
	printk("CLKENA\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CLKENA));
	printk("TMOUT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TMOUT));
	printk("CTYPE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CTYPE));
	printk("BLKSIZE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BLKSIZE));
	printk("BYTCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BYTCNT));
	printk("INTMASK\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_INTMASK));
	printk("CMDARG\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CMDARG));
	printk("CMD\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CMD));
	printk("RESP0\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP0));
	printk("RESP1\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP1));
	printk("RESP2\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP2));
	printk("RESP3\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RESP3));
	printk("MINTSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_MINTSTS));
	printk("RINTSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_RINTSTS));
	printk("STATUS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_STATUS));
	printk("FIFOTH\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_FIFOTH));
	printk("CDETECT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_CDETECT));
	printk("WRTPRT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_WRTPRT));
	printk("GPIO\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_GPIO));
	printk("TCBCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TCBCNT));
	printk("TBBCNT\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_TBBCNT));
	printk("DEBNCE\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DEBNCE));
	printk("USRID\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_USRID));
	printk("VERID\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_VERID));
	printk("HCON\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_HCON));
	printk("BMOD\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BMOD));
	printk("PLDMND\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_PLDMND));
	printk("DBADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DBADDR));
	printk("IDSTS\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_IDSTS));
	printk("IDINTEN\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_IDINTEN));
	printk("DSCADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_DSCADDR));
	printk("BUFADDR\t=0x%08x\n",
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BUFADDR));
	printk("\n");
}

/* debuglog attribute is Read-Write and enable debug 
 *  messages if debugging has been compiled in
 */
static ssize_t dwcmsh_debuglog_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	return sprintf(buf, "%d\n", loglevel);
}

static ssize_t dwcmsh_debuglog_store(struct device *dev, 
		struct device_attribute* attr, const char *buf, size_t count)
{	
	loglevel = simple_strtoul(buf, NULL, 0);
	printk(KERN_INFO "DWC_MSH debug level set to %d\n", loglevel);
	return count;
}
DEVICE_ATTR(debuglog, 0644, dwcmsh_debuglog_show, dwcmsh_debuglog_store);

static ssize_t dwcmsh_regrw_show(struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	uint32_t offset = 0;

	offset = simple_strtoul(buf, NULL, 0);
	printk("Read at offset 0x%x is = 0x%08x(%d)\n",
			offset,
			DWCMSH_READ_REG(pdev->reg_addr, offset),
			DWCMSH_READ_REG(pdev->reg_addr, offset));

	return 1;
}
/* support functions for the reading and writing to sysfs */
static void advance_ptr(char **ptr, uint32_t buffer, unsigned long count)
{
	char *buf = (char *)buffer;
	// while(isspace(**ptr) && (*ptr-buf<count)) { // will this work
	while(((**ptr==' ') || (**ptr=='\t') || (**ptr=='\r') || 
				(**ptr=='\n')) && (*ptr-buf<count)) {
		(*ptr)++;
	}
}

/* get and return the next argument as an unsigned long */
static int get_ularg(char **ptr, uint32_t buffer, 
		unsigned long count, uint32_t *arg)
{
	char *buf = (char *)buffer;
	advance_ptr(ptr, buffer, count);
	if (*ptr-buf >= count) {
		printk("Invalid argument string\n");
		return -1;
	}
	else {
		*arg = simple_strtoul(*ptr, ptr, 0);
		if(*arg < 0) {
			printk(KERN_ERR "invalid argument\n");
			return -1;
		}
	}
	return 0;
}

static ssize_t dwcmsh_regrw_store(struct device *dev, 
		struct device_attribute* attr, const char *buf, size_t count)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	uint32_t offset = 0, value;
	char *ptr = (char *)buf;

	get_ularg(&ptr, (uint32_t)buf, count, &offset);
	get_ularg(&ptr, (uint32_t)buf, count, &value);

	DWCMSH_WRITE_REG(pdev->reg_addr, offset, value);

	return count;
}
DEVICE_ATTR(regrw, 0644, dwcmsh_regrw_show, dwcmsh_regrw_store);


#ifdef SD_MEM_LOG_ENABLE
void dwcmsh_dump_memlogs(uint32_t offset)
{
	/* 
	 * printk can't print a really big buffer so we just chunk
	 *  through our log buffer
	 */
	uint32_t bytes_read = 0, total, start;
	char pbuf[257];

	if (offset == 0xdeadbeaf) {
		start = (uint32_t)memlog_buf;
		total = strlen((char*)start);
		if (total > 524288) {
			start = (uint32_t)memlog_buf + total - 524288;
			total = 524288;
		}
	}
	else {
		start = (uint32_t)memlog_buf+offset;
		total = strlen((char*)start);
	}
	printk("dumping: start 0x%x, size %u\n", start, total);

	if (total != 0) {
		while (bytes_read < total) {
			memset(pbuf, 0x0, 257);
			if (bytes_read+256 > total)
				strncpy(pbuf, (char*)(start+bytes_read), total-bytes_read);
			else
				strncpy(pbuf, (char*)(start+bytes_read), 256);
			bytes_read += 256;
			printk("%s", pbuf);
		}
		printk("\nmemory log dump complete\n");
	}
	else
		printk("memory log is empty\n");
}

#define DBG_CMD_MEMLOG_DISABLE 	0
#define DBG_CMD_MEMLOG_ENABLE 	1
#define DBG_CMD_MEMLOG_DUMP 	2
#define DBG_CMD_MEMLOG_CLEAR	3
#define DBG_CMD_MEMLOG_DUMP_OFFSET	4
#define DBG_CMD_MEMLOG_DUMP_END		5
static ssize_t dwcmsh_memlog_store(struct device *dev, 
		struct device_attribute* attr, const char *buf, size_t count)
{	
	uint32_t cmd;
	cmd = simple_strtoul(buf, NULL, 0);
	switch (cmd) {
		case DBG_CMD_MEMLOG_DISABLE:
			use_memlog = 0;
			printk("Memory logging disabled\n");
			break;
		case DBG_CMD_MEMLOG_ENABLE:
			use_memlog = 1;
			printk("Memory logging enabled\n");
			break;
		case DBG_CMD_MEMLOG_DUMP:
			dwcmsh_dump_memlogs(0);
			break;
		case DBG_CMD_MEMLOG_CLEAR:
			memlog_buf_len = 0;
			memset(memlog_buf, 0x0, MEMLOG_BUFFER_SIZE);
			break;
			/* need to see how to get second arg */
		case DBG_CMD_MEMLOG_DUMP_OFFSET:
			dwcmsh_dump_memlogs(0);
			break;
		case DBG_CMD_MEMLOG_DUMP_END:
			dwcmsh_dump_memlogs(0xdeadbeaf);
			break;
	}

	return count;
}
DEVICE_ATTR(memlog, 0644, NULL, dwcmsh_memlog_store);
#endif
#endif

/* bmod attribute is Read-Write. It returns and 
 *   set the values of the BMOD register
 */
static ssize_t dwcmsh_bmod_show (struct device *dev, 
		struct device_attribute* attr, char *buf)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);

	return sprintf (buf, "0x%08x\n", 
			DWCMSH_READ_REG(pdev->reg_addr, DWCMSH_REG_BMOD));
}

static ssize_t dwcmsh_bmod_store (struct device *dev, 
		struct device_attribute* attr, const char *buf, size_t count)
{	
	struct mmc_host *p_host = dev_get_drvdata(dev);
	dwc_msh_data_t *pdev = (dwc_msh_data_t *)mmc_priv(p_host);
	uint32_t bmod= simple_strtoul(buf, NULL, 0);

	DWCMSH_WRITE_REG(pdev->reg_addr, DWCMSH_REG_BMOD, bmod);
	return count;
}
DEVICE_ATTR(bmod, 0644, dwcmsh_bmod_show, dwcmsh_bmod_store);

/********* Core Functions************/

static inline void dwcmsh_init_sg(dwc_msh_data_t *p_drv_data, 
		struct mmc_data *data)
{
	p_drv_data->cur_sg = data->sg;
	p_drv_data->num_sg = data->sg_len;

	p_drv_data->offset = 0;
	p_drv_data->remain = p_drv_data->cur_sg->length;
}

static inline char* dwcmsh_kmap_sg(dwc_msh_data_t *p_drv_data)
{
	p_drv_data->mapped_sg = 
		kmap_atomic(sg_page(p_drv_data->cur_sg), KM_BIO_SRC_IRQ);
	return p_drv_data->mapped_sg + p_drv_data->cur_sg->offset;
}

static inline void dwcmsh_kunmap_sg(dwc_msh_data_t *p_drv_data)
{
	kunmap_atomic(p_drv_data->mapped_sg, KM_BIO_SRC_IRQ);
}

static inline int dwcmsh_next_sg(dwc_msh_data_t *p_drv_data)
{
	/*
	 * Skip to next SG entry.
	 */
	p_drv_data->cur_sg++;
	p_drv_data->num_sg--;

	/*
	 * Any entries left?
	 */
	if (p_drv_data->num_sg > 0) {
		p_drv_data->offset = 0;
		p_drv_data->remain = p_drv_data->cur_sg->length;
	}

	return p_drv_data->num_sg;
}

/* DMA handler */
void dwcmsh_dma_handler(mobi_dma_handle dmah, mobi_dma_event dma_event, 
		void *data)
{
	switch (dma_event) {
		case MOBI_DMA_EVENT_TRANSFER_COMPLETE:
			dprintk1("%lu: Transfer complete\n", jiffies);
			dprintk5("\tstatus 0x%x bytes xfer %d\n",
					mobi_dma_get_status(dmah), mobi_dma_get_bytes(dmah));
			break;
		case MOBI_DMA_EVENT_TRANSFER_ERROR:
			error("DMA Transfer Error");
			break;
	}
	dma_waitq_flag = dma_event;
	wake_up_interruptible(&dma_waitq);
}


/*Get the DMA channel */
static int dwcmsh_dma_channel_config(dwc_msh_data_t *p_drv_data,
		struct mmc_data *data)
{
	int ret = 0;
	uint32_t src_config_flags, dst_config_flags;
	//struct mg_dma_config_data cfg_data;

	if (p_drv_data->dma_handle == -1 || p_drv_data->dedicated_dma == 0) {

		p_drv_data->dma_handle = -1;
		if ((p_drv_data->dma_handle = 
					mobi_dma_request("sdmmc", MOBI_DMA_O_NONE)) < 0) {
			ret = p_drv_data->dma_handle;
			error(" %s DMA Channel get failed",
					(ret == -EINVAL ? "Invalid Argument":"No Free channel"));
			return ret;
		}

		if ((ret = mobi_dma_setup_handler(p_drv_data->dma_handle,
						(void *)dwcmsh_dma_handler, NULL)))
			goto err;

		src_config_flags = 
			MOBI_DMA_CONFIG_TRANSFER_WIDTH_32 | 
			MOBI_DMA_CONFIG_BURST_SIZE_8;

		dst_config_flags = src_config_flags;
	}

	/* if setting up everytime, if first entry or if current mode is not
	 * the same as the last, then we have to do the setup
	 */
	if (p_drv_data->dedicated_dma == 0 || 
			(p_drv_data->prev_dma_mode == DMA_MODE_NONE) || 
			((data->flags & MMC_DATA_READ) && 
			 (p_drv_data->prev_dma_mode == DMA_MODE_WRITE)) ||
			((data->flags & MMC_DATA_WRITE) && 
			 (p_drv_data->prev_dma_mode == DMA_MODE_READ))) {
		/* 
		 * XXX according to the docs, writing to a fixed address means 
		 * we are bursting....  But we don't have enough address space 
		 * for multiblock writes.  doc has other suggestion about 
		 * auto-reload that might work, need to investigate, see
		 * pg 51 of the databook
		 *
		 * hmm, after I did this things didn't seem to speed up though, 
		 * perhaps to much time list processing? but this was on an 
		 * unloaded system, bursting could be good on loaded system. to try
		 * src & dst need address increment and uncomment RELOAD flag and 
		 * set max_blk_size to 2049 in XFER
		 */
		if (data->flags & MMC_DATA_READ) {
			/* src_config_flags |= MOBI_DMA_CONFIG_RELOAD_START_ADDR; */
			src_config_flags |= MOBI_DMA_CONFIG_ADDRADJ_NONE;
            dst_config_flags |= MOBI_DMA_CONFIG_ADDRADJ_INC;
			p_drv_data->prev_dma_mode = DMA_MODE_READ;
		}
		else {
			/* dst_config_flags |= MOBI_DMA_CONFIG_RELOAD_START_ADDR; */
            src_config_flags |= MOBI_DMA_CONFIG_ADDRADJ_INC;
			dst_config_flags |= MOBI_DMA_CONFIG_ADDRADJ_NONE;
			p_drv_data->prev_dma_mode = DMA_MODE_WRITE;
		}

		if ((ret = mobi_dma_config(p_drv_data->dma_handle, DMA_CONFIG_SRC,
						src_config_flags, 
						NULL))) {
			error("Failed to configure DMA source");
			goto err;
		}
		if ((ret = mobi_dma_config(p_drv_data->dma_handle, DMA_CONFIG_DST,
						dst_config_flags, 
						NULL))) {
			error("Failed to configure DMA destination");
			goto err;
		}
		if ((ret = mobi_dma_config(p_drv_data->dma_handle, DMA_CONFIG_XFER,
						MOBI_DMA_CONFIG_DATA_WIDTH_8,
						NULL))) {
			error("Failed to configure DMA xfer");
			goto err;
		}
	}

	return ret;

err:
	if (p_drv_data->dma_handle >= 0) {
		mobi_dma_free(p_drv_data->dma_handle);
		p_drv_data->dma_handle = -1;
	}

	return ret;
}

static int dwcmsh_dma_data_config(dwc_msh_data_t *p_drv_data, 
		struct mmc_data *data)
{
	int ret = 0;
	int dma_bufs = 0;
	struct scatterlist *sg = data->sg;

	/* Reading from Device */
	if (data->flags & MMC_DATA_READ) {
		dprintk4("DMA read from device\n");
		p_drv_data->dma_mode = DMA_FROM_DEVICE; 
	}
	else {
		dprintk4("DMA write to device\n");
		p_drv_data->dma_mode = DMA_TO_DEVICE; 
	}
	dprintk3("sg: 0x%p, sg_len %d, size %d, blocks %d\n", 
			data->sg, data->sg_len, p_drv_data->size, data->blocks);

	dma_bufs = dma_map_sg(NULL, sg, data->sg_len, 
			data->flags & MMC_DATA_READ ? DMA_FROM_DEVICE : DMA_TO_DEVICE);

	/* debug
	for (dma_bufs=0;dma_bufs<data->sg_len;dma_bufs++, sg++) {
		dprintk1("buf %d: dma_address 0x%x, len %d\n", 
				dma_bufs, sg_dma_address(sg), sg_dma_len(sg));
	}
	*/

	/* TOTRY:
	 * 	we may not be bursting since we are only using a single
	 * 	src/dst address and no incr for the fifo.  so and address 
	 * 	>= 0x100 will go to the fifo.  unfortunately, our sd address 
	 * 	space is only 4K and we lose 0x100 to registers so we can't
	 * 	do nice 4k buffers. soo... lets try this.  lets say that we 
	 * 	have a max block size of 2K, which fits into our remaining 
	 * 	address space and goes into the max buffer of 32768 evenly.
	 * 	so create a mobi_dma_list here, parse each sg list entry,
	 * 	entries that are greater than 2K will be broken into multiple
	 * 	list entries that have the same sd src/dst address but we allow
	 * 	the address to be incremented.  this should allow us burst 
	 * 	data
	 * 	- one thought is add a way to set the max block size
	 * 	so the dma.c can create the sglist.  the problem there is if
	 * 	the address is supposed to incremented and the sglist entry is 
	 * 	greater than 2K the address will be wrong.  but I think this
	 * 	is the case of src/dst address reload.  don't have a way to 
	 * 	set this flag either.
	 */
	ret = mobi_dma_setup_sglist(p_drv_data->dma_handle, 
			data->sg, 
			dma_bufs,
			p_drv_data->size, 
			(uint32_t)p_drv_data->phys_reg_addr + DWCMSH_REG_DATA, 
			p_drv_data->dma_mode);

	if (ret < 0)
		error("DMA %s setup failed, %s",
				(data->flags & MMC_DATA_READ ? "Read" : "Write"),
				(ret == -EINVAL ? "Invalid Argument" : "No memory"));

	return ret;
}

static int dwcmsh_prepare_data(dwc_msh_data_t *p_drv_data, 
		struct mmc_data *data)
{
	int ret = 0;

	/* XXX data == NULL, why, when, is it an error? */
	if (data == NULL)
		return 0;

	/* Calculate size */
	p_drv_data->size = data->blksz * data->blocks;

	/* write BLKSIZE in bytes */
	dprintk2("BLKSIZE %d, BYTCNT %d\n", data->blksz, p_drv_data->size);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_BLKSIZE, data->blksz);

	/* write BYTCNT in bytes should be multiple of BLKSIZ*/
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
			DWCMSH_REG_BYTCNT, p_drv_data->size);

	if (p_drv_data->flags & DWCMSH_USE_DMA){
		if ((ret = dwcmsh_dma_channel_config(p_drv_data, data)) >= 0) {
			ret = dwcmsh_dma_data_config(p_drv_data, data);
		}
		else {
			error("Unable to setup DMA transfer");
		}
		/* make sure we set the interrupt mask to match
		 *  the transfer mode
		 */
		if (ret >= 0)
			set_interrupt_mask(p_drv_data);
	}
	else {
		dwcmsh_init_sg(p_drv_data, data);
	}
	return ret;
}

#define DMA_XFER_ABORT   	0x1
#define DMA_XFER_SHUTDOWN   0x2
static void dwcmsh_dma_cleanup(dwc_msh_data_t *p_drv_data, 
		struct mmc_data *data, uint8_t flags)
{
	/* what happens if unmap is called on a list that has not been mapped?? */
	if (data)
		dma_unmap_sg(NULL, data->sg, data->sg_len, 
				data->flags & MMC_DATA_READ ? DMA_FROM_DEVICE : DMA_TO_DEVICE);

	/* there could be a timeout for something that does not use dma and the 
	 *  timeout function calls this one so don't bother doing dma functions
	 *  if we don't have a valid handle
	 */
	if (p_drv_data->dedicated_dma == 0 && p_drv_data->dma_handle != -1) {
		dprintk4("%s flag set\n", 
				(flags & DMA_XFER_ABORT) ? "abort" : "shutdown");
		if (flags & DMA_XFER_ABORT)
			mobi_dma_abort(p_drv_data->dma_handle);
		else
			mobi_dma_disable(p_drv_data->dma_handle);

		mobi_dma_free(p_drv_data->dma_handle);
		p_drv_data->dma_handle = -1;
	}
}
/*
 * Description: Send command to card
 * Param: pointer to dwc_msh_data_t and mmc_command structure
 * Returns: Nothing
 * Remarks: 
 */
static int dwcmsh_send_command(dwc_msh_data_t *p_drv_data, 
		struct mmc_command *cmd)
{
	u32 cmdr;
	struct mmc_data *data = cmd->data;
	int ret = 0;
	int timer;

	p_drv_data->isr = 0;
	cmdr = cmd->opcode; 

	/*No Response is expected */
	if (mmc_resp_type(cmd) == MMC_RSP_NONE){
		cmdr &= ~DWCMSH_CMD_RSPE;
		/*Card Number- slot number of card */
		cmdr |= DWCMSH_CMD_CNUM & ~DWCMSH_CMD_CNUM; 
		cmdr |= DWCMSH_CMD_START_CMD; /*Start bit */
		cmdr &= ~DWCMSH_CMD_UPD_CLK; /*Clear Update clock bit */
		cmdr &= ~DWCMSH_CMD_RSPL;
		cmdr &= ~DWCMSH_CMD_DATAE;

		cmdr |= DWCMSH_CMD_RSPCRC;

		if (cmd->opcode == 0)
			/* CMD0 Send Initialization seuqence */
			cmdr |= DWCMSH_CMD_SND_INIT; 
		else
			cmdr &= ~DWCMSH_CMD_SND_INIT; 

		if (p_drv_data->flags  & DWCMSH_SENT_STOP) {
			cmdr |= DWCMSH_CMD_STOP_ABT; 
			cmdr &= ~DWCMSH_CMD_WAIT_PRV_DATA;
		}
		else
			cmdr &= ~DWCMSH_CMD_STOP_ABT; 

		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMDARG, cmd->arg);
	}
	else {
		/*Response is expected */
		cmdr |= DWCMSH_CMD_RSPE;
		cmdr &= ~DWCMSH_CMD_DATAE;
		cmdr |= DWCMSH_CMD_START_CMD; /*Start bit */
		cmdr &= ~DWCMSH_CMD_UPD_CLK; /*Clear Update clock bit */
		cmdr |= DWCMSH_CMD_CNUM & ~DWCMSH_CMD_CNUM; /*Card number */

		if (cmd->opcode != SD_APP_OP_COND)
			cmdr |= DWCMSH_CMD_RSPCRC;

		/*Set 1 for long(R2) response, else 0*/
		if (mmc_resp_type(cmd) == MMC_RSP_R2)
			cmdr |= DWCMSH_CMD_RSPL;
		else
			cmdr &= ~DWCMSH_CMD_RSPL;

		if (cmd->opcode == 0)
			/* CMD0 Send Initialization sequence */
			cmdr |= DWCMSH_CMD_SND_INIT;
		else
			cmdr &= ~DWCMSH_CMD_SND_INIT; 

		if (p_drv_data->flags  & DWCMSH_SENT_STOP) {
			cmdr |= DWCMSH_CMD_STOP_ABT; 
			cmdr &= ~DWCMSH_CMD_WAIT_PRV_DATA;
		}
		else
			cmdr &= ~DWCMSH_CMD_STOP_ABT; 

		/* high speed mode cannot be enabled */
		if (cmd->opcode == 6 && p_drv_data->disable_hs) {
			if (cmd->arg & 0x80000001) {
				dprintk2("OVERRIDE high-speed mode\n");
				cmd->arg &= ~(0x1);
			}
		}
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMDARG, cmd->arg);
	}

	if (data) {

		p_drv_data->data = cmd->data;
		ret = dwcmsh_prepare_data(p_drv_data, cmd->data);

		if (ret < 0 && (p_drv_data->flags & DWCMSH_USE_DMA)) {
			dwcmsh_dma_cleanup(p_drv_data, data, DMA_XFER_ABORT);
			p_drv_data->data->error = -EIO;
			/* doing this should get the err back up to the next level */
			tasklet_schedule(&p_drv_data->finish_tasklet);
			goto out;
		}

		/*Set data start and direction */
		if (data->flags & MMC_DATA_READ)
			cmdr &= ~(DWCMSH_CMD_RW);
		if (data->flags & MMC_DATA_WRITE)
			cmdr |= DWCMSH_CMD_RW;

		/*Single/Multiple Block data transfer mode */
		if (data->blocks > 1) {
			cmdr &= ~(DWCMSH_CMD_TNFM);
			if ((data->blksz * data->blocks) > 0)
				cmdr |=  DWCMSH_CMD_SND_AUTO_STOP;
		}

		/*Stream data transfer mode */
		if (data->flags & MMC_DATA_STREAM)
			cmdr |= DWCMSH_CMD_TNFM;

		dprintk4("Data Present: %s %s access\n", 
				(data->flags & MMC_DATA_STREAM) ? "Streaming" : 
				((data->blocks > 1)? "Multi block":"Single block"),
				(data->flags & MMC_DATA_READ) ? "Read" : "Write");

		/*Start cmd and Update clk bit */
		cmdr |= DWCMSH_CMD_START_CMD; /*Start bit */
		/*Clear Update clock bit */
		cmdr |= DWCMSH_CMD_UPD_CLK & ~(DWCMSH_CMD_UPD_CLK); 
		cmdr |= DWCMSH_CMD_CNUM & ~DWCMSH_CMD_CNUM; /*Card number */


		if (p_drv_data->flags & DWCMSH_SENT_STOP) {
			cmdr |= DWCMSH_CMD_STOP_ABT; 
			cmdr &= ~DWCMSH_CMD_WAIT_PRV_DATA;
		}
		else
			cmdr &= ~DWCMSH_CMD_STOP_ABT; 

		if (cmd->opcode == 0)
			/* CMD0 Send Initialization sequence */
			cmdr |= DWCMSH_CMD_SND_INIT; 
		else
			cmdr &= ~DWCMSH_CMD_SND_INIT; 

		/*Set 1 for long(R2) response, else 0*/
		if (mmc_resp_type(cmd) == MMC_RSP_R2)
			cmdr |= DWCMSH_CMD_RSPL;
		else
			cmdr &= ~DWCMSH_CMD_RSPL;

		cmdr |= DWCMSH_CMD_RSPE;
		cmdr |= DWCMSH_CMD_DATAE;

		if (cmd->opcode != SD_APP_OP_COND)
			cmdr |= DWCMSH_CMD_RSPCRC;

		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMDARG, cmd->arg);

		if (p_drv_data->flags & DWCMSH_USE_DMA) {
			dprintk1("%lu: Preparing to start %s DMA transfer\n", 
					jiffies,
					(data->flags & MMC_DATA_READ) ? "read" : "write");
			if ((ret = mobi_dma_enable(p_drv_data->dma_handle)) < 0) {
				p_drv_data->data->error = -EIO;
				dwcmsh_dma_cleanup(p_drv_data, data, DMA_XFER_ABORT);
				goto out;
			}
		}
	}

			//(mmc_resp_type(cmd) == MMC_RSP_NONE ? 0 : 1),
	dprintk1("%lu: SEND CMD%-2d cmdReg 0x%08X, cmdArg 0x%08X, %s data, %d bytes\n", 
			jiffies,
			cmd->opcode, cmdr, cmd->arg, 
			(data ? ((data->flags & MMC_DATA_READ) ? "read" : "write") : "w/o"),
			(data ? (data->blksz * data->blocks) : 0));


	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMD, cmdr);
	mod_timer(&p_drv_data->timer, jiffies + 10 * HZ);
	if (data) {
		if (p_drv_data->flags & DWCMSH_USE_DMA) {
			/* don't return until we get error or xfer complete */
			wait_event_interruptible(dma_waitq, dma_waitq_flag != 0);
			if (dma_waitq_flag == MOBI_DMA_EVENT_TRANSFER_ERROR) {
				if (p_drv_data->data)
					p_drv_data->data->error = -EIO;
				dwcmsh_dma_cleanup(p_drv_data, data, DMA_XFER_ABORT);
			}
			else {
				dprintk1("dma transferred %d bytes\n", 
						mobi_dma_get_bytes(p_drv_data->dma_handle));
				if (p_drv_data->data)
					p_drv_data->data->error = 0;
				dwcmsh_dma_cleanup(p_drv_data, data, DMA_XFER_SHUTDOWN);
			}
			/* 
			 * this is a bit screwy right now. most of the time we get the DTO but
			 * during init we dont' for one of the commands, so we wait and no
			 * matter what do finish_data
			 */
			if (dma_waitq_flag == MOBI_DMA_EVENT_TRANSFER_COMPLETE) {
				dprintk3("wait for DTO\n");
				timer = wait_event_interruptible_timeout(dto_waitq, dto_waitq_flag != 0, 10);
				dprintk3("timer is %d, dto_waitq_flag = %d\n", timer, dto_waitq_flag);
				if (dto_waitq_flag != 0 || timer > 0) {
					dto_waitq_flag = 0;
				}
				else {
					dprintk1("DTO event timeout\n");
				}
				dprintk3("calling finish_data\n");
				dwcmsh_finish_data(p_drv_data); 
			}
			dma_waitq_flag = 0;
		}
		else {
			/* 
			 * we push PIO write data here, we will grab PIO read
			 *  data when get the RXDR intr in the isr routine
			 *
			 *  so in theory, we are supposed to prime the fifo before
			 *  sending the cmd for a write.  removed old command write above
			 *  and moved to after fifo start but this seems to hang.  sending
			 *  the cmd first was causing an interrupt that was requesting data
			 *  which then was writen so by the time we got to this code the
			 *  transfer had already started.  
			 *  don't know why write then cmd is hanging but things seem to
			 *  work if we just send cmd, wait for intr and then write data
			 *  ?????
			 */
			if (data->flags & MMC_DATA_WRITE) {
				/*dwcmsh_transfer_pio(p_drv_data);*/
				DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
						DWCMSH_REG_RINTSTS, DWCMSH_RINTSTS_TXDR);
			}
		}
	}

out:
	return ret;
}

/* Description: Set DWCMSH IP parameter
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks: Set DWCMSH IP parameters which need to configured one time.
 */
static inline void dwcmsh_set_IP_param(dwc_msh_data_t *p_drv_data)
{

	uint32_t fifo_depth, rx_wm, tx_wm, fifoth_reg =0;

	DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
			DWCMSH_REG_DEBNCE, DWCMSH_DEBNCE_DCNT);

	/* Calculate FIFO_DEPTH */
	fifo_depth = ((DWCMSH_READ_REG(p_drv_data->reg_addr, 
					DWCMSH_REG_FIFOTH) & 0x0fff0000) >> 16) + 1;

	// During repeatedly module load/unload DWC should be 
	// reset so correct fifo_depth is read or keep rx and 
	// tx watermark fixed

	/* burst size of 8 */
	fifoth_reg |= 2 << 28;

	/* Calculate RX watermark */
	//rx_wm = (fifo_depth/2)-1;
	rx_wm = 7;

	/* Calculate TX watermark */
	//tx_wm = (fifo_depth/2);
	tx_wm = 8;

	fifoth_reg |= rx_wm << 16;
	fifoth_reg |= tx_wm;
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_FIFOTH, fifoth_reg );

	dprintk3("fifo_depth  0x%08X, tx_wm 0x%08X rx_wm 0x%08X tmout 0x%08X\n", 
			fifo_depth, tx_wm, rx_wm, 
			DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_TMOUT)); 
}

/* Description: Check any CMD in progress,Wait for CIU to accept command 
 * Param: pointer to dwc_msh_data_t structure
 * Returns: <0 Failure, 0 SUCCESS
 * Remarks: 
 */
static inline int dwcmsh_cmd_busy(dwc_msh_data_t *p_drv_data)
{	
	unsigned long timeout = CMD_TIMEOUT_MS;
	/*Wait until start_cmd is cleared-CIU accepts command */
	while (DWCMSH_READ_REG(p_drv_data->reg_addr,
				DWCMSH_REG_CMD) & DWCMSH_CMD_START_CMD) {	
		if (timeout == 0)
			return -ETIMEDOUT;
		timeout--;
		mdelay(1);
	}
	return 0;
}

/*
 * Description:Process the next step in the request
 * Param: Pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_process_next(dwc_msh_data_t *p_drv_data)
{
	if (p_drv_data->mrq->cmd) {
		p_drv_data->flags |= DWCMSH_SENT_CMD;
		dwcmsh_send_command(p_drv_data, p_drv_data->mrq->cmd);
	} 
	else if (p_drv_data->mrq->stop || 
			(p_drv_data->mrq->cmd->opcode == MMC_STOP_TRANSMISSION)) {
		p_drv_data->flags |= DWCMSH_SENT_STOP;
		dwcmsh_send_command(p_drv_data, p_drv_data->mrq->stop);
	}
	else 
		dprintk1("Invalid request\n");
}

/*
 * Description: Handle an MMC request
 * Param: pointer to mmc_host and mmc_request structure
 * Returns: Nothing
 * Remarks: This is an callback function register with MMC layer
 */
static void dwc_msh_request(struct mmc_host *mmc, struct mmc_request *mrq)
{
	dwc_msh_data_t *p_drv_data=(dwc_msh_data_t *)mmc_priv(mmc);
	p_drv_data->mrq = mrq;
	p_drv_data->cmd = mrq->cmd;

	/*Check if card is present */
	if (p_drv_data->card_detect & DWCMSH_CDETECT_CD) {
		p_drv_data->cmd->error = -ETIMEDOUT;
		tasklet_schedule(&p_drv_data->finish_tasklet);
	}
	/* Check any cmd in progress */
	else if (dwcmsh_cmd_busy(p_drv_data) < 0) {
		error("dwcmsh_process_next: Command Busy");
		mmc_request_done(p_drv_data->mmc, p_drv_data->mrq);
		return;
	}
	else 
		dwcmsh_process_next(p_drv_data);
}

/*
 * Description: Reset DWCMSH controller 
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_reset(dwc_msh_data_t *p_drv_data)
{
	/* Wait max 100 ms */
	unsigned long timeout = 100;

	DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
			DWCMSH_REG_CTRL, DWCMSH_CTRL_CTRLRST);
	/* hw clears the bit when it's done */
	while (DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CTRL) & 
			DWCMSH_CTRL_CTRLRST) {

		if (timeout == 0) {
			error("%s: Reset never completed.", 
					mmc_hostname(p_drv_data->mmc));
			return;
		}
		timeout--;
		mdelay(1);
	}

	if (p_drv_data->flags & DWCMSH_USE_DMA) {
		/* After reset, this get cleared so Enable again */
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
				DWCMSH_REG_CTRL, DWCMSH_CTRL_DMAE);
	}
	dwcmsh_enable_int(p_drv_data);
}

static void set_interrupt_mask(dwc_msh_data_t *p_drv_data)
{
	uint32_t intmask;

	/*Clears pending Interrupts in Raw Interrupt register */
	//DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_RINTSTS, 0xffff);

	/*Unmasked interrupts we are interested */ 
	intmask = DWCMSH_RINTSTS_CD | DWCMSH_RINTSTS_RE |  
		DWCMSH_RINTSTS_CDONE | DWCMSH_RINTSTS_DTO  |
		DWCMSH_RINTSTS_RCRC  | DWCMSH_RINTSTS_DCRC |
		DWCMSH_RINTSTS_RTO   | DWCMSH_RINTSTS_DRTO | 
		DWCMSH_RINTSTS_HTO   | DWCMSH_RINTSTS_FRUN |
		DWCMSH_RINTSTS_HLE   | DWCMSH_RINTSTS_SBE  | 
		DWCMSH_RINTSTS_EBE; // |DWCMSH_RINTSTS_ACD;

	if(!(p_drv_data->flags & DWCMSH_USE_DMA))
		intmask |=  DWCMSH_RINTSTS_TXDR | DWCMSH_RINTSTS_RXDR;

	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_INTMASK, intmask);
}

/*
 * Description: Enable Interrupts on DWCMSH controller
 * Param: Pointer to dwc_msh_data_t structure
 * Returns: Nothing 
 * Remarks: 
 */
static void dwcmsh_enable_int(dwc_msh_data_t *p_drv_data)
{
	uint32_t reg;

	set_interrupt_mask(p_drv_data);
	/*Enable global interrupts */
	reg = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CTRL);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
			DWCMSH_REG_CTRL, reg | DWCMSH_CTRL_IE);
}

/*
 * Description: Disable Interrupts on DWCMSH controller
 * Param: Pointer to dwc_msh_data_t structure
 * Returns: Nothing 
 * Remarks: 
 */
static void dwcmsh_disable_int(dwc_msh_data_t *p_drv_data)
{	
	uint32_t reg;
	reg=DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CTRL);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CTRL, 
			reg & ~DWCMSH_CTRL_IE);
}

/*
 * Description: Calculate clock divider value 
 * Param: pointer to dwc_msh_data_t structure and unsigned int
 * Returns: Nothing
 * Remarks:
 */
static int dwcmsh_set_clock(dwc_msh_data_t *p_drv_data, unsigned int clock)
{
	int clkdiv = 0;
	u32 cmdr;
	unsigned long timeout;
	u32	clkdiv_reg;

	/* Expect the card not to be busy */
	if (DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_STATUS) & 
			DWCMSH_STAT_DATA_BUSY) {	
		error("SD/MMC: Card is busy, cannot program the clock.");
		return -EBUSY;
	};
	if (clock != 0) {	
		/* Calculate clock divider */
		if ((p_drv_data->cclock % (clock * 2)) == 0)
			clkdiv = ((p_drv_data->cclock / clock) / 2) - 1;
		else 
			clkdiv = (p_drv_data->cclock / clock) / 2;

		/* 
		 * we don't want to go over the requested rate so make
		 * the divider bigger
		 */
		if (clkdiv) {
			while ((p_drv_data->cclock/(2*clkdiv)) > clock) 
				clkdiv++;
		}
		dprintk1("requested clock %u, cclock_in %u "
				"clkdiv = %d cclock_out = %u\n", 
				clock, p_drv_data->cclock, clkdiv, 
				p_drv_data->cclock / (clkdiv ? 2*clkdiv : 1));
	}
	else 
		dprintk1("clock is being disabled!\n");

	/* Program new clock
	 * 1. Check card is busy or not.
	 * 2. Disable all clocks.Set start_cmd, update_clk_reg_only and 
	 * 	 wait_prv_data bit in CMD register
	 * 3. Wait for CMD acceptance by CIU.
	 * 4. Programme CLKDIV and CLKSRC register.Set start_cmd,
	 * 	 update_clk_reg_only,wait_prv_data bit in CMD register 
	 * 5. Wait for CMD acceptance by CIU.
	 * 6. Enable clocks.Set start_cmd, update_clk_reg_only and 
	 *  	 wait_prv_data bit in CMD register
	 * 7. Wait for CMD acceptance by CIU.
	 */
	/*Set start_cmd, update_clk_regs_only and wait_prv_data bit 
	 * and Write to command register */
	cmdr = DWCMSH_CMD_START_CMD | 
		DWCMSH_CMD_UPD_CLK | DWCMSH_CMD_WAIT_PRV_DATA;
	timeout = CMD_TIMEOUT_MS;
	/* Disable Clock */
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CLKENA, 0x00000000);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMD, cmdr);
	if (dwcmsh_cmd_busy(p_drv_data) < 0) {	
		error("Failed to disable the clock");
		return -ETIMEDOUT;
	}
	/* If clock is 0, we exit here */
	if(clock == 0)
		return 0;	
	/* Programm CLKDIV and CLKSRC */
	clkdiv_reg = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CLKDIV);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
			DWCMSH_REG_CLKDIV,(clkdiv_reg & ~(DWCMSH_CLKDIV_0)) |clkdiv);
	/*Clock divider source - clock_div_0 */
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CLKSRC, 0x0);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMD, cmdr);
	if (dwcmsh_cmd_busy(p_drv_data) < 0) {	
		error("Clock update to CLKDIV or CLKSRC NOT completed");
		return -ETIMEDOUT;
	}
	/* Enable Clock */
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CLKENA,DWCMSH_CLKENA_EN);
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CMD, cmdr);
	if (dwcmsh_cmd_busy(p_drv_data) < 0) {	
		error("Clock Enabling NOT completed");
		return -ETIMEDOUT;
	};
	return 0;
}

/*
 * Description: Enable/Disable power to the Card.
 * Param: pointer to dwc_msh_data_t structure and int flag.
 * flag=0 power OFF card.
 * flag=1 power ON card.
 * Returns: Nothing
 * Remarks:
 */
static inline void dwcmsh_power_card(dwc_msh_data_t *p_drv_data, int flag)
{
	u32 pwr;

	/* Power off the card */
	if (flag == 0){
		/*Power off the card */
		dprintk3("powering off\n");
		pwr = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_PWREN);
		pwr &= ~DWCMSH_PWREN_EN;
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_PWREN, pwr);
	} 
	else {
		/* Power on the card */
		pwr = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_PWREN);
		if (pwr & DWCMSH_PWREN_EN) {
			/*Already Power ON to card, Do nothing */
		} 
		else {
			/*Power up the card */
			dprintk3("powering on\n");
			pwr |= DWCMSH_PWREN_EN;
			DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_PWREN, pwr);
			mdelay(10);
		}
	}
}

/*
 * Description: MMC callback function
 * Param: pointer to mmc_host and mmc_ios structure
 * Returns: Nothing
 * Remarks: This is used by MMC layer to change physical 
 * parameters of controller
 */
static void dwc_msh_set_ios(struct mmc_host *mmc, struct mmc_ios *ios)
{
	u32 ctype;

	dwc_msh_data_t *p_drv_data=(dwc_msh_data_t *)mmc_priv(mmc);

	if (!p_drv_data)
		dprintk4("p_drv_data is NULL\n");

	p_drv_data->bus_mode = ios->bus_mode;

	/*Check if any cmd in progress. we are not suppose 
	 * to write certain register during busy*/
	if (dwcmsh_cmd_busy(p_drv_data) < 0) {
		error("dwc_msh_set_ios: Command Busy");
		return;
	}

	/*Card Type */
	ctype = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CTYPE);
	if (ios->bus_width == MMC_BUS_WIDTH_4) {
		dprintk4("Setting controller bus width to 4\n");
		ctype |= DWCMSH_CTYPE_4BIT;
		ctype &= ~DWCMSH_CTYPE_8BIT;
	}
	else {
		dprintk4("Setting controller bus width to 1\n");
		ctype &= ~DWCMSH_CTYPE_4BIT;
		ctype &= ~DWCMSH_CTYPE_8BIT;
	}
	DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_CTYPE, ctype);

	/* Power */
	if (ios->power_mode == MMC_POWER_OFF) {
		dwcmsh_power_card(p_drv_data, 0);
	}
	else {
		/* Check presence of card */
		if (!(p_drv_data->card_detect & DWCMSH_CDETECT_CD)) {
			dwcmsh_power_card(p_drv_data, 1);
		}
	}
	/* Change Clock */
	dwcmsh_set_clock(p_drv_data, ios->clock);
}

/*
 * Description: MMC callback function 
 * Param: pointer to mmc_host structure
 * Returns: int 
 * 1= Read only  
 * 0= Read/write
 * Remarks: This is used specially for SD cards to read swith on card. 
 */
int dwc_msh_get_ro(struct mmc_host *mmc)
{
	int read_only =0;
	dwc_msh_data_t *p_drv_data=(dwc_msh_data_t *)mmc_priv(mmc);

	if (DWCMSH_READ_REG(p_drv_data->reg_addr,
				DWCMSH_REG_WRTPRT) & DWCMSH_WRTPRT_WP ){
		read_only = 1;
		warning("%s: Card is Read-only\n", mmc_hostname(mmc));
	} 
	else {
		warning("%s: Card is Read-Write  \n", mmc_hostname(mmc));
	}
	return read_only;
}

/*
 * Description: MMC callback function
 * Param: pointer to mmc_host structure
 * Returns: int
 * 1= card present
 * 0= card absent
 * Remarks: This is function used to find out SD card present or not
 */
int dwc_msh_get_cd(struct mmc_host *mmc)
{
	uint32_t cdetect;
	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)mmc_priv(mmc);
	cdetect = DWCMSH_READ_REG(p_drv_data->reg_addr,
			DWCMSH_REG_CDETECT) && DWCMSH_CDETECT_CD ;
	return (cdetect == 1) ? 0 : 1;
}

static struct mmc_host_ops dwc_msh_ops = {
	.request	= dwc_msh_request,
	.set_ios	= dwc_msh_set_ios,
	.get_ro		= dwc_msh_get_ro,
	.get_cd		= dwc_msh_get_cd,
};

/* 
 * Description: Finish Data processing
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_finish_data(dwc_msh_data_t *p_drv_data)
{	
	struct mmc_data *data = NULL;

	if (p_drv_data->data) {
		data = p_drv_data->data;
		p_drv_data->data = NULL;
		data->bytes_xfered = data->blksz * data->blocks;
	}

	/* if running in PIO mode */
	if(!(p_drv_data->flags & DWCMSH_USE_DMA)) {
		if (p_drv_data->size != 0) {
			error("%s: %d bytes were left untransferred.",
					mmc_hostname(p_drv_data->mmc), p_drv_data->size);
			if (data)
				data->error = -EIO;
		}
	}
	dprintk3("schedule tasklet_finish\n");
	tasklet_schedule(&p_drv_data->finish_tasklet);
}

/*
 * Description: Finish Command processing
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing.
 * Remarks: Copy command response and indicate back to MMC layer 
 */
static void dwcmsh_finish_command(dwc_msh_data_t *p_drv_data)
{	
	if (p_drv_data->cmd){
		if (p_drv_data->cmd->flags & MMC_RSP_PRESENT) {
			if (p_drv_data->cmd->flags & MMC_RSP_136) {
				/* read long response */
				p_drv_data->cmd->resp[0] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP3);
				p_drv_data->cmd->resp[1] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP2);
				p_drv_data->cmd->resp[2] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP1);
				p_drv_data->cmd->resp[3] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP0);
				dprintk4("long resp[3:0] 0x%08x 0x%08x 0x%08x 0x%08x\n",
						p_drv_data->cmd->resp[3],
						p_drv_data->cmd->resp[2], 
						p_drv_data->cmd->resp[1], 
						p_drv_data->cmd->resp[0]);
			} 
			else {
				p_drv_data->cmd->resp[0] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP0);
				p_drv_data->cmd->resp[1] = 
					DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_RESP1);
				dprintk4("short resp[1:0] 0x%08x 0x%08x\n",
						p_drv_data->cmd->resp[1], p_drv_data->cmd->resp[0]);
			}
		}
		p_drv_data->cmd->error = 0;
		if (p_drv_data->cmd->data) {
			dprintk3("set data\n");
			p_drv_data->data = p_drv_data->cmd->data;
		}
		else {
			dprintk3("schedule tasklet_finish\n");
			tasklet_schedule(&p_drv_data->finish_tasklet);
		}
	}
}

/****************** TASKLET *********************/

/*
 * Description: Tasklet for Card insert/Remove event
 * Param: unsigned long
 * Returns: Nothing
 * Remarks: Acquire DMA channel when card is present. 
 */
static void dwcmsh_tasklet_card(unsigned long param)
{	
	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)param;
	unsigned long flags;
	int delay = -1;
	uint32_t cdetect_reg;

	spin_lock_irqsave(&p_drv_data->lock, flags);
	cdetect_reg = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_CDETECT);
	if (cdetect_reg ^ p_drv_data->card_detect) {	
		dprintk2("inside cdetect_reg 0x%x, card_detect 0x%x\n", 
				cdetect_reg, p_drv_data->card_detect);
		p_drv_data->card_detect = cdetect_reg;
		/*Card Removed */
		if (cdetect_reg & DWCMSH_CDETECT_CD) {	
			if (p_drv_data->mrq) {	
				error("%s: Card removed during transfer!", 
						mmc_hostname(p_drv_data->mmc));
				DWCMSH_WRITE_REG(p_drv_data->reg_addr,
				DWCMSH_REG_CTRL, DWCMSH_READ_REG(
				p_drv_data->reg_addr, DWCMSH_REG_CTRL) |
				DWCMSH_CTRL_CTRLRST | DWCMSH_CTRL_FIFORST);
				p_drv_data->mrq->cmd->error = -EIO;
				tasklet_schedule(&p_drv_data->finish_tasklet);
			}
			delay = 0;
			/*Power off card */
			dwcmsh_power_card(p_drv_data, 0);
			dprintk2("SD/MMC card removed\n"); 
		}
		else {
			/*Card Inserted */
			dprintk2("SD/MMC card inserted\n"); 
			delay = 10;
			/*Power on card */ 
			dwcmsh_power_card(p_drv_data, 1);
		}
	} 
	spin_unlock_irqrestore(&p_drv_data->lock, flags);
	if (delay != -1)
		mmc_detect_change(p_drv_data->mmc, msecs_to_jiffies(delay));
}

/*
 * Description: Tasklet for request finish
 * Param: unsigned long
 * Returns: Nothing
 * Remarks: Request done.
 */
static void dwcmsh_tasklet_finish(unsigned long param)
{
	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)param;
	unsigned long flags;
	struct mmc_request *mrq;

	spin_lock_irqsave(&p_drv_data->lock, flags);

	del_timer(&p_drv_data->timer);
	mrq = p_drv_data->mrq;
	if (!mrq) { 
		dprintk1("mrq is null, return now!\n");
		spin_unlock_irqrestore(&p_drv_data->lock, flags);
		return;
	}

	/*Upon error conditions, controller need to be reset */
	if ((mrq->cmd->error) || 
			(mrq->data && ((mrq->data->error) || 
						   (mrq->data->stop && (mrq->data->stop->error))))) {
		//TBD do we need to reset controller
		dprintk1("Finishing CMD%d, some kind of error\n\n", mrq->cmd->opcode);
		/*
		   dprintk1("Finishing CMD%d, cmd err %d, data err %d, stop err %d\n\n", 
		   mrq->cmd->opcode, 
		   mrq->cmd->error,
		   (mrq->data ? mrq->data->error : 0),
		   (mrq->data->stop ? mrq->data->stop->error : 0));
		 */
	}
	else 
		dprintk1("%lu: Finishing CMD%d, no errors\n\n", 
				jiffies, mrq->cmd->opcode);

	p_drv_data->mrq = NULL;
	p_drv_data->cmd = NULL;
	p_drv_data->data = NULL;
	p_drv_data->flags &= ~DWCMSH_SENT_CMD;
	p_drv_data->flags &= ~DWCMSH_SENT_STOP;

	spin_unlock_irqrestore(&p_drv_data->lock, flags);
	mmc_request_done(p_drv_data->mmc, mrq);
}

/*
 * Description: PIO read
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing.
 * Remarks:
 */
static void dwcmsh_read_block_pio(dwc_msh_data_t *p_drv_data)
{
	int blksize, chunk_remain;
	u32 data;
	char *buffer;
	int  size;

	blksize = p_drv_data->data->blksz;
	chunk_remain = 0;
	data = 0;

	buffer = dwcmsh_kmap_sg(p_drv_data) + p_drv_data->offset;
	dprintk5("blksize %d, size %d, offset 0x%x, buffer 0x%p\n", 
			blksize, p_drv_data->size, p_drv_data->offset, buffer);

	while (blksize) {
		if (chunk_remain == 0) {
			data = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_DATA);
			chunk_remain = min(blksize, 4);
		}
		size = min(p_drv_data->size, p_drv_data->remain);
		size = min(size, chunk_remain);
		chunk_remain -= size;
		blksize -= size;
		p_drv_data->offset += size;
		p_drv_data->remain -= size;
		p_drv_data->size -= size;

		while (size) {
			*buffer = data & 0xFF;
			buffer++;
			data >>= 8;
			size--;
		}

		if (p_drv_data->remain == 0) {
			dwcmsh_kunmap_sg(p_drv_data);
			if (dwcmsh_next_sg(p_drv_data) == 0) {
				return;
			}
			buffer = dwcmsh_kmap_sg(p_drv_data);
		}

		//ATTENTION!!!
		while (DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_STATUS) & 
				DWCMSH_STAT_FIFO_EMTY); 
	}
	dwcmsh_kunmap_sg(p_drv_data);
}

/*
 * Description: PIO write
 * Param: Pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_write_block_pio(dwc_msh_data_t *p_drv_data)
{
	u32 data=0, stat, filled_loc;
	char *buffer;
	int fifo_count=0, cnt;

	buffer = dwcmsh_kmap_sg(p_drv_data) + p_drv_data->offset;
	dprintk5("size %d, offset 0x%x, buffer 0x%p\n", 
			p_drv_data->size, p_drv_data->offset, buffer);

	while (p_drv_data->size) {

		stat = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_STATUS);
		filled_loc = stat;
		filled_loc <<=2;
		filled_loc >>=19;

		if ((stat & DWCMSH_STAT_FIFOTX_WM))
			fifo_count = 8; //Push fifo_depth - tx_wm data into FIFO

		if (!(stat & DWCMSH_STAT_FIFO_FULL) || !(stat & DWCMSH_STAT_FIFO_EMTY)){
			fifo_count = 16 - filled_loc; //Calculate empty location in FIFO
		}

		if (stat & DWCMSH_STAT_FIFO_FULL){
			fifo_count = 0; //FULL 
		}

		for(cnt=0; cnt < fifo_count; cnt++) { 

			p_drv_data->offset += 4;
			p_drv_data->remain -= 4;
			p_drv_data->size -= 4;

			data = *(u32*)buffer;
			buffer +=4;
			DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_DATA,data);

			if (p_drv_data->remain == 0) {
				dwcmsh_kunmap_sg(p_drv_data);
				if (dwcmsh_next_sg(p_drv_data) == 0){ 
					return;
				}
				buffer = dwcmsh_kmap_sg(p_drv_data);
			}
		} //for loop end 
	} //while loop end

	dwcmsh_kunmap_sg(p_drv_data);
}

/*
 * Description: PIO transfer 
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_transfer_pio(dwc_msh_data_t *p_drv_data)
{
	if (!p_drv_data->data) {
		dprintk3("data is NULL\n");
		return;
	}
	if (p_drv_data->size == 0) {
		dprintk3("data->size == 0\n");
		return;
	}
	dprintk3("%s %d bytes\n", 
			((p_drv_data->data->flags & MMC_DATA_WRITE) ? "write" : "read"),
			p_drv_data->size);

	while (p_drv_data->size > 0 && p_drv_data->data != NULL) {
		mod_timer(&p_drv_data->timer, jiffies + 30 * HZ);
		/* Check the presence of card before read/write */
		if (dwc_msh_get_cd(p_drv_data->mmc) == 0)
			return ;
		if (p_drv_data->data->flags & MMC_DATA_WRITE) 
			dwcmsh_write_block_pio(p_drv_data); 
		else
			dwcmsh_read_block_pio(p_drv_data); 

	}
	dprintk3("done....\n");
}

/*
 * Description: Handle Data irq events
 * Param: pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_data_irq(dwc_msh_data_t *p_drv_data)
{
	uint32_t intmask = (p_drv_data->isr & DWCMSH_INT_DATA_MASK);
	p_drv_data->isr &= ~(intmask);

	dprintk2("intmask: 0x%08x\n", intmask);
	if (!p_drv_data->data) {

		dprintk0("Got data interrupt even though no "
				"data operation was in progress: 0x%08x\n",
				intmask);

		/*Data end interrupt */
		if (intmask & DWCMSH_RINTSTS_DTO) 
			dprintk0("Got DTO interrupt, continuing\n");

		if (intmask & DWCMSH_RINTSTS_HTO) 
			dprintk0("Got HTO interrupt, returning\n");

		dprintk1("Status reg 0x%08x\n",
				(DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_STATUS)));

		return;
	}

	if (intmask & DWCMSH_RINTSTS_DRTO) {
		p_drv_data->data->error = -ETIMEDOUT;
		dprintk1("Data read timeout error(DRTO)\n");
	}
	else if (intmask & DWCMSH_RINTSTS_DCRC) {
		p_drv_data->data->error = -EILSEQ;
		dprintk1("Data CRC error(DCRC)\n");
	}
	else if (intmask & (DWCMSH_RINTSTS_EBE | DWCMSH_RINTSTS_SBE)) {
		p_drv_data->data->error = -EIO;
		dprintk1("Start/End bit error(EBE/SBE)\n");
	}
	else if (intmask & DWCMSH_RINTSTS_FRUN) {
		p_drv_data->data->error = -EILSEQ;
		dprintk1("FIFO over/under error(FRUN)\n");
	}

	if (p_drv_data->data->error) {
		dprintk1("Error, call finish data\n");
		dwcmsh_finish_data(p_drv_data);
	}
	else {

#ifdef DW_SDMMC_DEBUG
		dprintk2("data intmask: 0x%08x\n", intmask);
		if (intmask & DWCMSH_RINTSTS_RXDR)
			dprintk2("Receive Data Request(RXDR)\n");
		if (intmask & DWCMSH_RINTSTS_TXDR)
			dprintk2("Transmit Data Request(TXDR)\n");
		if (intmask & DWCMSH_RINTSTS_DTO)
			dprintk2("Data transfer over(DTO)\n");
#endif
		/* in PIO mode, this transfer PIO is always run here */
		if(!(p_drv_data->flags & DWCMSH_USE_DMA) && 
				p_drv_data->size != 0 && p_drv_data->data != NULL) {
			dprintk2("call transfer_pio\n"); 
			dwcmsh_transfer_pio(p_drv_data);
		}

		if (intmask & DWCMSH_RINTSTS_DTO) {
			if(p_drv_data->flags & DWCMSH_USE_DMA) {
				dprintk3("Wakeup dto waitq\n");
				dto_waitq_flag = 1;
				wake_up_interruptible(&dto_waitq);
			}
			else {
				dprintk3("call finish_data\n");
				dwcmsh_finish_data(p_drv_data); 
			}
		}
	}
}

/*
 * Description: Handle command irq events
 * Param: Pointer to dwc_msh_data_t structure
 * Returns: Nothing
 * Remarks:
 */
static void dwcmsh_cmd_irq(dwc_msh_data_t *p_drv_data)
{
	uint32_t intmask = (p_drv_data->isr & DWCMSH_INT_CMD_MASK);
	p_drv_data->isr &= ~(intmask);

	dprintk2("intmask: 0x%08x\n", intmask);
	if (!p_drv_data->cmd) {
		dprintk0("Got command interrupt even though no "
				"command operation was in progress: 0x%08x\n",
				intmask);
		return;
	}
	/*Check for error conditions */
	if (intmask & DWCMSH_RINTSTS_RTO){
		dprintk1("Cmd timeout error(RTO)\n");
		p_drv_data->cmd->error = -ETIMEDOUT;
		tasklet_schedule(&p_drv_data->finish_tasklet);
	}
	else if (intmask & DWCMSH_RINTSTS_RCRC){
		dprintk1("Cmd crc error(RCRC)\n");
		p_drv_data->cmd->error = -EILSEQ;
		tasklet_schedule(&p_drv_data->finish_tasklet);
	}
	else if (intmask & 
			(DWCMSH_RINTSTS_RE | DWCMSH_RINTSTS_EBE | DWCMSH_RINTSTS_SBE)) {
#ifdef DW_SDMMC_DEBUG
		if (intmask & DWCMSH_RINTSTS_RE)
			dprintk1("Response Error(RE)\n");
		if (intmask & DWCMSH_RINTSTS_EBE)
			dprintk1("End-bit(read)/No-crc error(EBE)\n");
		if (intmask & DWCMSH_RINTSTS_SBE)
			dprintk1("Start-bit error(SBE)\n");
#endif
		p_drv_data->cmd->error = -EIO;
		tasklet_schedule(&p_drv_data->finish_tasklet);
	}
	else {
		/* If no error, Command done, Response is valid*/
		if (intmask & DWCMSH_RINTSTS_CDONE) {
			dprintk2("CDONE, call finish_command\n");
			dwcmsh_finish_command(p_drv_data);
		}
	}
}

/*
 * Description: Handle CMD/DATA interrupts events
 * Param: unsigned long
 * REturns: Nothing
 * Remarks:
 */
static void dwcmsh_tasklet_cmd_data(unsigned long param)
{
	unsigned long flags;
	u32 intmask;
	u32 tmpmask;

	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)param;

	spin_lock_irqsave(&p_drv_data->lock, flags);
	intmask = p_drv_data->isr;

	if (intmask & DWCMSH_INT_CMD_MASK) {
		dprintk4("...running. calling cmd_irq\n");
		dwcmsh_cmd_irq(p_drv_data);
	}
	if (intmask & DWCMSH_INT_DATA_MASK) {
		dprintk4("...running. calling data_irq\n");
		dwcmsh_data_irq(p_drv_data);

		if(!(p_drv_data->flags & DWCMSH_USE_DMA)) {
			tmpmask = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_INTMASK);
			tmpmask |= DWCMSH_RINTSTS_RXDR;
			tmpmask |= DWCMSH_RINTSTS_TXDR;
			DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_INTMASK, tmpmask);
		}
	}
	spin_unlock_irqrestore(&p_drv_data->lock, flags);
}

/*
 * Description: Interrupt Handler
 * Param: integer, void pointer, pointer to pt_regs structure
 * Returns: irqreturns_t
 * REmarks:
 */
static irqreturn_t dwc_msh_irq(int irq, void *dev_id)
{
	irqreturn_t result;
	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)dev_id;
	uint32_t intmask;
	uint32_t tmpmask;

	intmask = DWCMSH_READ_REG(p_drv_data->reg_addr, DWCMSH_REG_MINTSTS);
	if (!intmask) {
		result = IRQ_NONE;
		goto out;
	}
	p_drv_data->isr |= intmask;

	if (!(p_drv_data->mrq)) 
		dprintk1("p_drv_data->mrq is NULL,  intmask: 0x%08x, isr: 0x%08x\n",
				intmask, p_drv_data->isr);
	else
		dprintk1("CMD%d: intmask: 0x%08x, isr: 0x%08x\n",
				p_drv_data->mrq->cmd->opcode, intmask, p_drv_data->isr);

	if (intmask & (DWCMSH_RINTSTS_CD)) {
		dprintk3("RINTSTS_CD, schedule card_tasklet\n");
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
				DWCMSH_REG_RINTSTS, DWCMSH_RINTSTS_CD);
		tasklet_schedule(&p_drv_data->card_tasklet);
		intmask &= ~(DWCMSH_RINTSTS_CD);
	}

	if (intmask & (DWCMSH_INT_CMD_MASK | DWCMSH_INT_DATA_MASK)) {
		dprintk3("CMD/DATA, schedule cmd_data_tasklet\n");
		tasklet_schedule(&p_drv_data->cmd_data_tasklet);

		if (intmask & DWCMSH_RINTSTS_RXDR) {
			tmpmask = DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_INTMASK);
			tmpmask &= ~DWCMSH_RINTSTS_RXDR;
			DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_INTMASK, tmpmask);
		}
		if (intmask & DWCMSH_RINTSTS_TXDR) {
			tmpmask = DWCMSH_READ_REG(p_drv_data->reg_addr,DWCMSH_REG_INTMASK);
			tmpmask &= ~DWCMSH_RINTSTS_TXDR;
			DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_INTMASK, tmpmask);
		}

		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_RINTSTS, 
				(intmask & (DWCMSH_INT_CMD_MASK | DWCMSH_INT_DATA_MASK)));

		intmask &= ~(DWCMSH_INT_CMD_MASK | DWCMSH_INT_DATA_MASK);
	}

	/* Hardware lock error This occurs when CMD queue is full*/
	if (intmask & (DWCMSH_RINTSTS_HLE)) {
		if (p_drv_data->mrq) {
			p_drv_data->mrq->cmd->error = -ETIMEDOUT;
			dprintk3("RINTSTS_HLE, schedule finish_tasklet\n");
			tasklet_schedule(&p_drv_data->finish_tasklet);
		}
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
				DWCMSH_REG_RINTSTS, DWCMSH_RINTSTS_HLE);

		warning("%s: HLE.. reseting Controller\n", 
				mmc_hostname(p_drv_data->mmc));
		/* Reset controller */
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
				DWCMSH_REG_CTRL, DWCMSH_CTRL_CTRLRST);

		dwcmsh_enable_int(p_drv_data);
		intmask &= ~(DWCMSH_RINTSTS_HLE);
	}

	if (intmask & (DWCMSH_RINTSTS_ACD)) {
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, 
				DWCMSH_REG_RINTSTS, DWCMSH_RINTSTS_ACD);
		intmask &= ~(DWCMSH_RINTSTS_ACD);
	}

	if (intmask) {
		dprintk3("clear unhandled interrupt: 0x%08x\n", intmask);
		DWCMSH_WRITE_REG(p_drv_data->reg_addr, DWCMSH_REG_RINTSTS, intmask);
	}

	result = IRQ_HANDLED;

out:
	return result;
}

/*
 * Description: Timer handler
 * Param: unsigned long
 * Returns: nothing
 * Remarks: Timeout for no activity 
 */
static void dwcmsh_timeout_timer(unsigned long data)
{
	unsigned long flags;
	dwc_msh_data_t *p_drv_data = (dwc_msh_data_t *)data;
	struct mmc_data *mmc_data = NULL;

	spin_lock_irqsave(&p_drv_data->lock, flags);

	if (p_drv_data->mrq){
		error("%s: Timeout waiting for Hardware interrupts", 
				mmc_hostname(p_drv_data->mmc));
		dprintk3("***** Timeout waiting for Hardware interrupts ******\n");
#if DW_SDMMC_DEBUG
		dump_regs(p_drv_data);
#endif

		if (p_drv_data->data){
			p_drv_data->data->error = -ETIMEDOUT;
			mmc_data = p_drv_data->mrq->cmd->data,
					 dwcmsh_finish_data(p_drv_data);
		}
		else {
			if (p_drv_data->cmd)
				p_drv_data->cmd->error = -ETIMEDOUT;
			else
				p_drv_data->mrq->cmd->error = -ETIMEDOUT;

			tasklet_schedule(&p_drv_data->finish_tasklet);
		}
	}
	spin_unlock_irqrestore(&p_drv_data->lock, flags);
}

static int device_powerup(void)
{
	return(mobi_reset_disable(RESET_ID_SDMMC));
}

static void device_powerdown(void)
{
	mobi_reset_enable(RESET_ID_SDMMC);
}

/*
 * Driver Core
 */
static int dwcmsh_drv_probe(struct amba_device *pdev, struct amba_id *id)
{	
	int irq, timeout;
	u32 *p_iomem;
	int err=0;
	u32 sig;
	dwcmsh_reg_hcon_t cfg;
	dwc_msh_data_t *p_drv_data;
	struct mmc_host *p_host;
	struct dwc_sdmmc_device_data_t *pdata = 
		(struct dwc_sdmmc_device_data_t *)pdev->dev.platform_data;

	dprintk3("DWC Mobile Storage driver probe\n");
	irq = pdev->irq[0];
	if (irq < 0) {	
		error("No IRQ attached to this device, not supported by this driver.");
		err = -ENODEV;
		goto err_res;
	};

	err = device_powerup();
	if (err) {	
		error("Unable to take device out of reset");
		err = -ENODEV;
		goto err_res;
	};

	err = amba_request_regions(pdev, DRIVER_NAME);
	if(err) {	
		error("No IO memory region found, probe failed.");
		err = -ENODEV;
		goto err_res;
	};

	p_iomem = ioremap(pdev->res.start, pdev->res.end-pdev->res.start+1);
	if (p_iomem == NULL) {	
		error("Failed to remap the device IO memory (0x%08x-0x%08x).",
				pdev->res.start, pdev->res.end-pdev->res.start+1);
		err = -ENOMEM;
		goto err_remap;
	};
	sig = DWCMSH_READ_REG(p_iomem, DWCMSH_REG_VERID);
	if (!DWCMSH_CHECK_SIGNATURE(sig)) {	
		warning("Unrecognized device 0x%p signature: 0x%08x.",pdev,sig);
		err = -ENOSYS;
		goto err_notfound;
	};
	dprintk2("Found DWC Mobile Storage with ID 0x%08x, userID 0x%08x\n",
			sig,DWCMSH_READ_REG(p_iomem, DWCMSH_REG_USRID));
	/* Now we initialize the hardware and the driver structures */
	p_host = mmc_alloc_host(sizeof(dwc_msh_data_t),&(pdev->dev));
	if(p_host == NULL) {	
		error("Unable to allocate memory for the driver data");
		err = -ENOMEM;
		goto err_host_alloc;
	};
	platform_set_drvdata(pdev, p_host);
	
	/* Performance FIXES for 2.6.30 kernel */ 
	/* To get more no blocks in single write request specify max block size and count supported by controller */
	p_host->max_req_size = 65536 * 512;
   p_host->max_blk_size = 65536;
	p_host->max_blk_count = 512;
	
	p_drv_data = (dwc_msh_data_t *)mmc_priv(p_host);
	if(p_drv_data == NULL) {	
		error("Unexpected NULL pointer to the private data");
		err = -ENOMEM;
		goto err_priv_alloc;
	};
	cfg.reg = DWCMSH_READ_REG(p_iomem, DWCMSH_REG_HCON);
	p_drv_data->bus_width = 1<<(cfg.fields.h_data_width+1);

	if (pdata->use_dma || (use_dma == 1)){
		p_drv_data->dma_enable = cfg.fields.dma_interface;
		p_drv_data->flags |= DWCMSH_USE_DMA;
	}
	else {
		p_drv_data->dma_enable = 0;
		p_drv_data->flags &= ~DWCMSH_USE_DMA;
	}

	p_drv_data->disable_hs = (disable_highspeed | pdata->disable_highspeed);

	/* keep a dedicated channel or request for each transaction */
	p_drv_data->dedicated_dma = 
		(dedicated_dma_channel | pdata->dedicated_dma_channel);

	printk(KERN_INFO "DW SD/MMC driver loaded, running in %s mode.\n", 
			(p_drv_data->dma_enable != 0 ? 
			 (dedicated_dma_channel == 1 ? "dedicated dma" : "dma") : "pio"));

	p_drv_data->dma_handle = -1;
	p_drv_data->dma_width = 1<<(cfg.fields.ge_dma_data_width+1);

	p_drv_data->sdmmc_clk = clk_get(&pdev->dev, pdata->clk_name);
	if (override_cclock_rate == 0)
		clk_set_rate(p_drv_data->sdmmc_clk, pdata->clk_rate);
	else {
		clk_set_rate(p_drv_data->sdmmc_clk, override_cclock_rate);
	}
	p_drv_data->cclock = clk_get_rate(p_drv_data->sdmmc_clk);
	dprintk1("Input card clock = %u\n", p_drv_data->cclock);

	p_drv_data->prev_dma_mode = DMA_MODE_NONE;

	p_drv_data->phys_reg_addr = pdev->res.start;
	p_drv_data->reg_addr = p_iomem;
	p_drv_data->card_detect = 1; /* reg shadow, 1 means not detected */
	spin_lock_init(&p_drv_data->lock);

	/* Finally we advertise ourself to the MMC/SD block driver */
	p_host->ops = &dwc_msh_ops;
	p_host->f_min = 400000;   /* Controller Min Freq 400 Khz */
	p_host->f_max = 25000000; /* not HS */

#if DW_SDMMC_DEBUG
	if (fmax_clock_rate != 0) {
		p_host->f_max = fmax_clock_rate;
		printk("Setting fmax clock rate to %d\n",
				fmax_clock_rate);
	}
#endif

	p_host->ocr_avail = MMC_VDD_32_33; /* Voltage level Mask */
#if DW_SDMMC_DEBUG
	if (one_bit_enable == 0)
#endif
		p_host->caps = MMC_CAP_4_BIT_DATA;

	p_drv_data->mmc = p_host;
	p_drv_data->bus_mode = 0;

	p_host->max_hw_segs = 16;
	p_host->max_phys_segs = 16;
	p_host->max_seg_size = 64*512;

	/* Init Tasklets*/
	tasklet_init(&p_drv_data->card_tasklet,
			dwcmsh_tasklet_card, (unsigned long)p_drv_data);
	tasklet_init(&p_drv_data->finish_tasklet,
			dwcmsh_tasklet_finish, (unsigned long)p_drv_data);
	tasklet_init(&p_drv_data->cmd_data_tasklet,
			dwcmsh_tasklet_cmd_data, (unsigned long)p_drv_data);

	setup_timer(&p_drv_data->timer, dwcmsh_timeout_timer, (long)p_drv_data);

	/* Reset the hardware: this will power off card and disable clock */
	dwcmsh_reset(p_drv_data);

	/* Wait for the hardware to be ready ... if it doesn't bail out */
	timeout = WAITFORREADY_TIMEOUT_MS;
	while (DWCMSH_READ_REG(p_drv_data->reg_addr,
				DWCMSH_REG_STATUS) & DWCMSH_STAT_DATA_BUSY) {	
		if (timeout == 0) {	
			error("Card is busy. It may be that card is not working"
					" or a hardware error.");
			err=-EIO;
			goto untasklet;
		};
		timeout--;
		mdelay(1);
	};
	/*Set other DWCMSH IP parameters */
	dwcmsh_set_IP_param(p_drv_data);
	/* Allocate Interrupt */ 
	err = request_irq(irq, dwc_msh_irq, IRQF_DISABLED, DRIVER_NAME, p_drv_data);
	if(err<0) {
		error("Failed to request Interrupt");
		goto untasklet;
	}
	p_drv_data->irq = irq;

	err=mmc_add_host(p_host);
	if(err<0)
		goto err_mmc_add_host;

	/* Now we can safely enable interrupts */
	dwcmsh_enable_int(p_drv_data);

	/* Now we create all the attribute files */
	err = device_create_file(&pdev->dev, &dev_attr_hcon);
	if(err < 0)
		goto err_create_file_hcon;
	err = device_create_file(&pdev->dev, &dev_attr_regs);
	if (err < 0)
		goto err_create_file_regs;
	err = device_create_file(&pdev->dev, &dev_attr_info);
	if(err < 0)
		goto err_create_file_info;
	err = device_create_file(&pdev->dev, &dev_attr_dma_enable);
	if(err < 0)
		goto err_create_file_dma_enable;

#if DW_SDMMC_DEBUG
	err = device_create_file(&pdev->dev, &dev_attr_regrw);
	err = device_create_file(&pdev->dev, &dev_attr_debuglog);
	if(err < 0)
		goto err_create_file_debug;

#ifdef SD_MEM_LOG_ENABLE
	err = device_create_file(&pdev->dev, &dev_attr_memlog);
	if(err < 0)
		goto err_create_file_debug;

#ifdef USE_CODEC_ADDR_FOR_MEMLOG
	memlog_buf = ioremap(MEMLOG_CODEC_OFFSET, MEMLOG_BUFFER_SIZE);
#else
	memlog_buf = (char*)vmalloc(MEMLOG_BUFFER_SIZE);
#endif
	if (memlog_buf == NULL) {
		printk(KERN_ERR "Error allocating memory for log buffer\n");
		goto err_create_file_debug;
	}
	else {
		printk(KERN_INFO "DW SD debug memory logging available\n");
		memlog_buf_len = 0;
	}
#endif
#endif
	err = device_create_file(&pdev->dev, &dev_attr_bmod);
	if(err < 0)
		goto err_create_file_bmod;

	/* Last, we run the card detection tasklet */
	tasklet_schedule(&p_drv_data->card_tasklet);

	dprintk3("DWC Mobile Storage Host IP driver initialized.\n");
	return 0;

err_create_file_bmod:
#if DW_SDMMC_DEBUG
	device_remove_file(&pdev->dev, &dev_attr_debuglog);
#ifdef SD_MEM_LOG_ENABLE
	device_remove_file(&pdev->dev, &dev_attr_memlog);
	if (memlog_buf != NULL)
#ifdef USE_CODEC_ADDR_FOR_MEMLOG
		iounmap(memlog_buf);
#else
	vfree(memlog_buf);
#endif
#endif
err_create_file_debug:
#endif
	device_remove_file(&pdev->dev, &dev_attr_dma_enable);
err_create_file_dma_enable:
	device_remove_file(&pdev->dev, &dev_attr_info);
err_create_file_info:
	device_remove_file(&pdev->dev, &dev_attr_regs);
err_create_file_regs:
	device_remove_file(&pdev->dev, &dev_attr_hcon);
err_create_file_hcon:
	dwcmsh_disable_int(p_drv_data);
	mmc_remove_host(p_host);
err_mmc_add_host:
	free_irq(p_drv_data->irq, p_drv_data);
untasklet:
	clk_set_rate(p_drv_data->sdmmc_clk, 0);
	tasklet_kill(&p_drv_data->card_tasklet);
	tasklet_kill(&p_drv_data->finish_tasklet);
	tasklet_kill(&p_drv_data->cmd_data_tasklet);
	del_timer(&p_drv_data->timer);
err_priv_alloc:
	mmc_free_host(p_host);
err_host_alloc:
err_notfound:
	iounmap(p_iomem);
	p_iomem = NULL;
err_remap:
	amba_release_regions(pdev);
err_res:
	device_powerdown();
	error("Failed to initialize driver: error %d.",err);
	return err;
}

static int dwcmsh_drv_remove(struct amba_device *pdev)
{	
	struct mmc_host *p_host;
	dwc_msh_data_t *priv;

	dprintk3("entered\n");
	p_host=(struct mmc_host *)platform_get_drvdata(pdev);
	priv = (dwc_msh_data_t *)mmc_priv(p_host);
	/* Finishes pending transaction and abort them if necessary
	   then put the IP in reset */
	dwcmsh_disable_int(priv);
	/* power it down */
	dwcmsh_power_card(priv, 0); 

	/* in dedicated channel mode, we need to make sure we free the channel */
	if (priv->dma_handle != -1) {
		mobi_dma_abort(priv->dma_handle);
		mobi_dma_free(priv->dma_handle);
	}

	device_remove_file(&pdev->dev, &dev_attr_bmod);
	device_remove_file(&pdev->dev, &dev_attr_hcon);
#if DW_SDMMC_DEBUG
	device_remove_file(&pdev->dev, &dev_attr_debuglog);
	device_remove_file(&pdev->dev, &dev_attr_regrw);
#ifdef SD_MEM_LOG_ENABLE
	device_remove_file(&pdev->dev, &dev_attr_memlog);
	if (memlog_buf != NULL)
#ifdef USE_CODEC_ADDR_FOR_MEMLOG
		iounmap(memlog_buf);
#else
	vfree(memlog_buf);
#endif
#endif
#endif
	device_remove_file(&pdev->dev, &dev_attr_dma_enable);
	device_remove_file(&pdev->dev, &dev_attr_regs);
	device_remove_file(&pdev->dev, &dev_attr_info);
	mmc_remove_host(p_host);
	free_irq(priv->irq, priv);	
	tasklet_kill(&priv->card_tasklet);
	tasklet_kill(&priv->finish_tasklet);
	tasklet_kill(&priv->cmd_data_tasklet);
	del_timer_sync(&priv->timer);
	clk_set_rate(priv->sdmmc_clk, 0);
	iounmap(priv->reg_addr);
	amba_release_regions(pdev);
	mmc_free_host(p_host);
	platform_set_drvdata(pdev, NULL);
	device_powerdown();

	return 0;
}

#ifdef CONFIG_PM
static int dwcmsh_drv_suspend(struct amba_device *dev, pm_message_t state)
{
	return 0;
}

static int dwcmsh_drv_resume(struct amba_device *dev)
{
	return 0;
}

#else
#define dwcmsh_drv_suspend NULL
#define dwcmsh_drv_resume NULL
#endif

static struct amba_id dwcmsh_ids[] __initdata = {
	{
		.id     = DW_SDMMC_AMBA_DEVID,
		.mask   = 0xffffffff,
	},
	{ 0, 0 },
};

static struct amba_driver dwcmsh_driver = {
	.probe		= dwcmsh_drv_probe,
	.remove		= dwcmsh_drv_remove,
	.suspend	= dwcmsh_drv_suspend,
	.resume		= dwcmsh_drv_resume,
	.id_table	= dwcmsh_ids,
	.drv	= {
		.name	= DW_SDMMC_AMBA_NAME,
	},
};

static int __init dwcmsh_init(void)
{	
	int ret;

	ret = amba_driver_register(&dwcmsh_driver);
	if(ret < 0) {	
		error("DW Mobile Storage driver failed to load, error %d.", ret);
		return ret;
	}

	/* if not set locally or by modparam, then init */
#if defined(LOCAL_DW_SDMMC_DEBUG_ENABLE) || defined(CONFIG_DWCMSH_DEBUG)
	if (loglevel == -1)
#if defined(CONFIG_DWCMSH_DEBUG)
		loglevel = CONFIG_DWCMSH_DEBUG_LEVEL;
#else
	loglevel = 0;
#endif
#endif
	if (loglevel != -1) 
		printk(KERN_INFO "DW SD/MMC debug enabled, loglevel is %d\n", loglevel);

	return 0;
}

static void __exit dwcmsh_exit(void)
{	
	amba_driver_unregister(&dwcmsh_driver);
	printk(KERN_INFO "DW SD/MMC driver unloaded.\n");
}

/*
 * Module declaration
 */
module_init(dwcmsh_init);
module_exit(dwcmsh_exit);

module_param(use_dma, int, 0644);
MODULE_PARM_DESC(use_dma, "Use DMA for data transfers(default 0)");

module_param(disable_highspeed, int, 0644);
MODULE_PARM_DESC(disable_highspeed, "Disable setting card into highspeed mode(default 1)");

module_param(dedicated_dma_channel, int, 0644);
MODULE_PARM_DESC(dedicated_dma_channel, "Allocate a dedeciated dma channel(default 0)");

module_param(override_cclock_rate, int, 0644);
MODULE_PARM_DESC(override_cclock_rate, "Bypass protocol max card clock with new setting");

MODULE_DESCRIPTION("DesignWare Mobile Storage Host IP driver (IP v2.00a)");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Remi Machet, Mobilygen Corp.");
MODULE_VERSION("1.00");

