#define DRIVER_NAME 	"cadpwm"
#define DRIVER_DESC 	"Driver for the Cadence PWM component"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:2.1"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/clk.h>
#include <linux/err.h>

#include <linux/amba/bus.h>

#include <asm/atomic.h>
#include <asm/io.h>

#include <linux/cadpwm.h>

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define MARK 			{ printk("<1>---%s(%d)\n", __func__, __LINE__); }
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define MARK 			{ do {} while (0) }
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

struct cadpwm_drv_data
{
	struct amba_device 		*dev;
	void __iomem 			*base;
	
	struct mutex			access_lock;
	
	struct cadpwm_block_data	*bdata;
	struct clk 			*clock;
	
	char				saved_enable;
	char 				hide_info;
};

/*-------------------------------------------------------------------------*/

static inline u32 apb_read32(void __iomem *base, int reg)
{
	return ioread32(base + reg);
}

static inline void apb_write32(void __iomem *base, int reg, u32 value)
{
	iowrite32(value, base + reg);
}

static inline u16 apb_read16(void __iomem *base, int reg)
{
	return ioread16(base + reg);
}

static inline void apb_write16(void __iomem *base, int reg, u16 value)
{
	iowrite16(value, base + reg);
}

static inline u8 apb_read8(void __iomem *base, int reg)
{
	return ioread8(base + reg);
}

static inline void apb_write8(void __iomem *base, int reg, u8 value)
{
	iowrite8(value, base + reg);
}

/*-------------------------------------------------------------------------*/

static unsigned int cadpwm_get_enabled(struct cadpwm_drv_data *ddata)
{
	unsigned int ret;
	mutex_lock(&ddata->access_lock);
	ret = (apb_read32(ddata->base, CADPWM_IC_CONTROL) == 1);
	mutex_unlock(&ddata->access_lock);
	return ret;
}

static void cadpwm_set_enabled(struct cadpwm_drv_data *ddata, unsigned int enabled)
{
	mutex_lock(&ddata->access_lock);
	if (enabled != 0 && enabled != 1)
		dev_warn(&ddata->dev->dev, "%s: %u passed, assuming you "
			"want to ENABLE block\n", __func__, enabled);
	apb_write32(ddata->base, CADPWM_IC_CONTROL, enabled ? 1 : 0);
	mutex_unlock(&ddata->access_lock);
}

static inline unsigned long roundup_div(const unsigned long numerator,
	const unsigned long denominator)
{
	return ( numerator == 0 
		? 0
		: 1 + ( numerator - 1 ) / denominator );
}

static unsigned long cadpwm_get_dutycycle(struct cadpwm_drv_data *ddata)
{
	unsigned long period, ht, dc = 1;
	mutex_lock(&ddata->access_lock);
	period = apb_read32(ddata->base, CADPWM_IC_PERIOD);
	ht = apb_read32(ddata->base, CADPWM_IC_HIGHTIME);
	if (ht > period) {
		dc = 1000;
	} else if (ht <= CADPWM_HIGHTIME_MIN) {
		dc = 1;
	} else {
		/* Order of operations is VERY important to avoid an overflow. */
		dc = CADPWM_DUTY_CYCLE_RATIO / roundup_div(period, ht);
		if (dc < 1) dc = 1;
	}
	mutex_unlock(&ddata->access_lock);
	return dc;
}

static unsigned long cadpwm_get_dutycycle_min(struct cadpwm_drv_data *ddata)
{
	return 1;
}

static unsigned long cadpwm_get_dutycycle_max(struct cadpwm_drv_data *ddata)
{
	return CADPWM_DUTY_CYCLE_RATIO;
}

static void cadpwm_set_dutycycle(struct cadpwm_drv_data *ddata, unsigned long dutycycle)
{
	unsigned long p, a;
	if (dutycycle > cadpwm_get_dutycycle_max(ddata) ||
		dutycycle < cadpwm_get_dutycycle_min(ddata)) {
		dev_err(&ddata->dev->dev, "%s: cannot reach that "
			"duty cycle: %lu\n", __func__, dutycycle);
		return;
	}
	p = apb_read32(ddata->base, CADPWM_IC_PERIOD);
	if (dutycycle == CADPWM_DUTY_CYCLE_RATIO / 2)
		a = (p >> 1);
	else if (p >= ULONG_MAX / CADPWM_DUTY_CYCLE_RATIO)
		/*
		 * Avoid an overflow with p * dutycycle (p may be equal to 
		 * CADPWM_PERIOD_MAX == ULONG_MAX), divide it first.
		 */
		a = dutycycle * (p / CADPWM_DUTY_CYCLE_RATIO);
	else
		a = (dutycycle * p) / CADPWM_DUTY_CYCLE_RATIO;
	mutex_lock(&ddata->access_lock);
	apb_write32(ddata->base, CADPWM_IC_HIGHTIME,
		min(max(a, CADPWM_HIGHTIME_MIN), CADPWM_HIGHTIME_MAX));
	mutex_unlock(&ddata->access_lock);
}

static unsigned long cadpwm_get_frequency(struct cadpwm_drv_data *ddata)
{
	unsigned long period, freq = 0;
	mutex_lock(&ddata->access_lock);
	period = apb_read32(ddata->base, CADPWM_IC_PERIOD);
	if (period)
		freq = roundup_div(clk_get_rate(ddata->clock), period);
	mutex_unlock(&ddata->access_lock);
	return freq;
}

static unsigned long cadpwm_get_frequency_min(struct cadpwm_drv_data *ddata)
{
	unsigned long freq;
	mutex_lock(&ddata->access_lock);
	freq = roundup_div(clk_get_rate(ddata->clock), CADPWM_PERIOD_MAX);
	mutex_unlock(&ddata->access_lock);
	return freq;
}

static unsigned long cadpwm_get_frequency_max(struct cadpwm_drv_data *ddata)
{
	unsigned long freq;
	mutex_lock(&ddata->access_lock);
	freq = roundup_div(clk_get_rate(ddata->clock), CADPWM_PERIOD_MIN);
	mutex_unlock(&ddata->access_lock);
	return freq;
}

static void cadpwm_set_frequency(struct cadpwm_drv_data *ddata, unsigned long freq)
{
	unsigned long new_freq, dc, enabled;
	if (!freq || freq < cadpwm_get_frequency_min(ddata)
		|| freq > cadpwm_get_frequency_max(ddata)) {
		dev_err(&ddata->dev->dev, "%s: invalid frequency "
			"value: %lu\n", __func__, freq);
		return;
	}
	dc = cadpwm_get_dutycycle(ddata);
	mutex_lock(&ddata->access_lock);
	enabled = (apb_read32(ddata->base, CADPWM_IC_CONTROL) == 1);
	if (enabled)
		apb_write32(ddata->base, CADPWM_IC_CONTROL, 0);
	apb_write32(ddata->base, CADPWM_IC_PERIOD, 
		min(max(roundup_div(clk_get_rate(ddata->clock), freq), 
		CADPWM_PERIOD_MIN), CADPWM_PERIOD_MAX));
	if (enabled)
		apb_write32(ddata->base, CADPWM_IC_CONTROL, 1);
	mutex_unlock(&ddata->access_lock);
	new_freq = cadpwm_get_frequency(ddata);
	if (!new_freq) {
		dev_err(&ddata->dev->dev, "%s: could not set "
			"frequency to %luHz\n", __func__, freq);
	} else {
		cadpwm_set_dutycycle(ddata, dc);
		if (!ddata->hide_info) {
			dev_info(&ddata->dev->dev, "frequency set to %luHz (duty cycle "
				"kept to %lu/%lu)\n", new_freq, dc, CADPWM_DUTY_CYCLE_RATIO);
			if (!cadpwm_get_enabled(ddata))
				dev_warn(&ddata->dev->dev, "block is not enabled, "
					"echo 1 in enabled to do so\n");
		}
	}
}

static unsigned long cadpwm_get_period(struct cadpwm_drv_data *ddata)
{
	unsigned long p;
	mutex_lock(&ddata->access_lock);
	p = apb_read32(ddata->base, CADPWM_IC_PERIOD);
	mutex_unlock(&ddata->access_lock);
	return p;
}

static unsigned long cadpwm_get_hightime(struct cadpwm_drv_data *ddata)
{
	unsigned long h;
	mutex_lock(&ddata->access_lock);
	h = apb_read32(ddata->base, CADPWM_IC_HIGHTIME);
	mutex_unlock(&ddata->access_lock);
	return h;
}

static void cadpwm_set_period(struct cadpwm_drv_data *ddata, unsigned long period)
{
	if (period < CADPWM_PERIOD_MIN || period > CADPWM_PERIOD_MAX) {
		dev_err(&ddata->dev->dev, "%s: invalid period value: %lu. "
			"Min. is %lu, max. is %lu\n", __func__, period,
			CADPWM_PERIOD_MIN, CADPWM_PERIOD_MAX);
		return;
	}
	mutex_lock(&ddata->access_lock);
	apb_write32(ddata->base, CADPWM_IC_PERIOD, period);
	mutex_unlock(&ddata->access_lock);
}

static void cadpwm_set_hightime(struct cadpwm_drv_data *ddata, unsigned long hightime)
{
	if (hightime < CADPWM_HIGHTIME_MIN || hightime > CADPWM_HIGHTIME_MAX) {
		dev_err(&ddata->dev->dev, "%s: invalid hightime value: %lu. "
			"Min. is %lu, max. is %lu\n", __func__, hightime,
			CADPWM_HIGHTIME_MIN, CADPWM_HIGHTIME_MAX);
		return;
	}
	mutex_lock(&ddata->access_lock);
	apb_write32(ddata->base, CADPWM_IC_HIGHTIME, hightime);
	mutex_unlock(&ddata->access_lock);
}

/*-------------------------------------------------------------------------*/

void cadpwm_get(const char *block_id, char *enabled, 
	unsigned long *frequency, unsigned long *dutycycle)
{
	struct amba_device *dev;
	if (unlikely(!block_id || !*block_id))
		return;
	dev = amba_find_device(block_id, NULL, 0, 0);
	if (dev) {
		struct cadpwm_drv_data *ddata = amba_get_drvdata(dev);
		if (frequency)
			*frequency = cadpwm_get_frequency(ddata);
		if (dutycycle)
			*dutycycle = cadpwm_get_dutycycle(ddata);
		if (enabled)
			*enabled = cadpwm_get_enabled(ddata);
	} else {
		error("%s: device with id '%s' not found", __func__, block_id);
	}
}
EXPORT_SYMBOL(cadpwm_get);

void cadpwm_set(const char *block_id, char enabled, 
	unsigned long frequency, unsigned long dutycycle)
{
	struct amba_device *dev;
	if (unlikely(!block_id || !*block_id))
		return;
	dev = amba_find_device(block_id, NULL, 0, 0);
	if (dev) {
		struct cadpwm_drv_data *ddata = amba_get_drvdata(dev);
		mutex_lock(&ddata->access_lock);
		ddata->hide_info = 1;
		mutex_unlock(&ddata->access_lock);
		if (!enabled)
			cadpwm_set_enabled(ddata, 0);
		if (frequency)
			cadpwm_set_frequency(ddata, frequency);
		if (dutycycle)
			cadpwm_set_dutycycle(ddata, dutycycle);
		if (enabled)
			cadpwm_set_enabled(ddata, 1);
		mutex_lock(&ddata->access_lock);
		ddata->hide_info = 0;
		mutex_unlock(&ddata->access_lock);
	} else {
		error("%s: device with id '%s' not found", __func__, block_id);
	}
}
EXPORT_SYMBOL(cadpwm_set);

void cadpwm_get_minmax(const char *block_id, unsigned long *min_frequency, 
	unsigned long *max_frequency, unsigned long *min_dutycycle, 
	unsigned long *max_dutycycle)
{
	struct amba_device *dev;
	if (unlikely(!block_id || !*block_id))
		return;
	dev = amba_find_device(block_id, NULL, 0, 0);
	if (dev) {
		struct cadpwm_drv_data *ddata = amba_get_drvdata(dev);
		if (min_frequency)
			*min_frequency = cadpwm_get_frequency_min(ddata);
		if (max_frequency)
			*min_frequency = cadpwm_get_frequency_max(ddata);
		if (min_dutycycle)
			*min_frequency = cadpwm_get_dutycycle_min(ddata);
		if (max_dutycycle)
			*min_frequency = cadpwm_get_dutycycle_max(ddata);
	} else {
		error("%s: device with id '%s' not found", __func__, block_id);
	}
}
EXPORT_SYMBOL(cadpwm_get_minmax);

/*-------------------------------------------------------------------------*/

#define CADPWM_DECLARE_R_DEV_ATTR_FN(name, fmt, unit, type, getfn) \
static ssize_t dev_attr_show_ ## name(struct device *dev, \
	struct device_attribute *attr, char *buf) \
{ \
	int ret = 0; struct cadpwm_drv_data *ddata = dev_get_drvdata(dev); \
	ret += snprintf(buf + ret, PAGE_SIZE - ret, fmt, getfn(ddata)); \
	ret += snprintf(buf + ret, PAGE_SIZE - ret, unit "\n"); \
	return ret; \
}
#define CADPWM_DECLARE_W_DEV_ATTR_FN(name, fmt, type, setfn) \
static ssize_t dev_attr_store_ ## name(struct device *dev, struct device_attribute *attr, \
	const char *buf, size_t count) \
{ \
	struct cadpwm_drv_data *ddata = dev_get_drvdata(dev); type tmp; \
	if (sscanf(buf, fmt, &tmp) == 1) { setfn(ddata, tmp); } \
	else { dev_err(&ddata->dev->dev, "%s: invalid input, expected input " \
		"format for %s is %" fmt "\n", __func__, #name); } \
	return count; \
}

#define CADPWM_DECLARE_RW_DEV_ATTR(name, fmt, unit, type, getfn, setfn) \
	CADPWM_DECLARE_R_DEV_ATTR_FN(name, fmt, unit, type, getfn) \
	CADPWM_DECLARE_W_DEV_ATTR_FN(name, fmt, type, setfn) \
static DEVICE_ATTR(name, 0644, dev_attr_show_ ## name, dev_attr_store_ ## name);

#define CADPWM_DECLARE_R_DEV_ATTR(name, fmt, unit, type, getfn) \
	CADPWM_DECLARE_R_DEV_ATTR_FN(name, fmt, unit, type, getfn) \
static DEVICE_ATTR(name, 0444, dev_attr_show_ ## name, NULL);

#define CADPWM_CREATE_DEV_ATTR(name) \
	if (unlikely(ret = device_create_file(&dev->dev, &dev_attr_ ## name))) { \
		dev_err(&dev->dev, "%s: unable to create device " \
			"attribute file %s\n", __func__, #name); \
		goto err_devattr_ ## name; \
	}
	
#define CADPWM_REMOVE_DEV_ATTR(name) { device_remove_file(&dev->dev, &dev_attr_ ## name); }

#define CADPWM_CREATE_DEV_ATTR_END(name) err_devattr_ ## name: CADPWM_REMOVE_DEV_ATTR(name)


CADPWM_DECLARE_RW_DEV_ATTR(frequency, "%lu", "\nHz", unsigned long, 
	cadpwm_get_frequency, cadpwm_set_frequency)
	
CADPWM_DECLARE_RW_DEV_ATTR(enabled, "%u", "", unsigned int, 
	cadpwm_get_enabled, cadpwm_set_enabled)
	
CADPWM_DECLARE_RW_DEV_ATTR(dutycycle, "%lu", "\n/" CADPWM_DUTY_CYCLE_RATIO_STRING, 
	unsigned long, cadpwm_get_dutycycle, cadpwm_set_dutycycle)
	
CADPWM_DECLARE_R_DEV_ATTR(frequency_min, "%lu", "\nHz", unsigned long, 
	cadpwm_get_frequency_min)
	
CADPWM_DECLARE_R_DEV_ATTR(frequency_max, "%lu", "\nHz", unsigned long, 
	cadpwm_get_frequency_max)
	
CADPWM_DECLARE_R_DEV_ATTR(dutycycle_min, "%lu", "\n/" CADPWM_DUTY_CYCLE_RATIO_STRING, 
	unsigned long, cadpwm_get_dutycycle_min)
	
CADPWM_DECLARE_R_DEV_ATTR(dutycycle_max, "%lu", "\n/" CADPWM_DUTY_CYCLE_RATIO_STRING, 
	unsigned long, cadpwm_get_dutycycle_max)
	
CADPWM_DECLARE_RW_DEV_ATTR(period, "%lu", "", unsigned long, 
	cadpwm_get_period, cadpwm_set_period)
	
CADPWM_DECLARE_RW_DEV_ATTR(hightime, "%lu", "", unsigned long, 
	cadpwm_get_hightime, cadpwm_set_hightime)


static int cadpwm_probe(struct amba_device *dev, struct amba_id *id)
{
	int 				ret;
	struct cadpwm_drv_data		*ddata;
	struct cadpwm_block_data 	*bdata = dev->dev.platform_data;
	
	if (unlikely(!bdata))
		return -EINVAL;
	
	ddata = kzalloc(sizeof(struct cadpwm_drv_data), GFP_KERNEL);
	if (unlikely(!ddata)) {
		ret = -ENOMEM;
		goto err_alloc;
	}
	ddata->dev = dev;
	ddata->bdata = bdata;
	
	amba_set_drvdata(dev, ddata);
	
	ddata->base = ioremap(dev->res.start, dev->res.end - dev->res.start + 1);
	if (unlikely(!ddata->base)) {
		dev_err(&dev->dev, "%s: unable to remap memory (address=0x%08x, size=%d)\n",
			__func__, dev->res.start, dev->res.end - dev->res.start + 1);
		ret = -EBUSY;
		goto err_ioremap;
	}
	
	/* Disable component & interrupts */
	apb_write32(ddata->base, CADPWM_IC_CONTROL, 0);
	apb_write32(ddata->base, CADPWM_IC_IR_DISABLE, ~0);
	
	ddata->clock = clk_get(&dev->dev, ddata->bdata->clock_id);
	if (unlikely(IS_ERR(ddata->clock))) {
		ret = PTR_ERR(ddata->clock);
		dev_err(&dev->dev, "%s: failed to get clock id=%s, "
			"error is %d\n", __func__, 
			bdata->clock_id, ret);
		goto err_clkget;
	}
	
	mutex_init(&ddata->access_lock);
	ddata->saved_enable = 0;
	ddata->hide_info = 1;
	
	/* Set min frequency and 50% duty cycle */
	cadpwm_set_frequency(ddata, cadpwm_get_frequency_min(ddata));
	cadpwm_set_dutycycle(ddata, CADPWM_DUTY_CYCLE_RATIO / 2);
	
	ddata->hide_info = 0;
	
	CADPWM_CREATE_DEV_ATTR(frequency);
	CADPWM_CREATE_DEV_ATTR(enabled);
	CADPWM_CREATE_DEV_ATTR(dutycycle);
	CADPWM_CREATE_DEV_ATTR(frequency_min);
	CADPWM_CREATE_DEV_ATTR(frequency_max);
	CADPWM_CREATE_DEV_ATTR(dutycycle_min);
	CADPWM_CREATE_DEV_ATTR(dutycycle_max);
	CADPWM_CREATE_DEV_ATTR(period);
	CADPWM_CREATE_DEV_ATTR(hightime);
	
	dev_info(&dev->dev, "device ready\n");
	
	return 0;
	
	CADPWM_CREATE_DEV_ATTR_END(hightime);
	CADPWM_CREATE_DEV_ATTR_END(period);
	CADPWM_CREATE_DEV_ATTR_END(dutycycle_max);
	CADPWM_CREATE_DEV_ATTR_END(dutycycle_min);
	CADPWM_CREATE_DEV_ATTR_END(frequency_max);
	CADPWM_CREATE_DEV_ATTR_END(frequency_min);
	CADPWM_CREATE_DEV_ATTR_END(dutycycle);
	CADPWM_CREATE_DEV_ATTR_END(enabled);
	CADPWM_CREATE_DEV_ATTR_END(frequency);
	
	clk_put(ddata->clock);
err_clkget:
	iounmap(ddata->base);
err_ioremap:
 	kfree(ddata);
err_alloc:
	return ret;
}

static int cadpwm_remove(struct amba_device *dev)
{
	struct cadpwm_drv_data 	*ddata = amba_get_drvdata(dev);
	
	dev_info(&dev->dev, "removing device\n");
	
	CADPWM_REMOVE_DEV_ATTR(hightime);
	CADPWM_REMOVE_DEV_ATTR(period);
	CADPWM_REMOVE_DEV_ATTR(dutycycle_max);
	CADPWM_REMOVE_DEV_ATTR(dutycycle_min);
	CADPWM_REMOVE_DEV_ATTR(frequency_max);
	CADPWM_REMOVE_DEV_ATTR(frequency_min);
	CADPWM_REMOVE_DEV_ATTR(dutycycle);
	CADPWM_REMOVE_DEV_ATTR(enabled);
	CADPWM_REMOVE_DEV_ATTR(frequency);
	
	clk_put(ddata->clock);
	
	iounmap(ddata->base);
	kfree(ddata);
	
	return 0;
}

#ifdef CONFIG_PM

static int cadpwm_suspend(struct amba_device *dev, pm_message_t msg)
{
#ifdef CADPWM_DISABLE_ON_SUSPEND
	struct cadpwm_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "suspending\n");
	ddata->saved_enable = cadpwm_get_enabled(ddata);
	cadpwm_set_enabled(ddata, 0);
#endif
	return 0;
}

static int cadpwm_resume(struct amba_device *dev)
{
#ifdef CADPWM_DISABLE_ON_SUSPEND
	struct cadpwm_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "resuming\n");
	cadpwm_set_enabled(ddata, ddata->saved_enable);
#endif
	return 0;
}

#endif

/*-------------------------------------------------------------------------*/

static struct __initdata amba_id cadpwm_ids[] = {
	{
		.id	= CADPWM_ID,
		.mask	= CADPWM_ID_MASK,
	},
	{ 0, 0 },
};

static struct __initdata amba_driver cadpwm_drv = {
	.drv		= {
		.name	= DRIVER_NAME,
	},
	.probe		= cadpwm_probe,
	.remove		= cadpwm_remove,
#ifdef CONFIG_PM
	.suspend	= cadpwm_suspend,
	.resume		= cadpwm_resume,
#endif
	.id_table	= cadpwm_ids,
};

static int __init cadpwm_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	if (unlikely(ret = amba_driver_register(&cadpwm_drv)))
		error("could not register AMBA driver");

	return ret;
}

static void __exit cadpwm_exit(void)
{
	amba_driver_unregister(&cadpwm_drv);
}


module_init(cadpwm_init);
module_exit(cadpwm_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
