/******************************************************************************
 * mhif-merlin.c
 *
 * This implements the master host interface configuration platform driver.
 * The information about each device associated with the chip selects are
 * obtained through devices registered in /sys/devices/platform/ 
 *
 * The misc driver is not used because the misc class is shared with other 
 * devices and misc has only one major number.  
 *
 * The major number is dynamically allocated. It is unique for this driver
 * and is not shared with other drivers.  The char device is registered
 * using the cdev and class device.
 ******************************************************************************/

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include <asm/io.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <linux/moduleparam.h>
#include <asm/bitops.h>
#include <asm/uaccess.h>
#include <linux/delay.h> 				// for mdelay() 
#include <linux/interrupt.h> 			// request_irq()

#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/err.h>
#include <linux/proc_fs.h>

#include <linux/mhif.h>
#include <linux/sbus.h> 				// ioctls to do 8/16/32 bit read/writes

#include <linux/mobi_mhif_device.h>

#include <asm/arch/platform.h>
#include <asm/arch/timex.h>				// system clock frequency info
#include <asm/arch/mobi_clock.h>		// AHB clock freq through mobi_clock_get_rate(CLOCK_ID_AHB) 
#include <asm/arch/mobi_hmux.h>			// host mux platform driver  
#include <asm/arch/mobi_dma.h>	
#include <asm/arch/mobi_reset.h>
#include <asm/arch/mobi_qccsisc.h>

#include "mg3500_mhif_regs.h"

// In MHIF space, the registers and data space is selected by bit 24 on/off.
#define MHIF_REG_BASE  MHIF_BASE | 0x01000000
#define MHIF_REG_SIZE  0x40 

struct device;
struct class_device;

struct mhifdevice  {
	int minor;
	const char *name;					// mhif.0 device name 
	struct device *dev;					// -> platform_device.dev -^
	struct class_device *class;			//
	struct semaphore sem;				// mutual exclusion 
	struct cdev cdev;					// char device structure
	// additional fields for driver's own use
	void __iomem *virt_base;			// device memory pio xfer base address
	u32	phys_base;						// device memory dma xfer base address
	u32 file_size;						// device memory block size
	mobi_dma_handle   dma_handle;		// dma handle 
	wait_queue_head_t dma_waitq;		// dma interrupt waiting and notification
	int	dma_waitq_flag;					// dma interrupt wait queue flag 
	u8 *kernbuf;						// 4K word-aligned kmalloc kernel space buffer
};

//int mhif_major = 0; 					//major number is dynamically allocated
int mhif_nr_devs = MHIF_MAX_DEVS; 
module_param(mhif_nr_devs, int, S_IRUGO);

//static struct mhifdevice * devlistpio[MHIF_MAX_DEVS]; //one for each cs
//static struct mhifdevice * devlistdma[MHIF_MAX_DEVS]; //one for each cs
// Note:
// devlistpio[] and devlistdma[] array index = cs#
// cs# = minor# / 2

//static struct class * mhif_class; 		// created in the init(), one for all cs

// open() private_data struct allocated per open() as needed by user app 
#if 0
struct open_pdata {
	struct mhifdevice * mdev;
	u_int8_t	endian;			// 0=little, 1=big
	u_int8_t	addr_inc;		// 0=no auto increment. 1=auto increment of address
	int reg_ofs;				// config reg_a_N address offset (use for switching open's)
	int reg_val;				// config reg_a_N register value to program
};
#endif

static int debug = 0;
module_param(debug, int, 0);	//module_param(debug, int, S_IRUGO | S_IWUGO);
MODULE_PARM_DESC(debug, "Enable debug");
#define dprintk(a...)	if (debug) { printk(a); }

void __iomem *mhif_reg_base;

/******************************************************************************
 * this is used by both .probe and .remove for each of the device 
 ******************************************************************************/
struct mhif_cs_info {
	struct resource	*res;

	// these variables from struct map_info (map.h) are all that's needed here
	char *name;
	unsigned long size;
	unsigned long phys;	
	void __iomem *virt;
};

#if 0
/******************************************************************************
 * close() for /dev/mhif.N
 ******************************************************************************/
static int mhif_close_pio(struct inode *inode, struct file *filp)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);

	// free the kernel space buffer used by read() write() for this cs device 
	kfree(mdev->kernbuf);
	kfree(odata); //free the container that was holding the mdev

	up(&mdev->sem);
	return 0;
}

/******************************************************************************
 * close for /dev/mhif.N.dma
 ******************************************************************************/
static int mhif_close_dma(struct inode *inode, struct file *filp)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);

	//releasing allocated write DMA
	mobi_dma_free(mdev->dma_handle); 
	mdev->dma_handle = -1;

	// free the kernel space buffer used by read() write() for this cs device 
	kfree(mdev->kernbuf);

	kfree(odata); //free the container that was holding the mdev

	up(&mdev->sem);
	return 0;
}

/******************************************************************************
 * seek() for /dev/mhif.N and /dev/mhif.N.dma
 ******************************************************************************/
static loff_t mhif_llseek(struct file *filp, loff_t offset, int origin)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	loff_t newpos;
	
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);

	switch (origin)
	{
	case 0: /* SEEK_SET */
		newpos = offset;
		break;
	case 1: /* SEEK_CUR */
		newpos = filp->f_pos + offset;
		break;
	default: 
		return -EINVAL;
	}

	if ((newpos >= mdev->file_size) || (newpos < 0))
	{
		printk(KERN_ERR "Error: file seek out of bound\n");		
		up(&mdev->sem);
	 	return -EINVAL;
	}
	
	filp->f_pos = newpos;

	up(&mdev->sem);
	return newpos;
}

/******************************************************************************
 * ioctl() 
 ******************************************************************************/
static int mhif_ioctl(struct inode *inode, struct file *filp,
			      unsigned int cmd, unsigned long arg, int use_dma)
{
	int retval = 0;

	struct open_pdata *odata;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	sbus_ioaccess_t *io;
	sbus_endianmode_t *en;
	void *mempos; 				// mapped memory address of device for io
	int cs;
	int reg_ofs;
	int reg_a_val;
	
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);
	mobi_hmux_request();		// grap host mux semaphore ownership

	cs_data = mdev->dev->platform_data;

	cs = MINOR(inode->i_rdev) >> 1;
	dprintk(KERN_INFO "ioctl device number cs = %d\n", cs);

	switch (cs)
	{
	case 0: reg_ofs = MHIF_CFGA0_REG; break;
	case 1: reg_ofs = MHIF_CFGA1_REG; break;
	case 2: reg_ofs = MHIF_CFGA2_REG; break;
	case 3: reg_ofs = MHIF_CFGA3_REG; break;
	case 4: reg_ofs = MHIF_CFGA4_REG; break;
	case 5: reg_ofs = MHIF_CFGA5_REG; break;
	default: retval = -EINVAL; goto exit;
	}		

	if (copy_from_user(mdev->kernbuf, (char*)arg, sizeof(sbus_ioaccess_t)))
	{
		printk(KERN_ERR "copy_from_user failed\n");
		retval = -EFAULT;
		goto exit;
	}
	io = (sbus_ioaccess_t*)mdev->kernbuf;

	dprintk(KERN_INFO "cmd=0x%08X io->address=0x%08lX  io->value=0x%08lX \n", cmd, io->address, io->value);

	mempos = mdev->virt_base + io->address;

	switch(cmd) {

	case SBUS_IOCW_BYTE:
		dprintk("iocw08 [0x%08X]=0x%02lX\n", (u32)io->address, io->value);
		if (io->address+1 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		writeb(io->value, mempos);
		break;
			
	case SBUS_IOCW_SHORT:
		dprintk("iocw16 [0x%08X]=0x%04lX\n", (u32)io->address, io->value);
		if (io->address+2 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		if (odata->endian) // if big endian
			writew(cpu_to_be16(io->value), mempos);
		else
			writew(cpu_to_le16(io->value), mempos);
		break;

	case SBUS_IOCW_LONG:
		dprintk("iocw32 [0x%08X]=0x%08lX\n", (u32)io->address, io->value);
		if (io->address+4 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		if (odata->endian) // if big endian
			writel(cpu_to_be32(io->value), mempos);
		else
			writel(cpu_to_le32(io->value), mempos);
		break;

	case SBUS_IOCR_BYTE:
		if (io->address+1 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		io->value = readb(mempos);
		dprintk("iocr08 [0x%08X]=0x%02lX\n", (u32)io->address, io->value);
		break;

	case SBUS_IOCR_SHORT:
		if (io->address+2 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		if (odata->endian) // if big endian
			io->value = be16_to_cpu(readw(mempos));
		else
			io->value = le16_to_cpu(readw(mempos));
		dprintk("iocr16 [0x%08X]=0x%04lX\n", (u32)io->address, io->value);
		break;

	case SBUS_IOCR_LONG:
		if (io->address+4 > mdev->file_size)
		{
			retval = -EFAULT;
			goto exit;
		}
		if (odata->endian) // if big endian
			io->value = be32_to_cpu(readl(mempos));
		else
			io->value = le32_to_cpu(readl(mempos));
		dprintk("iocr32 [0x%08X]=0x%08lX\n", (u32)io->address, io->value);
		break;

	case SBUS_IOCS_IOBASE:
		// this is an optional additional offset within the open file space.
		retval = -EFAULT;
		break;
		
	case SBUS_IOCS_ADINCR:	// ADINCR ioctl takes an unsigned char parameter
		odata->addr_inc = mdev->kernbuf[0];	// 0=no auto increment. 1=auto increment of address
		dprintk(KERN_INFO "ioctl ADINCR = %d\n", odata->addr_inc);

		reg_a_val = MHIFREG_READL(reg_ofs) & ~(1<<31);
		if (odata->addr_inc)
			reg_a_val |= (1<<31);
		dprintk(KERN_INFO "MHIF_CFGA%d_REG=0x%08X\n", cs, reg_a_val);
		MHIFREG_WRITEL(reg_a_val, reg_ofs);	

		odata->reg_ofs = reg_ofs;
		odata->reg_val = reg_a_val;

		break;

	case SBUS_IOCS_ENDIANESS:
		if (copy_from_user(mdev->kernbuf, (char*)arg, sizeof(sbus_endianmode_t)))
		{
			printk(KERN_ERR "copy_from_user failed\n");
			retval = -EFAULT;
			goto exit;
		}
		en = (sbus_endianmode_t*)mdev->kernbuf;
	
		//dprintk(KERN_INFO "en->device  =%d\n", en->device); 		//0=little, 1=big endian 
		//dprintk(KERN_INFO "en->wordsize=%d\n", (u32)en->wordsize); 	//size of a word in bits

		odata->endian = en->device;	//0=little, 1=big endian
		break;

	case SBUS_IOCS_TXSIZE:

	case SBUS_IOCS_LOCK:

	case SBUS_IOCS_UNLOCK:

	default:
		retval = -ENOIOCTLCMD;
		break;
	}

	if (copy_to_user((char*)arg, mdev->kernbuf, sizeof(sbus_ioaccess_t)))
	{
		printk(KERN_ERR "copy_to_user failed\n");
		retval = -EFAULT;
		goto exit;
	}

exit:
	mobi_hmux_release();	// return host mux semaphore ownership
	up(&mdev->sem);
	return retval;
}



/******************************************************************************
 * ioctl for /dev/mhif.N 
 * the byte/word/long single read/writes uses address offsets within each of 
 * the cs devices.
 ******************************************************************************/
static int mhif_ioctl_pio(struct inode *inode, struct file *filp,
						  unsigned int cmd, unsigned long arg)
{
	return mhif_ioctl(inode, filp, cmd, arg, 0);
}


/******************************************************************************
 * ioctl for /dev/mhif.N.dma  
 * the byte/word/long single read/writes uses address offsets within each of 
 * the cs devices.
 ******************************************************************************/
static int mhif_ioctl_dma(struct inode *inode, struct file *filp,
						  unsigned int cmd, unsigned long arg)
{
	return mhif_ioctl(inode, filp, cmd, arg, 1);
}


/******************************************************************************
 * write() for /dev/mhif.N
 * For a 8-bit device, we do 8-bit wide writes, 16-bit for 16-bit wide.
 * 32-bit operation on a 16/8 bit device is not recommended or supported by hw.
 *
 * For an 16-bit wide device, the transfer size specified should be an even 
 * number.  If an odd number transfer size is given, here in the driver we
 * will reject it by returning an error and not perform the transfer.
 *
 * In DMA mode, the write should wait until the transfer is complete before
 * returning back to the user.  For large size transfer, could use dma list
 * to transfer multiple 4K blocks.
 *
 * In PIO mode, we can process in 4K chunks if the incoming user's buffer is 
 * bigger than 4K. 
 ******************************************************************************/
static ssize_t mhif_write_pio(struct file *filp, const char *userbuf, size_t len, loff_t *ppos)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	int retval=0, status=0;
	u16 *ptr16;
	u8 *ptr8;
	void *mempos; 							// phys memory address of device write location
	int xferlen; 							// size of transfer in each iteration 

	odata = (struct open_pdata*)filp->private_data;	// mhifdevice structure pointer 
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);						// no one else can rd/wr this device 
	mobi_hmux_request();					// grap host mux semaphore ownership

	// restore config setting for THIS open() channel handle
	MHIFREG_WRITEL(odata->reg_val, odata->reg_ofs);	

	cs_data = mdev->dev->platform_data;		// cs config parameters structure

	if (*ppos+len > mdev->file_size)
	{
		dprintk(KERN_ERR "File position out of bound\n");
		status = -EINVAL;
		goto exit;
	}

	dprintk("wr len=%d cs=%d, w=%d f_pos=0x%08X data:", 
			len, mdev->minor>>1, cs_data->width, (u32)*ppos);

	if (cs_data->width > 1) { 				// reject odd length 16-bit wide transfer 
		if (len & 1) {
			dprintk(KERN_ERR "Invalid length for 16-bit device\n");
			status = -EINVAL;
			goto exit;
		}		
	}

	// here we do PIO transfer iterations of up to 4KB at a time
	mempos = mdev->virt_base + *ppos; 
	retval = len;

	while (len > 0)							// consume in 4K chunks until len is consumed 
	{
		xferlen = (len>4096? 4096:len);
		if (copy_from_user(mdev->kernbuf, userbuf, xferlen))
		{
			printk(KERN_ERR "copy_from_user failed\n");
			retval = -EFAULT;
			goto exit;
		}

		userbuf += xferlen;
		len -= xferlen;

		if (odata->addr_inc)
		{
			*ppos += xferlen;
		}

		// it's an 8-bit or 16-bit device, so use the correct type of write 
		ptr8 = mdev->kernbuf;
		if (cs_data->width > 1) // width:2 = 16-bit hword
		{
			ptr16=(s16*)mdev->kernbuf;

			if (odata->endian) // if big endian
			{
				while (xferlen>=2)
				{
					dprintk("%04X ", *ptr16);
					writew(cpu_to_be16(*ptr16++), mempos);
					if (odata->addr_inc)
						mempos += 2;	
					xferlen -= 2;
				}
				ptr8 = (s8*)ptr16;	
			}
			else // little endian
			{
				while (xferlen>=2)
				{
					dprintk("%04X ", *ptr16);
					writew(cpu_to_le16(*ptr16++), mempos);
					if (odata->addr_inc)
						mempos += 2;	
					xferlen -= 2;
				}
				ptr8 = (s8*)ptr16;
			}
		}

		while (xferlen-- > 0)	// width:1 = 8-bit byte read
		{
			dprintk(KERN_INFO "%02X ", *ptr8);
			writeb(*ptr8++, mempos);
			if (odata->addr_inc)
				mempos++;
		}
		dprintk("\n");
	}

exit:
	mobi_hmux_release();	// return host mux semaphore ownership
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * write() for /dev/mhif.N.dma
 * For a 8-bit device, we do 8-bit wide writes, 16-bit for 16-bit wide.
 * 32-bit operation on a 16/8 bit device is not recommended or supported by hw.
 *
 * For an 16-bit wide device, the transfer size specified should be an even 
 * number.  If an odd number transfer size is given, here in the driver we
 * will reject it by returning an error and not perform the transfer.
 *
 * In DMA mode, the write should wait until the transfer is complete before
 * returning back to the user.  For large size transfer, could use dma list
 * to transfer multiple 4K blocks.
 ******************************************************************************/
static ssize_t mhif_write_dma(struct file *filp, const char *userbuf, size_t len, loff_t *ppos)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	int retval=0, status=0;
	dma_addr_t src_dma_addr, dst_dma_addr;
	int xferlen; 							// size of transfer in each iteration 
	int dmacfg;

	odata = (struct open_pdata*)filp->private_data;	 
	mdev  = (struct mhifdevice*)odata->mdev;// mhifdevice structure pointer
	down(&mdev->sem);						// no one else can rd/wr this device 
	mobi_hmux_request();					// grap host mux semaphore ownership

	// restore config setting for THIS open() channel handle
	MHIFREG_WRITEL(odata->reg_val, odata->reg_ofs);	

	cs_data = mdev->dev->platform_data;		// cs config parameters structure

	if (*ppos+len > mdev->file_size)
	{
		dprintk(KERN_ERR "File position out of bound\n");
		status = -EINVAL;
		goto exit_dma;
	}

	dprintk("wr len=%d cs=%d, w=%d f_pos=0x%08X\n", 
			len, mdev->minor>>1, cs_data->width, (u32)*ppos);

	if (cs_data->width > 1) { 				// reject odd length 16-bit wide transfer 
		if (len & 1) {
			dprintk(KERN_ERR "Invalid length for 16-bit device\n");
			status = -EINVAL;
			goto exit_dma;
		}		
	}

	// using DMA API to do dma
	dmacfg = MOBI_DMA_CONFIG_TRANSFER_WIDTH_32|MOBI_DMA_CONFIG_BURST_SIZE_8;

	if (odata->endian) // 0=little, 1=big
		dmacfg |= MOBI_DMA_CONFIG_ENDIAN_BE;
	else
		dmacfg |= MOBI_DMA_CONFIG_ENDIAN_LE;

	if (odata->addr_inc) 
		dmacfg |= MOBI_DMA_CONFIG_ADDRADJ_INC;
	else
		dmacfg |= MOBI_DMA_CONFIG_ADDRADJ_NONE;

	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_SRC, dmacfg, NULL))) 
	{
	    printk(KERN_ERR "DMA src fail\n");
	    goto exit_dma;
    }

	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_DST,
					((cs_data->width>1)?
					 (MOBI_DMA_CONFIG_TRANSFER_WIDTH_16|MOBI_DMA_CONFIG_ADDRADJ_INC|MOBI_DMA_CONFIG_BURST_SIZE_8) :
					 (MOBI_DMA_CONFIG_TRANSFER_WIDTH_8 |MOBI_DMA_CONFIG_ADDRADJ_INC|MOBI_DMA_CONFIG_BURST_SIZE_8)),
					 NULL ) )) 
	{
	   	printk(KERN_ERR "DMA dst fail\n");
	    goto exit_dma;
	}
	
	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_XFER,
			MOBI_DMA_CONFIG_DATA_WIDTH_16,
			NULL))) 
	{
	    printk(KERN_ERR "DMA config xfer fail\n");
	    goto exit_dma;
	}


	dst_dma_addr = (dma_addr_t)mdev->phys_base + *ppos;
	retval = len;

	while ((len > 0) && (status>=0))		// consume in 4K chunks until len is consumed 
	{
		xferlen = (len>4096? 4096:len);
		/* XXX avoid!! */
		if (copy_from_user(mdev->kernbuf, userbuf, xferlen)) {
			printk(KERN_ERR "copy_from_user failed\n");
			status = -EFAULT;
			goto exit_dma;
		}

		// Convert virtual to physical. Now buffer is owned by the DMA and the CPU must not touch it.
	    src_dma_addr = dma_map_single(NULL, (void*)(mdev->kernbuf), xferlen, DMA_TO_DEVICE);
	    if (src_dma_addr <= 0) {
			printk(KERN_ERR "dma_map_single failed\n");
			status = src_dma_addr;
			goto exit_dma;
	    }
			
		if ((status = mobi_dma_setup_single(mdev->dma_handle, src_dma_addr, dst_dma_addr, xferlen))) {
			printk(KERN_ERR "Failed mobi_dma_setup_single\n");
			goto exit_dma_unmap_single;
		}
			
		status = mobi_dma_enable(mdev->dma_handle); // this kicks start the DMA
		if (status < 0) {
			printk(KERN_ERR "Error %d mobi_dma_enable() failed to start dma\n", status);
			goto exit_dma_unmap_single;				
		}

		// sleep until done
   		wait_event_interruptible(mdev->dma_waitq, mdev->dma_waitq_flag != 0);
		if (mdev->dma_waitq_flag < 0) {		// if error reported by handler
			printk(KERN_ERR "DMA transfer error\n");
			status = -EFAULT;
			// fall through to clean up and return with error 
		}
		mdev->dma_waitq_flag = 0;

		// clean up after either the completion or if mobi_dma_enable() failed to start DMA
		mobi_dma_disable(mdev->dma_handle);

exit_dma_unmap_single:	
		dma_unmap_single(NULL, src_dma_addr, xferlen, DMA_TO_DEVICE);

		if (odata->addr_inc)
		{
			dst_dma_addr += xferlen;  		// go to next dest block 
			*ppos += xferlen;
		}
		userbuf += xferlen;
		len -= xferlen;
	}
exit_dma:	// return number of bytes transferred, or error code if any
	if (status < 0)
		retval = status;		

	mobi_hmux_release();					// return host mux semaphore ownership
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * read() for pio
 ******************************************************************************/
static ssize_t mhif_read_pio(struct file *filp, char *userbuf, size_t len, loff_t *ppos)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	int retval=0, status=0;
	u16 *ptr16;
	u16 val16;
	s8 *ptr8;
	void *mempos; 							// phys memory address of device write location
	int xferlen; 							// size of transfer in each iteration 
	int cnt;								// loop counter for xferlen
		
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);
	mobi_hmux_request();					// grap host mux semaphore ownership

	// restore config setting for THIS open() channel handle
	MHIFREG_WRITEL(odata->reg_val, odata->reg_ofs);	

	cs_data = mdev->dev->platform_data;
	
	if (*ppos+len > mdev->file_size)
	{
		dprintk(KERN_ERR "File position out of bound\n");
		status = -EINVAL;
		goto exit;
	}

	dprintk(KERN_INFO "rd len=%d cs=%d, w=%d f_pos=0x%08X data:", 
					  len, mdev->minor>>1, cs_data->width, (u32)*ppos);

	if (cs_data->width > 1) { 				// reject odd length 16-bit wide transfer 
		if (len & 1) {
			dprintk(KERN_ERR "Invalid length for 16-bit device\n");
			status = -EINVAL;
			goto exit;
		}		
	}

	mempos = (void*)mdev->virt_base + *ppos;
	retval = len;

	while (len > 0)							// consume in 4K chunks until len is consumed 
	{
		xferlen = (len>4096? 4096:len);
		cnt = xferlen;

		// it's an 8-bit or 16-bit device, so use the correct type of write 
		ptr8 = mdev->kernbuf;
		if (cs_data->width > 1) // width:2 = 16-bit hword
		{
			ptr16=(s16*)mdev->kernbuf;
			
			if (odata->endian) // if big endian
			{
				while (cnt>=2)
				{
					val16 = be16_to_cpu(readw(mempos));
					dprintk("%04X ", val16); 				   
					cnt -= 2;
					*ptr16++ = val16;
					if (odata->addr_inc)
						mempos += 2;    
				}
				ptr8 = (s8*)ptr16;				
			}
			else // little endian  
			{
				while (cnt>=2)
				{
					val16 = le16_to_cpu(readw(mempos));  	
					dprintk("%04X ", val16); 				   
					cnt -= 2;
					*ptr16++ = val16;
					if (odata->addr_inc)
						mempos += 2;    
				}
				ptr8 = (s8*)ptr16;				
			}
		}
		while (cnt-- > 0)	// width:1 = 8-bit byte read
		{
			*ptr8 = readb(mempos);
			dprintk("%02X ", *ptr8);
			ptr8++;
			mempos++;
		}
		dprintk("\n");
		
		if (copy_to_user(userbuf, mdev->kernbuf, xferlen))
		{
			printk(KERN_ERR "copy_to_user failed (pio)\n");
			retval = -EFAULT;
			goto exit;
		}
		userbuf += xferlen;
		len -= xferlen;
		if (odata->addr_inc)
			*ppos += xferlen;
	}

exit:
	mobi_hmux_release();					// return host mux semaphore ownership
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * read() for dma
 * Note: for DMA, the hardware requires that the memory side of the transfer 
 * to be 32-bit in width.
 ******************************************************************************/
static ssize_t mhif_read_dma(struct file *filp, char *userbuf, size_t len, loff_t *ppos)
{
	struct open_pdata *odata;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	int retval=0, status=0;
	dma_addr_t src_dma_addr, dst_dma_addr;
	int xferlen; 							// size of transfer in each iteration 
	int dmacfg;
		
	odata = (struct open_pdata*)filp->private_data;
	mdev  = (struct mhifdevice*)odata->mdev;
	down(&mdev->sem);
	mobi_hmux_request();					// grap host mux semaphore ownership

	// restore config setting for THIS open() channel handle
	MHIFREG_WRITEL(odata->reg_val, odata->reg_ofs);	

	cs_data = mdev->dev->platform_data;

	if (*ppos+len > mdev->file_size)
	{
		dprintk(KERN_ERR "File position out of bound\n");
		status = -EINVAL;
		goto exit_dma;
	}

	dprintk(KERN_INFO "rd len=%d cs=%d, w=%d f_pos=0x%08X\n", 
					  len, mdev->minor>>1, cs_data->width, (u32)*ppos);

	if (cs_data->width > 1) { 				// reject odd length 16-bit wide transfer 
		if (len & 1) {
			dprintk(KERN_ERR "Invalid length for 16-bit device\n");
			status = -EINVAL;
			goto exit_dma;
		}		
	}

	// use DMA API to do DMA
	dmacfg = MOBI_DMA_CONFIG_TRANSFER_WIDTH_32|MOBI_DMA_CONFIG_BURST_SIZE_8;

	if (odata->endian) // 0=little, 1=big
		dmacfg |= MOBI_DMA_CONFIG_ENDIAN_BE;
	else
		dmacfg |= MOBI_DMA_CONFIG_ENDIAN_LE;

	if (odata->addr_inc) 
		dmacfg |= MOBI_DMA_CONFIG_ADDRADJ_INC;
	else
		dmacfg |= MOBI_DMA_CONFIG_ADDRADJ_NONE;

	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_DST, dmacfg, NULL))) 
	{
	    printk(KERN_ERR "DMA src fail\n");
	    goto exit_dma;
    }

	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_SRC,
				((cs_data->width>1)?
				 (MOBI_DMA_CONFIG_TRANSFER_WIDTH_16|MOBI_DMA_CONFIG_ADDRADJ_INC|MOBI_DMA_CONFIG_BURST_SIZE_8) :
				 (MOBI_DMA_CONFIG_TRANSFER_WIDTH_8 |MOBI_DMA_CONFIG_ADDRADJ_INC|MOBI_DMA_CONFIG_BURST_SIZE_8)),
				 NULL ))) {
	    printk(KERN_ERR "DMA src fail\n");
	    goto exit_dma;
    }
	
	if ((status = mobi_dma_config(mdev->dma_handle, DMA_CONFIG_XFER,
			MOBI_DMA_CONFIG_DATA_WIDTH_16,
			NULL))) 
	{
	    printk(KERN_ERR "DMA config xfer fail\n");
	    goto exit_dma;
	}


	src_dma_addr = (dma_addr_t)mdev->phys_base + *ppos;
	retval = len;

	/* ACK! copies suck map the pages */
	while ((len > 0) && (status>=0))		// consume in 4K chunks until len is consumed 
	{
		xferlen = (len>4096? 4096:len);

		// Convert virtual to physical. Now buffer is owned by the DMA and the CPU must not touch it.
	    dst_dma_addr = dma_map_single(NULL, (void*)(mdev->kernbuf), xferlen, DMA_FROM_DEVICE);
	    if (dst_dma_addr <= 0) {
			printk(KERN_ERR "dma_map_single failed\n");
			status = dst_dma_addr;
			goto exit_dma;
	    }

		if ((status = mobi_dma_setup_single(mdev->dma_handle, src_dma_addr, dst_dma_addr, xferlen))) {
			printk(KERN_ERR "Failed mobi_dma_setup_single\n");
			goto exit_dma_unmap_single;
		}
			
		status = mobi_dma_enable(mdev->dma_handle); 	// this kicks starts the DMA
		if (status < 0) {
			printk(KERN_ERR "Error: mobi_dma_enable() returns %d\n", status);
			goto exit_dma_unmap_single;
		}

		// sleep until done
   		wait_event_interruptible(mdev->dma_waitq, mdev->dma_waitq_flag != 0); 
		if (mdev->dma_waitq_flag < 0) {		// if error reported by handler
			printk(KERN_ERR "DMA transfer error\n");
			status = -EFAULT;
			// fall through to clean up and return with error 
		}
		mdev->dma_waitq_flag = 0;			// isr completed so reset the flag 

		// clean up after either the completion or mobi_dma_enable() failure
		mobi_dma_disable(mdev->dma_handle);

exit_dma_unmap_single:			
		dma_unmap_single(NULL, dst_dma_addr, xferlen, DMA_FROM_DEVICE);
			
		if (status>=0)
		{
			if (copy_to_user(userbuf, mdev->kernbuf, xferlen)) {
				printk(KERN_ERR "copy_to_user failed (dma)\n");
				status = -EFAULT;
				goto exit_dma;
			}			
			if (odata->addr_inc)
			{
				src_dma_addr += xferlen; 	// go to next src block 
				*ppos += xferlen;
			}
			userbuf += xferlen;
			len -= xferlen;
		}
	}
exit_dma: // return number of bytes transferred, or error code if any
	if (status < 0)
		retval = status;

	mobi_hmux_release();					// return host mux semaphore ownership
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * write dma interrupt handler
 ******************************************************************************/
void dma_handler(mobi_dma_handle dmah, mobi_dma_event dma_event, void *mdev_voidptr)
{
	struct mhifdevice *mdev;
	mobi_dma_status status = 0;

	mdev = (struct mhifdevice *)mdev_voidptr;

	mdev->dma_waitq_flag = 0;
	switch (dma_event) 
	{
		case MOBI_DMA_EVENT_TRANSFER_COMPLETE:
			status = mobi_dma_get_status(dmah);
			mdev->dma_waitq_flag = 1;
			//printk(KERN_INFO "dma completed status=%d\n", status);
			break;
		case MOBI_DMA_EVENT_TRANSFER_ERROR:
			mdev->dma_waitq_flag = -1;
			//printk(KERN_INFO "dma transfer error\n"); 
			break;
	}
    wake_up_interruptible((wait_queue_head_t*)&(mdev->dma_waitq));
}


/******************************************************************************
 * open() for /dev/mhif.N
 ******************************************************************************/
static int mhif_open_pio(struct inode *inode, struct file *filp)
{
	int retval=0;
	int cs;
	struct mhifdevice *mdev;
	struct merlin_mhif_data *cs_data;
	struct open_pdata *odata;

	cs = MINOR(inode->i_rdev) >> 1;
	dprintk(KERN_INFO "device number cs = %d\n", cs);

	if (cs >= MHIF_MAX_DEVS)
		return -EINVAL;
		
	// tell read/write/ioctl the address of this device's data
	mdev = (struct mhifdevice*)devlistpio[cs];
	down(&mdev->sem);						// exclusive usage
	
	cs_data = mdev->dev->platform_data;		// to access (cs_data->endian)
	filp->f_pos = 0;

	//--------------------------------------------------------------------------
	// allocate a 4K kernel space buffer for use by read(), write()	
	//--------------------------------------------------------------------------
	mdev->kernbuf = kmalloc(4096, GFP_KERNEL);
	if (mdev->kernbuf == NULL) {
		printk(KERN_ERR "kmalloc failed\n");
		retval = -ENOMEM;
		goto exit;
	}
	
	//--------------------------------------------------------------------------
	// allocate an open_private_data struct for this open instance 
	//--------------------------------------------------------------------------
	odata = kmalloc(sizeof(struct open_pdata), GFP_KERNEL);
	if (odata == NULL) {
		printk(KERN_ERR "kmalloc failed\n");
		kfree(mdev->kernbuf);
		retval = -ENOMEM;
		goto exit;
	}
	
	odata->mdev = mdev;						// put mdev "inside" odata
	odata->endian = cs_data->endian;		// copy platform driver properties
	odata->addr_inc = cs_data->addr_inc;	// copy platform driver properties
	filp->private_data = odata;				// odata is our main container 
	
exit:
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * open() for /dev/mhif.N.dma
 ******************************************************************************/
static int mhif_open_dma(struct inode *inode, struct file *filp)
{
	int retval=0;
	int cs;
	struct mhifdevice *mdev;
//	struct merlin_mhif_data *cs_data;
	struct open_pdata *odata;

	cs = MINOR(inode->i_rdev) >> 1;
	dprintk(KERN_INFO "device number cs = %d\n", cs);

	if (cs >= MHIF_MAX_DEVS)
		return -EINVAL;
		
	// tell read/write/ioctl the address of this device's data
	mdev = (struct mhifdevice*)devlistdma[cs];	
	down(&mdev->sem);						// exclusive usage

//	cs_data = mdev->dev->platform_data;
	filp->f_pos = 0;

	//--------------------------------------------------------------------------
	// setup read/write DMA for this cs device
	//--------------------------------------------------------------------------
	// allocate a handle for this dma channel
    if ((mdev->dma_handle = mobi_dma_request("mhif", MOBI_DMA_O_NONE)) < 0) 
	{
		printk(KERN_ERR "mobi_dma_request() not able to allocate a free DMA\n");
		retval = -EINVAL;
		goto exit;
	}
	else // successful, so install dma isr handler
	{
		mdev->dma_waitq_flag = 0;
		init_waitqueue_head(&(mdev->dma_waitq));
		if ((retval = mobi_dma_setup_handler(mdev->dma_handle, (void *)dma_handler, (void*)mdev)))
		{
			printk(KERN_ERR "mobi_dma_setup_handler() for write\n");
			goto exit;
		}
	}

	//--------------------------------------------------------------------------
	// allocate a 4K kernel space buffer for use by read(), write()	
	//--------------------------------------------------------------------------
	mdev->kernbuf = kmalloc(4096, GFP_KERNEL);
	if (mdev->kernbuf == NULL) {
		printk(KERN_ERR "kmalloc failed\n");
		retval = -ENOMEM;
		goto exit;
	}
		
	//--------------------------------------------------------------------------
	// allocate an open_private_data struct for this open instance 
	//--------------------------------------------------------------------------
	odata = kmalloc(sizeof(struct open_pdata), GFP_KERNEL);
	if (odata == NULL) {
		printk(KERN_ERR "kmalloc failed\n");
		kfree(mdev->kernbuf);
		retval = -ENOMEM;
		goto exit;
	}

	odata->mdev = mdev;			// put mdev "inside" odata
	filp->private_data = odata;	// odata is our main container 
		
exit:
	up(&mdev->sem);
	return retval;
}


/******************************************************************************
 * user access functions
 ******************************************************************************/
static const struct file_operations mhif_fops_pio = {
	.owner		= THIS_MODULE,
	.llseek		= mhif_llseek,
	.ioctl		= mhif_ioctl_pio,
	.open		= mhif_open_pio,
	.release	= mhif_close_pio,
	.write		= mhif_write_pio,
	.read		= mhif_read_pio,
};


static const struct file_operations mhif_fops_dma = {
	.owner		= THIS_MODULE,
	.llseek		= mhif_llseek,
	.ioctl		= mhif_ioctl_dma,
	.open		= mhif_open_dma,
	.release	= mhif_close_dma,
	.write		= mhif_write_dma,
	.read		= mhif_read_dma,
};

#endif 

int bitsat(unsigned int value, unsigned int mask)
{
	return (value > mask ? mask : value);
}

/******************************************************************************
 * program the mhif hardware registers based on config parameters 
 ******************************************************************************/
int mhif_program_regs(int cs, 
		struct merlin_mhif_data *cs_data, struct resource *res)
{
	int reg_ofs;
	int reg_a_val, reg_b_val;
	int addr24mask;
	int addr24size;	/* size of the memory device in number of bytes */
	int clkns;		/* clock period in ns, used for computing clock cycles  */

	addr24size = res->end - res->start + 1; /* size of memory device */

	switch (cs) {
		case 0: reg_ofs = MHIF_CFGA0_REG; break;
		case 1: reg_ofs = MHIF_CFGA1_REG; break;
		case 2: reg_ofs = MHIF_CFGA2_REG; break;
		case 3: reg_ofs = MHIF_CFGA3_REG; break;
		case 4: reg_ofs = MHIF_CFGA4_REG; break;
		case 5: reg_ofs = MHIF_CFGA5_REG; break;
		default: return -EINVAL;
	}		

	/* XXX need to try this with the real clock,  the AHB has probably 
	 * changed a couple times now but the values based on a clkns of 1
	 * still seem to be working but it seems sketchy
	 */
	/*clkns = (1000000000/mobi_clock_get_rate(CLOCK_ID_AHB)); */
	dprintk(KERN_INFO "bus_clk_ns=%d ns\n", cs_data->bus_clk_ns);
	clkns = cs_data->bus_clk_ns;
	if (clkns == 0) {
		printk(KERN_ERR "mhif.%d bus clock config setting is invalid\n", cs);
		return -EINVAL;
	}

	dprintk(KERN_INFO "width=%d endian=%d bux_mux=%d dma_wen=%d "
			"wait_en=%d addr_inc=%d\n", 
			cs_data->width, cs_data->endian, cs_data->bus_mux, cs_data->dma_wen, 
			cs_data->wait_en, cs_data->addr_inc);

	dprintk(KERN_INFO "ale_setup_ns=%d ale_width_ns=%d ale_hold_ns=%d\n", 
			cs_data->ale_setup_ns, cs_data->ale_width_ns, cs_data->ale_hold_ns);

	dprintk(KERN_INFO "rd_wait_ns=%d wr_hold_ns=%d rw_setup_ns=%d "
			" rw_hold_ns=%d\n", 
			cs_data->rd_wait_ns, cs_data->wr_hold_ns, 
			cs_data->rw_setup_ns, cs_data->rw_hold_ns);

	dprintk(KERN_INFO "dev_idle_ns=%d prescaler=%d timeout_en=%d dev_timeout_ns=%d\n", 
			cs_data->dev_idle_ns, cs_data->prescaler, 
			cs_data->timeout_en, cs_data->dev_timeout_ns);

	dprintk(KERN_INFO "ale_on_fall_edge=%d wait_active_low=%d "
			"dreq_active_low=%d\n", 
			cs_data->ale_on_fall_edge, 
			cs_data->wait_active_low, cs_data->dreq_active_low);

	addr24mask = -addr24size;		// calc mask based on device size
	if (addr24size < 0x1000)		// if size is too small then
		addr24mask = -0x1000;		// round up to the resolution
	if (addr24size==0)				// if size is 0 then
		addr24mask = -1;			// set the correct mask for 0
	
	reg_a_val = ((res->start & 0x00FFF000) >>12)// AHB[23:12] address base
			  | ((addr24mask & 0x00FFF000))		// address mask
			  | ((cs_data->width >1? 1:0)<<24)	// width: 1=8bit, 2=16bit
			  | (0<<25)   //((cs_data->endian&3)<<25)	// normally little endian. Use DMA api and bridge instead
			  | ((cs_data->bus_mux>0? 1:0)<<27)	// 
			  | ((cs_data->dma_wen>0? 1:0)<<28) //
			  | ((cs_data->wait_en>0? 1:0)<<29) //
			  | ((cs_data->addr_inc>0? 1:0)<<31); 		//use DMA api and bridge instead 

	// reg_a_val = 0x81FFF000; // debug test value
	dprintk(KERN_INFO "MHIF_CFGA%d_REG=0x%08X\n", cs, reg_a_val);
	MHIFREG_WRITEL(reg_a_val, reg_ofs);	

	 reg_b_val = (bitsat(cs_data->ale_setup_ns/clkns,0x3) ) 		// 2 ALE setup time
			  | ((bitsat(cs_data->ale_width_ns/clkns,0x3)) <<2)		// 2 ALE width time
			  | ((bitsat(cs_data->ale_hold_ns/clkns,0x3)) <<4)		// 2 ALE hold time
			  | ((bitsat(cs_data->rd_wait_ns/clkns,0xF)) <<6)		// 4 read data out sampling delay (waitstates)
			  | ((bitsat(cs_data->wr_hold_ns/clkns,0x3)) <<10)		// 2 write hold data delay in ns
			  | ((bitsat(cs_data->dev_timeout_ns/clkns,0x7F)) <<12)	// 7 device wait time out delay in ns
			  | ((bitsat(cs_data->dev_idle_ns/clkns,0x3)) <<19)		// 2 device inter-access chip sel idle time in ns
			  | ((bitsat(cs_data->prescaler,0x3)) <<21)				// 2 prescaler for ale_setup,width,hold, rd_wait, wr_hold, idle
			  | ((bitsat(cs_data->timeout_en,0x1)) <<23)			// 1 timeout enable 0=off, 1=on
			  | ((bitsat(cs_data->rw_setup_ns/clkns,0x3))<<24)		// 2 delay between cs and write in ns
			  | ((bitsat(cs_data->ale_on_fall_edge,0x1)) <<26)		// 1 ale strobe on 0=rise, 1=fall edge
			  | ((1) <<27)											// 1 chip select. Documentation says 0=active high, 1=active low 
			  | ((bitsat(cs_data->wait_active_low,0x1)) <<28)		// 1 MHIF_WAIT input 0=active high, 1=active low
			  | ((bitsat(cs_data->dreq_active_low,0x1)) <<29)		// 1 MHIF_DMARQ input 0=active high, 1=active low
			  | ((bitsat(cs_data->rw_hold_ns/clkns,0x3))<<30);		// 2 read hold time in ns for cs, rd, addr

	// reg_b_val = 0xFF7FFFFF; // debug test value
	dprintk(KERN_INFO "MHIF_CFGB%d_REG=0x%08X\n", cs, reg_b_val);
	MHIFREG_WRITEL(reg_b_val, reg_ofs+4);

	return 0;
}

static irqreturn_t mhif_isr(int irq, void *dev_id)
{

	/* clear interrupt. Here we don't need atomic_set() 
	 * writing 1 bit clears the int, writing 0 bit does nothing 
	 */
	/* XXX 32 bit access on 16bit register */
	if (MHIFREG_READL(MHIF_INT_PEND_REG) & MHIF_INTEN_UNMAPPED)
		MHIFREG_WRITEL(MHIF_INTEN_UNMAPPED, MHIF_INT_PEND_REG);

	/* XXX 32 bit access on 16bit register */
	/* XXX disables interrupts instead of clearing */
	if (MHIFREG_READL(MHIF_INT_PEND_REG) & MHIF_INTEN_TIMEOUT) 
		MHIFREG_WRITEL(MHIF_INTEN_TIMEOUT, MHIF_INT_EN_CLR_REG);

	return IRQ_HANDLED;
}

/* this is invoked upon rmmod */
static int mhif_remove(struct platform_device *pdev)
{
	struct mhif_cs_info *info;

#if 0
	//=============================================================
	// do the equivalent of: misc_deregister(devlist[pdev->id])

	// remove DMA driver
	if (devlistdma[pdev->id] != NULL)
	{
		struct mhifdevice * d;
		d = devlistdma[pdev->id];	
		cdev_del(&d->cdev);
		class_device_destroy(mhif_class, MKDEV(mhif_major, d->minor));
		printk(KERN_INFO "/dev/%s\n", d->name);
		kfree(devlistdma[pdev->id]);
		devlistdma[pdev->id] = NULL;
	}

	// remove PIO driver
	if (devlistpio[pdev->id] != NULL)
	{
		struct mhifdevice * d;
		d = devlistpio[pdev->id];	
		cdev_del(&d->cdev);
		class_device_destroy(mhif_class, MKDEV(mhif_major, d->minor));
		printk(KERN_INFO "/dev/%s\n", d->name);
		kfree(devlistpio[pdev->id]);
		devlistpio[pdev->id] = NULL;
	}
#endif

	//=============================================================

	info = platform_get_drvdata(pdev);
	if (info == NULL)
		return 0;
	platform_set_drvdata(pdev, NULL);

	if (info->virt != NULL)
		iounmap((void *)info->virt);

	return 0;
}


#if 0
/******************************************************************************
 * Set up the char_dev structure for this device.
 ******************************************************************************/
static void mhif_setup_cdev(struct mhifdevice *mdev, 			 
							const struct file_operations *mhif_fops) // input table
{
	int err; 
	dev_t devno = MKDEV(mhif_major, mdev->minor); // minors start at 0

	cdev_init(&mdev->cdev, mhif_fops);
	mdev->cdev.owner = THIS_MODULE;
	mdev->cdev.ops = mhif_fops;
	if ((err=cdev_add (&mdev->cdev, devno, 1)))
	{
		printk(KERN_NOTICE "Error %d adding mhif at minor %d", err, mdev->minor);
	}

	printk(KERN_INFO "cdev(%d,%d) /dev/%s\n", mhif_major, mdev->minor, mdev->name);

	// the last step is to connect our driver to the mhif class 
	mdev->class = class_device_create(mhif_class, NULL, devno, mdev->dev, "%s", mdev->name);
	if (IS_ERR(mdev->class)) {
		err = PTR_ERR(mdev->class);
		goto exit;
	}
	return;
exit:
	printk(KERN_ERR "mhif_setup_cdev fail\n");
	return;
}
#endif


/******************************************************************************
 * mhif_probe()
 * create 1 info structure, 2 mhifdevice structures
 * setup 2 cdev's
 ******************************************************************************/
static int mhif_probe(struct platform_device *pdev)
{
	struct merlin_mhif_data *merlin_mhif_data;
	struct mhif_cs_info *info;
	int err;
	int cs;
	u32 regval;

#if 0
    struct mhifdevice *mdev_pio;	// data for setting up user file_operations
    struct mhifdevice *mdev_dma; 	// data for setting up user file_operations
	char *dev_name_dma;			// str holding dev name like "mhif.0.dma"
#endif

	merlin_mhif_data = pdev->dev.platform_data;
	if (merlin_mhif_data == NULL)
		return -ENODEV;

	info = kmalloc(sizeof(struct mhif_cs_info), GFP_KERNEL);
	if (info == NULL) {
		err = -ENOMEM;
		goto err_out;
	}
	memset(info, 0, sizeof(*info));

	/* store info address inside pdev */
	platform_set_drvdata(pdev, info);	

	info->name = pdev->dev.bus_id;								
	info->phys = pdev->resource->start;		
	info->size = pdev->resource->end - pdev->resource->start + 1;	

	dprintk(KERN_NOTICE "info->name=%s\n",    info->name);
	dprintk(KERN_NOTICE "info->phys=%08lX\n", info->phys);
	dprintk(KERN_NOTICE "info->size=%08lX\n", info->size);

	info->virt = ioremap(info->phys, info->size);  
	if (info->virt == NULL) {
		dev_err(&pdev->dev, "Failed to ioremap memory region\n");
		err = -EIO;
		goto err_out;
	}

	if (pdev->id > MHIF_MAX_DEVS) {
		err = -ENOMEM;
		goto err_out;
	}

#if 0
	//=========================================================================
	// for each device, dynamically allocate the equivalent of this:
	// static struct miscdevice mhif_miscdev = {
	//	.minor		= 0,			 
	//	.name		= "mhif.0",
	//	.fops		= &mhif_fops,
	// };

		// this device id entry already exist? 
		if ((devlistpio[pdev->id]) || (devlistdma[pdev->id])) {		
			err = -EBUSY;
			goto err_out;	
		}
	
		//---------------------------------------------------------------------
		// allocate 1st mhifdevice struct 
		//---------------------------------------------------------------------
		mdev_pio = kmalloc(sizeof(struct mhifdevice), GFP_KERNEL);
		if (mdev_pio == NULL) {
			err = -ENOMEM;
			goto err_out;
		}
		memset(mdev_pio, 0, sizeof(*mdev_pio));
		mdev_pio->dev   	= &pdev->dev;			// associate device with this driver 
		mdev_pio->minor 	= (pdev->id <<1)+0,		// one device number into 2 minor numbers
		mdev_pio->name  	= pdev->dev.bus_id;		// device name "mhif.<id>"
		mdev_pio->virt_base = info->virt;			// virtual base address for pio
		mdev_pio->phys_base = info->phys;			// physical base address for dma
		mdev_pio->file_size = info->size;			// block size of device memory
		//do the equivalent of: misc_register(mdev_pio);
		init_MUTEX(&mdev_pio->sem);
		mhif_setup_cdev(mdev_pio, &mhif_fops_pio); // minor number to have 2 cdevs
	
		devlistpio[pdev->id] = mdev_pio;			// table of device structs, for remove()

		//---------------------------------------------------------------------
		// allocate 2nd mhifdevice struct and space for string holding dev name
		//---------------------------------------------------------------------
		mdev_dma = kmalloc(sizeof(struct mhifdevice) + 16, GFP_KERNEL);
		if (mdev_dma == NULL) {
			err = -ENOMEM;
			goto err_out;
		}
		dev_name_dma = ((char*)(mdev_dma) + sizeof(struct mhifdevice));
		strcpy(dev_name_dma, pdev->dev.bus_id);		// /dev/"mhif.0"
		strcat(dev_name_dma, ".dma"); 				// /dev/"mhif.0.dma"
	
		memset(mdev_dma, 0, sizeof(*mdev_dma));
		mdev_dma->dev   	= &pdev->dev;			// associate device with this driver 
		mdev_dma->minor 	= (pdev->id <<1)+1,		// one device number into 2 minor numbers
		mdev_dma->name  	= dev_name_dma;			// /dev/"mhif.0.dma" pdev->dev.bus_id;
		mdev_dma->virt_base = info->virt;			// virtual base address for pio
		mdev_dma->phys_base = info->phys;			// physical base address for dma
		mdev_dma->file_size = info->size;			// block size of device memory

		//do the equivalent of: misc_register(mdev_dma);
		init_MUTEX(&mdev_dma->sem);
		mhif_setup_cdev(mdev_dma, &mhif_fops_dma); 	// minor number to have 2 cdevs

		devlistdma[pdev->id] = mdev_dma;			// table of device structs, for remove()
	//=========================================================================

#endif
	if (mhif_program_regs(pdev->id,	
				pdev->dev.platform_data, pdev->resource) !=0) {
		err = -ENOMEM;
		goto err_out;
	}
	
	/* set appropriate bit in MHIF_CS_ENABLE reg to enable that cs */
	/* chip sel number */
	cs = pdev->id;	
	regval=MHIFREG_READL(MHIF_CS_ENABLE);		
	regval=regval | (1<<cs);
	MHIFREG_WRITEL(regval, MHIF_CS_ENABLE);	

	if (MHIFREG_READL(MHIF_CS_ENABLE) != regval) {
		err=-EFAULT;
		goto err_out;
	}
	return 0;

err_out:
	mhif_remove(pdev);
	if (info) 
		kfree(info);

	return err;
}

static void mhif_shutdown(struct platform_device *pdev)
{
	return;
}

static int mhif_suspend(struct platform_device *pdev, pm_message_t message)
{
	return 0;
}

static int mhif_resume(struct platform_device *pdev)
{
	return 0;
}

/******************************************************************************
 *
 ******************************************************************************/
static struct platform_driver mhif_driver = {
	.probe	    = mhif_probe,
	.remove	    = mhif_remove,
	.shutdown   = mhif_shutdown,
	.suspend    = mhif_suspend,
	.resume     = mhif_resume,
	.driver	    = {
	    .name = MHIF_PLATFORM_NAME,
	},
};

/******************************************************************************
 * For debug purpose only
 ******************************************************************************/
#if 0
int mhif_reg_test(void)
{
	printk(KERN_ALERT "Registers R/W test\n");
	MHIFREG_WRITEL(0xABCD3264, MHIF_CFGA0_REG);		
	printk(KERN_ALERT "writing CFGB0\n");
	MHIFREG_WRITEL(0x12345678, MHIF_CFGB0_REG);	
	if ((0xABCD3264!=MHIFREG_READL(MHIF_CFGA0_REG)) || (0x12345678!=MHIFREG_READL(MHIF_CFGB0_REG)))
	{
		printk(KERN_INFO "Registers R/W FAIL\n");
		return -1;
	}

	MHIFREG_WRITEL(0x00000000, MHIF_CFGA0_REG);		
	MHIFREG_WRITEL(0x00000000, MHIF_CFGB0_REG);	
	if ((0x00000000!=MHIFREG_READL(MHIF_CFGA0_REG)) || (0x00000000!=MHIFREG_READL(MHIF_CFGB0_REG)))
	{
		printk(KERN_INFO "Registers R/W FAIL\n");
		return -1;
	}

	printk(KERN_INFO "Registers R/W PASS\n");
	return 0;
}
#endif

static int __init mhif_init(void)
{
	int err;
	/* 
	dev_t dev = 0; 
	int i
	*/

	mhif_reg_base = ioremap(MHIF_REG_BASE, MHIF_REG_SIZE);
	/* XXX not all devices */
	MHIFREG_WRITEL(0x3, MHIF_INT_EN_CLR_REG); /* disable interrupts */

	/* 
	 * the MHIF_WAIT timeout interrupt needs the MHIF_CS_ENABLE register to 
	 * tell it which cs's are actually in use for a real device.  We need to 
	 * set the corresponding cs bits to 0 for unused cs's, and 1 for the ones 
	 * in use.
	 */
	MHIFREG_WRITEL(0x0, MHIF_CS_ENABLE);	//start with 0, let probe set each bit.

#if 0
	// initialize the list of pointers to each device mhifdev structs
	for (i=0; i<MHIF_MAX_DEVS; i++)
		devlistpio[i] = NULL;
	for (i=0; i<MHIF_MAX_DEVS; i++)
		devlistdma[i] = NULL;

	//-----------------------------------------------------------------------
	// create a class called 'mhif'
	//-----------------------------------------------------------------------
	mhif_class = class_create(THIS_MODULE, "mhif");
	if (IS_ERR(mhif_class))
	{
		printk(KERN_INFO "mhif_init class_create failed\n");  
		return PTR_ERR(mhif_class);
	}
	//-----------------------------------------------------------------------
	// grab several minor numbers and one dynamically allocated major number 
	//-----------------------------------------------------------------------
	err = alloc_chrdev_region(&dev, 0, mhif_nr_devs, "mhif"); // 2nd param is minor #
	mhif_major = MAJOR(dev);	// remember our allocated major number

	if (err < 0) {
		printk(KERN_WARNING "mhif: can't get major %d\n", mhif_major);
		class_destroy(mhif_class);	// undo the created class 
		return err;
	}
#endif

#if 0
	mhif_reg_test();
#endif
	err = platform_driver_register(&mhif_driver);

	err = request_irq(MOBI_IRQ_MHIF, mhif_isr, IRQF_DISABLED, "mhif",  0);//&cb);
	if (err != 0) {
		printk(KERN_INFO "### request_irq() failed!!! ### Return=%d\n", err);
		// recovery TBD	
	}

	/* XXX why when don't know which device selected */
	MHIFREG_WRITEL(0x2, MHIF_INT_EN_SET_REG); //enable the timeout interrupt

	return err;
}

/******************************************************************************
 *
 ******************************************************************************/
static void __exit mhif_exit(void)
{
	/* dev_t devno; */

	MHIFREG_WRITEL(0x3, MHIF_INT_EN_CLR_REG); 
	free_irq(MOBI_IRQ_MHIF,  0);

	/* platform_driver to unregister itself, runs .REMOVE */
	platform_driver_unregister(&mhif_driver);

#if 0
	//-------------------------------------------------------
	// give back the several minor numbers we grabbed in init 
	//-------------------------------------------------------
	devno = MKDEV(mhif_major, 0); // 2nd param is minor number
	// mhif_exit() is never called if registering failed
	unregister_chrdev_region(devno, mhif_nr_devs);

	//-------------------------------------------------------
	// remove the class we created
	//-------------------------------------------------------
	class_destroy(mhif_class);	// undo the created class 
#endif
}

/******************************************************************************/
module_init(mhif_init);
module_exit(mhif_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Peter Hsiang");
MODULE_DESCRIPTION("Master host interface driver");
