/*
 *  drivers/mhif/mg3500_mhif.c
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/clk.h>
#include <linux/amba/bus.h>
#include <linux/io.h>

#include <linux/mobi_mhif_device.h>
#include <mach/mobi_reset.h>
#include <mach/platform.h>
#include <mach/mhif_regs.h>

#define PFX "MHIF"

static void __iomem *mhif_base;

#define WRITE32(data, offset) \
	iowrite32((uint32_t) data, mhif_base+(uint32_t)offset)

#define READ32(offset) \
	ioread32(mhif_base+(uint32_t)offset)

#define WRITE16(data, offset) \
	iowrite16((uint16_t) data, mhif_base+(uint32_t)offset)

#define READ16(offset) \
	ioread16(mhif_base+(uint32_t)offset)

#if 0
#define dprintk(x...)   printk(x)
#else
#define dprintk(x...)
#endif

static irqreturn_t mhif_interrupt(int irq, void *dev_id)
{
	uint16_t got = READ16(MOBI_MHIF_INT_PEND_OFFSET);
	uint16_t mask = 0x0, dev = 0x0;

	if (got & MOBI_MHIF_INT_PEND_UM_MASK) {
		printk(KERN_WARNING PFX "Attempted access to unmapped device\n");
		WRITE16(MOBI_MHIF_INT_PEND_UM_MASK, MOBI_MHIF_INT_PEND_OFFSET);
	} else {
		if (got & MOBI_MHIF_INT_PEND_WTO_0_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_0_MASK;
			dev = 0;
		} else if (got & MOBI_MHIF_INT_PEND_WTO_1_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_1_MASK;
			dev = 1;
		} else if (got & MOBI_MHIF_INT_PEND_WTO_2_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_2_MASK;
			dev = 2;
		} else if (got & MOBI_MHIF_INT_PEND_WTO_3_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_3_MASK;
			dev = 3;
		} else if (got & MOBI_MHIF_INT_PEND_WTO_4_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_4_MASK;
			dev = 4;
		} else if (got & MOBI_MHIF_INT_PEND_WTO_5_MASK) {
			mask = MOBI_MHIF_INT_PEND_WTO_5_MASK;
			dev = 5;
		}

		/* clear the interrupt */
		WRITE16(mask, MOBI_MHIF_INT_PEND_OFFSET);
		printk(KERN_ERR PFX "Wait timeout for CS %d\n", dev);
		/*
		 * XXX would be useful if a device had a callback so we could send
		 * this error to it otherwise it is lost.  Need API to do that...
		 */
	}

	return IRQ_HANDLED;
}

static int config_mhif_devices(struct amba_device *pdev)
{
	int cs, i;
	int32_t addr_size, addr_mask;
	mobi_mhif_dev_configa_t configa;
	mobi_mhif_dev_configb_t configb;
	uint32_t ns_per_cycle;
	uint16_t intr_enable = 0x0, cs_enable = 0x0;
	struct mhif_device_t *mdev = NULL;
	struct mhif_driver_data_t *pdata = pdev->dev.platform_data;

	struct clk *mhif_clk = NULL;
	mhif_clk = clk_get(&pdev->dev, pdata->clk_name);
	ns_per_cycle = (1000000000/clk_get_rate(mhif_clk));

	dprintk("ns_per_cycle %d\n", ns_per_cycle);
	dprintk("number of devices %d\n", pdata->num_devices);
	for (i = 0; i < pdata->num_devices; i++) {
		mdev = &pdata->devices[i];
		cs = mdev->chip_select;

		dprintk("MHIF: Register device %s on CS %d\n", mdev->dev_name, cs);
		dprintk("MHIF: resource_start 0x%08x, resource_end 0x%08x\n",
				mdev->resource_start, mdev->resource_end);
		/* XXX hmmmm... I don't get it ??? */
		addr_size = mdev->resource_end - mdev->resource_start + 1;
		/* XXX probably need to check for max size */
		if (addr_size == 0)
			/* if size is 0 then set the correct mask for 0 */
			addr_mask = -1;
		else if (addr_size < 0x1000)
			/* if size is too small then round up to the resolution */
			addr_mask = -0x1000;
		else
			/* calc mask based on device size */
			addr_mask = -addr_size;

		dprintk("MHIF: address_size = 0x%08x, addr_mask 0x%08x\n",
				addr_size, addr_mask);

		/* setup DevConfigA for device i */
		configa.r = 0x0;
		/* only want bits 12-23 of AHB address */
		configa.b.addrbase =
			(mdev->resource_start >> MOBI_MHIF_DEV_CONFIGA_ADDRBASE_SIZE) &
			MOBI_MHIF_DEV_CONFIGA_ADDRBASE_MASK;
		configa.b.addrmask = MOBI_MHIF_DEV_CONFIGA_ADDRMASK_R(addr_mask);
		dprintk("MHIF config reg, addrbase 0x%08x, addrmask 0x%08x\n",
				configa.b.addrbase, configa.b.addrmask);

		configa.b.dwb = mdev->width;
		/* This can be done here but recommend using the bridge instead */
		configa.b.endianess   = mdev->endian;
		configa.b.busmux  = mdev->bus_mux;
		configa.b.dmawen  = mdev->dma_wen;
		configa.b.waiten  = mdev->wait_en;
		configa.b.addrinc = mdev->addr_inc;

		dprintk("CS %d, devconfigA = 0x%x\n", cs, configa.r);
		WRITE32(configa.r, MOBI_MHIF_DEV_CONFIGA_OFFSET+(cs*8));

/* if delay is larger than max, the only give max */
/* largest bit field is 7 bits, so just cast to 8 to get correct comparison */
#define FLD_MAX(fld) \
		((MOBI_MHIF_DEV_CONFIGB_ ## fld ## _MASK) >> \
		 (MOBI_MHIF_DEV_CONFIGB_ ## fld ## _SHIFT))
#define BITSAT(want, max) ((uint8_t)want > (uint8_t)max ? max : want);

		/* setup DevConfigB for device i */
		configb.r = 0x0;
		configb.b.cnt_as = BITSAT((mdev->ale_setup_ns/ns_per_cycle),
				FLD_MAX(CNT_AS));
		configb.b.cnt_aw = BITSAT((mdev->ale_width_ns/ns_per_cycle),
				FLD_MAX(CNT_AW));
		configb.b.cnt_ah = BITSAT((mdev->ale_hold_ns/ns_per_cycle),
				FLD_MAX(CNT_AS));
		configb.b.cnt_data = BITSAT((mdev->rd_wait_ns/ns_per_cycle),
				FLD_MAX(CNT_DATA));
		configb.b.cnt_wh = BITSAT((mdev->wr_hold_ns/ns_per_cycle),
				FLD_MAX(CNT_WH));
		configb.b.cnt_to = BITSAT((mdev->dev_timeout_ns/ns_per_cycle),
				FLD_MAX(CNT_TO));
		configb.b.cnt_idle = BITSAT((mdev->dev_idle_ns/ns_per_cycle),
				FLD_MAX(CNT_IDLE));
		configb.b.cnt_ps = BITSAT((mdev->prescaler),
				FLD_MAX(CNT_PS));
		configb.b.to_en = BITSAT((mdev->timeout_en),
				FLD_MAX(TO_EN));
		configb.b.wr_delay = BITSAT((mdev->wr_delay_ns/ns_per_cycle),
				FLD_MAX(WR_DELAY));
		configb.b.ale_en = BITSAT((mdev->ale_edge),
				FLD_MAX(ALE_EN));
		configb.b.cs_lvl = BITSAT((mdev->cs_lvl),
				FLD_MAX(CS_LVL));
		configb.b.wait_lvl = BITSAT((mdev->wait_lvl),
				FLD_MAX(WAIT_LVL));
		configb.b.dmarq_lvl = BITSAT((mdev->dmarq_lvl),
				FLD_MAX(DMARQ_LVL));
		configb.b.rd_hold = BITSAT((mdev->rd_hold_ns),
				FLD_MAX(RD_HOLD));

		dprintk("CS %d, devconfigB = 0x%08x\n", cs, configb.r);
		WRITE32(configb.r, MOBI_MHIF_DEV_CONFIGB_OFFSET+(cs*8));

		/*
		 *  *_0(for dev 0) is lowest bit to be set, just shift for
		 * each incoming CS
		 */
		intr_enable |= (MOBI_MHIF_INT_EN_SET_WTO_0_MASK << cs);
		cs_enable |= (MOBI_MHIF_CS_ENABLE_DEV_0_MASK << cs);
	}

	dprintk("intr_enable 0x%x, cs_enable 0x%x\n", intr_enable, cs_enable);
	/* enable intr for selected CS and UM */
	/* once again, should these really be turned on since currently no
	 * event is returned to a device ????
	 */
	WRITE16((intr_enable|MOBI_MHIF_INT_EN_SET_UM_MASK), MOBI_MHIF_INT_EN_SET_OFFSET);
	WRITE16(cs_enable, MOBI_MHIF_CS_ENABLE_OFFSET);

	return 0;
}

static int device_powerup(void)
{
	return mobi_reset_disable(RESET_ID_MHIF);
}

static void device_powerdown(void)
{
	mobi_reset_enable(RESET_ID_MHIF);
}

static int mhif_probe(struct amba_device *pdev, struct amba_id *id)
{
	int rc = 0;

	dprintk("in mhif_probe\n");
	rc = device_powerup();
	if (rc) {
		printk(KERN_ERR PFX "Could not take device out of reset\n");
		return -EIO;
	}

	rc = amba_request_regions(pdev, PFX);
	if (rc) {
		printk(KERN_ERR PFX "No IO memory region found, probe failed.");
		rc = -ENODEV;
		goto power_down;
	}

	dprintk("mhif_base start 0x%x, end 0x%x\n", pdev->res.start, pdev->res.end);
	mhif_base = ioremap(pdev->res.start, pdev->res.end-pdev->res.start+1);
	if (!mhif_base) {
		printk(KERN_ERR PFX "MOBI_MHIF failed to ioremap register space\n");
		rc = -EIO;
		goto err_ioremap;
	}

	/* disable interrupts */
	WRITE16(MOBI_MHIF_INT_EN_CLR_MASK, MOBI_MHIF_INT_EN_CLR_OFFSET);
	rc = request_irq(pdev->irq[0], mhif_interrupt, IRQF_DISABLED, PFX, NULL);
	if (rc) {
		printk(KERN_ERR PFX "Can't get IRQ for MOBI_MHIF\n");
		goto err_irq;
	}

	/*
	 * the MOBI_MHIF_WAIT timeout interrupt needs the MOBI_MHIF_CS_ENABLE register to
	 * tell it which cs's are actually in use for a real device.  We need to
	 * set the corresponding cs bits to 0 for unused cs's, and 1 for the ones
	 * in use.
	 */
	WRITE16(0x0, MOBI_MHIF_CS_ENABLE_OFFSET);

	/* OK good to go, parse the dev list */
	config_mhif_devices(pdev);

	return 0;

err_irq:
	iounmap(mhif_base);
	mhif_base = 0x0;
err_ioremap:
	amba_release_regions(pdev);
power_down:
	device_powerdown();

	return rc;
}

static int mhif_remove(struct amba_device *pdev)
{
	/* disable interrupts */
	WRITE16(MOBI_MHIF_INT_EN_CLR_MASK, MOBI_MHIF_INT_EN_CLR_OFFSET);
	free_irq(pdev->irq[0], NULL);

	if (mhif_base)
		iounmap(mhif_base);
	amba_release_regions(pdev);

	device_powerdown();
	return 0;
}

static void mhif_shutdown(struct amba_device *pdev)
{
	return;
}

static int mhif_suspend(struct amba_device *pdev, pm_message_t message)
{
	return 0;
}

static int mhif_resume(struct amba_device *pdev)
{
	return 0;
}

static struct amba_id mhif_ids[] = {
	{
		.id     = MHIF_AMBA_DEVID,
		.mask   = MHIF_AMBA_DEVID_MASK,
	},
	{ 0, 0 },
};

static struct amba_driver mhif_driver = {
	.probe    = mhif_probe,
	.remove   = mhif_remove,
	.shutdown = mhif_shutdown,
#ifdef CONFIG_PM
	.suspend  = mhif_suspend,
	.resume   = mhif_resume,
#else
	.suspend    = NULL,
	.resume     = NULL,
#endif
	.id_table   = mhif_ids,
	.drv    = {
		.name   = MHIF_AMBA_NAME,
	},
};

static int __init mhif_init(void)
{
	return(amba_driver_register(&mhif_driver));
}

static void __exit mhif_exit(void)
{
	amba_driver_unregister(&mhif_driver);
}
module_init(mhif_init);
module_exit(mhif_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Maxim-IC");
MODULE_DESCRIPTION("MG3500 master host interface driver");
