/*HEADER_START*/
/*
 *  Copyright (C) 2007 Mobilygen Corp., All Rights Reserved.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*HEADER_STOP*/
#define DRIVER_NAME 	"dwapbi2c"
#define DRIVER_DESC 	"Driver for the DW_apb_i2c component"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"3:2.2"

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/clk.h>

#include <linux/amba/bus.h>
#include <linux/i2c.h>

#include <asm/atomic.h>

#include <linux/i2c/dwapbi2c.h>

#define USE_MOBI_DMA

#ifdef USE_MOBI_DMA
#include <mach/mobi_dma.h>
#endif

int disp_err = 0;
module_param(disp_err, int, 0644);
MODULE_PARM_DESC(disp_err, "0=no errors printed, 1=errors with description");

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

struct dmahelper_data {
#ifdef USE_MOBI_DMA
	mobi_dma_handle			dmah;
#endif
	char 				write;
	dma_addr_t 			mem_addr;
	int				size;
};

struct dwapbi2c_drv_data {
	struct amba_device 		*dev;
	void __iomem 			*base;
	
	atomic_t			stop;
	struct mutex			access_lock;
	
	struct dwapbi2c_bus_data	*bdata;
	struct clk 			*clock;
	
	struct i2c_adapter 		adapter;
	
	wait_queue_head_t		wait_queue;
	atomic_t			completion_pending;
	struct i2c_msg 			*messages;
	int				msg_count, cur_msg_index;
	unsigned long			cur_buf_index, cur_rd_buf_index;
	u16				abort_reg, error_stat;
	
	char				master_code;
	u32				txfifo_depth, rxfifo_depth;
	char				has_dynamic_tar_update;
	char 				can_restart;
	unsigned long			transfer_timeout;
	char				i2c_speed;
	char				is_slave;
	u16				slave_addr;
	u8				sda_setup_delay;
	
	char				use_dma;
	atomic_t			tx_dma_error, rx_dma_error;
	struct dmahelper_data 		*dma_tx, *dma_rx;
	u8				*dma_tx_buf, *dma_rx_buf;
	int				dma_tx_buf_order, dma_rx_buf_order;
};

static atomic_t 			cur_mastercode;
static u8				smbus_dummy_byte;

/*-------------------------------------------------------------------------*/

static inline u32 apb_read32(void __iomem *base, int reg)
{
	return ioread32(base + reg);
}

static inline void apb_write32(void __iomem *base, int reg, u32 value)
{
	iowrite32(value, base + reg);
}

static inline u16 apb_read16(void __iomem *base, int reg)
{
	return ioread16(base + reg);
}

static inline void apb_write16(void __iomem *base, int reg, u16 value)
{
	iowrite16(value, base + reg);
}

static inline u8 apb_read8(void __iomem *base, int reg)
{
	return ioread8(base + reg);
}

static inline void apb_write8(void __iomem *base, int reg, u8 value)
{
	iowrite8(value, base + reg);
}

/* 
 * Checks if an I2C address is not reserved:
 * - 0000 000 0 General Call
 * - 0000 000 1 START byte - for slow micros without I2C h/w
 * - 0000 001 X CBUS address - a different bus protocol
 * - 0000 010 X Reserved for different bus format
 * - 0000 011 X Reserved for future purposes
 * - 0000 1XX X Hs-mode master code
 * - 1111 1XX X Reserved for future purposes
 * - 1111 0XX X 10-bit slave addressing
 */
static inline int is_reserved_address(u16 addr)
{
	/* Only minimum restriction. After all, one will see by himself if his bus
	 * setup does not allow an address. */
	return (addr == 0 || addr == 1 /*|| (addr & 0xf0) == 0
		|| (addr & 0xf0) == 0xf0*/);
}

static int wait_for_master_completion(struct dwapbi2c_drv_data *ddata)
{
	unsigned long comp_timeout_j;
	u8 status;
	status = apb_read8(ddata->base, DWAPBI2C_IC_STATUS);
	comp_timeout_j = jiffies + msecs_to_jiffies(ddata->transfer_timeout);
	while ((status & DWAPBI2C_STATUS_MST_ACTIVITY)
		|| !(status & DWAPBI2C_STATUS_TFE)) {
		if (unlikely(atomic_read(&ddata->stop)))
			return -ECANCELED;
		msleep(0);
		if (unlikely(!time_before_eq(jiffies, comp_timeout_j)))
			return -ETIMEDOUT;
		status = apb_read8(ddata->base, DWAPBI2C_IC_STATUS);
	}
	return 0;
}

static int wait_for_slave_completion(struct dwapbi2c_drv_data *ddata)
{
	unsigned long comp_timeout_j;
	u8 status;
	status = apb_read8(ddata->base, DWAPBI2C_IC_STATUS);
	comp_timeout_j = jiffies + msecs_to_jiffies(ddata->transfer_timeout);
	while ( status & DWAPBI2C_STATUS_SLV_ACTIVITY) {
		if (unlikely(atomic_read(&ddata->stop)))
			return -ECANCELED;
		msleep(0);
		if (unlikely(!time_before_eq(jiffies, comp_timeout_j)))
			return -ETIMEDOUT;
		status = apb_read8(ddata->base, DWAPBI2C_IC_STATUS);
	}
	return 0;
}

#define ulmin(a, b) min((unsigned long)(a), (unsigned long)(b))

static int dwapbi2c_write_max(struct dwapbi2c_drv_data *ddata, unsigned long count)
{
u32 reg;
struct i2c_msg *cur_msg;
u32 write_max = 0;

    while( ddata->cur_msg_index < ddata->msg_count) {
	cur_msg = &ddata->messages[ddata->cur_msg_index];

	if (unlikely(!cur_msg->len || !cur_msg->buf))
		return -EINVAL;
	write_max = ulmin(cur_msg->len, ddata->cur_buf_index
		+ ulmin(count, ddata->txfifo_depth - (reg = apb_read32(ddata->base, DWAPBI2C_IC_TXFLR))));

	/* if no room in FIFO, exit */
	if( !((ddata->txfifo_depth-1) - reg)) break;

	if (cur_msg->flags & I2C_M_RD) {
		for (; ddata->cur_buf_index < write_max; ddata->cur_buf_index++)
			apb_write16(ddata->base, DWAPBI2C_IC_DATA_CMD,
				DWAPBI2C_DATA_CMD_CMD_READ);
		/* end of data, disable TX and wait to RX finish */
		if( ((ddata->cur_msg_index + 1) >= ddata->msg_count) &&
		     (ddata->cur_buf_index      >= cur_msg->len)) {
			apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK,
				apb_read16(ddata->base, DWAPBI2C_IC_INTR_MASK) & ~DWAPBI2C_INTR_TX_EMPTY);
			return 0;
		}
	} else {
		for (; ddata->cur_buf_index < write_max; ddata->cur_buf_index++)
			apb_write16(ddata->base, DWAPBI2C_IC_DATA_CMD,
				DWAPBI2C_DATA_CMD_CMD_WRITE
				| (u16)cur_msg->buf[ddata->cur_buf_index]);
		/* end of data */
		if( ((ddata->cur_msg_index + 1) >= ddata->msg_count) &&
		     (ddata->cur_buf_index      >= cur_msg->len)) return -ENODATA;
	}
	if( ddata->cur_buf_index >= cur_msg->len) {
	    ddata->cur_msg_index++;
	    ddata->cur_buf_index = 0;
	}
    }
    return 0;
}

static int dwapbi2c_read_max(struct dwapbi2c_drv_data *ddata, unsigned long count)
{
	u8 b;
	struct i2c_msg *cur_msg = &ddata->messages[ddata->cur_msg_index];
	u32 read_max = ulmin(cur_msg->len, ddata->cur_rd_buf_index
		+ ulmin(count, apb_read32(ddata->base, DWAPBI2C_IC_RXFLR)));
	for (; ddata->cur_rd_buf_index < read_max; ddata->cur_rd_buf_index++) {
		b = apb_read16(ddata->base, DWAPBI2C_IC_DATA_CMD)
			& DWAPBI2C_DATA_CMD_DATA_MASK;
		if ((cur_msg->flags & I2C_M_RD) && cur_msg->len && cur_msg->buf)
			cur_msg->buf[ddata->cur_rd_buf_index] = b;
	}
	/* end of data */
	if( ((ddata->cur_msg_index + 1) >= ddata->msg_count) &&
	     (ddata->cur_rd_buf_index   >= cur_msg->len)) return -ENODATA;
	return 0;
}

static const char* dwapbi2c_get_abort_string(u16 abort_reg)
{
	if (abort_reg & 0x001)
		return "no ack received for this 7bit address (wrong addr.?)";
	else if (abort_reg & 0x0002)
		return "no ack received for the 1st byte of the 10bit addr. (wrong addr.?)";
	else if (abort_reg & 0x0004)
		return "no ack received for the 2nd byte of the 10bit addr. (wrong addr.?)";
	else if (abort_reg & 0x0008)
		return "data bytes were not ack'ed by the slave device";
	else if (abort_reg & 0x0010)
		return "no ack received for the general call";
	else if (abort_reg & 0x0020)
		return "general call error (see doc.)";
	else if (abort_reg & 0x0040)
		return "high speed master code was ack'ed and should not, try lowering the speed"; 
	else if (abort_reg & 0x0080)
		return "a start byte was ack'ed and should not";
	else if (abort_reg & 0x0100)
		return "restart must be enabled for high speed mode";
	else if (abort_reg & 0x0200)
		return "restart must be enabled to send a start byte";
	else if (abort_reg & 0x0400)
		return "restart must be enabled for reading in 10bit mode";
	else if (abort_reg & 0x0800)
		return "tried to initiate a master mode operation while in slave mode";
	else if (abort_reg & 0x1000)
		return "lost arbitration";
	else if (abort_reg & 0x2000)
		return "slave needs flushing the TX FIFO";
	else if (abort_reg & 0x4000)
		return "lost the bus while transmitting in slave mode";
	else if (abort_reg & 0x8000)
		return "invalid read command in slave mode";
	return "unknown reason";
}

#ifdef USE_MOBI_DMA
static void dwapbi2c_dma_event(mobi_dma_handle dmah, mobi_dma_event e, void *data)
{
	struct dwapbi2c_drv_data *ddata = data;
	if (e == MOBI_DMA_EVENT_TRANSFER_COMPLETE) {
		if (!atomic_dec_return(&ddata->completion_pending))
			ddata->cur_msg_index = ddata->msg_count;
		wake_up_interruptible(&ddata->wait_queue);
	} else if (e == MOBI_DMA_EVENT_TRANSFER_ERROR) {
		if (ddata->dma_tx && dmah == ddata->dma_tx->dmah)
			atomic_set(&ddata->tx_dma_error, 1);
		else
			atomic_set(&ddata->rx_dma_error, 1);
		atomic_set(&ddata->completion_pending, 0);
		wake_up_interruptible(&ddata->wait_queue);
	}
}
#endif

static struct dmahelper_data *dmahelper_setup(struct dwapbi2c_drv_data *ddata, const char *dma_id,
	int write, u8 *bytes, int size, dma_addr_t ip_addr, int data_is_16bits)
{
#ifdef USE_MOBI_DMA
	int ret, ip_flags, mem_flags;
	struct dmahelper_data *hdata = kzalloc(sizeof(struct dmahelper_data), GFP_KERNEL);
	if (unlikely(!hdata))
		return NULL;
	
	hdata->size = size;
	hdata->write = write;
	
	hdata->dmah = mobi_dma_request(dma_id, MOBI_DMA_O_NONE);
	if (unlikely(hdata->dmah < 0))
		goto err_request;

	ret = mobi_dma_setup_handler(hdata->dmah, dwapbi2c_dma_event, ddata);
	if (unlikely(ret))
		goto err_handler;
	
	ip_flags = mem_flags = 0;
	
	mem_flags |= MOBI_DMA_CONFIG_ADDRADJ_INC;
	mem_flags |= MOBI_DMA_CONFIG_BURST_SIZE_8;
	mem_flags |= MOBI_DMA_CONFIG_TRANSFER_WIDTH_32;
	
	ip_flags |= MOBI_DMA_CONFIG_ADDRADJ_NONE;
	ip_flags |= MOBI_DMA_CONFIG_BURST_SIZE_4;
	ip_flags |= (data_is_16bits ? MOBI_DMA_CONFIG_TRANSFER_WIDTH_16
		: MOBI_DMA_CONFIG_TRANSFER_WIDTH_8);

        if (unlikely(mobi_dma_config(hdata->dmah, DMA_CONFIG_DST, write ? ip_flags : mem_flags, NULL)
		|| mobi_dma_config(hdata->dmah, DMA_CONFIG_SRC, write ? mem_flags : ip_flags, NULL)
		|| mobi_dma_config(hdata->dmah, DMA_CONFIG_XFER, MOBI_DMA_CONFIG_DATA_WIDTH_8, NULL)))
		goto err_config;
	
	hdata->mem_addr = dma_map_single(NULL, (void *)bytes, size,
		write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
	if (unlikely(dma_mapping_error(NULL, hdata->mem_addr)))
		goto err_map;
	
	if (unlikely(mobi_dma_setup_single(hdata->dmah,
		(uint32_t)(write ? hdata->mem_addr : ip_addr),
		(uint32_t)(write ? ip_addr : hdata->mem_addr),
		size)))
		goto err_setup;
	
	return hdata;

err_setup:
	dma_unmap_single(NULL, hdata->mem_addr, hdata->size,
		hdata->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
err_map:
err_config:
err_handler:
    	mobi_dma_free(hdata->dmah);
err_request:
    	kfree(hdata);
#endif
	return NULL;
}

static inline int dmahelper_start(struct dmahelper_data *hdata)
{
#ifdef USE_MOBI_DMA
	return mobi_dma_enable(hdata->dmah);
#endif
}

static void dmahelper_free(struct dmahelper_data *hdata)
{
#ifdef USE_MOBI_DMA
	dma_unmap_single(NULL, hdata->mem_addr, hdata->size,
		hdata->write ? DMA_TO_DEVICE : DMA_FROM_DEVICE);
	mobi_dma_disable(hdata->dmah);
	mobi_dma_free(hdata->dmah);
	kfree(hdata);
#endif
}

static const char *wait_for_main_completion(struct dwapbi2c_drv_data *ddata, int is_last_wait)
{
	const char *stop_reason = NULL;
	int ret;
	
	if (ddata->transfer_timeout == DWAPBI2C_TRANSFER_TIMEOUT_INFINITE)
		ret = wait_event_interruptible(ddata->wait_queue,
			(atomic_read(&ddata->completion_pending) <= 0));
	else
		ret = wait_event_interruptible_timeout(ddata->wait_queue,
			(atomic_read(&ddata->completion_pending) <= 0),
			msecs_to_jiffies(ddata->transfer_timeout));
	
	apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK, DWAPBI2C_INTR_TX_ABRT);
	atomic_set(&ddata->completion_pending, 0);
	
	if (unlikely((ret <= 0 && ddata->transfer_timeout != DWAPBI2C_TRANSFER_TIMEOUT_INFINITE)
		|| (ddata->transfer_timeout == DWAPBI2C_TRANSFER_TIMEOUT_INFINITE && ret < 0))) {
		stop_reason = (ret == 0) ? "operation timed out" : "operation interrupted";
		if( ret == -ERESTARTSYS) {
			stop_reason = NULL;
		}
	}
	else if (unlikely(ddata->abort_reg))
		stop_reason = "operation aborted by component";
	else if (unlikely(ddata->error_stat & DWAPBI2C_INTR_TX_OVER))
		stop_reason = "TX FIFO overflow";
	else if (unlikely(ddata->error_stat & DWAPBI2C_INTR_RX_UNDER))
		stop_reason = "RX FIFO underflow";
	else if (unlikely(ddata->error_stat & DWAPBI2C_INTR_RX_OVER))
		stop_reason = "RX FIFO overflow";
	else if (unlikely(ddata->is_slave
		&& wait_for_slave_completion(ddata)))
		stop_reason = "slave completion timeout";
	else if (unlikely(is_last_wait && !ddata->is_slave
		&& wait_for_master_completion(ddata)))
		stop_reason = "master completion timeout";
	else if (unlikely(atomic_read(&ddata->tx_dma_error) && atomic_read(&ddata->rx_dma_error)))
		stop_reason = "RX and TX DMA failed";
	else if (unlikely(atomic_read(&ddata->tx_dma_error)))
		stop_reason = "TX DMA failed";
	else if (unlikely(atomic_read(&ddata->rx_dma_error)))
		stop_reason = "RX DMA failed";	
	
	return stop_reason;
}

static int dwapbi2c_access(struct i2c_adapter *adapter, struct i2c_msg *messages, int count)
{
	struct dwapbi2c_drv_data 		*ddata;
	struct i2c_msg 				*cur_msg = messages;
	int					needs_dynamic_tar_update = 0,
						i = 0, lcnt = 8, hcnt = 6, ret = 0;
	u16					sar = 0, tar = 0, intr_mask = 0;
	u8					con = 0;
	const char 				*stop_reason;
	
	if (unlikely(!count || !messages || !adapter->algo_data))
		return -EINVAL;

	ddata = amba_get_drvdata(((struct amba_device *)adapter->algo_data));
	if (unlikely(!ddata))
		return -EINVAL;
		
	if (unlikely(atomic_read(&ddata->stop)))
		return -ECANCELED;
	
	if (ddata->slave_addr)
		sar = ddata->slave_addr & DWAPBI2C_SAR_ADDRESS_MASK;
	else
#define GENERATE_SLAVE_ADDR(periphid) ((DWAPBI2C_DEFAULT_SLAVE_ADDR_BASE \
	+ ((periphid & 0xff) << 1)) & DWAPBI2C_SAR_ADDRESS_MASK)
		sar = GENERATE_SLAVE_ADDR(ddata->dev->periphid);
	
	if (unlikely(is_reserved_address(sar))) {
		dev_err(&ddata->dev->dev, "%s: cannot use reserved I2C "
			"address for component: 0x%x\n", __func__, sar);
		return -EINVAL;
	}
	sar >>= 1; /* Get rid of LSB */
	
	if (unlikely(mutex_lock_interruptible(&ddata->access_lock)))
		return -ECANCELED;

	ddata->messages = messages;
	ddata->msg_count = count;
	ddata->cur_msg_index = 0;
		
	for (i=0; i<count; i++) {
		if (unlikely(!messages[i].len
			|| !messages[i].buf)) {
			/*dev_err(&ddata->dev->dev, "%s: invalid I2C message: buf=0x%p, len=%d\n",
					__func__, messages[i].buf, messages[i].len);
			ret = -EINVAL;
			goto out;*/
			/* Workaround for SMBUS_QUICK command to work. */
			messages[i].len = 1;
			if (!messages[i].buf) {
				smbus_dummy_byte = 0;
				messages[i].buf = &smbus_dummy_byte;
			}
			messages[i].flags = I2C_M_RD;
		}
		if (ddata->is_slave)
			continue;
		if (i && ((messages[0].flags & I2C_M_TEN) != (messages[i].flags & I2C_M_TEN)
			|| messages[0].addr != messages[i].addr))
			needs_dynamic_tar_update = 1;
		/* Don't check for target address validity. 
		 * The component will complain itself it is has to. */
		/*if (needs_dynamic_tar_update || !i) {
			if (unlikely(is_reserved_address(messages[i].addr << 1))) {
				dev_err(&ddata->dev->dev, "%s: invalid (reserved) target I2C address: 0x%x\n",
					__func__, messages[i].addr << 1);
				ret = -EINVAL;
				goto out;
			}
		}*/
	}

	/* Disable dw_apb_i2c to start configuring it for the transfer */
	apb_write8(ddata->base, DWAPBI2C_IC_ENABLE, 0);
	
	ddata->dma_rx = NULL;
	ddata->dma_tx = NULL;
	atomic_set(&ddata->tx_dma_error, 0);
	atomic_set(&ddata->rx_dma_error, 0);
	ddata->abort_reg = 0;
	ddata->error_stat = 0;
	
	/* Assign slave address for this component */
	apb_write16(ddata->base, DWAPBI2C_IC_SAR, sar);
	
	if (ddata->is_slave) {
		con &= ~DWAPBI2C_CON_SADDRESSING_MASK;
		if (messages->flags & I2C_M_TEN)
			con |= DWAPBI2C_CON_SADDRESSING_10BIT;
		else
			con |= DWAPBI2C_CON_SADDRESSING_7BIT;
	} else {
		if (ddata->has_dynamic_tar_update == 2) {
			/* Detect if component supports dynamic TAR update (MASTER MODE) */
			u16 tar;
			ddata->has_dynamic_tar_update = 1;
			tar = apb_read16(ddata->base, DWAPBI2C_IC_TAR);
			apb_write16(ddata->base, DWAPBI2C_IC_TAR, tar & ~DWAPBI2C_TAR_MADDRESSING_MASK);
			con = apb_read8(ddata->base, DWAPBI2C_IC_CON);
			apb_write8(ddata->base, DWAPBI2C_IC_CON, con | DWAPBI2C_CON_MADDRESSING_MASK);
			con = apb_read8(ddata->base, DWAPBI2C_IC_CON);
			if (con & DWAPBI2C_CON_MADDRESSING_MASK)
				ddata->has_dynamic_tar_update = 0;
		}
		
		con = DWAPBI2C_CON_SLAVE_MODE_DISABLED;
		con |= DWAPBI2C_CON_MASTER_MODE_ENABLED;
		if (ddata->can_restart)
			con |= DWAPBI2C_CON_RESTART_ENABLED;
	
		if (!ddata->has_dynamic_tar_update) {
			if (needs_dynamic_tar_update) {
				dev_err(&ddata->dev->dev, "%s: changing target address during "
					"transfer is not supported\n", __func__);
				ret = -EPERM;
				goto out;
			} else {
				con &= ~DWAPBI2C_CON_MADDRESSING_MASK;
				if (messages->flags & I2C_M_TEN)
					con |= DWAPBI2C_CON_MADDRESSING_10BIT;
				else
					con |= DWAPBI2C_CON_MADDRESSING_7BIT;
			}
		}
	}
	
#define HZ_TO_NS4(hz) (4000000000UL/(unsigned long)(hz))
#define GET_CLKCNT(d) (1UL + ((unsigned long)(d) * 4UL - 1UL) \
	/ HZ_TO_NS4(clk_get_rate(ddata->clock)))

	/* Speed settings */
	con &= ~DWAPBI2C_CON_SPEED_MASK;
	switch (ddata->i2c_speed) {
	
	case DWAPBI2C_CON_SPEED_FAST: /* 400Kbit/s */
		con |= DWAPBI2C_CON_SPEED_FAST;
		apb_write8(ddata->base, DWAPBI2C_IC_CON, con);
		
		hcnt = (int)GET_CLKCNT(DWAPBI2C_FS_SCL_HIGH_MIN_NS) - 8;
		lcnt = (int)GET_CLKCNT(DWAPBI2C_FS_SCL_LOW_MIN_NS) - 1;
		
		if (hcnt < DWAPBI2C_IC_FS_SCL_HCNT_MIN)
			hcnt = DWAPBI2C_IC_FS_SCL_HCNT_MIN;
		if (lcnt < DWAPBI2C_IC_FS_SCL_LCNT_MIN)
			lcnt = DWAPBI2C_IC_FS_SCL_LCNT_MIN;
		
		debug("fs_scl_hcnt=%d", hcnt);
		debug("fs_scl_lcnt=%d", lcnt);
		
		apb_write16(ddata->base, DWAPBI2C_IC_FS_SCL_HCNT, hcnt);
		apb_write16(ddata->base, DWAPBI2C_IC_FS_SCL_LCNT, lcnt);
		
		break;
		
	case DWAPBI2C_CON_SPEED_HIGH: /* 3.4Mbit/s*/
		con |= DWAPBI2C_CON_SPEED_HIGH;
		apb_write8(ddata->base, DWAPBI2C_IC_CON, con);
		
		hcnt = (int)GET_CLKCNT(DWAPBI2C_HS_SCL_HIGH_MIN_NS) - 8;
		lcnt = (int)GET_CLKCNT(DWAPBI2C_HS_SCL_LOW_MIN_NS) - 1;
		
		if (hcnt < DWAPBI2C_IC_HS_SCL_HCNT_MIN)
			hcnt = DWAPBI2C_IC_HS_SCL_HCNT_MIN;
		if (lcnt < DWAPBI2C_IC_HS_SCL_LCNT_MIN)
			lcnt = DWAPBI2C_IC_HS_SCL_LCNT_MIN;
		
		apb_write16(ddata->base, DWAPBI2C_IC_HS_SCL_HCNT, hcnt);
		apb_write16(ddata->base, DWAPBI2C_IC_HS_SCL_LCNT, lcnt);
		
		debug("hs_scl_hcnt=%d", hcnt);
		debug("hs_scl_lcnt=%d", lcnt);
		
		/*
		 * For high speed transfer we need to specify a master code,
		 * which value must be between 0 and 7.
		 */
		apb_write16(ddata->base, DWAPBI2C_IC_HS_MADDR, ddata->master_code
			& DWAPBI2C_HS_MASTERCODE_MASK);
		
		break;
	
	case DWAPBI2C_CON_SPEED_LOW: /* 10Kbit/s */
	case DWAPBI2C_CON_SPEED_STANDARD: /* 100Kbit/s */
	default:
		con |= DWAPBI2C_CON_SPEED_STANDARD;
		apb_write8(ddata->base, DWAPBI2C_IC_CON, con);
		
		hcnt = (int)GET_CLKCNT(DWAPBI2C_SS_SCL_HIGH_MIN_NS) - 8;
		lcnt = (int)GET_CLKCNT(DWAPBI2C_SS_SCL_LOW_MIN_NS) - 1;
		
		if (ddata->i2c_speed == DWAPBI2C_CON_SPEED_LOW) {
			hcnt *= 10;
			lcnt *= 10;
		}
		
		if (hcnt < DWAPBI2C_IC_SS_SCL_HCNT_MIN)
			hcnt = DWAPBI2C_IC_SS_SCL_HCNT_MIN;
		if (lcnt < DWAPBI2C_IC_SS_SCL_LCNT_MIN)
			lcnt = DWAPBI2C_IC_SS_SCL_LCNT_MIN;
		
		debug("ss_scl_hcnt=%d", hcnt);
		debug("ss_scl_lcnt=%d", lcnt);
		
		apb_write16(ddata->base, DWAPBI2C_IC_SS_SCL_HCNT, hcnt);
		apb_write16(ddata->base, DWAPBI2C_IC_SS_SCL_LCNT, lcnt);
		
		break;
	}
	
	apb_write8(ddata->base, DWAPBI2C_IC_SDA_SETUP, ddata->sda_setup_delay);
	
	if (!ddata->txfifo_depth || !ddata->rxfifo_depth) {
		u32 comp = apb_read32(ddata->base, DWAPBI2C_IC_COMP_PARAM_1);
		ddata->txfifo_depth = 1 + ((comp & DWAPBI2C_COMP_PARAM_1_TXDEPTH_MASK)
			>> DWAPBI2C_COMP_PARAM_1_TXDEPTH_LSB);
		ddata->rxfifo_depth = 1 + ((comp & DWAPBI2C_COMP_PARAM_1_RXDEPTH_MASK)
			>> DWAPBI2C_COMP_PARAM_1_RXDEPTH_LSB);
	}
	
	if (ddata->use_dma && !needs_dynamic_tar_update) {
		int total_len_tx = 0, total_len_rx = 0;
		
		for (i=0; i<count; i++) {
			if (messages[i].flags & I2C_M_RD) {
				total_len_rx += messages[i].len;
				if (!ddata->is_slave)
					total_len_tx += messages[i].len;
			} else {
				total_len_tx += messages[i].len;
			}
		}
		
		if (total_len_tx) {
			int this_order_tx;
			
			if (!ddata->is_slave)
				total_len_tx <<= 1;
			
			this_order_tx = get_order(total_len_tx);
			
			if (ddata->dma_tx_buf && this_order_tx > ddata->dma_tx_buf_order) {
				free_pages((unsigned long)ddata->dma_tx_buf, ddata->dma_tx_buf_order);
				ddata->dma_tx_buf = NULL;
			}
			if (!ddata->dma_tx_buf) {
				ddata->dma_tx_buf = (u8 *)__get_dma_pages(GFP_KERNEL, this_order_tx);
				ddata->dma_tx_buf_order = ddata->dma_tx_buf ? this_order_tx : 0;
			}
			if (ddata->dma_tx_buf) {
				int j, k = 0;
				for (i=0; i<count; i++) {
					for (j=0; j<messages[i].len; j++) {
						if (messages[i].flags & I2C_M_RD) {
							if (!ddata->is_slave)
								((u16 *)ddata->dma_tx_buf)[k++] = (u16)messages[i].buf[j]
									| DWAPBI2C_DATA_CMD_CMD_READ;
						} else {
							((u16 *)ddata->dma_tx_buf)[k++] = (u16)messages[i].buf[j]
									| DWAPBI2C_DATA_CMD_CMD_WRITE;
						}
					}
				}
				ddata->dma_tx = dmahelper_setup(ddata, DRIVER_NAME "_tx", 1,
					ddata->dma_tx_buf, total_len_tx,
					ddata->dev->res.start + DWAPBI2C_IC_DATA_CMD,
					(ddata->is_slave == 0)); 
			}
		}
		
		if (total_len_rx && (ddata->dma_tx || !total_len_tx)) {
			int this_order_rx = get_order(total_len_rx);
			
			if (ddata->dma_rx_buf && this_order_rx > ddata->dma_rx_buf_order) {
				free_pages((unsigned long)ddata->dma_rx_buf, ddata->dma_rx_buf_order);
				ddata->dma_rx_buf = NULL;
			}
			if (!ddata->dma_rx_buf) {
				ddata->dma_rx_buf = (u8 *)__get_dma_pages(GFP_KERNEL, this_order_rx);
				ddata->dma_rx_buf_order = ddata->dma_rx_buf ? this_order_rx : 0;
			}
			if (ddata->dma_rx_buf) {
				memset(ddata->dma_rx_buf, ~0, total_len_rx);
				ddata->dma_rx = dmahelper_setup(ddata, DRIVER_NAME "_rx", 0,
					ddata->dma_rx_buf, total_len_rx,
					ddata->dev->res.start + DWAPBI2C_IC_DATA_CMD,
					0);
			}
			if (!ddata->dma_rx && ddata->dma_tx) {
				dmahelper_free(ddata->dma_tx);
				ddata->dma_tx = NULL;
			}
		}
	}
	
	if (!ddata->is_slave) {
		tar = 0;
		if (ddata->has_dynamic_tar_update) {
			if (messages->flags & I2C_M_TEN)
				tar |= DWAPBI2C_TAR_MADDRESSING_10BIT;
			else
				tar |= DWAPBI2C_TAR_MADDRESSING_7BIT;
		}
		tar |= (messages->addr & DWAPBI2C_TAR_ADDRESS_MASK);
		apb_write16(ddata->base, DWAPBI2C_IC_TAR, tar);
	}
	
	if (ddata->dma_tx || ddata->dma_rx) {
		int j = 0;
		
		apb_write8(ddata->base, DWAPBI2C_IC_DMA_CR, (ddata->dma_tx ? DWAPBI2C_DMA_CR_TDMAE : 0)
			| (ddata->dma_rx ? DWAPBI2C_DMA_CR_RDMAE : 0));
		apb_write8(ddata->base, DWAPBI2C_IC_DMA_TDLR, ddata->txfifo_depth - 5);
		apb_write8(ddata->base, DWAPBI2C_IC_DMA_RDLR, 3);
		
		intr_mask = DWAPBI2C_INTR_TX_ABRT | DWAPBI2C_INTR_TX_OVER
			| DWAPBI2C_INTR_RX_UNDER | DWAPBI2C_INTR_RX_OVER;
		apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK, intr_mask);
		
		atomic_set(&ddata->completion_pending, 0);
		if (ddata->dma_tx)
			atomic_inc(&ddata->completion_pending);
		if (ddata->dma_rx)
			atomic_inc(&ddata->completion_pending);
		
		apb_write8(ddata->base, DWAPBI2C_IC_ENABLE, 1);
		
		/* You have to start DMA *after* component is enabled. */
		if (ddata->dma_tx)
			dmahelper_start(ddata->dma_tx);
		if (ddata->dma_rx)
			dmahelper_start(ddata->dma_rx);
		
		stop_reason = wait_for_main_completion(ddata, 1);
		
		if (ddata->dma_tx)
			dmahelper_free(ddata->dma_tx);
		if (ddata->dma_rx) {
			dmahelper_free(ddata->dma_rx);
			for (i=0; i<count; i++) {
				if (messages[i].flags & I2C_M_RD) {
					memcpy(messages[i].buf, ddata->dma_rx_buf + j,
						messages[i].len);
					j += messages[i].len;
				}
			}
		}
		
		if (unlikely(stop_reason))
			goto err_stop;
		else /* Success. */
			goto out;
			
	} else if (ddata->use_dma) {
		dev_err(&ddata->dev->dev, "%s: could not setup DMA transfer, "
			"aborting\n", __func__);
		ret = -EFAULT;
		goto out;
	}

	apb_write8(ddata->base, DWAPBI2C_IC_DMA_CR, 0);

	/* increase it to 3/4 of fifo depth */
	apb_write8(ddata->base, DWAPBI2C_IC_TX_TL, (ddata->txfifo_depth*3)/4);

	{
		ddata->cur_msg_index = 0;
		cur_msg = &messages[ddata->cur_msg_index];
		
		if (!ddata->is_slave && ddata->has_dynamic_tar_update) {
			u16 newtar = tar;
			newtar &= ~DWAPBI2C_TAR_MADDRESSING_MASK;
			if (cur_msg->flags & I2C_M_TEN)
				newtar |= DWAPBI2C_TAR_MADDRESSING_10BIT;
			else
				newtar |= DWAPBI2C_TAR_MADDRESSING_7BIT;
			newtar &= ~DWAPBI2C_TAR_ADDRESS_MASK;
			newtar |= cur_msg->addr & DWAPBI2C_TAR_ADDRESS_MASK;
			if (newtar != tar) {
				if (unlikely(wait_for_master_completion(ddata))) {
					stop_reason = "master completion timeout";
					ret = -ETIME;
					goto err_stop;
				}
				apb_write16(ddata->base, DWAPBI2C_IC_TAR, newtar);
				tar = newtar;
			}
		}
		
		atomic_set(&ddata->completion_pending, 1);
		ddata->cur_buf_index = 0;
		ddata->cur_rd_buf_index = 0;
		ddata->abort_reg = 0;
		ddata->error_stat = 0;
		
		apb_read8(ddata->base, DWAPBI2C_IC_CLR_INTR);
		
		intr_mask = DWAPBI2C_INTR_TX_ABRT | DWAPBI2C_INTR_TX_OVER
			| DWAPBI2C_INTR_RX_UNDER | DWAPBI2C_INTR_RX_OVER;
		if (ddata->is_slave) {
			if (cur_msg->flags & I2C_M_RD)
				intr_mask |= DWAPBI2C_INTR_RX_FULL;
			intr_mask |= DWAPBI2C_INTR_RD_REQ;
		} else {
			intr_mask |= DWAPBI2C_INTR_TX_EMPTY | DWAPBI2C_INTR_RX_FULL;
		}
		apb_write8(ddata->base, DWAPBI2C_IC_RX_TL, 0);

		/* Start transfer */
		apb_write8(ddata->base, DWAPBI2C_IC_ENABLE, 1);
		apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK, intr_mask);
		
		/* Wait for its completion */
		if (unlikely(stop_reason = wait_for_main_completion(ddata, 1)))
			goto err_stop;
	}
	
	goto out;
	
err_stop:
	cur_msg = &messages[ddata->cur_msg_index];
	if( disp_err) {
	  if (ddata->dma_tx || ddata->dma_rx) {
		dev_err(&ddata->dev->dev, "%s: %s, address: 0x%x, TX/RX DMA: %s/%s\n",
			ddata->is_slave ? "slave" : "master",
			stop_reason, ddata->is_slave ? (sar << 1) : (messages->addr << 1),
			ddata->dma_tx ? "yes" : "no", ddata->dma_rx ? "yes" : "no");
		dev_err(&ddata->dev->dev, "disable DMA to get extended error information\n");
	  } else {
		u16 last_addr;
		if (cur_msg->flags & I2C_M_RD)
			last_addr = ddata->is_slave ? (sar << 1) : ((cur_msg->addr << 1) + 1);
		else
			last_addr = ddata->is_slave ? ((sar << 1) + 1) : (cur_msg->addr << 1);
  		dev_err(&ddata->dev->dev, "%s: %s after %d/%d messages, bytes sent/read/total: %lu/%lu/%d, "
			"op/address: %s/0x%x\n", ddata->is_slave ? "slave" : "master", stop_reason,
			ddata->cur_msg_index, count, ddata->cur_buf_index, ddata->cur_rd_buf_index,
			cur_msg->len, (cur_msg->flags & I2C_M_RD) ? "receive" : "transmit", last_addr);
	  }
	
	  if (ddata->abort_reg)
		dev_err(&ddata->dev->dev, "reason: %s (0x%x)\n",
			dwapbi2c_get_abort_string(ddata->abort_reg), ddata->abort_reg);
	
	  dev_err(&ddata->dev->dev, "status register: 0x%x, interrupt status register: 0x%x, intr. mask: 0x%x\n",
		apb_read8(ddata->base, DWAPBI2C_IC_STATUS),
		apb_read16(ddata->base, DWAPBI2C_IC_RAW_INTR_STAT),
		intr_mask);
	  dev_err(&ddata->dev->dev, "you can try changing bus settings with /sys/devices/%s/config\n",
		dev_name(&ddata->dev->dev));
	}
	
	if (!ret) {
		/* Try to be a little more specific in case no ack was received. */
		if (ddata->abort_reg & 0x0007)
			ret = -ENXIO;
		else
			ret = -EIO;
	}
	
out:
	/* Disable component */
	apb_write8(ddata->base, DWAPBI2C_IC_ENABLE, 0);
	
	mutex_unlock(&ddata->access_lock);
	
	if (!ddata->dma_tx) {
		for (i = 0; i < count; i++) {
			if (messages[i].flags & I2C_M_RD) {
				/* Mark failed reads with 0xff. */
				if (i == ddata->cur_msg_index)
					memset(messages[i].buf + ddata->cur_rd_buf_index, ~0,
						messages[i].len - ddata->cur_rd_buf_index);
				else if (i > ddata->cur_msg_index)
					memset(messages[i].buf, ~0, messages[i].len);
				
				/* Prevent people from accessing this dummy buffer. */
				if (messages[i].buf == &smbus_dummy_byte)
					messages[i].buf = NULL;
			}
			
		}
	}
	
	return ret ? ret : ddata->cur_msg_index;
}

#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
static irqreturn_t dwapbi2c_interrupt(int irq, void *dev_id, struct pt_regs *regs)
#else
static irqreturn_t dwapbi2c_interrupt(int irq, void *dev_id)
#endif
{
	struct dwapbi2c_drv_data 		*ddata = dev_id;
	u16					stat;
	int					read;

	if (unlikely(!ddata))
		return IRQ_NONE;
	
	if (unlikely(!atomic_read(&ddata->completion_pending)
		|| atomic_read(&ddata->stop)))
		goto wakeup;
	
	stat = apb_read16(ddata->base, DWAPBI2C_IC_INTR_STAT);
	read = (ddata->messages[ddata->cur_msg_index].flags & I2C_M_RD);
	
	if (unlikely((stat & DWAPBI2C_INTR_RX_OVER)
		|| (stat & DWAPBI2C_INTR_RX_UNDER)
		|| (stat & DWAPBI2C_INTR_TX_OVER))) {
		ddata->error_stat = stat;
		goto wakeup;
	} else if (stat & DWAPBI2C_INTR_TX_ABRT) {
		ddata->abort_reg = apb_read32(ddata->base, DWAPBI2C_IC_TX_ABRT_SOURCE);
		if (ddata->abort_reg & DWAPBI2C_TXABRT_SLVFLUSH) {
			ddata->abort_reg = 0;
			apb_read8(ddata->base, DWAPBI2C_IC_CLR_TX_ABRT);
		} else {
			goto wakeup;
		}
	}
	
	if (stat & DWAPBI2C_INTR_RD_REQ) {
		apb_read8(ddata->base, DWAPBI2C_IC_CLR_RD_REQ);
		apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK,
			apb_read16(ddata->base, DWAPBI2C_IC_INTR_MASK) | DWAPBI2C_INTR_TX_EMPTY);
	}
	
	if (stat & DWAPBI2C_INTR_RX_FULL) {
		if( dwapbi2c_read_max (ddata, ~0) == -ENODATA) goto wakeup;
	}
	if ((stat & DWAPBI2C_INTR_TX_EMPTY)
		|| (stat & DWAPBI2C_INTR_RD_REQ)) {
		if( dwapbi2c_write_max(ddata, ~0) == -ENODATA) goto wakeup;
	}

	return IRQ_HANDLED;
	
wakeup:
	apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK, 0);
	atomic_set(&ddata->completion_pending, 0);
	wake_up_interruptible(&ddata->wait_queue);
	return IRQ_HANDLED;
}

static u32 dwapbi2c_function(struct i2c_adapter *adapter)
{
	return I2C_FUNC_I2C | I2C_FUNC_10BIT_ADDR 
		| I2C_FUNC_SMBUS_EMUL;
}

static struct i2c_algorithm dwapbi2c_algorithm = {
	.master_xfer	= dwapbi2c_access,
	.functionality	= dwapbi2c_function,
};

/*-------------------------------------------------------------------------*/

static ssize_t dev_attr_show_config(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret = 0;
	struct dwapbi2c_drv_data *ddata = dev_get_drvdata(dev);
	const char *s;
	u32 sar;
	mutex_lock(&ddata->access_lock);
	ret += snprintf(buf + ret, PAGE_SIZE - ret, DRIVER_NAME " " DRIVER_VERSION ": %s Bus Configuration\n\n",
		dev_name(dev));
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (r) Input Clock Rate:       %luHz\n",
		clk_get_rate(ddata->clock));
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (M) Master/Slave Mode:      %s\n",
		ddata->is_slave ? "SLAVE" : "MASTER");
	switch (ddata->i2c_speed) {
		default:
		case DWAPBI2C_CON_SPEED_LOW: s = "LOW (10kbit/s)"; break;
		case DWAPBI2C_CON_SPEED_STANDARD: s = "STANDARD (100kbit/s)"; break;
		case DWAPBI2C_CON_SPEED_FAST: s = "FAST (400kbit/s)"; break;
		case DWAPBI2C_CON_SPEED_HIGH: s = "HIGH SPEED (3400kbit/s)"; break;
	}
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (S) I2C Speed:              %s\n", s);
	if (ddata->slave_addr)
		sar = ddata->slave_addr & DWAPBI2C_SAR_ADDRESS_MASK;
	else
		sar = GENERATE_SLAVE_ADDR(ddata->dev->periphid);
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (A) Slave Address:          0x%x (transmitting on 0x%x)\n",
		(sar & ~1), sar | 1);
	if (ddata->transfer_timeout == DWAPBI2C_TRANSFER_TIMEOUT_INFINITE)
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (T) Transfer Timeout:       INFINITE\n");
	else
		ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (T) Transfer Timeout:       %lumsecs\n",
			ddata->transfer_timeout);
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (R) Master Restart Feature: %s\n",
		ddata->can_restart ? "Enabled" : "Disabled");
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (D) Try Use DMA:            %s\n",
		ddata->use_dma ? "YES" : "NO");
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "  (t) SDA Setup Delay:        %d periods (%luns)\n",
		ddata->sda_setup_delay, (unsigned long)ddata->sda_setup_delay * 1000UL 
		/ (clk_get_rate(ddata->clock) / 1000000UL));
	ret += snprintf(buf + ret, PAGE_SIZE - ret, "\necho '<letter> <value>' in this "
		"file to change value.\n");
	mutex_unlock(&ddata->access_lock);
	return ret;
}

static ssize_t dev_attr_store_config(struct device *dev, struct device_attribute *attr, 
	const char *buf, size_t count)
{
	struct dwapbi2c_drv_data *ddata = dev_get_drvdata(dev);
	char c = 0;
	char v[256];
	mutex_lock(&ddata->access_lock);
	if (sscanf(buf, "%c %256s", &c, v) == 2) {
		switch (c) {
		case 'D': {
			int i;
			if (sscanf(v, "%i", &i) != 1)
				goto err_input;
			else
				ddata->use_dma = (i != 0);
			break;
		}
		case 'M':
			if (!strcmp(v, "MASTER") || !strcmp(v, "M"))
				ddata->is_slave = 0;
			else if (!strcmp(v, "SLAVE") || !strcmp(v, "S"))
				ddata->is_slave = 1;
			else
				dev_err(&ddata->dev->dev, "%s: invalid input: value must be "
					"either 'M[ASTER]' or 'S[LAVE]'\n", __func__);
			break;
		case 'S':
			if (!strcmp(v, "LOW") || !strcmp(v, "L"))
				ddata->i2c_speed = DWAPBI2C_CON_SPEED_LOW;
			else if (!strcmp(v, "STANDARD") || !strcmp(v, "S"))
				ddata->i2c_speed = DWAPBI2C_CON_SPEED_STANDARD;
			else if (!strcmp(v, "FAST") || !strcmp(v, "F"))
				ddata->i2c_speed = DWAPBI2C_CON_SPEED_FAST;
			else if (!strcmp(v, "HIGH") || !strcmp(v, "H"))
				ddata->i2c_speed = DWAPBI2C_CON_SPEED_HIGH;
			else
				dev_err(&ddata->dev->dev, "%s: invalid input: value must be either 'L[OW]', "
					"'S[TANDARD]', 'F[AST]' or 'H[IGH]'\n", __func__);
			break;
		case 'A': {
			u32 addr = 0;
			if (sscanf(v, "0x%x", &addr) == 1 || sscanf(v, "%x", &addr) == 1) {
				if (is_reserved_address(addr))
					dev_err(&ddata->dev->dev, "%s: invalid address: 0x%x\n", __func__, addr);
				else
					ddata->slave_addr = (addr & ~1);
			} else {
				dev_err(&ddata->dev->dev, "%s: invalid address\n", __func__);
			}
		}	break;
		case 'T': {
			char b[16] = { 0 };
			if (sscanf(v, "%3s", b) == 1 && (!strcmp(b, "inf") || !strcmp(b, "INF"))) {
				ddata->transfer_timeout = DWAPBI2C_TRANSFER_TIMEOUT_INFINITE;
			} else {
				unsigned long t = 0;
				sscanf(v, "%lu", &t);
				if (t >= 1)
					ddata->transfer_timeout = t;
				else
					dev_err(&ddata->dev->dev, "%s: invalid timeout value\n", __func__);
			}
		}	break;
		case 't': {
			unsigned long t = 0;
			if (sscanf(v, "%lu", &t) == 1 && t <= 255)
				ddata->sda_setup_delay = t & 0xff;
			else
				dev_err(&ddata->dev->dev, "%s: invalid SDA setup delay, it is expressed "
					"in clock periods and must be between 0 and 255\n", __func__);
		}	break;
		case 'R': {
			unsigned long e = 1;
			if (sscanf(v, "%lu", &e) == 1)
				ddata->can_restart = e ? 1 : 0;
			else
				dev_err(&ddata->dev->dev, "%s: invalid restart feature enable/disable "
					"command. echo 'R 0' or 'R 1'\n", __func__);
		}	break;
		case 'r': {
			unsigned long r = 12000000;
			if (sscanf(v, "%lu", &r) != 1 || r < 1000000 || clk_set_rate(ddata->clock, r))
				dev_err(&ddata->dev->dev, "%s: could not set "
					"clock rate to %luHz\n", __func__, r);
		}	break;
		default:
			goto err_input;
		}
		goto out;
	}
err_input:
	dev_err(&ddata->dev->dev, "%s: invalid input\n", __func__);
out:
  	mutex_unlock(&ddata->access_lock);
	return count;
}

static DEVICE_ATTR(config, 0644, dev_attr_show_config, dev_attr_store_config);

/*-------------------------------------------------------------------------*/

int dwapbi2c_configure(const char *bus_id, int i2c_speed, int is_slave, 
	u16 slave_addr, unsigned long transfer_timeout, int use_dma,
	u8 sda_setup_delay)
{
	struct amba_device *dev;
	if (unlikely(!bus_id || !*bus_id))
		return -EINVAL;
	dev = amba_find_device(bus_id, NULL, 0, 0);
	if (dev) {
		struct dwapbi2c_drv_data *ddata = amba_get_drvdata(dev);
		mutex_lock(&ddata->access_lock);
		ddata->is_slave = (is_slave != 0);
		if (i2c_speed)
			ddata->i2c_speed = i2c_speed;
		if (!is_reserved_address(slave_addr))
			ddata->slave_addr = slave_addr;
		if (transfer_timeout)
			ddata->transfer_timeout = transfer_timeout;
		if (use_dma >= 0)
			ddata->use_dma = (use_dma != 0);
		if (sda_setup_delay > 0)
			ddata->sda_setup_delay = sda_setup_delay;
		mutex_unlock(&ddata->access_lock);
		return 0;
	}
	return -ENODEV;
}
EXPORT_SYMBOL(dwapbi2c_configure);

/*-------------------------------------------------------------------------*/

static int dwapbi2c_probe(struct amba_device *dev, struct amba_id *id)
{
	int 					ret, i;
	struct dwapbi2c_drv_data		*ddata;
	struct dwapbi2c_bus_data	 	*bdata = dev->dev.platform_data;
	
	if (unlikely(!bdata))
		return -EINVAL;
	
	ddata = kzalloc(sizeof(struct dwapbi2c_drv_data), GFP_KERNEL);
	if (unlikely(!ddata)) {
		ret = -ENOMEM;
		goto err_alloc;
	}
	ddata->dev = dev;
	ddata->bdata = bdata;
	
	amba_set_drvdata(dev, ddata);
	
	ddata->base = ioremap(dev->res.start, dev->res.end - dev->res.start + 1);
	if (unlikely(!ddata->base)) {
		dev_err(&dev->dev, "%s: unable to remap memory (address=0x%08x, size=%d)\n",
			__func__, dev->res.start, dev->res.end - dev->res.start + 1);
		ret = -EBUSY;
		goto err_ioremap;
	}
	
	/* Disable component & interrupts */
	apb_write8(ddata->base, DWAPBI2C_IC_ENABLE, 0);
	apb_write16(ddata->base, DWAPBI2C_IC_INTR_MASK, 0);
	
	for (i=0; i<AMBA_NR_IRQS; i++) { 
		if (dev->irq[i] != NO_IRQ) {
			if (unlikely(ret = request_irq(dev->irq[i], dwapbi2c_interrupt,
				IRQF_DISABLED, DRIVER_NAME, (void*)ddata))) {
				dev_err(&dev->dev, "%s: could not get shared interrupt "
					"%d, error is %d\n", __func__, dev->irq[i], ret);
				goto err_irq;
			}
		}
	}
	
	ddata->clock = clk_get(&dev->dev, ddata->bdata->clock_id);
	if (unlikely(IS_ERR(ddata->clock))) {
		ret = PTR_ERR(ddata->clock);
		dev_err(&dev->dev, "%s: failed to get clock id=%s, "
			"error is %d\n", __func__, 
			bdata->clock_id, ret);
		goto err_clkget;
	}
	
	/* ddata initialization */
	sprintf(ddata->adapter.name, DRIVER_NAME ".%s", dev_name(&dev->dev));
	ddata->adapter.id = I2C_HW_DWAPBI2C;
	ddata->adapter.algo = &dwapbi2c_algorithm;
	ddata->adapter.algo_data = (void*)dev;
	ddata->adapter.class = I2C_CLASS_HWMON | I2C_CLASS_SPD;
	ddata->adapter.dev.parent = &dev->dev;
	ddata->has_dynamic_tar_update = 2; /* means undetermined */
	ddata->can_restart = 1;
	ddata->master_code = atomic_inc_return(&cur_mastercode) - 1;
	ddata->transfer_timeout = bdata->transfer_timeout ?
		bdata->transfer_timeout : DWAPBI2C_DEFAULT_TRANSFER_TIMEOUT_MSECS;
	atomic_set(&ddata->stop, 0);
	atomic_set(&ddata->completion_pending, 0);
	mutex_init(&ddata->access_lock);
	init_waitqueue_head(&ddata->wait_queue);
	
	ddata->use_dma = (bdata->use_dma != 0);
	ddata->is_slave = (bdata->is_slave != 0);
	ddata->i2c_speed = bdata->i2c_speed;
	ddata->slave_addr = bdata->slave_addr;
	
	ddata->sda_setup_delay = apb_read8(ddata->base, DWAPBI2C_IC_SDA_SETUP);
	
	if (unlikely(ret = device_create_file(&dev->dev, &dev_attr_config))) {
		dev_err(&dev->dev, "%s: unable to create "
			"bus config file\n", __func__);
		goto err_devfile;
	}

	if (unlikely(ret = i2c_add_adapter(&ddata->adapter))) {
		dev_err(&dev->dev, "%s: I2C adapter registration "
			"failed\n", __func__);
		goto err_i2cadd;
	}
	
	dev_info(&dev->dev, "device ready\n");
	
	return 0;

err_i2cadd:
 	device_remove_file(&dev->dev, &dev_attr_config);
err_devfile:
 	clk_put(ddata->clock);
err_clkget:
err_irq:
 	while (--i >= 0) {
		if (dev->irq[i] != NO_IRQ)
 			free_irq(dev->irq[i], (void *)ddata);
	}
	iounmap(ddata->base);
err_ioremap:
 	kfree(ddata);
err_alloc:
	return ret;
}

static int dwapbi2c_remove(struct amba_device *dev)
{
	int ret, i;
	struct dwapbi2c_drv_data *ddata = amba_get_drvdata(dev);
	
	dev_info(&dev->dev, "removing device\n");
	
	/* Make sure no transfer is ongoing */
	atomic_set(&ddata->stop, 1);
	if (unlikely(ret = mutex_lock_interruptible(&ddata->access_lock))) {
		atomic_set(&ddata->stop, 0);
		return ret;
	}
	
	amba_set_drvdata(dev, NULL);
	
	for (i=0; i<AMBA_NR_IRQS; i++) {
		if (dev->irq[i] != NO_IRQ)
 			free_irq(dev->irq[i], (void *)ddata);
	}
	
	device_remove_file(&dev->dev, &dev_attr_config);
	
	if (unlikely(ret = i2c_del_adapter(&ddata->adapter)))
		dev_err(&dev->dev, "%s: unable to unregister I2C adapter, "
			"(error is %d) removing anyway\n", __func__, ret);
	
	if (ddata->dma_tx_buf)
		free_pages((unsigned long)ddata->dma_tx_buf, ddata->dma_tx_buf_order);
	if (ddata->dma_rx_buf)
		free_pages((unsigned long)ddata->dma_rx_buf, ddata->dma_rx_buf_order);
	
	clk_put(ddata->clock);
	mutex_unlock(&ddata->access_lock);
	iounmap(ddata->base);
	kfree(ddata);
	
	return 0;
}

#ifdef CONFIG_PM

static int dwapbi2c_suspend(struct amba_device *dev, pm_message_t msg)
{
	int ret;
	struct dwapbi2c_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "suspending -- all I2C messages will be discarded\n");
	atomic_set(&ddata->stop, 1);
	if (unlikely(ret = mutex_lock_interruptible(&ddata->access_lock))) {
		atomic_set(&ddata->stop, 0);
		return ret;
	}
	mutex_unlock(&ddata->access_lock);
	return 0;
}

static int dwapbi2c_resume(struct amba_device *dev)
{
	struct dwapbi2c_drv_data *ddata = amba_get_drvdata(dev);
	dev_info(&dev->dev, "resuming\n");
	atomic_set(&ddata->stop, 0);
	return 0;
}

#endif

/*-------------------------------------------------------------------------*/

static struct __initdata amba_id dwapbi2c_ids[] = {
	{
		.id	= DWAPBI2C_ID,
		.mask	= DWAPBI2C_ID_MASK,
	},
	{ 0, 0 },
};

static struct __initdata amba_driver dwapbi2c_drv = {
	.drv		= {
		.name	= DRIVER_NAME,
	},
	.probe		= dwapbi2c_probe,
	.remove		= dwapbi2c_remove,
#ifdef CONFIG_PM
	.suspend	= dwapbi2c_suspend,
	.resume		= dwapbi2c_resume,
#endif
	.id_table	= dwapbi2c_ids,
};

static int __init dwapbi2c_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	atomic_set(&cur_mastercode, 0);
	
	if (unlikely(ret = amba_driver_register(&dwapbi2c_drv)))
		error("could not register AMBA driver");

	return ret;
}

static void __exit dwapbi2c_exit(void)
{
	amba_driver_unregister(&dwapbi2c_drv);
}

module_init(dwapbi2c_init);
module_exit(dwapbi2c_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
