#define DRIVER_NAME 	"gpiospi"
#define DRIVER_DESC 	"SPI over GPIO driver"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:4.1"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/platform_device.h>

#include <linux/gpio-core.h>

#include <linux/spi/spi.h>
#include <linux/spi/spi_bitbang.h>

#include <linux/gpiospi.h>

/*-------------------------------------------------------------------------*/

#define error(format, arg...)	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...) 	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) 	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) 	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif

/*-------------------------------------------------------------------------*/

struct gpiospi_spi_device {
	struct list_head		list;
	struct spi_device		*dev;
};

struct gpiospi_slave_data {
	struct list_head		list;
	struct gpiospi_master_data	*mdata;
	struct gpiospi_chip_data	*cdata;
	struct spi_device		*dev;
	struct gpio_client_pin		*cs_, *strobe, *alt_mosi, *alt_miso;
};

struct gpiospi_master_data {
	struct platform_device		*dev;
	struct gpiospi_bus_data		*bdata;
	struct list_head		slaves;
	struct gpio_client		*gpiocli;
	int				initializing;
	struct gpio_client_pin		*sclk, *mosi, *miso;
	struct list_head 		spi_devices;
	struct spi_bitbang		*bitbang;
};

/*-------------------------------------------------------------------------*/

static inline struct gpiospi_chip_data *spidev_to_cdata(struct spi_device *spi)
{
	return ((struct gpiospi_slave_data *)spi->controller_data)->cdata;
}

static inline struct gpiospi_slave_data *spidev_to_sdata(struct spi_device *spi)
{
	return spi->controller_data;
}

/*-------------------------------------------------------------------------*/

static inline void gpiospi_set_value(struct gpio_client_pin *pin, int on)
{
	gpio_direction_output(pin->hdrv, pin->index, on);
}

static inline int gpiospi_get_value(struct gpio_client_pin *pin)
{
	return gpio_get_value(pin->hdrv, pin->index);
}

static inline void setsck(struct spi_device *spi, int on)
{
	struct gpiospi_master_data *mdata = spidev_to_sdata(spi)->mdata;
	if (likely(mdata->sclk)) {
		gpiospi_set_value(mdata->sclk, on);}
	else
		dev_err(&mdata->dev->dev, "clock pin is not available?!");
}

static inline void setmosi(struct spi_device *spi, int on)
{
	struct gpiospi_slave_data *sdata = spidev_to_sdata(spi);
	if (sdata->alt_mosi)
		gpiospi_set_value(sdata->alt_mosi, on);
	else if (sdata->mdata->mosi)
		gpiospi_set_value(sdata->mdata->mosi, on);
	else
		dev_err(&sdata->mdata->dev->dev, "no MOSI pin available?!");
}

static inline u32 getmiso(struct spi_device *spi)
{
	struct gpiospi_slave_data *sdata = spidev_to_sdata(spi);
	if (sdata->alt_miso)
		return gpiospi_get_value(sdata->alt_miso);
	else if (sdata->mdata->miso)
		return gpiospi_get_value(sdata->mdata->miso);
	return 0;
}

#define spidelay(x) ndelay(x)

#define	EXPAND_BITBANG_TXRX
#include <linux/spi/spi_bitbang.h>

static u32 gpiospi_txrx_mode0(struct spi_device *spi,
	unsigned nsecs, u32 word, u8 bits)
{
	return bitbang_txrx_be_cpha0(spi, nsecs, 0, word, bits);
}

static u32 gpiospi_txrx_mode1(struct spi_device *spi,
	unsigned nsecs, u32 word, u8 bits)
{
	return bitbang_txrx_be_cpha1(spi, nsecs, 0, word, bits);
}

static u32 gpiospi_txrx_mode2(struct spi_device *spi,
	unsigned nsecs, u32 word, u8 bits)
{
	return bitbang_txrx_be_cpha0(spi, nsecs, 1, word, bits);
}

static u32 gpiospi_txrx_mode3(struct spi_device *spi,
	unsigned nsecs, u32 word, u8 bits)
{
	return bitbang_txrx_be_cpha1(spi, nsecs, 1, word, bits);
}

static void gpiospi_do_strobe(struct gpiospi_slave_data *sdata)
{
	if (sdata->cdata->strobe_predelay_ns > 0)
		ndelay(sdata->cdata->strobe_predelay_ns);
		
	gpiospi_set_value(sdata->strobe, 1);
	ndelay(sdata->cdata->strobe_delay_ns);
	gpiospi_set_value(sdata->strobe, 0);
	
	if (sdata->cdata->strobe_postdelay_ns > 0)
		ndelay(sdata->cdata->strobe_postdelay_ns);
}

static void gpiospi_delesect_all_slaves(struct gpiospi_master_data *mdata, int default_delay)
{
	struct gpiospi_slave_data *sdata;
	int max_delay = default_delay;
	
	list_for_each_entry(sdata, &mdata->slaves, list) {
		/* Ensure strobe is LOW */
		if (sdata->strobe) {
			gpiospi_set_value(sdata->strobe, 0);
			if (sdata->cdata->strobe_postdelay_ns > max_delay)
				max_delay = sdata->cdata->strobe_postdelay_ns;
		}
		
		/* Ensure CS_ is HIGH */
		if (sdata->cs_) {
			gpiospi_set_value(sdata->cs_, 1);
			if (sdata->cdata->cs__postdelay_ns > max_delay)
				max_delay = sdata->cdata->cs__postdelay_ns;
		}
	}
			
	ndelay(max_delay);
}

#define HZ_TO_NS(hz) (1000000000 / (hz))

static void gpiospi_chipselect(struct spi_device *spi, int value)
{
	struct gpiospi_slave_data *sdata = spidev_to_sdata(spi);

	/* chip DEselect: pulse strobe now if needed */
	if (value == BITBANG_CS_INACTIVE) {
		if (sdata->strobe)
			gpiospi_do_strobe(sdata);
		
		if (sdata->cs_) {
			gpiospi_set_value(sdata->cs_, 1);
			if (sdata->cdata->cs__postdelay_ns > 0)
				ndelay(sdata->cdata->cs__postdelay_ns);
		}
		
	} else if (value == BITBANG_CS_ACTIVE) {
		/*
		 * SAFETY FIRST: deselect any previously selected chip 
		 * (eg. if some external driver made a chip selected...).
		 * DESELECTS also this CURRENT chip select to start over 
		 * a new transfer.
		 */
		gpiospi_delesect_all_slaves(sdata->mdata, HZ_TO_NS(spi->max_speed_hz));
	
		if (sdata->cs_) {
			gpiospi_set_value(sdata->mdata->sclk, ((spi->mode & 2) != 0));
			
			/* Actually select the chip */
			gpiospi_set_value(sdata->cs_, 0);
			
			/*
			 * Wait the equivalent of one clock period before 
			 * sending data, required by most chips.
			 */
			ndelay(HZ_TO_NS(spi->max_speed_hz));
		}
	}
}


static int gpiospi_setup_transfer(struct spi_device *spi, struct spi_transfer *t)
{
	struct gpiospi_slave_data 	*sdata = spidev_to_sdata(spi);
	struct gpiospi_master_data 	*mdata = sdata->mdata;
	struct gpiospi_chip_data 	*cdata = sdata->cdata;
	
	if (sdata->alt_miso)
		gpio_direction_input(sdata->alt_miso->hdrv, sdata->alt_miso->index);
	else if (mdata->miso)
		gpio_direction_input(mdata->miso->hdrv, mdata->miso->index);
	
	if (mdata->initializing) {
		dev_info(&mdata->dev->dev, "new chip(%d): max speed %lu%sHz, driver: %s, name: %s\n",
			spi->chip_select, (unsigned long)(spi->max_speed_hz < 10000 ?
			spi->max_speed_hz : (spi->max_speed_hz / 1000)),
			spi->max_speed_hz < 10000 ? "" : "K", spi->modalias,
			cdata->name ? cdata->name : "");	
		sdata->dev = spi;
		list_add_tail(&sdata->list, &mdata->slaves);
	}
	
	return spi_bitbang_setup_transfer(spi, t);
}

/*-------------------------------------------------------------------------*/

static int gpiospi_gpiocli_probe(struct gpio_client *client)
{
	int 				ret, i;
	struct spi_master 		*master;
	struct gpiospi_master_data 	*mdata = client->parent_data;
	struct platform_device 		*dev = mdata->dev;
	
	master = spi_alloc_master(&mdata->dev->dev, sizeof(struct spi_bitbang));
	if (unlikely(!master)) {
		dev_err(&dev->dev, "%s: failed to allocate SPI master\n", __func__);
		return -ENOMEM;
	}
	
	mdata->bitbang = spi_master_get_devdata(master);
	mdata->bitbang->master = spi_master_get(master);
	mdata->bitbang->master->bus_num = dev->id;
        mdata->bitbang->chipselect = gpiospi_chipselect;
	mdata->bitbang->setup_transfer = gpiospi_setup_transfer;
	mdata->bitbang->txrx_word[SPI_MODE_0] = gpiospi_txrx_mode0;
	mdata->bitbang->txrx_word[SPI_MODE_1] = gpiospi_txrx_mode1;
	mdata->bitbang->txrx_word[SPI_MODE_2] = gpiospi_txrx_mode2;
	mdata->bitbang->txrx_word[SPI_MODE_3] = gpiospi_txrx_mode3;
	
	mdata->initializing = 1;
	
	/* Start bitbang adapter which will perform the spi_register_master call. */
	if (unlikely(ret = spi_bitbang_start(mdata->bitbang))) {
		dev_err(&dev->dev, "%s: spi_bitbang_start failed with error %d\n", __func__, ret);
		goto err_bitbang;
	}
	
	INIT_LIST_HEAD(&mdata->spi_devices);
	for (i=0; i<mdata->bdata->num_chips; i++) {
		struct gpiospi_spi_device *d_spi_dev =
			kzalloc(sizeof(struct gpiospi_spi_device), GFP_KERNEL);
		if (unlikely(!d_spi_dev))
			continue;
		mdata->bdata->chips[i].info.bus_num = master->bus_num;
		d_spi_dev->dev = spi_new_device(master, &mdata->bdata->chips[i].info);
		if (likely(d_spi_dev->dev))
			list_add_tail(&d_spi_dev->list, &mdata->spi_devices);
		else
			kfree(d_spi_dev);
	}
	
	mdata->initializing = 0;
	
	return 0;
	
err_bitbang:
	spi_master_put(mdata->bitbang->master);
	put_device(&dev->dev);
	return ret;
}

static int gpiospi_gpiocli_remove(struct gpio_client *client)
{
	struct gpiospi_master_data 	*mdata = client->client_data;
	struct platform_device 		*dev = mdata->dev;
	struct gpiospi_spi_device 	*entry, *tmp_entry;
	
	dev_info(&dev->dev, "removing bus\n");
	
	/* Unregister SPI devices currently on the bus. */
	list_for_each_entry_safe(entry, tmp_entry, &mdata->spi_devices, list) {
		if (entry->dev)
			spi_unregister_device(entry->dev);
		list_del(&entry->list);
		kfree(entry);
	}
	
	/* stop the bitbang adapter (stops the work queue and unregisters master) */
	spi_bitbang_stop(mdata->bitbang);
	
	/* release memory allocated for master */
	spi_master_put(mdata->bitbang->master);
	put_device(&dev->dev);
	
	return 0;
}

static struct gpio_client_pin *gpiospi_add_pin_unique(struct gpio_client_pin *pins, 
	unsigned gpio, int cur_i, const char *fmt, const char *bus_id)
{
	int i;
	for (i=0; i<cur_i; i++)
		if (pins[i].gpio == gpio)
			return &pins[i];
	pins[cur_i].gpio = gpio;
	snprintf(pins[cur_i].label, sizeof(pins[cur_i].label), fmt, bus_id);
	return &pins[cur_i];
}

static void gpiospi_release(struct device *dev)
{
	int 				j;
	struct gpio_client 		*gpiocli = dev_get_drvdata(dev);
	struct gpiospi_master_data 	*mdata = gpiocli->parent_data;
	dev_set_drvdata(dev, NULL);
	for (j=0; j<mdata->bdata->num_chips; j++)
		kfree(mdata->bdata->chips[j].info.controller_data);
	kfree(gpiocli->match_pins);
	kfree(mdata);
	kfree(gpiocli);
}

static int gpiospi_probe(struct platform_device *dev)
{
	int 				ret, i = 0, j;
	struct gpio_client 		*gpiocli;
	struct gpiospi_master_data 	*mdata;
	struct gpiospi_bus_data 	*bdata = dev->dev.platform_data;
	
	gpiocli = kzalloc(sizeof(struct gpio_client), GFP_KERNEL);
	if (!gpiocli)
		return -ENOMEM;
	platform_set_drvdata(dev, gpiocli);
	gpiocli->probe = gpiospi_gpiocli_probe;
	gpiocli->remove = gpiospi_gpiocli_remove;
	gpiocli->match_group = bdata->gpio_group;
	
	mdata = kzalloc(sizeof(struct gpiospi_master_data), GFP_KERNEL);
	if (!mdata) {
		ret = -ENOMEM;
		goto err_alloc_mdata;
	}
	INIT_LIST_HEAD(&mdata->slaves);
	mdata->dev = dev;
	mdata->bdata = bdata;
	
	gpiocli->parent_data = mdata;
	mdata->gpiocli = gpiocli;
	
	/*
	 * Now fill the gpiocli->match_pins with the GPIOs we need: 
	 * the gpio_client probe function will not be called until
	 * all these GPIOs are available (ie. when relevant controller
	 * drivers are loaded).
	 */
	
	gpiocli->match_pins = kzalloc(sizeof(struct gpio_client_pin) 
		* (3 + bdata->num_chips * 4), GFP_KERNEL);
	if (!gpiocli->match_pins) {
		ret = -ENOMEM;
		goto err_alloc_gpioclip;
	}
	
	mdata->sclk = gpiospi_add_pin_unique(gpiocli->match_pins, bdata->pin_sclk, 
		i++, "%s.SCLK", dev->dev.bus_id);
	
	if (bdata->pin_mosi >= 0)
		if ((mdata->mosi = gpiospi_add_pin_unique(gpiocli->match_pins, bdata->pin_mosi, 
			i, "%s.MOSI", dev->dev.bus_id)))
			i++;
	if (bdata->pin_miso >= 0)
		if ((mdata->miso = gpiospi_add_pin_unique(gpiocli->match_pins, bdata->pin_miso, 
			i, "%s.MISO", dev->dev.bus_id)))
			i++;
	
	for (j=0; j<bdata->num_chips; j++) {
		struct gpiospi_slave_data *sdata = kzalloc(sizeof(struct gpiospi_slave_data), GFP_KERNEL);
		if (!sdata)
			continue;
		sdata->mdata = mdata;
		bdata->chips[j].info.controller_data = sdata;
		sdata->cdata = &bdata->chips[j].cdata;
		if (sdata->cdata->pin_cs_ >= 0)
			if ((sdata->cs_ = gpiospi_add_pin_unique(gpiocli->match_pins, sdata->cdata->pin_cs_, 
				i, "%s.CS", dev->dev.bus_id)))
				i++;
		if (sdata->cdata->pin_strobe >= 0)
			if ((sdata->strobe = gpiospi_add_pin_unique(gpiocli->match_pins, sdata->cdata->pin_strobe, 
				i, "%s.STROBE", dev->dev.bus_id)))
				i++;
		if (sdata->cdata->alt_pin_mosi >= 0)
			if ((sdata->alt_mosi = gpiospi_add_pin_unique(gpiocli->match_pins, sdata->cdata->alt_pin_mosi, 
				i, "%s.ALTMOSI", dev->dev.bus_id)))
				i++;
		if (sdata->cdata->alt_pin_miso >= 0)
			if ((sdata->alt_miso = gpiospi_add_pin_unique(gpiocli->match_pins, sdata->cdata->alt_pin_miso, 
				i, "%s.ALTMISO", dev->dev.bus_id)))
				i++;
	}
	gpiocli->pin_count = i;
	
	if (unlikely(ret = gpio_client_register(gpiocli))) {
		dev_err(&dev->dev, "%s: failed to register GPIO client, "
			"error is %d\n", __func__, ret);
		goto err_reg;
	}
	
	if (!dev->dev.release)
		dev->dev.release = gpiospi_release;
	
	return 0;
	
err_reg:
	for (j=0; j<bdata->num_chips; j++)
		kfree(bdata->chips[j].info.controller_data);
	kfree(gpiocli->match_pins);
err_alloc_gpioclip:
	kfree(mdata);
err_alloc_mdata:
	kfree(gpiocli);
	return ret;
}

static int gpiospi_remove(struct platform_device *dev)
{
	int 				ret;
	struct gpio_client 		*gpiocli = platform_get_drvdata(dev);
	struct gpiospi_master_data 	*mdata = gpiocli->parent_data;
	
	if (unlikely(ret = gpio_client_unregister(gpiocli))) {
		dev_err(&mdata->dev->dev, "%s: failed to unregister GPIO client, "
			"error is %d\n", __func__, ret);
		return ret;
	}
	
	return 0;
}

/*-------------------------------------------------------------------------*/

static struct platform_driver gpiospi_drv = {
	.probe 		= gpiospi_probe,
        .remove		= gpiospi_remove,
        .driver		= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
        },
};

static int __init gpiospi_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)", 
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);
	
	if ((ret = platform_driver_register(&gpiospi_drv)) < 0)
		error("%s: platform_driver_register failed with error %d", __func__, ret);
	
	return ret;
}

static void __exit gpiospi_exit(void)
{
        platform_driver_unregister(&gpiospi_drv);
}

module_init(gpiospi_init);
module_exit(gpiospi_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");

