#define DRIVER_NAME 	"dwapbgpio"
#define DRIVER_DESC 	"Driver for the DW_apb_gpio component"
#define DRIVER_AUTHOR 	"Gregoire Pean <gpean@mobilygen.com>"
#define DRIVER_VERSION 	"1:3.5"

/*
 *  This file Copyright (C) 2007 Mobilygen Corp.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE	LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/vermagic.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/device.h>

#include <linux/amba/bus.h>

#include <asm/atomic.h>
#include <asm/io.h>

#include <linux/dwapbgpio.h>
#include <mach/gpio.h>
#include <mach/mobi_ioctrl.h>

#include <linux/gpio.h>

/*-------------------------------------------------------------------------*/
#define error(format, arg...) \
	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#define info(format, arg...)\
	printk(KERN_INFO DRIVER_NAME ": " format "\n" , ## arg)
#define warn(format, arg...)\
	printk(KERN_WARNING DRIVER_NAME ": " format "\n" , ## arg)

#ifdef DEBUG
#define debug(format, arg...) \
	printk(KERN_ERR DRIVER_NAME ": " format "\n" , ## arg)
#else
#define debug(format, arg...) 	do {} while (0)
#endif
/*-------------------------------------------------------------------------*/

/* You can choose not to create character devices in /dev with this */
static int no_dev_entries;
module_param(no_dev_entries, bool, S_IRUGO);

/*-------------------------------------------------------------------------*/

struct dwapbgpio_drv_data {
	struct amba_device 		*dev;
	void __iomem 			*base;
	struct dwapbgpio_block_data	*bdata;
	struct gpio_driver 	drv;
	short	port_a_last, port_b_last, port_c_last, port_d_last;
	int				restore_func;
};

/*-------------------------------------------------------------------------*/

static inline u32 apb_read32(void __iomem *base, int reg)
{
	return ioread32(base + reg);
}

static inline void apb_write32(void __iomem *base, int reg, u32 value)
{
	iowrite32(value, base + reg);
}

static inline u16 apb_read16(void __iomem *base, int reg)
{
	return ioread16(base + reg);
}

static inline void apb_write16(void __iomem *base, int reg, u16 value)
{
	iowrite16(value, base + reg);
}

static inline u8 apb_read8(void __iomem *base, int reg)
{
	return ioread8(base + reg);
}

static inline void apb_write8(void __iomem *base, int reg, u8 value)
{
	iowrite8(value, base + reg);
}

/*-------------------------------------------------------------------------*/

static void dwapbgpio_invalid_index(struct dwapbgpio_drv_data *ddata,
	const char *func, unsigned gpio)
{
	dev_err(&ddata->dev->dev, "%s: invalid GPIO index: %u\n", func, gpio);
}

/* GPIO framework exports */

static int dwapbgpio_request(struct gpio_chip *chip, unsigned gpio_index)
{
#ifdef CONFIG_ARCH_MERLIN
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int ret = mobi_ioctrl_get_gpio(ddata->bdata->gpio_id_offset +
			gpio_index,
		IOCTRL_GPIO_CONTROL_FUNCTION);
	if (unlikely(ret < 0))
		return ret;
	ddata->restore_func = -1;
	if (ret != IOCTRL_GPIO_FUNCTION_GPIO) {
		ddata->restore_func = ret;
		return mobi_ioctrl_set_gpio(ddata->bdata->gpio_id_offset +
				gpio_index,
			IOCTRL_GPIO_CONTROL_FUNCTION,
			IOCTRL_GPIO_FUNCTION_GPIO);
	}
#endif
	return 0;
}

static void dwapbgpio_free(struct gpio_chip *chip, unsigned gpio_index)
{
#ifdef CONFIG_ARCH_MERLIN
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	if (ddata->restore_func != -1) {
		mobi_ioctrl_set_gpio(ddata->bdata->gpio_id_offset + gpio_index,
			IOCTRL_GPIO_CONTROL_FUNCTION, ddata->restore_func);
		ddata->restore_func = -1;
	}
#endif
}

static int dwapbgpio_is_enabled(struct gpio_driver *drv, unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int ret = mobi_ioctrl_get_gpio(ddata->bdata->gpio_id_offset +
			gpio_index,
		IOCTRL_GPIO_CONTROL_FUNCTION);
	if (unlikely(ret < 0))
		return ret;
	return ret == IOCTRL_GPIO_FUNCTION_GPIO;
	return -1;
}

static int dwapbgpio_direction_input(struct gpio_chip *chip,
		unsigned gpio_index)
{
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int reg = -1, off;
	if (gpio_index < ddata->port_a_last) {
		reg = DWAPBGPIO_IC_SWPORTA_DDR;
		off = 0;
	} else if (gpio_index < ddata->port_b_last) {
		reg = DWAPBGPIO_IC_SWPORTB_DDR;
		off = ddata->port_a_last;
	} else if (gpio_index < ddata->port_c_last) {
		reg = DWAPBGPIO_IC_SWPORTC_DDR;
		off = ddata->port_b_last;
	} else if (gpio_index < ddata->port_d_last) {
		reg = DWAPBGPIO_IC_SWPORTD_DDR;
		off = ddata->port_c_last;
	}
	if (reg >= 0) {
		apb_write32(ddata->base, reg,
			apb_read32(ddata->base, reg)
			& ~(1 << ((int)gpio_index - off)));
		return 0;
	} else {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
}

static int dwapbgpio_direction_output(struct gpio_chip *chip,
		unsigned gpio_index, int value)
{
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int reg = -1, regr = -1, regw = -1, off;
	if (gpio_index < ddata->port_a_last) {
		reg = DWAPBGPIO_IC_SWPORTA_DDR;
		regw = DWAPBGPIO_IC_SWPORTA_DR;
		regr = DWAPBGPIO_IC_EXT_PORTA;
		off = 0;
	} else if (gpio_index < ddata->port_b_last) {
		reg = DWAPBGPIO_IC_SWPORTB_DDR;
		regw = DWAPBGPIO_IC_SWPORTB_DR;
		regr = DWAPBGPIO_IC_EXT_PORTB;
		off = ddata->port_a_last;
	} else if (gpio_index < ddata->port_c_last) {
		reg = DWAPBGPIO_IC_SWPORTC_DDR;
		regw = DWAPBGPIO_IC_SWPORTC_DR;
		regr = DWAPBGPIO_IC_EXT_PORTC;
		off = ddata->port_b_last;
	} else if (gpio_index < ddata->port_d_last) {
		reg = DWAPBGPIO_IC_SWPORTD_DDR;
		regw = DWAPBGPIO_IC_SWPORTD_DR;
		regr = DWAPBGPIO_IC_EXT_PORTD;
		off = ddata->port_c_last;
	}
	if (reg >= 0) {
		u32 v = (1 << ((int)gpio_index - off));
		if (value)
			apb_write32(ddata->base, regw,
					apb_read32(ddata->base, regr) | v);
		else
			apb_write32(ddata->base, regw,
					apb_read32(ddata->base, regr) & ~v);
		apb_write32(ddata->base, reg, apb_read32(ddata->base, reg) | v);
		return 0;
	} else {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
}

static int dwapbgpio_get_value(struct gpio_chip *chip, unsigned gpio_index)
{
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int reg = -1, off;
	if (gpio_index < ddata->port_a_last) {
		reg = DWAPBGPIO_IC_EXT_PORTA;
		off = 0;
	} else if (gpio_index < ddata->port_b_last) {
		reg = DWAPBGPIO_IC_EXT_PORTB;
		off = ddata->port_a_last;
	} else if (gpio_index < ddata->port_c_last) {
		reg = DWAPBGPIO_IC_EXT_PORTC;
		off = ddata->port_b_last;
	} else if (gpio_index < ddata->port_d_last) {
		reg = DWAPBGPIO_IC_EXT_PORTD;
		off = ddata->port_c_last;
	}
	if (reg >= 0) {
		return (apb_read32(ddata->base, reg) &
				(1 << ((int)gpio_index - off))) != 0;
	} else {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
}

void dwapbgpio_set_value(struct gpio_chip *chip,
		unsigned gpio_index, int value)
{
	struct gpio_driver *drv = container_of(chip, struct gpio_driver, chip);
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int reg = -1, regr = -1, off;
	if (gpio_index < ddata->port_a_last) {
		reg = DWAPBGPIO_IC_SWPORTA_DR;
		regr = DWAPBGPIO_IC_EXT_PORTA;
		off = 0;
	} else if (gpio_index < ddata->port_b_last) {
		reg = DWAPBGPIO_IC_SWPORTB_DR;
		regr = DWAPBGPIO_IC_EXT_PORTB;
		off = ddata->port_a_last;
	} else if (gpio_index < ddata->port_c_last) {
		reg = DWAPBGPIO_IC_SWPORTC_DR;
		regr = DWAPBGPIO_IC_EXT_PORTC;
		off = ddata->port_b_last;
	} else if (gpio_index < ddata->port_d_last) {
		reg = DWAPBGPIO_IC_SWPORTD_DR;
		regr = DWAPBGPIO_IC_EXT_PORTD;
		off = ddata->port_c_last;
	}
	if (reg >= 0) {
		u32 v = apb_read32(ddata->base, regr);
		if (value)
			v |= (1 << ((int)gpio_index - off));
		else
			v &= ~(1 << ((int)gpio_index - off));
		apb_write32(ddata->base, reg, v);
		return ;
	} else {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return ;
	}
}

static int dwapbgpio_get_direction(struct gpio_driver *drv, unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int reg = -1, off;
	if (gpio_index < ddata->port_a_last) {
		reg = DWAPBGPIO_IC_SWPORTA_DDR;
		off = 0;
	} else if (gpio_index < ddata->port_b_last) {
		reg = DWAPBGPIO_IC_SWPORTB_DDR;
		off = ddata->port_a_last;
	} else if (gpio_index < ddata->port_c_last) {
		reg = DWAPBGPIO_IC_SWPORTC_DDR;
		off = ddata->port_b_last;
	} else if (gpio_index < ddata->port_d_last) {
		reg = DWAPBGPIO_IC_SWPORTD_DDR;
		off = ddata->port_c_last;
	}
	if (reg >= 0) {
		return (apb_read32(ddata->base, reg) &
				(1 << ((int)gpio_index - off)))
			? GPIO_DIRECTION_OUTPUT : GPIO_DIRECTION_INPUT;
	} else {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
}

static int dwapbgpio_read_bank(struct gpio_driver *drv, u32 *data,
		int dword_count)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	int i = 0;
	if (ddata->bdata->port_a_count && i < dword_count)
		data[i++] = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTA);
	if (ddata->bdata->port_b_count && i < dword_count)
		data[i++] = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTB);
	if (ddata->bdata->port_c_count && i < dword_count)
		data[i++] = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTC);
	if (ddata->bdata->port_d_count && i < dword_count)
		data[i++] = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTD);
	return i;
}

static int dwapbgpio_write_bank(struct gpio_driver *drv, const u32 *data,
	const u32 *mask, int dword_count)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	u32 r;
	int i = 0;

	if (dword_count) {
		r = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTA);
		r &= ~(~data[i] & mask[i]);
		r |= data[i] & mask[i];
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTA_DR, r);
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTA_DDR, mask[i]);
		i++;
	}
	if (dword_count > 1) {
		r = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTB);
		r &= ~(~data[i] & mask[i]);
		r |= data[i] & mask[i];
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTB_DR, r);
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTB_DDR, mask[i]);
		i++;
	}
	if (dword_count > 2) {
		r = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTC);
		r &= ~(~data[i] & mask[i]);
		r |= data[i] & mask[i];
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTC_DR, r);
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTC_DDR, mask[i]);
		i++;
	}
	if (dword_count > 3) {
		r = apb_read32(ddata->base, DWAPBGPIO_IC_EXT_PORTD);
		r &= ~(~data[i] & mask[i]);
		r |= data[i] & mask[i];
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTD_DR, r);
		apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTD_DDR, mask[i]);
		i++;
	}

	return i;
}

static int dwapbgpio_interrupt_enable(struct gpio_driver *drv,
		unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	u32 gpio_shift;
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
	gpio_shift = (1 << gpio_index);
	apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTA_DDR,
		apb_read32(ddata->base, DWAPBGPIO_IC_SWPORTA_DDR)
		& ~gpio_shift);
	apb_write32(ddata->base, DWAPBGPIO_IC_INTMASK,
		apb_read32(ddata->base, DWAPBGPIO_IC_INTMASK)
		& ~gpio_shift);
	apb_write32(ddata->base, DWAPBGPIO_IC_INTEN,
		apb_read32(ddata->base, DWAPBGPIO_IC_INTEN)
		| gpio_shift);
	return 0;
}

static int dwapbgpio_interrupt_is_enabled(struct gpio_driver *drv,
		unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return 0;
	}
	return 0 != (apb_read32(ddata->base, DWAPBGPIO_IC_INTEN) &
			(1 << gpio_index));
}

static int dwapbgpio_interrupt_is_active(struct gpio_driver *drv,
		unsigned gpio_index, int irq)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return 0;
	}
	return 0 != (apb_read32(ddata->base, DWAPBGPIO_IC_INTSTATUS) &
			(1 << gpio_index));
}

static int dwapbgpio_interrupt_disable(struct gpio_driver *drv,
		unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
	apb_write32(ddata->base, DWAPBGPIO_IC_INTEN,
		apb_read32(ddata->base, DWAPBGPIO_IC_INTEN)
		& ~(1 << gpio_index));
	apb_write32(ddata->base, DWAPBGPIO_IC_INTMASK,
		apb_read32(ddata->base, DWAPBGPIO_IC_INTMASK)
		| (1 << gpio_index));
	return 0;
}

static int dwapbgpio_interrupt_configure(struct gpio_driver *drv,
		unsigned gpio_index,
	const struct gpio_interrupt_config *config)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	u32 gpio_shift;
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
	gpio_shift = (1 << gpio_index);

	if (config->type == GPIO_INT_TYPE_LEVEL) {
		apb_write32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL,
			apb_read32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL) &
			~gpio_shift);
	} else if (config->type == GPIO_INT_TYPE_EDGE) {
		apb_write32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL,
			apb_read32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL) |
			gpio_shift);
	} else {
		dev_err(&ddata->dev->dev, "%s: setting edge+level type is not "
			"supported by this controller\n", __func__);
		return -EINVAL;
	}

	if (config->polarity == GPIO_INT_POLARITY_LOW_OR_FALLING) {
		apb_write32(ddata->base, DWAPBGPIO_IC_INT_POLARITY,
			apb_read32(ddata->base, DWAPBGPIO_IC_INT_POLARITY) &
			~gpio_shift);
	} else if (config->polarity == GPIO_INT_POLARITY_HIGH_OR_RISING) {
		apb_write32(ddata->base, DWAPBGPIO_IC_INT_POLARITY,
			apb_read32(ddata->base, DWAPBGPIO_IC_INT_POLARITY) |
			gpio_shift);
	} else {
		dev_err(&ddata->dev->dev, "%s: setting both polarities is not "
			"supported by this controller\n", __func__);
		return -EINVAL;
	}

	if (config->debounce_on) {
		apb_write32(ddata->base, DWAPBGPIO_IC_DEBOUNCE,
			apb_read32(ddata->base, DWAPBGPIO_IC_DEBOUNCE) |
			gpio_shift);
	} else {
		apb_write32(ddata->base, DWAPBGPIO_IC_DEBOUNCE,
			apb_read32(ddata->base, DWAPBGPIO_IC_DEBOUNCE) &
			~gpio_shift);
	}

	return 0;
}

static int dwapbgpio_interrupt_get_config(struct gpio_driver *drv,
	unsigned gpio_index, struct gpio_interrupt_config *config)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	u32 gpio_shift;
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
	gpio_shift = (1 << gpio_index);

	if (apb_read32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL) & gpio_shift)
		config->type = GPIO_INT_TYPE_EDGE;
	else
		config->type = GPIO_INT_TYPE_LEVEL;

	if (apb_read32(ddata->base, DWAPBGPIO_IC_INT_POLARITY) & gpio_shift)
		config->polarity = GPIO_INT_POLARITY_HIGH_OR_RISING;
	else
		config->polarity = GPIO_INT_POLARITY_LOW_OR_FALLING;

	if (apb_read32(ddata->base, DWAPBGPIO_IC_DEBOUNCE) & gpio_shift)
		config->debounce_on = 1;
	else
		config->debounce_on = 0;

	return 0;
}

static int dwapbgpio_interrupt_clear(struct gpio_driver *drv,
		unsigned gpio_index)
{
	struct dwapbgpio_drv_data *ddata = gpio_get_drvdata(drv);
	if (gpio_index >= ddata->port_a_last) {
		dwapbgpio_invalid_index(ddata, __func__, gpio_index);
		return -EINVAL;
	}
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTA_EOI, 1 << gpio_index);
	return 0;
}

/*-------------------------------------------------------------------------*/

static void dwapbgpio_reset(struct dwapbgpio_drv_data *ddata)
{
	/* Set control to software */
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTA_CTL, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTB_CTL, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTC_CTL, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTD_CTL, 0);

	/* Set all pins of every port to input mode */
#if 0
	apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTA_DDR, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTB_DDR, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTC_DDR, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_SWPORTD_DDR, 0);
#endif

	/* Disable interrupts on port A, reset some defaults */
	apb_write32(ddata->base, DWAPBGPIO_IC_INTEN, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_INTMASK, ~0);
	apb_write32(ddata->base, DWAPBGPIO_IC_PORTA_EOI, ~0);
	apb_write32(ddata->base, DWAPBGPIO_IC_INT_POLARITY, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_INTTYPE_LEVEL, 0);
	apb_write32(ddata->base, DWAPBGPIO_IC_DEBOUNCE, 0);
}

static int dwapbgpio_probe(struct amba_device *dev, struct amba_id *id)
{
	int	ret;
	struct dwapbgpio_drv_data *ddata;
	struct dwapbgpio_block_data *bdata = dev->dev.platform_data;
	unsigned i;
	static  int nr_gpio = 0;

	if (unlikely(!bdata))
		return -EINVAL;

	ddata = kzalloc(sizeof(struct dwapbgpio_drv_data), GFP_KERNEL);
	if (unlikely(!ddata))
		return -ENOMEM;
	ddata->dev = dev;
	ddata->bdata = bdata;

	amba_set_drvdata(dev, ddata);

	ddata->base =
		ioremap(dev->res.start, dev->res.end - dev->res.start + 1);
	if (unlikely(!ddata->base)) {
		dev_err(&dev->dev,
				"%s: unable to remap memory "
				"(address=0x%08x, size=%d)\n",
			__func__, dev->res.start,
			dev->res.end - dev->res.start + 1);
		ret = -EBUSY;
		goto err_ioremap;
	}

	dwapbgpio_reset(ddata);

	/* Configure GPIO framework */

	gpio_set_drvdata(&ddata->drv, ddata);

	strncpy(ddata->drv.name, dev_name(&dev->dev), GPIO_DRV_NAME_MAXLEN);
	if (bdata->group)
		strncpy(ddata->drv.group, bdata->group, GPIO_DRV_NAME_MAXLEN);
	ddata->drv.dev = &dev->dev;
	ddata->drv.no_dev_entries = no_dev_entries;
	ddata->drv.bank_size = bdata->port_a_count + bdata->port_b_count
		+ bdata->port_c_count + bdata->port_d_count;
	ddata->port_a_last = ddata->bdata->port_a_count;
	ddata->port_b_last = ddata->port_a_last + bdata->port_b_count;
	ddata->port_c_last = ddata->port_b_last + bdata->port_c_count;
	ddata->port_d_last = ddata->port_c_last + bdata->port_d_count;

	ddata->drv.chip.request = dwapbgpio_request;
	ddata->drv.chip.free = dwapbgpio_free;	
	ddata->drv.chip.direction_input = dwapbgpio_direction_input;
	ddata->drv.chip.direction_output = dwapbgpio_direction_output;
	ddata->drv.chip.get = dwapbgpio_get_value;
	ddata->drv.chip.set = dwapbgpio_set_value;

	ddata->drv.is_enabled = dwapbgpio_is_enabled;
	ddata->drv.get_direction = dwapbgpio_get_direction;
	ddata->drv.read_bank = dwapbgpio_read_bank;
	ddata->drv.write_bank = dwapbgpio_write_bank;
	ddata->drv.interrupt_is_active = dwapbgpio_interrupt_is_active;
	ddata->drv.interrupt_enable = dwapbgpio_interrupt_enable;
	ddata->drv.interrupt_is_enabled = dwapbgpio_interrupt_is_enabled;
	ddata->drv.interrupt_disable = dwapbgpio_interrupt_disable;
	ddata->drv.interrupt_clear = dwapbgpio_interrupt_clear;
	ddata->drv.interrupt_configure = dwapbgpio_interrupt_configure;
	ddata->drv.interrupt_get_config = dwapbgpio_interrupt_get_config;
	
	ddata->drv.chip.base = nr_gpio;
	ddata->drv.chip.dev = &ddata->dev->dev;
	ddata->drv.chip.ngpio = ddata->drv.bank_size;
	nr_gpio += ddata->drv.bank_size;
	
	gpiochip_add(&ddata->drv.chip);
	
	gpio_driver_alloc_bank(&ddata->drv, ddata->drv.bank_size);
	if (unlikely(!ddata->drv.bank)) {
		ret = -ENOMEM;
		goto err_alloc_bank;
	}

	for (i = 0; i < ddata->drv.bank_size; i++)
		/* We assign pin IDs here */
		gpio_driver_set_pininfo(&ddata->drv, i,
			i + bdata->gpio_id_offset,
			i < ddata->port_a_last ? dev->irq[0] : NO_IRQ);

	ret = gpio_driver_register(&ddata->drv);
	if (unlikely(ret)) {
		dev_err(&dev->dev, "%s: GPIO driver registration "
			"failed with error %d\n", __func__, ret);
		goto err_register;
	}

	dev_info(&dev->dev, "ready to control %d pins (offset %d)\n",
		ddata->drv.bank_size, bdata->gpio_id_offset);

	return 0;

err_register:
	kfree(ddata->drv.bank);
err_alloc_bank:
	iounmap(ddata->base);
err_ioremap:
	kfree(ddata);
	return ret;
}

static int dwapbgpio_remove(struct amba_device *dev)
{
	int 				ret;
	struct dwapbgpio_drv_data 	*ddata = amba_get_drvdata(dev);

	dev_info(&dev->dev, "removing device\n");


	gpiochip_remove(&ddata->drv.chip);
	ret = gpio_driver_unregister(&ddata->drv);
	if (unlikely(ret)) {
		dev_err(&dev->dev, "%s: GPIO driver unregistration failed "
			"with error %d, removing anyway\n", __func__, ret);
	}

	amba_set_drvdata(dev, NULL);

	kfree(ddata->drv.bank);
	iounmap(ddata->base);
	kfree(ddata);

	return ret;
}

#ifdef CONFIG_PM

static int dwapbgpio_suspend(struct amba_device *dev, pm_message_t msg)
{
	return 0;
}

static int dwapbgpio_resume(struct amba_device *dev)
{
	return 0;
}

#endif

/*-------------------------------------------------------------------------*/

static struct __initdata amba_id dwapbgpio_ids[] = {
	{
		.id	= DWAPBGPIO_ID,
		.mask	= DWAPBGPIO_ID_MASK,
	},
	{ 0, 0 },
};

static struct __initdata amba_driver dwapbgpio_drv = {
	.drv		= {
		.name	= DRIVER_NAME,
	},
	.probe		= dwapbgpio_probe,
	.remove		= dwapbgpio_remove,
#ifdef CONFIG_PM
	.suspend	= dwapbgpio_suspend,
	.resume		= dwapbgpio_resume,
#endif
	.id_table	= dwapbgpio_ids,
};

static int __init dwapbgpio_init(void)
{
	int ret;

	debug(DRIVER_DESC " compiled for Linux %s, version %s (%s at %s)",
		UTS_RELEASE, DRIVER_VERSION, __DATE__, __TIME__);

	ret = amba_driver_register(&dwapbgpio_drv);
	if (unlikely(ret))
		error("could not register AMBA driver");

	return ret;
}

static void __exit dwapbgpio_exit(void)
{
	amba_driver_unregister(&dwapbgpio_drv);
}


module_init(dwapbgpio_init);
module_exit(dwapbgpio_exit);

/*-------------------------------------------------------------------------*/

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
