/*
 * Cryptographic API.
 *
 * Support for Elliptic hardware crypto engine.
 *
 * Copyright (c) 2009  Stephan Lachowsky <stephan.lachowsky@maxim-ic.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/crypto.h>
#include <crypto/algapi.h>
#include <crypto/sha.h>

#include <mach/platform.h>
#include <mach/mobi_reset.h>
#include <mach/dma-list.h>

static int use_dma;
module_param(use_dma, int, 0);
MODULE_PARM_DESC(use_dma, "Enable DMA");

#define ELLIPTIC_CRA_PRIORITY 300
#define ELLIPTIC_COMPOSITE_PRIORITY 400

struct elliptic_sha_ctx {
	u64 total_length;
	u32 ctrl;
	u32 *ptr;
	u32 length;
	u32 digestsize;
};

/* Registers */

#define SHA_CTRL_REG				0x0000

#define SHA_CTRL_MSG_BEGIN	(1 << 2)
#define SHA_CTRL_MSG_END	(1 << 3)
#define SHA_CTRL_RET_CTX	(1 << 4)
#define SHA_CTRL_STR_CTX	(1 << 5)
#define SHA_CTRL_STR_2MSG	(1 << 6)
#define SHA_CTRL_SHA1_MODE	(4 << 7)
#define SHA_CTRL_SHA224_MODE	(0 << 7)
#define SHA_CTRL_SHA256_MODE	(1 << 7)

#define SHA_STATUS_REG				0x0004

#define SHA_STATUS_DONE		1

#define SHA_CTRL1_REG				0x0008

#define SHA_CTRL1_MSG_SIZE(s)	((s) << 0)		/* length of current block in bytes */
#define SHA_CTRL1_MSG_TOTAL(t)	((t) << 16)		/* total length of message in bytes */

#define SHA_CTRL2_REG				0x000C
#define SHA_IRQ_EN_REG				0x0018
#define SHA_IRQ_ACK_REG				0x001C
#define SHA_CTX_PAGE				0x0800
#define SHA_MSG_PAGE				0x2000

#define SHA_MSGMEM_SIZE		512

#define SHA_MAX_HW_PADDING_SIZE 0x10000

#define SHA_OP_TIMEOUT 0x50000

static void __iomem *_iobase;

#define SHA_READL(x) readl(_iobase + (x))
#define SHA_WRITEL(v, x) writel((v), _iobase + (x))

static inline void msg_update(struct elliptic_sha_ctx *ctx, const u8 *v, int len)
{
	int i;
	u32 d;
	u32 *p = ctx->ptr;

	if (ctx->length & 3) {
		i = 4 - (ctx->length & 3);
		i = i > len ? len : i;
		ctx->length += i;
		len -= i;
		d = *--p;
		d >>= 8*i;
		while (i--)
			d = (d<<8) | *v++;
		*p++ = d;
	}

	for (i = 0; i < len; i += 4, v += 4) {
		d = ((v[0] << 24) | (v[1] << 16) | (v[2] << 8) | (v[3] << 0));
		*p++ = d;
	}

	ctx->ptr = p;
	ctx->length += len;
}

static void wait_core(void)
{
	u32 status;
	u32 counter = SHA_OP_TIMEOUT;
	do
		status = SHA_READL(SHA_STATUS_REG);
	while (!(status & SHA_STATUS_DONE) && --counter);
}

static void elliptic_sha_init(struct crypto_tfm *tfm)
{
	struct elliptic_sha_ctx *ctx = crypto_tfm_ctx(tfm);
	ctx->ctrl = SHA_CTRL_SHA1_MODE | SHA_CTRL_MSG_BEGIN;
	ctx->ptr = _iobase + SHA_MSG_PAGE;
	ctx->length = 0;
	ctx->total_length = 0;
	ctx->digestsize = SHA1_DIGEST_SIZE;
}

static void elliptic_sha_update(struct crypto_tfm *tfm,
			const uint8_t *data, unsigned int length)
{
	struct elliptic_sha_ctx *ctx = crypto_tfm_ctx(tfm);

	while (ctx->length + length >= SHA_MSGMEM_SIZE) {
		unsigned int partial = SHA_MSGMEM_SIZE - ctx->length;
		msg_update(ctx, data, partial);

		ctx->ptr = _iobase + SHA_MSG_PAGE;
		ctx->length = 0;
		ctx->total_length += SHA_MSGMEM_SIZE;

		data += partial;
		length -= partial;

		SHA_WRITEL(SHA_MSGMEM_SIZE, SHA_CTRL1_REG);
		SHA_WRITEL(ctx->ctrl, SHA_CTRL_REG);
		wait_core();
		ctx->ctrl &= ~SHA_CTRL_MSG_BEGIN;
	}
	msg_update(ctx, data, length);
}

#define MEM_PHYS(ptr) (virt_to_phys((void *)(ptr)) | 0xF0000000)
#define SHA_PHYS(ptr) (SHA_BASE + ((unsigned)(ptr) - (unsigned)_iobase))
static inline dmac_lli_t *dma_data_node(dmac_lli_t *lli, void *data, void *sha, unsigned length)
{
	return dmac_lli_node(lli, MEM_PHYS(data), SHA_PHYS(sha), length);
}

static inline dmac_lli_t *dma_ctrl_node(dmac_lli_t *lli, dma_addr_t token, unsigned reg, unsigned length)
{
	return dmac_lli_node(lli, token, SHA_BASE + reg, length);
}

static void elliptic_sha_update_dma(struct crypto_tfm *tfm,
			const uint8_t *data, unsigned int length)
{
	struct elliptic_sha_ctx *ctx = crypto_tfm_ctx(tfm);
	dmac_lli_t *lli = NULL;
	dma_addr_t base, ctrl_token;

	/* handle unaligned in pio path */
	if ((ctx->length & 3) || ((unsigned)data & 3))
		return elliptic_sha_update(tfm, data, length);

	dmac_cache_reset();
	ctrl_token = dmac_cache_value(ctx->ctrl);

	base = dma_map_single(NULL, data, length, DMA_TO_DEVICE);

	while (ctx->length + length >= SHA_MSGMEM_SIZE) {
		unsigned int partial = SHA_MSGMEM_SIZE - ctx->length;
		/* data copy */
		lli = dma_data_node(lli, data, ctx->ptr, partial);

		ctx->ptr = _iobase + SHA_MSG_PAGE;
		ctx->length = 0;
		ctx->total_length += SHA_MSGMEM_SIZE;

		data += partial;
		length -= partial;

		/* ctrl node */
		lli = dma_ctrl_node(lli, ctrl_token, SHA_CTRL_REG, sizeof(u32));

		if (ctx->ctrl & SHA_CTRL_MSG_BEGIN) {
			ctx->ctrl &= ~SHA_CTRL_MSG_BEGIN;
			ctrl_token = dmac_cache_value(ctx->ctrl);
		}
	}
	/* data copy */
	if (length) {
		lli = dma_data_node(lli, data, ctx->ptr, (length+3)&~3);
		ctx->ptr += (length+3)/4;
		ctx->length += length;
	}

	SHA_WRITEL(SHA_MSGMEM_SIZE, SHA_CTRL1_REG);

	/* start dma */
	dmac_process_list();
	/* prevent race condition between core and cpu */
	wait_core();

	dma_unmap_single(NULL, base, length, DMA_TO_DEVICE);
}

static void elliptic_sha_final(struct crypto_tfm *tfm, uint8_t *out)
{
	struct elliptic_sha_ctx *ctx = crypto_tfm_ctx(tfm);
	u64 total_length = ctx->total_length + ctx->length;
	u32 ctrl, ctrl1;
	u32 i;

	if (total_length < SHA_MAX_HW_PADDING_SIZE) {
		/* For messages of this length we can have hardware do the
		 * padding */
		ctrl1 = SHA_CTRL1_MSG_TOTAL(total_length) |
			SHA_CTRL1_MSG_SIZE(ctx->length);
		ctrl = ctx->ctrl | SHA_CTRL_STR_CTX | SHA_CTRL_MSG_END;
	} else {
		u32 index, padlen;
		__be64 bits;
		static const u8 padding[64] = { 0x80, };

		bits = cpu_to_be64(total_length << 3);

		/* Pad out to 56 mod 64 */
		index = ctx->length & 0x3f;
		padlen = (index < 56) ? (56 - index) : ((64+56) - index);
		elliptic_sha_update(tfm, padding, padlen);

		/* Append length */
		elliptic_sha_update(tfm, (const u8 *)&bits, sizeof(bits));

		/* Do not enable the hardware padding */
		ctrl1 = SHA_CTRL1_MSG_SIZE(ctx->length);
		ctrl = ctx->ctrl | SHA_CTRL_STR_CTX;
	}

	SHA_WRITEL(ctrl1, SHA_CTRL1_REG);
	SHA_WRITEL(ctrl, SHA_CTRL_REG);
	wait_core();
	for (i = 0; i < ctx->digestsize/4; i++)
		((u32 *)out)[i] = be32_to_cpu(SHA_READL(SHA_CTX_PAGE + 8*i));

	/* Wipe context */
	memset(ctx, 0, sizeof *ctx);
}

static struct crypto_alg sha1_alg = {
	.cra_name		=	"sha1",
	.cra_driver_name	=	"sha1-elliptic",
	.cra_priority		=	ELLIPTIC_CRA_PRIORITY,
	.cra_flags		=	CRYPTO_ALG_TYPE_DIGEST,
	.cra_blocksize		=	SHA1_BLOCK_SIZE,
	.cra_ctxsize		=	sizeof(struct elliptic_sha_ctx),
	.cra_module		=	THIS_MODULE,
	.cra_alignmask          =       3,
	.cra_list		=	LIST_HEAD_INIT(sha1_alg.cra_list),
	.cra_u			=	{
		.digest = {
			.dia_digestsize	=	SHA1_DIGEST_SIZE,
			.dia_init	=	elliptic_sha_init,
			.dia_update	=	elliptic_sha_update,
			.dia_final 	=	elliptic_sha_final,
		}
	}
};

static void elliptic_sha256_init(struct crypto_tfm *tfm)
{
	struct elliptic_sha_ctx *ctx = crypto_tfm_ctx(tfm);
	ctx->ctrl = SHA_CTRL_SHA256_MODE | SHA_CTRL_MSG_BEGIN;
	ctx->ptr = _iobase + SHA_MSG_PAGE;
	ctx->length = 0;
	ctx->total_length = 0;
	ctx->digestsize = SHA256_DIGEST_SIZE;
}

static struct crypto_alg sha256_alg = {
	.cra_name		=	"sha256",
	.cra_driver_name	=	"sha256-elliptic",
	.cra_priority		=	ELLIPTIC_CRA_PRIORITY,
	.cra_flags		=	CRYPTO_ALG_TYPE_DIGEST,
	.cra_blocksize		=	SHA256_BLOCK_SIZE,
	.cra_ctxsize		=	sizeof(struct elliptic_sha_ctx),
	.cra_module		=	THIS_MODULE,
	.cra_list		=	LIST_HEAD_INIT(sha256_alg.cra_list),
	.cra_u			=	{
		.digest = {
			.dia_digestsize	=	SHA256_DIGEST_SIZE,
			.dia_init   	= 	elliptic_sha256_init,
			.dia_update 	=	elliptic_sha_update,
			.dia_final  	=	elliptic_sha_final,
		}
	}
};

static int __init elliptic_init(void)
{
	int ret = 0;

	mobi_reset_disable(RESET_ID_SHA);

	_iobase = ioremap(SHA_BASE, SHA_SIZE);

	if (_iobase == NULL) {
		ret = -ENOMEM;
		goto out;
	}

	SHA_WRITEL(0, SHA_CTRL2_REG);
	SHA_WRITEL(0, SHA_IRQ_EN_REG);

	if (use_dma) {
		if ((ret = dmac_alloc("elliptic-sha")) < 0)
			goto dma_err;
		dmac_set_handshake(DMAC_HANDSHAKE_SHA);

		sha1_alg.cra_digest.dia_update = elliptic_sha_update_dma;
		sha256_alg.cra_digest.dia_update = elliptic_sha_update_dma;
	}

	if ((ret = crypto_register_alg(&sha1_alg)))
		goto sha1_err;

	if ((ret = crypto_register_alg(&sha256_alg)))
		goto sha256_err;
out:
	return ret;

	crypto_unregister_alg(&sha256_alg);
sha256_err:
	crypto_unregister_alg(&sha1_alg);
sha1_err:
	if (use_dma)
		dmac_free();
dma_err:
	iounmap(_iobase);
	goto out;
}

static void __exit elliptic_exit(void)
{
	crypto_unregister_alg(&sha256_alg);
	crypto_unregister_alg(&sha1_alg);
	if (use_dma)
		dmac_free();
	iounmap(_iobase);
}

MODULE_AUTHOR("Stephan Lachowsky");
MODULE_DESCRIPTION("Elliptic Hardware SHA driver");
MODULE_LICENSE("GPL");

module_init(elliptic_init);
module_exit(elliptic_exit);
