/*
 * Cryptographic API.
 *
 * Support for Elliptic hardware crypto engine.
 *
 * Copyright (c) 2009  Stephan Lachowsky <stephan.lachowsky@maxim-ic.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/crypto.h>
#include <crypto/algapi.h>
#include <crypto/aes.h>

#include <mach/platform.h>
#include <mach/mobi_reset.h>
#include <mach/dma-list.h>

static int use_dma;
module_param(use_dma, int, 0);
MODULE_PARM_DESC(use_dma, "Enable DMA");

#define ELLIPTIC_CRA_PRIORITY 300
#define ELLIPTIC_COMPOSITE_PRIORITY 400

struct elliptic_aes_ctx {
	u32 ctrl;
	int key_len;
	u8 key[AES_MAX_KEY_SIZE];
};

/* Register definitions */

#define AES_CTRL_REG    0x0000

#define AES_CTRL_KEYSZ128 (0)
#define AES_CTRL_KEYSZ192 (1)
#define AES_CTRL_KEYSZ256 (2)
#define AES_CTRL_KEYSZMASK (3)

#define AES_CTRL_DECRYPT  (0<<2)
#define AES_CTRL_ENCRYPT  (1<<2)

#define AES_CTRL_ECB_MODE (0)
#define AES_CTRL_CBC_MODE (1<<3)
#define AES_CTRL_CTR_MODE (1<<4)

#define AES_CTRL_STR_IV      (1<<5)
#define AES_CTRL_RET_IV      (1<<6)
#define AES_CTRL_RET_INV_KEY (1<<7)
#define AES_CTRL_STR_INV_KEY (1<<8)
#define AES_CTRL_RET_FWD_KEY (1<<9)

#define AES_STATUS_REG  0x0004

#define AES_STATUS_DONE      (1<<0)
#define AES_STATUS_CCM_VALID (1<<8)

#define AES_CTRL1_REG   0x0008

#define AES_CTRL1_AAD_BLOCKS(s) ((s)<<0)
#define AES_CTRL1_TAG_SIZE(s)   ((s)<<9)
#define AES_CTRL1_MSG_SIZE(s)   ((s)<<16)
#define AES_CTRL1_MSG_END       (1<<29)
#define AES_CTRL1_MSG_BEGIN     (1<<30)
#define AES_CTRL1_CCM_MODE      (1<<31)

#define AES_IRQ_EN_REG  0x000C
#define AES_IRQ_ACK_REG 0x0010

#define AES_CTX_PAGE    0x0100
#define AES_MSG_PAGE    0x2000

#define AES_CTX_PAGE_CTR (AES_CTX_PAGE + 0x30)
#define AES_CTX_PAGE_IV  (AES_CTX_PAGE + 0x40)
#define AES_CTX_PAGE_MAC (AES_CTX_PAGE + 0x50)

#define AES_MSGMEM_SIZE 512

/*  A very large counter that is used to gracefully bail out of an
 *  operation in case of trouble
 */

#define AES_OP_TIMEOUT    0x50000

/* Static structures */

static void __iomem *_iobase;

#define AES_READL(x) readl(_iobase + (x))
#define AES_WRITEL(v, x) writel((v), _iobase + (x))

/* Write a field (either a writable key or IV or message) */
static inline void _writefield(u32 offset, const void *value, int len)
{
	int i;
	for (i = 0; i < (len+3)/4; i++)
		AES_WRITEL(swab32(((u32 *) value)[i]), offset + (i * 4));
}

/* Read a field (either a writable key or IV or message) */
static inline void _readfield(u32 offset, void *value, int len)
{
	int i;
	for (i = 0; i < (len+3)/4; i++)
		((u32 *) value)[i] = swab32(AES_READL(offset + (i * 4)));
}

static inline void set_ctx_key(struct elliptic_aes_ctx *ctx)
{
	_writefield(AES_CTX_PAGE, ctx->key, ctx->key_len);
}
static inline void elliptic_set_iv(u8 *iv)
{
	_writefield(AES_CTX_PAGE_IV, iv, AES_BLOCK_SIZE);
}
static inline void elliptic_get_iv(u8 *iv)
{
	_readfield(AES_CTX_PAGE_IV, iv, AES_BLOCK_SIZE);
}

static inline void wait_core(void)
{
	u32 status;
	u32 counter = AES_OP_TIMEOUT;
	do
		status = AES_READL(AES_STATUS_REG);
	while (!(status & AES_STATUS_DONE) && --counter);
}

static void elliptic_xcrypt(const void *src, void *dst, int len, u32 ctrl)
{
	u32 ctrl1 = AES_CTRL1_MSG_BEGIN | AES_CTRL1_MSG_END | AES_CTRL1_MSG_SIZE(len);

	_writefield(AES_MSG_PAGE, src, len);

	AES_WRITEL(ctrl1, AES_CTRL1_REG);

	/* Start the operation */
	AES_WRITEL(ctrl, AES_CTRL_REG);

	wait_core();

	_readfield(AES_MSG_PAGE, dst, len);
}

static int elliptic_setkey(struct crypto_tfm *tfm, const u8 *key, unsigned int len)
{
	struct elliptic_aes_ctx *ctx = crypto_tfm_ctx(tfm);

	ctx->ctrl &= ~AES_CTRL_KEYSZMASK;

	switch (len) {
	case 128/8:
		ctx->ctrl |= AES_CTRL_KEYSZ128;
		break;
	case 192/8:
		ctx->ctrl |= AES_CTRL_KEYSZ192;
		break;
	case 256/8:
		ctx->ctrl |= AES_CTRL_KEYSZ256;
		break;
	default:
		tfm->crt_flags |= CRYPTO_TFM_RES_BAD_KEY_LEN;
		return -EINVAL;
	}

	memcpy(ctx->key, key, len);
	ctx->key_len = len;

	if (use_dma)
		dma_cache_maint(ctx->key, len, DMA_TO_DEVICE);

	return 0;
}

static void elliptic_encrypt(struct crypto_tfm *tfm, u8 *out, const u8 *in)
{
	struct elliptic_aes_ctx *ctx = crypto_tfm_ctx(tfm);

	set_ctx_key(ctx);

	elliptic_xcrypt(in, out, AES_BLOCK_SIZE,
		ctx->ctrl | AES_CTRL_ENCRYPT);
}

static void elliptic_decrypt(struct crypto_tfm *tfm, u8 *out, const u8 *in)
{
	struct elliptic_aes_ctx *ctx = crypto_tfm_ctx(tfm);

	set_ctx_key(ctx);

	elliptic_xcrypt(in, out, AES_BLOCK_SIZE,
		ctx->ctrl | AES_CTRL_DECRYPT);
}

static inline dma_addr_t virt_to_dmac(void *addr)
{
	return virt_to_dma(NULL, addr) | 0xF0000000;
}
static inline dma_addr_t dmac_map_page(struct page *page,
		unsigned long offset, size_t size,
		enum dma_data_direction dir)
{
	void *addr = page_address(page)+offset;
	dma_cache_maint(addr, size, dir);
	return virt_to_dmac(addr);
}

static int elliptic_blk_xcrypt_dma(struct blkcipher_desc *desc,
				struct scatterlist *dst, struct scatterlist *src,
				unsigned int nbytes, u32 ctrl)
{
	struct elliptic_aes_ctx *ctx = crypto_blkcipher_ctx(desc->tfm);
	struct blkcipher_walk walk;
	int err;
	dmac_lli_t *lli;
	dma_addr_t ctrl1_token;
	dma_addr_t ctrl_token;
	u32 ctrl1_last = 0;

	dmac_cache_reset();
	ctrl_token = dmac_cache_value(ctrl);

	lli = dmac_lli_node(NULL, virt_to_dmac(ctx->key), AES_BASE + AES_CTX_PAGE, ctx->key_len);

	blkcipher_walk_init(&walk, dst, src, nbytes);
	err = blkcipher_walk_phys(desc, &walk);

	if (ctrl & AES_CTRL_RET_IV) {
		dma_cache_maint(walk.iv, AES_BLOCK_SIZE, DMA_BIDIRECTIONAL);
		lli = dmac_lli_node(lli, virt_to_dmac(walk.iv), AES_BASE + AES_CTX_PAGE_IV, AES_BLOCK_SIZE);
	}

	while ((nbytes = walk.nbytes)) {
		unsigned int count = (nbytes > AES_MSGMEM_SIZE) ?  AES_MSGMEM_SIZE : nbytes & ~(AES_BLOCK_SIZE-1);
		u32 ctrl1 = AES_CTRL1_MSG_BEGIN | AES_CTRL1_MSG_END | AES_CTRL1_MSG_SIZE(count);
		dma_addr_t src = dmac_map_page(walk.src.phys.page, walk.src.phys.offset, count, DMA_TO_DEVICE);
		dma_addr_t dst = dmac_map_page(walk.dst.phys.page, walk.dst.phys.offset, count, DMA_FROM_DEVICE);

		lli = dmac_lli_node(lli, src, AES_BASE + AES_MSG_PAGE, count);
		if (ctrl1 != ctrl1_last) {
			ctrl1_token = dmac_cache_value(ctrl1);
			ctrl1_last = ctrl1;
			lli = dmac_lli_node(lli, ctrl1_token, AES_BASE + AES_CTRL1_REG, sizeof(u32));
		}
		lli = dmac_lli_node(lli, ctrl_token, AES_BASE + AES_CTRL_REG, sizeof(u32));
		lli = dmac_lli_node(lli, AES_BASE + AES_MSG_PAGE, dst, count);

		if (walk.flags & 6) {
			/* we must process now when using an ephemeral alignment buffer*/
			dmac_process_list();
			lli = NULL;
		}

		err = blkcipher_walk_done(desc, &walk, nbytes-count);
	}

	if (ctrl & AES_CTRL_STR_IV)
		lli = dmac_lli_node(lli, AES_BASE + AES_CTX_PAGE_IV, virt_to_dmac(walk.iv), AES_BLOCK_SIZE);

	dmac_process_list();

	return err;
}

static int elliptic_blk_xcrypt(struct blkcipher_desc *desc,
				struct scatterlist *dst, struct scatterlist *src,
				unsigned int nbytes, u32 ctrl)
{
	struct elliptic_aes_ctx *ctx = crypto_blkcipher_ctx(desc->tfm);
	struct blkcipher_walk walk;
	int err;

	set_ctx_key(ctx);

	blkcipher_walk_init(&walk, dst, src, nbytes);
	err = blkcipher_walk_virt(desc, &walk);

	if (ctrl & AES_CTRL_RET_IV)
		elliptic_set_iv(walk.iv);

	while ((nbytes = walk.nbytes)) {
		unsigned int count = (nbytes > AES_MSGMEM_SIZE) ?  AES_MSGMEM_SIZE : nbytes & ~(AES_BLOCK_SIZE-1);
		elliptic_xcrypt(walk.src.virt.addr, walk.dst.virt.addr, count, ctrl);
		err = blkcipher_walk_done(desc, &walk, nbytes-count);
	}

	if (ctrl & AES_CTRL_STR_IV)
		elliptic_get_iv(walk.iv);

	return err;
}

static int elliptic_blk_encrypt(struct blkcipher_desc *desc,
				struct scatterlist *dst, struct scatterlist *src,
				unsigned int nbytes)
{
	struct elliptic_aes_ctx *ctx = crypto_blkcipher_ctx(desc->tfm);
	if (use_dma)
		return elliptic_blk_xcrypt_dma(desc, dst, src, nbytes, ctx->ctrl | AES_CTRL_ENCRYPT);
	return elliptic_blk_xcrypt(desc, dst, src, nbytes, ctx->ctrl | AES_CTRL_ENCRYPT);
}

static int elliptic_blk_decrypt(struct blkcipher_desc *desc,
				struct scatterlist *dst, struct scatterlist *src,
				unsigned int nbytes)
{
	struct elliptic_aes_ctx *ctx = crypto_blkcipher_ctx(desc->tfm);
	if (use_dma)
		return elliptic_blk_xcrypt_dma(desc, dst, src, nbytes, ctx->ctrl | AES_CTRL_DECRYPT);
	return elliptic_blk_xcrypt(desc, dst, src, nbytes, ctx->ctrl | AES_CTRL_DECRYPT);
}

static int elliptic_ecb_init(struct crypto_tfm *tfm)
{
	struct elliptic_aes_ctx *ctx = crypto_tfm_ctx(tfm);
	ctx->ctrl = AES_CTRL_ECB_MODE | AES_CTRL_RET_FWD_KEY;
	return 0;
}

static int elliptic_cbc_init(struct crypto_tfm *tfm)
{
	struct elliptic_aes_ctx *ctx = crypto_tfm_ctx(tfm);
	ctx->ctrl = AES_CTRL_CBC_MODE | AES_CTRL_RET_IV | AES_CTRL_STR_IV | AES_CTRL_RET_FWD_KEY;
	return 0;
}

static struct crypto_alg aes_alg = {
	.cra_name               =	"aes",
	.cra_driver_name	=	"aes-elliptic",
	.cra_priority           =	ELLIPTIC_CRA_PRIORITY,
	.cra_alignmask          =	3,
	.cra_flags		=	CRYPTO_ALG_TYPE_CIPHER,
	.cra_blocksize		=	AES_BLOCK_SIZE,
	.cra_ctxsize		=	sizeof(struct elliptic_aes_ctx),
	.cra_module		=	THIS_MODULE,
	.cra_list		=	LIST_HEAD_INIT(aes_alg.cra_list),
	.cra_init		=	elliptic_ecb_init,
	.cra_u			= {
		.cipher = {
			.cia_min_keysize	=	AES_MIN_KEY_SIZE,
			.cia_max_keysize	=	AES_MAX_KEY_SIZE,
			.cia_setkey		=	elliptic_setkey,
			.cia_encrypt		=	elliptic_encrypt,
			.cia_decrypt		=	elliptic_decrypt
		}
	}
};

static struct crypto_alg ecb_aes_alg = {
	.cra_name		=	"ecb(aes)",
	.cra_driver_name	=	"ecb-aes-elliptic",
	.cra_priority		=	ELLIPTIC_COMPOSITE_PRIORITY,
	.cra_flags		=	CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize		=	AES_BLOCK_SIZE,
	.cra_ctxsize		=	sizeof(struct elliptic_aes_ctx),
	.cra_alignmask		=	3,
	.cra_type		=	&crypto_blkcipher_type,
	.cra_module		=	THIS_MODULE,
	.cra_list		=	LIST_HEAD_INIT(ecb_aes_alg.cra_list),
	.cra_init		=	elliptic_ecb_init,
	.cra_u			=	{
		.blkcipher = {
			.min_keysize		=	AES_MIN_KEY_SIZE,
			.max_keysize		=	AES_MAX_KEY_SIZE,
			.setkey			=	elliptic_setkey,
			.encrypt		=	elliptic_blk_encrypt,
			.decrypt		=	elliptic_blk_decrypt,
		}
	}
};

static struct crypto_alg cbc_aes_alg = {
	.cra_name		=	"cbc(aes)",
	.cra_driver_name	=	"cbc-aes-elliptic",
	.cra_priority		=	ELLIPTIC_COMPOSITE_PRIORITY,
	.cra_flags		=	CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize		=	AES_BLOCK_SIZE,
	.cra_ctxsize		=	sizeof(struct elliptic_aes_ctx),
	.cra_alignmask		=	3,
	.cra_type		=	&crypto_blkcipher_type,
	.cra_module		=	THIS_MODULE,
	.cra_list		=	LIST_HEAD_INIT(cbc_aes_alg.cra_list),
	.cra_init		=	elliptic_cbc_init,
	.cra_u			=	{
		.blkcipher = {
			.min_keysize		=	AES_MIN_KEY_SIZE,
			.max_keysize		=	AES_MAX_KEY_SIZE,
			.ivsize			=	AES_BLOCK_SIZE,
			.setkey			=	elliptic_setkey,
			.encrypt		=	elliptic_blk_encrypt,
			.decrypt		=	elliptic_blk_decrypt,
		}
	}
};

static int __init elliptic_aes_init(void)
{
	int ret;

	mobi_reset_disable(RESET_ID_AES);

	_iobase = ioremap(AES_BASE, AES_SIZE);

	if (_iobase == NULL) {
		ret = -ENOMEM;
		goto out;
	}

	AES_WRITEL(0, AES_IRQ_EN_REG);

	if (use_dma) {
		if ((ret = dmac_alloc("elliptic-aes")) < 0)
			goto dma_err;
		dmac_set_handshake(DMAC_HANDSHAKE_AES);
	}

	if ((ret = crypto_register_alg(&aes_alg)))
		goto aes_err;

	if ((ret = crypto_register_alg(&ecb_aes_alg)))
		goto ecb_aes_err;

	if ((ret = crypto_register_alg(&cbc_aes_alg)))
		goto cbc_aes_err;

out:
	return ret;

	crypto_unregister_alg(&cbc_aes_alg);
cbc_aes_err:
	crypto_unregister_alg(&ecb_aes_alg);
ecb_aes_err:
	crypto_unregister_alg(&aes_alg);
aes_err:
	if (use_dma)
		dmac_free();
dma_err:
	iounmap(_iobase);
	goto out;
}

static void __exit elliptic_aes_exit(void)
{
	crypto_unregister_alg(&cbc_aes_alg);
	crypto_unregister_alg(&ecb_aes_alg);
	crypto_unregister_alg(&aes_alg);
	if (use_dma)
		dmac_free();
	iounmap(_iobase);
}

MODULE_AUTHOR("Stephan Lachowsky");
MODULE_DESCRIPTION("Elliptic Hardware AES driver");
MODULE_LICENSE("GPL");

module_init(elliptic_aes_init);
module_exit(elliptic_aes_exit);
