/*
 *  linux/drivers/char/qcc_arm.c
 *
 *  Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <asm/uaccess.h>

#include <linux/qccdev.h>

#include <mach/platform.h>
#include <mach/mobi_qcc.h>

#endif

#define DEV_NAME "qcc"

static struct class *dev_class = NULL;
int major;
static struct cdev cdev = {
    .kobj   =   {
		.name = DEV_NAME, 
	},
    .owner  =   THIS_MODULE,
};

#define MG_QCC_DEBUG 0

#if MG_QCC_DEBUG
#define dprintk(x...) printk(x);
#else
#define dprintk(x...) 
#endif

static int qcc_ioctl(struct inode *inode, struct file *file,
        unsigned int cmd, unsigned long arg)
{
	switch(cmd) {
	case QCCIOC_READ:
		{
			struct qcc_access read_cmd;
			copy_from_user(&read_cmd, 
					(char*)arg, sizeof(struct qcc_access));
			mobi_qcc_read(read_cmd.blockID, 
					read_cmd.addr, &(read_cmd.data), read_cmd.size);
			copy_to_user((char*)arg, &read_cmd, sizeof(struct qcc_access));
		}
		break;
	case QCCIOC_WRITE:
		{
			struct qcc_access write_cmd;
			copy_from_user(&write_cmd, 
					(char*)arg, sizeof(struct qcc_access));
			mobi_qcc_write(write_cmd.blockID, 
					write_cmd.addr, write_cmd.data, write_cmd.size);
		}
		break;
	case QCCIOC_WRITE_BLOCK:
		{
			struct qcc_write_block qwb;
			struct qcc_cmd_def *qcmd, *qcmd_head;
			unsigned long data;
			int i;

			copy_from_user(&qwb, (char*)arg, sizeof(struct qcc_write_block));

			qcmd = kmalloc(qwb.entries * sizeof(struct qcc_cmd_def), GFP_KERNEL);
			if (!qcmd)
				return -ENOMEM;
			qcmd_head = qcmd;

			copy_from_user(qcmd, 
					(struct qcc_cmd_def *)qwb.cmds,
					qwb.entries * sizeof(struct qcc_cmd_def));

			/* 
			 * would be better to have qcc api that does this so
			 * we don't have to switch back and forth....
			 */
			for (i=0; i < qwb.entries; i++, qcmd_head++) {
				/*
				printk("%s %2d, 0x%04x, %d, 0x%08x, 0x%08x\n",
						(qcmd_head->cmd == QCC_READ ? "read" : "write"),
						qcmd_head->blockID, qcmd_head->addr, 
						qcmd_head->size, qcmd_head->data, qcmd_head->mask);
				*/
				if (qcmd_head->cmd == QCC_WRITE)
					mobi_qcc_write(qcmd_head->blockID, 
							qcmd_head->addr, qcmd_head->data, qcmd_head->size);
				else if (qcmd_head->cmd == QCC_READ)
					mobi_qcc_read(qcmd_head->blockID, 
							qcmd_head->addr, &data, qcmd_head->size);

			}
			kfree(qcmd);
		}
		break;
	}
	return 0;
}

/* we don't have anything to do! */
static int qcc_open(struct inode *inode, struct file *filp)
{
	return 0;
}

static const struct file_operations dev_fops = {
    .owner  = THIS_MODULE,
    .llseek = no_llseek,
    .open   = qcc_open,
    .ioctl  = qcc_ioctl,
};

static int __devinit qcc_probe(struct platform_device *dev)
{
	dev_t devt_id = 0;
	int err = 0;

	/* 1 device, minor 0 */
	dev_class = class_create(THIS_MODULE, DEV_NAME);
	if (dev_class == NULL)
		return -EIO;

	err = alloc_chrdev_region(&devt_id, 0, 1, DEV_NAME);
	if (err) {
		class_destroy(dev_class);
		return err;
	}

	major = MAJOR(devt_id);

	cdev_init(&cdev, &dev_fops);
	err = cdev_add(&cdev, devt_id, 1);
	if (err) {
		unregister_chrdev_region(devt_id, 1);
		class_destroy(dev_class);
	}
	else {
		device_create(dev_class, NULL, devt_id, NULL, DEV_NAME);
	}

	return err;
}

static int __devexit qcc_remove(struct platform_device *pdev)
{
	dev_t devt_id = MKDEV(major, 0);

	device_destroy(dev_class, devt_id);
	class_destroy(dev_class);
	cdev_del(&cdev);
	unregister_chrdev_region(devt_id, 1);

	return 0;
}

static void qcc_shutdown(struct platform_device *dev) { }

#ifdef CONFIG_PM
static int qcc_suspend(struct platform_device *dev, pm_message_t state)
{
	return 0;
}

static int qcc_resume(struct platform_device *dev)
{
	return 0;
}
#else 
#define qcc_suspend NULL
#define qcc_resume  NULL
#endif

static struct platform_driver qcc_driver = {
	.probe		= qcc_probe,
	.remove		= qcc_remove,
	.suspend	= qcc_suspend,
	.resume		= qcc_resume,
	.shutdown	= qcc_shutdown,
	.driver		= {
		.name	= PLATFORM_NAME_QCC,
	},
};

static int __init qcc_init(void)
{
	return platform_driver_register(&qcc_driver);
}

static void __exit qcc_exit(void)
{
	platform_driver_unregister(&qcc_driver);
}
module_init(qcc_init);
module_exit(qcc_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Maxim qcc device driver");
