/*
 *  drivers/char/falcon_ppc_pci.c
 *
 *  Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/errno.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/major.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/pci.h>
#include <linux/spinlock.h>
#include <linux/uaccess.h>
#include <linux/debugfs.h>

/* shared resources between ARM and PPC */
#include <linux/falcon/qcc_defs.h>
#include <linux/falcon/qcc_regs.h>
#include <linux/falcon/qcc_chipctl_regs.h>
#include <linux/falcon/pci_bar0_offsets.h>
#include <linux/falcon/pci_bar2_offsets.h>

/* userspace interface */
#include <linux/qccdev.h>
#include <linux/fmem.h>

#define DRV_NAME "falcon"
#define PFX "FALCON-PCI "
#define DEV_NAME "falcon"

#define AHB_BAR0_TRANS_MASK	0xFFFC0000
#define AHB_BAR0_TRANS_ADDR	0xB8000000
#define AHB_BAR1_TRANS_MASK	0xFF000000
#define AHB_BAR1_LIN_TRANS_ADDR	0x00000000
/* special mode */
#define AHB_BAR1_PIO_TRANS_ADDR	0xB9000000

static struct class *dev_class;
static int major;
static struct cdev cdev = {
    .kobj   =   {
		.name = DEV_NAME,
	},
    .owner  =   THIS_MODULE,
};

static struct pci_falcondev *falcon;

struct pci_falcondev  {
	struct pci_dev *pdev;
	uint16_t vendor_id;
	uint16_t device_id;
	void __iomem *bar0_base;
	uint32_t bar0_size;
	void __iomem *bar1_base;
	uint32_t bar1_size;
	void __iomem *bar2_base;
	uint32_t bar2_size;

	uint8_t irq;

	void __iomem *bar0_qcc_base;
	void __iomem *bar0_pcie_phy_base;
	void __iomem *bar0_video_data_base;
	void __iomem *bar0_axi_mmu_base;
	void __iomem *bar0_video_regs_base;
	void __iomem *bar0_pcie_regs_base;
};

static struct bidtable_control bidtable_ctrl;

static struct dentry *d_fpci_debug;
static int8_t __attribute__((unused)) dbg_loglevel;

#define LOCAL_FALCON_PCI_DEBUG_ENABLE

#if defined(LOCAL_FALCON_PCI_DEBUG_ENABLE) || \
	defined(CONFIG_FALCON_PCI_DEBUG)

#define FALCON_PCI_DEBUG 1
#else
#define FALCON_PCI_DEBUG 0
#endif

#if FALCON_PCI_DEBUG
module_param(dbg_loglevel, byte, 0644);
MODULE_PARM_DESC(dbg_loglevel, "Set module debug level(0-5)");

#define DBGPFX "FPCI:"
#define falconpci_debug(n, fmt, args...)	\
	do { 						\
		if (dbg_loglevel >= n) { 	\
			printk(DBGPFX"%s: " fmt, __func__, ##args); \
		}						\
	} while (0)

#else
#define falconpci_debug(n, fmt, args...)	do {} while (0)
#endif

#define dprintk(x...)	printk(x)
#define dprintk1(fmt, x...)	falconpci_debug(1, fmt, ##x)
#define dprintk2(fmt, x...)	falconpci_debug(2, fmt, ##x)
#define dprintk3(fmt, x...)	falconpci_debug(3, fmt, ##x)
#define dprintk4(fmt, x...)	falconpci_debug(4, fmt, ##x)
#define dprintk5(fmt, x...)	falconpci_debug(5, fmt, ##x)

#if 1
#define fwritel(data, addr) \
	writel(data, addr)
#define fwritew(data, addr) \
	writew(data, addr)
#define fwriteb(data, addr) \
	writeb(data, addr)

#define freadl(addr) \
	readl(addr)
#define freadw(addr) \
	readw(addr)
#define freadb(addr) \
	readb(addr)

#else

/* XXX um, all the code that is accessing registers is using these
 * functions, need to figure out what is correct ????
 */
#define fwritel(data, addr) \
	out_be32(addr, data)
#define fwritew(data, addr) \
	out_be16(addr, data)
#define fwriteb(data, addr) \
	out_8(addr, data)

#define freadl(addr) \
	in_be32(addr)
#define freadw(addr) \
	in_be16(addr)
#define freadb(addr) \
	in_8(addr)

#endif

/*
 * we have 7 bidtable slots so were are going to have to lock the
 * table when using it otherwise we may be writing data to the wrong
 * place!
 */
spinlock_t bidtable_lock;
#define BID_TABLE_ENTRY_MASK ((uint64_t)0x00000000000000ffULL)

/*
 * lock an entry in the bidtable so it cannot be replaced
 * caller must be holding bidtable_lock
 */
static int32_t lock_bidtable_entry(uint8_t entry)
{
	if (hweight8(bidtable_ctrl.locked_entry_bitmask) <
			BIDTABLE_MAX_LOCKED_ENTRIES) {
		bidtable_ctrl.locked_entry_bitmask |= (0x1 << entry);
		dprintk1("locking bidtable entry %d\n", entry);
	} else {
		return -1;
	}

	return 0;
}
static void unlock_bidtable_entry(uint8_t entry)
{
	bidtable_ctrl.locked_entry_bitmask &= ~(0x1 << entry);
	dprintk1("unlocking bidtable entry %d\n", entry);
}
/*
 * if a given bid is active(in one of the bid table registers)
 * then then return which entry(0-3 is bidtable0, 4-6 is bidtable1)
 * otherwise return -1;
 *   this function and any corresponding read/writes should be called
 * while holding the bidtable_lock; otherwise, the bidtable entries
 * could change underneath you
 */
static int32_t find_bidtable_entry(uint8_t blockID)
{
	int i = 0;

	dprintk4("looking for bid 0x%x in bidtable...", blockID);
	/* note, all blkIDs/bids are 8 bits */
	while (i < BIDTABLE_MAX_SUPPORTED_BIDS &&
			(((bidtable_ctrl.table >> (i * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE))
			  & 0xFF) != blockID))
		i++;

	if (likely(i < BIDTABLE_MAX_SUPPORTED_BIDS)) {
		dprintk4("found\n");
		return i;
	}
	dprintk4("NOT found\n");
	return -1;
}

static void replace_bidtable_entry(uint8_t index, uint8_t newblockID)
{
	dprintk1("Will update index %d\n", index);
	dprintk1("old table 0x%014llx\n", bidtable_ctrl.table);
	/* clr and set */
	bidtable_ctrl.table &=
		~(BID_TABLE_ENTRY_MASK << (index * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));
	bidtable_ctrl.table |=
		(((uint64_t)newblockID) << (index * QCC_CHIPCTL_BIDTABLE0_BID0_SIZE));
	dprintk1("new table 0x%014llx\n", bidtable_ctrl.table);

	/* update */
	if (index < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
		dprintk1("update HW bid table0: 0x%08x\n",
				(uint32_t)(bidtable_ctrl.table & QCC_CHIPCTL_BIDTABLE0_MASK));
		fwritel((uint32_t)(bidtable_ctrl.table & QCC_CHIPCTL_BIDTABLE0_MASK),
				(falcon->bar0_qcc_base + QCC_CHIPCTL_BIDTABLE0_OFFSET));
	} else {
		dprintk1("update HW bid table1: 0x%08x\n",
				((uint32_t)(bidtable_ctrl.table >>
					QCC_CHIPCTL_BIDTABLE0_SIZE) & QCC_CHIPCTL_BIDTABLE1_MASK));
		fwritel(((uint32_t)(bidtable_ctrl.table >>
						QCC_CHIPCTL_BIDTABLE0_SIZE) & QCC_CHIPCTL_BIDTABLE1_MASK),
				falcon->bar0_qcc_base + QCC_CHIPCTL_BIDTABLE1_OFFSET);
	}
}

/*
 * for now, we just do a crude round robin replacement of blkID's.  later,
 * we may want to create a way to lock an entry in so it cannot be removed,
 *
 */
static int update_bidtable(uint8_t newblockID)
{
	int index = 0;
	int count = 0;

	dprintk1("adding bid 0x%02x(%d) to bidtable\n",
			newblockID, newblockID);
	while (count < BIDTABLE_MAX_SUPPORTED_BIDS) {
		if (bidtable_ctrl.locked_entry_bitmask &
				(0x1 << (bidtable_ctrl.update_index))) {
			count++;
			if (bidtable_ctrl.update_index + 1 ==
					BIDTABLE_MAX_SUPPORTED_BIDS)
				bidtable_ctrl.update_index = 0;
			else
				bidtable_ctrl.update_index++;

		} else {
			break;
		}
	}
	if (count == BIDTABLE_MAX_SUPPORTED_BIDS)
		return -1;

	index = bidtable_ctrl.update_index;
	replace_bidtable_entry(index, newblockID);

	if (bidtable_ctrl.update_index + 1 ==
			BIDTABLE_MAX_SUPPORTED_BIDS)
		bidtable_ctrl.update_index = 0;
	else
		bidtable_ctrl.update_index++;

	return index;
}

#if FALCON_PCI_DEBUG
#define PFXRW "PCI-RW "
#define PCI_READ_CONFIG    0x1
#define PCI_WRITE_CONFIG   0x2
#define PCI_READ_BAR0      0x3
#define PCI_WRITE_BAR0     0x4
#define PCI_READ_BAR2      0x5
#define PCI_WRITE_BAR2     0x6
int32_t falcon_pci_regrw(uint8_t cmd,
		uint32_t offset, uint32_t data, uint8_t size)
{
	int32_t ret = 0;
	uint32_t read_data;

	switch (cmd) {
	case PCI_READ_CONFIG:
		if (size == 1) {
			pci_read_config_byte(falcon->pdev,
					offset, (uint8_t *)&read_data);
			read_data >>= 24;
		} else if (size == 2) {
			pci_read_config_word(falcon->pdev,
					offset, (uint16_t *)&read_data);
			read_data >>= 16;
		} else if (size == 4) {
			pci_read_config_dword(falcon->pdev,
					offset, &read_data);
		} else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_config_read: offset 0x%x = 0x%08x(%d)\n",
						offset, read_data, read_data);
		break;
	case PCI_WRITE_CONFIG:
		if (size == 1)
			pci_write_config_byte(falcon->pdev,
					offset, (uint8_t)data & 0xff);
		else if (size == 2)
			pci_write_config_word(falcon->pdev,
					offset, (uint16_t)data & 0xffff);
		else if (size == 4)
			pci_write_config_dword(falcon->pdev,
					offset, data);
		else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_config_write: "
					"offset 0x%x, size %d, data 0x%08x(%d)\n",
					offset, size, data, data);
		break;
	case PCI_READ_BAR0:
		if (size == 1) {
			read_data = freadb(falcon->bar0_base + offset);
		} else if (size == 2) {
			read_data = freadw(falcon->bar0_base + offset);
		} else if (size == 4) {
			read_data = freadl(falcon->bar0_base + offset);
		} else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_read_bar0: offset 0x%x = 0x%08x(%d)\n",
						offset, read_data, read_data);
		break;
	case PCI_WRITE_BAR0:
		if (size == 1)
			fwriteb(data & 0xff, falcon->bar0_base + offset);
		else if (size == 2)
			fwritew(data & 0xffff, falcon->bar0_base + offset);
		else if (size == 4)
			fwritel(data, falcon->bar0_base + offset);
		else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_write_bar0: "
				"offset 0x%x, size %d, data 0x%08x(%d)\n",
				offset, size, data, data);
		break;
	case PCI_READ_BAR2:
		if (size == 1) {
			read_data = freadb(falcon->bar2_base + offset);
		} else if (size == 2) {
			read_data = freadw(falcon->bar2_base + offset);
		} else if (size == 4) {
			read_data = freadl(falcon->bar2_base + offset);
		} else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_read_bar2: offset 0x%x = 0x%08x(%d)\n",
						offset, read_data, read_data);
		break;
	case PCI_WRITE_BAR2:
		if (size == 1)
			fwriteb(data & 0xff, falcon->bar2_base + offset);
		else if (size == 2)
			fwritew(data & 0xffff, falcon->bar2_base + offset);
		else if (size == 4)
			fwritel(data, falcon->bar2_base + offset);
		else {
			printk(KERN_ERR PFXRW "Unsupported size: %u\n", size);
			ret = -EINVAL;
		}
		if (!ret)
			printk(KERN_ERR PFXRW "pci_write_bar2: "
				"offset 0x%x, size %d, data 0x%08x(%d)\n",
				offset, size, data, data);
		break;
	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}

#else
int32_t falcon_pci_regrw(uint8_t cmd,
		uint32_t offset, uint32_t data, uint8_t size)
{
	printk(KERN_ERR "ERROR: Falcon PCI debug disabled\n");
	return -EPERM;
}
#endif
EXPORT_SYMBOL(falcon_pci_regrw);


static int falcon_ioctl(struct inode *inode, struct file *file,
							unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	case QCCIOC_READ:
	{
		struct qcc_access read_cmd;
		void __iomem *read_addr;
		int32_t bidtable_entry;

		copy_from_user(&read_cmd,
				(char *)arg, sizeof(struct qcc_access));

		dprintk1("READ: bid 0x%02x(%d), addr 0x%x, size %d\n",
				read_cmd.blockID, read_cmd.blockID,
				read_cmd.addr, read_cmd.size);
		if (read_cmd.blockID > MAX_QCC_BID)
			return -EINVAL;

		if (read_cmd.blockID == QCC_BID_CHIPCTL) {
			/* we don't have to go thru the change
			 * to get to the chipctl registers
			 */
			dprintk1("CHIPCTL direct access\n");
			read_addr = falcon->bar0_qcc_base + read_cmd.addr;

		} else {
			spin_lock(&bidtable_lock);

			bidtable_entry = find_bidtable_entry(read_cmd.blockID);
			if (bidtable_entry < 0)
				bidtable_entry = update_bidtable(read_cmd.blockID);

			if (bidtable_entry < 0) {
				spin_unlock(&bidtable_lock);
				return -EBUSY;
			}
			dprintk1("Bidtable access, bidtable_entry index %d\n",
					bidtable_entry);

			/* the maximum slots in a bidtable register is 4(bidtable0) */
			if (bidtable_entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
				read_addr = falcon->bar0_qcc_base +
					(QCC_BID0_OFFSET * (bidtable_entry+1)) + read_cmd.addr;
			} else {
				read_addr = falcon->bar0_qcc_base +
					(QCC_BID0_OFFSET * (bidtable_entry+1)) + read_cmd.addr;
			}
			dprintk1("read_addr: 0x%08x\n", (uint32_t)read_addr);
		}
		if (read_cmd.size == 4)
			read_cmd.data = freadl(read_addr);
		else if (read_cmd.size == 2)
			read_cmd.data = freadw(read_addr) & 0xFFFF;
		else
			read_cmd.data = freadb(read_addr) & 0xFF;
		dprintk1("READ result: address 0x%08x = 0x%x\n",
				(uint32_t)read_addr, (uint32_t)read_cmd.data);

		if (read_cmd.blockID != QCC_BID_CHIPCTL)
			spin_unlock(&bidtable_lock);

		copy_to_user((char *)arg, &read_cmd,
				sizeof(struct qcc_access));
	}
	break;
	case QCCIOC_WRITE:
		{
			struct qcc_access write_cmd;
			void __iomem *write_addr;
			int32_t bidtable_entry;

			copy_from_user(&write_cmd,
					(char *)arg, sizeof(struct qcc_access));

			if (write_cmd.blockID > MAX_QCC_BID)
				return -EINVAL;

			dprintk1("WRITE: bid 0x%02x(%d), addr 0x%x, size %d, data 0x%x\n",
					write_cmd.blockID, write_cmd.blockID, write_cmd.addr,
					write_cmd.size, (uint32_t)write_cmd.data);
			if (write_cmd.blockID == QCC_BID_CHIPCTL) {
				/* we don't have to go thru the change to
				 * get to the chipctl registers */
				dprintk1("CHIPCTL direct access\n");
				write_addr = falcon->bar0_qcc_base + write_cmd.addr;

			} else {
				spin_lock(&bidtable_lock);

				bidtable_entry = find_bidtable_entry(write_cmd.blockID);
				if (bidtable_entry < 0)
					bidtable_entry = update_bidtable(write_cmd.blockID);

				if (bidtable_entry < 0) {
					spin_unlock(&bidtable_lock);
					return -EBUSY;
				}

				dprintk1("Bidtable access, bidtable_entry index %d\n",
						bidtable_entry);
				/* the maximum slots in a bidtable register is 4(bidtable0) */
				if (bidtable_entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
					write_addr = falcon->bar0_qcc_base +
						(QCC_BID0_OFFSET * (bidtable_entry+1)) + write_cmd.addr;
				} else {
					write_addr = falcon->bar0_qcc_base +
						(QCC_BID0_OFFSET * (bidtable_entry+1)) + write_cmd.addr;
				}
				dprintk1("write_addr: 0x%08x\n", (uint32_t)write_addr);
			}

			if (write_cmd.size == 4)
				fwritel(write_cmd.data, write_addr);
			else if (write_cmd.size == 2)
				fwritew(write_cmd.data & 0xFFFF, write_addr);
			else
				fwriteb(write_cmd.data & 0xFF, write_addr);

			if (write_cmd.blockID != QCC_BID_CHIPCTL)
				spin_unlock(&bidtable_lock);
		}
		break;
	case QCCIOC_WRITE_BLOCK:
		{
			/*
			 * NOTE: writes to chipcontrol in a block write will go thru hte
			 * qcc chain
			 */
			struct qcc_write_block qwb;
			struct qcc_cmd_def *qcmd, *qcmd_head;
			void __iomem *write_addr;
			int i;
			uint8_t prev_blockID = 0;
			int32_t bidtable_entry;

			copy_from_user(&qwb, (char *)arg, sizeof(struct qcc_write_block));

			dprintk1("WRITE_BLOCK\n");
			qcmd = kmalloc(qwb.entries * sizeof(struct qcc_cmd_def), GFP_KERNEL);
			if (!qcmd)
				return -ENOMEM;
			qcmd_head = qcmd;

			copy_from_user(qcmd,
					(struct qcc_cmd_def *)qwb.cmds,
					qwb.entries * sizeof(struct qcc_cmd_def));

			spin_lock(&bidtable_lock);

			bidtable_entry = find_bidtable_entry(qcmd_head->blockID);
			if (bidtable_entry < 0)
				bidtable_entry = update_bidtable(qcmd_head->blockID);

			if (bidtable_entry < 0) {
				spin_unlock(&bidtable_lock);
				kfree(qcmd);
				return -EBUSY;
			}
			prev_blockID = qcmd_head->blockID;

			if (lock_bidtable_entry(bidtable_entry) < 0) {
				spin_unlock(&bidtable_lock);
				kfree(qcmd);
				return -EBUSY;
			}
			dprintk1("starting bid 0x%02x(%d), locked bidtable_entry %d\n",
					qcmd_head->blockID, qcmd_head->blockID, bidtable_entry);

			for (i = 0; i < qwb.entries; i++, qcmd_head++) {

				dprintk1("WB: bid 0x%02x(%d), subaddr 0x%04x, data 0x%08x\n",
						qcmd_head->blockID, qcmd_head->blockID,
						qcmd_head->addr, qcmd_head->data);
				if (qcmd_head->blockID != prev_blockID) {
					replace_bidtable_entry(bidtable_entry, qcmd_head->blockID);
					prev_blockID = qcmd_head->blockID;
				}

				/* the maximum slots in a bidtable register is 4(bidtable0) */
				if (bidtable_entry < BIDTABLE_MAX_BIDS_IN_SINGLE_REGISTER) {
					write_addr = falcon->bar0_qcc_base +
						(QCC_BID0_OFFSET * (bidtable_entry+1)) + qcmd_head->addr;
				} else {
					write_addr = falcon->bar0_qcc_base +
						(QCC_BID0_OFFSET * (bidtable_entry+1)) + qcmd_head->addr;
				}

				if (qcmd_head->size == 4)
					fwritel(qcmd_head->data, write_addr);
				else if (qcmd_head->size == 2)
					fwritew(qcmd_head->data, write_addr);
				else
					fwriteb(qcmd_head->data, write_addr);

				qcmd++;
			}
			unlock_bidtable_entry(bidtable_entry);
			spin_unlock(&bidtable_lock);
			kfree(qcmd);
		}
		break;
		/* PIO memory r/w */
	case FMEMIOC_READ:
	case FMEMIOC_WRITE:
		{
			struct fmem_access fmem_cmd;
			uint32_t pci_addr, offset = 0, cur_addr, cur_size, addr_mask;
			uint32_t endianess_mask = 0x0, boundry;

			copy_from_user(&fmem_cmd,
					(char *)arg, sizeof(struct fmem_access));

			dprintk1("%s: mem_addr 0x%x, bar1_base 0x%08x, size %d\n",
				(cmd == FMEMIOC_WRITE ? "W" : "R"), fmem_cmd.mem_addr,
				(uint32_t)(falcon->bar1_base), fmem_cmd.size);

			/*
			 * if we want to disable the inherent endianess swap in the bridge, set
			 * bit 30.
			 */
			if (fmem_cmd.disable_swap)
				endianess_mask = 0x4000000;

			offset = fmem_cmd.mem_addr % falcon->bar1_size;
			pci_addr = (uint32_t)(falcon->bar1_base) + offset;
			dprintk1("%s: offset 0x%x, target pci_addr 0x%08x\n",
				(cmd == FMEMIOC_WRITE ? "W" : "R"), offset, pci_addr);

#define MY_ALIGN_UP(addr, size) \
			((addr & (size-1)) ? \
			 (((addr)+((size)-1))&(~((size)-1))) : (addr+size))

			addr_mask = ~(falcon->bar1_size-1);
			cur_addr = fmem_cmd.mem_addr;
			dprintk2("Will start access at address 0x%x, sizeof 0x%x(%d), rounding size 0x%x(%d)\n",
					cur_addr, fmem_cmd.size, fmem_cmd.size, falcon->bar1_size, falcon->bar1_size);
			dprintk2("fmem_cmd.buffer = 0x%x\n", (uint32_t)fmem_cmd.buffer);
			while (fmem_cmd.size != 0) {
				boundry = MY_ALIGN_UP(cur_addr, falcon->bar1_size);
				dprintk2("Boundry is: 0x%x(%d)\n", boundry, boundry);
				if (cur_addr + fmem_cmd.size > boundry) {
					cur_size = boundry - cur_addr;
					dprintk2("Access would cross boundry, break it up\n");
				} else {
					cur_size = fmem_cmd.size;
				}

				dprintk2("memory access @ address 0x%x, access_size 0x%x(%d)\n",
					cur_addr, cur_size, cur_size);

				/* set up pci->ahb translation registers */
				if ((endianess_mask | (cur_addr & addr_mask)) !=
						freadl((void *)((falcon->bar2_base) +
								PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET))) {

					dprintk1("Updating pci_ahb_bar1 translations registers\n");
					fwritel(addr_mask,
							(void *)((falcon->bar2_base) +
								PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET));

					fwritel((endianess_mask | (cur_addr & addr_mask)),
							(void *)((falcon->bar2_base) +
								PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET));

				}
				/* TODO after debug/bringup, move input if above */
				dprintk1("PCI_AHB_BAR1_MASK: 0x%08x\n",
						freadl((void *)((falcon->bar2_base) +
								PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET)));
				dprintk1("PCI_AHB_BAR1_TRANS: 0x%08x\n",
						freadl((void *)((falcon->bar2_base) +
								PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET)));

				if (cmd == FMEMIOC_WRITE) {
					dprintk1("W: calling copy_from_user\n");
					copy_from_user((void *)pci_addr,
							fmem_cmd.buffer, cur_size);
					//dprintk1("WR: first data byte 0x%x\n", *(uint32_t*)pci_addr);
				} else {
					dprintk1("R: calling copy_to_user\n");
					copy_to_user(fmem_cmd.buffer,
							(void *)pci_addr, cur_size);
					//dprintk1("R: first data byte 0x%x\n", *(uint8_t*)pci_addr);
				}
				/*
				 * if we enter the loop again, we will always be accessing
				 * from the start of the mem bar
				 */
				pci_addr = (uint32_t)(falcon->bar1_base);
				fmem_cmd.size -= cur_size;
				fmem_cmd.buffer += cur_size;
				cur_addr += cur_size;
				dprintk2("fmem_cmd.buffer 0x%x, fmem_cmd.size 0x%x, cur_addr 0x%x\n",
						(uint32_t)fmem_cmd.buffer, fmem_cmd.size, cur_addr);
			}
		}
		break;
	}
	return 0;
}

static irqreturn_t falcon_pci_isr(int irq, void *dev_id)
{
	uint32_t status;
	uint32_t err_addr;
	uint8_t valid = 0;

	status = readl(falcon->bar0_qcc_base + QCC_CHIPCTL_PCIMASTERINFO_OFFSET);
	err_addr = readl(falcon->bar0_qcc_base + QCC_CHIPCTL_PCIMASTERERRADDR_OFFSET);

	if ((status & QCC_CHIPCTL_PCIMASTERINFO_RESPONSEERRINTEN_MASK) &&
			(status & QCC_CHIPCTL_PCIMASTERINFO_RESPONSEINTSTATUS_MASK)) {
		dprintk2("Got Response Error interrupt @ 0x%x\n", err_addr);
		valid = 1;
	}
	if ((status & QCC_CHIPCTL_PCIMASTERINFO_DECODEERRINTEN_MASK) &&
			(status & QCC_CHIPCTL_PCIMASTERINFO_DECODEERRINTSTATUS_MASK)) {
		dprintk2("Got Decode Error interrupt @ 0x%x\n", err_addr);
		valid = 1;
	}
	if ((status & QCC_CHIPCTL_PCIMASTERINFO_SLAVEERRINTEN_MASK) &&
			(status & QCC_CHIPCTL_PCIMASTERINFO_SLAVEERRINTSTATUS_MASK)) {
		dprintk2("Got Slave Error interrupt @ 0x%x\n", err_addr);
		valid = 1;
	}
	if (!valid) {
		/* this would seem odd .. */
		dprintk2("got IRQ_NONE???\n");
		return IRQ_NONE;
	}

	/* clear the interrupt */
	writel(status,
			falcon->bar0_qcc_base + QCC_CHIPCTL_PCIMASTERINFO_OFFSET);

	return IRQ_HANDLED;
}

/* we don't have anything to do! */
static int falcon_open(struct inode *inode, struct file *filp)
{
	return 0;
}

static const struct file_operations dev_fops = {
    .owner  = THIS_MODULE,
    .llseek = no_llseek,
    .open   = falcon_open,
    .ioctl  = falcon_ioctl,
	/* would we gain anything by adding read,
	 * write, mmap??
	 */
};

static int __devinit falcon_probe(struct pci_dev *pdev,
				    const struct pci_device_id *ent)
{
	dev_t devt_id = 0;
	int err = 0;

	dprintk1("yea, we are being probed\n");
	falcon = kzalloc(sizeof(struct pci_falcondev), GFP_KERNEL);
	if (!falcon) {
		printk(KERN_ERR PFX "Failed to allocate memory\n");
		return ENOMEM;
	}

	/* 1 device, minor 0 */
	dev_class = class_create(THIS_MODULE, DEV_NAME);
	if (dev_class == NULL)
		return -EIO;

	err = alloc_chrdev_region(&devt_id, 0, 1, DEV_NAME);
	if (err) {
		class_destroy(dev_class);
		return err;
	}

	major = MAJOR(devt_id);

	cdev_init(&cdev, &dev_fops);
	err = cdev_add(&cdev, devt_id, 1);
	if (err) {
		unregister_chrdev_region(devt_id, 1);
		class_destroy(dev_class);
	} else {
		device_create(dev_class, NULL, devt_id, NULL, DEV_NAME);
	}

	/* Start PCI initialization */
	err = pci_enable_device(pdev);
	if (err) {
		printk(KERN_ERR PFX "pci_enable_device failed, aborting\n");
		goto err_out;
	}

	if (!(pci_resource_flags(pdev, 0) & IORESOURCE_MEM)) {
		printk(KERN_ERR PFX "BAR0 is not a mem resource\n");
		err = -ENODEV;
		goto err_out_disable;
	}

	if (!(pci_resource_flags(pdev, 2) & IORESOURCE_MEM)) {
		printk(KERN_ERR PFX "BAR2 is not a mem resource\n");
		err = -ENODEV;
		goto err_out_disable;
	}

	pci_read_config_word(pdev, PCI_VENDOR_ID, &falcon->vendor_id);
	pci_read_config_word(pdev, PCI_DEVICE_ID, &falcon->device_id);
	printk(KERN_INFO PFX "Found PCI%s vendorID: 0x%04x, deviceID: 0x%04x\n",
			pdev->is_pcie ? "e" : "", falcon->vendor_id, falcon->device_id);

	if (pdev->is_pcie == 1) {
		printk(KERN_ERR PFX "PCIe is not implemented yet\n");
		return -EINVAL;
	}

	pci_read_config_byte(pdev, PCI_INTERRUPT_LINE, &falcon->irq);
	if (falcon->irq == 0) {
		falcon->irq = pdev->irq;
		dprintk2("Got irq from pci_dev struct\n");
	}
	dprintk2("irq is %d\n", falcon->irq);
	err = request_irq(falcon->irq, falcon_pci_isr, IRQF_DISABLED, "falconpci", NULL);
	if (err < 0) {
		printk(KERN_ERR "Failed to request Interrupt");
		goto err_out_disable;
	}

	pci_set_master(pdev);
	if (pci_set_dma_mask(pdev, DMA_32BIT_MASK) != 0) {
		printk(KERN_ERR PFX "System does not support DMA,"
				" aborting\n");
		err = -EIO;
		goto err_out_disable1;
	}

	/* QCC and other falcon registers */
	falcon->bar0_size = pci_resource_len(pdev, 0);
	falcon->bar0_base =
		ioremap_nocache(pci_resource_start(pdev, 0),
				falcon->bar0_size);
	if (!falcon->bar0_base) {
		printk(KERN_ERR PFX "Cannot ioremap BAR0\n");
		err = -ENOMEM;
		goto err_out_disable1;
	} else {
		dprintk1("bar0_size = 0x%x\n", falcon->bar0_size);
	}

	/* for PIO to memory */
	falcon->bar1_size = pci_resource_len(pdev, 1);
	falcon->bar1_base =
		ioremap_nocache(pci_resource_start(pdev, 1),
				falcon->bar1_size);
	if (!falcon->bar1_base) {
		printk(KERN_ERR PFX "Cannot ioremap BAR1\n");
		err = -ENOMEM;
		goto err_out_disable2;
	} else {
		dprintk1("bar1_base 0x%p, bar1_size = 0x%x\n",
				falcon->bar1_base, falcon->bar1_size);
	}

	/* PCI config/control/dma registers */
	falcon->bar2_size = pci_resource_len(pdev, 2);
	falcon->bar2_base =
		ioremap_nocache(pci_resource_start(pdev, 2),
				falcon->bar2_size);
	if (!falcon->bar2_base) {
		printk(KERN_ERR PFX "Cannot ioremap BAR2\n");
		err = -ENOMEM;
		goto err_out_disable3;
	} else {
		dprintk1("bar2_base 0x%p, bar2_size = 0x%x\n",
				falcon->bar2_base, falcon->bar2_size);
	}

	if (pdev->is_pcie == 0) {
		/* some nice variable name for easy access */
		falcon->bar0_qcc_base        =
			falcon->bar0_base + BAR0_QCC_OFFSET;
		falcon->bar0_pcie_phy_base   =
			falcon->bar0_base + BAR0_PCIE_PHY_OFFSET;;
		falcon->bar0_video_data_base =
			falcon->bar0_base + BAR0_VIDEO_DATA_OFFSET;;
		falcon->bar0_axi_mmu_base    =
			falcon->bar0_base + BAR0_AXI_MMU_OFFSET;
		falcon->bar0_video_regs_base =
			falcon->bar0_base + BAR0_VIDEO_REGS_OFFSET;;
		falcon->bar0_pcie_regs_base  =
			falcon->bar0_base + BAR0_PCIE_REGS_OFFSET;
		dprintk1("bar0_base 0x%x, qcc_base 0x%x axi_mmu_base 0x%x\n",
				(uint32_t)falcon->bar0_base,
				(uint32_t)falcon->bar0_qcc_base,
				(uint32_t)falcon->bar0_axi_mmu_base);

		/* 0xB8000000 AHB QCC address - we need to replace upper 3 bytes */
		fwritel(AHB_BAR0_TRANS_MASK,
				(void *)((falcon->bar2_base) +
					PCI_BAR2_PCI_AHB_BAR0_MASK_OFFSET));
		fwritel(AHB_BAR0_TRANS_ADDR,
				(void *)((falcon->bar2_base) +
					PCI_BAR2_PCI_AHB_BAR0_TRANS_OFFSET));


		dprintk1("bar0 mask 0x%08x, trans 0x%08x\n",
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR0_MASK_OFFSET)),
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR0_TRANS_OFFSET)));

		fwritel(AHB_BAR1_TRANS_MASK,
				(void *)((falcon->bar2_base) +
					PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET));

		fwritel(AHB_BAR1_LIN_TRANS_ADDR,
				(void *)((falcon->bar2_base) +
					PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET));

		dprintk1("bar1 mask 0x%08x, trans 0x%08x\n",
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET)),
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET)));

#ifdef CONFIG_FALCON_IS_FPGA
		if (AHB_BAR0_TRANS_MASK !=
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR0_MASK_OFFSET))) {
			printk(KERN_ERR
					"AHB_BAR_TRANS_MASK readback/compare failed 0x%x != 0x%x\n",
					AHB_BAR0_TRANS_MASK,
					in_le32((void *)((falcon->bar2_base) +
							PCI_BAR2_PCI_AHB_BAR0_MASK_OFFSET)));
				err = -EIO;
		}
		if (AHB_BAR0_TRANS_ADDR !=
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR0_TRANS_OFFSET))) {
			printk(KERN_ERR
					"AHB_BAR0_TRANS_ADDR readback/compare failed 0x%x != 0x%x\n",
					AHB_BAR0_TRANS_ADDR,
					in_le32((void *)((falcon->bar2_base) +
							PCI_BAR2_PCI_AHB_BAR0_TRANS_OFFSET)));
				err = -EIO;
		}

		if (AHB_BAR1_TRANS_MASK !=
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET))) {
			printk(KERN_ERR
					"AHB_BAR1_TRANS_MASK readback/compare failed 0x%x != 0x%x\n",
					AHB_BAR1_TRANS_MASK,
					in_le32((void *)((falcon->bar2_base) +
							PCI_BAR2_PCI_AHB_BAR1_MASK_OFFSET)));
				err = -EIO;
		}
		if (AHB_BAR1_LIN_TRANS_ADDR !=
				in_le32((void *)((falcon->bar2_base) +
						PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET))) {
			printk(KERN_ERR
					"AHB_BAR1_TRANS_ADDR readback/compare failed 0x%x != 0x%x\n",
					AHB_BAR1_LIN_TRANS_ADDR,
					in_le32((void *)((falcon->bar2_base) +
							PCI_BAR2_PCI_AHB_BAR1_TRANS_OFFSET)));
				err = -EIO;
		}

		if (err)
			goto err_out_disable3;
#endif /* CONFIG_FALCON_IS_FPGA */

/* clear and enable pci interrupts */
#define PCI_AHB_INTERRUPT_ENABLE_MASK \
		(QCC_CHIPCTL_PCIMASTERINFO_RESPONSEINTSTATUS_MASK | \
		 QCC_CHIPCTL_PCIMASTERINFO_RESPONSEERRINTEN_MASK     | \
		 QCC_CHIPCTL_PCIMASTERINFO_DECODEERRINTSTATUS_MASK   | \
		 QCC_CHIPCTL_PCIMASTERINFO_DECODEERRINTEN_MASK       | \
		 QCC_CHIPCTL_PCIMASTERINFO_SLAVEERRINTSTATUS_MASK    | \
		 QCC_CHIPCTL_PCIMASTERINFO_SLAVEERRINTEN_MASK)

		/* enable interrupts that can generated from a pci write to falcon */
		if (falcon->irq) {
			dprintk2("write 0x%x to 0x%x, ",
				PCI_AHB_INTERRUPT_ENABLE_MASK,
					(uint32_t)falcon->bar0_qcc_base+QCC_CHIPCTL_PCIMASTERINFO_OFFSET);
			writel(PCI_AHB_INTERRUPT_ENABLE_MASK,
					falcon->bar0_qcc_base+QCC_CHIPCTL_PCIMASTERINFO_OFFSET);
			dprintk2("readback 0x%x\n",
					readl(falcon->bar0_qcc_base+QCC_CHIPCTL_PCIMASTERINFO_OFFSET));
		}

	} else {
		printk(KERN_WARNING "PCIe not implemented yet\n");
	}

	falcon->pdev = pdev;
	pci_set_drvdata(pdev, falcon);
	spin_lock_init(&bidtable_lock);

	return err;

err_out_disable3:
	iounmap(falcon->bar1_base);

err_out_disable2:
	iounmap(falcon->bar0_base);

err_out_disable1:
	if (falcon->irq != 0)
		free_irq(falcon->irq, NULL);	

err_out_disable:
	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);

err_out:
	device_destroy(dev_class, devt_id);
	class_destroy(dev_class);
	cdev_del(&cdev);
	unregister_chrdev_region(devt_id, 1);

	return err;
}

static void __devexit falcon_remove(struct pci_dev *pdev)
{
	dev_t devt_id = MKDEV(major, 0);

	device_destroy(dev_class, devt_id);
	class_destroy(dev_class);
	cdev_del(&cdev);
	unregister_chrdev_region(devt_id, 1);

	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);
	if (falcon->irq != 0) {
		free_irq(falcon->irq, NULL);	
		falcon->irq = 0;
	}
	if (falcon->bar0_base)
		iounmap(falcon->bar0_base);
	/* faked address so don't unmap */
	if (falcon->bar1_base)
		iounmap(falcon->bar1_base);
	if (falcon->bar2_base)
		iounmap(falcon->bar2_base);
	kfree(falcon);
}

static int falcon_suspend(struct pci_dev *pdev, pm_message_t state)
{
	return 0;
}

static int falcon_resume(struct pci_dev *pdev)
{
	return 0;
}

/**
 * bnx2x_io_error_detected - called when PCI error is detected
 * @pdev: Pointer to PCI device
 * @state: The current pci connection state
 *
 * This function is called after a PCI bus error affecting
 * this device has been detected.
 */
static pci_ers_result_t falcon_io_error_detected(struct pci_dev *pdev,
						pci_channel_state_t state)
{
	/* struct pci_falcondev *falcon = pci_get_drvdata(pdev); */

	pci_disable_device(pdev);

	/* Request a slot reset */
	return PCI_ERS_RESULT_NEED_RESET;
}

/**
 * falcon_io_slot_reset - called after the PCI bus has been reset
 * @pdev: Pointer to PCI device
 *
 * Restart the card from scratch, as if from a cold-boot.
 */
static pci_ers_result_t falcon_io_slot_reset(struct pci_dev *pdev)
{
	if (pci_enable_device(pdev)) {
		dev_err(&pdev->dev,
			"Cannot re-enable PCI device after reset\n");
		return PCI_ERS_RESULT_DISCONNECT;
	}

	pci_set_master(pdev);
	pci_restore_state(pdev);

	return PCI_ERS_RESULT_RECOVERED;
}

/**
 * falcon_io_resume - called when traffic can start flowing again
 * @pdev: Pointer to PCI device
 *
 * This callback is called when the error recovery driver tells us that
 * its OK to resume normal operation.
 */
static void falcon_io_resume(struct pci_dev *pdev)
{
	/* do something ? */
}

static struct pci_error_handlers falcon_err_handler = {
	.error_detected = falcon_io_error_detected,
	.slot_reset = falcon_io_slot_reset,
	.resume = falcon_io_resume,
};

static const struct pci_device_id falcon_pci_tbl[] = {
	{
		PCI_VENDOR_ID_MAXIM, PCI_DEVICE_ID_MAXIM_FALCON,
		PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0
	}
};

#if FALCON_PCI_DEBUG
#ifdef CONFIG_DEBUG_FS
static int __init falcon_pci_init_debugfs(void)
{
	d_fpci_debug = debugfs_create_dir("fpci", NULL);

	if (d_fpci_debug == NULL)
		return -ENOMEM;

	debugfs_create_u8("dbg_loglevel", 0664, d_fpci_debug, &dbg_loglevel);
	dprintk1("dbg_loglevel %d\n", dbg_loglevel);

	return 0;
}

static void __exit falcon_pci_remove_debugfs(void)
{
	if (d_fpci_debug)
		debugfs_remove_recursive(d_fpci_debug);
}
#else
static int __init falcon_pci_init_debugfs(void) { return 0; }
static void __exit falcon_pci_remove_debugfs(void) {}
#endif /* DEBUG_FS */
#endif /* FALCON_PCI_DEBUG */

static struct pci_driver falcon_pci_driver = {
	.name        = DRV_NAME,
	.id_table    = falcon_pci_tbl,
	.probe       = falcon_probe,
	.remove      = __devexit_p(falcon_remove),
	.suspend     = falcon_suspend,
	.resume      = falcon_resume,
	.err_handler = &falcon_err_handler,
};

static int __init falcon_init(void)
{
	printk(KERN_INFO PFX "Loading falcon PCI driver\n");
	dprintk1("PCI will try to find device matching: vendorID 0x%x, deviceID 0x%x\n",
			falcon_pci_tbl[0].vendor, falcon_pci_tbl[0].device);

#if FALCON_PCI_DEBUG
	falcon_pci_init_debugfs();
#endif

	return pci_register_driver(&falcon_pci_driver);
}

static void __exit falcon_exit(void)
{
#if FALCON_PCI_DEBUG
	falcon_pci_remove_debugfs();
#endif

	pci_unregister_driver(&falcon_pci_driver);
	printk(KERN_INFO PFX "Unloading falcon PCI driver\n");
}
module_init(falcon_init);
module_exit(falcon_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Maxim falcon ppc PCI driver");
