/*HEADER_START*/
/*
*  Copyright (C) 2006 Mobilygen Corp.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*HEADER_STOP*/

#ifndef DOXYGEN_SKIP

#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <linux/mtd/physmap.h>

#include <mach/platform.h>
#include <mach/irqs.h>
#include <mach/mg3500_devices.h>
#include <mach/mobi_ioctrl.h>

#include <linux/spi/spi.h>
#include <linux/spi/eeprom.h>
#include <linux/mobi_usb_device.h>
#include <linux/gpioi2c.h>
#include <linux/i2c-gpio.h>
#include <linux/jiffies.h>

#include "board_desc.h"

#endif

struct board_desc board_desc = {
	.bd_type = "merlinbup",
};

static void empty_platform_release(struct device *dev) { }

struct nand_board_config_t nand_config = {

	/* init data passed to nand device */
	.device_data.databus_16bit     = 0,
	.device_data.banks_per_system  = 1,
	.device_data.chips_per_bank    = 1,
	.device_data.use_dma = 0,
	.device_data.clk_name = "ahb",
	
	/*
	 * only set new timing values for those you want to change, default
	 * values will be use for remaining timing values.  Make sure to set 
	 * the corresponding bit in the bitmask so the device know which 
	 * timing value to use. for example:
	.device_data.board_timings[NAND_TIMING_TDS] = 32,
	.device_data.board_timings[NAND_TIMING_TCLH] = 64,
	.device_data.board_set_timing_bitmask = 
		  NAND_TIMING_TDS_SET 
		| NAND_TIMING_TCLH_SET,
	 */

	/* board specific stuff */
	.hmux.ce = HMUX_NAND_CE0,
	.hmux.cs = HMUX_HOST_CS1,
};

struct pata_board_config_t ide_config = {

    .device_data.gpio_shift = 7,
    .device_data.gpio = 0,
    .device_data.irq = MOBI_IRQ_GPIO0,

	.hmux0.ce = HMUX_CF_CE0,
	.hmux0.cs = HMUX_HOST_CS2,
	.hmux1.ce = HMUX_CF_CE1,
	.hmux1.cs = HMUX_HOST_CS3,
};

struct wdt_board_config_t wdt_config = {
	/* device driver platform data */
	.wdt_data.timeout_msec = 0,
	.wdt_data.timeout_sec  = WATCHDOG_DEFAULT_HEARTBEAT,
	.wdt_data.clk_name = "apb",

	/* board specific setup data, not passed to driver */
	.rsvd = 0,
};

/* 
 * define the size of the address space for each mhif device for this 
 * platform here.  unused devices must have a size of 0 defined, for now
 * use the macros MHIF_DEVICE<id>_START and MHIF_DEVICE<id>_END when
 * defining the resources for the individual devices in the array below.
 * the start address of devices will start after the end device of
 * the previous device
 */

/* FYI, the max size for NOR on merlinbup via mhif is 0x1000000 */
#define MHIF_DEVICE0_SIZE	0x1000000	/* dev0 is NOR */
#define MHIF_DEVICE1_SIZE	0x0
#define MHIF_DEVICE2_SIZE	0x0
#define MHIF_DEVICE3_SIZE	0x0
#define MHIF_DEVICE4_SIZE	0x0
#define MHIF_DEVICE5_SIZE	0x0

struct mhif_device_t mhif_devices[] = {
	{
		/* CS0 = NOR flash device definition Width=16-bit */
		.dev_name 		= "NOR",
		.chip_select 	= 0,
		.width  		= MHIF_DEVICE_DATA_WIDTH_16,	
		.endian 		= MHIF_DEVICE_LITTLE_ENDIAN,	
		.bus_mux		= MHIF_DEVICE_BUS_SEPARATE,	
		.dma_wen		= MHIF_DEVICE_DMAWAIT_DISABLE,	
		.wait_en		= MHIF_DEVICE_MHIFWAIT_DISABLE,	
		.addr_inc		= MHIF_DEVICE_ADDRINC_ENABLE,	

		.ale_setup_ns 	= 20,	
		.ale_width_ns 	= 20,	
		.ale_hold_ns 	= 20,
		/* read data out sampling delay (waitstates) in ns (max 195), */
		.rd_wait_ns 	= 100,
		.wr_hold_ns 	= 20,	
		/* device wait time out delay in ns (max 1600), 400 works*/
		.dev_timeout_ns = 400,	
		.dev_idle_ns 	= 20,	
		.prescaler 		= 0,	
		.timeout_en	 	= MHIF_DEVICE_TIMEOUT_DISABLE,	

		.wr_delay_ns 	= 20,	
		.ale_edge 	    = MHIF_DEVICE_ALE_STROBE_RISING,	
		.cs_lvl		    = MHIF_DEVICE_CS_ACTIVE_LOW,	
		.wait_lvl     	= MHIF_DEVICE_WAITEN_ACTIVE_HIGH,	
		.dmarq_lvl    	= MHIF_DEVICE_DMARQ_ACTIVE_HIGH,	
		.rd_hold_ns     = 20,	

		.resource_start  = MHIF_DEVICE0_START,
		.resource_end    = MHIF_DEVICE0_END,
	},
};

static  __attribute__((unused)) struct mhif_board_config_t mhif_config = 
{
	.mhif_data.devices	= mhif_devices,
	.mhif_data.num_devices	= ARRAY_SIZE(mhif_devices),
	.mhif_data.clk_name = "ahb",

	/* 
	 * on the merlinbup board to connect the NOR > 8M via mhif, 
	 * use a second CS to connect to the chip address bits, 
	 * ie mhif address bit 23 is connect to the nor address bit 22,
	 * contact customer support for more details on how to use this
	 */
	.ce12_hmux = {
		.ce    = HMUX_MHIF_ADDR_23_CE0,
		.cs    = HMUX_HOST_CS5,
	},

	/* not used in this design */
	.ce13_hmux = {
		.ce    = HMUX_UNUSED_CE,
	},

	/*
	 * even though we are not accessing the NOR through the
	 * aurora flash controller, we have to unassert WP in the 
	 * controller's control register to be able to access the NOR
	 * so, set to 1 if attaching a NOR to the MHIF, if you device
	 * is not a nor set this to 0
	 */
	.nor_attached = 1,
};

/* 
 * here we are doing the driver setup for a device we are 
 * going to access thru the MHIF.  for the merlinbup platform
 * we only have one device and it's a NOR chip.  We want to 
 * use the nor as a MTD block device, so setup everything here
 */
static struct resource mhif_resource_device0 = {
	.start  = MHIF_BASE,
	.end    = (MHIF_BASE+MHIF_DEVICE0_SIZE-1),
	.flags  = IORESOURCE_MEM,
};

static struct physmap_flash_data nor_physmap_data = {
	.width      = 2,
	.nr_parts   = 0,
};

static struct platform_device nor_physmap_flash = {
	.name       = "physmap-flash",
	.id         = -1 ,
	.dev        = {
		.platform_data  = &nor_physmap_data,
		.release		= empty_platform_release,
	},
	.resource       = &mhif_resource_device0,
	.num_resources  = 0,
};

struct sdmmc_board_config_t sdmmc_config = {

	/* device driver platform data */
	.device_data.use_dma 			= 0,
	.device_data.disable_highspeed 	= 1,
	.device_data.dedicated_dma_channel 	= 0,
	.device_data.clk_name 			= "sdmmc",
	.device_data.clk_rate 			= 25000000,

	/* board specific setup data, not passed to driver */
	.rsvd = 0,
};

struct gmac_board_config_t gmac_config = {

	/* device driver platform data */
	.gmac_data.clk_name = "ahb",
	.gmac_data.options = DWGMAC_O_NONE,
	/* values defined in include/linux/phy.h */
	.gmac_data.phy_interface = PHY_INTERFACE_MODE_MII,

	/* board specific setup data, not passed to driver */
	.rsvd = 0,
};

struct usb_board_config_t usb_config = {

	/* device driver platform data */
	.otg_data.options = 0x0,

	/* board specific setup data, not passed to driver */
	.rsvd = 0,
};

static void setup_ioctrl(void)
{
	return;
}

/*
 * SPI Busses and Devices
 */
static struct spi_eeprom eeprom0_data = {
	.name			= "EEPROM0",
	.byte_len		= 32768,
	.page_size 		= 64,
	.flags 			= EE_ADDR2,
};

struct spi_board_info spi0_devices[] = {
	/* SPI_0 MASTER bus: has 2 chipselects */
	{
		.bus_num	= 0,
		/* Test EEPROM on Merlin daughter board */
		.modalias	= "at25",
		.platform_data	= &eeprom0_data,
		/* Mode 3 is actually a mode 0 for the DW_apb_spi component 
		 *  (but mode 0 chip-unselects after every byte sent) */
		.mode		= 3, 
		.max_speed_hz   = 500000,
		.chip_select	= 0,
	},
	{
		.bus_num	= 0,
		.modalias	= "spidev",
		.platform_data  = NULL,
		.mode		= 3, 
		.max_speed_hz   = 10000,
		.chip_select	= 1,
	},
};

static struct dwapbssi_bus_data spi0_bus_data = {
	.devices		= spi0_devices,
	.num_devices	= ARRAY_SIZE(spi0_devices),
	.num_chipselect	= ARRAY_SIZE(spi0_devices),
	.use_tx_dma 	= 0,
	.use_rx_dma		= 0,
};

struct spi_board_info spi1_devices[] = {
	/* SPI_1 MASTER bus: has 2 chipselects */
	{
		.bus_num		= 1,
		.modalias		= "spidev",
		.platform_data  = NULL,
		.mode			= 3,
		.max_speed_hz   = 10000,
		.chip_select	= 0,
	},
	{
		.bus_num		= 1,
		.modalias		= "spidev",
		.platform_data  = NULL,
		.mode			= 3, 
		.max_speed_hz   = 10000,
		.chip_select	= 1,
	},
};

static struct dwapbssi_bus_data spi1_bus_data = {
	.devices		= spi1_devices,
	.num_devices	= ARRAY_SIZE(spi1_devices),
	.num_chipselect	= ARRAY_SIZE(spi1_devices),
};

struct spi_board_info spi2_devices[] = {
	/* SPI_2 SLAVE bus: MIPS <-> ARM communication */
	{
		.bus_num		= 2,
		.modalias		= "spidev",
		.platform_data  = NULL,
		.mode			= 3,
		.max_speed_hz   = 10000,
		.chip_select	= 0,
	},
};

static struct dwapbssi_bus_data spi2_bus_data = {
	.devices		= spi2_devices,
	.num_devices	= ARRAY_SIZE(spi2_devices),
	.num_chipselect	= ARRAY_SIZE(spi2_devices),
};

/*
 * I2C Busses
 */
static struct dwapbi2c_bus_data i2c0_bus_data = {
	.i2c_speed 		= DWAPBI2C_CON_SPEED_STANDARD,
};

static struct dwapbi2c_bus_data i2c1_bus_data = {
	.i2c_speed 		= DWAPBI2C_CON_SPEED_STANDARD,
};

/*
 * PWM Blocks
 */
struct cadpwm_block_data pwm0_block_data = {};
struct cadpwm_block_data pwm1_block_data = {};
struct cadpwm_block_data pwm2_block_data = {};

/*
 * GPIO Blocks
 */
struct dwapbgpio_block_data gpio0_block_data = {
	.group = "native",
};

struct dwapbgpio_block_data gpio1_block_data = {
	.group = "native",
};

struct dwapbgpio_block_data gpio2_block_data = {
	.group = "native",
};

#if defined(CONFIG_I2C_GPIO) || defined(CONFIG_I2C_GPIO_MODULE)
/*
 * I2C Over GPIO Alternative Devices
 *
 * Note: This code is not enabled by default.  The pins used
 * below are the actual I2C pins driven by the I2C controller. To
 * actually use these same pins for i2c-gpio you must use ioctrl
 * to set these pins a gpio.  Or you can change the pins to 
 * any gpios available
 */
static struct i2c_gpio_platform_data gpioi2c_bus0_data = {
	/* I2C_0 */
	.scl_pin        = 61, /* bank 2, index 21 */
	.sda_pin        = 62, /* bank 2, index 22 */
	.udelay = 5,
};
static struct i2c_gpio_platform_data gpioi2c_bus1_data = {
	/* I2C_1 (V01) */
	.scl_pin        = 34, /* bank 1, index 26 */
	.sda_pin        = 35, /* bank 1, index 27 */
	.udelay = 5,
};
static struct i2c_gpio_platform_data gpioi2c_bus2_data = {
	/* I2C_1 (V23) */
	.scl_pin        = 54, /* bank 2, index 14 */
	.sda_pin        = 55, /* bank 2, index 15 */
	.udelay = 5,
};
static struct platform_device gpioi2c_bus0_dev = {
	.name	= "i2c-gpio",
	.id 	= 4,
	.dev	= { 
		.platform_data = &gpioi2c_bus0_data 
	},
};
static struct platform_device gpioi2c_bus1_dev = {
	.name	= "i2c-gpio",
	.id 	= 5,
	.dev	= { 
		.platform_data = &gpioi2c_bus1_data 
	},
};
static struct platform_device gpioi2c_bus2_dev = {
	.name	= "i2c-gpio",
	.id 	= 6,
	.dev	= { 
		.platform_data = &gpioi2c_bus2_data 
	},
};
#endif


/*
 * USB device mode 
 */
static struct merlin_usb_data musb_0_data = {
	.width		= 2,			
	.endian		= 0,			
};

static struct platform_device merlin_musb_0_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 0,
	.dev    = {
		.platform_data	= &musb_0_data,		
		.release        = empty_platform_release,
	},
	.num_resources = 0,
};

static struct merlin_usb_data musb_1_data = {
	.width		= 2,			
	.endian		= 0,			
};

static struct platform_device merlin_musb_1_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 1,
	.dev    = {
		.platform_data	= &musb_1_data,		
		.release        = empty_platform_release,
	},
	.num_resources = 0,
};

static struct merlin_usb_data musb_2_data = {
	.width		= 2,			
	.endian		= 0,			
};

static struct platform_device merlin_musb_2_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 2,
	.dev    = {
		.platform_data	= &musb_2_data,		
		.release        = empty_platform_release,
	},
	.num_resources = 0,
};

static struct merlin_usb_data musb_3_data = {
	.width		= 2,
	.endian		= 0,
};

static struct platform_device merlin_musb_3_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 3,
	.dev    = {
		.platform_data	= &musb_3_data,
		.release		= empty_platform_release,
	},
	.num_resources = 0,
};

static struct merlin_usb_data musb_4_data = {
	.width		= 2,
	.endian		= 0,
};

static struct platform_device merlin_musb_4_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 4,
	.dev    = {
		.platform_data	= &musb_4_data,
		.release		= empty_platform_release,
	},
	.num_resources = 0,
};

static struct merlin_usb_data musb_5_data = {
	.width		= 2,
	.endian		= 0,
};

static struct platform_device merlin_musb_5_device = {
	.name   = MUSB_PLATFORM_NAME,
	.id     = 5,
	.dev    = {
		.platform_data	= &musb_5_data,
		.release		= empty_platform_release,
	},
	.num_resources = 0,
};

static struct platform_device mhifnor_device = {
	.name   = "mg_mhifnor",
	.id     = -1,
	.dev    = {
		.platform_data	= NULL,
		.release = empty_platform_release,
	},
	.num_resources = 0,
};

static int board_proc_rd_type(char *buf, char **start, 
		off_t offset, int count, int *eof, void *data)
{
	int len = 0;
	len += sprintf(buf+len,"%s\n", board_desc.bd_type);
	return len;
}

/* 
 * this array is a list of devices that are to be enabled on
 * a specific board.  devices will be added in order listed here
 *
 * XXX Do not change the order of spi/i2c registration
 */
struct board_device_desc board_devices[] = {
	{
		.device = BOARD_DEVICE_NAND,
		.devid = 0,
		.platform_data = &nand_config,
	},
	{
		.device = BOARD_DEVICE_WATCHDOG,
		.devid = 0,
		.platform_data = &wdt_config,
	},
	{
		.device = BOARD_DEVICE_SDMMC,
		.devid = 0,
		.platform_data = &sdmmc_config,
	},
	{
		.device = BOARD_DEVICE_GMAC,
		.devid = 0,
		.platform_data = &gmac_config,
	},
	{
		.device = BOARD_DEVICE_USB,
		.devid = 0,
		.platform_data = &usb_config,
	},
	{
		.device = BOARD_DEVICE_MHIF,
		.devid = 0,
		.platform_data = &mhif_config,
	},
	/* XXX Do not change the order of spi/i2c */
	/* spi2 disables spi0, spi1 disables i2c  */
	{
		.device = BOARD_DEVICE_SPI,
		.devid = 2,
		.platform_data = &spi2_bus_data,
	},
	{
		.device = BOARD_DEVICE_SPI,
		.devid = 1,
		.platform_data = &spi1_bus_data,
	},
	{
		.device = BOARD_DEVICE_SPI,
		.devid = 0,
		.platform_data = &spi0_bus_data,
	},
	{
		.device = BOARD_DEVICE_I2C,
		.devid = 0,
		.platform_data = &i2c0_bus_data,
	},
	{
		.device = BOARD_DEVICE_I2C,
		.devid = 1,
		.platform_data = &i2c1_bus_data,
	},
	{
		.device = BOARD_DEVICE_PWM,
		.devid = 0,
		.platform_data = &pwm0_block_data,
	},
	{
		.device = BOARD_DEVICE_PWM,
		.devid = 1,
		.platform_data = &pwm1_block_data,
	},
	{
		.device = BOARD_DEVICE_PWM,
		.devid = 2,
		.platform_data = &pwm2_block_data,
	},
	{
		.device = BOARD_DEVICE_GPIO,
		.devid = 0,
		.platform_data = &gpio0_block_data,
	},
	{
		.device = BOARD_DEVICE_GPIO,
		.devid = 1,
		.platform_data = &gpio1_block_data,
	},
	{
		.device = BOARD_DEVICE_GPIO,
		.devid = 2,
		.platform_data = &gpio2_block_data,
	},
#if 0
	/* ide requires gpio */
	{
		.device = BOARD_DEVICE_PATA,
		.devid = 0,
		.platform_data = &pata_config,
	},
#endif
};

static int __init merlinbup_board_init(void)
{	
	struct proc_dir_entry *board_proc_dir = proc_mkdir(BOARD_PROCDIR, NULL);
	create_proc_read_entry("type",
			S_IRUSR | S_IRGRP | S_IROTH,
			board_proc_dir,
			board_proc_rd_type,
			NULL);

	printk(KERN_ERR "Loading driver for board type: %s\n", 
			board_desc.bd_type);

	/* optional board specific ioctrl setup */
	setup_ioctrl();

	arch_add_board_devices(board_devices, ARRAY_SIZE(board_devices));

	/* mhif is setup, now register mtd device for nor */
	platform_device_register(&nor_physmap_flash);

#if defined(CONFIG_I2C_GPIO) || defined(CONFIG_I2C_GPIO_MODULE)
    gpioi2c_bus0_data.timeout = msecs_to_jiffies(3000);
	gpioi2c_bus1_data.timeout = msecs_to_jiffies(3000);
	gpioi2c_bus2_data.timeout = msecs_to_jiffies(3000);
	platform_device_register(&gpioi2c_bus0_dev);
	platform_device_register(&gpioi2c_bus1_dev);
	platform_device_register(&gpioi2c_bus2_dev);
#endif

	platform_device_register(&merlin_musb_0_device);
	platform_device_register(&merlin_musb_1_device);
	platform_device_register(&merlin_musb_2_device);
	platform_device_register(&merlin_musb_3_device);
	platform_device_register(&merlin_musb_4_device);
	platform_device_register(&merlin_musb_5_device);

	platform_device_register(&mhifnor_device);

	return 0;
}

static void __exit merlinbup_board_exit(void)
{
	return;
}
#if defined(CONFIG_MG_BOARD_MERLINBUP_MODULE)
module_init(merlinbup_board_init);
module_exit(merlinbup_board_exit);
#endif

#if defined(CONFIG_MG_BOARD_MERLINBUP)
arch_initcall(merlinbup_board_init);
#endif

MODULE_AUTHOR("Jeff Hane");
MODULE_DESCRIPTION("Driver for Mobilygen Merlin Bringup board");
MODULE_LICENSE("GPL");
