/*******************************************************************************
 * cryptodev.c
 *
 * Linux CryptoAPI user space interface module
 *
 * Copyright (c) 2008 Shasi Pulijala <spulijala@amcc.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *
 * Detail Description:
 * This file implements the /dev/crypto interface which is intended to
 * provide user space interface to the Linux CryptoAPI.
 *
 *******************************************************************************
 */
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/file.h>
#include <linux/miscdevice.h>
#include <linux/crypto.h>
#include <linux/mm.h>
#include <linux/highmem.h>
#include <linux/random.h>
#include <linux/syscalls.h>
#include <linux/scatterlist.h>
#include <linux/time.h>
#include <linux/unistd.h>
#include <linux/rtnetlink.h>
#include <linux/fdtable.h>
#include <linux/err.h>
#include <crypto/cryptodev.h>
#include <crypto/aead.h>
#include <crypto/authenc.h>
#include <crypto/hash.h>
#include <asm/uaccess.h>
#include <asm/ioctl.h>
#include <asm/scatterlist.h>

/*******************************************************************************
 * Forward declaration
 *******************************************************************************
 */
#define CRYPTODEV_DEBUG

#define SCATTERLIST_MAX (((CRYPTO_MAX_DATA_LEN + PAGE_SIZE - 1) >> PAGE_SHIFT) + 1)

/*******************************************************************************
 * Macro declaration
 *******************************************************************************
 */
/* /dev/crypto is a char block device with majar 10 and minor below */
#define	CRYPTODEV_MINOR				70

#define	CRYPTODEV_UI_SUPPORT_DRIVER		"0.1"

/*******************************************************************************
 * Module Parameters
 *******************************************************************************
 */
static int debug;
module_param(debug, int, 0644);
MODULE_PARM_DESC(debug, "0: normal, 1: verbose, 2: debug");

#ifdef CRYPTODEV_STATS
static int enable_stats;
module_param(enable_stats, int, 0644);
MODULE_PARM_DESC(enable_stats, "collect statictics about cryptodev usage");
#endif

static int use_zcopy = 1;
module_param(use_zcopy, int, 0644);
MODULE_PARM_DESC(use_zcopy, "enable zero-copy from userspace");

/*******************************************************************************
 * Debugging Macro's
 *******************************************************************************
 */
#define PFX "cryptodev: "

#ifndef CRYPTODEV_DEBUG
#define CRYPTODEV_HEXDUMP(b, l) \
		print_hex_dump(KERN_CONT, "", DUMP_PREFIX_OFFSET, \
				16, 1, (b), (l), false);
#define CRYPTODEV_PRINTK(level, severity, format, a...) \
	do { \
		if (level <= debug) \
			printk(severity PFX "%s[%u]: " format, \
			       current->comm, current->pid, ##a); \
	} while (0)
#else
#define CRYPTODEV_HEXDUMP(b, l)
#define CRYPTODEV_PRINTK(level, severity, format, a...)
#endif

/*******************************************************************************
 * Helper Structure
 *******************************************************************************
 */
#define CRYPTO_ACIPHER	0
#define CRYPTO_AHASH	1
#define CRYPTO_AEAD	2

#define tfm_ablkcipher	crt_tfm.acipher_tfm
#define tfm_aead	crt_tfm.aead_tfm
#define tfm_ahash	crt_tfm.ahash_tfm

struct csession {
	struct list_head entry;
	struct semaphore sem;
	union {
		struct crypto_ablkcipher *acipher_tfm;
		struct crypto_ahash 	 *ahash_tfm;
		struct crypto_aead 	 *aead_tfm;
	} crt_tfm;

	int 		mode;
	unsigned int 	sid;
};

/*******************************************************************************
 * Table Lookup for Algorithms name(Crypto/hash name)
 * Helper Structure
 *******************************************************************************
 */
char *algo_map_tbl[CRYPTO_ALGORITHM_MAX] = {
	[CRYPTO_DES_CBC]  	= "cbc(des)",
	[CRYPTO_3DES_CBC] 	= "cbc(des3_ede)",
	[CRYPTO_MD5_HMAC] 	= "hmac(md5)",
	[CRYPTO_BLF_CBC]  	= "cbc(blowfish)",
	[CRYPTO_CAST_CBC]		= "cbc(cast5)",
	[CRYPTO_SKIPJACK_CBC]	= "camellia",
	[CRYPTO_MD5_HMAC]		= "hmac(md5)",
	[CRYPTO_SHA1_HMAC]	= "hmac(sha1)",
	[CRYPTO_RIPEMD160_HMAC]	= "hmac(rmd160)",
	[CRYPTO_MD5_KPDK]		= "",
	[CRYPTO_SHA1_KPDK]	= "",
	[CRYPTO_RIJNDAEL128_CBC] = "cbc(aes)",
	[CRYPTO_AES_CBC]		= "cbc(aes)",
	[CRYPTO_ARC4]		= "ecb(arc4)",
	[CRYPTO_MD5]		= "md5",
	[CRYPTO_SHA1]		= "sha1",
	[CRYPTO_NULL_HMAC]	= "",
	[CRYPTO_NULL_CBC]		= "",
	[CRYPTO_DEFLATE_COMP]	= "deflate",
	[CRYPTO_SHA2_256_HMAC]	= "hmac(sha256)",
	[CRYPTO_SHA2_384_HMAC]	= "hmac(sha384)",
	[CRYPTO_SHA2_512_HMAC]	= "hmac(sha512)",
	[CRYPTO_CAMELLIA_CBC]	= "cbc(camellia)",
	[CRYPTO_SHA2_256]		= "sha256",
	[CRYPTO_SHA2_384]		= "sha384",
	[CRYPTO_SHA2_512]		= "sha512",
	[CRYPTO_RIPEMD160]	= "rmd160",
	[CRYPTO_SHA2_224]		= "sha224",
	[CRYPTO_AES_ECB]		= "ecb(aes)",
	[CRYPTO_AES_CTR]		= "ctr(aes)",
	[CRYPTO_AES_CCM]		= "ccm(aes)",
	[CRYPTO_AES_GCM]		= "gcm(aes)",
};

struct fcrypt {
	struct list_head list;
	struct semaphore sem;
};

struct async_result {
	struct completion completion;
	int err;
};

/*******************************************************************************
 * Function Declarations
 *******************************************************************************
 */
static int create_session_ablkcipher(char *alg_name,
				struct fcrypt *fcr,
				struct session_op *sop);
static int create_session_ahash(char *alg_name,
				struct fcrypt *fcr,
				struct session_op *sop);
static int create_session_aead(char *alg_name,
				struct fcrypt *fcr,
				struct session_op *sop);
static int cryptodev_run_acipher(struct csession *ses_ptr,
				struct crypt_op *cop);
static int cryptodev_run_ahash(struct csession *ses_ptr,
				struct crypt_op *cop);
static int cryptodev_run_aead(struct csession *ses_ptr,
				struct crypt_op *cop);

/*******************************************************************************
 * Asynchronous handling Routine
 *
 *******************************************************************************
 */
static void cryptodev_async_complete(struct crypto_async_request *req, int err)
{
	struct async_result *res = req->data;

	if (err == -EINPROGRESS)
		return;

	res->err = err;
	complete(&res->completion);
}

/*******************************************************************************
 * Prepare session for future use
 *
 *******************************************************************************
 */
static int cryptodev_create_session(struct fcrypt *fcr, struct session_op *sop)
{
	char alg_name[CRYPTO_MAX_ALG_NAME];
	int  mode = -1;
	int ret =  0;

	if (sop->mac > CRYPTO_ALGORITHM_MAX ||
		sop->cipher > CRYPTO_ALGORITHM_MAX) {
		printk(KERN_INFO PFX "algorithm not supported or "
			"not set\n");
		return -EINVAL;
	}

	if (sop->cipher && sop->mac) {
		mode = CRYPTO_AEAD;
		printk(KERN_INFO PFX "authenc(%s,%s) (Algorithm Chanining Mode"
			"not yet supported", algo_map_tbl[sop->mac],
			algo_map_tbl[sop->cipher]);
		return -EINVAL;
	} else if (sop->cipher) {
		if (sop->cipher == CRYPTO_AES_GCM ||
			sop->cipher == CRYPTO_AES_CCM)
			mode = CRYPTO_AEAD;
		else
			mode = CRYPTO_ACIPHER;
		strncpy(alg_name, algo_map_tbl[sop->cipher],
			 CRYPTO_MAX_ALG_NAME);
	} else if (sop->mac) {
		mode = CRYPTO_AHASH;
		strncpy(alg_name, algo_map_tbl[sop->mac], CRYPTO_MAX_ALG_NAME);
	}

	switch (mode) {
	case CRYPTO_ACIPHER:
		ret = create_session_ablkcipher(alg_name, fcr, sop);
		break;
	case CRYPTO_AHASH:
		ret = create_session_ahash(alg_name, fcr, sop);
		break;
	case CRYPTO_AEAD:
		ret = create_session_aead(alg_name, fcr, sop);
		break;
	default:
		printk(KERN_INFO PFX "Improper Mode Set(Not Cipher/Hash/Aead)");
		ret = -EINVAL;
		break;
	}
	return ret;
}

/*******************************************************************************
 * Routine for Creating a Session for the Combined Mode Implementation
 *
 *******************************************************************************
 */
static int create_session_aead(char *alg_name, struct fcrypt *fcr,
			struct session_op *sop)
{
	struct csession	*ses_new;
	struct csession	*ses_ptr;
	struct crypto_aead *tfm;
	char *keyp = NULL;
	size_t authsize;
	int ret = 0;

	tfm = crypto_alloc_aead(alg_name, 0, 0);
	if (IS_ERR(tfm)) {
		printk(KERN_INFO PFX "Failed to load aead"
			"transform for %s: %ld \n", alg_name, PTR_ERR(tfm));
		return -EINVAL;
	}

	crypto_aead_clear_flags(tfm, ~0);

	keyp = kmalloc(sop->keylen, GFP_KERNEL);
	if (unlikely(!keyp)) {
		crypto_free_aead(tfm);
		return -ENOMEM;
	}

	if (copy_from_user(keyp, sop->key, sop->keylen)) {
		printk(KERN_INFO PFX "Copy of Key Failed from"
		"User Space for %s\n", alg_name);
		kfree(keyp);
		crypto_free_aead(tfm);
		return -EFAULT;
	}

	ret = crypto_aead_setkey(tfm, keyp, sop->keylen);
	kfree(keyp);
	if (ret) {
		printk(KERN_INFO PFX
			"Setting key failed for %s-%zu: flags=0x%X\n",
			alg_name, sop->keylen*8,
			crypto_aead_get_flags(tfm));
		printk(KERN_INFO PFX
			"(see CRYPTO_TFM_RES_* in <linux/crypto.h> "
			"for details)\n");

		crypto_free_aead(tfm);
		return -EINVAL;
	}

	/* Supporting Authsize for ccm and gcm from mackeylen
	(no separate field for authsize) */
	authsize = sop->mackeylen;
	ret = crypto_aead_setauthsize(tfm, authsize);
	if (ret) {
		printk(KERN_INFO "failed to set authsize = %u\n", authsize);
		crypto_free_aead(tfm);
		return -EINVAL;
	}

	ses_new = kmalloc(sizeof(*ses_new), GFP_KERNEL);
	if (!ses_new) {
		crypto_free_aead(tfm);
		return -ENOMEM;
	}

	memset(ses_new, 0, sizeof(*ses_new));
	get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
	ses_new->tfm_aead = tfm;

	ses_new->mode = CRYPTO_AEAD;
	init_MUTEX(&ses_new->sem);

	down(&fcr->sem);

restart:
	list_for_each_entry(ses_ptr, &fcr->list, entry) {
		/* Check for duplicate SID */
		if (unlikely(ses_new->sid == ses_ptr->sid)) {
			get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
			goto restart;
		}
	}

	list_add(&ses_new->entry, &fcr->list);
	up(&fcr->sem);

	sop->ses = ses_new->sid;

	return 0;
}

/*******************************************************************************
 * Routine for Creating a Session for the Hash Implementation
 *
 *******************************************************************************
 */
static int create_session_ahash(char *alg_name, struct fcrypt *fcr,
			struct session_op *sop)
{
	struct csession	*ses_new;
	struct csession	*ses_ptr;
	struct crypto_ahash *tfm;
	char *keyp = NULL;
	int ret = 0;

	tfm = crypto_alloc_ahash(alg_name, 0, 0);
	if (IS_ERR(tfm)) {
		printk(KERN_INFO PFX "Failed to load ahash "
			"transform for %s: %ld \n", alg_name, PTR_ERR(tfm));
		return -EINVAL;
	}
	crypto_ahash_clear_flags(tfm, ~0);

	/* Copy the key(hmac) from user and set to TFM. */
	if (sop->mackey && (sop->mac != CRYPTO_MD5) && (sop->mac != CRYPTO_SHA1)
		&& (sop->mac != CRYPTO_SHA2_256)
		&& (sop->mac != CRYPTO_SHA2_384)
		&& (sop->mac != CRYPTO_SHA2_512)
		&& (sop->mac != CRYPTO_RIPEMD160)) {
		keyp = kmalloc(sop->mackeylen, GFP_KERNEL);
		if (unlikely(!keyp)) {
			crypto_free_ahash(tfm);
			return -ENOMEM;
		}

		if (copy_from_user(keyp, sop->mackey, sop->mackeylen)) {
			printk(KERN_INFO PFX "Copy of Key Failed from User"
			"space for %s\n", alg_name);
			kfree(keyp);
			crypto_free_ahash(tfm);
			return -EFAULT;
		}

		ret = crypto_ahash_setkey(tfm, keyp, sop->mackeylen);
		kfree(keyp);
		if (ret) {
			printk(KERN_INFO PFX
				"Setting key failed for %s-%zu: flags=0x%X\n",
				alg_name, sop->mackeylen * 8,
				crypto_ahash_get_flags(tfm));
			printk(KERN_INFO PFX
				"(see CRYPTO_TFM_RES_* in "
				"<linux/crypto.h> for details)\n");

			crypto_free_ahash(tfm);
			return -EINVAL;
		}
	}

	ses_new = kmalloc(sizeof(*ses_new), GFP_KERNEL);
	if (!ses_new) {
		crypto_free_ahash(tfm);
		return -ENOMEM;
	}

	memset(ses_new, 0, sizeof(*ses_new));
	get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
	ses_new->tfm_ahash = tfm;

	ses_new->mode = CRYPTO_AHASH;
	init_MUTEX(&ses_new->sem);

	down(&fcr->sem);

restart:
	list_for_each_entry(ses_ptr, &fcr->list, entry) {
		/* Check for duplicate SID */
		if (unlikely(ses_new->sid == ses_ptr->sid)) {
			get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
			goto restart;
		}
	}
	list_add(&ses_new->entry, &fcr->list);
	up(&fcr->sem);

	/* Fill in some values for the user. */
	sop->ses = ses_new->sid;

	return 0;
}

/*******************************************************************************
 * Routine for Creating a Session for the Crypto Implementation
 *
 *******************************************************************************
 */
static int create_session_ablkcipher(char *alg_name, struct fcrypt *fcr,
			struct session_op *sop)
{
	struct csession	*ses_new, *ses_ptr;
	struct crypto_ablkcipher *tfm;
	char *keyp = NULL;
	int ret = 0;

	tfm = crypto_alloc_ablkcipher(alg_name, 0, 0);
	if (IS_ERR(tfm)) {
		printk(KERN_INFO PFX "Failed to load crypto "
			"transform for %s: %ld\n", alg_name, PTR_ERR(tfm));
		return -EINVAL;
	}

	crypto_ablkcipher_clear_flags(tfm, ~0);

	/* Copy the key from user and set to TFM. */
	keyp = kmalloc(sop->keylen, GFP_KERNEL);
	if (unlikely(!keyp)) {
		crypto_free_ablkcipher(tfm);
		return -ENOMEM;

	}

	if (copy_from_user(keyp, sop->key, sop->keylen)) {
		printk(KERN_INFO PFX "Copy of Key Failed from User"
		"space for %s\n", alg_name);
		kfree(keyp);
		crypto_free_ablkcipher(tfm);
		return -EFAULT;
	}

	ret = crypto_ablkcipher_setkey(tfm, keyp, sop->keylen);
	kfree(keyp);
	if (ret) {
		printk(KERN_INFO PFX
			"Setting key failed for %s-%zu: flags=0x%X\n",
			alg_name, sop->keylen*8,
			crypto_ablkcipher_get_flags(tfm));
		printk(KERN_INFO PFX
			"(see CRYPTO_TFM_RES_* in <linux/crypto.h> for "
			"details)\n");

		crypto_free_ablkcipher(tfm);
		return -EINVAL;
	}

	ses_new = kmalloc(sizeof(*ses_new), GFP_KERNEL);
	if (!ses_new) {
		crypto_free_ablkcipher(tfm);
		return -ENOMEM;
	}

	memset(ses_new, 0, sizeof(*ses_new));
	get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
	ses_new->tfm_ablkcipher = tfm;

	ses_new->mode = CRYPTO_ACIPHER;
	init_MUTEX(&ses_new->sem);

	down(&fcr->sem);

restart:
	list_for_each_entry(ses_ptr, &fcr->list, entry) {
		/* Check for duplicate SID */
		if (unlikely(ses_new->sid == ses_ptr->sid)) {
			get_random_bytes(&ses_new->sid, sizeof(ses_new->sid));
			goto restart;
		}
	}
	list_add(&ses_new->entry, &fcr->list);
	up(&fcr->sem);

	/* Fill in some values for the user. */
	sop->ses = ses_new->sid;

	return 0;
}

/*******************************************************************************
 *  Everything that needs to be done when removing a session.
 *
 *******************************************************************************
 */
static inline void cryptodev_destroy_session(struct csession *ses_ptr)
{
	if (down_trylock(&ses_ptr->sem)) {
		CRYPTODEV_PRINTK(2, KERN_DEBUG,
			"Waiting for semaphore of sid=0x%08X\n",
			ses_ptr->sid);
		down(&ses_ptr->sem);
	}
	CRYPTODEV_PRINTK(2, KERN_DEBUG, "Removed session 0x%08X\n",
			 ses_ptr->sid);

	/* Check for mode and then delete */
	switch (ses_ptr->mode) {
	case CRYPTO_ACIPHER:
		crypto_free_ablkcipher(ses_ptr->tfm_ablkcipher);
		ses_ptr->tfm_ablkcipher = NULL;
		break;
	case CRYPTO_AHASH:
		crypto_free_ahash(ses_ptr->tfm_ahash);
		ses_ptr->tfm_ahash = NULL;
		break;
	case CRYPTO_AEAD:
		crypto_free_aead(ses_ptr->tfm_aead);
		ses_ptr->tfm_aead = NULL;
		break;
	}
	up(&ses_ptr->sem);
	kfree(ses_ptr);
}

/*******************************************************************************
 * Look up a session by ID and remove.
 *
 *******************************************************************************
 */
static int cryptodev_finish_session(struct fcrypt *fcr, u32 sid)
{
	struct csession  *tmp;
	struct csession  *ses_ptr;
	struct list_head *head;
	int ret = 0;

	down(&fcr->sem);
	head = &fcr->list;
	list_for_each_entry_safe(ses_ptr, tmp, head, entry) {
		if (ses_ptr->sid == sid) {
			list_del(&ses_ptr->entry);
			cryptodev_destroy_session(ses_ptr);
			break;
		}
	}

	if (!ses_ptr) {
		CRYPTODEV_PRINTK(1, KERN_ERR,
			"Session with sid=0x%08X not found!\n", sid);
		ret = -ENOENT;
	}
	up(&fcr->sem);

	return ret;
}

/*******************************************************************************
 * Remove all sessions when closing the file
 *
 *******************************************************************************
 */
static int cryptodev_finish_all_sessions(struct fcrypt *fcr)
{
	struct csession *tmp;
	struct csession *ses_ptr;

	down(&fcr->sem);
	list_for_each_entry_safe(ses_ptr, tmp, &fcr->list, entry) {
		list_del(&ses_ptr->entry);
		cryptodev_destroy_session(ses_ptr);
	}
	up(&fcr->sem);

	return 0;
}

/*******************************************************************************
 * Look up session by session ID. The returned session is locked.
 *******************************************************************************
 */
static struct csession *cryptodev_get_session_by_sid(struct fcrypt *fcr,
							u32 sid)
{
	struct csession *ses_ptr;

	down(&fcr->sem);
	list_for_each_entry(ses_ptr, &fcr->list, entry) {
		if (ses_ptr->sid == sid) {
			down(&ses_ptr->sem);
			break;
		}
	}
	up(&fcr->sem);

	return ses_ptr;
}

static void cryptodev_release_session(struct csession *session)
{
	if (session)
		up(&session->sem);
}

/*******************************************************************************
 * This is the main crypto function - feed it with plaintext
 * and get a ciphertext
 *******************************************************************************
 */
static int cryptodev_run(struct fcrypt *fcr, struct crypt_op *cop)
{

	struct csession *ses_ptr;
	int ret = 0;

	if (cop->op &&
	    (cop->op != COP_ENCRYPT && cop->op != COP_DECRYPT)) {
		printk(KERN_INFO PFX "invalid operation op=%u\n", cop->op);
		return -EINVAL;
	}

	ses_ptr = cryptodev_get_session_by_sid(fcr, cop->ses);
	if (!ses_ptr) {
		printk(KERN_INFO PFX "invalid session ID=0x%08X\n", cop->ses);
		return -EINVAL;
	}

	switch (ses_ptr->mode) {
	case CRYPTO_ACIPHER:
		ret = cryptodev_run_acipher(ses_ptr, cop);
		break;
	case CRYPTO_AHASH:
		ret = cryptodev_run_ahash(ses_ptr, cop);
		break;
	case CRYPTO_AEAD:
		ret = cryptodev_run_aead(ses_ptr, cop);
		break;
	}
	cryptodev_release_session(ses_ptr);

	return ret;

}

static int sg_setup_copy(void __user *buf, int size, struct scatterlist *sg, int write)
{
	void *data = (void *) __get_free_pages(GFP_KERNEL, get_order(size));

	if (unlikely(!data))
		return -ENOMEM;

	sg_init_one(sg, data, size);

	return 0;
}

static void sg_free_copy(struct scatterlist *sg, int write)
{
	free_pages((unsigned long)sg_virt(sg), get_order(sg->length));
}

static int sg_setup_page(void __user *buf, int size, struct scatterlist *sg, int write)
{
	struct vm_area_struct *vma;
	struct mm_struct *mm = current->mm;
	unsigned long start = (unsigned long)buf;

	vma = find_vma(mm, start);

	if (vma && vma->vm_start <= start && (vma->vm_flags & (VM_IO | VM_PFNMAP))) {
		unsigned long offset = start - vma->vm_start;
		unsigned long pfn = vma->vm_pgoff + (offset >> PAGE_SHIFT);
		struct page *page;
		if (!pfn_valid(pfn) || !pfn_valid(vma->vm_pgoff + ((offset + size) >> PAGE_SHIFT)))
			return -EFAULT;
		page = pfn_to_page(pfn);
		get_page(page);
		sg_set_page(sg, page, size, offset_in_page(offset));
		sg_mark_end(sg);
	} else {
		struct page *pages[SCATTERLIST_MAX];
		int i;
		unsigned long offset = offset_in_page(start);
		int nr_pages = (offset + size + PAGE_SIZE - 1) >> PAGE_SHIFT;
		nr_pages = get_user_pages_fast(start, nr_pages, write, pages);
		if (nr_pages < 0)
			return nr_pages;
		sg_init_table(sg, nr_pages);
		for (i = 0; i < nr_pages; i++) {
			unsigned int length = PAGE_SIZE - offset;
			length = length < size ? length : size;
			sg_set_page(&sg[i], pages[i], length, offset);
			size -= length;
			offset = 0;
		}
	}
	return 0;
}

static void sg_free_page(struct scatterlist *sg, int write)
{
	do {
		if (write)
			flush_dcache_page(sg_page(sg));
		put_page(sg_page(sg));
	} while (!sg_is_last(sg++));
}

static int sg_setup_user(void __user *buf, int size, struct scatterlist *sg, int write)
{
	return use_zcopy ? sg_setup_page(buf, size, sg, write) :
			   sg_setup_copy(buf, size, sg, write) ;
}

static void sg_free_user(struct scatterlist *sg, int write)
{
	if (use_zcopy)
		sg_free_page(sg, write);
	else
		sg_free_copy(sg, write);
}

/*******************************************************************************
 * This is the actual aead function that implements
 * the Combined mode
 *******************************************************************************
 */
static int cryptodev_run_aead(struct csession *ses_ptr, struct crypt_op *cop)
{
	char *ivp  = NULL;
	char __user *src;
	char __user *dst;
	struct scatterlist sg_src[SCATTERLIST_MAX];
	struct scatterlist sg_dst[SCATTERLIST_MAX];
	struct scatterlist asg[1];
	struct aead_request *req;
	struct async_result result;
	size_t bufsize;
	size_t ivsize;
	size_t authsize;
	int ret = 0;

	/* Checking the Input Length */
	bufsize = cop->len;
	if (cop->len > CRYPTO_MAX_DATA_LEN) {
		printk(KERN_INFO PFX "Maximum Data Size Exceeded: %d > %d\n",
			cop->len, CRYPTO_MAX_DATA_LEN);
		return -E2BIG;
	}

	init_completion(&result.completion);

	/* Setting the resquest */
	req = aead_request_alloc(ses_ptr->tfm_aead, GFP_KERNEL);
	if (!req) {
		printk(KERN_INFO PFX "failed to allocate request");
		return -EINVAL;
	}

	aead_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
				  cryptodev_async_complete, &result);

	src = cop->src;
	dst = cop->dst;

	authsize = crypto_aead_authsize(ses_ptr->tfm_aead);

	ivsize = crypto_aead_ivsize(ses_ptr->tfm_aead);

	ivp = kmalloc(ivsize, GFP_KERNEL);
	if (unlikely(!ivp)) {
		ret = -ENOMEM;
		goto out_req;
	}

	if (cop->iv && copy_from_user(ivp, cop->iv, ivsize)) {
		printk(KERN_INFO PFX "Copy of src iv Failed from User "
		"space for aead\n");
		ret = -EFAULT;
		goto out_iv;
	}

	if ((ret = sg_setup_user(src, bufsize + authsize, sg_src, 0)))
		goto out_iv;

	if ((ret = sg_setup_user(dst, bufsize + authsize, sg_dst, 1))) {
		sg_free_user(sg_src, 0);
		goto out_iv;
	}

	if (!use_zcopy && copy_from_user(sg_virt(sg_src), src, bufsize)) {
		printk(KERN_INFO PFX "Copy of src data Failed from User"
		"space for aead\n");
		ret = -EFAULT;
		goto out;
	}

	/* Additional Associated data set to 0 bytes */
	sg_init_one(&asg[0], ivp, 0);

	aead_request_set_crypt(req, sg_src, sg_dst, bufsize, ivp);
	aead_request_set_assoc(req, asg, 0);

	if (cop->op == COP_ENCRYPT)
		ret = crypto_aead_encrypt(req);
	else
		ret = crypto_aead_decrypt(req);
	switch (ret) {
	case 0:
		break;
	case -EINPROGRESS:
	case -EBUSY:
		ret = wait_for_completion_interruptible(
			&result.completion);
		if (!ret)
			ret = result.err;
		if (!ret) {
			INIT_COMPLETION(result.completion);
			break;
		}
		/* fall through */
	default:
		printk("%s () failed err=%d\n", "enc/dec", -ret);
		goto out;
	}

	CRYPTODEV_HEXDUMP(sg_virt(sg_dst), bufsize + authsize);
	if (!use_zcopy && copy_to_user(dst, sg_virt(sg_dst), bufsize + authsize)) {
		printk(KERN_INFO PFX "Copy of enc data Failed to User"
		"space for aead\n");
		ret = -EFAULT;
	}

out:
	sg_free_user(sg_src, 0);
	sg_free_user(sg_dst, 1);

out_iv:
	kfree(ivp);

out_req:
	aead_request_free(req);

	return ret;
}

/*******************************************************************************
 * This is the actual hash function that creates the
 * authenticated data
 *******************************************************************************
 */
static int cryptodev_run_ahash(struct csession *ses_ptr, struct crypt_op *cop)
{

	char __user *src;
	char __user *mac;
	struct scatterlist sg[SCATTERLIST_MAX];
	struct ahash_request *req;
	struct async_result result;
	size_t authsize;
	size_t bufsize;
	int ret = 0;
	char digest_result[64];

	/* Checking the Input Length */
	bufsize = cop->len;
	if (cop->len > CRYPTO_MAX_DATA_LEN) {
		printk(KERN_INFO PFX "Maximum Data Size Exceeded: %d > %d\n",
		 cop->len, CRYPTO_MAX_DATA_LEN);
		return -E2BIG;
	}

	init_completion(&result.completion);

	/* Setting the resquest */
	req = ahash_request_alloc(ses_ptr->tfm_ahash, GFP_KERNEL);
	if (!req) {
		printk(KERN_INFO PFX "failed to allocate request");
		return -EINVAL;
	}

	authsize = crypto_ahash_digestsize(ses_ptr->tfm_ahash);
	memset(digest_result, 0, 64);

	ahash_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
					cryptodev_async_complete, &result);

	src = cop->src;
	mac = cop->mac;

	if ((ret = sg_setup_user(src, bufsize, sg, 0)))
		goto out_req;

	if (!use_zcopy && copy_from_user(sg_virt(sg), src, bufsize)) {
		printk(KERN_INFO PFX "Copy of src data Failed from User"
		"space for hash\n");
		ret = -EFAULT;
		goto out;
	}

	ahash_request_set_crypt(req, sg, digest_result, bufsize);
	ret = crypto_ahash_digest(req);

	switch (ret) {
	case 0:
		break;
	case -EINPROGRESS:
	case -EBUSY:
		ret = wait_for_completion_interruptible(
			&result.completion);
		if (!ret)
			ret = result.err;
		if (!ret) {
			INIT_COMPLETION(result.completion);
			break;
		}
		/* fall through */
	default:
		printk(KERN_INFO PFX "%s failed err=%d\n", "enc/dec", -ret);
		goto out;
	}

	CRYPTODEV_HEXDUMP(digest_result, authsize);
	if (copy_to_user(mac, digest_result, authsize)) {
		printk(KERN_INFO PFX "Copy of mac data Failed to User"
		"space for hash\n");
		ret = -EFAULT;
	}

out:
	sg_free_user(sg, 0);

out_req:
	ahash_request_free(req);

	return ret;
}

/*******************************************************************************
 * This is the actual crypto function that creates the
 * encrypted or decrypted data
 *******************************************************************************
 */
static int cryptodev_run_acipher(struct csession *ses_ptr, struct crypt_op *cop)
{
	char *ivp = NULL;
	char __user *src;
	char __user *dst;
	struct scatterlist sg_src[17];
	struct scatterlist sg_dst[17];
	struct ablkcipher_request *req;
	struct async_result result;
	size_t bufsize;
	size_t ivsize;
	int ret = 0;

	/* Checking the Input Length */
	bufsize = cop->len;
	if (cop->len > CRYPTO_MAX_DATA_LEN) {
		printk(KERN_INFO PFX "Maximum Data Size Exceeded: %d > %d\n",
		 cop->len, CRYPTO_MAX_DATA_LEN);
		return -E2BIG;
	}

	init_completion(&result.completion);

	/* Setting the request */
	req = ablkcipher_request_alloc(ses_ptr->tfm_ablkcipher, GFP_KERNEL);
	if (!req) {
		printk(KERN_INFO PFX "failed to allocate request\n");
		return -EINVAL;
	}

	if (bufsize % crypto_ablkcipher_blocksize(ses_ptr->tfm_ablkcipher)) {
		printk(KERN_INFO PFX
			"data size (%zu) isn't a multiple of block size (%u)\n",
			bufsize, crypto_ablkcipher_blocksize
			(ses_ptr->tfm_ablkcipher));
		ret = -EINVAL;
		goto out_req;
	}

	ivsize = crypto_ablkcipher_ivsize(ses_ptr->tfm_ablkcipher);

	ivp = kmalloc(ivsize, GFP_KERNEL);
	if (unlikely(!ivp)) {
		ret = -ENOMEM;
		goto out_req;
	}

	if (cop->iv && copy_from_user(ivp, cop->iv, ivsize)) {
		printk(KERN_INFO PFX "Copy of src iv Failed from User "
		"space for crypto\n");
		ret = -EFAULT;
		goto out_iv;
	}

	ablkcipher_request_set_callback(req, CRYPTO_TFM_REQ_MAY_BACKLOG,
					cryptodev_async_complete, &result);

	src = cop->src;
	dst = cop->dst;

	if ((ret = sg_setup_user(src, bufsize, sg_src, 0)))
		goto out_iv;

	if ((ret = sg_setup_user(dst, bufsize, sg_dst, 1))) {
		sg_free_user(sg_src, 0);
		goto out_iv;
	}

	if (!use_zcopy && copy_from_user(sg_virt(sg_src), src, bufsize)) {
		ret = -EFAULT;
		goto out;
	}

	ablkcipher_request_set_crypt(req, sg_src, sg_dst, bufsize, ivp);

	if (cop->op == COP_ENCRYPT)
		ret = crypto_ablkcipher_encrypt(req);
	else
		ret = crypto_ablkcipher_decrypt(req);
	switch (ret) {
	case 0:
		break;
	case -EINPROGRESS:
	case -EBUSY:
		ret = wait_for_completion_interruptible(
			&result.completion);
		if (!ret)
			ret = result.err;
		if (!ret) {
			INIT_COMPLETION(result.completion);
			break;
		}
		/* fall through */
	default:
		printk(KERN_INFO PFX "%s failed err=%d\n", "enc/dec", -ret);
		goto out;
	}

	CRYPTODEV_HEXDUMP(sg_virt(sg_dst), bufsize);
	if (!use_zcopy && copy_to_user(dst, sg_virt(sg_dst), bufsize)) {
		printk(KERN_INFO PFX "Copy of enc data Failed to User"
		"space for crypto\n");
		ret = -EFAULT;
	}

out:
	sg_free_user(sg_src, 0);
	sg_free_user(sg_dst, 1);

out_iv:
	kfree(ivp);

out_req:
	ablkcipher_request_free(req);

	return ret;
}

 /*****************************************************************************
 * /dev/crypto function operation functions
 ******************************************************************************
 */
static int cryptodev_clonefd(struct file *filp)
{
	mm_segment_t fs;
	int fd;

	fs = get_fs();
	set_fs(get_ds());
	for (fd = 0; fd < files_fdtable(current->files)->max_fds; fd++)
		if (files_fdtable(current->files)->fd[fd] == filp)
			break;
	fd = __sys_dup(fd);
	set_fs(fs);
	return fd;
}

static int cryptodev_open(struct inode *inode, struct file *filp)
{
	struct fcrypt *fcr;

	fcr = kmalloc(sizeof(*fcr), GFP_KERNEL);
	if (!fcr)
		return -ENOMEM;

	memset(fcr, 0, sizeof(*fcr));
	init_MUTEX(&fcr->sem);
	INIT_LIST_HEAD(&fcr->list);
	filp->private_data = fcr;

	return 0;
}

static int cryptodev_release(struct inode *inode, struct file *filp)
{
	struct fcrypt *fcr = filp->private_data;

	if (fcr) {
		cryptodev_finish_all_sessions(fcr);
		kfree(fcr);
		filp->private_data = NULL;
	}
	return 0;
}

static int cryptodev_ioctl(struct inode *inode, struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	struct session_op sop;
	struct crypt_op   cop;
	struct fcrypt *fcr = filp->private_data;
	unsigned int  ses;
	int ret;
	int fd, feat;

	if (!fcr)
		BUG();

	switch (cmd) {
	case CRIOGET:
		fd = cryptodev_clonefd(filp);
		put_user(fd, (int *) arg);
		return IS_ERR_VALUE(fd) ? fd : 0;

	case CIOCGSESSION:
		if (copy_from_user(&sop, (void *) arg, sizeof(sop))) {
			printk(KERN_INFO PFX "Copy of Session data failed"
				"at CIOCGSESSION from user space\n");
			return -EFAULT;
		}
		ret = cryptodev_create_session(fcr, &sop);
		if (ret)
			return ret;
		if (copy_to_user((void *)arg, &sop, sizeof(sop))) {
			printk(KERN_INFO PFX "Copy of Session data failed"
				"at CIOCGSESSION to user space\n");
			return -EFAULT;
		}
		return 0;

	case CIOCFSESSION:
		get_user(ses, (u32 *) arg);
		return cryptodev_finish_session(fcr, ses);

	case CIOCCRYPT:
		if (copy_from_user(&cop, (void *) arg, sizeof(cop))) {
			printk(KERN_INFO PFX "Copy of src data failed"
				"at CIOCCRYPT from user space\n");
			return -EFAULT;
		}
		ret = cryptodev_run(fcr, &cop);
		if (copy_to_user((void *) arg, &cop, sizeof(cop))) {
			printk(KERN_INFO PFX "Copy of enc/dec/hash data failed"
				"at CIOCCRYPT to user space\n");
			return -EFAULT;
		}
		return ret;

	case CIOCASYMFEAT:
		/* No Asymmetric Algorithms Supported */
		feat = 0;
		if (copy_to_user((void *)arg, &feat, sizeof(feat))) {
			printk(KERN_INFO PFX "Copy of asymm algorithm data"
				" failed at CIOCASYMFEAT to user space\n");
			return -EFAULT;
		}
		return 0;

	default:
		printk(KERN_ERR PFX "un-supported command 0x%08X\n", cmd);
		return -EINVAL;
	}
}

struct file_operations cryptodev_fops = {
	.owner   = THIS_MODULE,
	.open    = cryptodev_open,
	.release = cryptodev_release,
	.ioctl   = cryptodev_ioctl,
};

struct miscdevice cryptodev = {
	.minor = CRYPTODEV_MINOR,
	.name  = "crypto",
	.fops  = &cryptodev_fops,
};

static int cryptodev_register(void)
{
	int rc;

	rc = misc_register(&cryptodev);
	if (rc) {
		printk(KERN_ERR PFX "registeration of /dev/crypto failed\n");
		return rc;
	}

	return 0;
}

static void cryptodev_deregister(void)
{
	misc_deregister(&cryptodev);
}

/*******************************************************************************
 * Module init/exit
 *******************************************************************************
 */
int __init init_cryptodev(void)
{
	int rc;

	rc = cryptodev_register();
	if (rc)
		return rc;

	printk(KERN_INFO PFX "CryptoAPI driver v%s loaded\n",
		CRYPTODEV_UI_SUPPORT_DRIVER);

	return 0;
}

void __exit exit_cryptodev(void)
{
	cryptodev_deregister();
	printk(KERN_INFO PFX "CryptoAPI driver v%s unloaded\n",
		CRYPTODEV_UI_SUPPORT_DRIVER);
}

module_init(init_cryptodev);
module_exit(exit_cryptodev);

MODULE_AUTHOR("Shasi Pulijala <spulijala@amcc.com>");
MODULE_DESCRIPTION("CryptoDev driver");
MODULE_LICENSE("Dual BSD/GPL");
