/**mmuinfo.c
* Procfs interface to read MMU configuration
 */

/*HEADER_START*/
/* Copyright 2006 Mobilygen, Remi Machet
*
*  This program is free software; you can redistribute  it and/or modify it
*  under  the terms of  the GNU General  Public License as published by the
*  Free Software Foundation;  either version 2 of the  License, or (at your
*  option) any later version.
*
*  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
*  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
*  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
*  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
*  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
*  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
*  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
*  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
*  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*  You should have received a copy of the  GNU General Public License along
*  with this program; if not, write  to the Free Software Foundation, Inc.,
*  675 Mass Ave, Cambridge, MA 02139, USA.
*
 */
/*HEADER_STOP*/

#include <linux/proc_fs.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/io.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/version.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,6,18)
#include <linux/config.h>
#endif

#ifdef CONFIG_PROC_FS

static int print_ttb(char *buf, char **start, off_t offset,int count, int *eof, void *data)
{   unsigned long TTB;
	int len=0;
	asm volatile("mrc p15, 0, %0, c2, c0\n" : "=r" (TTB) : );
	len+=sprintf(buf+len,"TTB address: 0x%08lx\n",TTB & 0xFFFFC000);
	*eof=1;
	return len;
}

static int print_mmap(char *buf, char **start, off_t offset,int count, int *eof, void *data)
{   
	static unsigned long i=0, j=0;
	static unsigned long *TTB, *STTB;
	int len=0;

	*start=buf;
	if(count<60)
		return 0;
	if(offset==0) {   
		i=0;j=0;

		asm volatile("mrc p15, 0, %0, c2, c0\n" : "=r" ((unsigned long)TTB) : );
		TTB=(unsigned long *)((unsigned long)TTB & 0xFFFFC000);
		len+=sprintf(buf+len,"TTB address: 0x%p\n",TTB);
		TTB=ioremap((unsigned long)TTB, 4096<<2);
	} 
	else if(TTB==NULL)    /* Someone needs to be reminded the last packet has eof set */
	{   
		*eof=1;
		return 0;
	};
	for(; i<4096; i++)
	{   
		switch(TTB[i] & 0x3)
		{   
			case 0:
				/* 1MB not allocated */
				break;
			case 2:
				len+=sprintf(buf+len,
						"0x%08lx->0x%08lx 1MB (AP=%ld, domain=%ld, fl=%c%c)\n",
						i<<20,TTB[i] & 0xFFF00000, (TTB[i]>>10) & 3, 
						(TTB[i]>>5) & 0xF,
						((TTB[i]>>3) & 1) ? 'C' : ' ',
						((TTB[i]>>2) & 1) ? 'B' : ' ');
				break;
			case 1:
				if(j==0)
					STTB=ioremap(TTB[i] & 0xFFFFFC00, 256<<2);
				for(; j<256; j++)
				{   
					if(len>=count-60)
						break;
					switch(STTB[j] & 0x3)
					{   
						case 0:
							/* 4kB not allocated */
							break;
						case 1:
							len+=sprintf(buf+len,
									"0x%08lx->0x%08lx 64kB (Perm=0x%02lx, fl=%c%c)\n",
									(i<<20) | (j<<12),STTB[j] & 0xFFFF0000,
									(STTB[j]>>4) & 0xFF,
									((STTB[j]>>3) & 1) ? 'C' : ' ',
									((STTB[j]>>2) & 1) ? 'B' : ' ');
							j+=15;  /* This one will be duplicated over the 64kB == 16*4kB */
							break;
						case 2:
							len+=sprintf(buf+len,
									"0x%08lx->0x%08lx 4kB (Perm=0x%02lx, fl=%c%c)\n",
									(i<<20) | (j<<12),STTB[j] & 0xFFFFF000,
									(STTB[j]>>4) & 0xFF,
									((STTB[j]>>3) & 1) ? 'C' : ' ',
									((STTB[j]>>2) & 1) ? 'B' : ' ');
							break;
						case 3:
							/* 4kB not allocated */
							break;
					};
				};
				if(j==256) 
				{   
					j=0;
					iounmap(STTB);
				};
				break;
			case 3:
				if(j==0)
					STTB=ioremap(TTB[i] & 0xFFFFF000, 256<<2);
				for(; j<1024; j++)
				{   if(len>=count-60)
					break;
					switch(STTB[j] & 0x3)
					{   case 0:
						/* 1kB not allocated */
						break;
						case 1:
						len+=sprintf(buf+len,
								"0x%08lx->0x%08lx 64kB (Perm=0x%02lx, fl=%c%c)\n",
								(i<<20) | (j<<10),
								STTB[j] & 0xFFFF0000,(STTB[j]>>4) & 0xFF,
								((STTB[j]>>3) & 1) ? 'C' : ' ',
								((STTB[j]>>2) & 1) ? 'B' : ' ');
						j+=63;  /* This one will be duplicated over the 64kB == 64*1kB */
						break;
						case 2:
						len+=sprintf(buf+len,
								"0x%08lx->0x%08lx 4kB (Perm=0x%02lx, fl=%c%c)\n",
								(i<<20) | (j<<10),
								STTB[j] & 0xFFFFF000,(STTB[j]>>4) & 0xFF,
								((STTB[j]>>3) & 1) ? 'C' : ' ',
								((STTB[j]>>2) & 1) ? 'B' : ' ');
						j+=3;  /* This one will be duplicated over the 4kB == 4*1kB */
						break;
						case 3:
						len+=sprintf(buf+len,
								"0x%08lx->0x%08lx 1kB (Perm=0x%01lx, fl=%c%c)\n",
								(i<<20) | (j<<10),
								STTB[j] & 0xFFFFFC00,(STTB[j]>>4) & 0x3,
								((STTB[j]>>3) & 1) ? 'C' : ' ',
								((STTB[j]>>2) & 1) ? 'B' : ' ');                        break;
						break;
					};
				};
				if(j==1024) 
				{   j=0;
					iounmap(STTB);
				};
				break;
		};
		if(len>=count-60)
		{   if(j==0) i++;
			break;
		};
	};
	buf[len]=0;
	if(i==4096)
	{   iounmap(TTB);
		TTB=NULL;
		i=0;
		*eof=1;
	};
	return len;
}

static int __init mmuinfo_init(void)
{   struct proc_dir_entry *mmu_dir_p;

	/* Create directory */
	mmu_dir_p=proc_mkdir("mmu",NULL);
	if(mmu_dir_p==NULL)
	{   printk(KERN_ERR "Failed to create MMU /proc interface directory\n");
		return -ENOMEM;
	};
	create_proc_read_entry("ttb", 0, mmu_dir_p, print_ttb, NULL);
	create_proc_read_entry("memorymap", 0, mmu_dir_p, print_mmap, NULL);
	printk(KERN_INFO "Merlin MMU debug interface initialized\n");
	return 0;
}

static void __exit mmuinfo_cleanup(void)
{   remove_proc_entry("mmu/ttb", NULL);
	remove_proc_entry("mmu/memorymap", NULL);
	remove_proc_entry("mmu", NULL);
}

module_init(mmuinfo_init);
module_exit(mmuinfo_cleanup);

#ifdef MODULE
MODULE_AUTHOR("Remi Machet");
MODULE_DESCRIPTION("Merlin MMU debug interface");
MODULE_LICENSE("GPL");
#endif	/* #ifdef MODULE */

#endif
