/*HEADER_START*/
/*
 * linux/arch/arm/mach-merlin/reset.c
 *
 * Copyright (C) Mobilygen Corp
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

/* enable doxygen */
/** \file
 * Driver for Mobilygen Reset block
 *
 * Many devices in the SOC have the ability to be put in reset.  This
 * can be used to conserve power when a device is not going to be used.
 * At bootup time, most devices are being held in reset and therfor
 * must be taken out of reset.  The normal sequence is during init/probe
 * a driver should called
 *
 * mobi_reset_disable(<device_id>)
 *
 * and at exit/shutdown call
 *
 * mobi_reset_enable(<device_id>)
 *
 */

#ifndef DOXYGEN_SKIP

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>

#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/delay.h>

#include <mach/platform.h>
#include <mach/mobi_qcc.h>

#include "reset.h"

static struct proc_dir_entry *reset_proc_dir;

static struct reset_dev merlin_resets[] = {
	{ .name	= "aes", 			.id = RESET_ID_AES, },
	{ .name	= "ahb_arm", 		.id = RESET_ID_AHB_ARM, },
	{ .name	= "ahb_arm_mmu", 	.id = RESET_ID_AHB_ARM_MMU, },
	{ .name	= "ahb_dma", 		.id = RESET_ID_AHB_DMA, },
	{ .name	= "ahb_dma", 		.id = RESET_ID_AHB_DMA_MMU, },
	{ .name	= "ahb_periph", 	.id = RESET_ID_AHB_PERIPH, },
	{ .name	= "apb_bridge", 	.id = RESET_ID_APB_BRIDGE, },
	{ .name	= "arm", 			.id = RESET_ID_ARM, },
	{ .name	= "cf",				.id = RESET_ID_CF, },
	{ .name	= "dmac", 			.id = RESET_ID_DMAC, },
	{ .name	= "eh2h_dma", 		.id = RESET_ID_EH2H_DMA, },
	{ .name	= "eh2h_periph", 	.id = RESET_ID_EH2H_PERIPH, },
	{ .name	= "gmac", 			.id = RESET_ID_GMAC, },
	{ .name	= "ictl", 			.id = RESET_ID_ICTL, },
	{ .name	= "mhif", 			.id = RESET_ID_MHIF, },
	{ .name	= "nand", 			.id = RESET_ID_NAND, },
	{ .name	= "otgphy", 		.id = RESET_ID_OTGPHY, },
	{ .name	= "preset", 		.id = RESET_ID_PRESET, },
	{ .name	= "qccsisc", 		.id = RESET_ID_QCCSISC, },
	{ .name	= "qdsm", 			.id = RESET_ID_QDSM, },
	{ .name	= "sdmmc", 			.id = RESET_ID_SDMMC, },
	{ .name	= "sha", 			.id = RESET_ID_SHA, },
	{ .name	= "usbh",			.id = RESET_ID_USBH, },
	{ .name	= "usbp", 			.id = RESET_ID_USBP, },
	{ .name	= "wdt", 			.id = RESET_ID_WDT, },
	{ .name	= "uartdbg", 		.id = RESET_ID_UARTDBG, },
	{ .name	= "uart01", 		.id = RESET_ID_UART01, },
	{ .name	= "pwm", 			.id = RESET_ID_PWM, },
	{ .name	= "ssi", 			.id = RESET_ID_SSI, },
	{ .name	= "gpio", 			.id = RESET_ID_GPIO, },
	{ .name	= "i2c", 			.id = RESET_ID_I2C, },
	{ .name	= "chip", 			.id = RESET_ID_CHIP, },
};
#endif /* DOXYGEN_SKIP */

/**
 * \brief Default timer value if self-clearing is enabled
 */
#define QCC_DEFAULT_RESET_TIMER_VALUE 0xff

/**
 * \brief reset_control:
 * 	Assert or deassert reset for a device
 *
 * \param device_id	- id of device to take action on
 * \param action	- enable or disable
 *
 * \retval -ENODEV - if invalid device_id is provided
 * \retval Zero    - upon success
 *
 */
static uint32_t reset_control(uint32_t device_id, uint8_t action)
{

	int ret = 0;

	switch (device_id) {
	case RESET_ID_AES:
	case RESET_ID_AHB_ARM:
	case RESET_ID_AHB_ARM_MMU:
	case RESET_ID_AHB_DMA:
	case RESET_ID_AHB_PERIPH:
	case RESET_ID_APB_BRIDGE:
	case RESET_ID_ARM:
	case RESET_ID_CF:
	case RESET_ID_DMAC:
	case RESET_ID_EH2H_DMA:
	case RESET_ID_EH2H_PERIPH:
	case RESET_ID_GMAC:
	case RESET_ID_ICTL:
	case RESET_ID_MHIF:
	case RESET_ID_NAND:
	case RESET_ID_PRESET:
	case RESET_ID_QCCSISC:
	case RESET_ID_QDSM:
	case RESET_ID_SDMMC:
	case RESET_ID_SHA:
	case RESET_ID_USBH:
	case RESET_ID_USBP:
	case RESET_ID_WDT:
	case RESET_ID_UARTDBG:
	case RESET_ID_UART01:
	case RESET_ID_PWM:
	case RESET_ID_SSI:
	case RESET_ID_GPIO:
	case RESET_ID_I2C:
		ret = mobi_qcc_write(QCC_BID_CHIPCTL,
				(action == RESET_STATE_ENABLE ?
				 QCC_CHIPCTL_SOC_RESETVEC_SET :
				 QCC_CHIPCTL_SOC_RESETVEC_CLR),
				device_id, 4 /* len */);
		break;
	case RESET_ID_OTGPHY:
		if (action == RESET_STATE_ENABLE) {
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SOC_RESETVEC_SET,
					RESET_ID_USBH, 4);
			if (ret >= 0)
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_SOC_RESETVEC_SET,
						RESET_ID_USBP, 4);
			if (ret >= 0)
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_SOC_RESETVEC_SET,
						RESET_ID_OTGPHY, 4);
		} else {
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SOC_RESETVEC_CLR,
					RESET_ID_OTGPHY, 4);
			if (ret >= 0) {
				/* wait for OTG PHY clock to start toggling */
				udelay(400);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_SOC_RESETVEC_CLR,
						RESET_ID_USBP, 4);
			}
			if (ret >= 0) {
				/* ??? 6 hclock cycles, how long? */
				udelay(50);
				ret = mobi_qcc_write(QCC_BID_CHIPCTL,
						QCC_CHIPCTL_SOC_RESETVEC_CLR,
						RESET_ID_USBH, 4);
			}
		}
		break;
	case RESET_ID_CHIP:
		if (action == RESET_STATE_ENABLE) {
			ret = mobi_qcc_write(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SWCHIP_RESET_SET,
					0x1, 1 /* len */);
		}
		break;
	default:
		return -ENODEV;
		break;
	}
	return ret;
}

/**
 * \brief mobi_reset_enable:
 *  	Put a device into reset
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval Zero   	- on success
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_enable(uint32_t device_id)
{
	return reset_control(device_id, RESET_STATE_ENABLE);
}
EXPORT_SYMBOL(mobi_reset_enable);

/**
 * \brief mobi_reset_disable:
 *  	Put a device into reset
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval Zero   	- on success
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_disable(uint32_t device_id)
{
	return reset_control(device_id, RESET_STATE_DISABLE);
}
EXPORT_SYMBOL(mobi_reset_disable);

/**
 * \brief mobi_reset_state:
 *  	Query the reset state of a device
 *
 * \param device_id - id of device to be put into reset
 *
 * \retval -ENODEV  	- if invalid device id is provided
 * \retval zero   	- if device is not in reset
 * \retval one   	- if device is in reset
 *
 * \remark	- Is Exported
 *
 */
uint32_t mobi_reset_state(uint32_t device_id)
{
	unsigned long reg_value = 0;
	int in_reset = 0;

	switch (device_id) {
	case RESET_ID_AES:
	case RESET_ID_AHB_ARM:
	case RESET_ID_AHB_ARM_MMU:
	case RESET_ID_AHB_DMA:
	case RESET_ID_AHB_PERIPH:
	case RESET_ID_APB_BRIDGE:
	case RESET_ID_ARM:
	case RESET_ID_CF:
	case RESET_ID_DMAC:
	case RESET_ID_EH2H_DMA:
	case RESET_ID_EH2H_PERIPH:
	case RESET_ID_GMAC:
	case RESET_ID_ICTL:
	case RESET_ID_MHIF:
	case RESET_ID_NAND:
	case RESET_ID_OTGPHY:
	case RESET_ID_PRESET:
	case RESET_ID_QCCSISC:
	case RESET_ID_QDSM:
	case RESET_ID_SDMMC:
	case RESET_ID_SHA:
	case RESET_ID_USBH:
	case RESET_ID_USBP:
	case RESET_ID_WDT:
	case RESET_ID_UARTDBG:
	case RESET_ID_UART01:
	case RESET_ID_PWM:
	case RESET_ID_SSI:
	case RESET_ID_GPIO:
	case RESET_ID_I2C:
		in_reset = mobi_qcc_read(QCC_BID_CHIPCTL,
				QCC_CHIPCTL_SOC_RESETVEC_SET,
				&reg_value,
				4);

		if (in_reset == 0)
			if (reg_value & device_id)
				in_reset = 1;
		break;
	case RESET_ID_CHIP:
		in_reset = mobi_qcc_read(QCC_BID_CHIPCTL,
				QCC_CHIPCTL_SWCHIP_RESET_SET,
				&reg_value,
				1);

		if (in_reset == 0)
			if (reg_value & RESET_STATE_ENABLE)
				in_reset = 1;
		break;
	default:
		in_reset = -ENODEV;
		break;
	}

	return in_reset;
}

#ifdef CONFIG_PROC_FS

/**
 * \brief reset_state_proc_rd:
 * 	Create a readable procfs entry for each register reset id.
 */
static int reset_state_proc_rd(char *buf, char **start,
		off_t offset, int count, int *eof, void *data)
{
	int i, len = 0;
	struct reset_dev *reset;
	unsigned long status_reg = 0;

	mobi_qcc_read(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_SOC_RESETVEC_SET, &status_reg, 4);
	len += sprintf(buf+len, "device        state\n");
	len += sprintf(buf+len, "-------------------\n");

	for (i = 0, reset = merlin_resets;
			i < ARRAY_SIZE(merlin_resets); i++, reset++) {

		len += sprintf(buf+len, "%-14s  ", reset->name);
		if (reset->id != RESET_ID_CHIP) {
			len += sprintf(buf+len, "%d",
					(status_reg & reset->id ? 1 : 0));
		} else {
			/* no bitmask for for SWCHIP */
			mobi_qcc_read(QCC_BID_CHIPCTL,
					QCC_CHIPCTL_SWCHIP_RESET_SET,
					&status_reg, 1);
			len += sprintf(buf+len, "%d",
					(status_reg & RESET_STATE_ENABLE ? 1 : 0));
		}
		len += sprintf(buf+len, "\n");
	}
	len += sprintf(buf+len, "\n");
	return len;
}
#endif

/**
 * \brief mobi_reset_init:
 * 	Called at system startup.  Priority is core_initcall
 */
static int __init mobi_reset_init(void)
{

#ifdef CONFIG_PROC_FS
	if (reset_proc_dir == NULL)
		reset_proc_dir = proc_mkdir("driver/reset", NULL);

	if (reset_proc_dir == NULL) {
		reset_proc_dir = proc_mkdir("driver", NULL);
		if (reset_proc_dir != NULL)
			proc_mkdir("driver/reset", NULL);
	}

	if (reset_proc_dir != NULL)
		create_proc_read_entry("driver/reset/status",
				S_IRUSR | S_IRGRP | S_IROTH,
				NULL, reset_state_proc_rd, NULL);
#endif

	/*
	 *  make sure the timers are disable otherwise things will
	 * come out of reset automatically
	 */
	mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_SOC_RESETTIMER_EN,
			0,
			4 /* len */);

	mobi_qcc_write(QCC_BID_CHIPCTL,
			QCC_CHIPCTL_SOC_RESETVEC_SET,
			SOC_DISABLE_ALL_MASK,
			4);

	return 0;

}
core_initcall(mobi_reset_init);
