#include <linux/module.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/mman.h>
#include <linux/shm.h>
#include <linux/vmalloc.h>
#include <linux/io.h>

#include <asm/system.h>
#include <asm/cacheflush.h>
#include <asm/tlbflush.h>
#include <asm/mmu_context.h>

/*
 * We need to ensure that framebuffer mappings are correctly aligned to
 * the PGDIR_SIZE. This gives us complete control over the global page
 * directory entry, allowing us to use fine page tables if we need to.
 */
unsigned long
get_fb_unmapped_area(struct file *filp, unsigned long start_addr,
		unsigned long len, unsigned long pgoff, unsigned long flags)
{
	unsigned long addr;

	/* need a multiple of PGDIR_SIZE */
	len = (len + PGDIR_SIZE-1) & PGDIR_MASK;

	/* aligned to PGDIR_SIZE */
	addr = get_unmapped_area(NULL, start_addr,
			len + PGDIR_SIZE-1, pgoff, flags);

	/* errors are encoded in the low bits, return them */
	if (addr & ~PAGE_MASK)
		return addr;

	/* otherwise return aligned addr */
	addr = (addr + PGDIR_SIZE-1) & PGDIR_MASK;
	return addr;
}
EXPORT_SYMBOL(get_fb_unmapped_area);

static void unmap_area_tiny(unsigned long addr, unsigned long size)
{
	unsigned long prot =
		PMD_TYPE_FINE | PMD_BIT4 | PMD_DOMAIN(DOMAIN_IO);
	unsigned long end = addr + size;
	pgd_t *pgd;

	flush_cache_vunmap(addr, end);
	pgd = pgd_offset_k(addr);
	do {
		pmd_t *pmd = pmd_offset(pgd, addr);

		WARN_ON((pmd[0] & ~PAGE_MASK) != prot);
		WARN_ON((pmd[1] & ~PAGE_MASK) != prot);
		__free_pages(pmd_page(pmd[0]), 1);
		pmd_clear(pmd);
		init_mm.context.kvm_seq++;

		addr += PGDIR_SIZE;
		pgd++;
	} while (addr < end);

	if (current->active_mm->context.kvm_seq !=
			init_mm.context.kvm_seq)

		__check_kvm_seq(current->active_mm);

	flush_tlb_kernel_range(addr, end);
}

int
remap_area_sections(struct vm_area_struct *vma,
		unsigned long addr, unsigned long pfn,
		    unsigned long size, unsigned long flags)
{
	unsigned long prot, end = addr + size;
	struct mm_struct *mm = vma->vm_mm;
	pgd_t *pgd;

	prot = PMD_TYPE_SECT | PMD_SECT_AP_WRITE |
			PMD_SECT_AP_READ | PMD_BIT4 |
			PMD_DOMAIN(DOMAIN_IO) |
	       (flags & (PTE_CACHEABLE | PTE_BUFFERABLE));

	pgd = pgd_offset(mm, addr);
	do {
		pmd_t *pmd = pmd_offset(pgd, addr);

		pmd[0] = __pmd(__pfn_to_phys(pfn) | prot);
		pfn += SZ_1M >> PAGE_SHIFT;
		pmd[1] = __pmd(__pfn_to_phys(pfn) | prot);
		pfn += SZ_1M >> PAGE_SHIFT;
		flush_pmd_entry(pmd);

		addr += PGDIR_SIZE;
		pgd++;
	} while (addr < end);

	return 0;
}

#define TINY_SHIFT 10
#define TINY_SIZE (1<<TINY_SHIFT)
#define TINY_MASK (~(TINY_SIZE-1))
#define __tiny_index(addr)	(((addr) >> TINY_SHIFT) & (2047))
#define pte_offset_tiny(pmd, addr) \
	((unsigned long *)(__va((*(pmd)) & ~(PAGE_SIZE-1))) + __tiny_index(addr))
#define tiny_pte(addr, prot) \
	((((unsigned long)addr)&TINY_MASK) | PTE_EXT_AP_URW_SRW | \
	  (prot & (PTE_CACHEABLE | PTE_BUFFERABLE)) | PTE_TYPE_EXT)

static inline void set_tiny_pte(unsigned long *pte, unsigned long val)
{
	*pte = val;
	asm volatile ("mcr	p15, 0, %0, c7, c10, 1" : : "r" (pte));
	asm volatile ("mcr	p15, 0, %0, c7, c10, 4" : : "r" (pte));
}

static int
remap_tiny_pte(pmd_t *pmd, unsigned long addr,
		unsigned long end, unsigned long phys_addr, pgprot_t prot)
{
	unsigned long *pte = pte_offset_tiny(pmd, addr);
	do {
		if (!pte_none(*pte))
			goto bad;
		set_tiny_pte(pte, tiny_pte(phys_addr, prot));
	} while (pte++, phys_addr += TINY_SIZE,
			addr += TINY_SIZE, addr != end);

	return 0;

 bad:
	printk(KERN_CRIT "%s: tiny page already exists\n",
			__func__);
	BUG();
}

static int
__remap_area_tiny(struct mm_struct *mm, unsigned long addr, unsigned long end,
		    unsigned long phys_addr, unsigned long flags)
{
	unsigned long prot = PMD_TYPE_FINE | PMD_BIT4 | PMD_DOMAIN(DOMAIN_IO);
	unsigned long pmde, next;
	pgd_t *pgd;

	pgd = pgd_offset(mm, addr);
	do {
		pmd_t *pmd = pmd_offset(pgd, addr);

		if (pmd[0] || pmd[1])
			goto bad;

		pmde = __get_free_pages(GFP_KERNEL|__GFP_REPEAT|__GFP_ZERO, 1);
		pmd[0] = __pmd(__pa(pmde) | prot);
		pmd[1] = __pmd(__pa(pmde + PAGE_SIZE) | prot);
		flush_pmd_entry(pmd);

		next = pmd_addr_end(addr, end);
		remap_tiny_pte(pmd, addr, next, phys_addr, flags);

		phys_addr += next - addr;
		addr = next;
		pgd++;
	} while (addr < end);

	return 0;
bad:
	printk(KERN_CRIT "%s: first level directory already exists\n",
			__func__);
	BUG();
}

int
remap_area_tiny(struct vm_area_struct *vma,
		unsigned long addr, unsigned long phys_addr,
		    unsigned long size, unsigned long flags)
{
	return __remap_area_tiny(vma->vm_mm, addr,
			addr+size, phys_addr, flags);
}

void __iomem *ioremap_area_tiny(unsigned long phys_addr,
		unsigned long size, unsigned long flags)
{
	int err;
	unsigned long addr;
	struct vm_struct *area;

	/* need PGDIR_SIZE alignment */
	size = (size + PGDIR_SIZE-1) & PGDIR_MASK;

	area = get_vm_area(size, VM_IOREMAP);
	if (!area)
		return NULL;

	addr = (unsigned long)area->addr;
	if (addr & (PGDIR_SIZE-1)) {
		printk(KERN_CRIT "%s: did not get properly aligned address %p\n",
				__func__, (void *)addr);
		vunmap((void *)addr);
		return NULL;
	}

	err = __remap_area_tiny(&init_mm, addr, addr+size, phys_addr, flags);

	if (err) {
		vunmap((void *)addr);
		return NULL;
	}

	flush_cache_vmap(addr, addr + size);
	return (void __iomem *)addr;
}
EXPORT_SYMBOL(ioremap_area_tiny);

void
iounmap_area_tiny(void __iomem *addr)
{
	struct vm_struct *area;

	if (!addr)
		return;

	write_lock(&vmlist_lock);
	for (area = vmlist; area != NULL; area = area->next) {
		if (area->addr == addr) {
			/* remove gaurd page from size */
			unmap_area_tiny((unsigned long)area->addr,
					area->size - PAGE_SIZE);
			break;
		}
	}
	write_unlock(&vmlist_lock);

	if (unlikely(!area)) {
		printk(KERN_ERR "Trying to iounmap_area_tiny() nonexistent vm area (%p)\n",
				addr);
		WARN_ON(1);
		return;
	}

	vunmap(area->addr);
}
EXPORT_SYMBOL(iounmap_area_tiny);
