/*HEADER_START*/
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/amba/bus.h>
#include <linux/io.h>
#include <asm/mach/irq.h>
#include <linux/mtd/physmap.h>

/* APIs */
#include <mach/mobi_reset.h>
#include <mach/mobi_clock.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_ioctrl.h>

#include <mach/platform.h>
#include <mach/mg3500_devices.h>
#include <mach/au_flash_regs.h>
#include <mach/mhif_regs.h>
#include <mach/ioctrl_regs.h>

/* notes:
 *
 *  chip driver will have an init function called at bootup
 *  it will:
 *  	1) register a platform driver, the probe of this platform
 *  	driver will be responsible for initializing any merlin/mobilygen
 *  	specic stuff
 *  	2) it will also register and amba_device_register
 *  	3) it will contain the add device function just as it does not
 *  	and do the same thing, assign the data, and call device_register
 *  	-- note this platform data will be split in the probe and the
 *  	actual amba platform data will have to be assigned there.
 *
 *  board driver will
 *  	1) initialize data structures for IP and merlin/board specific
 *  	2) call add device
 *
 *  device driver
 *   1) calls amba_driver register
 */

/*
 * We define our own amba device release function so that
 * our amba_device static structs will not be kfree'd().
 */
static void static_amba_device_release(struct device *dev)
{
	struct amba_device *adev = container_of(dev, struct amba_device, dev);
	if (adev->res.parent)
		release_resource(&adev->res);
}

static void empty_release(struct device *dev) {}
static int empty_suspend(struct platform_device *dev,
		pm_message_t state)
{
	return 0;
}
static int empty_resume(struct platform_device *dev)
{
	return 0;
}

/* ----------------------------------------------------------------
 *  Aurora NAND controller
 * ----------------------------------------------------------------*/
#if defined(CONFIG_MTD_NAND_AUMB3000) || \
	defined(CONFIG_MTD_NAND_AUMB3000_MODULE)

#define NANDREG_AMBA_NAME 	AUMB3000_NANDREG_AMBA_NAME
#define NANDREG_AMBA_PID 	AUMB3000_NANDREG_AMBA_DEVID
#define NANDDATA_AMBA_NAME 	AUMB3000_NANDDATA_AMBA_NAME
#define NANDDATA_AMBA_PID 	AUMB3000_NANDDATA_AMBA_DEVID
static struct aumb3000_nand_device_data_t nandreg_amba_pdata;

static struct mobi_hmux_drvdata nand_hmux_data;
static struct platform_device nand_hmux_device = {
	.name = "mobi_hmux",
	.id   = 0x0,
	.dev  = {
		.release       = empty_release,
		.platform_data = &nand_hmux_data,
	},
};

static int nand_arch_init(struct nand_board_config_t *config)
{
	uint32_t qcc_sysconfig_shadow;

	qcc_sysconfig_shadow =
		mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);

	qcc_sysconfig_shadow &= ~(MOBI_QCCSISC_SYS_CONFIG_NAND_FLASH_EN_MASK |
			MOBI_QCCSISC_SYS_CONFIG_NANDNOR_RAM_EN_MASK);
	mobi_qcc_writereg(qcc_sysconfig_shadow, MOBI_QCCSISC_SYS_CONFIG_OFFSET);

	mobi_reset_enable(RESET_ID_NAND);

	nand_hmux_device.id   = config->hmux.cs;
	nand_hmux_data.chipen = config->hmux.ce;

	return mobi_hmux_register(&nand_hmux_device);
}

/* non-arch specific code */
static struct amba_device nandreg_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = NANDREG_AMBA_NAME,
		.platform_data = &nandreg_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = FLASH_REG_BASE,
		.end    = FLASH_REG_END,
		.flags  = IORESOURCE_MEM,
	},
	.dma_mask	= ~0,
	.periphid 	= NANDREG_AMBA_PID,
};

static struct amba_device nanddata_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = NANDDATA_AMBA_NAME,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = FLASH_DATA_BASE,
		.end    = FLASH_DATA_END,
		.flags  = IORESOURCE_MEM,
	},
	.dma_mask	= ~0,
	.periphid 	= NANDDATA_AMBA_PID,
};

static int nand_platform_driver_probe(struct platform_device *pdev)
{
	int ret;
	struct nand_board_config_t *config =
		(struct nand_board_config_t *)pdev->dev.platform_data;

	memcpy(&nandreg_amba_pdata, &config->device_data, 
			sizeof(nandreg_amba_pdata));

	ret = nand_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&nandreg_amba_device, &iomem_resource);
	amba_device_register(&nanddata_amba_device, &iomem_resource);

	return 0;
}

static int nand_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&nanddata_amba_device);
	amba_device_unregister(&nandreg_amba_device);

	return 0;
}

static struct platform_driver nand_platform_driver = {
	.driver	= {
		.name	= PLATFORM_NAME_NAND,
		.owner	= THIS_MODULE,
	},
	.probe		= nand_platform_driver_probe,
	.remove		= nand_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
};

static struct nand_board_config_t nand_config_data;
static struct platform_device nand_platform_device = {
	.name	= PLATFORM_NAME_NAND,
	.id     = -1,
	.dev    = {
		.platform_data  = &nand_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int nand_register_platform_driver(void)
{
	return platform_driver_register(&nand_platform_driver);
}

static int add_board_device_nand(struct nand_board_config_t *data)
{
	nand_config_data = *data;
	return platform_device_register(&nand_platform_device);
}

#else
static int nand_register_platform_driver(void) { return 0; }
static int add_board_device_nand(
		struct nand_board_config_t *data) { return 0; }
#endif

/* ----------------------------------------------------------------
 *  DesignerWare Watchdog
 * ----------------------------------------------------------------*/
#if defined(CONFIG_DW_WATCHDOG) || \
	defined(CONFIG_DW_WATCHDOG_MODULE)

#define WDT_AMBA_NAME	DW_WDT_AMBA_NAME
#define WDT_AMBA_PID	DW_WDT_AMBA_DEVID
static struct dw_wdt_driver_data_t wdt_amba_pdata;

static int wdt_arch_init(struct wdt_board_config_t *config)
{
	mobi_reset_enable(RESET_ID_WDT);
	return 0;
}

static struct amba_device wdt_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = WDT_AMBA_NAME,
		.platform_data = &wdt_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = WDT_BASE,
		.end    = WDT_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_WDT, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= WDT_AMBA_PID,
};

static int wdt_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct wdt_board_config_t *config =
		(struct wdt_board_config_t *)pdev->dev.platform_data;

	memcpy(&wdt_amba_pdata, &config->wdt_data, sizeof(wdt_amba_pdata));

	ret = wdt_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&wdt_amba_device, &iomem_resource);

	return 0;
}

static int wdt_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&wdt_amba_device);
	return 0;
}

static struct platform_driver wdt_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_WDT,
		.owner  = THIS_MODULE,
	},
	.probe          = wdt_platform_driver_probe,
	.remove         = wdt_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct wdt_board_config_t wdt_config_data;
static struct platform_device wdt_platform_device = {
	.name = PLATFORM_NAME_WDT,
	.id   = -1,
	.dev  = {
		.platform_data  = &wdt_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int wdt_register_platform_driver(void)
{
	return platform_driver_register(&wdt_platform_driver);
}

static int add_board_device_watchdog(struct wdt_board_config_t *data)
{
	wdt_config_data = *data;
	return platform_device_register(&wdt_platform_device);
}

#else
static int wdt_register_platform_driver(void) { return 0; }
static int add_board_device_watchdog(
		struct wdt_board_config_t *data) { return 0; }
#endif

/* ----------------------------------------------------------------
 *  ATA
 * ----------------------------------------------------------------*/
#if defined(CONFIG_PATA_AUMB4000) || \
	defined(CONFIG_PATA_AUMB4000_MODULE)
static struct aumb4000_device_data_t pata_amba_pdata;

static struct mobi_hmux_drvdata pata_hmux_data_ce0;
static struct platform_device pata_hmux_device_ce0 = {
	.name = "mobi_hmux",
	.id   = 0x0,
	.dev  = {
		.platform_data = &pata_hmux_data_ce0,
		.release       = empty_release,
	},
};
static struct mobi_hmux_drvdata pata_hmux_data_ce1;
static struct platform_device pata_hmux_device_ce1 = {
	.name = "mobi_hmux",
	.id   = 0x0,
	.dev  = {
		.platform_data = &pata_hmux_data_ce1,
		.release       = empty_release,
	},
};

static int pata_arch_init(struct pata_board_config_t *config)
{
	int ret = 0;
	uint32_t qcc_sysconfig_shadow =
		mobi_qcc_readreg(MOBI_QCCSISC_SYS_CONFIG_OFFSET);

	if (mobi_reset_state(RESET_ID_GPIO) != 0) {
		printk(KERN_ERR "GPIO must be enabled before IDE\n");
		return -1;
	}

	pata_hmux_data_ce0.chipen = config->hmux0.ce;
	pata_hmux_device_ce0.id   = config->hmux0.cs;
	pata_hmux_data_ce1.chipen = config->hmux1.ce;
	pata_hmux_device_ce1.id   = config->hmux1.cs;
	if ((ret = mobi_hmux_register(&pata_hmux_device_ce0)) < 0) {
		return ret;
	}
	if ((ret = mobi_hmux_register(&pata_hmux_device_ce1)) < 0) {
		mobi_hmux_unregister(&pata_hmux_device_ce0) ;
		return ret;
	}

	MOBI_QCCSISC_SYS_CONFIG_CF_RAM_EN_CLR(qcc_sysconfig_shadow);
	mobi_qcc_writereg(qcc_sysconfig_shadow, MOBI_QCCSISC_SYS_CONFIG_OFFSET);
	mobi_reset_enable(RESET_ID_CF);

	return ret;
}

static struct amba_device pata_amba_device = {
	.dev = {
		.coherent_dma_mask = 0xffffffff,
		.init_name = "idereg",
		.platform_data = &pata_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start	= CF_REG_BASE,
		.end	= CF_REG_END,
		.flags	= IORESOURCE_MEM,
	},
	.dma_mask = 0xffffffff,
	.periphid = 0x00041114,
};

static int pata_platform_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct pata_board_config_t *config =
		(struct pata_board_config_t *)pdev->dev.platform_data;

	/* copy over platform data */
	memcpy(&pata_amba_pdata, &config->device_data, sizeof(pata_data));

	ret = pata_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&pata_amba_device, &iomem_resource);
	return ret;
}

static int pata_platform_remove(struct platform_device *pdev)
{
	amba_device_unregister(&pata_platform_device);
	return 0;
}

static struct platform_driver pata_platform_driver = {
	.driver  = {
		.name  = PLATFORM_NAME_IDE,
		.owner = THIS_MODULE,
	},
	.probe   = pata_platform_probe,
	.remove  = pata_platform_remove,
	.suspend = empty_suspend,
	.resume  = empty_resume,
};

static struct pata_board_config_t pata_config_data;
static struct platform_device pata_platform_device = {
	.name = PLATFORM_NAME_IDE,
	.id		= -1,
	.dev	= {
		.platform_data = &pata_config_data,
		.release = empty_release,
	},
	.num_resources  = 0
};

static int pata_register_platform_driver(void)
{
	return platform_driver_register(&pata_platform_driver);
}

static int add_board_device_pata(struct pata_board_config_t *data)
{
	pata_config_data = *data;
	return platform_device_register(&pata_platform_device);
}

#else
static int pata_register_platform_driver(void) { return 0; }
static int add_board_device_pata(
		struct pata_board_config_t *data) { return 0; }
#endif

#if defined(CONFIG_DW_STMMAC) || defined(CONFIG_DW_STMMAC_MODULE)

#define GMAC_AMBA_NAME	DW_GMAC_AMBA_NAME
#define GMAC_AMBA_PID	DW_GMAC_AMBA_DEVID
static struct dw_gmac_driver_data_t gmac_amba_pdata;

static int gmac_arch_init(struct gmac_board_config_t *config)
{
	mobi_reset_enable(RESET_ID_GMAC);
	return 0;
}

static struct amba_device gmac_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = GMAC_AMBA_NAME,
		.platform_data = &gmac_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = GMAC_BASE,
		.end    = GMAC_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_GMAC, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= GMAC_AMBA_PID,
};

static int gmac_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct gmac_board_config_t *config =
		(struct gmac_board_config_t *)pdev->dev.platform_data;

	memcpy(&gmac_amba_pdata, &config->gmac_data,
			sizeof(gmac_amba_pdata));

	ret = gmac_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&gmac_amba_device, &iomem_resource);

	return ret;
}

static int gmac_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&gmac_amba_device);
	return 0;
}

static struct platform_driver gmac_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_GMAC,
		.owner  = THIS_MODULE,
	},
	.probe          = gmac_platform_driver_probe,
	.remove         = gmac_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct gmac_board_config_t gmac_config_data;
static struct platform_device gmac_platform_device = {
	.name   = PLATFORM_NAME_GMAC,
	.id     = -1,
	.dev    = {
		.platform_data  = &gmac_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int gmac_register_platform_driver(void)
{
	return platform_driver_register(&gmac_platform_driver);
}

static int add_board_device_gmac(struct gmac_board_config_t *data)
{
	gmac_config_data = *data;
	return platform_device_register(&gmac_platform_device);
}

#else
static int gmac_register_platform_driver(void) { return 0; }
static int add_board_device_gmac(
		struct gmac_board_config_t *data) { return 0; }
#endif /* GMAC */

#if defined(CONFIG_DWCMSH) || defined(CONFIG_DWCMSH_MODULE)

#define SDMMC_AMBA_NAME	DW_SDMMC_AMBA_NAME
#define SDMMC_AMBA_PID	DW_SDMMC_AMBA_DEVID
static struct dwc_sdmmc_device_data_t sdmmc_amba_pdata;
static int sdmmc_arch_init(struct sdmmc_board_config_t *config)
{
	/* turn it off until actual device driver is loaded */
	mobi_clock_set_rate(CLOCK_ID_SDMMC, 0);
	return mobi_reset_enable(RESET_ID_SDMMC);
}

static struct amba_device sdmmc_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = SDMMC_AMBA_NAME,
		.platform_data = &sdmmc_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = SDMMC_BASE,
		.end    = SDMMC_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_SDMMC, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= SDMMC_AMBA_PID,
};

static int sdmmc_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct sdmmc_board_config_t *config =
		(struct sdmmc_board_config_t *)pdev->dev.platform_data;

	memcpy(&sdmmc_amba_pdata, &config->device_data, 
			sizeof(sdmmc_amba_pdata));

	ret = sdmmc_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&sdmmc_amba_device, &iomem_resource);

	return ret;
}

static int sdmmc_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&sdmmc_amba_device);
	return 0;
}

static struct platform_driver sdmmc_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_SDMMC,
		.owner  = THIS_MODULE,
	},
	.probe          = sdmmc_platform_driver_probe,
	.remove         = sdmmc_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct sdmmc_board_config_t sdmmc_config_data;
static struct platform_device sdmmc_platform_device = {
	.name   = PLATFORM_NAME_SDMMC,
	.id     = -1,
	.dev    = {
		.platform_data  = &sdmmc_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int sdmmc_register_platform_driver(void)
{
	return platform_driver_register(&sdmmc_platform_driver);
}

static int add_board_device_sdmmc(struct sdmmc_board_config_t *data)
{
	sdmmc_config_data = *data;
	return platform_device_register(&sdmmc_platform_device);
}

#else
static int add_board_device_sdmmc(
		struct sdmmc_board_config_t *data) { return 0; }
static int sdmmc_register_platform_driver(void) { return 0; }
#endif /* DWCMSH - sdmmc */

#if defined(CONFIG_USB_DW_MERLIN) || \
	defined(CONFIG_USB_DW_MERLIN_MODULE)

#define USB_AMBA_NAME	DW_OTG_AMBA_NAME
#define USB_AMBA_PID	DW_OTG_AMBA_DEVID
static struct dw_otg_driver_data_t usb_amba_pdata;

static int usb_arch_init(struct usb_board_config_t *config)
{
	/* disable Low leakage RAMs for USB */
	mobi_qcc_writereg(0x0, MOBI_QCCSISC_USB_RAM_CONFIG_OFFSET);
	return mobi_reset_enable(RESET_ID_OTGPHY);
}

/* must init, some usb code does look at this */
uint64_t usb_dev_dma_mask = ~0;
static struct amba_device usb_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = USB_AMBA_NAME,
		.platform_data = &usb_amba_pdata,
		.release = static_amba_device_release,
		.dma_mask = &usb_dev_dma_mask,
	},
	.res = {
		.start  = USB_BASE,
		.end    = USB_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_USB, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= USB_AMBA_PID,
};

static int usb_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct usb_board_config_t *config =
		(struct usb_board_config_t *)pdev->dev.platform_data;

	memcpy(&usb_amba_pdata, &config->otg_data,
			sizeof(usb_amba_pdata));

	ret = usb_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&usb_amba_device, &iomem_resource);

		return ret;
}

static int usb_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&usb_amba_device);
	return 0;
}

static struct platform_driver usb_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_OTG,
		.owner  = THIS_MODULE,
	},
	.probe          = usb_platform_driver_probe,
	.remove         = usb_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct usb_board_config_t usb_config_data;
static struct platform_device usb_platform_device = {
	.name   = PLATFORM_NAME_OTG,
	.id     = -1,
	.dev    = {
		.platform_data  = &usb_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int usb_register_platform_driver(void)
{
	return platform_driver_register(&usb_platform_driver);
}

static int add_board_device_usb(struct usb_board_config_t *data)
{
	usb_config_data = *data;
	return platform_device_register(&usb_platform_device);
}

#else
static int usb_register_platform_driver(void) { return 0; }
static int add_board_device_usb(
		struct usb_board_config_t *data) { return 0; }
#endif /* USB_DW_CORE */

#if defined(CONFIG_MG3500_MHIF) || defined(CONFIG_MG3500_MHIF_MODULE)
/*
 * so it turns out the hmux and the mhif have an undocumented connection!
 * a device that is connected via the mhif must use the hmux for a given
 * chip select.  the mhif can support six devices.  sooo... the hmux
 * chip select and the mhif device chip select much match. so just to
 * keep things simple, use an array of mhif-hmux devices and init
 * the array based on the chip select defined in the mhif_devices array
 * in the board driver, not that while the mhif has multiple chip selects
 * there is only one chip-enable in the hmux.  so all base mhif devices
 * get the hmux CE of MHIF or MHIF16(which sets another bit).  however,
 * we still have to define ce12/13 for devices that exceed the mhif
 * address space
 */
static struct mobi_hmux_drvdata mhif_hmux_data[] = {
	{ .chipen	= HMUX_UNUSED_CE, },
	{ .chipen	= HMUX_UNUSED_CE, },
	{ .chipen	= HMUX_UNUSED_CE, },
	{ .chipen	= HMUX_UNUSED_CE, },
	{ .chipen	= HMUX_UNUSED_CE, },
	{ .chipen	= HMUX_UNUSED_CE, },
};

static struct platform_device mhif_hmux_devices[] = {
	{
		.name = "mobi_hmux",
		.id   = 0,
		.dev  = {
			.platform_data = &mhif_hmux_data[0],
			.release       = empty_release,
		},
	},
	{
		.name = "mobi_hmux",
		.id   = 1,
		.dev  = {
			.platform_data = &mhif_hmux_data[1],
			.release       = empty_release,
		},
	},
	{
		.name = "mobi_hmux",
		.id   = 2,
		.dev  = {
			.platform_data = &mhif_hmux_data[2],
			.release       = empty_release,
		},
	},
	{
		.name = "mobi_hmux",
		.id   = 3,
		.dev  = {
			.platform_data = &mhif_hmux_data[3],
			.release       = empty_release,
		},
	},
	{
		.name = "mobi_hmux",
		.id   = 4,
		.dev  = {
			.platform_data = &mhif_hmux_data[4],
			.release       = empty_release,
		},
	},
	{
		.name = "mobi_hmux",
		.id   = 5,
		.dev  = {
			.platform_data = &mhif_hmux_data[5],
			.release       = empty_release,
		}
	}
};

static struct mobi_hmux_drvdata ce12_hmux_data;
static struct platform_device ce12_hmux_device = {
	.name = "mobi_hmux",
	.id   = 0x0,
	.dev  = {
		.platform_data = &ce12_hmux_data,
		.release       = empty_release,
	},
};
static struct mobi_hmux_drvdata ce13_hmux_data;
static struct platform_device ce13_hmux_device = {
	.name = "mobi_hmux",
	.id   = 0x0,
	.dev  = {
		.platform_data = &ce13_hmux_data,
		.release       = empty_release,
	},
};

/* note, the mhif is a bit odd, it can handle 6 devices but has
 * limited address space.  the hmux can be used to increase that
 * address space but ignore that for now.  so the way to access
 * the control registers in the mhif is to set bit 24.  other devices,
 * like the mtd mapping for nor, ioremaps the space below this to
 * get access to the devices.  for this reason, the resource below
 * is modified just to pass in the control register space.
 * data/device space will be passed in via the device structure and
 * then mapped via external drivers
 */
static struct mhif_driver_data_t mhif_amba_pdata;
static struct amba_device mhif_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = MHIF_AMBA_NAME,
		.platform_data = &mhif_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		/* set bit 24 to get to the ctrl address space */
		.start  = MHIF_CTRL_REG_BASE,
		.end    = MHIF_CTRL_REG_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_MHIF, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= MHIF_AMBA_DEVID,
};

static int mhif_platform_probe(struct platform_device *pdev)
{
	int ret = 0, i, cs;
	struct mhif_board_config_t *config =
		(struct mhif_board_config_t *)pdev->dev.platform_data;
	struct mhif_device_t *mhif_devices;
	uint8_t cs_used = 0;

	memcpy(&mhif_amba_pdata, &config->mhif_data, sizeof(mhif_amba_pdata));

	mobi_reset_enable(RESET_ID_MHIF);
	if (config->nor_attached == 1) {

		void __iomem *flash_reg_base =
			ioremap(FLASH_REG_BASE, FLASH_REG_SIZE);
		uint32_t regval;

		mobi_reset_disable(RESET_ID_NAND);

		regval = readl(flash_reg_base +
				(uint32_t)MOBI_FLASH_NAND_CONTROL_OFFSET);
		MOBI_FLASH_NAND_CONTROL_WP_CLR(regval);
		writel(regval, flash_reg_base +
				(uint32_t)MOBI_FLASH_NAND_CONTROL_OFFSET);

		iounmap(flash_reg_base);
	}

	mhif_devices = mhif_amba_pdata.devices;
	for (i = 0; i < mhif_amba_pdata.num_devices; i++, mhif_devices++) {

		if (cs_used & (1 << mhif_devices->chip_select)) {
			printk(KERN_ERR "MHIF devices cannot share chip selects\n");
			return -1;
		}
		cs = mhif_devices->chip_select;
		mhif_hmux_devices[cs].id = mhif_devices->chip_select;

		if (mhif_devices->width == MHIF_DEVICE_DATA_WIDTH_8)
			mhif_hmux_data[cs].chipen = HMUX_MHIF_CE0;
		else
			mhif_hmux_data[cs].chipen = HMUX_MHIF_16BIT_CE0;

		if ((ret = mobi_hmux_register(&mhif_hmux_devices[cs])) < 0) {
			printk(KERN_ERR "failed to register hmux\n");
			return ret;
		}
		cs_used |= (1 << cs);
	}

	if (config->ce12_hmux.ce != HMUX_UNUSED_CE) {
		ce12_hmux_data.chipen = config->ce12_hmux.ce;
		ce12_hmux_device.id   = config->ce12_hmux.cs;
		if (cs_used & (1 << ce12_hmux_device.id)) {
			printk(KERN_ERR "MHIF devices cannot share chip selects\n");
			goto mhif_hmux_err;
		}

		if ((ret = mobi_hmux_register(&ce12_hmux_device)) < 0) {
			goto mhif_hmux_err;
		}
		cs_used |= (1 << ce12_hmux_device.id);
	}

	if (config->ce13_hmux.ce != HMUX_UNUSED_CE) {
		ce13_hmux_data.chipen = config->ce13_hmux.ce;
		ce13_hmux_device.id   = config->ce13_hmux.cs;
		if (cs_used & (1 << ce13_hmux_device.id)) {
			printk(KERN_ERR "MHIF devices cannot share chip selects\n");
			goto mhif_hmux_err;
		}

		if ((ret = mobi_hmux_register(&ce13_hmux_device)) < 0) {
			goto mhif_addr22_err;
		}
		cs_used |= (1 << ce13_hmux_device.id);
	}

	amba_device_register(&mhif_amba_device, &iomem_resource);

	return ret;

mhif_addr22_err:
	if (config->ce12_hmux.ce != HMUX_UNUSED_CE)
		mobi_hmux_unregister(&ce12_hmux_device);

mhif_hmux_err:
	mhif_devices = mhif_amba_pdata.devices;
	for (i = 0; i < mhif_amba_pdata.num_devices; i++, mhif_devices++) {
		cs = mhif_devices->chip_select;
		mobi_hmux_unregister(&mhif_hmux_devices[cs]);
	}

	return -1;
}

static int mhif_platform_remove(struct platform_device *pdev)
{
	amba_device_unregister(&mhif_amba_device);
	return 0;
}

static struct platform_driver mhif_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_MHIF,
		.owner  = THIS_MODULE,
	},
	.probe          = mhif_platform_probe,
	.remove         = mhif_platform_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct mhif_board_config_t mhif_config_data;
static struct platform_device mhif_platform_device = {
	.name   = PLATFORM_NAME_MHIF,
	.id     = -1,
	.dev    = {
		.platform_data  = &mhif_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int mhif_register_platform_driver(void)
{
	return platform_driver_register(&mhif_platform_driver);
}

static int add_board_device_mhif(struct mhif_board_config_t *data)
{
	memcpy(&mhif_config_data, data, sizeof(struct mhif_board_config_t));
	return platform_device_register(&mhif_platform_device);
}

#else
static int mhif_register_platform_driver(void) { return 0; }
static int add_board_device_mhif(
		struct mhif_board_config_t *data) { return 0; }
#endif /* MHIF */

/* platform devices */
extern struct mg3500_ioctrl_data_t arch_ioctrl_data;
static struct platform_device ioctrl_platform_device = {
	.name = PLATFORM_NAME_IOCTRL,
	.id   = -1,
	.num_resources = 0,
	.dev  = {
		.platform_data = &arch_ioctrl_data,
		.release       = empty_release,
	},
};

static int add_device_ioctrl(void)
{
	return platform_device_register(&ioctrl_platform_device);
}

static struct platform_device codec_reset_platform_device = {
	.name   = PLATFORM_NAME_CODEC_RESET,
	.id     = -1,
	.dev    = {
		.platform_data	= NULL,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int add_device_codec_reset(void)
{
	return platform_device_register(&codec_reset_platform_device);
}

static struct platform_device qcc_platform_device = {
	.name   = PLATFORM_NAME_QCC,
	.id     = -1,
	.dev    = {
		.platform_data	= NULL,
		.release = empty_release,
	},
	.num_resources = 0,
};

int add_device_qcc(void)
{
	return platform_device_register(&qcc_platform_device);
}

/*
 * unlike most devices in this file, these are not hardware control
 * blocks. we want to have on every system so we add them here instead
 * of leaving to the board drivers
 */
static int arch_add_platform_devices(void)
{
	int ret = 0;

	if ((ret = add_device_ioctrl()) < 0) {
		printk(KERN_ERR "%s: error adding device ioctrl\n", __func__);
		return ret;

	} else if ((ret = add_device_codec_reset()) < 0) {
		printk(KERN_ERR "%s: error adding device codec_reset\n", __func__);
		return ret;

	} else if ((ret = add_device_qcc()) < 0) {
		printk(KERN_ERR "%s: error adding device qcc\n", __func__);
		return ret;
	}

	return ret;
}

struct amba_device_container {
	struct amba_device dev;
	char registered;
};

#if defined(CONFIG_I2C_DWAPBI2C) || defined(CONFIG_I2C_DWAPBI2C_MODULE)
#define _MHZ 1000000
static struct amba_device_container i2c0_amba_device = {
	.dev = {
		.res		= {
			.start	= I2C0_BASE,
			.end	= I2C0_BASE + I2C0_SIZE - 1,
			.flags	= IORESOURCE_MEM,
		},
		.dev		= {
			.init_name = "I2C_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq		= {
			MOBI_IRQ_I2C0,
			NO_IRQ,
		},
		.dma_mask	= ~0,
		.periphid 	= DWAPBI2C_ID | 0,
	},
};

static struct amba_device_container i2c1_amba_device = {
	.dev = {
		.res		= {
			.start	= I2C1_BASE,
			.end	= I2C1_BASE + I2C1_SIZE - 1,
			.flags	= IORESOURCE_MEM,
		},
		.dev		= {
			.init_name = "I2C_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq		= {
			MOBI_IRQ_I2C1,
			NO_IRQ,
		},
		.dma_mask	= ~0,
		.periphid 	= DWAPBI2C_ID | 1,
	},
};
/* This is for reenabling reset when all devices has been removed */
static int i2c_refcount = 0;
static int i2c_platform_driver_probe(struct platform_device *pdev)
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbi2c_bus_data *bdata = adev->dev.platform_data;
	int clock_id;
	unsigned long clock_rate = 3 * _MHZ;
	int ret;

	if (mobi_reset_state(RESET_ID_I2C) != 0)
		mobi_reset_disable(RESET_ID_I2C);

	if (adev->res.start == I2C0_BASE) {
		clock_id 	= CLOCK_ID_I2C0;
		bdata->clock_id = "i2c0";
	} else {
		clock_id 	= CLOCK_ID_I2C1;
		bdata->clock_id = "i2c1";
	}

	mobi_clock_set_rate(clock_id, clock_rate);
	if (unlikely(!mobi_clock_get_rate(clock_id))) {
		printk(KERN_ERR "%s: %s: could not set clock id=%d (%s) to %luHz\n",
				adev->dev.init_name, __func__, 
				clock_id, bdata->clock_id, clock_rate);
		return -EINVAL;
	}

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	i2c_refcount++;

	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int i2c_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) { 
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	mobi_clock_set_rate(adev->res.start == I2C0_BASE ?
			CLOCK_ID_I2C0 : CLOCK_ID_I2C1, 0);
	if (--i2c_refcount == 0)
		mobi_reset_enable(RESET_ID_I2C);

	return 0;
}

static struct platform_driver i2c0_pdrv = {
	.probe		= i2c_platform_driver_probe,
	.remove		= i2c_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
	.driver	= {
		.name	= "m_i2c0",
		.owner	= THIS_MODULE,
	},
};

static struct platform_device i2c0_pdev = {
	.name           = "m_i2c0",
	.id             = -1,
	.dev            = {
		.platform_data = &i2c0_amba_device,
		.release = empty_release,
	},
	.num_resources	= 0,
};

static struct platform_driver i2c1_pdrv = {
	.probe		= i2c_platform_driver_probe,
	.remove		= i2c_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
	.driver	= {
		.name	= "m_i2c1",
		.owner	= THIS_MODULE,
	},
};

static struct platform_device i2c1_pdev = {
	.name           = "m_i2c1",
	.id             = -1,
	.dev            = {
		.platform_data = &i2c1_amba_device,
		.release = empty_release,
	},
	.num_resources	= 0,
};

static int i2c_register_platform_driver(void) 
{ 
	int ret = platform_driver_register(&i2c0_pdrv); 
	if (ret) 
		return ret;

	return platform_driver_register(&i2c1_pdrv); 
}

static int add_board_device_i2c(
		struct dwapbi2c_bus_data *bdata, int devid) 
{
	int ret = 0;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						i2c0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						i2c1_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&i2c0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&i2c1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static inline int i2c_register_platform_driver(void) { return 0; };
static int add_board_device_i2c(
		struct dwapbi2c_bus_data *bdata, int devid) { return 0; };
#endif

#if defined(CONFIG_SPI_DWAPBSSI) || defined(CONFIG_SPI_DWAPBSSI_MODULE)

static struct amba_device_container spi0_amba_device = {
	.dev = {
		.res = {
			.start = SSI0_BASE,
			.end = SSI0_BASE + SSI0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 0,
	},
};

static struct amba_device_container spi1_amba_device = {
	.dev = {
		.res = {
			.start = SSI1_BASE,
			.end = SSI1_BASE + SSI1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 1,
	},
};

static struct amba_device_container spi2_amba_device = {
	.dev = {
		.res = {
			.start = SSI2_BASE,
			.end = SSI2_BASE + SSI2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 2,
	},
};

static int spi_refcount = 0;
static int spi_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbssi_bus_data *bdata = adev->dev.platform_data;
	int ret;
	bdata->clock_id = "ahb";

	if (mobi_reset_state(RESET_ID_SSI) != 0)
		mobi_reset_disable(RESET_ID_SSI);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	spi_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)
		 pdev->dev.platform_data)->registered = 1;
	return ret;
}

static int spi_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--spi_refcount == 0)
		mobi_reset_enable(RESET_ID_SSI);

	return 0;
}

static struct platform_driver spi0_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi0_pdev = {
	.name = "m_spi0",
	.id = -1,
	.dev = {
		.platform_data = &spi0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver spi1_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi1_pdev = {
	.name = "m_spi1",
	.id = -1,
	.dev = {
		.platform_data = &spi1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver spi2_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi2_pdev = {
	.name = "m_spi2",
	.id = -1,
	.dev = {
		.platform_data = &spi2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int add_board_device_spi(
		struct dwapbssi_bus_data *bdata, int devid) 
{
	int ret = 0;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						spi0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						spi1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						spi2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&spi0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&spi1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&spi2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int spi_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&spi0_pdrv);
	if (ret) 
		return ret;

	ret = platform_driver_register(&spi1_pdrv);
	if (ret) 
		return ret;

	return platform_driver_register(&spi2_pdrv);
}

#else
static int add_board_device_spi(
		struct dwapbssi_bus_data *bdata, int devid) { return 0; }; 
static int spi_register_platform_driver(void) { return 0; }; 
#endif

#if defined(CONFIG_CADENCE_PWM) || defined(CONFIG_CADENCE_PWM_MODULE)
static struct amba_device_container pwm0_amba_device = {
	.dev = {
		.res = {
			.start = PWM0_BASE,
			.end = PWM0_BASE + PWM0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 0,
	},
};

static struct amba_device_container pwm1_amba_device = {
	.dev = {
		.res = {
			.start = PWM1_BASE,
			.end = PWM1_BASE + PWM1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 1,
	},
};

static struct amba_device_container pwm2_amba_device = {
	.dev = {
		.res = {
			.start = PWM2_BASE,
			.end = PWM2_BASE + PWM2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 2,
	},
};

static int pwm_refcount = 0;
static int pwm_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct cadpwm_block_data *bdata = adev->dev.platform_data;
	int ret;

	bdata->clock_id = "apb";

	if (mobi_reset_state(RESET_ID_PWM) != 0)
		mobi_reset_disable(RESET_ID_PWM);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	pwm_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)
		 pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int pwm_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--pwm_refcount == 0)
		mobi_reset_enable(RESET_ID_PWM);

	return 0;
}

static struct platform_driver pwm0_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm0_pdev = {
	.name = "m_pwm0",
	.id = -1,
	.dev = {
		.platform_data = &pwm0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver pwm1_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm1_pdev = {
	.name = "m_pwm1",
	.id = -1,
	.dev = {
		.platform_data = &pwm1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};


static struct platform_driver pwm2_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm2_pdev = {
	.name = "m_pwm2",
	.id = -1,
	.dev = {
		.platform_data = &pwm2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int pwm_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&pwm0_pdrv);
	if (ret)
		return ret;

	ret = platform_driver_register(&pwm1_pdrv);
	if (ret)
		return ret;

	return platform_driver_register(&pwm2_pdrv);
}

static int add_board_device_pwm(
		struct cadpwm_block_data *bdata, int devid) 
{
	int ret;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						pwm0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						pwm1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						pwm2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;
	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;
	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&pwm0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&pwm1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&pwm2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static int pwm_register_platform_driver(void) { return 0; }; 
static int add_board_device_pwm(
		struct cadpwm_block_data *bdata, int devid) { return 0; }; 
#endif

#if defined(CONFIG_GPIO_DWAPBGPIO) || defined(CONFIG_GPIO_DWAPBGPIO_MODULE)
static struct amba_device_container gpio0_amba_device = {
	.dev = {
		.res = {
			.start = GPIO0_BASE,
			.end = GPIO0_BASE + GPIO0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 0,
	},
};

static struct amba_device_container gpio1_amba_device = {
	.dev = {
		.res = {
			.start = GPIO1_BASE,
			.end = GPIO1_BASE + GPIO1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 1,
	},
};

static struct amba_device_container gpio2_amba_device = {
	.dev = {
		.res = {
			.start = GPIO2_BASE,
			.end = GPIO2_BASE + GPIO2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 2,
	},
};

static int gpio_refcount = 0;
static int gpio_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbgpio_block_data *bdata = adev->dev.platform_data;
	int ret;

	bdata->port_a_count = 0;
	bdata->port_b_count = 0;
	bdata->port_c_count = 0;
	bdata->port_d_count = 0;

	switch (adev->res.start) {
	case GPIO0_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(0);
		bdata->gpio_id_offset = 0;
		break;
	case GPIO1_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(1);
		bdata->gpio_id_offset = mobi_ioctrl_get_gpio_bank_size(0);
		break;
	case GPIO2_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(2);
		bdata->gpio_id_offset = mobi_ioctrl_get_gpio_bank_size(0)
			+ mobi_ioctrl_get_gpio_bank_size(1);
		break;
	default:
		return -EINVAL;
	}

	if (!bdata->port_a_count)
		return -ENODEV;

	if (mobi_reset_state(RESET_ID_GPIO) != 0)
		mobi_reset_disable(RESET_ID_GPIO);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	gpio_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int gpio_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--gpio_refcount == 0)
		mobi_reset_enable(RESET_ID_GPIO);

	return 0;
}

static struct platform_driver gpio0_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio0_pdev = {
	.name = "m_gpio0",
	.id = -1,
	.dev = {
		.platform_data = &gpio0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver gpio1_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio1_pdev = {
	.name = "m_gpio1",
	.id = -1,
	.dev = {
		.platform_data = &gpio1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver gpio2_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio2_pdev = {
	.name = "m_gpio2",
	.id = -1,
	.dev = {
		.platform_data = &gpio2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int gpio_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&gpio0_pdrv);
	if (ret) 
		return ret;
	ret = platform_driver_register(&gpio1_pdrv);
	if (ret) 
		return ret;

	return platform_driver_register(&gpio2_pdrv);
}

static int add_board_device_gpio(
		struct dwapbgpio_block_data *bdata, int devid) 
{
	int ret;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						gpio0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						gpio1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						gpio2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&gpio0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&gpio1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&gpio2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static int gpio_register_platform_driver(void) { return 0; };
static int add_board_device_gpio(
		struct dwapbgpio_block_data *bdata, int devid) { return 0; }
#endif

int arch_add_board_devices(
		struct board_device_desc *board_devices, int num_devices)
{
	int i;
	int ret = 0;

	for (i = 0; i < num_devices; i++)
	{
		switch (board_devices->device) {
		case BOARD_DEVICE_NAND:
			ret = add_board_device_nand(
					(struct nand_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_WATCHDOG:
			ret = add_board_device_watchdog(
					(struct wdt_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_SDMMC:
			ret = add_board_device_sdmmc(
					(struct sdmmc_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_GMAC:
			ret = add_board_device_gmac(
					(struct gmac_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_USB:
			ret = add_board_device_usb(
					(struct usb_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_PATA:
			ret = add_board_device_pata(
					(struct pata_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_MHIF:
			ret = add_board_device_mhif(
					(struct mhif_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_SPI:
			ret = add_board_device_spi(
					(struct dwapbssi_bus_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_I2C:
			ret = add_board_device_i2c(
					(struct dwapbi2c_bus_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_PWM:
			ret = add_board_device_pwm(
					(struct cadpwm_block_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_GPIO:
			ret = add_board_device_gpio(
					(struct dwapbgpio_block_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		default:
			ret = -1;
		}
		if (ret) {
			printk(KERN_ERR "ERROR: Failed to add board device %s, devid %d\n",
					board_device_types_names[board_devices->device],
					board_devices->devid);
			return ret;
		}

		board_devices++;
	}
	return ret;
}
EXPORT_SYMBOL(arch_add_board_devices);

/* for possible future reference */
/*
void set_ahb_dma_master_priority(void)
{
	void __iomem *ahb_dma_base = ioremap(AHB_ARB_DMA_BASE, AHB_ARB_DMA_SIZE);
	uint32_t regval;
	int i;

	for (i=0;i<4;i++) {
		printk("Master %d has priority 0x%x\n", i+1, readl(ahb_dma_base + (0x4*i)));
	}
	regval = 0xf;
	writel(regval, ahb_dma_base+0x4);
}
*/

int __init mg3500_register_platform_drivers(void)
{

	if (wdt_register_platform_driver())
		return -1;
	if (nand_register_platform_driver())
		return -1;
	if (sdmmc_register_platform_driver())
		return -1;
	if (gmac_register_platform_driver())
		return -1;
	if (usb_register_platform_driver())
		return -1;
	if (mhif_register_platform_driver())
		return -1;
	if (i2c_register_platform_driver())
		return -1;
	if (spi_register_platform_driver())
		return -1;
	if (pwm_register_platform_driver())
		return -1;
	if (gpio_register_platform_driver())
		return -1;
	if (pata_register_platform_driver())
		return -1;

	if (arch_add_platform_devices());
		return -1;

	return 0;
}
