/*HEADER_START*/
/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*HEADER_STOP*/

#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/amba/bus.h>
#include <linux/io.h>
#include <asm/mach/irq.h>
#include <linux/mtd/physmap.h>

/* APIs */
#include <mach/mobi_reset.h>
#include <mach/mobi_clock.h>
#include <mach/mobi_qcc.h>
#include <mach/mobi_ioctrl.h>

#include <mach/platform.h>
#include <mach/mx64180_devices.h>
#include <mach/au_flash_regs.h>

/* notes:
 *
 *  chip driver will have an init function called at bootup
 *  it will:
 *  	1) register a platform driver, the probe of this platform
 *  	driver will be responsible for initializing any merlin/mobilygen
 *  	specic stuff
 *  	2) it will also register and amba_device_register
 *  	3) it will contain the add device function just as it does not
 *  	and do the same thing, assign the data, and call device_register
 *  	-- note this platform data will be split in the probe and the
 *  	actual amba platform data will have to be assigned there.
 *
 *  board driver will
 *  	1) initialize data structures for IP and merlin/board specific
 *  	2) call add device
 *
 *  device driver
 *   1) calls amba_driver register
 */

/*
 * We define our own amba device release function so that
 * our amba_device static structs will not be kfree'd().
 */
static void static_amba_device_release(struct device *dev)
{
	struct amba_device *adev = container_of(dev, struct amba_device, dev);
	if (adev->res.parent)
		release_resource(&adev->res);
}

static void empty_release(struct device *dev) {}
static int empty_suspend(struct platform_device *dev,
		pm_message_t state)
{
	return 0;
}
static int empty_resume(struct platform_device *dev)
{
	return 0;
}

/* ----------------------------------------------------------------
 *  Aurora NAND controller
 * ----------------------------------------------------------------*/
#if defined(CONFIG_MTD_NAND_AUMB3000) || \
	defined(CONFIG_MTD_NAND_AUMB3000_MODULE)

#define NANDREG_AMBA_NAME 	AUMB3000_NANDREG_AMBA_NAME
#define NANDREG_AMBA_PID 	AUMB3000_NANDREG_AMBA_DEVID
#define NANDDATA_AMBA_NAME 	AUMB3000_NANDDATA_AMBA_NAME
#define NANDDATA_AMBA_PID 	AUMB3000_NANDDATA_AMBA_DEVID
static struct aumb3000_nand_device_data_t nandreg_amba_pdata;

static int nand_arch_init(struct nand_board_config_t *config)
{
	return mobi_reset_enable(RESET_ID_NAND);
}


/* non-arch specific code */
static struct amba_device nandreg_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = NANDREG_AMBA_NAME,
		.platform_data = &nandreg_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = FLASH_REG_BASE,
		.end    = FLASH_REG_END,
		.flags  = IORESOURCE_MEM,
	},
	.dma_mask	= ~0,
	.periphid 	= NANDREG_AMBA_PID,
};

static struct amba_device nanddata_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = NANDDATA_AMBA_NAME,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = FLASH_DATA_BASE,
		.end    = FLASH_DATA_END,
		.flags  = IORESOURCE_MEM,
	},
	.dma_mask	= ~0,
	.periphid 	= NANDDATA_AMBA_PID,
};

static int nand_platform_driver_probe(struct platform_device *pdev)
{
	int ret;
	struct nand_board_config_t *config =
		(struct nand_board_config_t *)pdev->dev.platform_data;

	memcpy(&nandreg_amba_pdata, &config->device_data, 
			sizeof(nandreg_amba_pdata));

	ret = nand_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&nandreg_amba_device, &iomem_resource);
	amba_device_register(&nanddata_amba_device, &iomem_resource);

	return 0;
}

static int nand_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&nanddata_amba_device);
	amba_device_unregister(&nandreg_amba_device);

	return 0;
}

static struct platform_driver nand_platform_driver = {
	.driver	= {
		.name	= PLATFORM_NAME_NAND,
		.owner	= THIS_MODULE,
	},
	.probe		= nand_platform_driver_probe,
	.remove		= nand_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
};

static struct nand_board_config_t nand_config_data;
static struct platform_device nand_platform_device = {
	.name	= PLATFORM_NAME_NAND,
	.id     = -1,
	.dev    = {
		.platform_data  = &nand_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int nand_register_platform_driver(void)
{
	return platform_driver_register(&nand_platform_driver);
}

static int add_board_device_nand(struct nand_board_config_t *data)
{
	nand_config_data = *data;
	return platform_device_register(&nand_platform_device);
}

#else
static int nand_register_platform_driver(void) { return 0; }
static int add_board_device_nand(
		struct nand_board_config_t *data) { return 0; }
#endif

/* ----------------------------------------------------------------
 *  DesignerWare Watchdog
 * ----------------------------------------------------------------*/
#if defined(CONFIG_DW_WATCHDOG) || \
	defined(CONFIG_DW_WATCHDOG_MODULE)

#define WDT_AMBA_NAME	DW_WDT_AMBA_NAME
#define WDT_AMBA_PID	DW_WDT_AMBA_DEVID
static struct dw_wdt_driver_data_t wdt_amba_pdata;

static int wdt_arch_init(struct wdt_board_config_t *config)
{
	mobi_reset_enable(RESET_ID_WDT);
	return 0;
}

static struct amba_device wdt_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = WDT_AMBA_NAME,
		.platform_data = &wdt_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = WDT_BASE,
		.end    = WDT_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_WDT, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= WDT_AMBA_PID,
};

static int wdt_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct wdt_board_config_t *config =
		(struct wdt_board_config_t *)pdev->dev.platform_data;

	memcpy(&wdt_amba_pdata, &config->wdt_data, sizeof(wdt_amba_pdata));

	ret = wdt_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&wdt_amba_device, &iomem_resource);

	return 0;
}

static int wdt_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&wdt_amba_device);
	return 0;
}

static struct platform_driver wdt_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_WDT,
		.owner  = THIS_MODULE,
	},
	.probe          = wdt_platform_driver_probe,
	.remove         = wdt_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct wdt_board_config_t wdt_config_data;
static struct platform_device wdt_platform_device = {
	.name = PLATFORM_NAME_WDT,
	.id   = -1,
	.dev  = {
		.platform_data  = &wdt_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int wdt_register_platform_driver(void)
{
	return platform_driver_register(&wdt_platform_driver);
}

static int add_board_device_watchdog(struct wdt_board_config_t *data)
{
	wdt_config_data = *data;
	return platform_device_register(&wdt_platform_device);
}

#else
static int wdt_register_platform_driver(void) { return 0; }
static int add_board_device_watchdog(
		struct wdt_board_config_t *data) { return 0; }
#endif

/* ----------------------------------------------------------------
 *  SATA
 *  XXX 6/1 this is unverified/uncompiled code, just changed pata
 *  to sata to searve as placeholder
 * ----------------------------------------------------------------*/
#if defined(CONFIG_SATA_DWC) || \
	defined(CONFIG_SATA_DWC_MODULE)
static struct dw_sata_driver_data_t sata_amba_pdata;

static int sata_arch_init(struct pata_board_config_t *config)
{
	return mobi_reset_enable(RESET_ID_SATA);
}

static struct amba_device sata_amba_device = {
	.dev = {
		.coherent_dma_mask = 0xffffffff,
		.init_name = "idereg",
		.platform_data = &sata_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start	= SATA_REG_BASE,
		.end	= SATA_REG_END,
		.flags	= IORESOURCE_MEM,
	},
	.dma_mask = 0xffffffff,
	.periphid = 0x00041114,
};

static int sata_platform_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct sata_board_config_t *config =
		(struct sata_board_config_t *)pdev->dev.platform_data;

	/* copy over platform data */
	memcpy(&sata_amba_pdata, &config->device_data, sizeof(sata_data));

	ret = sata_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&sata_amba_device, &iomem_resource);
	return ret;
}

static int sata_platform_remove(struct platform_device *pdev)
{
	amba_device_unregister(&sata_platform_device);
	return 0;
}

static struct platform_driver sata_platform_driver = {
	.driver  = {
		.name  = PLATFORM_NAME_IDE,
		.owner = THIS_MODULE,
	},
	.probe   = sata_platform_probe,
	.remove  = sata_platform_remove,
	.suspend = empty_suspend,
	.resume  = empty_resume,
};

static struct sata_board_config_t sata_config_data;
static struct platform_device sata_platform_device = {
	.name = PLATFORM_NAME_SATA,
	.id		= -1,
	.dev	= {
		.platform_data = &sata_config_data,
		.release = empty_release,
	},
	.num_resources  = 0
};

static int sata_register_platform_driver(void)
{
	return platform_driver_register(&sata_platform_driver);
}

static int add_board_device_sata(struct sata_board_config_t *data)
{
	sata_config_data = *data;
	return platform_device_register(&sata_platform_device);
}

#else
static int sata_register_platform_driver(void) { return 0; }
static int add_board_device_sata(
		struct sata_board_config_t *data) { return 0; }
#endif

#if defined(CONFIG_DW_STMMAC) || defined(CONFIG_DW_STMMAC_MODULE)

#define GMAC_AMBA_NAME	DW_GMAC_AMBA_NAME
#define GMAC_AMBA_PID	DW_GMAC_AMBA_DEVID
static struct dw_gmac_driver_data_t gmac_amba_pdata;

static int gmac_arch_init(struct gmac_board_config_t *config)
{
	mobi_reset_enable(RESET_ID_GMAC);
	return 0;
}

static struct amba_device gmac_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = GMAC_AMBA_NAME,
		.platform_data = &gmac_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = GMAC_BASE,
		.end    = GMAC_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_GMAC, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= GMAC_AMBA_PID,
};

static int gmac_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct gmac_board_config_t *config =
		(struct gmac_board_config_t *)pdev->dev.platform_data;

	memcpy(&gmac_amba_pdata, &config->gmac_data,
			sizeof(gmac_amba_pdata));

	ret = gmac_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&gmac_amba_device, &iomem_resource);

	return ret;
}

static int gmac_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&gmac_amba_device);
	return 0;
}

static struct platform_driver gmac_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_GMAC,
		.owner  = THIS_MODULE,
	},
	.probe          = gmac_platform_driver_probe,
	.remove         = gmac_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct gmac_board_config_t gmac_config_data;
static struct platform_device gmac_platform_device = {
	.name   = PLATFORM_NAME_GMAC,
	.id     = -1,
	.dev    = {
		.platform_data  = &gmac_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int gmac_register_platform_driver(void)
{
	return platform_driver_register(&gmac_platform_driver);
}

static int add_board_device_gmac(struct gmac_board_config_t *data)
{
	gmac_config_data = *data;
	return platform_device_register(&gmac_platform_device);
}

#else
static int gmac_register_platform_driver(void) { return 0; }
static int add_board_device_gmac(
		struct gmac_board_config_t *data) { return 0; }
#endif /* GMAC */

#if defined(CONFIG_DWCMSH) || defined(CONFIG_DWCMSH_MODULE)

#define SDMMC_AMBA_NAME	DW_SDMMC_AMBA_NAME
#define SDMMC_AMBA_PID	DW_SDMMC_AMBA_DEVID
static struct dwc_sdmmc_device_data_t sdmmc_amba_pdata;
static int sdmmc_arch_init(struct sdmmc_board_config_t *config)
{
	/* turn it off until actual device driver is loaded */
	mobi_clock_set_rate(CLOCK_ID_SDMMC, 0);
	return mobi_reset_enable(RESET_ID_SDMMC);
}

static struct amba_device sdmmc_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = SDMMC_AMBA_NAME,
		.platform_data = &sdmmc_amba_pdata,
		.release = static_amba_device_release,
	},
	.res = {
		.start  = SDMMC_BASE,
		.end    = SDMMC_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_SDMMC, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= SDMMC_AMBA_PID,
};

static int sdmmc_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct sdmmc_board_config_t *config =
		(struct sdmmc_board_config_t *)pdev->dev.platform_data;

	memcpy(&sdmmc_amba_pdata, &config->device_data, 
			sizeof(sdmmc_amba_pdata));

	ret = sdmmc_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&sdmmc_amba_device, &iomem_resource);

	return ret;
}

static int sdmmc_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&sdmmc_amba_device);
	return 0;
}

static struct platform_driver sdmmc_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_SDMMC,
		.owner  = THIS_MODULE,
	},
	.probe          = sdmmc_platform_driver_probe,
	.remove         = sdmmc_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct sdmmc_board_config_t sdmmc_config_data;
static struct platform_device sdmmc_platform_device = {
	.name   = PLATFORM_NAME_SDMMC,
	.id     = -1,
	.dev    = {
		.platform_data  = &sdmmc_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int sdmmc_register_platform_driver(void)
{
	return platform_driver_register(&sdmmc_platform_driver);
}

static int add_board_device_sdmmc(struct sdmmc_board_config_t *data)
{
	sdmmc_config_data = *data;
	return platform_device_register(&sdmmc_platform_device);
}

#else
static int add_board_device_sdmmc(
		struct sdmmc_board_config_t *data) { return 0; }
static int sdmmc_register_platform_driver(void) { return 0; }
#endif /* DWCMSH - sdmmc */

#if defined(CONFIG_USB_DW_FALCON) || defined(CONFIG_USB_DW_FALCON_MODULE)

#define USB_AMBA_NAME	DW_OTG_AMBA_NAME
#define USB_AMBA_PID	DW_OTG_AMBA_DEVID
static struct dw_otg_driver_data_t usb_amba_pdata;

static int usb_arch_init(struct usb_board_config_t *config)
{
	return mobi_reset_enable(RESET_ID_OTGPHY);
}

/* must init, some usb code does look at this */
uint64_t usb_dev_dma_mask = ~0;
static struct amba_device usb_amba_device = {
	.dev = {
		.coherent_dma_mask = ~0,
		.init_name = USB_AMBA_NAME,
		.platform_data = &usb_amba_pdata,
		.release = static_amba_device_release,
		.dma_mask = &usb_dev_dma_mask,
	},
	.res = {
		.start  = USB_BASE,
		.end    = USB_END,
		.flags  = IORESOURCE_MEM,
	},
	.irq        = { MOBI_IRQ_USB, NO_IRQ },
	.dma_mask	= ~0,
	.periphid 	= USB_AMBA_PID,
};

static int usb_platform_driver_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct usb_board_config_t *config =
		(struct usb_board_config_t *)pdev->dev.platform_data;

	memcpy(&usb_amba_pdata, &config->otg_data,
			sizeof(usb_amba_pdata));

	ret = usb_arch_init(config);
	if (ret < 0)
		return ret;

	amba_device_register(&usb_amba_device, &iomem_resource);

		return ret;
}

static int usb_platform_driver_remove(struct platform_device *pdev)
{
	amba_device_unregister(&usb_amba_device);
	return 0;
}

static struct platform_driver usb_platform_driver = {
	.driver = {
		.name   = PLATFORM_NAME_OTG,
		.owner  = THIS_MODULE,
	},
	.probe          = usb_platform_driver_probe,
	.remove         = usb_platform_driver_remove,
	.suspend        = empty_suspend,
	.resume         = empty_resume,
};

static struct usb_board_config_t usb_config_data;
static struct platform_device usb_platform_device = {
	.name   = PLATFORM_NAME_OTG,
	.id     = -1,
	.dev    = {
		.platform_data  = &usb_config_data,
		.release        = empty_release,
	},
	.num_resources  = 0,
};

static int usb_register_platform_driver(void)
{
	return platform_driver_register(&usb_platform_driver);
}

static int add_board_device_usb(struct usb_board_config_t *data)
{
	usb_config_data = *data;
	return platform_device_register(&usb_platform_device);
}

#else
static int usb_register_platform_driver(void) { return 0; }
static int add_board_device_usb(
		struct usb_board_config_t *data) { return 0; }
#endif /* USB_DW_CORE */

/* platform devices */
extern struct faclon_ioctrl_data_t arch_ioctrl_data;
static struct platform_device ioctrl_platform_device = {
	.name = PLATFORM_NAME_IOCTRL,
	.id   = -1,
	.num_resources = 0,
	.dev  = {
		.platform_data = &arch_ioctrl_data,
		.release       = empty_release,
	},
};

static int add_device_ioctrl(void)
{
	return platform_device_register(&ioctrl_platform_device);
}

static struct platform_device codec_reset_platform_device = {
	.name   = PLATFORM_NAME_CODEC_RESET,
	.id     = -1,
	.dev    = {
		.platform_data	= NULL,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int add_device_codec_reset(void)
{
	return platform_device_register(&codec_reset_platform_device);
}

static struct platform_device qcc_platform_device = {
	.name   = PLATFORM_NAME_QCC,
	.id     = -1,
	.dev    = {
		.platform_data	= NULL,
		.release = empty_release,
	},
	.num_resources = 0,
};

int add_device_qcc(void)
{
	return platform_device_register(&qcc_platform_device);
}

/*
 * unlike most devices in this file, these are not hardware control
 * blocks. we want to have on every system so we add them here instead
 * of leaving to the board drivers
 */
static int arch_add_platform_devices(void)
{
	int ret = 0;

	if ((ret = add_device_ioctrl()) < 0) {
		printk(KERN_ERR "ioctrl error\n");
		return ret;

	} else if ((ret = add_device_codec_reset()) < 0) {
		printk(KERN_ERR "codec_reset error\n");
		return ret;

	} else if ((ret = add_device_qcc()) < 0) {
		printk(KERN_ERR "codec_reset error\n");
		return ret;
	}

	return ret;
}

struct amba_device_container {
	struct amba_device dev;
	char registered;
};

#if defined(CONFIG_I2C_DWAPBI2C) || defined(CONFIG_I2C_DWAPBI2C_MODULE)

#define _MHZ 1000000
static struct amba_device_container i2c0_amba_device = {
	.dev = {
		.res		= {
			.start	= I2C0_BASE,
			.end	= I2C0_BASE + I2C0_SIZE - 1,
			.flags	= IORESOURCE_MEM,
		},
		.dev		= {
			.init_name = "I2C_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq		= {
			MOBI_IRQ_I2C0,
			NO_IRQ,
		},
		.dma_mask	= ~0,
		.periphid 	= DWAPBI2C_ID | 0,
	},
};

static struct amba_device_container i2c1_amba_device = {
	.dev = {
		.res		= {
			.start	= I2C1_BASE,
			.end	= I2C1_BASE + I2C1_SIZE - 1,
			.flags	= IORESOURCE_MEM,
		},
		.dev		= {
			.init_name = "I2C_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq		= {
			MOBI_IRQ_I2C1,
			NO_IRQ,
		},
		.dma_mask	= ~0,
		.periphid 	= DWAPBI2C_ID | 1,
	},
};
/* This is for reenabling reset when all devices has been removed */
static int i2c_refcount = 0;
static int i2c_platform_driver_probe(struct platform_device *pdev)
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbi2c_bus_data *bdata = adev->dev.platform_data;
	int clock_id;
	unsigned long clock_rate = 3 * _MHZ;
	int ret;

	if (mobi_reset_state(RESET_ID_I2C) != 0)
		mobi_reset_disable(RESET_ID_I2C);

	if (adev->res.start == I2C0_BASE) {
		clock_id 	= CLOCK_ID_I2C0;
		bdata->clock_id = "i2c0";
	} else {
		clock_id 	= CLOCK_ID_I2C1;
		bdata->clock_id = "i2c1";
	}

	mobi_clock_set_rate(clock_id, clock_rate);
	if (unlikely(!mobi_clock_get_rate(clock_id))) {
		printk(KERN_ERR "%s: %s: could not set clock id=%d (%s) to %luHz\n",
				adev->dev.init_name, __func__, 
				clock_id, bdata->clock_id, clock_rate);
		return -EINVAL;
	}

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	i2c_refcount++;

	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int i2c_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) { 
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	mobi_clock_set_rate(adev->res.start == I2C0_BASE ?
			CLOCK_ID_I2C0 : CLOCK_ID_I2C1, 0);
	if (--i2c_refcount == 0)
		mobi_reset_enable(RESET_ID_I2C);

	return 0;
}

static struct platform_driver i2c0_pdrv = {
	.probe		= i2c_platform_driver_probe,
	.remove		= i2c_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
	.driver	= {
		.name	= "m_i2c0",
		.owner	= THIS_MODULE,
	},
};

static struct platform_device i2c0_pdev = {
	.name           = "m_i2c0",
	.id             = -1,
	.dev            = {
		.platform_data = &i2c0_amba_device,
		.release = empty_release,
	},
	.num_resources	= 0,
};

static struct platform_driver i2c1_pdrv = {
	.probe		= i2c_platform_driver_probe,
	.remove		= i2c_platform_driver_remove,
	.suspend	= empty_suspend,
	.resume		= empty_resume,
	.driver	= {
		.name	= "m_i2c1",
		.owner	= THIS_MODULE,
	},
};

static struct platform_device i2c1_pdev = {
	.name           = "m_i2c1",
	.id             = -1,
	.dev            = {
		.platform_data = &i2c1_amba_device,
		.release = empty_release,
	},
	.num_resources	= 0,
};

static int i2c_register_platform_driver(void) 
{ 
	int ret = platform_driver_register(&i2c0_pdrv); 
	if (ret) 
		return ret;

	return platform_driver_register(&i2c1_pdrv); 
}

static int add_board_device_i2c(
		struct dwapbi2c_bus_data *bdata, int devid) 
{
	int ret = 0;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						i2c0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						i2c1_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&i2c0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&i2c1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static inline int i2c_register_platform_driver(void) { return 0; };
static int add_board_device_i2c(
		struct dwapbi2c_bus_data *bdata, int devid) { return 0; };
#endif

#if defined(CONFIG_SPI_DWAPBSSI) || defined(CONFIG_SPI_DWAPBSSI_MODULE)

static struct amba_device_container spi0_amba_device = {
	.dev = {
		.res = {
			.start = SSI0_BASE,
			.end = SSI0_BASE + SSI0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 0,
	},
};

static struct amba_device_container spi1_amba_device = {
	.dev = {
		.res = {
			.start = SSI1_BASE,
			.end = SSI1_BASE + SSI1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 1,
	},
};

static struct amba_device_container spi2_amba_device = {
	.dev = {
		.res = {
			.start = SSI2_BASE,
			.end = SSI2_BASE + SSI2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "SPI_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_SSI2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBSSI_ID | 2,
	},
};

static int spi_refcount = 0;
static int spi_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbssi_bus_data *bdata = adev->dev.platform_data;
	int ret;
	bdata->clock_id = "ahb";

	if (mobi_reset_state(RESET_ID_SSI) != 0)
		mobi_reset_disable(RESET_ID_SSI);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	spi_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)
		 pdev->dev.platform_data)->registered = 1;
	return ret;
}

static int spi_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--spi_refcount == 0)
		mobi_reset_enable(RESET_ID_SSI);

	return 0;
}

static struct platform_driver spi0_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi0_pdev = {
	.name = "m_spi0",
	.id = -1,
	.dev = {
		.platform_data = &spi0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver spi1_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi1_pdev = {
	.name = "m_spi1",
	.id = -1,
	.dev = {
		.platform_data = &spi1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver spi2_pdrv = {
	.probe = spi_platform_driver_probe,
	.remove = spi_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_spi2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device spi2_pdev = {
	.name = "m_spi2",
	.id = -1,
	.dev = {
		.platform_data = &spi2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int add_board_device_spi(
		struct dwapbssi_bus_data *bdata, int devid) 
{
	int ret = 0;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						spi0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						spi1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						spi2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&spi0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&spi1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&spi2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int spi_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&spi0_pdrv);
	if (ret) 
		return ret;

	ret = platform_driver_register(&spi1_pdrv);
	if (ret) 
		return ret;

	return platform_driver_register(&spi2_pdrv);
}

#else
static int add_board_device_spi(
		struct dwapbssi_bus_data *bdata, int devid) { return 0; }; 
static int spi_register_platform_driver(void) { return 0; }; 
#endif

#if defined(CONFIG_CADENCE_PWM) || defined(CONFIG_CADENCE_PWM_MODULE)
static struct amba_device_container pwm0_amba_device = {
	.dev = {
		.res = {
			.start = PWM0_BASE,
			.end = PWM0_BASE + PWM0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 0,
	},
};

static struct amba_device_container pwm1_amba_device = {
	.dev = {
		.res = {
			.start = PWM1_BASE,
			.end = PWM1_BASE + PWM1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 1,
	},
};

static struct amba_device_container pwm2_amba_device = {
	.dev = {
		.res = {
			.start = PWM2_BASE,
			.end = PWM2_BASE + PWM2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "PWM_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_PWM2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = CADPWM_ID | 2,
	},
};

static int pwm_refcount = 0;
static int pwm_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct cadpwm_block_data *bdata = adev->dev.platform_data;
	int ret;

	bdata->clock_id = "apb";

	if (mobi_reset_state(RESET_ID_PWM) != 0)
		mobi_reset_disable(RESET_ID_PWM);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	pwm_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)
		 pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int pwm_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--pwm_refcount == 0)
		mobi_reset_enable(RESET_ID_PWM);

	return 0;
}

static struct platform_driver pwm0_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm0_pdev = {
	.name = "m_pwm0",
	.id = -1,
	.dev = {
		.platform_data = &pwm0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver pwm1_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm1_pdev = {
	.name = "m_pwm1",
	.id = -1,
	.dev = {
		.platform_data = &pwm1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};


static struct platform_driver pwm2_pdrv = {
	.probe = pwm_platform_driver_probe,
	.remove = pwm_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_pwm2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device pwm2_pdev = {
	.name = "m_pwm2",
	.id = -1,
	.dev = {
		.platform_data = &pwm2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int pwm_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&pwm0_pdrv);
	if (ret)
		return ret;

	ret = platform_driver_register(&pwm1_pdrv);
	if (ret)
		return ret;

	return platform_driver_register(&pwm2_pdrv);
}

static int add_board_device_pwm(
		struct cadpwm_block_data *bdata, int devid) 
{
	int ret;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						pwm0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						pwm1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						pwm2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;
	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;
	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&pwm0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&pwm1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&pwm2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static int pwm_register_platform_driver(void) { return 0; }; 
static int add_board_device_pwm(
		struct cadpwm_block_data *bdata, int devid) { return 0; }; 
#endif

#if defined(CONFIG_GPIO_DWAPBGPIO) || defined(CONFIG_GPIO_DWAPBGPIO_MODULE)
static struct amba_device_container gpio0_amba_device = {
	.dev = {
		.res = {
			.start = GPIO0_BASE,
			.end = GPIO0_BASE + GPIO0_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_0",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO0,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 0,
	},
};

static struct amba_device_container gpio1_amba_device = {
	.dev = {
		.res = {
			.start = GPIO1_BASE,
			.end = GPIO1_BASE + GPIO1_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_1",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO1,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 1,
	},
};

static struct amba_device_container gpio2_amba_device = {
	.dev = {
		.res = {
			.start = GPIO2_BASE,
			.end = GPIO2_BASE + GPIO2_SIZE - 1,
			.flags = IORESOURCE_MEM,
		},
		.dev = {
			.init_name = "GPIO_2",
			.coherent_dma_mask = ~0,
			.release = static_amba_device_release,
		},
		.irq = {
			MOBI_IRQ_GPIO2,
			NO_IRQ,
		},
		.dma_mask = ~0,
		.periphid = DWAPBGPIO_ID | 2,
	},
};

static int gpio_refcount = 0;
static int gpio_platform_driver_probe(struct platform_device *pdev) 
{
	struct amba_device *adev = 
		&((struct amba_device_container *)pdev->dev.platform_data)->dev;
	struct dwapbgpio_block_data *bdata = adev->dev.platform_data;
	int ret;

	bdata->port_a_count = 0;
	bdata->port_b_count = 0;
	bdata->port_c_count = 0;
	bdata->port_d_count = 0;

	switch (adev->res.start) {
	case GPIO0_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(0);
		bdata->gpio_id_offset = 0;
		break;
	case GPIO1_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(1);
		bdata->gpio_id_offset = mobi_ioctrl_get_gpio_bank_size(0);
		break;
	case GPIO2_BASE:
		bdata->port_a_count = mobi_ioctrl_get_gpio_bank_size(2);
		bdata->gpio_id_offset = mobi_ioctrl_get_gpio_bank_size(0)
			+ mobi_ioctrl_get_gpio_bank_size(1);
		break;
	default:
		return -EINVAL;
	}

	if (!bdata->port_a_count)
		return -ENODEV;

	if (mobi_reset_state(RESET_ID_GPIO) != 0)
		mobi_reset_disable(RESET_ID_GPIO);

	mobi_ioctrl_apply_macro(mobi_ioctrl_get_macro(adev->dev.init_name));

	gpio_refcount++;
	ret = amba_device_register(adev, &iomem_resource);
	if (likely(!ret))
		((struct amba_device_container *)pdev->dev.platform_data)->registered = 1;

	return ret;
}

static int gpio_platform_driver_remove(struct platform_device *pdev) 
{
	struct amba_device_container *__ac = pdev->dev.platform_data;
	struct amba_device *adev = &__ac->dev;

	if (__ac->registered) {
		amba_device_unregister(adev);
		__ac->registered = 0;
	}

	if (--gpio_refcount == 0)
		mobi_reset_enable(RESET_ID_GPIO);

	return 0;
}

static struct platform_driver gpio0_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio0",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio0_pdev = {
	.name = "m_gpio0",
	.id = -1,
	.dev = {
		.platform_data = &gpio0_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver gpio1_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio1",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio1_pdev = {
	.name = "m_gpio1",
	.id = -1,
	.dev = {
		.platform_data = &gpio1_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static struct platform_driver gpio2_pdrv = {
	.probe = gpio_platform_driver_probe,
	.remove = gpio_platform_driver_remove,
	.suspend = empty_suspend,
	.resume = empty_resume,
	.driver = {
		.name = "m_gpio2",
		.owner = THIS_MODULE,
	},
};
static struct platform_device gpio2_pdev = {
	.name = "m_gpio2",
	.id = -1,
	.dev = {
		.platform_data = &gpio2_amba_device,
		.release = empty_release,
	},
	.num_resources = 0,
};

static int gpio_register_platform_driver(void) 
{
	int ret;

	ret = platform_driver_register(&gpio0_pdrv);
	if (ret) 
		return ret;
	ret = platform_driver_register(&gpio1_pdrv);
	if (ret) 
		return ret;

	return platform_driver_register(&gpio2_pdrv);
}

static int add_board_device_gpio(
		struct dwapbgpio_block_data *bdata, int devid) 
{
	int ret;
	struct amba_device *adev = NULL;

	switch (devid) {
	case 0: adev = &(((struct amba_device_container *) 
						gpio0_pdev.dev.platform_data)->dev);
			break;
	case 1: adev = &(((struct amba_device_container *) 
						gpio1_pdev.dev.platform_data)->dev);
			break;
	case 2: adev = &(((struct amba_device_container *) 
						gpio2_pdev.dev.platform_data)->dev);
			break;
	default:
		return -EINVAL;
	}

	if (unlikely(!bdata))
		return -EINVAL;

	if (adev->dev.platform_data)
		return -EALREADY;

	adev->dev.platform_data = bdata;

	switch (devid) {
	case 0:
		if (unlikely(ret = platform_device_register(&gpio0_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 1:
		if (unlikely(ret = platform_device_register(&gpio1_pdev)))
			adev->dev.platform_data = NULL;
		break;
	case 2:
		if (unlikely(ret = platform_device_register(&gpio2_pdev)))
			adev->dev.platform_data = NULL;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

#else
static int gpio_register_platform_driver(void) { return 0; };
static int add_board_device_gpio(
		struct dwapbgpio_block_data *bdata, int devid) { return 0; }
#endif

int arch_add_board_devices(
		struct board_device_desc *board_devices, int num_devices)
{
	int i;
	int ret = 0;

	for (i = 0; i < num_devices; i++)
	{
		switch (board_devices->device) {
		case BOARD_DEVICE_NAND:
			ret = add_board_device_nand(
					(struct nand_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_WATCHDOG:
			ret = add_board_device_watchdog(
					(struct wdt_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_SDMMC:
			ret = add_board_device_sdmmc(
					(struct sdmmc_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_GMAC:
			ret = add_board_device_gmac(
					(struct gmac_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_USB:
			ret = add_board_device_usb(
					(struct usb_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_SATA:
			ret = add_board_device_sata(
					(struct sata_board_config_t *)board_devices->platform_data);
			break;
		case BOARD_DEVICE_SPI:
			ret = add_board_device_spi(
					(struct dwapbssi_bus_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_I2C:
			ret = add_board_device_i2c(
					(struct dwapbi2c_bus_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_PWM:
			ret = add_board_device_pwm(
					(struct cadpwm_block_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		case BOARD_DEVICE_GPIO:
			ret = add_board_device_gpio(
					(struct dwapbgpio_block_data *)board_devices->platform_data, 
					board_devices->devid);
			break;
		default:
			ret = -1;
		}
		if (ret) {
			printk(KERN_ERR "ERROR: Failed to add board device %s, devid %d\n",
					board_device_types_names[board_devices->device],
					board_devices->devid);
			return ret;
		}

		board_devices++;
	}
	return ret;
}
EXPORT_SYMBOL(arch_add_board_devices);

/* for possible future reference */
/*
void set_ahb_dma_master_priority(void)
{
	void __iomem *ahb_dma_base = ioremap(AHB_ARB_DMA_BASE, AHB_ARB_DMA_SIZE);
	uint32_t regval;
	int i;

	for (i=0;i<4;i++) {
		printk("Master %d has priority 0x%x\n", i+1, readl(ahb_dma_base + (0x4*i)));
	}
	regval = 0xf;
	writel(regval, ahb_dma_base+0x4);
}
*/

int __init mx64180_register_platform_drivers(void)
{

	if (wdt_register_platform_driver())
		return -1;
	if (nand_register_platform_driver())
		return -1;
	if (sdmmc_register_platform_driver())
		return -1;
	if (gmac_register_platform_driver())
		return -1;
	if (usb_register_platform_driver())
		return -1;
	if (i2c_register_platform_driver())
		return -1;
	if (spi_register_platform_driver())
		return -1;
	if (pwm_register_platform_driver())
		return -1;
	if (gpio_register_platform_driver())
		return -1;
	if (sata_register_platform_driver())
		return -1;

	if (arch_add_platform_devices());
		return -1;

	return 0;
}
