/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


/// \defgroup NonRealTimeVideoPreProcessorControlObject Non-Realtime Video Pre-Processor
/// \{

/// \page nvpp Non-Realtime Video Pre-Processor
/// \latexonly
/// \chapter{Non-Realtime Video Pre-Processor}
/// \endlatexonly
///
/// The NVPP object is memory-to-memory version of the LVPP.  As such, it has a
/// very similar API for all pixel-processing configuration, and only differs
/// in video input.  For obvious reasons, there are no configuration parameters
/// related to synchronizing to an external video source.  Furthermore, the NVPP
/// object can operate on both 4:2:0 and 4:2:2 video frames.
/// \section lvpp_status Status Block
///
/// The NVPP status block contains information related to state and performance.  Unlike
/// the LVPP where some status information is contained in the SystemControl object's
/// status block, the NVPP status is entirely contained in this block.
///
/// \latexonly
/// \input{structVPP__STATUS}
/// \endlatexonly


#ifndef __QNVPP_HH
#define __QNVPP_HH

#include "qglobal.h"

//////////////////////////////////////////////////////////////////////
//
// Commands
//

/// \brief The start command puts the object into the RUN state and
/// allows input frames to be processed.
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_START                    1

/// \brief The stop command stops the video capture.
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_STOP                     2

/// \brief This command stops video capture and sends a flush message to 
/// all downstream objects.  When those objects receive the flush message
/// they will automatically stop and flush their internal buffers.
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_FLUSH                    3

/// \brief This command is used to set a double-buffered configuration parameter.
/// \param argument arg[0] = param
/// \param argument arg[1] = value
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_PARAM                4

/// \brief This command activates all double-buffered configuration parameters set since the last activate.
/// \param argument none
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_ACTIVATE_CFG             5

/// \brief This command is used to set a double-buffered configuration parameter 
/// of the form Q_NVPP_CMP_REGION_*
/// \param argument arg[0] = region
/// \param argument arg[1] = value
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_REGION_PARAM         6

/// \brief This command is used to set a double-buffered configuration parameter of the
/// form Q_NVPP_CMP_REGION_FEATURE_*
/// \param argument arg[0] = region
/// \param argument arg[1] = value
/// \param argument arg[2] = param
/// \param argument arg[3] = value
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_REGION_FEATURE_PARAM 7

/// \brief This command is used to reset all region features for a particular region.
/// \param argument arg[0] = region
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_RESET_REGION_FEATURES    8

/// \brief This command is used to set up to 16-entries in the 
/// gamma lookup table.  The first entry to be set is specified
/// by the index argument, the number of entries to set is specified
/// by the length argument, and up to 16 subsequent entries are the
/// gamma values.
/// \param argument arg[0] = index
/// \param argument arg[1] = length
/// \param argument arg[2...17] = gamma 
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_GAMMA_LUT            9

/// \brief This command binds the input of the NVPP object to the output
/// of an upstream object.
/// \param ID Object ID of the connecting object
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_BIND_INPUT              10

/// \brief This command unbinds the input of the NVPP object.
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_UNBIND_INPUT            11

/// \brief This command is used to set a double-buffered DIS configuration parameter.
/// \param argument arg[0] = param
/// \param argument arg[1] = value
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_DIS_PARAM           12

/// \brief This command activates all double-buffered DIS configuration parameters set since the last activate.
/// \param argument none
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_ACTIVATE_DIS_CFG        13

/// \brief This command is used to set up to 16-entries in the 
/// temporal filter lookup table.  The first entry to be set is specified
/// by the index argument, the number of entries to set is specified
/// by the length argument, and up to 16 subsequent entries are the
/// gamma values.
/// \param argument arg[0] = index
/// \param argument arg[1] = length
/// \param argument arg[2...17] = gamma 
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_SET_TF_LUT           14

/// \brief The resume command puts the object into the RUN state and
/// allows input frames to be continue processed.
/// \return Code:  1 for success, 0 for failure
#define Q_NVPP_CMD_RESUME                  15

#define Q_NVPP_MAX_CMD                     16

///////////////////////////////////////////////////////////////////////
//
// Command Payloads
// 

typedef struct
{
    int param;
    int value;
} STRUCT_Q_NVPP_CMD_SET_PARAM;

typedef struct
{
    int region;
    int param;
    int value;
} STRUCT_Q_NVPP_CMD_SET_REGION_PARAM;

typedef struct
{
    int region;
    int feature;
    int param;
    int value;
} STRUCT_Q_NVPP_CMD_SET_REGION_FEATURE_PARAM;

typedef struct
{
    int region;
} STRUCT_Q_NVPP_CMD_RESET_REGION_FEATURES;

typedef struct
{
    int index;
    int length;
    unsigned char gamma[16];
} STRUCT_Q_NVPP_CMD_SET_GAMMA_LUT;

typedef struct
{
    int index;
    int length;
    unsigned char tflut[16];
} STRUCT_Q_NVPP_CMD_SET_TF_LUT;

typedef struct
{
    int ctrlObjID;
} STRUCT_Q_NVPP_CMD_BIND_INPUT;

typedef struct
{
    unsigned int width;
    unsigned int height;
    unsigned int interlaced;
    unsigned int is420;
} STRUCT_Q_NVPP_CMD_SET_INPUT_FORMAT;

//////////////////////////////////////////////////////////////////////
//
// Command Parameter Values
//

#define Q_NVPP_CMP_SOME_PARAMETER                    0

//////////////////////////////////////////////////////////////////////
//
// Events
//
#define NVPP_EVENT(ID) ((NVPP_CTRLOBJ_TYPE << 16) | ID)

#define Q_NVPP_EV_VIDEO_FRAME_CAPTURED  NVPP_EVENT(1) 
#define Q_NVPP_EV_VIDEO_INPUT_STOPDONE  NVPP_EVENT(2) 


typedef struct {
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        objIndex;
} STRUCT_Q_NVPP_EV_FRAME_CAPTURED;

typedef struct {
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        objIndex;
} STRUCT_Q_NVPP_EV_VIDEO_INPUT;

//////////////////////////////////////////////////////////////////////
//
// Configuration parameters
//
// Generated from mediaware.h VPP_CFG s/VPP_CFG/Q_NVPP_CMP/
/// \brief This parameter sets the field offset of the second field. This only affects interlaced input.
///        This value needs to be programmed with the (height of the first field + vblank of second field).
/// \param value Integer number.
///
#define Q_NVPP_CMP_FIELD_OFFSET                      0
/// \brief This parameter sets the width of the crop rectangle relative to the crop x offset.
/// \param value Integer number.
///
#define Q_NVPP_CMP_CROP_WIDTH                        1
/// \brief This parameter sets the height of the crop rectangle relative to the crop y offset.
/// \param value Integer number.
///
#define Q_NVPP_CMP_CROP_HEIGHT                       2
/// \brief This parameter sets the horizontal offset of the crop rectangle relative to the start of active video.
/// \param value Integer number.
///
#define Q_NVPP_CMP_CROP_OFFSET_X                     3
/// \brief This parameter sets the vertical offset of the crop rectangle relative to the start of active video.
/// \param value Integer number.
///
#define Q_NVPP_CMP_CROP_OFFSET_Y                     4
/// \brief This parameter sets the x offset of the output picture.
/// \param value Integer value.
///
#define Q_NVPP_CMP_PICT_OFFSET_X                     5
/// \brief This parameter sets the y offset of the output picture.
/// \param value Integer value.
///
#define Q_NVPP_CMP_PICT_OFFSET_Y                     6
/// \brief This parameter sets the width of the output picture.
/// \param value Integer value.
///
#define Q_NVPP_CMP_PICT_WIDTH                        7
/// \brief This parameter sets the height of the output picture.
/// \param value Integer value.
///
#define Q_NVPP_CMP_PICT_HEIGHT                       8
/// \brief This parameter sets whether the output picture is progressive.
/// \param value 0 = Interlaced, 1 = Progressive.
///
#define Q_NVPP_CMP_DEST_PROG                         9
/// \brief This parameter sets whether the output picture aligned to macro block boundaries.
///        When connecting to the encoder this needs to be set to 1 as the encoder operates on
///        macro block sizes.
/// \param value 0 = not aligned to MB boundaires, 1 = aligned to MB boundaries.
///
#define Q_NVPP_CMP_SCALE_MB_ALIGN                   10
/// \brief This parameter selects one of the several motion transfer function co-efficient sets available.
/// \param value 0 = OFF, 1 = Weak, 2 = Medium, 3 = Strong.
///
#define Q_NVPP_CMP_TF_LEVEL                         11
/// \brief This parameter sets the attenuation for luma motion adaptive attenuation of the output of the motion non-linear transfer function. This parameter is set when it lies between the High and Low thresholds enabled by the parameters TF_LUMA_LOW and TF_LUMA_HIGH.
/// \param value 0 = Gain 0 Complete attenuation\n
///              1 = Gain 1/8\n
///              2 = Gain 1/4\n
///              3 = Gain 1/2\n
///              4 = Gain 5/8\n
///              5 = Gain 3/4\n
///              6 = Gain 7/8\n
///              7 = Gain 8/8 Motion signal not attenuated\n
///
#define Q_NVPP_CMP_TF_ATTENUATION_Y0                12
/// \brief This parameter sets the attenuation for luma motion adaptive attenuation of the output of the motion non-linear transfer function. This parameter is set when it does not lie between the High and Low thresholds set by the parameters TF_LUMA_LOW and TF_LUMA_HIGH.
/// \param value 0 = Gain 0 Complete attenuation\n
///              1 = Gain 1/8\n
///              2 = Gain 1/4\n
///              3 = Gain 1/2\n
///              4 = Gain 5/8\n
///              5 = Gain 3/4\n
///              6 = Gain 7/8\n
///              7 = Gain 8/8 Motion signal not attenuated\n
///
#define Q_NVPP_CMP_TF_ATTENUATION_Y1                13
/// \brief This parameter determines the chroma motion adaptive attenuation of the motion non-linear transfer function. This parameter is used when it lies within the chosen color range enabled by the parameter TF_CHROMA_COLOR_SELECT.
/// \param value 0 = Gain 0 Complete attenuation\n
///              1 = Gain 1/8\n
///              2 = Gain 1/4\n
///              3 = Gain 1/2\n
///              4 = Gain 5/8\n
///              5 = Gain 3/4\n
///              6 = Gain 7/8\n
///              7 = Gain 8/8 Motion signal not attenuated\n
///
#define Q_NVPP_CMP_TF_ATTENUATION_C0                14
/// \brief This parameter determines the chroma motion adaptive attenuation of the motion non-linear transfer function. This parameter is used when it does not belong to any color range enabled by the parameter TF_CHROMA_COLOR_SELECT.
/// \param value 0 = Gain 0 Complete attenuation\n
///              1 = Gain 1/8\n
///              2 = Gain 1/4\n
///              3 = Gain 1/2\n
///              4 = Gain 5/8\n
///              5 = Gain 3/4\n
///              6 = Gain 7/8\n
///              7 = Gain 8/8 Motion signal not attenuated\n
///
#define Q_NVPP_CMP_TF_ATTENUATION_C1                15
/// \brief This parameter indicates the low threshold of the attenuation factor to be used motion detected signal depending on the input value of luma.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_TF_LUMA_LOW                      16
/// \brief This parameter indicates the high threshold of the attenuation factor to be used motion detected signal depending on the input value of luma.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_TF_LUMA_HIGH                     17
/// \brief This parameter indicates the amount of motion luma signal to be combined with the motion chroma signal.
/// \param value 0 = Ycorr 0\n
///              1 = Ycorr 1/8\n
///              2 = Ycorr 1/4\n
///              3 = Ycorr 1/2\n
///              4 = Ycorr 5/8\n
///              5 = Ycorr 3/4\n
///              6 = Ycorr 7/8\n
///              7 = Ycorr 8/8\n
///
#define Q_NVPP_CMP_TF_CHROMA_YCORR                  18
/// \brief This parameter indicates the which color regions to enable for application of temporal filter.
/// \param value [0] 0 = Disable color 0, 1 = Enable color 0\n
///              [1] 0 = Disable color 1, 1 = Enable color 1\n
///              [2] 0 = Disable color 2, 1 = Enable color 2\n
///              [3] 0 = Disable color 3, 1 = Enable color 3\n
///
#define Q_NVPP_CMP_TF_CHROMA_COLOR_SELECT           19
/// \brief This is a Horizontal Edge-Preserving Noise Reduction Command that selects one of the many filter configurations available.
/// \param value Integer number 0 - 4.
///
#define Q_NVPP_CMP_HEA_LEVEL                        20
/// \brief This parameter is used to set the adaptive threshold for edge preserving noise reduction filter. Threshold applies difference between center pixel and the immediate neighboring pixels.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_HEA_THRESHOLD_0                  21
/// \brief This parameter is used to set the adaptive threshold for edge preserving noise reduction filter. Threshold applies difference between center pixel and the farthest pixel neighbors. (within 5 pixel radii).
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_HEA_THRESHOLD_1                  22
/// \brief This parameter indicates the input pixel aspect ratio in the X direction.
/// \param value Integer number.
///
#define Q_NVPP_CMP_PIXEL_AR_X                       23
/// \brief This parameter indicates the input pixel aspect ratio in the Y direction.
/// \param value Integer number.
///
#define Q_NVPP_CMP_PIXEL_AR_Y                       24
/// \brief This parameter skips the aspect ratio calculations and uses the pixel aspect ratio
///        values directly.
/// \param value 0 = take scaling effects into account, 1 = ignore scaling effects.
///
#define Q_NVPP_CMP_PIXEL_AR_FIXED                   25
/// \brief This parameter sets the Ka matrix element in the [2x2] hue/saturation matrix.
/// \param value 10-bit fixed point in 2.8, 2's complement
///
#define Q_NVPP_CMP_HUE_SAT_CB_KA                    26
/// \brief This parameter sets the Kb matrix element in the [2x2] hue/saturation matrix.
/// \param value 10-bit fixed point in 2.8, 2's complement
///
#define Q_NVPP_CMP_HUE_SAT_CB_KB                    27
/// \brief This parameter sets the Kc matrix element in the [2x2] hue/saturation matrix.
/// \param value 10-bit fixed point in 2.8, 2's complement
///
#define Q_NVPP_CMP_HUE_SAT_CR_KC                    28
/// \brief This parameter sets the Kd matrix element in the [2x2] hue/saturation matrix.
/// \param value 10-bit fixed point in 2.8, 2's complement
///
#define Q_NVPP_CMP_HUE_SAT_CR_KD                    29
/// \brief This parameter sets the bypass for all statistics output.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_ALL_STAT               30
/// \brief This parameter sets the bypass for color detection.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_COLOR_DETECT           31
/// \brief This parameter is deprecated.
///
#define Q_NVPP_CMP_HW_BYPASS_DEINT                  32
/// \brief This parameter sets the bypass for edge enhance.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_EDGE_ENHANCE           33
/// \brief This parameter sets the bypass for gamma.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_GAMMA                  34
/// \brief This parameter sets the bypass for hue/saturation.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HUESAT                 35
/// \brief This parameter sets the bypass for the horizontal scaler.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HSCALE                 36
/// \brief This parameter is deprecated.
///
#define Q_NVPP_CMP_HW_BYPASS_VSCALE                 37
/// \brief This parameter sets the bypass for the horizontal impulse noise reduction.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HINR                   38
/// \brief This parameter sets the bypass for the horizontal edge enhancement.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HEDGE                  39
/// \brief This parameter sets the bypass for the luma temporal filter.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_TFILTY                 40
/// \brief This parameter sets the bypass for the chroma temporal filter.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_TFILTC                 41
/// \brief This parameter sets the bypass for the temporal filter analysis.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_TSTAT                  42
/// \brief This parameter sets the bypass for the high frequency statistics.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HFREQ                  43
/// \brief This parameter sets the bypass for the histogram statistics.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_HIST                   44
/// \brief This parameter sets the bypass for the MB statistics output.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_MB_STAT                45
/// \brief This parameter sets the bypass for the motion vertical filter.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_MOT_VFILT              46
/// \brief This parameter sets the bypass for the motion edge detection.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_MOT_EDGE               47
/// \brief This parameter sets the bypass for the region generation.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_REGION                 48
/// \brief This parameter sets the bypass for the spatial edge detection.
/// \param value 0 = Enable, 1 = Disable (Bypass)
///
#define Q_NVPP_CMP_HW_BYPASS_SPAT_EDGE              49
/// \brief This parameter sets the PME downscale enable.
/// \param value 0 = Disable, 1 = Enable;
///
#define Q_NVPP_CMP_PMEDS_EN                         50
/// \brief This parameter sets the PME downscale horizontal scale factor.
/// \param value 1, 2, 4;
///
#define Q_NVPP_CMP_PMEDS_H                          51
/// \brief This parameter sets the PME downscale vertical scale factor.
/// \param value 1, 2, 4;
///
#define Q_NVPP_CMP_PMEDS_V                          52
/// \brief This parameter indicates the start pixel of each horizontal line during video capture.
///        This is typically set to 0 for embedded sync (EAV/SAV) and set to the video input
///        horizontal blanking for external sync.
/// \param value Integer number.
///
#define Q_NVPP_CMP_START_HPIXEL                     53
/// \brief This parameter sets the first line to be captured to memory in each frame. This value
///        needs to match the vertical blanking of the video input.
/// \param value Integer number.
///
#define Q_NVPP_CMP_START_VLINE                      54
/// \brief Enables Y correction for HD to SD color space conversion and vice-versa.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_NVPP_CMP_YCORR_ENABLE                     55
/// \brief Contribution of Cb to corrected Y for HD to SD color space conversion and vice-versa.
/// \param value 7-bits, 2's complement.
///
#define Q_NVPP_CMP_YCORR_CB_KE                      56
/// \brief Contribution of Cr to corrected Y for HD to SD color space conversion and vice-versa.
/// \param value 7-bits, 2's complement.
///
#define Q_NVPP_CMP_YCORR_CR_KF                      57
/// \brief Color 0 Hue Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 0 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR0_HUE_LOW                   58
/// \brief Color 0 Hue Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 0 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR0_HUE_HIGH                  59
/// \brief Color 0 Saturation Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 0 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR0_SAT_LOW                   60
/// \brief Color 0 Saturation Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 0 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR0_SAT_HIGH                  61
/// \brief Color 1 Hue Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 1 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR1_HUE_LOW                   62
/// \brief Color 1 Hue Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 1 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR1_HUE_HIGH                  63
/// \brief Color 1 Saturation Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 1 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR1_SAT_LOW                   64
/// \brief Color 1 Saturation Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 1 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR1_SAT_HIGH                  65
/// \brief Color 2 Hue Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 2 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR2_HUE_LOW                   66
/// \brief Color 2 Hue Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 2 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR2_HUE_HIGH                  67
/// \brief Color 2 Saturation Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 2 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR2_SAT_LOW                   68
/// \brief Color 2 Saturation Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 2 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR2_SAT_HIGH                  69
/// \brief Color 3 Hue Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 3 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR3_HUE_LOW                   70
/// \brief Color 3 Hue Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 3 detection by the Hue comparator.
///
#define Q_NVPP_CMP_COLOR3_HUE_HIGH                  71
/// \brief Color 3 Saturation Lower Boundary Level.
/// \param value 8-bit value for the lower boundary of color 3 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR3_SAT_LOW                   72
/// \brief Color 3 Saturation Upper Boundary Level.
/// \param value 8-bit value for the upper boundary of color 3 detection by the Saturation comparator.
///
#define Q_NVPP_CMP_COLOR3_SAT_HIGH                  73
/// \param value Integer number 0 - 3. 0 means no change to MB average.
///
#define Q_NVPP_CMP_GSTAT_HFMB_GAIN                  74
/// \brief This parameter sets the gain to boost the vertical high frequency detection.
/// \param value 4-bit integer and 2-bit fractional number.
///
#define Q_NVPP_CMP_GSTAT_VF_GAIN                    75
/// \brief This parameter sets the gain to boost the horizontal high frequency detection.
/// \param value 4-bit integer and 2-bit fractional number.
///
#define Q_NVPP_CMP_GSTAT_HF_GAIN                    76
/// \brief This parameter sets the mode in which the low frequency statistics are calculated.
/// \param value 0 = Average of horizontal and vertical frequency.\n
///              1 = Horizontal frequency.\n
///              2 = Vertical frequency.\n
///              3 = Reserved.\n
///
#define Q_NVPP_CMP_GSTAT_HV_LOW                     77
/// \brief This parameter sets the mode in which the high frequency statistics are calculated.
/// \param value 0 = Average of horizontal and vertical frequency.\n
///              1 = Horizontal frequency.\n
///              2 = Vertical frequency.\n
///              3 = Reserved.\n
///
#define Q_NVPP_CMP_GSTAT_HV_HIGH                    78
/// \brief This parameter sets the threshold between the two strongest and adjacent neighboring
///        directions to be considered an intermediate direction.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_GEDGE_NEIGHBOR                   79
/// \brief This parameter sets the lowest MB average value to be considered a valid edge.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_GEDGE_LOW                        80
/// \brief This parameter sets the gain in calculating Temporal Edge MB averages.
/// \param value Integer number 0 - 3.
///
#define Q_NVPP_CMP_GEDGE_MOT_MB_GAIN                81
/// \brief This parameter sets the gain in calculating Spatial Edge MB averages.
/// \param value Integer number 0 - 3.
///
#define Q_NVPP_CMP_GEDGE_MB_GAIN                    82
/// \brief This parameter defines the adaptive threshold for luma horizontal impulse noise reduction filters.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_HINR_THRESHOLD_Y                 83
/// \brief This parameter defines the adaptive threshold for chroma horizontal impulse noise reduction filters.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_HINR_THRESHOLD_C                 84
/// \brief This parameter controls the Sobel edge values. Average value of all Sobel edge is less than the threshold then the output of the smoothening FIR is set to zero.
/// \param value Integer number 0 - 31. 31 turns off edge enhancement.
///
#define Q_NVPP_CMP_EDGE_ENHANCE_THRESHOLD           85
/// \brief This parameter controls the Attenuation to final enhancement value.
/// \param value Integer number 0 - 7.
///
#define Q_NVPP_CMP_EDGE_RSHIFT                      86
/// \brief This parameter controls the Gain applied to the left edge detected.
/// \param value 0 = no shift, 1 = 1 bit shift, 2 = 2 bit shift, 3 = 3 bit shift
///
#define Q_NVPP_CMP_EDGE_LSHIFT_LEFT                 87
/// \brief This parameter controls the Gain applied to the right edge detected.
/// \param value 0 = no shift, 1 = 1 bit shift, 2 = 2 bit shift, 3 = 3 bit shift
///
#define Q_NVPP_CMP_EDGE_LSHIFT_RIGHT                88
/// \brief This parameter controls the Gain applied to the horizontal edge detected.
/// \param value 0 = no shift, 1 = 1 bit shift, 2 = 2 bit shift, 3 = 3 bit shift
///
#define Q_NVPP_CMP_EDGE_LSHIFT_VERT                 89
/// \brief This parameter controls the Gain applied to the right edge detected.
/// \param value 0 = no shift, 1 = 1 bit shift, 2 = 2 bit shift, 3 = 3 bit shift
///
#define Q_NVPP_CMP_EDGE_LSHIFT_HORZ                 90
/// \brief This parameter is utilized to choose between 4 tap and 3tap or 6tap filter for decimation.
/// \param value 0 = Internal 4 tap filters are used for 2:1 and 4:1 decimation, 1 = 2:1 prescaler used first and then 3tap filter used for final scaling and 6tap filter used for 4:1 decimation.
///
#define Q_NVPP_CMP_PMEDS_FILTER_SEL                 91
/// \brief This parameter helps in bypassing the vertical filtering operation for decimation.
/// \param value 0 = Enable filtering, 1 = Scaling done without filtering just decimation.
///
#define Q_NVPP_CMP_PMEDS_FILTER_V_BYPASS            92
/// \brief This parameter helps in bypassing the horizontal filtering operation for decimation.
/// \param value 0 = Enable filtering, 1 = Scaling done without filtering just decimation.
///
#define Q_NVPP_CMP_PMEDS_FILTER_H_BYPASS            93
/// \brief This parameter sets the number of ticks (duration of each frame) of the video input.
/// \param value Integer number.
///
#define Q_NVPP_CMP_STC_PER_FRAME                    94
/// \brief This parameter sets the number of ticks (duration of each frame) of the video frames
///        at the output of the VPP. If VPP_CFG_STC_PER_OUTPUT_FRAME is twice VPP_CFG_STC_PER_FRAME
///        then the VPP has effectively halved the frame rate. The reverse is also true.
/// \param value Integer number.
///
#define Q_NVPP_CMP_STC_PER_OUTPUT_FRAME             95
/// \brief This parameter sets the VPP to wait for black frames and then skip them until a non
///        black frame. This is helpful when you want to repeat a capture sequence from a live
///        source as it will wait for the initial black frame and then start.
/// \param value 0 = OFF, 1 = ON.
///
#define Q_NVPP_CMP_BLACK_FRAME_SKIP                 96
/// \brief This parameter sets the VPP capture a maximum number of frames. This is helpful when
///        you want to repeat a capture sequence from a live source as it will control exactly
///        how many frames to capture.
/// \param value Integer number.
///
#define Q_NVPP_CMP_MAX_CAP_FRAMES                   97
/// \brief This parameter allows the VPP to work in a fast scale mode. Normally the VPP takes 2
///        clock cycles to process a pixel. In fast scale mode the VPP will take 1 clock cycle
///        to process a pixel. This mode is needed when the video input pixel rate is more than
///        half the VPP clock frequency.
/// \param value 0 = Normal mode, 1 = Fast scale mode.
///
#define Q_NVPP_CMP_FAST_SCALE_MODE                  98
/// \brief This parameter sets the VPP to increment clock ticks when a capture overflow occurs.
/// \param value 0 = Normal mode, 1 = Enable STC increment on capture overflow.
///
#define Q_NVPP_CMP_ENABLE_STC_INC_ON_OVERFLOW       99
/// \brief This parameter sets the number of lines per field/frame when running in interval capture
///        mode. It must be a multiple of 8 for interlaced video output and a multiple of 16 for
///        progressive video output.
/// \param value Integer number.
///
#define Q_NVPP_CMP_INTERVAL_INT_LINES              100
/// \brief This parameter overrides the internal calculation of the BW of the horizontal scaling filter. 
///         A value of 0 uses the internal calculation which is based on horizontal input to output ratio.
///         Bandwidth of the filter can be adjusted in steps of 0.05 starting from 0.05 going upto 1.0, 
///         using integer values going from 1 to 20. A value of 1 represents filter BW of 0.05,
///         2 represnts a BW of 0.10, while 20 represents a BW of 1.0.
/// \param value Integer number 0-20.
#define Q_NVPP_CMP_HORZ_LUMABW_OVERRIDE            101
/// \brief This parameter overrides the internal calculation of the Luma BW of the vertical filter to be used for
///         scaling. A value of 0 uses the internal calculation which is based on input to output ratio.
///         Bandwidth of the filter can be adjusted in steps of 0.05 starting from 0.05 going upto 1.0, 
///         using integer values going from 1 to 20. A value of 1 represents filter BW of 0.05,
///         2 represnts a BW of 0.10, while 20 represents a BW of 1.0.
/// \param value Integer number 0-20.
#define Q_NVPP_CMP_VERT_LUMABW_OVERRIDE            102
/// \brief This parameter overrides the internal calculation of the Chroma BW of the vertical filter to be used for
///         scaling. A value of 0 uses the internal calculation which is based on input to output ratio.
///         Bandwidth of the filter can be adjusted in steps of 0.05 starting from 0.05 going upto 1.0, 
///         using integer values going from 1 to 20. A value of 1 represents filter BW of 0.05,
///         2 represnts a BW of 0.10, while 20 represents a BW of 1.0.
/// \param value Integer number 0-20.
#define Q_NVPP_CMP_VERT_CHROMABW_OVERRIDE          103
/// \brief This parameter sets the VPP to use the PTS values from the upstream component instead
///        of recalculating it. This is used in transcoding applications. This option is only
///        valid for NVPP and not for LVPP.
/// \param value 0 = STC is recalculated, 1 = STC is passed through from upstream component.
#define Q_NVPP_CMP_PTS_PASSTHROUGH                 104
/// \brief This parameter sets the mode for the deinterlacer.
/// \param value 0 = OFF, 1 = 1 field mode, 2 = 2 field mode, 3 = 3 field mode.
#define Q_NVPP_CMP_DEINT_MODE                      105
/// \brief This parameter enables A/V synch between this VPP and audio input 0
/// \param value 0 = OFF, 1 = ON
#define Q_NVPP_CMP_AV_SYNCH_ENABLE                 106
/// \brief This parameter enables the scaler to use one field of the source picture and scales
///        it to a progressive frame. For example you would use it for 720x480i to 352x288p.
///        This feature improves performance with minor losses in scaling quality.
///        This feature only works for NVPP and not LVPP and only on interlaced source pictures.
/// \param value 0 = OFF (Both fields are used), 1 = Use top field, 2 = Use bottom field.
#define Q_NVPP_CMP_SCALE_FIELD_MODE                107
/// \brief This parameter sets how much padding is done for the PME downscale image and it should
///        match the encoder mode.
/// \param value 0 = Field Mode, 1 = Frame Mode.
#define Q_NVPP_CMP_PMEDS_FRAME_MODE                108
/// \brief This parameter sets the source format. For example this allows a frame to set as
///        progressive even though it was captured over an interlaced interface. This feature
///        is not implemented yet.
/// \param value.
#define Q_NVPP_CMP_SOURCE_FORMAT                   109
/// \brief This parameter sets the video input to drop frames before they are captured. For
///        fractional frame drops this should be used over using the STC_PER_OUTPUT_FRAME as
///        dropping frames before capture saves memory bandwidth. The STC_PER_FRAME and
///        STC_PER_OUTPUT_FRAME must be set with the decimation factor in mind. For example
///        to use this feature to go from 30fps to 15fps, INPUT_DECIMATION:STC_PER_FRAME:STC_PER_OUTPUT_FRAME
///        will change from 1:3000:3000 to 2:6000:6000.
/// \param value Integer number greater or equal to 1.
#define Q_NVPP_CMP_INPUT_DECIMATION                110
/// \brief This parameter sets the upper 32 bits of the starting PTS of video capture.
/// \param value.
#define Q_NVPP_CMP_START_PTS_HIGH                  111
/// \brief This parameter sets the lower 32 bits of the starting PTS of video capture.
/// \param value.
#define Q_NVPP_CMP_START_PTS_LOW                   112
/// \brief Add an offset to all video PTS values.  This has no effect on the audio PTS or
///        on A/V sync, video PTS for each captured frame simply has a higher value.  This
///        means that when the stream is played back, video will be delayed relative to
///        audio.  This can be used to correct for audio/video skew in the input.
/// \param Offset 32 bit PTS offset
#define Q_NVPP_CMP_STC_OFFSET                      113
/// \brief This parameter enables variable frame rate mode.  In this mode, lvpp accepts video with variable frame rate.
///  It stamps all incoming video frames with a real time clock.  Since the clock is real time and the stamping process
/// is a software based, the resulting video time stamps contain a small amount of jitter.
/// \param value 0 = disable, 1 = enable.
#define Q_NVPP_CMP_VARIABLE_FRAME_RATE             114
/// \brief This parameter enables the NTP clock. This clock will periodically reset the QMM's realtime clock from a 
///        queried time on the host. This is done to prevent time-drift. Must be used with variable frame rate enabled.
/// \param value 0 = disable, 1 = enable.
#define Q_NVPP_CMP_NTP_CLOCK_ENABLE                115

// Generated from mediaware.h VPP_REGION_CFG s/VPP_REGION_CFG/Q_NVPP_CMP_REGION/
/// \brief This parameter sets the VPP region similarity step size.
/// \param value Integer number 0 - 15.
///
#define Q_NVPP_CMP_REGION_SIM_STEP                   0
/// \brief This parameter sets the VPP region spatial smoothing filter.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_NVPP_CMP_REGION_SMOOTH_EN                  1
/// \brief This parameter sets the VPP region median filter mode.
/// \param value 0 = Median filter disabled.\n
///              1 = Horizontal 3-tap median filter.\n
///              2 = Vertical 3-tap median filter.\n
///              3 = Horizontal/Vertical 5-tap median filter.\n
///
#define Q_NVPP_CMP_REGION_MED_FILT                   2
/// \brief This parameter sets the VPP region temporal smoothing filter.
/// \param value 0 = Disable, 1 = Enable.
///
#define Q_NVPP_CMP_REGION_TFILT_EN                   3

// Generated from mediaware.h VPP_REGION_FEATURE_CFG s/VPP_REGION_FEATURE_CFG/Q_NVPP_CMP_REGION_FEATURE/
/// \brief This parameter sets the VPP region detection feature select.
/// \param value  0 = Disable feature selection.\n
///               1 = Luma low frequency average..\n
///               2 = Luma high frequency average.\n
///               3 = Luma vertical high frequency average. (for MBAFF)\n
///               4 = Luma motion average.\n
///               5 = Luma spatial edge average.\n
///               6 = Luma motion edge average.\n
///               7 = Chroma high frequency average.\n
///               8 = Color 0 Sum.\n
///               9 = Color 1 Sum.\n
///              10 = Color 2 Sum.\n
///              11 = Color 3 Sum.\n
///
#define Q_NVPP_CMP_REGION_FEATURE_SELECT             0
/// \brief This parameter sets the VPP region feature low threshold. Region will qualify if
///        greater than or equal to this threshold.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_REGION_FEATURE_LOW                1
/// \brief This parameter sets the VPP region feature high threshold. Region will qualify if
///        less than or equal to this threshold.
/// \param value Integer number 0 - 255.
///
#define Q_NVPP_CMP_REGION_FEATURE_HIGH               2

// Generated from mediaware.h DIS_CFG s/DIS_CFG/Q_NVPP_CMP_DIS/
/// \brief This parameter is used to enable Digital Image Stabilization.
/// \param value 0:Disable 1:Enable.
#define Q_NVPP_CMP_DIS_ENABLE                        0
/// \brief This parameter controls the minimum relative percentile value of luma motion for a macroblock in a  
///  frame required to classify a block for image stabilization. If relative luma motion percentile value for a macroblock is below this threshold, 
///  the macroblock will not be used for image stabilization.
/// \param value is [0-100].
#define Q_NVPP_CMP_DIS_LUMAMOT_LOW_PERCENT           1
/// \brief This parameter controls the maximum relative percentile value of luma motion for a macroblock in a  
///  frame required to classify a block for image stabilization. If relative luma motion percentile value for a macroblock is above this threshold, 
///  the macroblock will not be used for image stabilization.
/// \param value is [0-100].
#define Q_NVPP_CMP_DIS_LUMAMOT_HIGH_PERCENT          2
/// \brief This parameter controls the minimum relative percentile value of luma edge for a macroblock in a  
///  frame required to classify a block for image stabilization. If relative luma edge percentile value for a macroblock is below this threshold, 
///  the macroblock will not be used for image stabilization.
/// \param value is [0-100].
#define Q_NVPP_CMP_DIS_LUMAEDGE_LOW_PERCENT          3
/// \brief This parameter controls the maximum relative percentile value of luma edge for a macroblock in a  
///  frame required to classify a block for image stabilization. If relative luma edge percentile value for a macroblock is above this threshold, 
///  the macroblock will not be used for image stabilization.
/// \param value is [0-100].
#define Q_NVPP_CMP_DIS_LUMAEDGE_HIGH_PERCENT         4
/// \brief This parameter enables the computation of motion vectors on the downscaled image. For HD images
///  this value should be set to 1.
/// \param value 0:Disable 1:Enable.
#define Q_NVPP_CMP_DIS_USE_DOWNSCALED_MOTION_VECTOR   5

//////////////////////////////////////////////////////////////////////
//
// Configuration parameter values
//

/// \}
#endif
