/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


#ifndef __QBOX_HH
#define __QBOX_HH

/* QBOX declarations */
// QBox: Quarc specific Box

#define QBOX_VERSION_NUM 1

//////////////////////////////////////////////////////////////////////////////
// version: 0
//
// SDL
// aligned(8) class QBox extends FullBox('qbox', version = 0, boxflags) {
//     unsigned short sample_stream_type;
//     unsigned short sample_stream_id;
//     unsigned long sample_flags;
//     unsigned long sample_cts;
//     unsigned char sample_data[];
// }
//
// equivalent to 
// typedef struct {
//     unsigned long box_size;
//     unsigned long box_type; // "qbox"
//     unsigned long box_flags; // (version << 24 | boxflags)
//     unsigned short sample_stream_type;
//     unsigned short sample_stream_id;
//     unsigned long sample_flags;
//     unsigned long sample_cts;
//     unsigned char sample_data[];
// } QBox;
//
// version 0 does not use large box
//
// box_flags
// 31 - 24         23 - 0
// version         boxflags
//
// boxflags
// 0x01 sample_data present after box header.
// otherwise sample_data contain four bytes address and four byte size info for
// the actual data.
// 0x02 this is the last sample
// 0x04 next qbox is word aligned
// 0x08 audio only
// 0x10 video only
// 0x20 stuffing packet (i.e. no meaningful data included)
//
// sample_stream_type:
// 0x01 AAC audio. sample_data contain audio frame or configuration info.
// 0x02 H.264 video. sample_data contain video frame or configuration info.
// It consists of 4 bytes length, NAL pairs and possible padding of 0s at the 
// end to make sample size word aligned.
// 0x05 H.264 video. sample_data contain video slice or configuration info.
// 0x06 MP1 audio. sample_data contain audio frame.
// 0x09 G.711 audio. sample_data contains one audio frame.
//
// sample_stream_id:
// 0, 1, ... or just the stream type
//
// sample_flags:
// 0x01 configuration info. sample_data contain configuration info.
// 0x02 cts present. 90 kHz cts present.
// 0x04 sync point. ex. I frame.
// 0x08 disposable. ex. B frame.
// 0x10 mute. Sample is mute/black.
// 0x20 cts base increment. By 2^32.
// 0x40 QBoxMeta present before configuration info or sample data.
// 0x80 sample contain end of sequence NALU.
// 0x100 sample contain end of stream NALU.
// 0x200 qmed
// 0xFF000000 padding mask, sample_data contain paddings "in front".
// sample_size include padding.
//
//////////////////////////////////////////////////////////////////////////////

#define QBOX_FLAGS_SAMPLE_DATA_PRESENT 0x1
#define QBOX_FLAGS_LAST_SAMPLE 0x2
#define QBOX_FLAGS_PADDING_4 0x4
#define QBOX_FLAGS_AUDIO_ONLY 0x8
#define QBOX_FLAGS_VIDEO_ONLY 0x10
#define QBOX_FLAGS_STUFFING_PACKET 0x20

#define QBOX_SAMPLE_TYPE_AAC 0x1
#define QBOX_SAMPLE_TYPE_QAC 0x1
#define QBOX_SAMPLE_TYPE_H264 0x2
#define QBOX_SAMPLE_TYPE_QPCM 0x3
#define QBOX_SAMPLE_TYPE_DEBUG 0x4
#define QBOX_SAMPLE_TYPE_H264_SLICE 0x5
#define QBOX_SAMPLE_TYPE_QMA 0x6
#define QBOX_SAMPLE_TYPE_VIN_STATS_GLOBAL 0x7
#define QBOX_SAMPLE_TYPE_VIN_STATS_MB 0x8
#define QBOX_SAMPLE_TYPE_Q711 0x9
#define QBOX_SAMPLE_TYPE_Q722 0xa
#define QBOX_SAMPLE_TYPE_Q726 0xb
#define QBOX_SAMPLE_TYPE_Q728 0xc
#define QBOX_SAMPLE_TYPE_JPEG 0xd
#define QBOX_SAMPLE_TYPE_MPEG2_ELEMENTARY 0xe
#define QBOX_SAMPLE_TYPE_USER_METADATA 0xf
#define QBOX_SAMPLE_TYPE_MAX 0x10

#define QBOX_SAMPLE_FLAGS_CONFIGURATION_INFO 0x01
#define QBOX_SAMPLE_FLAGS_CTS_PRESENT 0x02
#define QBOX_SAMPLE_FLAGS_SYNC_POINT 0x04
#define QBOX_SAMPLE_FLAGS_DISPOSABLE 0x08
#define QBOX_SAMPLE_FLAGS_MUTE 0x10
#define QBOX_SAMPLE_FLAGS_BASE_CTS_INCREMENT 0x20
#define QBOX_SAMPLE_FLAGS_META_INFO 0x40
#define QBOX_SAMPLE_FLAGS_END_OF_SEQUENCE 0x80
#define QBOX_SAMPLE_FLAGS_END_OF_STREAM 0x100
#define QBOX_SAMPLE_FLAGS_QMED_PRESENT 0x200
#define QBOX_SAMPLE_FLAGS_PKT_HEADER_LOSS 0x400
#define QBOX_SAMPLE_FLAGS_PKT_LOSS 0x800
#define QBOX_SAMPLE_FLAGS_120HZ_CLOCK 0x1000
#define QBOX_SAMPLE_FLAGS_TS                0x1000
#define QBOX_SAMPLE_FLAGS_TS_FRAME_START    0x2000
#define QBOX_SAMPLE_FLAGS_PADDING_MASK 0xFF000000

#define QBOX_VERSION(box_flags)                                         \
    ((box_flags) >> 24)
#define QBOX_BOXFLAGS(box_flags)                                        \
    (((box_flags) << 8) >> 8)
#define QBOX_FLAGS(v, f)                                                \
    (((v) << 24) | (f))
#define QBOX_SAMPLE_PADDING(sample_flags)                               \
    (((sample_flags) & QBOX_SAMPLE_FLAGS_PADDING_MASK) >> 24)
#define QBOX_SAMPLE_FLAGS_PADDING(sample_flags, padding)                \
    ((sample_flags) | ((padding) << 24))

typedef struct
{
    unsigned long v : 8;
    unsigned long f : 24;
} QBoxFlag;

typedef struct
{
    unsigned long samplerate;
    unsigned long samplesize;
    unsigned long channels;
} QBoxMetaA;

typedef struct
{
    unsigned long width;
    unsigned long height;
    unsigned long gop;
    unsigned long frameticks;
} QBoxMetaV;

typedef union
{
    QBoxMetaA a;
    QBoxMetaV v;
} QBoxMeta;

typedef struct
{
    unsigned long addr;
    unsigned long size;
} QBoxSample;

//////////////////////////////////////////////////////////////////////////////
// version: 1
//
// 64 bits cts support

typedef struct
{
    unsigned long ctslow;
    unsigned long addr;
    unsigned long size;
} QBoxSample1;

#ifdef    __cplusplus
/* - since qbox is to be used in streaming, data are stored in big endian.
   the calss is endinness aware.
   - cast QBox to memory to read/write its member.
*/
class QBox
{
  public:
    unsigned long HeaderSize();
    unsigned long BoxSize();
    unsigned long BoxType();
    unsigned long BoxFlags();
    unsigned char Version();
    unsigned long Flags();
    unsigned short SampleStreamType();
    unsigned short SampleStreamId();
    unsigned long SampleFlags();
    unsigned long SampleCTS();
    uint64_t SampleCTS64();
    unsigned long SampleAddr();
    unsigned long SampleSize();
    void BoxSize(unsigned long size);
    void BoxType(unsigned long type);
    void BoxFlags(unsigned long flags);
    void Version(unsigned char v);
    void Flags(unsigned long f);
    void SampleStreamType(unsigned short type);
    void SampleStreamId(unsigned short id);
    void SampleFlags(unsigned long flags);
    void SampleCTS(unsigned long cts);
    void SampleCTS64(uint64_t cts);
    void SampleAddr(unsigned long addr);
    void SampleSize(unsigned long size);

    void Dump();

  protected:
    unsigned long mBoxSize;
    unsigned long mBoxType; // "qbox"
    union {
        unsigned long value; // (version << 24 | boxflags)
        QBoxFlag field;
    } mBoxFlags;
    unsigned short mSampleStreamType;
    unsigned short mSampleStreamId;
    unsigned long mSampleFlags;
    unsigned long mSampleCTS;
    union {
        unsigned char data[];
        QBoxSample info;
        QBoxSample1 info1;
    } mSample;
};

inline unsigned long str2int(const char* s) {
    return (s[0] << 24) | (s[1] << 16) | (s[2] << 8) | s[3];
}
inline void int2str(unsigned long i, char* s) {
    s[0] = (char)((i >> 24) & 0xFF); s[1] = (char)((i >> 16) & 0xFF); 
    s[2] = (char)((i >> 8) & 0xFF); s[3] = (char)(i & 0xFF); s[4] = 0;
}
// This is for version 0 only, obselete
inline unsigned long HeaderSize(unsigned long flags)
{
    return (flags & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
        sizeof(QBox) - sizeof(QBoxSample1) : sizeof(QBox)- sizeof(unsigned long);
}
inline unsigned long HeaderSize(unsigned long flags, unsigned long version)
{
    if (version == 0)
        return (flags & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            sizeof(QBox) - sizeof(QBoxSample1) : sizeof(QBox) - sizeof(unsigned long);
    else
        return (flags & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            sizeof(QBox) - sizeof(QBoxSample) : sizeof(QBox);
}

#ifdef _WIN32
#define LITTLE_ENDIAN 0
#define BIG_ENDIAN    1
#define __BYTE_ORDER LITTLE_ENDIAN
#else
#include <endian.h>
#endif

#define BE8(a) (a)
#if __BYTE_ORDER == BIG_ENDIAN
#define BE16(a) (a)
#define BE24(a) (a)
#define BE32(a) (a)
#define BE64(a) (a)
#else
#define BE16(a)                                                             \
    ((((a)>>8)&0xFF) |                                                      \
    (((a)<<8)&0xFF00))
#define BE24(a)                                                             \
    ((((a)>>16)&0xFF) |                                                     \
    ((a)&0xFF00) |                                                          \
    (((a)<<16)&0xFF0000))
#define BE32(a)                                                             \
    ((((a)>>24)&0xFF) |                                                     \
    (((a)>>8)&0xFF00) |                                                     \
    (((a)<<8)&0xFF0000) |                                                   \
    ((((a)<<24))&0xFF000000))
#define BE64(a)                                                             \
    (BE32(((a)>>32)&0xFFFFFFFF) |                                           \
    ((BE32((a)&0xFFFFFFFF)<<32)&0xFFFFFFFF00000000))
#endif

inline unsigned long QBox::HeaderSize()
{
    if (Version() == 0)
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            sizeof(QBox) - sizeof(QBoxSample1) : sizeof(QBox) - sizeof(unsigned long);
    else
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            sizeof(QBox) - sizeof(QBoxSample) : sizeof(QBox);
}
inline unsigned long QBox::BoxSize()
{
    return BE32(mBoxSize);
}
inline unsigned long QBox::BoxType()
{
    return BE32(mBoxType);
}
inline unsigned long QBox::BoxFlags()
{
    return BE32(mBoxFlags.value);
}
inline unsigned char QBox::Version()
{
    return BE8(mBoxFlags.field.v);
}
inline unsigned long QBox::Flags()
{
    return BE24(mBoxFlags.field.f);
}
inline unsigned short QBox::SampleStreamType()
{
    return BE16(mSampleStreamType);
}
inline unsigned short QBox::SampleStreamId()
{
    return BE16(mSampleStreamId);
}
inline unsigned long QBox::SampleFlags()
{
    return BE32(mSampleFlags);
}
inline unsigned long QBox::SampleCTS()
{
    if (Version() == 0)
        return BE32(mSampleCTS);
    else
        return BE32(mSample.info1.ctslow);
}
inline uint64_t QBox::SampleCTS64()
{
    if (Version() == 0)
        return BE32(mSampleCTS);
    else
        return ((uint64_t)BE32(mSampleCTS) << 32) | BE32(mSample.info1.ctslow);
}
inline unsigned long QBox::SampleAddr()
{
    if (Version() == 0)
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            (unsigned long)mSample.data : BE32(mSample.info.addr); 
    else
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            (unsigned long)(mSample.data + 4) : BE32(mSample.info1.addr); 
}
inline unsigned long QBox::SampleSize()
{
    if (Version() == 0)
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ?
            BoxSize() - sizeof(QBox) + sizeof(QBoxSample1) : BE32(mSample.info.size);
    else
        return (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) ? 
            BoxSize() - sizeof(QBox) + sizeof(QBoxSample) : BE32(mSample.info1.size);
}
inline void QBox::BoxSize(unsigned long size)
{
    mBoxSize = BE32(size);
}
inline void QBox::BoxType(unsigned long type)
{
    mBoxType = BE32(type);
}
inline void QBox::BoxFlags(unsigned long flags)
{
    mBoxFlags.value = BE32(flags);
    if (!(BoxFlags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT)) {
        BoxSize(sizeof(QBox));
    }
}
inline void QBox::Version(unsigned char v)
{
    mBoxFlags.field.v = BE8(v);
}
inline void QBox::Flags(unsigned long f)
{
    mBoxFlags.field.f = BE24(f);
    if (!(BoxFlags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT)) {
        BoxSize(sizeof(QBox));
    }
}
inline void QBox::SampleStreamType(unsigned short type)
{
    mSampleStreamType = BE16(type);
}
inline void QBox::SampleStreamId(unsigned short id)
{
    mSampleStreamId = BE16(id);
}
inline void QBox::SampleFlags(unsigned long flags)
{
    mSampleFlags = BE32(flags);
}
inline void QBox::SampleCTS(unsigned long cts)
{
    if (Version() == 0)
        mSampleCTS = BE32(cts);
    else {
        mSampleCTS = 0;
        mSample.info1.ctslow = BE32(cts);
    }
}
inline void QBox::SampleCTS64(uint64_t cts)
{
    if (Version() == 0)
        mSampleCTS = BE32((unsigned long)(cts & 0xFFFFFFFF));
    else {
        mSampleCTS = BE32((unsigned long)(cts >> 32));
        mSample.info1.ctslow = BE32((unsigned long)(cts & 0xFFFFFFFF));
    }
}
inline void QBox::SampleAddr(unsigned long addr)
{
    if (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) {
        fprintf(stderr, "sample should store in addr 0x%x\n", (unsigned int)SampleAddr());
    }
    else {
        if (Version() == 0)
            mSample.info.addr = BE32(addr);
        else
            mSample.info1.addr = BE32(addr);
    }
}
inline void QBox::SampleSize(unsigned long size)
{
    if (Flags() & QBOX_FLAGS_SAMPLE_DATA_PRESENT) {
        if (Version() == 0)
            BoxSize(sizeof(QBox) - sizeof(QBoxSample1) + size);
        else
            BoxSize(sizeof(QBox) - sizeof(QBoxSample) + size);
    }
    else {
        if (Version() == 0)
            mSample.info.size = BE32(size);
        else
            mSample.info1.size = BE32(size);
    }
}
inline void QBox::Dump()
{
#if 1
    char type[5] = {0};
    int2str(BoxType(), type);
    fprintf(stdout, "Box: Size %d, Header Size %d, Type %s, Flags 0x%x(v %2d, f 0x%x) \nSample: Type 0x%x, Stream Id 0x%x, Flags 0x%x, CTS 0x%x-%08x, Addr 0x%x, Size %d\n", (int)BoxSize(), (int)HeaderSize(), type, (unsigned int)BoxFlags(), (int)Version(), (unsigned int)Flags(), (unsigned int)SampleStreamType(), (unsigned int)SampleStreamId(), (unsigned int)SampleFlags(), (unsigned int)(SampleCTS64() >> 32), (unsigned int)(SampleCTS64() & 0xFFFFFFFF), (unsigned int)SampleAddr(), (int)SampleSize());
#endif
}

#endif /* cplusplus */

#endif  /* __QBOX_HH */
/*******************************************************************************
* vi: set shiftwidth=4 tabstop=8 softtabstop=4 expandtab nosmarttab:
*******************************************************************************/
