/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2010 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
* Version: SDK7r26554
*
*******************************************************************************/


/// \defgroup AudioEncoderControlObject Audio Encoder
/// \{

/// \page audenc Audio Encoder
/// \latexonly
/// \chapter{Audio Encoder}
/// \input{audio_encoder}
/// \endlatexonly
///
/// \section avcencstatus Status Block
///
/// The Audio Encoder status block contains information related to state and performance.

#ifndef __QAUDENC_HH
#define __QAUDENC_HH

#include "qglobal.h"

//////////////////////////////////////////////////////////////////////
//
// Commands
//

/// \brief Begin recording audio from the source that the audio encoder is bound to.
///        An encoded bitstream will be generated from PCM samples, and the host will
///        receive bitstream block ready events containing the location of each audio frame.
#define Q_AUDENC_CMD_RECORD                             1

/// \brief This command puts the encoder into the IDLE state.  Generation of the encoded
///        bitstream stops.
#define Q_AUDENC_CMD_STOP                               2

/// \brief Gracefully stop the encoding.  Audio input frames that have already started
///        capturing (up to two) will finish capturing, and the audio sample amplitude
///        will ramp down to avoid pops and clicks at the end of encoding.  These frames
///        will be encoded.  After all frames have been encoded and bitstream block
///        ready events sent to the host, a flush event is sent to the host.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AUDENC_CMD_FLUSH                              3

/// \brief Signal the encoder that the host has finished reading a bitstream block, so
///        the encoder may reuse that space for future encoding.  The host will receive
///        bitstream block ready events specifying the location of the encoded bitstream
///        in memory.  The encoded bitstream is stored in a ring buffer, so the
///        host does not have to sent bitstream block done for every bitstream block
///        ready.  Instead, sending bitstream block done for a specific bitstream block
///        releases that block and ALL previously sent blocks.  For example, it is possible
///        to send bitstream block done only for every fifth block.
/// \param address Address field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
/// \param offset Offset field from the bitstream block ready event, specifying the block
///        (and all previous blocks) that the host is done processing
/// \return Code:  1 = Success, 0 = Failure
#define Q_AUDENC_CMD_BITSTREAM_BLOCK_DONE               4

/// \brief Specify the input source for the audio encoder.  The encoder must be bound to the
///        SAIN (Stereo Audio Input) component to capture live audio.  The encoder must be
///        bound to an input before recording is started.
/// \param objID Object ID of the component the audio encoder should bind to.
/// \param portType Source object port to bind to (defined in source object API)
/// \return Code:  1 = Success, 0 = Failure
#define Q_AUDENC_CMD_BIND_AUDIO_INPUT                   5

/// \brief Unbind the encoder from its audio input source.  Unbind must be called while the
///        encoder is idle (Stop command has been sent).  This allows the encoder to be bound
///        to a different input, either a StereoAudioInput object or an AudioSplitter object.
/// \param handle Object ID of the producer object
/// \param port Port ID of the producer object
/// \return Code:  1 = Success, 0 = Failure

#define Q_AUDENC_CMD_UNBIND_AUDIO_INPUT                 6 

/// \brief Set a value in the encoder configuration array.  Encoder configuration is
///        stored in double buffered arrays of 32 bit parameters, listed in the Q_AUDENC_CMP_
///        defines.  This API sets entries in the inactive array.  They will have no effect
///        until the Q_AUDENC_CMD_ACTIVATE command is sent.  This allows many parameters to
///        be set sequentially but take effect at the same time.
/// 
///        Parameters take effect at frame boundaries, never while a frame is being encoded.
///        Many parameters can be set on the fly, but most should be set before encoding begins.
/// \param param Parameter defined in Q_AUDENC_CMP_ list.
/// \param value 32 bit value
/// \return Code:  1 = Success, 0 = Failure
#define Q_AUDENC_CMD_SET_PARAM                          7

/// \brief Activate any changed settings in the encoder configuration array.  Activate takes
///        effect between frames, and the pending configuration array (modified by any set param
///        calls since the last activate) is copied into the active configuration array.
/// \return Code:  1 = Success, 0 = Failure
#define Q_AUDENC_CMD_ACTIVATE                           8 

/// \brief This command sets the stream ID of the encoded 
/// stream
#define Q_AUDENC_CMD_SET_STREAM_ID                      9

/// \brief set the number of blocks reported in each bitstream block ready event. bitstream
///        block ready event will wait until that many QBoxes of data have been encoded 
///        before sending them all in one event.
/// \param value blocks per event (1 to 5)
#define Q_AUDENC_CMD_SET_ENC_TRANSFER_SIZE              10 


/// \brief undocumented
#define Q_AUDENC_MAX_CMD                                11


///////////////////////////////////////////////////////////////////////
//
// Command Payloads
// 

typedef struct
{
    int objID;
    int portType;
} STRUCT_Q_AUDENC_CMD_BIND_AUDIO_INPUT;

typedef struct
{
    unsigned int address;
    unsigned int size;
} STRUCT_Q_AUDENC_CMD_BITSTREAM_BLOCK_DONE;

typedef struct
{
    unsigned int param;
    int value;
} STRUCT_Q_AUDENC_CMD_SET_PARAM;

typedef struct
{
    unsigned int streamID;
} STRUCT_Q_AUDENC_CMD_SET_STREAM_ID;

typedef struct
{
    int value;
} STRUCT_Q_AUDENC_CMD_SET_ENC_TRANSFER_SIZE;

//////////////////////////////////////////////////////////////////////
//
// Command Parameter Values
//


//////////////////////////////////////////////////////////////////////
//
// Events
//

#define AUDENC_EVENT(ID) ((AUDENC_CTRLOBJ_TYPE << 16) | ID)

/// \brief The bitstream block ready event is used to notify the host that new encoded
///        bitstream data is available to be read.  Events are generated for every 1-5
///        encoded blocks (configurable using Q_AUDENC_CMD_SET_AUDIO_ENC_TRANSFER_SIZE).
///        Each block contains a single QBox which normally contains all NALUs for one
///        encoded video frame.
/// \param streamID Stream ID of the encoder which generated the events.
/// \param numFrames Number of frames (bitstream blocks) in this event
/// \param frameNAddr Start address of frame N in codec memory
/// \param frameNSize Size of frame N, in bytes
#define Q_AUDENC_EV_BITSTREAM_BLOCK_READY       AUDENC_EVENT(1)

#define Q_AUDENC_EV_RESERVED0                   AUDENC_EVENT(2)

/// \brief A Flush operation on the encoder has completed.  Flush is started by sending
///        the encoder flush command, or sending a Flush command to the LVPP or NVPP which
///        the encoder is bound to.  The event is sent once the encoder has finished
///        encoding any remaining frames, and bitstream block ready events have been sent
///        to the host for all of those frames.
/// \param streamID Stream ID of the encoder which is flushed.
#define Q_AUDENC_EV_BITSTREAM_FLUSHED           AUDENC_EVENT(3)

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
    unsigned int        numFrames;
    unsigned int        frame0Addr;
    unsigned int        frame0Size;
    unsigned int        frame1Addr;
    unsigned int        frame1Size;
    unsigned int        frame2Addr;
    unsigned int        frame2Size;
    unsigned int        frame3Addr;
    unsigned int        frame3Size;
    unsigned int        frame4Addr;
    unsigned int        frame4Size;
} STRUCT_Q_AUDENC_EV_BITSTREAM_BLOCK_READY;

typedef struct
{
    CONTROLOBJECT_ID    controlObjectId;
    EVENT_ID            eventId;
    unsigned int        timestamp;
    unsigned int        streamID;
} STRUCT_Q_AUDENC_EV_BITSTREAM_FLUSHED;

//////////////////////////////////////////////////////////////////////
//
// Single buffered Configuration parameters

/// \brief When set the encoder will acknowledge its own bitstream blocks
/// which removes the need for the host to send BITSTREAM_BLOCK_DONE 
/// commands.  Note that this means the encoder can overwrite un-read
/// data and the host can receive a corrupted bitstream
#define Q_AUDENC_CFG_BBLOCK_SELF_ACK                  1


//////////////////////////////////////////////////////////////////////
//
// Double Configuration parameters
//
// define audio encoder's parameters 

/// /brief Number of channels in the audio samples to encoded.  Set to 2 for stereo, 1 for mono.
/// For AAC-LC, ER-AAC-LD, MPEG 1 and 2 layer 2 it can be mono or stereo
/// but for G.711, G.722, G.726, G.728 it should be mono.
#define Q_AUDENC_CMP_NUM_CHANNELS                    1 
/// \brief Sampling rate of the audio samples to be encoded.
/// Some codecs require specific sample rates while for others it 
/// is flexible. For example : 
/// - AAC-LC ,  MPEG 1 and 2 layer 2 : flexible
/// - ER-AAC-LD - flexible, but the minimum available is 22.05Khz
/// - G.711 - 8 Khz
/// - G.722 - 16 Khz
/// - G.726 - 8 Khz
/// - G.728 - 8 Khz
#define Q_AUDENC_CMP_SAMPLE_RATE                     2 
/// \brief Sample size (in bits) of the audio samples to be encoded.
#define Q_AUDENC_CMP_SAMPLE_SIZE                     3 
/// \brief Select bitrate of the audio encode, in bits.
/// For some codecs there is a specific set of choices, for example
/// AAC-LC - flexible
/// ER-AAC-LD - flexible
/// MPEG-1 layer 2 which supports sampling frequencies of 32, 44.1 and 48 Khz
/// the minimum bitrate available is 32000 bits/sec.
/// MPEG-2 layer 2 which supports sampling frequencies of 16, 22.05 and 24 Khz
/// the minimum bitrate available is 8000 bits/sec.
/// G.711 - flexible
/// G.722 - 48000, 56000 or 64000 bits/sec.
/// G.726 - 16000, 24000, 32000 or 40000 bits/sec.
#define Q_AUDENC_CMP_BIT_RATE                        4 
/// \brief Select the law to use for the Q711 decoder.
/// \param law Q_AUDENC_CFP_Q711_A_LAW or Q_AUDENC_CFP_Q711_U_LAW
#define Q_AUDENC_CMP_Q711_LAW                        5
/// \brief set AAC audio-object-type
/// \param 2 - AAC-LC; 23 - ER-AAC-LD
#define Q_AUDENC_CMP_QAC_AUDIO_OBJECT_TYPE           6
/// \brief this parameter is reserved
#define Q_AUDENC_CMP_QAC_BITRATE_MODE                7
/// \brief Select the endianness to use with the PCM encoder/decoder.
#define Q_AUDENC_CMP_QPCM_ENDIANNESS                 8
/// \brief Enable TS stream
#define Q_AUDENC_CMP_TS_ENABLE                       9
/// \brief set TS stream id
#define Q_AUDENC_CMP_TS_STREAM_ID                    10
/// \brief Enable TS PCR master
#define Q_AUDENC_CMP_TS_PCR_MASTER                   11
/// \brief Enable audio only TS stream
#define Q_AUDENC_CMP_TS_AUDIO_ONLY                   12



//////////////////////////////////////////////////////////////////////
//
// Configuration parameter values
//
// Generated from mediaware.h ME_AUDIO_CODEC_TYPE s/ME_AUDIO_CODEC/Q_AUDENC_CFP_CODEC/
#define Q_AUDENC_CFP_CODEC_NONE                      0
#define Q_AUDENC_CFP_CODEC_QAC                       1
#define Q_AUDENC_CFP_CODEC_QPCM                      2
#define Q_AUDENC_CFP_CODEC_QMA                       3
#define Q_AUDENC_CFP_CODEC_Q711                      4
#define Q_AUDENC_CFP_CODEC_Q722                      5
#define Q_AUDENC_CFP_CODEC_Q726                      6
#define Q_AUDENC_CFP_CODEC_Q728                      7
#define Q_AUDENC_CFP_CODEC_ADTS                      8
#define Q_AUDENC_CFP_CODEC_NUM_TYPES                 9

/// \brief Configures the encoder for mono encoding
#define Q_AUDENC_CFP_CHANNEL_MONO                    1
/// \brief Configures the encoder for stereo encoding
#define Q_AUDENC_CFP_CHANNEL_STEREO                  2

/// \brief Configures the encoder for A-Law G.711 encoding
#define Q_AUDENC_CFP_Q711_A_LAW                      1 
/// \brief Configures the encoder for U-Law G.711 encoding
#define Q_AUDENC_CFP_Q711_U_LAW                      2

/// \brief Configures the pcm encoder for little-endian
#define Q_AUDENC_CFP_QPCM_LITTLE_ENDIAN              1
/// \brief Configures the pcm encoder for big-endian
#define Q_AUDENC_CFP_QPCM_BIG_ENDIAN                 2

//////////////////////////////////////////////////////////////////////
//
// Status block
//

typedef struct
{
#ifdef _WIN32
    int dummy;
#endif
} AUDENC_STATUS;

/// \}
#endif
