#!/bin/sh

CFGVERSION=1
EEPROM="/sys/bus/spi/devices/spi0.0/eeprom"
EEPROM_SIZE_KB=32
ID=0
MAC_ADDR=
IP_ADDR=
NAME=
IMAGE=

usage()
{
	echo 
	echo "USAGE: init_eeprom.sh <options>"
	echo "where options are:"
	echo "	--bootloader <image_name>  - name if bootloader to write to eeprom"
	echo "	  if bootloader is not given, only host config data will be written"
	echo "	--eeprom <eeprom> - path to eeprom, default is /sys/bus/spi/devices/spi0.0/eeprom"
	echo "	--ip - static ip address(if required)"
	echo "	--hostname - hostname of board"
	echo "	--mac - mac address of board"
	exit
}


if [ $# -eq 0 ]
then
	usage
fi

while [ $# -ne 0 ]
do
	if [ $1 == "--eeprom" ]
	then
		EEPROM=$2
		echo "ERROR: Unrecognized option"
		shift;shift;
	elif [ $1 == "--size" ]
	then
		EEPROM_SIZE_KB=$2
		shift;shift
	elif [ $1 == "--id" ]
	then
		ID=$2
		shift;shift
	elif [ $1 == "--mac" ]
	then
		MAC_ADDR=$2
		shift;shift
	elif [ $1 == "--ip" ]
	then
		IP_ADDR=$2
		shift;shift

	elif [ $1 == "--hostname" ]
	then
		NAME=$2
		shift;shift
	elif [ $1 == "--bootloader" ]
	then
		IMAGE=$2
		shift;shift
	elif [ $1 == "--help" -o $1 == "-h" ]
	then
		usage
	else
		echo "ERROR: Unrecognized option"
		usage
	fi
done

source /etc/sysconfig/config

if [ "X${MAC_ADDR}" == "X" ]
then
	if [ "X${CONFIG_NETWORK_MAC_INCONFIG}" == "Xy" ]
	then
		MAC_ADDR="${CONFIG_NETWORK_MACADDR}"
	else
		if [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmerlinbup" ]
		then
			if [ ${ID} -eq 0 ]
			then
				echo "ERROR:  ID cannot equal 0 for merlinbup boards"
				exit
			elif [  ${ID} -gt 127 ]
			then
				echo "ERROR:  ID cannot be bigger than 127 for EVP boards"
				exit
			else
				MAC_ADDR="00:18:80:80:03:$(printf "%02x" $((ID+0x80)))";
			fi
		elif [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmg3500evp" ]
		then
			if [ ${ID} -eq 0 ]
			then
				echo "ERROR:  ID cannot equal 0 for EVP boards"
				exit
			elif [  ${ID} -gt 4095 ]
			then
				echo "ERROR:  ID cannot be bigger than 4095 for EVP boards"
				exit
			elif [  ${ID} -lt 255  ]
			then
				MAC_ADDR="00:18:80:01:30:$(printf "%02x" $((ID)))";
			else
				ID_HIGH=$((${ID}&0xf00))
				let "ID_HIGH >>= 8"
				ID_LOW=$((${ID}&0xff))
				MAC_ADDR="00:18:80:01:3$(printf "%01x" $((ID_HIGH))):$(printf "%02x" $((ID_LOW)))"
			fi
		else
		    # customer boards, no default way to deterime MAC based on ID
			echo "ERROR: No default MAC for board type ${CONFIG_BOARD_TYPE_NAME}"
			exit
		fi
	fi
fi

if [ "X${IP_ADDR}" == "X" ]
then
	if [ "X${CONFIG_NETWORK_IPADDR_STATIC}" == "Xy" ]
	then
		IP_ADDR="${CONFIG_NETWORK_IPADDR}"
	else
		# just some bogus IP for a place holder in dynamic and custom
		IP_ADDR="0.0.0.0"
	fi
fi

if [ "X${NAME}" == "X" ]
then
	if [ "X${CONFIG_HOSTNAME_INCONFIG}" == "Xy" ]
	then
		NAME="${CONFIG_HOSTNAME}"
	else
		if [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmerlinbup" ]
		then
			if [ ${ID} -eq 0 ]
			then
				echo "ERROR:  ID cannot equal 0 for merlinbup boards"
				exit
			else
				NAME="merlin$(printf "%d" $((ID)))";
			fi
		elif [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmg3500evp" ]
		then
			if [ ${ID} -eq 0 ]
			then
				echo "ERROR:  ID cannot equal 0 for EVP boards"
				exit
			else
				NAME="mg3500evp$(printf "%d" $((ID)))";
			fi
		else
		    # customer boards, no default way to deterime MAC based on ID
			echo "ERROR: No default name for board type ${CONFIG_BOARD_TYPE_NAME}"
			exit
		fi

	fi
fi

echo
echo "SPI Data"
echo "--------"
if [ $ID -ne 0 ]
then
	echo "ID: ${ID}"
fi
echo "Hostname: ${NAME}"
echo "MAC Address: ${MAC_ADDR}"
if [ "X${IP_ADDR}" != "X0.0.0.0" ]
then
	echo "IP Address: ${IP_ADDR}"
fi
if [ "X${IMAGE}" != "X" ]
then 
	echo "Bootloader image: ${IMAGE}"
else
	echo "Bootloader image: none provided, will not be written"
fi
echo 

CHOICE=
echo -ne "Is this correct[y/N]? "
read CHOICE

if [ "X$CHOICE" != "XY" -a "X$CHOICE" != "Xy" ]
then
	echo "Aborting...."
	exit
fi

if [ "X${IMAGE}" != "X" ]
then
	if [ ! -e ${IMAGE} ]
	then
		echo "ERROR: bootloader image ${IMAGE} does not exist"
		exit
	fi
fi

echo "Erasing eeprom, please wait..."
dd if=/dev/zero of=${EEPROM} count=$((EEPROM_SIZE_KB)) bs=1024 > /dev/null 2>&1

if [ "X${IMAGE}" != "X" ]
then
	cp ${IMAGE} /tmp/boot.bin
	# pad out the bootloader image to 12K
	size=`ls -l /tmp/boot.bin |awk '{print $5}'`
	pad=$((12 * 1024))
	pad=$(($pad - $size))
else
	# if no bootloader, pad out 12K anyway, this will allow tftp
	# bootloaders not running from eeprom to find the CFG info(mac)
	pad=$((12 * 1024))
fi
dd if=/dev/zero count=1 bs=$pad >> /tmp/boot.bin
	
echo -ne "\
\nCFG$(printf "%02d" $((CFGVERSION)))\n\
ethaddr=${MAC_ADDR}\n\
hostname=${NAME}\n\
ipaddr=${IP_ADDR}\n\
" >> /tmp/boot.bin

echo -ne "Programming eeprom, please wait some more..."
dd if=/tmp/boot.bin of=${EEPROM} bs=1024 
rm -f /tmp/boot.bin
	
echo " done"

exit 0;
