/*******************************************************************************
 * cryptodev.h
 *
 * Linux CryptoAPI user space interface module
 *
 * Copyright (c) 2008 Shasi Pulijala <spulijala@amcc.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *
 * Detail Description:
 * This file defines ioctl structures for the Linux CryptoAPI interface. It
 * provides user space applications accesss into the Linux CryptoAPI
 * functionalities.
 *
 *******************************************************************************
 */
#ifndef __CRYPTODEV_H__
#define __CRYPTODEV_H__

#include <linux/types.h>

/* Crypto and Hash Algorithms */

#define CRYPTO_ALGORITHM_MIN		1
#define CRYPTO_DES_CBC			1
#define CRYPTO_3DES_CBC		2
#define CRYPTO_BLF_CBC			3
#define CRYPTO_CAST_CBC		4
#define CRYPTO_SKIPJACK_CBC		5
#define CRYPTO_MD5_HMAC		6
#define CRYPTO_SHA1_HMAC		7
#define CRYPTO_RIPEMD160_HMAC	8
#define CRYPTO_MD5_KPDK		9
#define CRYPTO_SHA1_KPDK		10
#define CRYPTO_RIJNDAEL128_CBC	11 /* 128 bit blocksize */
#define CRYPTO_AES_CBC			11 /* 128 bit blocksize  */
#define CRYPTO_ARC4			12
#define CRYPTO_MD5			13
#define CRYPTO_SHA1			14
#define CRYPTO_NULL_HMAC		15
#define CRYPTO_NULL_CBC		16
#define CRYPTO_DEFLATE_COMP		17 /* Deflate compression algorithm */
#define CRYPTO_SHA2_256_HMAC		18
#define CRYPTO_SHA2_384_HMAC		19
#define CRYPTO_SHA2_512_HMAC		20
#define CRYPTO_CAMELLIA_CBC		21
#define CRYPTO_SHA2_256		22
#define CRYPTO_SHA2_384		23
#define CRYPTO_SHA2_512		24
#define CRYPTO_RIPEMD160		25
#define CRYPTO_SHA2_224                 26
#define CRYPTO_AES_ECB                  27
#define CRYPTO_AES_CTR                  28
#define CRYPTO_AES_CCM                  29
#define CRYPTO_AES_GCM			30
#define CRYPTO_ALGORITHM_MAX		31 /* Keep last */

/**
 * @struct session_op
 * @brief ioctl parameter to create a session
 *
 *******************************************************************************
 */
struct session_op {
	__u32	cipher;				/* e.g. CRYPTO_DES_CBC */
	__u32	mac;				/* e.g. CRYPTO_MD5_HMAC */
	__u32	keylen;				/* cipher key */
	char	*key;
	int	mackeylen;			/* mac key length*/
	char	*mackey;			/* mackey(hmac)/authsize
						(ccm, gcm) */

	/* Return values */
	__u32 	ses;				/* session ID */
};

#define CRYPTO_MAX_DATA_LEN		64*1024
/**
 * @struct crypt_op
 * @brief ioctl parameter to request a crypt/decrypt operation against a session
 *
 *******************************************************************************
 */
struct crypt_op {
	__u32	ses;
	__u16	op;				/* i.e. COP_ENCRYPT */
#define COP_NONE	0
#define COP_ENCRYPT	1
#define COP_DECRYPT	2
	__u16	flags;
#define	COP_F_BATCH	0x0008			/* Batch op if possible */
	u_int		len;
	caddr_t		src, dst;		/* become sg inside kernel */
	caddr_t		mac;			/* must be big enough for
						chosen MAC */
	caddr_t		iv;
};


/*
 * Parameters for looking up a crypto driver/device by
 * device name or by id.  The latter are returned for
 * created sessions (crid) and completed key operations.
 */
struct crypt_find_op {
	int		crid;		/* driver id + flags */
	char		name[32];	/* device/driver name */
};

/* bignum parameter, in packed bytes, ... */
struct crparam {
	caddr_t		crp_p;
	u_int		crp_nbits;
};

#define CRK_MAXPARAM	8

struct crypt_kop {
	u_int		crk_op;		/* ie. CRK_MOD_EXP or other */
	u_int		crk_status;	/* return status */
	u_short		crk_iparams;	/* # of input parameters */
	u_short		crk_oparams;	/* # of output parameters */
	u_int		crk_crid;	/* NB: only used by CIOCKEY2 (rw) */
	struct crparam	crk_param[CRK_MAXPARAM];
};

#define CRK_ALGORITM_MIN        0
#define CRK_MOD_EXP             0
#define CRK_MOD_EXP_CRT         1
#define CRK_DSA_SIGN            2
#define CRK_DSA_VERIFY          3
#define CRK_DH_COMPUTE_KEY      4
#define CRK_ALGORITHM_MAX       4 /* Keep updated - see below */

#define CRF_MOD_EXP             (1 << CRK_MOD_EXP)
#define CRF_MOD_EXP_CRT         (1 << CRK_MOD_EXP_CRT)
#define CRF_DSA_SIGN            (1 << CRK_DSA_SIGN)
#define CRF_DSA_VERIFY          (1 << CRK_DSA_VERIFY)
#define CRF_DH_COMPUTE_KEY      (1 << CRK_DH_COMPUTE_KEY)

/* clone original filedescriptor */
#define CRIOGET         _IOWR('c', 100, unsigned int)

/* create crypto session */
#define CIOCGSESSION    _IOWR('c', 101, struct session_op)

/* finish crypto session */
#define CIOCFSESSION    _IOW('c', 102, unsigned int)

/* request encryption/decryptions of a given buffer */
#define CIOCCRYPT       _IOWR('c', 103, struct crypt_op)

/* ioctl()s for asym-crypto. Not yet supported. */
#define CIOCKEY         _IOWR('c', 104, void *)
#define CIOCASYMFEAT    _IOR('c', 105, unsigned int)

#endif
