/*
 * arch/arm/mach-falcon/include/mach/mobi_ioctrl.h
 *
 *  Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef MOBI_IOCTRL_H
#define MOBI_IOCTRL_H

#include <linux/types.h>

typedef enum {
	/* DS group 0 */
	IOCTRL_DS_PWM0         = 0,
	IOCTRL_DS_PWM1         = 1,
	IOCTRL_DS_PWM2         = 2,
	IOCTRL_DS_GPIO0        = 3,
	IOCTRL_DS_GPIO1        = 4,
	IOCTRL_DS_GPIO2        = 5,
	IOCTRL_DS_GPIO3        = 6,
	IOCTRL_DS_GPIO4        = 7,
	IOCTRL_DS_GPIO5        = 8,
	IOCTRL_DS_GPIO6        = 9,
	IOCTRL_DS_GPIO7        = 10,
	IOCTRL_DS_PCICLK       = 11,
	IOCTRL_DS_PCICTL       = 12,
	IOCTRL_DS_PCIAD        = 13,
	IOCTRL_DS_OSC          = 14,
	IOCTRL_DS_CLKIN        = 15,
	/* DS group 1 */     
	IOCTRL_DS_ETHTXCLK     = 16,     
	IOCTRL_DS_ETHMD        = 17,
	IOCTRL_DS_ETH          = 18,
	IOCTRL_DS_SDMMCCLK     = 19,
	IOCTRL_DS_SDMMC        = 20,
	IOCTRL_DS_NAND         = 21,
	IOCTRL_DS_SPI          = 22,
	IOCTRL_DS_I2C          = 23,
	IOCTRL_DS_UART0DATA    = 24,
	IOCTRL_DS_UART0CTL     = 25,
	IOCTRL_DS_UART1        = 26,
	IOCTRL_DS_DBGUART      = 27,
	IOCTRL_DS_BS0          = 28,
	IOCTRL_DS_BS1          = 29,
	IOCTRL_DS_BS2          = 30,
	IOCTRL_DS_RSVD0        = 31,
	/* DS group 2 */      
	IOCTRL_DS_VIDOUTINCLK0  = 32,
	IOCTRL_DS_VIDOUTINCLK1  = 33,
	IOCTRL_DS_VIDOUTOUTCLK0 = 34,
	IOCTRL_DS_VIDOUTHLFCLK0 = 35,
	IOCTRL_DS_VIDOUTOUTCLK1 = 36,
	IOCTRL_DS_VIDOUTHLFCLK1 = 37,
	IOCTRL_DS_V0I2C         = 38,
	IOCTRL_DS_V1I2C         = 39,
	IOCTRL_DS_V0SPI         = 40,
	IOCTRL_DS_V1SPI         = 41,
	IOCTRL_DS_AUDMCLK       = 42,
	IOCTRL_DS_AUD0          = 43,
	IOCTRL_DS_AUD1          = 44,
	IOCTRL_DS_AUD2          = 45,
	IOCTRL_DS_AUD1OSPDIF    = 46,
	IOCTRL_DS_AUD1ISPDIF    = 47,
	/* DS group 3 */      
	IOCTRL_DS_VID0DATA      = 48,    
	IOCTRL_DS_VID1DATA      = 49,
	IOCTRL_DS_VID2DATA      = 50,
	IOCTRL_DS_VID3DATA      = 51,
	IOCTRL_DS_VID4DATA      = 52,
	IOCTRL_DS_VID5DATA      = 53,
	IOCTRL_DS_VID6DATA      = 54,
	IOCTRL_DS_VID7DATA      = 55,
	IOCTRL_DS_VIDGENCLK0    = 56,
	IOCTRL_DS_VIDGENCLK1    = 57,
	IOCTRL_DS_VIDGENCLK2    = 58,
	IOCTRL_DS_VIDINCLK0     = 59,
	IOCTRL_DS_VIDINCLK1     = 60,
	IOCTRL_DS_VIDINCLK2     = 61,
	IOCTRL_DS_VIDINCLK3     = 62,
	IOCTRL_DS_END           = 63,
} ioctrl_ds_group_t;

typedef enum {
	IOCTRL_GPIO_MODE_AGPIO       = 0,
	IOCTRL_GPIO_MODE_I2C         = 1,
	IOCTRL_GPIO_MODE_V0I2C       = 2,
	IOCTRL_GPIO_MODE_V1I2C       = 3,
	IOCTRL_GPIO_MODE_V0SPI       = 4,
	IOCTRL_GPIO_MODE_V1SPI       = 5,
	IOCTRL_GPIO_MODE_SPI0        = 6,
	IOCTRL_GPIO_MODE_SPI1        = 7,
	IOCTRL_GPIO_MODE_BS0         = 8,
	IOCTRL_GPIO_MODE_BS1         = 9,
	IOCTRL_GPIO_MODE_BS2         = 10,
	IOCTRL_GPIO_MODE_UART0CTL    = 11,
	IOCTRL_GPIO_MODE_UART0DATA   = 12,
	IOCTRL_GPIO_MODE_UART1       = 13,
	IOCTRL_GPIO_MODE_DBGUART     = 14,
	IOCTRL_GPIO_MODE_AUD0        = 15,
	IOCTRL_GPIO_MODE_AUD1        = 16,
	IOCTRL_GPIO_MODE_AUD2        = 17,
	IOCTRL_GPIO_MODE_SD          = 18,
	IOCTRL_GPIO_MODE_PWM         = 19,
	IOCTRL_GPIO_MODE_PCI         = 20,
	IOCTRL_GPIO_MODE_END         = 21,
} ioctrl_gpio_mode_group_t;

typedef enum {
	IOCTRL_FUNC_DBG_UART	= 0,
	IOCTRL_FUNC_I2C_CFG	= 1,
	IOCTRL_FUNC_V23_MOSI	= 2,
	IOCTRL_FUNC_V23_MCLK	= 3,
	IOCTRL_FUNC_V01_MOSI	= 4,
	IOCTRL_FUNC_V01_MCLK	= 5,
	IOCTRL_FUNC_SPI_MOSI	= 6,
	IOCTRL_FUNC_SPI_MCLK	= 7,
	IOCTRL_FUNC_SPI_MSS0	= 8,
	IOCTRL_FUNC_SPI_MSS1	= 9,
	IOCTRL_FUNC_SPI_MASTER	= 10,
	IOCTRL_FUNC_END
} ioctrl_func_t;

typedef enum {
	/* FUNC_DBG_UART */
	IOCTRL_FUNC_ACTIVE_ARMDBG	= 0,
	IOCTRL_FUNC_ACTIVE_QMMDBG	= 1,

	/* FUNC_I2C_CFG */
	IOCTRL_FUNC_ACTIVE_I2CV01	= 0,
	IOCTRL_FUNC_ACTIVE_I2CV23	= 1,

	/* FUNC - V23_MOSI, V01_MOSI */
	IOCTRL_FUNC_ACTIVE_VXXMOSI	= 0,
	IOCTRL_FUNC_ACTIVE_SDA		= 1,

	/* FUNC - V23_MCLK, V01_MCLK */
	IOCTRL_FUNC_ACTIVE_VXXMCLK	= 0,
	IOCTRL_FUNC_ACTIVE_SCL		= 1,

	/* FUNC - SPI_MOSI */
	IOCTRL_FUNC_ACTIVE_SPIMOSI	= 0,
	IOCTRL_FUNC_ACTIVE_BS_DATA	= 1,

	/* FUNC - SPI_MCLK */
	IOCTRL_FUNC_ACTIVE_SPIMCLK	= 0,
	IOCTRL_FUNC_ACTIVE_BS_CLK	= 1,

	/* FUNC - SPI_MSS0 */
	IOCTRL_FUNC_ACTIVE_MSS0		 = 0,
	IOCTRL_FUNC_ACTIVE_BS_ENABLE = 1,

	/* FUNC - SPI_MSS1 */
	IOCTRL_FUNC_ACTIVE_MSS1		= 0,
	IOCTRL_FUNC_ACTIVE_BS_REQ	= 1,

	/* FUNC - SPI_MASTER */
	IOCTRL_FUNC_ACTIVE_SPISLAVE	 = 0,
	IOCTRL_FUNC_ACTIVE_SPIMASTER = 1,

} ioctrl_func_active_t;

typedef enum {
	IOCTRL_DRIVESTRENGTH_WEAKEST	= 0,
	IOCTRL_DRIVESTRENGTH_WEAK   	= 1,
	IOCTRL_DRIVESTRENGTH_STRONG 	= 2,
	IOCTRL_DRIVESTRENGTH_STRONGEST 	= 3,
	IOCTRL_DRIVESTRENGTH_DEFAULT	= 0,
} ioctrl_drivestrength_t;

typedef enum {
	IOCTRL_GPIO_CONTROL_FUNCTION	= 0,
	IOCTRL_GPIO_CONTROL_PULLUP	    = 1,
	IOCTRL_GPIO_CONTROL_PULLDOWN	= 2,
} ioctrl_gpiocontrol_t;

typedef enum {
	IOCTRL_GPIO_FUNCTION_PRIMARY	= 0,
	IOCTRL_GPIO_FUNCTION_GPIO	    = 1,
} ioctrl_gpioselect_t;

typedef enum {
	IOCTRL_GPIO_MODE_FUNCTIONAL	= 0,
	IOCTRL_GPIO_MODE_GPIO   	= 1,
} ioctrl_gpiomode_t;

typedef enum {
	IOCTRL_GPIO_PULLUP_DISABLED	= 0,
	IOCTRL_GPIO_PULLUP_ENABLED	= 1,
} ioctrl_gpiopullup_t;

typedef enum {
	IOCTRL_GPIO_PULLDOWN_DISABLED	= 0,
	IOCTRL_GPIO_PULLDOWN_ENABLED	= 1,
} ioctrl_gpiopulldown_t;

/* Macro stuff. */

struct ioctrl_macro_func_s {
	ioctrl_func_t 		func;
	ioctrl_func_active_t 	active;
};
#define IOCTRL_DECLARE_MACRO_FUNC(func_, active_) \
{ .func = func_, .active = active_ },
#define IOCTRL_DECLARE_MACRO_FUNC_END() { .func = IOCTRL_FUNC_END }

struct ioctrl_macro_gpio_s {
	const char		*name;
	uint32_t		gpio;
	ioctrl_gpioselect_t	sel;
	ioctrl_gpiopullup_t 	pu;
	ioctrl_gpiopulldown_t	pd;
};
#define IOCTRL_DECLARE_MACRO_GPIO_U(gpio_, sel_) \
{ .name = #gpio_, .gpio = gpio_, .sel = sel_, \
	.pu = IOCTRL_GPIO_PULLUP_ENABLED, .pd = IOCTRL_GPIO_PULLDOWN_DISABLED },
#define IOCTRL_DECLARE_MACRO_GPIO_D(gpio_, sel_) \
{ .name = #gpio_, .gpio = gpio_, .sel = sel_, \
	.pu = IOCTRL_GPIO_PULLUP_DISABLED, .pd = IOCTRL_GPIO_PULLDOWN_ENABLED },
#define IOCTRL_DECLARE_MACRO_GPIO_N(gpio_, sel_) \
{ .name = #gpio_, .gpio = gpio_, .sel = sel_, \
	.pu = IOCTRL_GPIO_PULLUP_DISABLED, .pd = IOCTRL_GPIO_PULLDOWN_DISABLED },
#define IOCTRL_DECLARE_MACRO_GPIO_END() { .gpio = -1 }

struct ioctrl_macro_s {
	const char 				*name;
	const struct 	ioctrl_macro_func_s	*func;
	const struct 	ioctrl_macro_gpio_s	*gpio;
};

/* API */

/**
 * \brief mobi_ioctrl_set_drivestrength:
 * 	Set the drivestrength for a given group.
 *
 * \param grp: defines group to access
 * \param drivestrength: new drivestrength for group
 *
 * \retval -EINVAL - if group is invalid
 * \retval Negetive value - qcc access failure
 * \retval Zero - upon success
 *
 *
 */
int mobi_ioctrl_set_drivestrength(ioctrl_ds_group_t grp, 
		ioctrl_drivestrength_t drivestrength);

/**
 * \brief mobi_ioctrl_gpio_pullup:
 * 	Used to control the pullup on a gpio pin
 *
 * \param gpio :  gpio number to change
 * \param enable :  enable or disable the pullup
 *
 * \retval -ENODEV - gpio does not exist
 * \retval -EINVAL - invalid enable
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Only one of pullup or pulldown can be enabled at the same time.
 * Enabling  one will disable the other. 
 */
int mobi_ioctrl_gpio_pullup(uint32_t gpio, uint8_t enable);

/**
 * \brief mobi_ioctrl_gpio_pulldown:
 * 	Used to control the pulldown on a gpio pin
 *
 * \param gpio :  gpio number to change
 * \param enable :  enable or disable the pullup
 *
 * \retval -ENODEV - gpio does not exist
 * \retval -EINVAL - invalid enable
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Only one of pullup or pulldown can be enabled at the same time.
 * Enabling  one will disable the other. 
 */
int mobi_ioctrl_gpio_pulldown(uint32_t gpio, uint8_t enable);

/**
 * \brief mobi_ioctrl_gpio_mode:
 * 	Used to control the mode of a function pin group
 *
 * \param grp :  functional pin group to change
 * \param mode : functional or gpio mode
 *
 * \retval -ENODEV - grp does not exist
 * \retval -EINVAL - invalid mode
 * \retval other negetive values - qcc read/write failed
 * \retval Zero - upon success
 *
 * \remark
 *   Some GPIOs can have alternative functions other than GPIO. The GPIO
 * have been put into functional groups which can be controlled via this
 * function.  If the group is in GPIO mode, these pins can be controlled
 * via the GPIO controller.  If set in FUNCTIONAL mode, these pins would
 * be controlled by a different controller, for exampel I2C
 */
int mobi_ioctrl_gpio_mode(ioctrl_gpio_mode_group_t grp, uint8_t mode);


int mobi_ioctrl_get_gpio_bank_size(uint8_t bank);
int mobi_ioctrl_get_gpio_bankindex(uint32_t gpio, uint8_t *bank);

void mobi_ioctrl_set_macros(const struct ioctrl_macro_s *macros);
const struct ioctrl_macro_s *mobi_ioctrl_get_macro(const char *name);
int mobi_ioctrl_apply_macro(const struct ioctrl_macro_s *macro);
int mobi_ioctrl_macro_is_active(const struct ioctrl_macro_s *macro);

int mobi_ioctrl_get_gpio(uint32_t gpio, ioctrl_gpiocontrol_t ctrl);

int mobi_ioctrl_get_function(ioctrl_func_t func);
int mobi_ioctrl_set_function(ioctrl_func_t func, ioctrl_func_active_t active);

#endif  /* MOBI_IOCTRL_H */
