/*
 * arch/arm/plat-mgx/include/mach/mobi_timer.h
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *  Copyright (C) 2009 Maxim IC
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef _PLAT_MGX_INCLUDE_MOBI_TIMER_H
#define _PLAT_MGX_INCLUDE_MOBI_TIMER_H

/* flags */
/* we have one timer that can trigger the FIQ, ask for it explicitly */
#define TIMER_USE_FIQ 	  	0x1
/* only countdown one time */
#define TIMER_COUNT_ONCE	0x2
/* this won't set any interrupt */
#define TIMER_NOINT		0x4

typedef int32_t timer_handle;

struct mobi_timeval {
	time_t          tv_hour;        /* hours        */
	time_t          tv_min;         /* minutes      */
	time_t          tv_sec;         /* seconds      */
	suseconds_t     tv_usec;        /* microseconds */
};

/**
 * \brief mobi_request_timer
 *  	Try to find a free timer.
 *
 * \param func	- callback function
 * \param flags	- option flags
 * 	- TIMER_COUNT_OUNCE - timer will only trigger once
 * 	- TIMER_USE_FIQ     - request timer the can trigger FIQ
 *	- TIMER_NOINT       - this will mask the interrupt (func is ignored and not called)
 *
 * \retval -ENODEV - if no free timer is found
 * \retval timer_handle  - id of timer
 *
 */
timer_handle mobi_request_timer(void (*func)(void), uint32_t flags);

/**
 * \brief mobi_set_timer
 *  	Program a timer for a give time interval.
 *
 * \param td - timer handle
 * \param tv - mobi_timeval with initial starting value define
 * \param flags - accepted flags
 *  		- TIMER_COUNT_ONCE: only trigger the counter one time
 *
 * \retval -ENODEV
 * 	- if invalid timer handle is provided
 * \retval -EINVAL
 * 	- if startvalue is great than 32 bit for 32 bit timer
 * 	- if td is a 32bit timer and try to set 64BIT_COUNTER flag
 * \retval -EIO
 * 	- if timer clock is disabled or rate cannot be determined
 *  \retval Zero
 *  	- on success
 *
 * \remark
 *  	Convert time into ticks which will be programmed into
 *  the LoadCount, enables interrupts and starts the timer.
 */
int32_t mobi_set_timer(timer_handle td, struct mobi_timeval tv, uint32_t flags);

/**
 * \brief mobi_restart_timer
 * 	If a process wants to restart the timer.
 *
 * \param td - timer_handle
 *
 * \retval -ENODEV - if invalid td is given
 * \retval Zero    - upon success
 *
 * \remark
 *   This applies to contineous and one-shot counters.  Timer will
 * be disabled, reloaded with start value and restarted.  Note,
 * the start value cannot be changed, must delete and create new
 * timer to do this.
 *
 */
int32_t mobi_restart_timer(timer_handle td);

/**
 * \brief mobi_get_timer
 * 	Return the current value of the timer.
 *
 * \param td  - timer handle
 * \param tv  - empty mobi_timeval struct
 *
 * \retval -ENODEV
 * 	- given an invalid timer handle
 * \retval -EACCESS
 * 	- if timer has not been enabled
 * \retval Zero
 * 	- up success and tv will contain current timer counts
 *
 * \remark
 *   Current value is returned in provided mobi_timeval struct.
 *  Do not have to be holding lock to read this value but
 *  timer must be enabled.
 */
int32_t mobi_get_timer(timer_handle td, struct mobi_timeval *tv);

/**
 * \brief mobi_del_timer
 * 	Disable and release selected timer
 *
 * \param td - timer handle
 *
 * \retval -ENODEV
 * 	- if td is invalid
 * \retval -EACCESS
 * 	- if caller is not the timer owner
 * \retval -EIO
 * 	- if timer clock is disabled or rate cannot be determined
 * \retval Zero
 * 	- upon success
 * 	
 * \remark
 *   Caller must be the current owner of the timer.  After successful
 * completion, timer is avail for use by other processes.
 *
 */
int32_t mobi_del_timer(timer_handle td);

/**
 * \brief mobi_request_tick
 * 	Set up a timer to be used a precise tick
 *
 * \param t	- none
 *
 * \retval	- -ENODEV if already requested
 *
 */
int mobi_request_tick(void);

/**
 * \brief mobi_get_tick
 * 	Get the current tick
 *
 * \retval	- the current tick value
 *
 */
unsigned long mobi_get_tick(void);

/**
 * \brief mobi_release_tick
 * 	Release a successfuly requested tick
 *
 */
void mobi_release_tick(void);

#endif
