/*
 * include/asm-arm/arch-merlin/mobi_dma.h
 *
 *  Copyright (C) 2006 Mobilygen Corp.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/** \file
 *  Header file for Mobilygen DMA API
 */

#ifndef __ASM_ARCH_MOBI_DMA_H
#define __ASM_ARCH_MOBI_DMA_H

#include <linux/dma-mapping.h>

/**
 * \brief the config flag type defines what is being 
 * configured.  src and dst are relevant only to the 
 * specified direction. xfer applies to the transfer 
 * as a whole and is not directly associated to a
 * direction.
 */
#define DMA_CONFIG_SRC			0x1
#define DMA_CONFIG_DST			0x2
#define DMA_CONFIG_XFER			0x4

/**
 * \brief the width of the node.  
 *  default: WIDTH_32
 *  config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_TRANSFER_WIDTH_8		0x00000001
#define MOBI_DMA_CONFIG_TRANSFER_WIDTH_16		0x00000002
#define MOBI_DMA_CONFIG_TRANSFER_WIDTH_32		0x00000004
#define MOBI_DMA_CONFIG_TRANSFER_WIDTH_MASK		0x00000007

/**
 * \brief for devices that support burst transactions, 
 * defines the number of data items to be written in 
 * a single burst.  
 *  default:  BURST_SIZE_8
 *  config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_BURST_SIZE_1		0x00000008
#define MOBI_DMA_CONFIG_BURST_SIZE_4		0x00000010
#define MOBI_DMA_CONFIG_BURST_SIZE_8		0x00000020
#define MOBI_DMA_CONFIG_BURST_SIZE_16		0x00000040
#define MOBI_DMA_CONFIG_BURST_SIZE_32		0x00000080
#define MOBI_DMA_CONFIG_BURST_SIZE_64		0x00000100
#define MOBI_DMA_CONFIG_BURST_SIZE_128		0x00000200
#define MOBI_DMA_CONFIG_BURST_SIZE_256		0x00000400
#define MOBI_DMA_CONFIG_BURST_SIZE_MASK		0x000007F8

/**
 * \brief the type of handshaking will be used.
 *  default:  HANDSHAKE_HW
 *  config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_HANDSHAKE_HW		0x00000800
#define MOBI_DMA_CONFIG_HANDSHAKE_SW		0x00001000
#define MOBI_DMA_CONFIG_HANDSHAKE_MASK		0x00001800

/** 
 *  \brief the dma engine will addjust the src or dst 
 *  addresses in the direction specified by this define.
 *  default:  ADDRADJ_INC
 *  config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_ADDRADJ_INC 		0x00002000
#define MOBI_DMA_CONFIG_ADDRADJ_DEC 		0x00004000
#define MOBI_DMA_CONFIG_ADDRADJ_NONE		0x00008000
#define MOBI_DMA_CONFIG_ADDRADJ_MASK		0x0000E000

/**
 * \brief the hardware is capable of doing endian swapping 
 *  during dma operations.  use this define to set 
 *  the endianess of a device.
 *  default:  ENDIAN_LE
 *  config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_ENDIAN_BE		0x00010000
#define MOBI_DMA_CONFIG_ENDIAN_LE		0x00020000
#define MOBI_DMA_CONFIG_ENDIAN_MASK		0x00030000

/** 
 * \brief used to define the width of the data, this is used
 *  to define how data is moved when swapping endianess.
 *  for example, two 32 bit devices with different 
 *  endianess and a data width of 8 would result in a 
 *  full reversal:  B0B1B2B3 becomes B3B2B1B0.
 *  default:  DATA_WIDTH_8
 *  config flag type: XFER
 */
#define MOBI_DMA_CONFIG_DATA_WIDTH_8		0x00040000
#define MOBI_DMA_CONFIG_DATA_WIDTH_16		0x00080000
#define MOBI_DMA_CONFIG_DATA_WIDTH_32		0x00100000
#define MOBI_DMA_CONFIG_DATA_WIDTH_MASK		0x001C0000

/** 
 * \brief  used to define the count and interval param for
 * scatter/gather list.  
 *   for a single block transfer, s/g can only be set on 
 * one node, src or dst.
 *   for a multiblock transfer, s/g can be set on either
 * the src or dst.  the param values are set via this config
 * flag and s/g is enable on a given block in a mobi_dma_list
 * by setting one of src or dst flags.  
 *   It is possible to have different count and interval values
 * for the src and dst.  However, these value *cannot* be different
 * if you set the REVERSE_DIR flag and and SG_ENABLE flag on a given
 * block.
 *   The count and interval must be set at config time in the
 * cfg_data argument
 * config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_SG_ENABLE			0x00200000
#define MOBI_DMA_CONFIG_SG_ENABLE_MASK		0x00200000

/** 
 * \brief  used to define if the start address should 
 * be reloaded for each node in a dma list.  
 *
 * config flag type: SRC & DST
 */
#define MOBI_DMA_CONFIG_RELOAD_START_ADDR			0x00400000
#define MOBI_DMA_CONFIG_RELOAD_START_ADDR_MASK		0x00400000

/** 
 * \brief  used to define if the start address should 
 * be reloaded for each node in a dma list.  
 *
 * config flag type: XFER 
 */
#define MOBI_DMA_CONFIG_MAX_BLK_SIZE		0x00800000
#define MOBI_DMA_CONFIG_MAX_BLK_SIZE_MASK	0x00800000

/**
 * \brief used to define the data for a config flag that 
 * requires data.  
 */

/**
 * \brief Defines params needed to set up a 
 * scatter/gather list for the DMA controller
 *
 * \remark
 *   count - defines the number of contiguous transfers
 * of size TRANSFER_WIDTH
 *   interval - specifies the address increment/decrement 
 * in multiples of TRANSFER_WIDTH
 *
 *   the direction that addresses are changed is determined
 * by the ADDRADJ setting
 *
 * for example, a transfer width of 32, count = 4, 
 * interval = 2 would result in 4 32-bit words transfer to
 * every other address.
 */
struct sg_params {
	uint32_t count;
	uint32_t interval;
};

struct mg_dma_config_data {
	/* flag MOBI_DMA_CONFIG_SG_ENABLE */
	struct sg_params sg_params;
	/* flag MOBI_DMA_CONFIG_MAX_BLK_SIZE */	
	int32_t max_blk_size;
};

/**
 * \brief return type for dma status
 */
typedef int32_t mobi_dma_status;

/** 
 * \brief  Defines the status of the current DMA transfer
 */
/* currently transferring data */
#define MOBI_DMA_STATUS_XFER_RUNNING 		0x00000001
/* dma controller has completed transfer */
#define MOBI_DMA_STATUS_XFER_IDLE 			0x00000002 
/* channel is enabled via mobi_dma_enable */
#define MOBI_DMA_STATUS_CHANNEL_ENABLED 	0x00000004 
/* channel is disabled via mobi_dma_disable */
#define MOBI_DMA_STATUS_CHANNEL_DISABLED 	0x00000008 
/* got an error */
#define MOBI_DMA_STATUS_ERROR 				0x00000010 

/**
 * \brief return type of a dma event
 */
typedef uint32_t mobi_dma_event;
/**
 * \brief Defines possible DMA events
 */
#define MOBI_DMA_EVENT_TRANSFER_COMPLETE			0x1
#define MOBI_DMA_EVENT_TRANSFER_ERROR				0x2
#define MOBI_DMA_EVENT_SRC_TRANSACTION_COMPLETE		0x4
#define MOBI_DMA_EVENT_DST_TRANSACTION_COMPLETE		0x8

/**
 * \brief  Handle for open DMA channel
 */
typedef int32_t mobi_dma_handle;

/* see mobi_dma_setup_mlist for details on this structure */
struct mobi_dma_list {
	uint32_t src_addr;
	uint32_t dst_addr;
	int32_t  size;	    	// size in bytes for this list entry
	uint32_t xfer_flags; 	// flags that apply to the general xfer of this entry.
	uint32_t src_flags; 	// flags that apply to the src node
	uint32_t dst_flags; 	// flags that apply to the dst node
	struct mobi_dma_list *next;
};

/* flags for mobi_dma_list */
#define MOBI_DMA_LIST_FLAG_NONE			0x0

/* 
 *  XFR_FLAGS apply to the general transfer of the given block in 
 *  the mobi_dma_list(in other words, not src or dst specific)
 */
/* reverse direction, ie. swap src and dst nodes 
 *   An example of how this flag would be used is dma(write) a 
 * series of commands and data, where the device will then 
 * perform some operation on the data.  The device may then
 * want to dma back some status or data.  This can be done in
 * single mobi_dma_list by setting the REVERSE_DIR flag.
 *   Do not reverse the nodes(src & dst) in the list entry.  The
 * dma API determines the node type(mem, device) at the start of 
 * the transfer.  It does not re-evaluate this information for
 * every entry in the mobi_dma_list which would be a very time 
 * consuming process.  The API will take the necessary step to 
 * swap the nodes.
 *  WARNING:  Pay attention to the direction when mapping a buffer
 * that will be used with the REVERSE flag.  For example, if the
 * src node is memory and the REVERSE flag is set, the src node 
 * will become a dst node and it must have it direction set 
 * correctly
 */
#define MOBI_DMA_LIST_XFER_FLAG_REVERSE_DIR		0x00000001

/* ************************************************************ */
/* Flags below this point only apply to the src and dst nodes and
 *  cannot be combined with the XFER flags above
 */
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_1		0x00000001
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_4		0x00000002
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_8		0x00000004
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_16		0x00000008
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_32		0x00000010
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_64		0x00000020
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_128		0x00000040
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_256		0x00000080
#define MOBI_DMA_LIST_NODE_FLAG_BURST_SIZE_MASK		0x000000FF

/* determines the direction an address will be adjusted */
#define MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_INC			0x00000100
#define MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_DEC			0x00000200
#define MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_NONE		0x00000400
#define MOBI_DMA_LIST_NODE_FLAG_ADDRADJ_MASK		0x00000700

/* 
 *  use to enable s/g on a given block.  the sg_params, count
 *  and interval must be set using the CONFIG_SG_ENABLE flag.  Note
 *  if you define different values for src and dst nodes, then you 
 *  cannot enable REVERSE_DIR and s/g on the same block!
 */
#define MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE			0x00000800
#define MOBI_DMA_LIST_NODE_FLAG_SG_ENABLE_MASK		0x00000800


/* DMA options flags, passed to mobi_dma_request */
#define MOBI_DMA_O_NONE			0x00000000

/* ============================================================== */
/* API function descriptions                                      */
/* ============================================================== */

/**
 * \brief mobi_dma_get_max_blk_size:
 * 	Returns the maximum size for a single dma transaction
 *
 * \param dmah  : DMA handler
 * \param status: pointer to mobi_dma_status will will be
 * 	           updated with current status flags
 *
 * \retval -EINVAL - if invalid dma_handle provided
 * \retval -ENODEV - if dma_handle has not been allocated
 * \retval maximum size in bytes
 *
 * \remark
 * 	 In this case, a single DMA transaction, is defined from
 * 	the software's perspective.  The hardware will break these
 * 	transactions into much smaller ones on the bus.  Examples of 
 * 	a single transaction would be a single buffer(less than 
 * 	max_blk_size) sent to mobi_dma_single or a single node in a
 * 	mobi_dma_list or scatter/gather list. DMA request must be 
 * 	checked that the size requested is not greater than the max
 * 	and if it is, the DMAC driver will have to break this into 
 * 	multiple transactions.  The value returned from this function
 * 	can be used to check the size of your transfer or list nodes.
 *   
 */
int32_t mobi_dma_get_max_blk_size(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_get_bytes:
 * 	Returns then number of bytes transfer so far.
 *
 * \param dmah  : DMA handler
 * \param status: pointer to mobi_dma_status will will be
 * 	           updated with current status flags
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval number of bytes transfer
 *
 * \remark
 *   Will returned the number of bytes transfered at the 
 * time this function is called.  If the transfer is still
 * running the current total will be returned or if transfer
 * has completed will return the total number of bytes 
 * transfer.  function can be called at any time but is
 * best called after calling mobi_dma_disable.  Note that
 * calling this function while a multiblock transfer is running
 * is not recommended due to the method that must be used
 * to calculate the running total(it could be slow for large
 * list)
 */
int32_t mobi_dma_get_bytes(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_get_status:
 * 	Returns the status of the channel
 *
 * \param dmah  : DMA handler
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval mobi_dma_status  - current status of channel/transfer
 *
 *  \remark
 *  \verbatim
 *  if channel is valid return values are
 *   	MOBI_DMA_STATUS_XFER_RUNNING
 *   		DMA transfer is currently running 
 *   	MOBI_DMA_STATUS_XFER_IDLE
 *   		The DMA controller has completed the transfer.  
 *   		It is possible to retrieve an idle status before 
 *   		receiving an interrupt.  this could happen in the 
 *   		brief time that the driver is doing post-processing 
 *   		after the xfer complete and before sending the 
 *   		interrupt.
 *   	MOBI_DMA_STATUS_CHANNEL_ENABLED
 *   		DMA channel has been enabled via mobi_dma_enable
 *   	MOBI_DMA_STATUS_CHANNEL_DISABLED
 *   		DMA channel has been disabled via mobi_dma_disable
 *   	MOBI_DMA_STATUS_ERROR
 *   		An error has occurred during the tranfer.  
 *  \endverbatim
 *
 */
mobi_dma_status mobi_dma_get_status(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_config
 * 	Configure the dma channel for transfer.  flags can apply to 
 * three differerent "areas", src, dst and xfer.  apply flags to the
 * src and dst that explicitly define that interface.  For example, the
 * transfer_width of a device. xfer flags apply to the transfer as a 
 * whole and is not necassarily something that specifically applies to 
 * the source or destination.  In addition, some flags need some 
 * addition parameters to be defined in order to apply the flag.  This
 * function is used to set the appropriate flags to configure the transfer.
 *
 * 	\param dmah  - DMA channel handle
 * 	\param what  - what are we configuring
 * 	\param flags - configuration flags
 * 	\param cfg_data  - mg_dma_config_data_t* pointer to data associated with a flag
 *
 * 	These flags apply when what is DMA_CONFIG_SRC or DMA_CONFIG_DST
 * 	    - MOBI_DMA_CONFIG_TRANSFER_WIDTH_*    - MUST be set
 * 	    	- 8, 16 or 32, for devices typically FIFO width
 * 	    - MOBI_DMA_CONFIG_BURST_SIZE_*	   - default of 8
 * 	    	- 1, 4, 8, 16, 32, burst transaction length. the
 * 	    	number of items of size 'WIDTH' to be used for each
 * 	    	burst transaction
 * 	    - MOBI_DMA_CONFIG_HANDSHAKE_*   	   - default is hw
 * 	    	- HW, SW, define the handshaking interface
 * 	    - MOBI_DMA_CONFIG_ADDRADJ	   	   - default increment	
 * 	    	- inc, dec, none, on devices, could be accessing a 
 * 	    	FIFO or some type of buffer.  If access is not to 
 * 	    	a fixed address, then need to know how to inc/dec 
 * 	    	the device address.  Devices with FIFOs should set this 
 * 	    	to NONE.  
 * 	    - MOBI_DMA_CONFIG_ENDIAN_*
 * 	    	- set the endianess of a device, BE or LE.  Default
 * 	    	is LE. Cannont be set for mem->mem or dev->dev transfers
 * 	    	How the bytes will be swapped will depend on the width 
 * 	    	of the src, dest and data width.
 * 	    - MOBI_DMA_CONFIG_SG_ENABLE
 * 	    	- this flags is used to define the params needed for
 * 	    	a scatter/gather transfer.  data should point to a
 * 	    	struct sg_params variable that has been initialized. this
 * 	    	flag must be used to set the param irregardless of the
 * 	    	type of transfer(single or multi block).  See flag
 * 	    	description for more details
 *
 * 	These flags apply when what is DMA_CONFIG_XFER
 * 	    - MOBI_DMA_CONFIG_DATA_WIDTH_*
 * 	    	- set the size of data element to be operated upon.
 * 	    	this value is used when determining how data will
 * 	    	be swapped when the src & dst are of different 
 * 	    	endianess.  For example, a 32-bit src & dst with
 * 	    	different endianess should have a data width of
 * 	    	8 which would result in B3B2B1B0 <-> B0B1B2B3 
 *
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 *
 * \remark 
 *   these settings are persistant until the channel is released.  when
 *  using a mobi_dma_list values set here will be the default if no
 *  flags have been set in a mobi_dma_list nodes flag fields.  this 
 *  function can be called multiple times with different flags and data.
 *
 */
int32_t mobi_dma_config(mobi_dma_handle dmah, uint8_t what,
		uint32_t flags, struct mg_dma_config_data *cfg_data);

/**
 * \brief mobi_dma_setup_mlist:
 * 	setup DMA channel based on client provided mobi_dma_list
 *
 * \param dmah: DMA channel number
 * \param mlist: pointer to mobi_dma_list
 * \param nents: number of entries in mlist
 * \param dma_length: total length of the transfer request in bytes
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval -EIO    - if internal resource request fails
 * \retval Zero   - upon success
 *
 * \remark
 * \verbatim
 *   This function allows a client to define a dma transfer
 * using the mobi_dma_list structure.  The structure has this
 * definition:
 * 	struct mobi_dma_list {
 *      	uint32_t src_addr;
 *              uint32_t dst_addr;
 *       	int32_t  size;	    // size in bytes for this list entry
 *	        uint32_t xfr_flags; // flags that apply to the general xfer of this entry.
 *	        uint32_t src_flags; // flags that apply to the src node
 *	        uint32_t dst_flags; // flags that apply to the dst node
 *		struct mobi_dma_list *next;
 *	};
 *
 *   This function should be used when accessing memory that is discontiguous, 
 * or when trying to do multiple operations in a single transaction.  For 
 * example, programming a device to perform an operation and then dma-ing
 * the data to the device.  This sequence can be programmined into a 
 * mobi_dma_list and would be considered a single dma transfer.
 *   The client creates a linked list of type mobi_dma_list and passes
 * this list to the function.  There are no restrictions on the transfer
 * type; in other words, can do mem->mem, mem->per, per->mem or per->per.
 * The src and dst must remain the same device/target for the duration of
 * the transfer. See mobi_dma.h for explanation of the flags.
 * \endverbatim
 *
 */
int32_t
mobi_dma_setup_mlist(mobi_dma_handle dmah,
		struct mobi_dma_list *mlist,
		uint32_t nents,
		uint32_t dma_length);

/**
 * \brief mobi_dma_setup_sglist:
 * 	setup DMA channel using sglist to/from a device
 *
 * \param dmah: DMA channel number
 * \param sg: pointer to the scatter-gather list/vector
 * \param nents: scatter-gather list number of entries count
 * \param dma_length: total length of the transfer request in bytes
 * \param dev_addr: physical device port address
 * \param dmamode: DMA transfer mode, DMA_FROM_DEVICE or DMA_TO_DEVICE 
 *
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval Zero   - upon success
 *
 * \remark
 *   This function can be used to setup a transfer between peripherals
 * and memory using a "classic" sglist.  The sglist can be created
 * using dma_map_sg and other related functions.
 *
 *
 */
int32_t
mobi_dma_setup_sglist(mobi_dma_handle dmah,
		struct scatterlist *sg,
		uint32_t nents,
		uint32_t dma_length,
		uint32_t dev_addr,
		enum dma_data_direction dmamode);

/**
 * \brief mobi_dma_setup_single:
 * 	  Setup DMA channel for a single block transfer. 
 *
 * \param dmah: dma channel handle
 * \param dma_src_addr: the DMA/physical memory address of the src data block
 * \param dma_dst_addr: the DMA/physical memory address of the destination target
 * \param dma_length: length of the data block in bytes
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval -EIO    - if internal resource request fails
 * \retval Zero    - upon success
 *
 * \remark
 *   A block is defined as either a tranfer of less than or equal 
 * to MAX_DMA_BLOCK_SIZE or a single block of contiguous memory 
 * greater than MAX_DMA_BLOCK_SIZE.  If doing a transfer of 
 * greater than MAX_DMA_BLOCK_SIZE, the src and dst cannot be 
 * the same type(mem->mem or per->per).  
 *
 */
int32_t mobi_dma_setup_single(mobi_dma_handle dmah, 
		uint32_t dma_src_addr, uint32_t dma_dst_addr, 
		uint32_t dma_length);

/**
 * \brief mobi_dma_setup_sglist:
 * 	setup DMA channel using sglist to/from a device
 *
 * \param dmah: DMA channel number
 * \param sg: pointer to the scatter-gather list/vector
 * \param nents: scatter-gather list number of entries count
 * \param dma_length: total length of the transfer request in bytes
 * \param dev_addr: physical device port address
 * \param dmamode: DMA transfer mode, DMA_FROM_DEVICE or DMA_TO_DEVICE 
 *
 * \retval -EINVAL - if invalid argument provided
 * \retval -ENOMEM - if internal memory allocation fails
 * \retval Zero   - upon success
 *
 * \remark
 *   This function can be used to setup a transfer between peripherals
 * and memory using a "classic" sglist.  The sglist can be created
 * using dma_map_sg and other related functions.
 *
 *
 */
int32_t
mobi_dma_setup_sglist(mobi_dma_handle dmah,
		struct scatterlist *sglist,
		uint32_t nents,
		uint32_t dma_length,
		uint32_t dev_addr,
		enum dma_data_direction dmamode);


/**
 * \brief mobi_dma_setup_handler:
 * 	setup DMA channel end notification handlers
 *
 * \param dmah: DMA channel number
 * \param handler: the pointer to the client's callback handler
 * \param data: user specified value to be passed back to the handlers
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 */
int
mobi_dma_setup_handler(mobi_dma_handle dmah, 
		void (*handler)(mobi_dma_handle, mobi_dma_event, void*),
		void *data);

/**
 * \brief mobi_dma_swhs_sglreq:
 *   Starts a single transaction for the src/dst node when 
 * using the sw handshaking interface.  
 *
 * \param dmah: DMA handle
 * \param target:  DMA_SRC_NODE or DMA_DST_NODE
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero   - upon success
 *
 * \remark
 *   This should be called when the client want to initiate a 
 *   transfer that is not a multiple of the burst size.  See the 
 *   DMAC databook for more details on using software handshake.
 *
 */
int32_t mobi_dma_swhs_sglreq(mobi_dma_handle dmah, uint32_t target);

/**
 * \brief mobi_dma_swhs_req:
 *   Starts a transaction for the src/dst node when using
 * 	the sw handshaking interface.  
 *
 *
 * \param dmah: DMA handle
 * \param target:  DMA_SRC_NODE or DMA_DST_NODE
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero   - upon success
 *
 * \remark
 *   This should be called when the client want to initiate a 
 * transfer that is is equal or greater than the burst size.  
 * It is not necessary to request a single transaction when 
 * calling this function.  See the DMAC databook for more 
 * details on using software handshake.
 */
int32_t mobi_dma_swhs_req(mobi_dma_handle dmah, uint32_t target);

/**
 * \brief mobi_dma_enable:
 *   starts DMA transfer
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EBUSY  - cannot find an unlocked master bus
 *
 */
int32_t mobi_dma_enable(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_disable:
 *   Disable DMA channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 *
 * \remark
 *   This function should be called after a transfer has completed.
 * If a transfer is still running and an error has not been 
 * encountered an attempt is made to let the transfer complete and 
 * disable the channel cleanly.  Note this does not release the channel 
 * only disables it. 
 *
 */
int32_t mobi_dma_disable(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_abort:
 *   Abort transfer on DMA channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval Zero    - upon success
 *
 * \remark
 * 		This function is used to immediately abort a running transfer.  
 * 	No effort is made to shutdown cleanly and any data in transit will
 * 	be lost.
 *
 */
int32_t mobi_dma_abort(mobi_dma_handle dmah);

/**
 * \brief mobi_dma_request:
 *   Request/allocate a DMA channel.
 *
 * \param name: client's unique non-NULL identification
 * \param options: DMA option flags(MOBI_DMA_O_*)
 *
 * \retval -EINVAL - if name is NULL
 * \retval -ENODEV - if a DMA channel is not available
 * \retval Zero    - upon success
 *
 */
mobi_dma_handle mobi_dma_request(const char *name, uint32_t options);

/**
 * \brief mobi_dma_free:
 *   Release previously acquired channel
 *
 * \param dmah: DMA handle
 *
 * \retval -ENODEV - if invalid or unallocated dma_handle provided
 * \retval -EPERM  - if channel has not been disabled
 * \retval non-Zero  - mobi_dma_handle
 *
 */
int mobi_dma_free(mobi_dma_handle dmah);

#endif
