#!/bin/sh

# Copyright 2008 Maxim IC
#
# Merlin distribution configuration script.
# This script is not supposed to be called directly,
# it barely check for command line errors and expect
# the environment to be properly setup.
# When exiting the script return 0 if it worked and
# a positive number otherwise. If successful it will also
# generate several images and a result file named config.result

DIST_NAME="merlin";
TOOLCHAIN=arm-${DIST_NAME}-linux-uclibc;
KEEPTMP=0;
CONFIG_FILE=;
FAKEROOT=
WORK_DIR=;
POST_PKG_INSTALL_TARBALLS=
TMP_DIR=/tmp/${DIST_NAME}-install.$$
while [ ! -z "$1" ]; do
	case $1 in
		"--keeptmp") KEEPTMP=1; shift 1;;
		"--tmp") TMP_DIR=$2; shift 2;;
		"--fakeroot") FAKEROOT=$2; shift 2;;
		"--postpkg") POST_PKG_INSTALL_TARBALLS="${POST_PKG_INSTALL_TARBALLS} $2"
			shift 2;;
		*) 
			if [ -z "${CONFIG_FILE}" ]; then
				CONFIG_FILE=$1;
			elif [ -z "${WORK_DIR}" ]; then
				WORK_DIR=$1;
			else
				echo "ERROR: too many arguments: $1";
				echo "try --help";
				return 1;
			fi
			shift 1;;
	esac
done
ROOTFS_DIR=${WORK_DIR}/rootfs
PACKAGES_DIR=${WORK_DIR}/packages
CONFIG_DIR=${WORK_DIR}/config
IMAGES_DIR=${WORK_DIR}/images
EXTRAS_DIR=${WORK_DIR}/extras
HOST_DIR=${WORK_DIR}/$(echo $(uname -m)-$(uname -s) | awk '{ print tolower($1) }')
RESULT_FILE=${CONFIG_DIR}/config.result
SVN_TAG=$(cat ${WORK_DIR}/svntag);
NAND_PADDING_KB=0
# The host directory must be an absolute path before it is added to PATH
if [ "X${HOST_DIR:0:1}" != "X/" ]; then
	HOST_DIR=$(pwd)/${HOST_DIR};
fi
export PATH="${HOST_DIR}/bin:${PATH}"


modify_file()
{	local FILENAME=$1;
	local EXPR=$2;
	local TMP_FILE=${TMP_DIR}/$(basename ${FILENAME}).$$;
	cp ${FILENAME} ${TMP_FILE}
	if [ $? -ne 0 ]; then
		return 1;
	fi
	sed -e "${EXPR}" ${TMP_FILE} > ${FILENAME}
	if [ $? -ne 0 ]; then
		return 2;
	fi
	return 0;
}

clean_and_exit()
{	CODE=$1;
	echo -ne "Cleaning temporary files ... ";
	if [ -d ${TMP_DIR} ]; then 
		if [ $((KEEPTMP)) -ne 1 ]; then
			rm -rf ${TMP_DIR}; 
		fi
	fi
	echo -ne "done\n";
	exit ${CODE};
}

get_flash_size_kB()
{
	local CHIPSIZE=$1

	case "${CHIPSIZE:${#CHIPSIZE}-1:1}" in 
		"k") kbytes=$((${CHIPSIZE:0:${#CHIPSIZE}-1}/8));;
		"M") kbytes=$((${CHIPSIZE:0:${#CHIPSIZE}-1}*1024/8));;
		"G") kbytes=$((${CHIPSIZE:0:${#CHIPSIZE}-1}*1024*1024/8));;
		*) kbytes=$(($((${CHIPSIZE}+1023))/1024/8));;
	esac
 	FLASH_SIZE_KB=$((kbytes));
	return 0;
}

add_user_accounts()
{	accounts_list=$*;
	uid=500;
	gid=500;
	for account in ${accounts_list}; do
		user=$(echo ${account} | sed -e 's/\([^:]*\):\([^:]*\)[:]*\(.*\)/\1/');
		pass=$(echo ${account} | sed -e 's/\([^:]*\):\([^:]*\)[:]*\(.*\)/\2/');
		extra=$(echo ${account} | sed -e 's/\([^:]*\):\([^:]*\)[:]*\(.*\)/\3/');
		pass=$(cryptpwd ${pass});
		if [ -n "${extra}" ]; then
			new_uid=$(echo ${extra} | sed -e 's/\([^:]*\)[:]*\(.*\)/\1/');
			if [ -n "${new_uid}" ]; then
				old_uid=$uid
				uid=$new_uid
				new_gid=$(echo ${extra} | sed -e 's/\([^:]*\)[:]*\(.*\)/\2/');
				if [ -n "${new_gid}" ]; then
					old_gid=$gid
					gid=$new_gid
				fi
			fi
		fi
		echo "${user}:x:${uid}:${gid}:${user}:/home/${user}:/bin/sh" >> ${ROOTFS_DIR}/etc/passwd;
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create user ${user}.\n";
			return 1;
		fi
		echo "${user}:${pass}:12773:0:99999:7:::" >> ${ROOTFS_DIR}/etc/shadow;
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create user ${user}.\n";
			return 2;
		fi
		echo "${user}:x:${gid}:" >> ${ROOTFS_DIR}/etc/group;
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create group ${user}.\n";
			return 3;
		fi
		mkdir -p ${ROOTFS_DIR}/home/${user};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create user ${user}.\n";
			return 4;
		fi
		if [ -n "${new_uid}" ]; then
			uid=$old_uid
		else
			uid=$((${uid}+1));
		fi
		if [ -n "${new_gid}" ]; then
			gid=$old_gid
		else
			gid=$((${gid}+1));
		fi
	done
	return 0;
}

create_resolv_conf()
{
	if [ "X${CONFIG_NETWORK_RESOLVE_DOT_CONF}" != "X" ]
	then
		for word in ${CONFIG_NETWORK_RESOLVE_DOT_CONF}
		do
			case $word in 
				"search") echo -ne "\n$word " >> ${ROOTFS_DIR}/etc/resolv.conf
						  ;;
				"nameserver") echo -ne "\n$word " >> ${ROOTFS_DIR}/etc/resolv.conf
						  ;;
				"domain") echo -ne "\n$word " >> ${ROOTFS_DIR}/etc/resolv.conf
						  ;;
				"sortlist") echo -ne "\n$word " >> ${ROOTFS_DIR}/etc/resolv.conf
						  ;;
				"options") echo -ne "\n$word " >> ${ROOTFS_DIR}/etc/resolv.conf
						  ;;
					  *) echo -ne "$word " >> ${ROOTFS_DIR}/etc/resolv.conf
			esac
		done
	    echo "" >> ${ROOTFS_DIR}/etc/resolv.conf
	fi
	return 0;
}

setup_chkconfig()
{
# going to handle any and all drivers/apps/services controllable by chkconfig in this function

	mkdir -p ${ROOTFS_DIR}/var/config/

	if [ "X${CONFIG_CHKCONFIG_WATCHDOG}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/watchdog
	else
		echo off > ${ROOTFS_DIR}/var/config/watchdog
	fi
	if [ -n "${CONFIG_CHKCONFIG_WATCHDOG_MODPARAMS}" ]
	then
		echo "MODPARAMS=\"${CONFIG_CHKCONFIG_WATCHDOG_MODPARAMS}\"" >> ${ROOTFS_DIR}/var/config/watchdog.options
	fi

	if [ "X${CONFIG_CHKCONFIG_SDCARD}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/sdcard
	else
		echo off > ${ROOTFS_DIR}/var/config/sdcard
	fi
	if [ -n "${CONFIG_CHKCONFIG_SDCARD_MODPARAMS}" ]
	then
		echo "MODPARAMS=\"${CONFIG_CHKCONFIG_SDCARD_MODPARAMS}\"" >> ${ROOTFS_DIR}/var/config/sdcard.options
	fi

	if [ "X${CONFIG_CHKCONFIG_USB}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/usb
	else
		echo off > ${ROOTFS_DIR}/var/config/usb
	fi
	if [ -n "${CONFIG_CHKCONFIG_USB_MODPARAMS}" ]
	then
		echo "MODPARAMS=\"${CONFIG_CHKCONFIG_USB_MODPARAMS}\"" >> ${ROOTFS_DIR}/var/config/usb.options
	fi

	if [ "X${CONFIG_CHKCONFIG_NETWORK}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/network
	else
		echo off > ${ROOTFS_DIR}/var/config/network
	fi
	if [ -n "${CONFIG_CHKCONFIG_NETWORK_MODPARAMS}" ]
	then
		echo "MODPARAMS=\"${CONFIG_CHKCONFIG_NETWORK_MODPARAMS}\"" >> ${ROOTFS_DIR}/var/config/network.options
	fi
	if [ -n "${CONFIG_CHKCONFIG_NETWORK_IFCONFIG_OPTIONS}" ]
	then
		echo "IFCONFIG_OPTS=\"${CONFIG_CHKCONFIG_NETWORK_IFCONFIG_OPTIONS}\"" >> ${ROOTFS_DIR}/var/config/network.options
	fi

	if [ "X${CONFIG_CHKCONFIG_NFS}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/nfs
	else
		echo off > ${ROOTFS_DIR}/var/config/nfs
	fi

	if [ "X${CONFIG_CHKCONFIG_EXTRA_PARTITIONS}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/partitions
	else
		echo off > ${ROOTFS_DIR}/var/config/partitions
	fi

	if [ "X${CONFIG_CHKCONFIG_FRAMEBUFFER}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/fb
	else
		echo off > ${ROOTFS_DIR}/var/config/fb
	fi
	if [ -n "${CONFIG_CHKCONFIG_FRAMEBUFFER_MODPARAMS}" ]
	then
		echo "MODPARAMS=\"${CONFIG_CHKCONFIG_FRAMEBUFFER_MODPARAMS}\"" >> ${ROOTFS_DIR}/var/config/fb.options
	fi

	# applications/services
	if [ "X${CONFIG_CHKCONFIG_NTP}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/ntp
	else
		echo off > ${ROOTFS_DIR}/var/config/ntp
	fi

	if [ "X${CONFIG_CHKCONFIG_APPWEB}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/appweb
	else
		echo off > ${ROOTFS_DIR}/var/config/appweb
	fi

	if [ "X${CONFIG_CHKCONFIG_FTP}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/ftp
	else
		echo off > ${ROOTFS_DIR}/var/config/ftp
	fi

	if [ "X${CONFIG_CHKCONFIG_SYSCTL_VM}" == "Xy" ]
	then
		echo on > ${ROOTFS_DIR}/var/config/vm
	else
		echo off > ${ROOTFS_DIR}/var/config/vm
	fi
# we want to get the percentage base on linux ram not system ram, the kernel
# does this based on system ram, so for 2.6.20 we tweak it here for now since
# the algorithm changed in later kernels
	echo "${CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS}" |grep -sq 'vm\.dirty_background_ratio'
	if [ $? -ne 0 ]
	then
		linux_ram_MB=$((CONFIG_LINUX_RAM_SIZE*1024*1024))
		system_ram_MB=$((CONFIG_SYSTEM_RAM_SIZE*1024*1024))
		linux_tenp=$(echo "$linux_ram_MB * .1" | bc)
		linux_tenp=`echo $linux_tenp |cut -d'.' -f1`
		total_percent=$(echo "$linux_tenp / $system_ram_MB * 100" | bc -l)
		result=`echo "$total_percent" |cut -d'.' -f1`
		result=$((result+1))
		if [ -n "${CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS}" ]
		then
			CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS="${CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS}\nvm.dirty_background_ratio=${result}"
		else
			CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS="vm.dirty_background_ratio=${result}"
		fi
	fi
	if [ -n "${CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS}" ]
	then
		echo -e "${CONFIG_CHKCONFIG_SYSCTL_VM_OPTIONS}" >> ${ROOTFS_DIR}/var/config/vm.options
	fi
}

signal_handler()
{	echo -ne "\nError: script interrupted by a signal.\n";
	clean_and_exit 255;
}
extract_packages()
{
	echo "Extracting packages ..."
	# Check if the directory containing the packages exists
	if [ ! -e  ${PACKAGES_DIR} ]; then
		echo "ERROR: ${PACKAGES_DIR}/ does not exist !"
		clean_and_exit 3;
	fi

	echo -ne "base.${TOOLCHAIN}.tar "
	tar -x -C ${ROOTFS_DIR} -f ${PACKAGES_DIR}/base.${TOOLCHAIN}.tar
	pkglist="$(grep CONFIG_PKG ${CONFIG_FILE} | grep "=y" | grep -v "^#" | grep -v CONFIG_PKG_NAME)"
	for pkg in $pkglist; do
		pkg_cfg_var=$(echo $pkg | cut -d "=" -f 1)
		pkg_name_var=$(echo ${pkg_cfg_var}_NAME)
		pkg_name=$(eval echo $`echo $pkg_name_var`)
		
		grep -e "^${pkg_name}[ ]\+" ${PACKAGES_DIR}/info > /dev/null
		if [ "X$?" != "X0" ]; then
			echo -e "\nERROR: No package found for ${pkg_name}!"
			echo "This may mean ${pkg_name} has not been compiled and/or packaged."
			echo "To create ${pkg_name}.tar, go to dist/${pkg_name} and type make DIST=${DIST_NAME} all install."
			echo "Note: if you don't want to use packages, unset USE_PACKAGE variables in dist.mk"
			clean_and_exit 3;
		fi

		tar_list="$(grep -e "^${pkg_name}[ ]\+" ${PACKAGES_DIR}/info | awk '{print $5}')"
		if [ "X${tar_list}" == "X" ]; then
			echo -n "Package ${pkg_name} was found in packages/info but the command below, used to find "
			echo "the tarballs corresponding to the package returned nothing:"
			echo -e "\tgrep -e "^${pkg_name}[ ]\+" ${PACKAGES_DIR}/info | awk '{print \$5}'"
			echo "It may be a syntax error in the info file or an unexpected case (bug)"
			clean_and_exit 3;
		fi
		for tar in ${tar_list}; do
			echo -n "${tar} "
			tar -x -C ${ROOTFS_DIR} -f ${PACKAGES_DIR}/${tar}
			if [ $? -ne 0 ]; then
				echo -e "\nERROR:Unable to untar ${PACKAGES_DIR}/${pkg_name}.tar !\n";
				clean_and_exit 3;
			fi
		done
	done

	echo -ne "done\n";
}

# Setup a trap in case CtrlC is pressed
trap signal_handler SIGKILL SIGINT SIGHUP SIGQUIT SIGTERM

echo -ne "Merlin configuration script\nParsing config file ... "

source ${CONFIG_FILE};
if [ $? -ne 0 ]; then
	echo -ne "ERROR\nConfig file is corrupted.\n";
	clean_and_exit 1;
fi

if [ "X${DIST_NAME}" == "Xmerlin" ]
then
	if [ "X${CONFIG_FOR_MERLIN_DIST}" != "Xy" ]; then
		echo -ne "ERROR\nThis script and the config file version do not match.\n"
		clean_and_exit 1;
	fi
fi

if [ "X${DIST_NAME}" == "Xfalcon" ]
then
	if [ "X${CONFIG_FOR_FALCON_DIST}" != "Xy" ]; then
		echo -ne "ERROR\nThis script and the config file version do not match.\n"
		clean_and_exit 1;
	fi
fi


if [ ! -d ${TMP_DIR} ]; then
	mkdir -p ${TMP_DIR};
	if [ $? -ne 0 ]; then 
		echo -ne "ERROR\nFailed to create temporary storage ${TMP_DIR}.\n"
		clean_and_exit 2; 
	fi
fi
echo -ne "done\n"

echo "Configuring distribution, please wait ... "

# Extract packages, only if USE_PACKAGE is set in dist.mk
if [ -e ${CONFIG_DIR}/USE_PACKAGE ]; then
	extract_packages

	if [ -n "${POST_PKG_INSTALL_TARBALLS}" ]
	then
		echo -ne "Extracting extra packages ...\n"
		for pkg in ${POST_PKG_INSTALL_TARBALLS}
		do
			ext=$(echo ${pkg} | sed -e 's/.*\.//g');
			if [ "X${ext}" = "Xtgz" ] || [ "X${ext}" = "Xgz" ]
			then
				tar_cmd="tar -z";
			elif [ "X${ext}" = "Xbz2" ]
			then
				tar_cmd="tar -j";
			elif [ "X${ext}" = "Xtar" ]
			then
				tar_cmd="tar";
			else
				echo -ne "ERROR\nCould not detect ${pkg} extension.\n"
				clean_and_exit 4;
			fi
			echo -ne "   ${pkg}\n"
			# note, these tarballs are expected to have ./rootfs in the directory path, 
            # generally we only will ever see files generated in the sw repos
			${tar_cmd} -x -f ${pkg} -C ${WORK_DIR} .
			if [ $? -ne 0 ]
			then
				echo -ne "ERROR\nPackage ${pkg} is corrupted  when using ${tar_cmd} !\n\n";
				clean_and_exit 5;
			fi
		done
		echo -ne "done\n";
	fi
fi

INITRD_FILES_LIST=${CONFIG_DIR}/initrd_files.lst;
INITRD_MODULES_LIST="";

if [ "X${CONFIG_INITRD_EXTRA_FILES}" != "X" ]
then
	for mod in ${CONFIG_INITRD_EXTRA_FILES}
	do
		echo $mod >> ${INITRD_FILES_LIST}
	done
fi

if [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmerlinproto" ]; then
	CHIP_FAMILY="fpga"
elif [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmax700proto" ]; then
	CHIP_FAMILY="max700proto"
elif [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmax700dini" ]; then
	CHIP_FAMILY="dini"
else
	if [ "X${CONFIG_FOR_FALCON_DIST}" == "Xy" ]
	then
		CHIP_FAMILY="max700"
	else
		CHIP_FAMILY="mg3500"
	fi
fi

# CONFIG_KERNELPART_SIZE_KB
# int
# Set to 0 to let the script detect the best value
if [ -z "${CONFIG_KERNELPART_SIZE_KB}" ]; then
	CONFIG_KERNELPART_SIZE_KB=0;
fi

# CONFIG_INITRDPART_SIZE_KB
# int
# Set to 0 to  the script detect the best value
if [ -z "${CONFIG_INITRDPART_SIZE_KB}" ]; then
	CONFIG_INITRDPART_SIZE_KB=0;
fi
#
# CONFIG_MBOOTPART_SIZE_KB
# int
# Set to 0 to let the script detect the best value
if [ -z "${CONFIG_MBOOTPART_SIZE_KB}" ]; then
	CONFIG_MBOOTPART_SIZE_KB=0;
fi

# CONFIG_ROOTFSPART_SIZE_KB
# int
# Set to 0 to let the script detect the best value
if [ -z "${CONFIG_ROOTFSPART_SIZE_KB}" ]; then
	CONFIG_ROOTFSPART_SIZE_KB=0;
fi

if [ "X${CONFIG_BOARD_TYPE_NAME}" == "X" ]
then
	echo -ne "ERROR\nUnknown board type.\n";
	clean_and_exit 20;
else
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}board_${CONFIG_BOARD_TYPE_NAME}\n"
fi

# OK, all supported configs of mboot are going to be compiled, so....
# need to figure out which one we need to use, append the env. and move it
# to the generic name mboot.bin.  this is a bit ugly but everything is
# compiled before this point and there is no other obvious way right now
if [ "X${CONFIG_BOARD_TYPE_NAME}" == "Xmerlinproto" ]
then
	MBOOT_BOARD_TYPE=fpga
else
	MBOOT_BOARD_TYPE=chip
fi

if [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_ddr
elif [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_nand
elif [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_nor
elif [ "X${CONFIG_BOOT_DEVICE_XMODEM}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_xmodem
elif [ "X${CONFIG_BOOT_DEVICE_JTAG}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_jtag
elif [ "X${CONFIG_BOOT_DEVICE_ENET_TFTP}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_tftp
elif [ "X${CONFIG_BOOT_DEVICE_MEMTEST}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_full_memtest
elif [ "X${CONFIG_BOOT_DEVICE_DDRTEST}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_ddrtest
elif [ "X${CONFIG_BOOT_DEVICE_USB}" == "Xy" ]; then
	MBOOT_TYPE=mboot_${MBOOT_BOARD_TYPE}_usb
else
	echo -ne "ERROR\nFailed to discover bootloader image.\n";
	clean_and_exit 20;
fi

# Figure out which kernel image to use
if [ "X${CONFIG_INSTALL_LINUX_ZIMAGE}" = "Xy" ]
then
	KERNEL_IMAGE_SRC="${IMAGES_DIR}/zImage.${CHIP_FAMILY}.img";
elif [ "X${CONFIG_INSTALL_LINUX_IMG}" = "Xy" ]
then
	KERNEL_IMAGE_SRC="${IMAGES_DIR}/vmlinux.${CHIP_FAMILY}.img";
elif [ "X${CONFIG_BOOT_IMAGE_TYPE_THIRD_STAGE}" = "Xy" ]
then
	KERNEL_IMAGE_SRC=
else
	echo -ne "ERROR\nInternal error, unrecognized kernel image type.\n";
	clean_and_exit 20;
fi
if [ ! -e "${KERNEL_IMAGE_SRC}" ]
then
	echo "ERROR kernel image not found: ${KERNEL_IMAGE_SRC}"
	clean_and_exit 20
fi

# Decide what to build on the config file
# Later on another pass will decide of the final config
BUILD_MBOOT=1;
BUILD_KERNEL=1;
BUILD_INITRD=1;
if [ "X${CONFIG_ROOTFS_TYPE_NONE}" == "Xy" ]; then
	BUILD_ROOTFSIMG=0;
	BUILD_ROOTFSTAR=0;
elif [ "X${CONFIG_ROOTFS_TYPE_IMAGE}" == "Xy" ]; then
	BUILD_ROOTFSIMG=1;
	BUILD_ROOTFSTAR=0;
elif [ "X${CONFIG_ROOTFS_TYPE_ARCHIVE}" == "Xy" ]; then
	BUILD_ROOTFSIMG=0;
	BUILD_ROOTFSTAR=1;
else
	# For compatibility reasons we assume it means CONFIG_ROOTFS_TYPE_NONE
	CONFIG_ROOTFS_TYPE_NONE="y";
	BUILD_ROOTFSIMG=0;
	BUILD_ROOTFSTAR=0;
fi
if [ -n "${CONFIG_BOOTIMAGE_NAME}" ]; then
	BUILD_BOOTIMG=1;
else
	BUILD_BOOTIMG=0;
fi

# Check NOR config
if [ "X${CONFIG_NOR_DEVICE}" == "Xy" ]; then
	get_flash_size_kB ${CONFIG_NOR_CHIPSIZE};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR:\nFailed to retrieve NOR device size, ${CONFIG_NOR_CHIPSIZE} invalid.\n";
		clean_and_exit 4;
	fi
	# Check if the address lines config is correct
	if [ "X${CONFIG_NOR_AD_24}" == "Xy" ]; then
		echo -ne "ERROR:\n32MB NOR flashes are not supported yet.\n";
		clean_and_exit 5;
	elif [ "X${CONFIG_NOR_AD_23}" == "Xy" ] && [ $((FLASH_SIZE_KB)) -gt 16384 ]; then
		echo -ne "ERROR:\nYour flash need a 24rd address line to your NOR flash. Please fix your config.\n";
		clean_and_exit 5;
	elif [ "X${CONFIG_NOR_AD_22}" == "Xy" ] && [ $((FLASH_SIZE_KB)) -gt 8192 ]; then
		echo -ne "ERROR:\nYour flash need a 23rd address line to your NOR flash. Please fix your config.\n";
		clean_and_exit 6;
	fi		
fi

# CONFIG_UBOOTIMAGE_NAME
# string 
MBOOT_IMAGE_NAME=${CONFIG_MBOOTIMAGE_NAME};
if [ -z "${MBOOT_IMAGE_NAME}" ]; then
	MBOOT_IMAGE_NAME=${TMP_DIR}/mboot.bin;
fi

# CONFIG_KERNELIMAGE_NAME
# string 
KERNEL_IMAGE_NAME=${CONFIG_KERNELIMAGE_NAME};
if [ -z "${KERNEL_IMAGE_NAME}" ]; then
	KERNEL_IMAGE_NAME=${TMP_DIR}/kernel.img;
fi

# CONFIG_INITRDIMAGE_NAME
# string 
INITRD_IMAGE_NAME=${CONFIG_INITRDIMAGE_NAME};
if [ -z "${INITRD_IMAGE_NAME}" ]; then
	INITRD_IMAGE_NAME=${TMP_DIR}/initrd.img;
fi

# CONFIG_ROOTFSIMAGE_NAME
# string 
ROOTFS_IMAGE_NAME=${CONFIG_ROOTFSIMAGE_NAME};
if [ -z "${ROOTFS_IMAGE_NAME}" ]; then
	ROOTFS_IMAGE_NAME=${TMP_DIR}/rootfs.bin;
fi

# CONFIG_ROOTFSARCHIVE_NAME
# string 
ROOTFS_ARCHIVE_NAME=${CONFIG_ROOTFSARCHIVE_NAME};
if [ -z "${ROOTFS_ARCHIVE_NAME}" ]; then
	ROOTFS_ARCHIVE_NAME=${TMP_DIR}/rootfs.tgz;
fi

# CONFIG_BOOTIMAGE_NAME
# string 
BOOT_IMAGE_NAME=${CONFIG_BOOTIMAGE_NAME};
if [ -z "${BOOT_IMAGE_NAME}" ]; then
	BOOT_IMAGE_NAME=${TMP_DIR}/boot.bin;
fi

# CONFIG_INSTALL_HOST
# bool 
if [ "X${CONFIG_INSTALL_HOST}" == "Xy" ]; then
	# CONFIG_INSTALL_MULTIPLATFORM
	# bool 
	if [ "X${CONFIG_INSTALL_MULTIPLATFORM}" == "Xy" ]; then
		HOST_DIR="${WORK_DIR}/*-*";
	else
		if [ ! -d ${HOST_DIR} ]; then
			echo -ne "ERROR\nHost type $(basename ${HOST_DIR}) is not supported, contact Maxim.\n"
			clean_and_exit 8;
		fi
	fi
	# CONFIG_HOSTDIR
	# string 
	if [ -z "${CONFIG_HOSTDIR}" ]; then
		echo -ne "ERROR\nHost installation enabled but no directory specified.\n";
		clean_and_exit 9;
	fi
	if [ -d ${CONFIG_HOSTDIR} ]; then
		echo -ne "\nERROR: ${CONFIG_HOSTDIR} already exist. Make sure this is really the directory you want to use and erase it.";
		clean_and_exit 10;
	fi
fi

#
# Setup the boot device
#
BOOTDEV="unknown";
BOOTDEV_BLOCK_SIZE_B=4;
BOOTDEV_SIZE_KB=0;	# That's all we need
BOOTDEV_MTDPARTS="";
BOOTDEV_USED_SPACE_KB=0;
if [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ] || \
	   [ "X${CONFIG_BOOT_DEVICE_XMODEM}" == "Xy" ] || \
	   [ "X${CONFIG_BOOT_DEVICE_JTAG}" == "Xy" ]   || \
	   [ "X${CONFIG_BOOT_DEVICE_ENET_TFTP}" == "Xy" ] || \
	   [ "X${CONFIG_BOOT_DEVICE_USB}" == "Xy" ]	
then
	BOOTDEV="ram";
	BOOTDEV_SIZE_KB=$((CONFIG_LINUX_RAM_SIZE*1024));

elif [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
then
	BOOTDEV="nand";
	if [ $((CONFIG_NAND_PAGESIZE)) -eq 2048 ]
	then
		BOOTDEV_BLOCK_SIZE_B=$((CONFIG_NAND_PAGESIZE*64));
	else
		BOOTDEV_BLOCK_SIZE_B=$((CONFIG_NAND_PAGESIZE*32));
	fi
	get_flash_size_kB ${CONFIG_NAND_CHIPSIZE};
	if [ $? -ne 0 ]
	then
		echo -ne "ERROR:\nFailed to retrieve NAND device size, ${CONFIG_NAND_CHIPSIZE} invalid.\n";
		clean_and_exit 4;
	fi
	BOOTDEV_SIZE_KB=${FLASH_SIZE_KB};

elif [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ]
then
	BOOTDEV="nor";
	BOOTDEV_BLOCK_SIZE_B=$((CONFIG_NOR_BLOCKSIZE_KB*1024));
	get_flash_size_kB ${CONFIG_NOR_CHIPSIZE};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR:\nFailed to retrieve NOR device size, ${CONFIG_NOR_CHIPSIZE} invalid.\n";
		clean_and_exit 4;
	fi
	BOOTDEV_SIZE_KB=${FLASH_SIZE_KB};

elif [ "X${CONFIG_BOOT_DEVICE_MEMTEST}" == "Xy" ] || [ "X${CONFIG_BOOT_DEVICE_DDRTEST}" == "Xy" ]
then
	if [ $((CONFIG_MBOOTPART_SIZE_KB)) -gt 0 ]; then
		BOOTDEV_SIZE_KB=$((CONFIG_MBOOTPART_SIZE_KB*$((CONFIG_MBOOT_BOOTLOADER_COPIES))))
	else
		BOOTDEV_SIZE_KB=$((CONFIG_MBOOT_BOOTLOADER_COPIES*16));	# 16kB is all we need
	fi
	# Disable the build of kernel, initrd and rootfs
	BUILD_MBOOT=1;
	BUILD_KERNEL=0;
	BUILD_INITRD=0;
	BUILD_ROOTFSIMG=0;
	BUILD_ROOTFSTAR=0;
fi

# Define boot device partition map. Use the MTD format
# First partition: bootloader, size=CONFIG_MBOOTPART_SIZE_KB or BOOTDEV_BLOCK_SIZE_B or 16kB
# In boot from RAM you cannot have anything after mboot in the 1st MB
# NOTE: we use the file name because at this point in the script it will be defined if this partition exists
if [ $((BUILD_MBOOT)) -eq 1 ]
then
	if [ $((CONFIG_MBOOTPART_SIZE_KB)) -eq 0 ] && [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ]
	then
		if [ $((BOOTDEV_BLOCK_SIZE_B)) -lt 1048576 ]
		then
			# Must align on CONFIG_MBOOTPART_SIZE_KB but bigger than 1MB
			CONFIG_MBOOTPART_SIZE_KB=$(($((1048576-$(($((1048576-$((BOOTDEV_BLOCK_SIZE_B))))%$((BOOTDEV_BLOCK_SIZE_B))))+1023))/1024));
		else
			CONFIG_MBOOTPART_SIZE_KB=$(($((BOOTDEV_BLOCK_SIZE_B+1023))/1024));
		fi

	elif [ $((CONFIG_MBOOTPART_SIZE_KB)) -lt 1024 ] && [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xy" ]
	then
		echo -ne "ERROR\nBootloader partition is too small for RAM boot: $((CONFIG_MBOOTPART_SIZE_KB))kB.(min 1MB)\n";
		clean_and_exit 80;

	elif [ $((CONFIG_MBOOTPART_SIZE_KB)) -eq 0 ]
	then
		if [ $((BOOTDEV_BLOCK_SIZE_B)) -lt 16384 ]
		then
			# Must align on CONFIG_MBOOTPART_SIZE_KB but bigger than 16kB
			CONFIG_MBOOTPART_SIZE_KB=$(($((16384-$(($((16384-$((BOOTDEV_BLOCK_SIZE_B))))%$((BOOTDEV_BLOCK_SIZE_B))))+1023))/1024));
		else
			CONFIG_MBOOTPART_SIZE_KB=$(($((BOOTDEV_BLOCK_SIZE_B+1023))/1024));
			if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
			then
				NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
				CONFIG_MBOOTPART_SIZE_KB=$((CONFIG_MBOOTPART_SIZE_KB+NAND_PADDING_KB))
			fi
		fi

	elif [ $((CONFIG_MBOOTPART_SIZE_KB)) -lt 16 ]
	then
		echo -ne "ERROR\nBootloader partition is too small: $((CONFIG_MBOOTPART_SIZE_KB))kB.\n";
		clean_and_exit 81;

	elif [ $(($((CONFIG_MBOOTPART_SIZE_KB*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
	then
		echo -ne "ERROR\nBootloader partition size must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
		clean_and_exit 82;
	else
		if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
		then
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			CONFIG_MBOOTPART_SIZE_KB=$((CONFIG_MBOOTPART_SIZE_KB+NAND_PADDING_KB))
		fi
	fi

	for ((image=1 ; image <= CONFIG_MBOOT_BOOTLOADER_COPIES ; image++)); do
		# For NOR devices the bootloader must be at the end
		if [ "X${CONFIG_BOOT_DEVICE_NOR}" != "Xy" ]; then
			BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((CONFIG_MBOOTPART_SIZE_KB))k(mboot$((image)))"
		fi
		BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+CONFIG_MBOOTPART_SIZE_KB));
	done
else
	CONFIG_MBOOTPART_SIZE_KB=0;
fi

if [ $((CONFIG_SYSCFGPART_SIZE_KB)) -ne 0 ] && [ "X${CONFIG_BOOT_DEVICE_RAM}" == "Xn" ]
then
	echo "ERROR:  System Config Partition currently not supported, skipping..."

: << COMMENTBLOCK
	if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
	then
		if [ $(($((CONFIG_SYSCFGPART_SIZE_KB*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
		then
			echo -ne "ERROR\nSystem Config partition size must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
			clean_and_exit 82;
		else
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			CONFIG_SYSCFGPART_SIZE_KB=$((CONFIG_SYSCFGPART_SIZE_KB+NAND_PADDING_KB))
			BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((CONFIG_SYSCFGPART_SIZE_KB))k(syscfg)"
			BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+CONFIG_SYSCFGPART_SIZE_KB));
		fi
	else
		echo -ne "ERROR\nSystem Config partition only support on NAND flash\n";
		clean_and_exit 82;
	fi
COMMENTBLOCK
fi

# Second partition: kernel, size=CONFIG_KERNELPART_SIZE_KB or BOOTDEV_BLOCK_SIZE_B or kernel image size
if [ $((BUILD_KERNEL)) -eq 1 ]
then
	KERNELPART_MINSIZE_KB=$(($(($(ls --block-size $((BOOTDEV_BLOCK_SIZE_B)) -l ${KERNEL_IMAGE_SRC} | sed -e 's/[^ ]*[ ]\+[^ ]*[ ]\+[^ ]*[ ]\+[^ ]*[ ]\+\([0-9]*\).*/\1/')*$((BOOTDEV_BLOCK_SIZE_B))+1023))/1024));

	if [ $((CONFIG_KERNELPART_SIZE_KB)) -eq 0 ]
	then
		CONFIG_KERNELPART_SIZE_KB=$((KERNELPART_MINSIZE_KB));
		if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
		then
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			CONFIG_KERNELPART_SIZE_KB=$((CONFIG_KERNELPART_SIZE_KB+NAND_PADDING_KB))
		fi

	elif [ $((CONFIG_KERNELPART_SIZE_KB)) -lt $((KERNELPART_MINSIZE_KB)) ]
	then
		echo -ne "ERROR\n Kernel partition ($((CONFIG_KERNELPART_SIZE_KB))kB)) is too small for the kernel ($((KERNELPART_MINSIZE_KB))kB).\n";
		clean_and_exit 85;

	elif [ $(($((CONFIG_KERNELPART_SIZE_KB*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
	then
		echo -ne "ERROR\nKernel partition size must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
		clean_and_exit 86;

	else
		# given a size, pad it by N blocks
		if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
		then
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			CONFIG_KERNELPART_SIZE_KB=$((CONFIG_KERNELPART_SIZE_KB+NAND_PADDING_KB))
		fi
	fi

	if [ $((CONFIG_KERNELPART_SIZE_KB-NAND_PADDING_KB)) -lt $((KERNELPART_MINSIZE_KB)) ]
	then
		echo "WARNING: Kernel size is using some NAND bad block padding, suggest increasing partition size"
		echo "-------------------------------------------------"
		echo "Config size  : $((CONFIG_KERNELPART_SIZE_KB-NAND_PADDING_KB))"
		echo "NAND Padding : ${NAND_PADDING_KB}"
		echo "Actual size  : ${MATHRES}"
		echo "-------------------------------------------------"
	fi

	for ((image=1 ; image <= CONFIG_MBOOT_KERN_COPIES ; image++)); do
		BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((CONFIG_KERNELPART_SIZE_KB))k(kernel$((image)))"
		BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+CONFIG_KERNELPART_SIZE_KB));
	done
else
	CONFIG_KERNELPART_SIZE_KB=0;
fi

if [ "X${CONFIG_SPLASH_SCREEN}" == "Xy" ]
then
	# Enable splash screen 
	SD_SPLASH_INITRD_SIZE_BYTES=$((196608+368640));
	HD_SPLASH_INITRD_SIZE_BYTES=$((491520+942080));
	if [ "X${CONFIG_BOOT_DEVICE_NAND}" != "Xy" ]; then
		echo "Splash screen supported only when boot device is nand\n";
	else		
		# we always run from initrd, so put utils in now
		if [ "X${CONFIG_OUTPUT_1080I}" == "Xy" ]
		then
			echo /usr/bin/setAD9889 >> ${INITRD_FILES_LIST}
		else
			echo /usr/bin/setADV739x >> ${INITRD_FILES_LIST}
		fi
		
		echo /usr/bin/i2crw >> ${INITRD_FILES_LIST}
		echo /usr/bin/i2cexp >> ${INITRD_FILES_LIST}
		echo /usr/bin/splash >> ${INITRD_FILES_LIST}
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}qcc\ni2c-dev\ni2c-mux\ndwapbi2c\ni2c-core\n"

		if [ "X${CONFIG_OUTPUT_480I}" == "Xy" ] || [ "X${CONFIG_OUTPUT_576I}" == "Xy" ]
		then
			SPLASH_SIZE_KB=$(($((SD_SPLASH_INITRD_SIZE_BYTES/BOOTDEV_BLOCK_SIZE_B))*1024))
		else
			SPLASH_SIZE_KB=$(($((HD_SPLASH_INITRD_SIZE_BYTES/BOOTDEV_BLOCK_SIZE_B))*1024))
		fi

		if [ "X${CONFIG_SPLASH_SCREEN_IN_INITRD}" == "Xy" ]
		then
			if [ "X${CONFIG_SPLASH_TILED_IMAGES}" == "Xy" ]
			then
				if [ -e ${CONFIG_SPLASH_Y_TILED_IMAGE_NAME} ]
				then
					cp ${CONFIG_SPLASH_Y_TILED_IMAGE_NAME} ${ROOTFS_DIR}/etc/sysconfig/splash.y
				else
					echo "ERROR: Could not find splash image - ${CONFIG_SPLASH_Y_TILED_IMAGE_NAME}"
					clean_and_exit 87;
				fi
				if [ -e ${CONFIG_SPLASH_UV_TILED_IMAGE_NAME} ]
				then
					cp ${CONFIG_SPLASH_UV_TILED_IMAGE_NAME} ${ROOTFS_DIR}/etc/sysconfig/splash.uv
				else
					echo "ERROR: Could not find splash image - ${CONFIG_SPLASH_UV_TILED_IMAGE_NAME}"
					clean_and_exit 87;
				fi
			else
				## Use Default maxim_XxY_logo.yuv file
				##
				if [ "X${CONFIG_SPLASH_SINGLE_IMAGE_NAME}" == "X" ]
				then
					if [ "X${CONFIG_OUTPUT_480I}" == "Xy" ] || [ "X${CONFIG_OUTPUT_576I}" == "Xy" ]
					then
						CONFIG_SPLASH_SINGLE_IMAGE_NAME="${IMAGES_DIR}/maxim_720x480_logo.yuv"
					else
						CONFIG_SPLASH_SINGLE_IMAGE_NAME="${IMAGES_DIR}/maxim_1280x720_logo.yuv"
					fi
				else
					## User has provided a YUV file ...
					if [ -e ${IMAGES_DIR}/${CONFIG_SPLASH_SINGLE_IMAGE_NAME} ]
					then
						## Create the Y and UV files from the given YUV file
						##
						CONFIG_SPLASH_SINGLE_IMAGE_NAME=${IMAGES_DIR}/${CONFIG_SPLASH_SINGLE_IMAGE_NAME}
					else 
						## User has provided a path to a non-existent YUV file ... bail
						##
						echo "ERROR: Could not find splash image - ${CONFIG_SPLASH_SINGLE_IMAGE_NAME}"
					clean_and_exit 87;
					fi
				fi

				if [ "X${CONFIG_OUTPUT_480I}" == "Xy" ] || [ "X${CONFIG_OUTPUT_576I}" == "Xy" ]
				then
					width=720;height=480
				else
					width=1280;height=720
				fi
				f2l ${CONFIG_SPLASH_SINGLE_IMAGE_NAME} ${ROOTFS_DIR}/etc/sysconfig/splash ${width} ${height}
				if [ $? -ne 0 ]
				then
					echo "ERROR: f2l failed to created tiled splash images"
					clean_and_exit 87;
				fi
			fi

		elif [ "X${CONFIG_SPLASH_SCREEN_IN_FS}" == "Xy" ]
		then
			# there is some asumption that the rootfs type will match the type we will use
			# for the splash fs so that those module will be imported... probably OK, but..
			# To be completed...
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			SPLASH_SIZE_KB=$((SPLASH_SIZE_KB+NAND_PADDING_KB))

			# XXX need to be added to extra partitions
            # name splash, size, type yaffs
			BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((SPLASH_SIZE_KB))k(splash)"
			BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+SPLASH_SIZE_KB));
		fi	
	fi
fi
    
# Third partition: initrd, size=CONFIG_INITRDPART_SIZE_KB, we cannot detect that one
if [ $((BUILD_INITRD)) -eq 1 ]; then
	if [ $((CONFIG_INITRDPART_SIZE_KB)) -eq 0 ]
	then
		echo -ne "ERROR\nInitrd partition size cannot equal zero\n";
		clean_and_exit 90;

	elif [ $(($((CONFIG_INITRDPART_SIZE_KB*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
	then
		echo -ne "ERROR: Init RAM disk partition size($((CONFIG_INITRDPART_SIZE_KB*1024))) must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
		clean_and_exit 91;
	else
		if [ "X${CONFIG_SPLASH_SCREEN}" == "Xy" ] && [ "X${CONFIG_SPLASH_SCREEN_IN_INITRD}" == "Xy" ] 
		then
			CONFIG_INITRDPART_SIZE_KB=$((CONFIG_INITRDPART_SIZE_KB+SPLASH_SIZE_KB));
		fi
		
		# given a size, pad it by N blocks
		if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
		then
			NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
			CONFIG_INITRDPART_SIZE_KB=$((CONFIG_INITRDPART_SIZE_KB+NAND_PADDING_KB))
		fi
	fi

	for ((image=1 ; image <= CONFIG_MBOOT_INITRD_COPIES ; image++)); do
		BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((CONFIG_INITRDPART_SIZE_KB))k(initrd$((image)))"
		BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+CONFIG_INITRDPART_SIZE_KB));
	done
else
	CONFIG_INITRDPART_SIZE_KB=0;
fi

# Remove the ',' at the beginning
if [ "X${BOOTDEV_MTDPARTS:0:1}" == "X," ]; then
	# Strip the leading ','
	BOOTDEV_MTDPARTS=${BOOTDEV_MTDPARTS:1};
fi

#
# Setup the root device
#
ROOTDEV="unknown";
ROOTDEV_MTDPARTS=;
ROOTDEV_FS=;
ROOTDEV_SIZE_KB=0;
ROOTDEV_BLOCK_SIZE_B=4;
ROOTDEV_USED_SPACE_KB=0;
# CONFIG_ROOTFS_DEVICE_RAM
# bool
if [ "X${CONFIG_ROOTFS_DEVICE_RAM}" == "Xy" ]; then
	ROOTDEV="ram";
	# In this mode the content of a tar file is inserted inside the initrd image
	# and extracted after the tmpfs filesystem is mounted
	BUILD_ROOTFSTAR=1;
# CONFIG_ROOTFS_DEVICE_NAND
# bool
# nand partions are defined below
elif [ "X${CONFIG_ROOTFS_DEVICE_NAND}" == "Xy" ]; then
	ROOTDEV="nand";
	if [ $((CONFIG_NAND_PAGESIZE)) -eq 2048 ]; then
		ROOTDEV_BLOCK_SIZE_B=$((CONFIG_NAND_PAGESIZE*64));
	else
		ROOTDEV_BLOCK_SIZE_B=$((CONFIG_NAND_PAGESIZE*32));
	fi
	get_flash_size_kB ${CONFIG_NAND_CHIPSIZE};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR:\nFailed to retrieve NAND device size, ${CONFIG_NAND_CHIPSIZE} invalid.\n";
		clean_and_exit 4;
	fi
	ROOTDEV_SIZE_KB=${FLASH_SIZE_KB};
    INITRD_MODULES_LIST="${INITRD_MODULES_LIST}mtdchar\nmtdblock\ncmdlinepart\n";
    INITRD_MODULES_LIST="${INITRD_MODULES_LIST}nand\naumb3000_nand\n";
# CONFIG_ROOTFS_DEVICE_NOR
# bool
# nor partions are defined below
elif [ "X${CONFIG_ROOTFS_DEVICE_NOR}" == "Xy" ]; then
	ROOTDEV="nor";
	ROOTDEV_BLOCK_SIZE_B=$((CONFIG_NOR_BLOCKSIZE_KB*1024));
	get_flash_size_kB ${CONFIG_NOR_CHIPSIZE};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR:\nFailed to retrieve NOR device size, ${CONFIG_NOR_CHIPSIZE} invalid.\n";
		clean_and_exit 4;
	fi
	ROOTDEV_SIZE_KB=${FLASH_SIZE_KB};
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}cfi_probe\ncfi_util\ncfi_cmdset_0001\ncfi_cmdset_0002\n";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}mtdblock\ncmdlinepart\n";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}mhif\nmg3500_flash\n";
# CONFIG_ROOTFS_DEVICE_NFS
# bool
elif [ "X${CONFIG_ROOTFS_DEVICE_NFS}" == "Xy" ]; then
	ROOTDEV="nfs";
	if [ "X${CONFIG_NETWORK_MAC_INSPI}" == "Xy" ]; then
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}at25\ndwapbssi\n"
	fi
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}dw_stmmac\nlibphy\nnfs\naf_packet\n"
	if [ "X${CONFIG_NETWORK_IPADDR_DYNAMIC}" == "Xy" ]; then
		echo /sbin/udhcpc >> ${INITRD_FILES_LIST};
		echo /usr/share/udhcpc/default.script >> ${INITRD_FILES_LIST};
	fi
	echo /sbin/route >> ${INITRD_FILES_LIST};
	echo /sbin/ifconfig >> ${INITRD_FILES_LIST};
else
	echo "ERROR: Internal error, ROOTFS undefined.";
	clean_and_exit 5;
fi
# CONFIG_ROOTFS_JFFS2
# bool
if [ "X${CONFIG_ROOTFS_JFFS2}" == "Xy" ]; then
	ROOTDEV_FS="jffs2";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}jffs2\n";
# CONFIG_ROOTFS_CRAMFS
# bool
elif [ "X${CONFIG_ROOTFS_CRAMFS}" == "Xy" ]; then
	ROOTDEV_FS="cramfs";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}cramfs\n";
# CONFIG_ROOTFS_EXT2
# bool
elif [ "X${CONFIG_ROOTFS_EXT2}" == "Xy" ]; then
	ROOTDEV_FS="ext2";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}ext2\n";
# CONFIG_ROOTFS_EXT3
# bool
elif [ "X${CONFIG_ROOTFS_EXT3}" == "Xy" ]; then
	ROOTDEV_FS="ext3";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}ext3\n";
# CONFIG_ROOTFS_JFS
# bool
elif [ "X${CONFIG_ROOTFS_JFS}" == "Xy" ]; then
	ROOTDEV_FS="jfs";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}jfs\n";
# CONFIG_ROOTFS_JFS
# bool
elif [ "X${CONFIG_ROOTFS_XFS}" == "Xy" ]; then
	ROOTDEV_FS="xfs";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}xfs\n";
# CONFIG_ROOTFS_TMPFS
# bool
elif [ "X${CONFIG_ROOTFS_TMPFS}" == "Xy" ]; then
	ROOTDEV_FS="tmpfs";
# CONFIG_ROOTFS_NFS
# bool
elif [ "X${CONFIG_ROOTFS_NFS}" == "Xy" ]; then
	ROOTDEV_FS="nfs";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}nfs\n";
elif [ "X${CONFIG_ROOTFS_YAFFS}" == "Xy" ]; then
	ROOTDEV_FS="yaffs2";
	INITRD_MODULES_LIST="${INITRD_MODULES_LIST}yaffs2\n";
fi
# Now we build the root device partition table using MTD format
if [ "${ROOTDEV}" == "${BOOTDEV}" ]
then
	ROOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB));
fi

if ( [ $((BUILD_ROOTFSIMG)) -eq 1 ] || [ $((BUILD_ROOTFSTAR)) -eq 1 ] ) && [ "X${ROOTDEV}" != "Xram" ]
then
	if [ $((CONFIG_ROOTFSPART_SIZE_KB)) -eq 0 ] && [ $((ROOTDEV_SIZE_KB)) -eq 0 ]
	then
		echo -ne "ERROR\nBoth the root filesystem partition and device size are unknown.\n";
		clean_and_exit 101;

	elif [ $((CONFIG_ROOTFSPART_SIZE_KB)) -eq 0 ]
	then
		tmp_size=$((ROOTDEV_SIZE_KB-ROOTDEV_USED_SPACE_KB));
		tmp_size=$((tmp_size*1024));
		tmp_size=$((tmp_size-$((tmp_size%$((ROOTDEV_BLOCK_SIZE_B))))));
		CONFIG_ROOTFSPART_SIZE_KB=$(($((tmp_size+1023))/1024));

	elif [ $(($((CONFIG_ROOTFSPART_SIZE_KB*1024))%$((ROOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
	then
		echo -ne "ERROR\nRoot filesystem partition size must be a multiple of $((ROOTDEV_BLOCK_SIZE_B))B.\n";
		clean_and_exit 102;
	fi

	if [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
	then
		NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
		CONFIG_ROOTFSPART_SIZE_KB=$((CONFIG_ROOTFSPART_SIZE_KB+NAND_PADDING_KB))
	fi

	ROOTDEV_MTDPARTS="${ROOTDEV_MTDPARTS},$((CONFIG_ROOTFSPART_SIZE_KB))k(rootfs)";
	ROOTDEV_USED_SPACE_KB=$((ROOTDEV_USED_SPACE_KB+CONFIG_ROOTFSPART_SIZE_KB));
	
	EXTRA_MTDPART_DEFS=""
	if [ "X${CONFIG_NAND_EXTRA_PARTITIONS_SINGLE}" == "Xy" ] || [ "X${CONFIG_NAND_EXTRA_PARTITIONS_CUSTOM}" == "Xy" ]
	then
		if [ "X${CONFIG_NAND_EXTRA_PARTITIONS_CUSTOM}" == "Xy" ]
		then
			OLD_IFS=${IFS}
			IFS=","
			for partition in ${CONFIG_NAND_EXTRA_PARTITIONS_CUSTOM_DEF}
			do
				psize=`echo ${partition}|cut -d':' -f1`
				pname=`echo ${partition}|cut -d':' -f2`
				pfstype=`echo ${partition}|cut -d':' -f3`

				if [ "X${pfstype}" != "Xraw" -a "X${pfstype}" != "Xyaffs2" ]
				then
					echo "ERROR: Unsupported partition type ${pfstype}"
					clean_and_exit 81;
				fi
				# for now, require user to define size on block boundry, but maybe it would be nice
				# to just round this up for the user?
				if [ $(($((psize*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
				then
					echo -ne "ERROR: Size of partition \'$pname\' must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
					clean_and_exit 82;
				else
					NAND_PADDING_KB=$(($(($((BOOTDEV_BLOCK_SIZE_B*CONFIG_NAND_BADBLOCK_PADDING))+1023))/1024));
					psize=$((psize+NAND_PADDING_KB))
				fi
				tmp_size=$((ROOTDEV_SIZE_KB-ROOTDEV_USED_SPACE_KB));
				if [ $psize -gt $tmp_size ]
				then
					echo -ne "ERROR\nThere is not enough space available to create a partition $pname.\n";
					clean_and_exit 102;
				fi
				tmp_size=$((psize*1024));
				tmp_size=$((tmp_size-$((tmp_size%$((ROOTDEV_BLOCK_SIZE_B))))));
				DATAPART_SIZE_KB=$(($((tmp_size+1023))/1024));
				ROOTDEV_MTDPARTS="${ROOTDEV_MTDPARTS},${DATAPART_SIZE_KB}k(${pname})";
				ROOTDEV_USED_SPACE_KB=$((ROOTDEV_USED_SPACE_KB+DATAPART_SIZE_KB));
				if [ "X${EXTRA_MTDPART_DEFS}" != "X" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS},"
				fi
				EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}$pname:$pfstype"
			done
			IFS=${OLD_IFS}
		fi

		# always try to add any remaining space to spare data partition
		tmp_size=$((ROOTDEV_SIZE_KB-ROOTDEV_USED_SPACE_KB));
		tmp_size=$((tmp_size*1024));
		tmp_size=$((tmp_size-$((tmp_size%$((ROOTDEV_BLOCK_SIZE_B))))));
		DATAPART_SIZE_KB=$(($((tmp_size+1023))/1024));
		if [ $((DATAPART_SIZE_KB)) -ne 0 ]
		then

			if [ "X${CONFIG_NAND_REMAINING_SPACE_PART_TYPE_IGNORE}" != "Xy" ]
			then
				 # the '-' will tell tell mtd to use all remaing space
				ROOTDEV_MTDPARTS="${ROOTDEV_MTDPARTS},-(data)";
				ROOTDEV_USED_SPACE_KB=$((ROOTDEV_USED_SPACE_KB+DATAPART_SIZE_KB));
				if [ "X${EXTRA_MTDPART_DEFS}" != "X" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS},"
				fi
				if [ "X${CONFIG_NAND_REMAINING_SPACE_PART_TYPE_YAFFS}" == "Xy" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}data:yaffs2"
				elif [ "X${CONFIG_NAND_REMAINING_SPACE_PART_TYPE_JFFS2}" == "Xy" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}data:jffs2"
				else
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}data:raw"
				fi
			fi
		fi
	fi
fi

if [ "${ROOTDEV}" == "${BOOTDEV}" ]; then
	BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS}${ROOTDEV_MTDPARTS}";
	ROOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS}";
	BOOTDEV_USED_SPACE_KB=$((ROOTDEV_USED_SPACE_KB));
# Remove the ',' at the beginning
elif [ -n "${ROOTDEV_MTDPARTS}" ]; then
	# Strip the leading ','
	ROOTDEV_MTDPARTS=${ROOTDEV_MTDPARTS:1};
fi

# If booting from NOR we now add the bootloader
if [ $((BUILD_MBOOT)) -eq 1 ] && [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ]; then
	# We check how empty the boot device is (remember that mboot has already been
	# accounted for above
	BOOTDEV_REMAINING_SIZE_KB=$((BOOTDEV_SIZE_KB-BOOTDEV_USED_SPACE_KB));
	if [ $((BOOTDEV_REMAINING_SIZE_KB)) -gt 0 ]
	then
		EXTRA_MTDPART_DEFS=""
		if [ "X${CONFIG_NOR_EXTRA_PARTITIONS_SINGLE}" == "Xy" ] || [ "X${CONFIG_NOR_EXTRA_PARTITIONS_CUSTOM}" == "Xy" ]
		then
			if [ "X${CONFIG_NOR_EXTRA_PARTITIONS_CUSTOM}" == "Xy" ]
			then
				OLD_IFS=${IFS}
				IFS=","
				for partition in ${CONFIG_NOR_EXTRA_PARTITIONS_CUSTOM_DEF}
				do
					psize=`echo ${partition}|cut -d':' -f1`
					pname=`echo ${partition}|cut -d':' -f2`
					pfstype=`echo ${partition}|cut -d':' -f3`

					if [ "X${pfstype}" != "Xraw" -a "X${pfstype}" != "Xjffs2" ]
					then
						echo "ERROR: Unsupported partition type ${pfstype}"
						clean_and_exit 81;
					fi
					# for now, require user to define size on block boundry, but maybe it would be nice
					# to just round this up for the user?
					if [ $(($((psize*1024))%$((BOOTDEV_BLOCK_SIZE_B)))) -ne 0 ]
					then
						echo -ne "ERROR: Size of partition \'$pname\' must be a multiple of $((BOOTDEV_BLOCK_SIZE_B))B.\n";
						clean_and_exit 82;
					fi
					tmp_size=$((BOOTDEV_SIZE_KB-BOOTDEV_USED_SPACE_KB));
					if [ $psize -gt $tmp_size ]
					then
						echo -ne "ERROR\nThere is not enough space available to create a partition $pname.\n";
						clean_and_exit 102;
					fi
					tmp_size=$((psize*1024));
					tmp_size=$((tmp_size-$((tmp_size%$((BOOTDEV_BLOCK_SIZE_B))))));
					DATAPART_SIZE_KB=$(($((tmp_size+1023))/1024));
					BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},${DATAPART_SIZE_KB}k(${pname})";
					BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+DATAPART_SIZE_KB));
					if [ "X${EXTRA_MTDPART_DEFS}" != "X" ]
					then
						EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS},"
					fi
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}$pname:$pfstype"
				done
				IFS=${OLD_IFS}
			fi

			# always try to add any remaining space to spare data partition
			tmp_size=$((BOOTDEV_SIZE_KB-BOOTDEV_USED_SPACE_KB));
			tmp_size=$((tmp_size*1024));
			tmp_size=$((tmp_size-$((tmp_size%$((ROOTDEV_BLOCK_SIZE_B))))));
			DATAPART_SIZE_KB=$(($((tmp_size+1023))/1024));
			if [ $((DATAPART_SIZE_KB)) -ne 0 ]
			then

				BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((DATAPART_SIZE_KB))k(data)"
				BOOTDEV_USED_SPACE_KB=$((BOOTDEV_USED_SPACE_KB+DATAPART_SIZE_KB));
				if [ "X${EXTRA_MTDPART_DEFS}" != "X" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS},"
				fi
				if [ "X${CONFIG_NOR_REMAINING_SPACE_PART_TYPE_JFFS2}" == "Xy" ]
				then
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}data:jffs2"
				else
					EXTRA_MTDPART_DEFS="${EXTRA_MTDPART_DEFS}data:raw"
				fi
			fi
		fi

	fi
	for ((image=1 ; image <= CONFIG_MBOOT_BOOTLOADER_COPIES ; image++)); do
		BOOTDEV_MTDPARTS="${BOOTDEV_MTDPARTS},$((CONFIG_MBOOTPART_SIZE_KB))k(mboot$((image)))"
	done
	# Remove the ',' at the beginning
	if [ "X${BOOTDEV_MTDPARTS:0:1}" == "X," ]; then
		# Strip the leading ','
		BOOTDEV_MTDPARTS=${BOOTDEV_MTDPARTS:1};
	fi
fi

# Finally we check that the partitions fit in the device
if [ $((BOOTDEV_USED_SPACE_KB)) -gt $((BOOTDEV_SIZE_KB)) ]; then
	echo -ne "ERROR\nBoot device too small: $((BOOTDEV_SIZE_KB))kB, needs $((BOOTDEV_USED_SPACE_KB))kB.\n";
	echo -ne "Partition table of the boot device is ${BOOTDEV_MTDPARTS}.\n";
	clean_and_exit 100;
fi
if [ $((ROOTDEV_USED_SPACE_KB)) -gt $((ROOTDEV_SIZE_KB)) ] && [ $((ROOTDEV_SIZE_KB)) -ne 0 ]; then
	echo -ne "ERROR\nRoot device too small: $((ROOTDEV_SIZE_KB))kB, needs $((ROOTDEV_USED_SPACE_KB))kB.\n";
	echo -ne "Partition table of the root device is ${ROOTDEV_MTDPARTS}.\n";
	clean_and_exit 100;
elif [ $((ROOTDEV_SIZE_KB)) -eq 0 ]; then
	ROOTDEV_SIZE_KB=$((CONFIG_ROOTFSPART_SIZE_KB));
fi

if [ $((BOOTDEV_USED_SPACE_KB)) -gt 0 ]; then
	echo "Boot device partition table: ${BOOTDEV_MTDPARTS}" >> ${RESULT_FILE}
fi
if [ $((ROOTDEV_USED_SPACE_KB)) -gt 0 ] && [ "${ROOTDEV}" != "${BOOTDEV}" ]; then
	echo "Root device partition table: ${ROOTDEV_MTDPARTS}" >> ${RESULT_FILE}
fi

echo -ne "done\n"

if [ "X${CONFIG_INSTALL_TARGET}" == "Xy" ]; then

	if [ $((BUILD_KERNEL)) -eq 1 ]; then
		echo -ne "Generating kernel image ... "
		if [ ! -d "$(dirname ${KERNEL_IMAGE_NAME})" ]; then
			mkdir -p $(dirname ${KERNEL_IMAGE_NAME});
		fi
		# Now we generate the image
		cp ${KERNEL_IMAGE_SRC} ${KERNEL_IMAGE_NAME};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate kernel image.\n";
			clean_and_exit 21;
		fi
		chmod 664 ${KERNEL_IMAGE_NAME};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to chnage access permissions on kernel image.\n";
			clean_and_exit 22;
		fi
		echo "kernel image: ${KERNEL_IMAGE_NAME}" >> ${RESULT_FILE}
		echo -ne "done\n"
	fi
	
	if [ $((BUILD_MBOOT)) -eq 1 ]; then
		echo -ne "Generating bootloader image ... "
		if [ ! -d "$(dirname ${MBOOT_IMAGE_NAME})" ]; then
			mkdir -p $(dirname ${MBOOT_IMAGE_NAME});
		fi
		# generate the mboot environment config file.
		${CONFIG_DIR}/mbootconfig.sh ${CONFIG_FILE} ${WORK_DIR} ${BOOTDEV_MTDPARTS} ${CONFIG_KERNELPART_SIZE_KB} ${TMP_DIR}
		if [ $? -ne 0 ] || [ ! -e ${TMP_DIR}/mboot.cfg ]
		then
			echo -ne "ERROR\nFailed to generate bootloader configuration.\n";
			clean_and_exit 20;
		fi
		# now generate the environment binary
		cfg2env -c "${TMP_DIR}/mboot.cfg" -e "${TMP_DIR}/mboot.env"
		#cp -f "${TMP_DIR}/mboot.cfg" /tmp
		if [ $? -ne 0 ]
		then
			echo -ne "ERROR\nFailed to generate bootloader environment.\n";
			clean_and_exit 20;
		fi
		# save a copy of the env file in the images dir
		cp ${TMP_DIR}/mboot.env ${IMAGES_DIR}/${MBOOT_TYPE}.env
		ERR=$?
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate bootloader image.\n";
			clean_and_exit 20;
		fi
		# now rename to simple, common name
		cp ${IMAGES_DIR}/${MBOOT_TYPE}.bin ${IMAGES_DIR}/mboot.bin
		ERR=$?
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate bootloader image.\n";
			clean_and_exit 20;
		fi
		# now "magically" join the bootloader and it's environment
		cat ${TMP_DIR}/mboot.env >> ${IMAGES_DIR}/mboot.bin
		ERR=$?
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate bootloader image.\n";
			clean_and_exit 20;
		fi
		if [ "X${CONFIG_BOOT_DEVICE_NOR}" == "Xy" ]; then
			# In this special case we pad mboot. Here we assume mboot partition is always 16kB or more
			dd if=/dev/zero of=${MBOOT_IMAGE_NAME} bs=1024 count=$((CONFIG_MBOOTPART_SIZE_KB-16)) >& /dev/null;
			ERR=$?
			if [ $((ERR)) -eq 0 ]; then
				cat ${IMAGES_DIR}/mboot.bin >> ${MBOOT_IMAGE_NAME};
				ERR=$?
			fi
		else
			# Now copy the image
			cp ${IMAGES_DIR}/mboot.bin ${MBOOT_IMAGE_NAME}
			ERR=$?
		fi
		if [ $((ERR)) -eq 0 ]; then
			chmod 664 ${MBOOT_IMAGE_NAME};
			ERR=$?;
		fi
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate bootloader image.\n";
			clean_and_exit 20;
		fi
		echo "bootloader image: ${MBOOT_IMAGE_NAME}" >> ${RESULT_FILE}
		echo -ne "done\n"
	fi
	
	# We always do it because some of those tools may end up in initrd
	echo -ne "Configuring target root filesystem ... "
	# If necessary copy System.map for the right kernel on the root filesystem
	if [ "X${CONFIG_INSTALL_SYSTEMMAP}" == "Xy" ]; then 
		cp -f ${EXTRAS_DIR}/System.${CHIP_FAMILY}.map ${ROOTFS_DIR}/boot/System.map;
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nCould not copy the System.map onto the filesystem, distribution is broken ?\n";
			clean_and_exit 23;
		fi
	fi
	# Set the root password
	PASS=$(cryptpwd ${CONFIG_ROOT_PASSWD})
	modify_file ${ROOTFS_DIR}/etc/shadow "s|\(root:\)[^:]*\(:.*\)|\1${PASS}\2|"
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to modify shadow file.\n";
		clean_and_exit 3;
	fi
	# CONFIG_EXTRA_ACCOUNT
	# string 
	if [ -n "${CONFIG_EXTRA_ACCOUNT}" ]; then
		add_user_accounts ${CONFIG_EXTRA_ACCOUNT};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create extra user accounts in list: ${CONFIG_EXTRA_ACCOUNT}\n";
			clean_and_exit 23;
		fi		
	fi
	# Remove unnecessary kernel modules
	for dir in ${ROOTFS_DIR}/lib/modules/* ${HOST_DIR}/${TOOLCHAIN}/lib/modules/*; do
		MATHRES=$(echo ${dir} | grep -c ${DIST_NAME}-${CHIP_FAMILY});
		if [ $((MATHRES)) -eq 0 ]; then
			rm -r ${dir};
			if [ $? -ne 0 ]; then
				echo -ne "ERROR\nFailed to remove directory ${dir}\n";
				clean_and_exit 22;
			fi
		fi
	done

	# Change permission on busybox (should be moved elsewhere)
	chmod 644 ${ROOTFS_DIR}/etc/busybox.conf
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to change permissions on /etc/busybox.conf.\n";
		clean_and_exit 21;
	fi
	chmod ugo+s ${ROOTFS_DIR}/bin/busybox
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to change permissions on /etc/busybox.conf.\n";
		clean_and_exit 21;
	fi

	# Create /etc/modules
	for mod in ${CONFIG_CORE_KERNEL_MODULES}; do
		echo ${mod} >> ${ROOTFS_DIR}/etc/modules;
	done
	for mod in ${CONFIG_EXTRA_KERNEL_MODULES}; do
		echo ${mod} >> ${ROOTFS_DIR}/etc/modules;
	done

    # mostly for static configs
	create_resolv_conf

	# setup the chkconfig files
	setup_chkconfig

	# Copy the CONFIG_* file. This is a bit tricky because we want all
	# variables and other shell completion to occur now !
	ERR=0;
	echo -ne "" > ${ROOTFS_DIR}/etc/sysconfig/config
	echo "#######################################################################" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#    DO NOT MODIFY!!  DO NOT MODIFY!! DO NOT MODIFY!!" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#  This file is automatically generated and should not" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#  be modified.  Modifying this file may result in a " >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#  system that is unustable or cannot boot" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#" >> ${ROOTFS_DIR}/etc/sysconfig/config;
	echo "#######################################################################" >> ${ROOTFS_DIR}/etc/sysconfig/config;

	if [ $? -ne 0 ]; then ERR=1; fi
	while read LINE; do
		WRITE_LINE=$(eval echo ${LINE} | sed -e 's/=\(.*\)/=\"\1\"/')
		if [ -n "${WRITE_LINE}" ]; then
			echo ${WRITE_LINE} >> ${ROOTFS_DIR}/etc/sysconfig/config;
			if [ $? -ne 0 ]; then ERR=2; fi
		fi
	done < ${CONFIG_FILE} ;
	if [ $((ERR)) -ne 0 ]; then
		echo -ne "ERROR\nFailed to import config on rootfs.\n";
		clean_and_exit 6;
	fi
	chmod 444 ${ROOTFS_DIR}/etc/sysconfig/config;

	# Create the CONFIGMTD_* file
	ERR=0;
	echo "CFGDEVICES_BOOTDEV=\"${BOOTDEV}\"" > ${ROOTFS_DIR}/etc/sysconfig/devices;
	if [ $? -ne 0 ]
	then 
		ERR=1
	fi
	echo "CFGDEVICES_BOOTDEV_SIZE_KB=\"${BOOTDEV_SIZE_KB}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_BOOTDEV_MTDPARTS=\"${BOOTDEV_MTDPARTS}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_BOOTDEV_BLOCKSIZE=\"${BOOTDEV_BLOCK_SIZE_B}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;

	echo "CFGDEVICES_ROOTDEV=\"${ROOTDEV}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_ROOTDEV_FS=\"${ROOTDEV_FS}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_ROOTDEV_SIZE_KB=\"${ROOTDEV_SIZE_KB}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_ROOTDEV_MTDPARTS=\"${ROOTDEV_MTDPARTS}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_ROOTDEV_BLOCKSIZE=\"${ROOTDEV_BLOCK_SIZE_B}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	echo "CFGDEVICES_ROOTDEV_EXTRA_MTDPARTS=\"${EXTRA_MTDPART_DEFS}\"" >> ${ROOTFS_DIR}/etc/sysconfig/devices;
	if [ $((ERR)) -ne 0 ]
	then
		echo -ne "ERROR\nFailed to import mtd config on rootfs.\n"
		clean_and_exit 7
	fi

	# Remove files that are not always necessary
	for file in ${CONFIG_REMOVE_FILE_LIST}
	do
		rootfs_len=`echo ${ROOTFS_DIR} | wc -c`
		fullpath_file=${ROOTFS_DIR}${file}
		for fpfile in ${fullpath_file}
		do
			one=`echo ${fpfile} |cut -c${rootfs_len}-200`
			if [ -d ${fpfile} ]
			then
				rm -rf ${fpfile}
				if [ $((ERR)) -ne 0 ]
				then
					echo -ne "WARNING:\nCannot remove directory ${one}, continue anyway ... "
				fi
			elif [ -e ${fpfile} ]
			then
				rm -f ${fpfile}
				if [ $((ERR)) -ne 0 ]
				then
					echo -ne "WARNING:\nCannot remove ${one}, continue anyway ... "
				fi
			fi
		done
	done

	# Make sure all files needed for the upgrade are in initrd
	if [ "X${CONFIG_UPGRADE_FTP}" == "Xy" ] || [ "X${CONFIG_UPGRADE_CIFS}" == "Xy" ] || [ "X${CONFIG_UPGRADE_NFS}" == "Xy" ]; then
		if [ "X${CONFIG_NETWORK_MAC_INSPI}" == "Xy" ]; then
			INITRD_MODULES_LIST="${INITRD_MODULES_LIST}at25\ndwapbssi\n"
		fi
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}dw_stmmac\nlibphy\naf_packet\n"
		if [ "X${CONFIG_NETWORK_IPADDR_DYNAMIC}" == "Xy" ]; then
			echo /sbin/udhcpc >> ${INITRD_FILES_LIST};
			echo /usr/share/udhcpc/default.script >> ${INITRD_FILES_LIST};
		fi
		echo /sbin/route >> ${INITRD_FILES_LIST};
		echo /sbin/ifconfig >> ${INITRD_FILES_LIST};
		if [ "X${CONFIG_UPGRADE_FTP}" == "Xy" ]; then
			echo /usr/bin/ftpget >> ${INITRD_FILES_LIST};
		fi
		if [ "X${CONFIG_UPGRADE_HTTP}" == "Xy" ]; then
			echo /usr/bin/wget >> ${INITRD_FILES_LIST};
		fi
		if [ "X${CONFIG_UPGRADE_CIFS}" == "Xy" ]; then
			INITRD_MODULES_LIST="${INITRD_MODULES_LIST}cifs\n"
		fi
		if [ "X${CONFIG_UPGRADE_NFS}" == "Xy" ]; then
			INITRD_MODULES_LIST="${INITRD_MODULES_LIST}nfs\n"
		fi
		if [ "X${CONFIG_ROOTFS_DEVICE_NAND}" == "Xy" ] || [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]; then
			echo /bin/flash_eraseall >> ${INITRD_FILES_LIST};
			echo /bin/nandwrite >> ${INITRD_FILES_LIST};
		fi
	fi
	if [ "X${CONFIG_UPGRADE_UART}" == "Xy" ]; then
		echo /bin/lrz >> ${INITRD_FILES_LIST};
	    # busybox links 
		echo /bin/mv >> ${INITRD_FILES_LIST};
		echo /usr/bin/dirname >> ${INITRD_FILES_LIST};
		echo /usr/bin/basename >> ${INITRD_FILES_LIST};
		if [ "X${CONFIG_ROOTFS_DEVICE_NAND}" == "Xy" ] || [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]; then
			echo /bin/flash_eraseall >> ${INITRD_FILES_LIST};
			echo /bin/nandwrite >> ${INITRD_FILES_LIST};
		fi
	fi
	if [ "X${CONFIG_UPGRADE_USB}" == "Xy" ] 
	then
		# USB stuff
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}dwc_otg\nusbcore\nusb-libusual\nusb-storage\n"
		# scsi stuff 
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}scsi_mod\nsd_mod\n"
		# filesystem stuff
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}jfs\next2\next3\njbd\n"
		# add fat to initrd because a USB could be very handy for upgrades
		INITRD_MODULES_LIST="${INITRD_MODULES_LIST}fat\nvfat\nnls_cp437\nnls_ascii\nnls_iso8859-1\n"
	    # busybox links 
		echo /usr/bin/dirname >> ${INITRD_FILES_LIST};
		echo /usr/bin/basename >> ${INITRD_FILES_LIST};
		echo /lib/udev/vol_id >> ${INITRD_FILES_LIST};
		if [ "X${CONFIG_ROOTFS_DEVICE_NAND}" == "Xy" ] || [ "X${CONFIG_BOOT_DEVICE_NAND}" == "Xy" ]
		then
			echo /bin/flash_eraseall >> ${INITRD_FILES_LIST};
			echo /bin/nandwrite >> ${INITRD_FILES_LIST};
		fi
	fi

	if [ "X${CONFIG_UPGRADE_AUTO}" == "Xy" ] 
	then
		if [ "X${CONFIG_UPGRADE_AUTO_SCRIPT}" != "X" ] 
		then
			cp -f ${CONFIG_UPGRADE_AUTO_SCRIPT} ${ROOTFS_DIR}/etc/sysconfig/helpers/autoupdate
		fi
	fi
	if [ "X${CONFIG_UPGRADE_CUSTOM}" == "Xy" ] 
	then
		if [ "X${CONFIG_UPGRADE_CUSTOM_SCRIPT}" != "X" ] 
		then
			cp -f ${CONFIG_UPGRADE_CUSTOM_SCRIPT} ${ROOTFS_DIR}/etc/sysconfig/helpers/customupdate
		fi
	fi
	echo -ne "done\n"
	
	# Generate the root filesystem image if it is a binary image
	if [ $((BUILD_ROOTFSIMG)) -eq 1 ]; then
		echo -ne "Generating root filesystem image ... "
		if [ ! -d "$(dirname ${ROOTFS_IMAGE_NAME})" ]; then
			mkdir -p $(dirname ${ROOTFS_IMAGE_NAME});
		fi
		MKROOTFS_ARGS="${ROOTFS_DIR} ${ROOTFS_IMAGE_NAME} --devtable ${CONFIG_DIR}/devtab --inplace --quiet";
		MKROOTFS_ARGS="${MKROOTFS_ARGS} --fs ${ROOTDEV_FS} --tmp ${TMP_DIR}/mkrootfs.img-${ROOTDEV_FS}";
		if [ "X${FAKEROOT}" != "X" ]
		then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --fakeroot ${FAKEROOT}"
		fi
		if [ $((CONFIG_ROOTFSPART_SIZE_KB)) -ne 0 ]; then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --size $((CONFIG_ROOTFSPART_SIZE_KB *1024))"
		fi
		if [ $((KEEPTMP)) -eq 1 ]; then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --keeptmp";
		fi
		mkrootfs.sh ${MKROOTFS_ARGS};
		ERR=$?;
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate root filesystem image.\n";
			clean_and_exit 40;
		fi
		if [ $((CONFIG_ROOTFSPART_SIZE_KB)) -eq 0 ]; then
			CONFIG_ROOTFSPART_SIZE_KB=$(du -kL ${ROOTFS_IMAGE_NAME} | sed -e 's/[^0-9]*\([0-9]*\).*/\1/');
		else
			MATHRES=$(du -kL ${ROOTFS_IMAGE_NAME} | sed -e 's/[^0-9]*\([0-9]*\).*/\1/')
			if [ $((CONFIG_ROOTFSPART_SIZE_KB)) -lt $((MATHRES)) ]; then
				echo -ne "ERROR\nGenerated root fs image does not fit in partition (${CONFIG_ROOTFSPART_SIZE_KB}kB).\n";
				clean_and_exit 41;
			fi
		fi
		echo "root filesystem image (type ${ROOTDEV_FS}): ${ROOTFS_IMAGE_NAME}" >> ${RESULT_FILE}
		echo -ne "done\n"
	fi

	# Generate the root filesystem image if it is a tar file
	if [ $((BUILD_ROOTFSTAR)) -eq 1 ]; then
		echo -ne "Generating root filesystem archive ... "
		if [ ! -d "$(dirname ${ROOTFS_ARCHIVE_NAME})" ]; then
			mkdir -p $(dirname ${ROOTFS_ARCHIVE_NAME});
		fi
		MKROOTFS_ARGS="${ROOTFS_DIR} ${ROOTFS_ARCHIVE_NAME} --devtable ${CONFIG_DIR}/devtab --inplace --quiet";
		MKROOTFS_ARGS="${MKROOTFS_ARGS} --fs tar --tmp ${TMP_DIR}/mkrootfs.archive";
		if [ $((CONFIG_ROOTFSPART_SIZE_KB)) -ne 0 ]; then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --size $((CONFIG_ROOTFSPART_SIZE_KB *1024))"
		fi
		if [ "X${FAKEROOT}" != "X" ]
		then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --fakeroot ${FAKEROOT}"
		fi
		if [ $((KEEPTMP)) -eq 1 ]; then
			MKROOTFS_ARGS="${MKROOTFS_ARGS} --keeptmp";
		fi
		mkrootfs.sh ${MKROOTFS_ARGS};
		ERR=$?;
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate root filesystem archive.\n";
			clean_and_exit 42;
		fi
		echo "root filesystem archive: ${ROOTFS_ARCHIVE_NAME}" >> ${RESULT_FILE}
		echo -ne "done\n"
		# If the root filesystem is a RAM disk we must install the archive in the initrd
		if [ "${ROOTDEV}" == "ram" ]; then
			ROOTFS_ARCHIVE_EXT=$(basename ${ROOTFS_ARCHIVE_NAME} | sed -e 's/.*\.//g');
			case ${ROOTFS_ARCHIVE_EXT} in
				"bz2") bzip2 -c -d ${ROOTFS_ARCHIVE_NAME} | gzip -c > ${EXTRAS_DIR}/rootfs.tgz;;
				"gz" | "tgz") cp ${ROOTFS_ARCHIVE_NAME} ${EXTRAS_DIR}/rootfs.tgz;;
				*) gzip -c ${ROOTFS_ARCHIVE_NAME} > ${EXTRAS_DIR}/rootfs.tgz;;
			esac
			if [ $? -ne 0 ]; then
				echo -ne "ERROR: Failed to copy root filesystem tar file into extras.\n";
				clean_and_exit 43;
			fi
			echo /rootfs.tgz ../extras/rootfs.tgz >> ${INITRD_FILES_LIST};
			if [ $? -ne 0 ]; then
				echo -ne "ERROR: Failed to add root filesystem tar file into initrd image file list.\n";
				clean_and_exit 44;
			fi
		fi
	fi

	if [ $((BUILD_INITRD)) -eq 1 ]; then
		echo -ne "Generating initrd image ... "
		if [ ! -d "$(dirname ${INITRD_IMAGE_NAME})" ]; then
			mkdir -p $(dirname ${INITRD_IMAGE_NAME});
		fi
		# Now we generate the initrd image
		# use extended echo so that \n embedded in INITRD_MODULES_LIST
		# become multiple lines, this way modparams can be added here!
		if [ -z "${INITRD_FILES_LIST}" ]; then
			echo "ERROR: Internal error, missing initrd files list.";
			clean_and_exit 29;
		fi

		echo -e "${INITRD_MODULES_LIST}" >> ${TMP_DIR}/modules.lst
		if [ "X${CONFIG_INITRD_EXTRA_KERNEL_MODULES}" != "X" ]
		then
			echo /etc/initrd_modules >> ${INITRD_FILES_LIST};
			for mod in ${CONFIG_INITRD_EXTRA_KERNEL_MODULES}; do
				echo ${mod} >> ${ROOTFS_DIR}/etc/initrd_modules;
				echo ${mod} >> ${TMP_DIR}/modules.lst
			done
		fi
		
		MKINITRDSH_ARGS="${ROOTFS_DIR} ${TMP_DIR}/initrd.bin --files ${INITRD_FILES_LIST} --modules ${TMP_DIR}/modules.lst --init ${EXTRAS_DIR}/init --devtable ${CONFIG_DIR}/devtab --tmp ${TMP_DIR}/mkinitrd --quiet --initramfs"
		if [ "X${FAKEROOT}" != "X" ]
		then
			MKINITRDSH_ARGS="${MKINITRDSH_ARGS} --fakeroot ${FAKEROOT}"
		fi
		if [ $((KEEPTMP)) -eq 1 ]; then
			MKINITRDSH_ARGS="${MKINITRDSH_ARGS} --keeptmp";
		fi
		if [ "X${CONFIG_KERNEL_MODULES_BUILTIN}" == "Xy" ]
		then
			MKINITRDSH_ARGS="${MKINITRDSH_ARGS} --builtin"
		fi
		mkinitrd.sh ${MKINITRDSH_ARGS};
		ERR=$?;
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to create initrd image, mkinitrd.sh returned error $ERR.\n";
			clean_and_exit 30;
		fi
		mkimage -A arm -O linux -T ramdisk -C none -n "initrd ${DIST_NAME}-${SVN_TAG}" -d ${TMP_DIR}/initrd.bin ${INITRD_IMAGE_NAME} > /dev/null;
		if [ $? -ne 0 ]; then ERR=3; fi
		chmod 664 ${INITRD_IMAGE_NAME};
		if [ $? -ne 0 ]; then ERR=4; fi
		if [ $((ERR)) -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate U-Boot compatible initrd image, error code is ${ERR}.\n";
			clean_and_exit 31;
		fi
		# now double check the what was generated will fit! take into account if nand padding has been added
		# nand_padding will be 0 for non-nand configs
		MATHRES=$(du -kL ${INITRD_IMAGE_NAME} | sed -e 's/[^0-9]*\([0-9]*\).*/\1/');
		if [ $((CONFIG_INITRDPART_SIZE_KB-NAND_PADDING_KB)) -lt $((MATHRES)) ]; then
			echo -ne "ERROR: Generated initrd image(${MATHRES}kB) does not fit in partition(${CONFIG_INITRDPART_SIZE_KB}kB).\n";
			clean_and_exit 32;
		fi		
		echo "initrd image: ${INITRD_IMAGE_NAME}" >> ${RESULT_FILE}
		echo -ne "done\n"
	fi
	
	# If requested we create a boot device image
	if [ $((BUILD_BOOTIMG)) -eq 1 ]; then
		echo -ne "Generating boot device image ... ";
		if [ ! -d $(dirname ${BOOT_IMAGE_NAME}) ]; then
			mkdir -p $(dirname ${BOOT_IMAGE_NAME});
			if [ $? -ne 0 ]; then
				echo -ne "ERROR\nFailed to create $(dirname ${BOOT_IMAGE_NAME}).\n";
				clean_and_exit 12;
			fi
		fi		
		# Parse ROOTDEV_MTDPARTS to figure out what needs to be stored in that file
		# Any non-recognized name is assumed to be filled separately by the user
		# and is initialized to 0xFF.
		MKMTDIMAGE_PARTS=;
		OLD_IFS=${IFS};
		IFS=",";
		for partition in ${BOOTDEV_MTDPARTS}; do
			name=$(echo ${partition} | sed -e 's/[^\(]*.\([^\)]*\).*/\1/');
			if [ "X$(echo ${name} | sed -e 's/mboot[0-9]*/FOUND_MBOOT/')" == "XFOUND_MBOOT" ]; then
				MKMTDIMAGE_PARTS="${MKMTDIMAGE_PARTS} ${name}=${MBOOT_IMAGE_NAME}";
			elif [ "X$(echo ${name} | sed -e 's/kernel[0-9]*/FOUND_KERNEL/')" == "XFOUND_KERNEL" ]; then
				MKMTDIMAGE_PARTS="${MKMTDIMAGE_PARTS} ${name}=${KERNEL_IMAGE_NAME}";
			elif [ "X$(echo ${name} | sed -e 's/initrd[0-9]*/FOUND_INITRD/')" == "XFOUND_INITRD" ]; then
				MKMTDIMAGE_PARTS="${MKMTDIMAGE_PARTS} ${name}=${INITRD_IMAGE_NAME}";
			elif [ "X$(echo ${name} | sed -e 's/rootfs[0-9]*/FOUND_ROOTFS/')" == "XFOUND_ROOTFS" ]; then
				# Make sure we have a rootfs image
				if [ $((BUILD_ROOTFSIMG)) -eq 1 ]; then
					MKMTDIMAGE_PARTS="${MKMTDIMAGE_PARTS} ${name}=${ROOTFS_IMAGE_NAME}";
				fi
			fi
		done
		IFS=${OLD_IFS};
		mkmtdimage --image ${BOOT_IMAGE_NAME} ${BOOTDEV_MTDPARTS} ${MKMTDIMAGE_PARTS}
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to generate boot device image.\n";
			clean_and_exit 50;
		fi
		chmod 664 ${BOOT_IMAGE_NAME};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to change access permissions on boot device image.\n";
			clean_and_exit 51;
		fi
		echo "Boot device image: ${BOOT_IMAGE_NAME}" >> ${RESULT_FILE}
	fi
fi

if [ "X${CONFIG_INSTALL_HOST}" == "Xy" ]; then
	echo -ne "Installing host tools and development files ... "
	for kmakefile in ${HOST_DIR}/${TOOLCHAIN}/lib/modules/*/build/Makefile; do
		ksrc=$(dirname $(dirname ${kmakefile} | sed -e "s|${WORK_DIR}|${CONFIG_HOSTDIR}|"))/source;
		kbld=$(dirname ${kmakefile} | sed -e "s|${WORK_DIR}|${CONFIG_HOSTDIR}|");
		modify_file ${kmakefile} "s|\(^[ \t]*KERNELSRC[ \t]*:=\).*|\1${ksrc}|";
		modify_file ${kmakefile} "s|\(^[ \t]*KERNELOUTPUT[ \t]*:=\).*|\1${kbld}|";
		modify_file ${kmakefile} "s|\(^[ \t]*MAKEARGS[ \t]*:=[ \t]*-C\).*|\1 ${ksrc}|";
	done
	if [ ! -d ${CONFIG_HOSTDIR} ]; then
		mkdir -p ${CONFIG_HOSTDIR};
		if [ $? -ne 0 ]; then
			echo -ne "ERROR\nFailed to create ${CONFIG_HOSTDIR}.\n";
			clean_and_exit 200;
		fi
	fi
	cp -a ${HOST_DIR} ${CONFIG_HOSTDIR}/;
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to copy host files into ${CONFIG_HOSTDIR}.\n";
		clean_and_exit 205;
	fi
	chmod -R go+r ${CONFIG_HOSTDIR};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to change access rights in ${CONFIG_HOSTDIR}.\n";
		clean_and_exit 210;
	fi
	chmod -R go-w ${CONFIG_HOSTDIR};
	if [ $? -ne 0 ]; then
		echo -ne "ERROR\nFailed to change access rights in ${CONFIG_HOSTDIR}.\n";
		clean_and_exit 211;
	fi
	echo "Host files installed in ${CONFIG_HOSTDIR}" >> ${RESULT_FILE}
	echo "Host files installed for platform $(basename ${HOST_DIR})" >> ${RESULT_FILE}
	echo -ne "done\n"
fi
	
clean_and_exit 0;
