/*
 *  ejsHandler.c -- Ejscript web framework handler.
 *
 *  The ejs handler supports the Ejscript web framework for applications using server-side Javascript. 
 *
 *  Copyright (c) All Rights Reserved. See details at the end of the file.
 */
/********************************** Includes **********************************/

#include    "http.h"
#include    "ejs.h"

#if BLD_FEATURE_EJS
/*********************************** Locals ***********************************/

#if BLD_FEATURE_MULTITHREAD && TODO
static void ejsWebLock(void *lockData);
static void ejsWebUnlock(void *lockData);
#endif

/***************************** Forward Declarations *****************************/

#if UNUSED
static void createSession(void *handle, int timeout);
static void destroySession(void *handle);
#endif

static void error(void *handle, int code, cchar *msg);
#if UNUSED
static int  mapToStorage(void *handle, char *path, int len, cchar *url);
static int  readFile(void *handle, char **buf, int *len, cchar *path);
#endif
static void redirect(void *handle, int code, cchar *url);
static void setCookie(void *handle, cchar *name, cchar *value, int lifetime, cchar *path, bool secure);
static void setHeader(void *handle, bool allowMultiple, cchar *key, cchar *value);
static int  writeBlock(void *handle, cchar *buf, int size);

/************************************* Code ***********************************/
/*
 *  When we come here, we've already matched either a location block or an extension
 */
static bool matchEjs(MaConn *conn, MaStage *handler, cchar *url)
{
    MaRequest   *req;
    MaLocation  *loc;
    cchar       *ext;

    ext = conn->response->extension;
    req = conn->request;
    loc = req->location;

    if (loc->flags & (MA_LOC_APP | MA_LOC_APP_DIR)) {

        /*
         *  Send non-ejs content under web to another handler, typically the file handler.
         */
        if (strncmp(&req->url[loc->prefixLen], "web/", 4) == 0) {
            if (!(ext && strcmp(ext, "ejs") == 0)) {
                return 0;
            }
        } else {
            if (ext && strcmp(ext, "ejs") == 0) {
                maFormatBody(conn, "Bad Request", "Can't serve *.ejs files outside web directory");
                maFailRequest(conn, MPR_HTTP_CODE_BAD_REQUEST, "Can't server *.ejs files outside web directory");
            }
        }
    }
    return 1;
}


/*
 *  Currently handling: MA_STAGE_GET | MA_STAGE_HEAD | MA_STAGE_POST | MA_STAGE_PUT
 */
static void openEjs(MaQueue *q)
{
    MaRequest       *req;
    MaResponse      *resp;
    MaConn          *conn;
    MaAlias         *alias;
    MaLocation      *location;
    EjsWeb          *web;
    cchar           *sep, *prefix;
    char            *urlBase, *dir, *app, *url, *cp;
    int             flags, locFlags;

    flags = 0;
    url = 0;

    conn = q->conn;
    resp = conn->response;
    req = conn->request;
    alias = req->alias;
    location = req->location;
    prefix = alias->prefix;
    locFlags = location->flags;
    
#if UNUSED
    if (resp->extension && resp->extension[0]) {
        locFlags = 0;
    }
#endif

    if (locFlags & MA_LOC_APP) {
        app = mprStrTrim(mprStrdup(q, prefix), "/");
        url = &req->url[alias->prefixLen];
        dir = mprStrdup(resp, alias->filename);
        if (*url != '/') {
            url--;
        }
        urlBase = mprStrdup(resp, prefix);
        
    } else if (locFlags & MA_LOC_APP_DIR) {
        url = &req->url[alias->prefixLen];
        app = mprStrdup(resp, url);
        if ((cp = strchr(app, '/')) != 0) {
            url = mprStrdup(resp, cp);
            *cp = '\0';
        }
        sep = prefix[strlen(prefix) - 1] == '/' ? "" : "/";
        mprAllocStrcat(resp, &dir, -1, NULL, alias->filename, sep, app, NULL);
        mprAllocStrcat(resp, &urlBase, -1, NULL, prefix, sep, app, NULL);

    } else {
        app = 0;
        dir = mprStrdup(resp, alias->filename);
        url = &req->url[alias->prefixLen];
        flags |= EJS_WEB_FLAG_STAND_ALONE;
        if (*url != '/') {
            url--;
        }        
        urlBase = mprStrdup(resp, prefix);
    }
    mprStrTrim(urlBase, "/");
    mprStrTrim(dir, "/");
    
    if (location->flags & MA_LOC_BROWSER) {
        flags |= EJS_WEB_FLAG_BROWSER_ERRORS;
    }
    if (location->flags & MA_LOC_AUTO_SESSION) {
        flags |= EJS_WEB_FLAG_SESSION;
    }

    //  TODO - why have extension here?
    /*
     *  Var         Stand-Alone             App                         AppDir
     *  app         0                       carmen                      carmen
     *  dir         /Users/mob/....         /Users/mob/hg/carmen        /Users/mob/hg/carmen
     *  urlBase                             /xg/carmen                  /carmen
     *  url                                 stock                       stock
     */
    web = ejsCreateWebRequest(req, q->stage->stageData, conn, app, dir, urlBase, url, req->cookie, flags);
    if (web == 0) {
        maFailRequest(conn, MPR_HTTP_CODE_INTERNAL_SERVER_ERROR, "Can't create Ejs web object for %s", req->url);
        return;
    }
    q->queueData = web;
    maSetHeader(conn, 0, "Last-Modified", req->host->currentDate);
    maDontCacheResponse(conn);
}


/*
 *  The routine runs when all input data has been received.
 */
static void runEjs(MaQueue *q)
{
    MaConn      *conn;
    EjsWeb      *web;
    char        *msg;

    conn = q->conn;
    web = q->queueData;

    maPutForService(q, maCreateHeaderPacket(conn), 0);

    if (ejsProcessWebRequest((EjsWeb*) q->queueData, &msg) < 0) {
        //  TODO - refactor. Want request failed to have an option which says send this output to the client also.
        if (web->flags & EJS_WEB_FLAG_BROWSER_ERRORS) {
            maFormatBody(conn, "Request Failed", "%s", msg);
        }
        maFailRequest(conn, MPR_HTTP_CODE_BAD_GATEWAY, msg);
        mprFree(msg);
    }
    maPutForService(q, maCreateEndPacket(conn), 1);
}


/****************************** Control Callbacks ****************************/

#if UNUSED
static void createSession(void *handle, int timeout)
{
    maCreateSession(handle, timeout);
}
#endif


/*
 *  Define form variables using any query data
 */
static void defineFormVars(void *handle)
{
    MprHash     *hp;
    MaConn      *conn;
    MaRequest   *req;
    Ejs         *ejs;

    conn = (MaConn*) handle;
    req = conn->request;
    mprAssert(req->formVars);

    ejs = ((EjsWeb*) maGetHandlerQueueData(conn))->ejs;

    hp = mprGetFirstHash(req->formVars);
    while (hp) {
        ejsDefineFormVar(ejs, hp->key, hp->data);
        hp = mprGetNextHash(req->formVars, hp);
    }
}


#if UNUSED
static void destroySession(void *handle)
{
    maDestroySession(handle);
}
#endif


static void discardOutput(void *handle)
{
    MaConn      *conn;

    conn = (MaConn*) handle;
    maDiscardData(conn->response->queue[MA_QUEUE_SEND].nextQ->nextQ, 0);
}


static void error(void *handle, int code, cchar *msg)
{
    maFailRequest(handle, code, "%s", msg);
}


static EjsVar *createString(Ejs *ejs, cchar *value)
{
    if (value == 0) {
        return ejs->nullValue;
    }
    return (EjsVar*) ejsCreateString(ejs, value);
}


#if UNUSED
static EjsVar *createCookies(Ejs *ejs, MaConn *conn)
{
    MprHash     *hp;
    EjsVar      *cookies;
    EjsName     qname;
    
    if (conn->request->cookie) {
        cookies = (EjsVar*) ejsCreateSimpleObject(ejs);
        for (hp = 0; (hp = mprGetNextHash(conn->request->cookies, hp)) != 0; ) {
            ejsSetPropertyByName(ejs, cookies, ejsName(&qname, "", hp->key), (EjsVar*) ejsCreateString(ejs, hp->data));
        }
        return cookies;
    }
    return ejs->nullValue;
}
#endif


static EjsVar *createHeaders(Ejs *ejs, MprHashTable *table)
{
    MprHash     *hp;
    EjsVar      *headers, *header;
    EjsName     qname;
    int         index;
    
    headers = (EjsVar*) ejsCreateArray(ejs, mprGetHashCount(table));
    for (index = 0, hp = 0; (hp = mprGetNextHash(table, hp)) != 0; ) {
        header = (EjsVar*) ejsCreateSimpleObject(ejs);
        ejsSetPropertyByName(ejs, header, ejsName(&qname, "", hp->key), (EjsVar*) ejsCreateString(ejs, hp->data));
        ejsSetProperty(ejs, headers, index++, header);
    }
    return headers;
}


static EjsVar *getRequestVar(void *handle, int field)
{
    Ejs         *ejs;
    EjsWeb      *web;
    MaConn      *conn;
    MaRequest   *req;

    conn = handle;
    req = conn->request;
    ejs = ((EjsWeb*) maGetHandlerQueueData(conn))->ejs;

    switch (field) {
    case ES_ejs_web_Request_accept:
        return createString(ejs, req->accept);

    case ES_ejs_web_Request_acceptCharset:
        return createString(ejs, req->acceptCharset);

    case ES_ejs_web_Request_acceptEncoding:
        return createString(ejs, req->acceptEncoding);

    case ES_ejs_web_Request_authAcl:
    case ES_ejs_web_Request_authGroup:
    case ES_ejs_web_Request_authUser:
        //  TODO 
        return (EjsVar*) ejs->undefinedValue;

    case ES_ejs_web_Request_authType:
        return createString(ejs, req->authType);

    case ES_ejs_web_Request_connection:
        return createString(ejs, req->connection);

    case ES_ejs_web_Request_contentLength:
        return (EjsVar*) ejsCreateNumber(ejs, req->length);

    case ES_ejs_web_Request_cookies:
        if (req->cookie) {
            return (EjsVar*) ejsCreateCookies(ejs);
        } else {
            return ejs->nullValue;
        }
        break;

    case ES_ejs_web_Request_extension:
        return createString(ejs, req->parsedUri->ext);

    case ES_ejs_web_Request_files:
        //  TODO 
        return (EjsVar*) ejs->undefinedValue;

    case ES_ejs_web_Request_headers:
        return (EjsVar*) createHeaders(ejs, conn->request->headers);

    case ES_ejs_web_Request_hostName:
        return createString(ejs, req->hostName);

    case ES_ejs_web_Request_method:
        return createString(ejs, req->methodName);

    case ES_ejs_web_Request_mimeType:
        return createString(ejs, req->mimeType);

    case ES_ejs_web_Request_pathInfo:
        return createString(ejs, req->extraPath);

    case ES_ejs_web_Request_pathTranslated:
        return createString(ejs, req->pathTranslated);

    case ES_ejs_web_Request_pragma:
        return createString(ejs, req->pragma);

    case ES_ejs_web_Request_query:
        return createString(ejs, req->parsedUri->query);

    case ES_ejs_web_Request_originalUri:
        return createString(ejs, req->parsedUri->originalUri);

    case ES_ejs_web_Request_referrer:
        return createString(ejs, req->referer);

    case ES_ejs_web_Request_remoteAddress:
        return createString(ejs, conn->remoteIpAddr);

    case ES_ejs_web_Request_remoteHost:
#if BLD_FEATURE_REVERSE_DNS && BLD_UNIX_LIKE
        {
            /*
             *  This feature has denial of service risks. Doing a reverse DNS will be slower,
             *  and can potentially hang the web server. Use at your own risk!!  Not supported for windows.
             */
            struct addrinfo *result;
            char            name[MPR_MAX_STRING];
            int             rc;

            if (getaddrinfo(remoteIpAddr, NULL, NULL, &result) == 0) {
                rc = getnameinfo(result->ai_addr, sizeof(struct sockaddr), name, sizeof(name), NULL, 0, NI_NAMEREQD);
                freeaddrinfo(result);
                if (rc == 0) {
                    return createString(ejs, name);
                }
            }
        }
#endif
        return createString(ejs, conn->remoteIpAddr);

#if UNUSED
    case ES_ejs_web_Request_scriptName:
        return createString(ejs, req->url);
#endif

    case ES_ejs_web_Request_sessionID:
        web = ejs->handle;
        return createString(ejs, web->session ? web->session->id : "");

    case ES_ejs_web_Request_url:
        return createString(ejs, req->url);

    case ES_ejs_web_Request_userAgent:
        return createString(ejs, req->userAgent);
    }

    ejsThrowOutOfBoundsError(ejs, "Bad property slot reference");
    return 0;
}


static EjsVar *getHostVar(void *handle, int field)
{
    Ejs         *ejs;
    MaConn      *conn;
    MaHost      *host;
    EjsWeb      *web;

    conn = handle;
    host = conn->host;
    ejs = ((EjsWeb*) maGetHandlerQueueData(conn))->ejs;

    switch (field) {
    case ES_ejs_web_Host_documentRoot:
        return createString(ejs, host->documentRoot);

    case ES_ejs_web_Host_name:
        return createString(ejs, host->name);

    case ES_ejs_web_Host_protocol:
        return createString(ejs, host->secure ? "https" : "http");

    case ES_ejs_web_Host_isVirtualHost:
        return (EjsVar*) ejsCreateBoolean(ejs, host->flags & MA_HOST_VHOST);

    case ES_ejs_web_Host_isNamedVirtualHost:
        return (EjsVar*) ejsCreateBoolean(ejs, host->flags & MA_HOST_NAMED_VHOST);

    case ES_ejs_web_Host_software:
        return createString(ejs, MA_SERVER_NAME);

    case ES_ejs_web_Host_logErrors:
        web = ejs->handle;
        return (EjsVar*) ((web->flags & EJS_WEB_FLAG_BROWSER_ERRORS) ? ejs->falseValue : ejs->trueValue);
    }

    ejsThrowOutOfBoundsError(ejs, "Bad property slot reference");
    return 0;
}


static EjsVar *getResponseVar(void *handle, int field)
{
    Ejs         *ejs;
    MaConn      *conn;
    MaResponse  *resp;

    conn = handle;
    resp = conn->response;
    ejs = ((EjsWeb*) maGetHandlerQueueData(conn))->ejs;

    switch (field) {
    case ES_ejs_web_Response_code:
        return (EjsVar*) ejsCreateNumber(ejs, resp->code);

#if UNUSED
    case ES_ejs_web_Response_contentLength:
        //  TODO - this is meaningless as the data is still being generated when the script is run
        return (EjsVar*) ejsCreateNumber(ejs, resp->length);

    case ES_ejs_web_Response_cookies:
        //  TODO - these are being defined as headers
        return ejs->undefinedValue;

    case ES_ejs_web_Response_etag:
        return (EjsVar*) createString(ejs, resp->etag);
#endif

    case ES_ejs_web_Response_filename:
        return (EjsVar*) createString(ejs, resp->filename);

    case ES_ejs_web_Response_headers:
        return (EjsVar*) createHeaders(ejs, conn->response->headers);

    case ES_ejs_web_Response_mimeType:
        return (EjsVar*) createString(ejs, resp->mimeType);

    default:
        ejsThrowOutOfBoundsError(ejs, "Bad property slot reference");
        return 0;
    }
}


static EjsVar *getVar(void *handle, int collection, int field)
{
    switch (collection) {
    case EJS_WEB_REQUEST_VAR:
        return getRequestVar(handle, field);
    case EJS_WEB_RESPONSE_VAR:
        return getResponseVar(handle, field);
    case EJS_WEB_HOST_VAR:
        return getHostVar(handle, field);
    default:
        return 0;
    }
}


#if UNUSED
/*
 *  Used to process include files only. Not for URIs that come over the web.
 *  TODO - doesn't seem right!
 *  TODO - since we don't do includes, can this be removed?
 */
static int mapToStorage(void *handle, char *path, int len, cchar *url)
{
    char    *file;

    mprAssert(0);
    file = maMapUriToStorage(handle, url);
    if (file == 0) {
        return MPR_ERR_NOT_FOUND;
    }
    mprStrcpy(path, len, file);
    mprFree(file);

    return 0;
}


//  TODO - who uses this
static int readFile(void *handle, char **pbuf, int *len, cchar *path)
{
    MaConn          *conn;
    MaResponse      *resp;
    MprFile         *file;
    char            *buf;
    int             size;

    mprAssert(0);
    mprAssert(path && *path);

    conn = (MaConn*) handle;
    resp = conn->response;

    file = mprOpen(conn, path, O_RDONLY | O_BINARY, 0666);
    if (file == 0) {
        return MPR_ERR_CANT_OPEN;
    }

    size = ((int) resp->fileInfo.size) * sizeof(char);
    buf = (char*) mprAlloc(resp, size + 1);
    buf[size] = '\0';

    if (mprRead(file, buf, size) != size) {
        mprFree(file);
        mprFree(buf);
        return MPR_ERR_CANT_READ;
    }

    *pbuf = buf;
    if (len) {
        *len = size;
    }
    mprFree(file);
    return size;
}
#endif


static void redirect(void *handle, int code, cchar *url)
{
    maRedirect(handle, code, url);
}


static void setCookie(void *handle, cchar *name, cchar *value, int lifetime, cchar *path, bool secure)
{
    maSetCookie(handle, name, value, lifetime, path, secure);
}


static void setHeader(void *handle, bool allowMultiple, cchar *key, cchar *value)
{
    maSetHeader(handle, allowMultiple, key, value);
}


static void setHttpCode(void *handle, int code)
{
    maSetResponseCode(handle, code);
}


static void setMimeType(void *handle, cchar *mimeType)
{
    maSetResponseMimeType(handle, mimeType);
}


static int writeBlock(void *handle, cchar *buf, int size)
{
    MaConn      *conn;
    MaQueue     *q;

    conn = (MaConn*) handle;

    /*
     *  We write to the service queue of the handler
     */
    q = conn->response->queue[MA_QUEUE_SEND].nextQ;
    mprAssert(q->stage->flags & MA_STAGE_HANDLER);

    return maWriteBlock(q, buf, size, 1);
}


#if BLD_FEATURE_MULTITHREAD && TODO
static void ejsWebLock(void *lockData)
{
    MprMutex    *mutex = (MprMutex*) lockData;

    mprAssert(mutex);
    mutex->lock();
}


static void ejsWebUnlock(void *lockData)
{
    MprMutex    *mutex = (MprMutex*) lockData;

    mprAssert(mutex);

    mutex->unlock();
}
#endif /* BLD_FEATURE_MULTITHREAD */


#if BLD_FEATURE_CONFIG_PARSE
static int parseEjs(MaHttp *http, cchar *key, char *value, MaConfigState *state)
{
    MaLocation      *location;
    MaServer        *server;
    MaHost          *host;
    char            *prefix, *path;
    int             flags;
    
    host = state->host;
    server = state->server;
    location = state->location;
    
    flags = location->flags & (MA_LOC_BROWSER | MA_LOC_AUTO_SESSION | MA_LOC_EXTRA_PATH);

    if (mprStrcmpAnyCase(key, "EjsApp") == 0) {
        if (mprStrcmpAnyCase(value, "on") == 0) {
            location->flags |= MA_LOC_APP;
        } else {
            location->flags &= ~MA_LOC_APP;
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsAppDir") == 0) {
        if (mprStrcmpAnyCase(value, "on") == 0) {
            location->flags |= MA_LOC_APP_DIR;
        } else {
            location->flags &= ~MA_LOC_APP_DIR;
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsAppAlias") == 0) {
        if (maSplitConfigValue(server, &prefix, &path, value, 1) < 0 || path == 0 || prefix == 0) {
            return MPR_ERR_BAD_SYNTAX;
        }
        location = maCreateLocationAlias(http, state, prefix, path, "ejsHandler", MA_LOC_APP | flags);
        if (location == 0) {
            return MPR_ERR_BAD_SYNTAX;
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsAppDirAlias") == 0) {
        if (maSplitConfigValue(server, &prefix, &path, value, 1) < 0 || path == 0 || prefix == 0) {
            return MPR_ERR_BAD_SYNTAX;
        }
        location = maCreateLocationAlias(http, state, prefix, path, "ejsHandler", MA_LOC_APP_DIR | flags);
        if (location == 0) {
            return MPR_ERR_BAD_SYNTAX;
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsErrors") == 0) {
        if (mprStrcmpAnyCase(value, "browser") == 0) {
            location->flags |= MA_LOC_BROWSER;
        } else {
            location->flags &= ~MA_LOC_BROWSER;
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsSessionTimeout") == 0) {
        if (value == 0) {
            return MPR_ERR_BAD_SYNTAX;
        }
        if (! mprGetDebugMode(http)) {
            location->sessionTimeout = atoi(mprStrTrim(value, "\""));
        }
        return 1;

    } else if (mprStrcmpAnyCase(key, "EjsSession") == 0) {
        if (mprStrcmpAnyCase(value, "on") == 0) {
            location->flags |= MA_LOC_AUTO_SESSION;
        } else {
            location->flags &= ~MA_LOC_AUTO_SESSION;
        }
        return 1;
    }

    return 0;
}
#endif


/*
 *  Dynamic module initialization
 */
MprModule *mprEjsHandlerInit(MaHttp *http)
{
    MprModule       *module;
    MaStage         *handler;
    EjsWebControl   *control;

    module = mprCreateModule(http, "ejsHandler", BLD_VERSION, 0, 0, 0);
    if (module == 0) {
        return 0;
    }

    handler = maCreateHandler(http, "ejsHandler", 
            MA_STAGE_GET | MA_STAGE_HEAD | MA_STAGE_POST | MA_STAGE_PUT | MA_STAGE_FORM_VARS | \
            MA_STAGE_VIRTUAL | MA_STAGE_EXTRA_PATH);
    if (handler == 0) {
        mprFree(module);
        return 0;
    }
	http->ejsHandler = handler;
    handler->match = matchEjs;
    handler->open = openEjs;
    handler->run = runEjs;
    handler->parse = parseEjs;

    /*
     *  Setup the control block
     */
    handler->stageData = control = mprAllocObjZeroed(handler, EjsWebControl);

#if UNUSED
    control->createSession = createSession;
    control->destroySession = destroySession;
    control->getSession = getSession;
#endif
    control->defineFormVars = defineFormVars;
    control->discardOutput = discardOutput;
    control->error = error;
    control->getVar = getVar;
#if UNUSED
    control->mapToStorage = mapToStorage;
    control->readFile = readFile;
#endif
    control->redirect = redirect;
    control->setCookie = setCookie;
    control->setHeader = setHeader;
    control->setHttpCode = setHttpCode;
    control->setMimeType = setMimeType;
    control->write = writeBlock;

#if BLD_FEATURE_MULTITHREAD && FUTURE
    /*
     *  This mutex is used very sparingly and must be an application global lock.
     */
    mutex = mprCreateLock(control);
    control->lock = ejsWebLock;
    control->unlock = ejsWebUnlock;
    control->lockData = mutex;
#endif

    if (ejsOpenWebService(control) < 0) {
        return 0;
    }
    return module;
}


#else
void __ejsWebHandlerDummy() {}
#endif /* BLD_FEATURE_EJS */


/*
 *  @copy   default
 *
 *  Copyright (c) Embedthis Software LLC, 2003-2009. All Rights Reserved.
 *  Copyright (c) Michael O'Brien, 1993-2009. All Rights Reserved.
 *
 *  This software is distributed under commercial and open source licenses.
 *  You may use the GPL open source license described below or you may acquire
 *  a commercial license from Embedthis Software. You agree to be fully bound
 *  by the terms of either license. Consult the LICENSE.TXT distributed with
 *  this software for full details.
 *
 *  This software is open source; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. See the GNU General Public License for more
 *  details at: http://www.embedthis.com/downloads/gplLicense.html
 *
 *  This program is distributed WITHOUT ANY WARRANTY; without even the
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 *  This GPL license does NOT permit incorporating this software into
 *  proprietary programs. If you are unable to comply with the GPL, you must
 *  acquire a commercial license to use this software. Commercial licenses
 *  for this software and support services are available from Embedthis
 *  Software at http://www.embedthis.com
 *
 *  @end
 */
