/**
 * i2crw: versatile I2C communication tool.
 */

#include <stdio.h>
#include <string.h>
#include <i2c.h>
#include <stdlib.h>

void help(void)
{
	printf("i2crw [...options...] <busDevPath> <slaveChipAddress> [<subAddress>] [<byte1> <byte2> ...]\n\n");
	printf("  <busDevPath> is the device node path, e.g. /dev/i2c-0 (mandatory)\n");
	printf("  <slaveChipAddress> is the target chip address, e.g. 0x98 (mandatory if acting as master)\n");
	printf("  <subAddress> is an optional sub-address, e.g. 0xab\n");
	printf("  <byteX> are optional data bytes, e.g. 0xab, 198...\n\n");
	printf("Available options:\n");
	printf("  --page                        - Specify a page(in hex) for devices the require it\n");
	printf("  --blk <size>, --count <size>  - Read a block of data of size <size> (otherwise only 1 byte is read)\n");
	printf("  --no-subaddr                  - No sub-address needed/provided\n");
	printf("  --subaddr <8|16>              - Size of the sub-address in bits, default is 8\n");
	printf("  --16bits                      - <byte1>, <byte2> ... are actually 16bit words\n");
	printf("  --sccb                        - Target chip uses the SCCB protocol (a variant of I2C)\n");
	printf("                                  In this protocol the read operation is done in two transactions:\n");
	printf("                                  register addr. is first sent alone, then a restart cond. occurs, then the value can be read.\n");
	printf("  --slave-read <size>           - Read <size> bytes from the i2c slave device.\n");
	printf("                                  This request may stall until we are contacted by a master.\n"); 
	printf("                                  YOU CANNOT SPECIFY A CHIP ADDRESS to talk to with this command.\n");
	printf("  --slave-write                 - Specify bytes to be sent when i2c device is slave and is contacted\n");
	printf("                                  by a master on the bus. Same remarks as above.\n");
	printf("  --check                       - Use when reading data to check that the data read match <byte1> <byte2> etc.\n");
	printf("                                  Works only for master reading. Program will return the number of mismatches, 0 on success.\n");
	printf("  --quiet                       - Do not display any messages\n");
	printf("  --raw                         - Received bytes will be printed directly to STDOUT. This implies --quiet\n\n");
}

inline int roundup_div(const int numerator, const int denominator)
{
	return ( numerator == 0 
		? 0
		: 1 + ( numerator - 1 ) / denominator );
}

#define BYTES_PER_LINE 16
void print_bytes(unsigned char *buf, int count, int offset, int raw)
{
	int i, j;
	if (raw) {
		fwrite(buf, count, 1, stdout);
		return;
	}
	for (i=0; i<roundup_div(count, BYTES_PER_LINE); i++) {
		if (offset >= 0)
			printf("byte %03u, offset 0x%02x: ", (unsigned)(i*BYTES_PER_LINE), offset + i*BYTES_PER_LINE);
		else
			printf("byte %03u: ", (unsigned)(i*BYTES_PER_LINE));
		for (j=i*BYTES_PER_LINE; j<count && j < (i+1)*BYTES_PER_LINE; j++)
			printf("0x%02x ", buf[j]);
		printf("\n");
	}
}

#define BUFSIZE 32768

int main(int argc,char *argv[])
{
	i2c_handle_t fd;
	int i2c_addr = -1;
	int sub_addr = -1;
	int blk_size = -1;
	int page = -1;
	int i = 1;
	int ret = I2C_SUCCESS;
	char *bus = NULL;
	int write = 0;
	char quiet = 0, check = 0, option16 = 0;
	i2c_dev_t type = I2CDEV_STD;
	int slave_read = 0, slave_write = 0;
	unsigned char buf[BUFSIZE];
	memset(buf, 0, BUFSIZE);
	
	if (argc == 1) {
		help();
		return(0);
	}
	
	while (i < argc) {
		if ((strcmp("--help", argv[i]) == 0) ||  (strcmp("-h", argv[i]) == 0)) {
			help();
			return(0);
			
		} else if (strcmp("--sccb", argv[i]) == 0) {
			if (type!=I2CDEV_STD) {
				fprintf(stderr, "Invalid setting: this device type is not supported in SCCB mode: %d\n", type);
				return -1;
			}
			type = I2CDEV_SCCB;
			
		} else if (strcmp("--no-subaddr", argv[i]) == 0) {
			if (type!=I2CDEV_STD) {
				fprintf(stderr, "Invalid setting: you cannot combine --no-subadd with --sccb or --subaddr\n");
				return -1;
			}
			type = I2CDEV_NOSUBADDR;
			
		} else if (strcmp("--subaddr", argv[i]) == 0) {
			int subaddr_size=atol(argv[++i]);
			switch (subaddr_size) {
			case 8: 
				break;
			case 16:
				if(type!=I2CDEV_STD) {
					fprintf(stderr, "Invalid setting: this device type is not supported "
						"with 16bits addresses: %d\n", type);
					return -1;
				}
				type = I2CDEV_ADDR16;
				break;
			default:
				fprintf(stderr, "Invalid subaddress size: %d bits\n", subaddr_size);
				return -1;
			}

		} else if (strcmp("--page", argv[i]) == 0) {
			page = strtol(argv[++i], NULL, 16);
			if (page < 0) {
				fprintf(stderr, "page must be greater than or equal to zero\n");
				return -1;
			}

		} else if (strcmp("--16bits", argv[i]) == 0) {
			option16 = 1;
			
		} else if (strcmp("--blk", argv[i]) == 0 || strcmp("--count", argv[i]) == 0) {
			if (i == argc - 1 || sscanf(argv[++i], "%i", &blk_size) != 1 || blk_size < 0 || blk_size > BUFSIZE) {
				fprintf(stderr, "%s: please specify a valid byte count to block read (up to %d)\n", argv[0], BUFSIZE);
				return -1;
			}
			
		} else if (strcmp("--quiet", argv[i]) == 0) {
			if (!quiet)
				quiet = 1;
			
		} else if (strcmp("--raw", argv[i]) == 0) {
			quiet = 2;
			
		} else if (strcmp("--check", argv[i]) == 0) {
			check = 1;
			
		} else if (strcmp("--slave-read", argv[i]) == 0) {
			if (i == argc - 1 || sscanf(argv[++i], "%i", &slave_read) != 1 || slave_read < 0 || slave_read > BUFSIZE) {
				fprintf(stderr, "%s: please specify a valid byte count to raw read (up to %d)\n", argv[0], BUFSIZE);
				return -1;
			}
			
		} else if (strcmp("--slave-write", argv[i]) == 0) {
			if (i == argc - 1) {
				fprintf(stderr, "%s: please specify bytes to raw write (up to %d)\n", argv[0], BUFSIZE);
				return -1;
			}
			for (i++; i<argc; i++) {
				int x;
				if (sscanf(argv[i], "%i", &x) != 1 || (x & 0xff) != x) {
					fprintf(stderr, "%s: invalid byte value '%s', aborting\n", argv[0], argv[i]);
					return -1;
				}
				buf[slave_write++] = (x & 0xff);
			}
			
		} 
		else if (strcmp("--bus", argv[i]) == 0) 
			bus = argv[++i];
		else if (bus == NULL) 
			bus = argv[i];
			
		else if (strcmp("--addr", argv[i]) == 0)
			sscanf (argv[++i], "%i", &i2c_addr);
		else if (i2c_addr == -1)
			sscanf (argv[i], "%i", &i2c_addr);
			
		else if ((strcmp("--saddr", argv[i]) == 0) && type != I2CDEV_NOSUBADDR) {
			sscanf (argv[++i], "%i", &sub_addr);
			if (blk_size == -1)
				blk_size = 1;

		} 
		else if ((sub_addr == -1) && (type != I2CDEV_NOSUBADDR)) {
			sscanf (argv[i], "%i", &sub_addr);
			if (blk_size == -1)
				blk_size = 1;
		}
		else {
			int x;
			if (sscanf(argv[i], "%i", &x) != 1 || (!option16 && (x & 0xff) != x) || (option16 && (x & 0xffff) != x)) {
				fprintf(stderr, "%s: invalid byte value '%s', aborting\n", argv[0], argv[i]);
				return -1;
			}
			if (option16)
				buf[write++] = ((x & 0xff00) >> 8);
			buf[write++] = (x & 0xff);
		}
		i++;
	}

	if (slave_read) {
		FILE *fp;
		if (!quiet)
			printf("Going to read %d bytes from %s...\n", slave_read, bus);
		fp = fopen(bus, "rb");
		if (!fp) {
			fprintf(stderr, "%s: could not open %s for reading\n", argv[0], bus);
			return -1;
		}
		fread(buf, slave_read, 1, fp);
		fclose(fp);
		print_bytes(buf, slave_read, -1, quiet == 2);
		printf("\n");
		return 0;
	}
	
	if (slave_write) {
		FILE *fp;
		if (!quiet)
			printf("Going to write %d bytes to %s...\n", slave_write, bus);
		fp = fopen(bus, "wb");
		if (!fp) {
			fprintf(stderr, "%s: could not open %s for writing\n", argv[0], bus);
			return -1;
		}
		fwrite(buf, slave_write, 1, fp);
		fclose(fp);
		printf("\n");
		return 0;
	}
	
	if (check) {
		if (write != blk_size) {
			fprintf(stderr, "%s: error, when using --check, you have to provide"
					" as many bytes to check against as the byte count to read,\n", argv[0]);
			return -1;
		}
		write = 0;
	}
	
	fd = i2c_open(bus, i2c_addr, type);
	if (fd < 0) {   
		fprintf(stderr, "%s: failed to open I2C device: error %d\n", argv[0], fd);
		return -1;
	}
	
	if (write > 0) {
		if (!quiet) {
			printf("Going to write %d bytes to address 0x%x, ", write, i2c_addr);
			if (page != -1)
				printf("page 0x%x, subaddress 0x%x on %s...\n", 
					page, sub_addr, bus);
			else
				printf("subaddress 0x%x on %s...\n", sub_addr, bus);
		}

		if (page != -1)
			ret = i2c_write_page_block(fd, page, sub_addr, buf, write);
		else
			ret = i2c_write_block(fd, sub_addr, buf, write);
		i2c_close(fd);
		if (ret < 0) {
			fprintf(stderr, "%s: write failed with error %d\n", argv[0], ret);
			return ret;
		}
		
	} else if (blk_size > 0) {
		unsigned char *bufr = malloc(blk_size);
		if (!bufr) {
			fprintf(stderr, "%s: not enough memory!", argv[0]);
			return -1;
		}
		memset(bufr, 0, blk_size);
		if (!quiet) {
			printf("Going to read %d bytes at address 0x%x, ", blk_size, i2c_addr);
			if (page != -1)
				printf("page 0x%x, subaddress 0x%x on %s...\n", 
					page, sub_addr, bus);
			else
				printf("subaddress 0x%x on %s...\n", sub_addr, bus);
		}

		if (page != -1)
			ret = i2c_read_page_block(fd, page, sub_addr, bufr, blk_size);
		else
			ret = i2c_read_block(fd, sub_addr, bufr, blk_size);

		i2c_close(fd);
		if (ret < 0) {
			fprintf(stderr, "%s: read failed with error %d\n", argv[0], ret);
			free(bufr);
			return ret;
		}
		print_bytes(bufr, blk_size, sub_addr, quiet == 2);
		if (check) {
			ret = 0;
			if (!quiet)
				printf("Checking read...\n");
			for (i = 0; i < blk_size; i++) {
				if (buf[i] != bufr[i]) {
					ret++;
					if (!quiet)
						printf("!!! byte %03u: expected 0x%02x, got 0x%02x\n", 
								i, buf[i], bufr[i]);
				}
			}
			if (!quiet)
				printf("=== %d/%d bytes OK, %.02f%% errors ===\n", 
						blk_size - ret, blk_size, 100.f * (float)ret / (float)blk_size);
		}
		free(bufr);
	} else {
		fprintf(stderr, "%s: invalid options\n", argv[0]);
	}
	
	if (!quiet)
		printf("\n");
	
	return ret;
}
