/**i2crw.c
 * Example of how to use I2C linux kernel driver
 */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <i2c.h>

#define VERBOSE(arg...) if(verbose) printf(arg)

#define ERR_SYSTEM  (-1)
#define ERR_I2C     (-2)
#define ERR_PARSE   (-3)

#define MAX_NDATA       256
#define READ_BUFFER_ALLOC_SIZE  1024
#define MAXSTRINGSIZE   256
typedef struct  {
    int line;
    char cmd;
    int addr;
    unsigned int size;
    unsigned char data[MAX_NDATA];
    unsigned int u32[MAX_NDATA];
    char string[MAXSTRINGSIZE];
} cmd_t;


#define END_OF_LINE(buf)    ((*buf=='\n') || (*buf=='\r') || (*buf=='#') || (*buf=='\a') || (*buf==0))
#define END_OF_FILE(buf)    ((*buf=='\a') || (*buf==0))
#define IS_SPACE(buf)       ((*buf==' ') || (*buf=='\t'))
#define MARKERS_STRING(buf) (((*buf=='"') || (*buf=='\'')) && (*(buf-1)!='\\'))
#define SKIP_SPACES(buf)    { while(IS_SPACE(buf)) buf++; }
#define SKIP_COMMENT(buf)   if(*buf=='#') while((!END_OF_LINE(buf)) || (*buf=='#')) buf++;
#define FETCH_STRING(buf,str,line,arg) \
                            do { \
                                int len; \
                                char *start; \
                                if(!MARKERS_STRING(buf)) \
                                {   fprintf(stderr,"Syntax error line #%d, expected string as argument %d\n",line,arg); \
                                    return ERR_PARSE; \
                                }; \
                                start=++buf; \
                                while((!MARKERS_STRING(buf)) && (!END_OF_FILE(buf))) buf++; \
                                if(END_OF_FILE(buf)) \
                                {   fprintf(stderr,"Syntax error line #%d, non terminated string\n",line); \
                                    return ERR_PARSE; \
                                }; \
                                len=buf-start; \
                                buf++; \
                                if(len>=MAXSTRINGSIZE) len=MAXSTRINGSIZE-1; \
                                memcpy(str,start,len); \
                                str[len]=0; \
                            } while(0)
#define FETCH_INT(buf,intval,line,arg) \
                            do  { \
                                int val;\
                                if(sscanf(buf,"%i",&val)<1) \
                                {   fprintf(stderr,"Syntax error line #%d, expected integer as argument %d\n",line,arg); \
                                    return ERR_PARSE; \
                                }; \
                                intval=val; \
                                while((!IS_SPACE(buf)) && (!END_OF_LINE(buf))) buf++; \
                            } while(0)
#define GET_INT_VAL(buf,intval,line,arg) \
                            do  { \
                                int val;\
                                if(sscanf(buf,"%d",&val)<1) \
                                {   fprintf(stderr,"Syntax error line #%d, expected integer as argument %d\n",line,arg); \
                                    return ERR_PARSE; \
                                }; \
                                intval=val; \
                                while((!IS_SPACE(buf)) && (!END_OF_LINE(buf))) buf++; \
                            } while(0)
                                
                                
int parse_line(char **buffer, cmd_t *command)
{   char *string=*buffer;
    int count;
    char loop;
    
    command->line++;
    SKIP_SPACES(string);
    do
    {   loop=0;
        switch(*string)
        {   case 'Q':       /* quit command */
            case 'q':
                string++;
                SKIP_SPACES(string);
                if(!END_OF_LINE(string))
                    FETCH_STRING(string,command->string,command->line,1);
                else command->string[0]=0;
            case 0:         /* End of file */
            case '\a':
                command->cmd='Q';
                break;
            case 'A':
            case 'a':
                string++;
                command->cmd='A';
                SKIP_SPACES(string);
                FETCH_INT(string,command->addr,command->line,1);
                SKIP_SPACES(string);
                if(!END_OF_LINE(string))
                    FETCH_INT(string,command->data[0],command->line,2);
                else command->data[0]=0;
                break;
            case 'C':
            case 'c':
                string++;
                command->cmd='C';
                SKIP_SPACES(string);
                FETCH_INT(string,command->addr,command->line,1);
                SKIP_SPACES(string);
                for(count=0; (!END_OF_LINE(string)) && (!MARKERS_STRING(string)) ; count++)
                {   FETCH_INT(string,command->data[count],command->line,count+2);
                    SKIP_SPACES(string);
                };
                command->size=count;
                if(!END_OF_LINE(string))
                    FETCH_STRING(string,command->string,command->line,count+2);
                else command->string[0]=0;
                break;
            case 'D':
            case 'd':
                string++;
                command->cmd = 'D';
                SKIP_SPACES( string );
                for( count = 0; ! END_OF_LINE( string ) ; count++ )
                {
					GET_INT_VAL( string, command->u32[count], command->line, count+2 );
                    SKIP_SPACES( string );
                };
                command->size = count;
                break;
            case 'R':
            case 'r':
                string++;
                command->cmd='R';
                SKIP_SPACES(string);
                FETCH_INT(string,command->addr,command->line,1);
                SKIP_SPACES(string);
                FETCH_INT(string,command->size,command->line,1);
                SKIP_SPACES(string);
                if(!END_OF_LINE(string))
                    FETCH_STRING(string,command->string,command->line,2);
                else command->string[0]=0;
                break;
            case 'W':
            case 'w':
                string++;
                command->cmd='W';
                SKIP_SPACES(string);
                FETCH_INT(string,command->addr,command->line,1);
                SKIP_SPACES(string);
                for(count=0; !END_OF_LINE(string) ; count++)
                {   FETCH_INT(string,command->data[count],command->line,count+2);
                    SKIP_SPACES(string);
                };
                command->size=count;
                break;
            case '#':
                SKIP_COMMENT(string);
                command->line++;
                if(!END_OF_FILE(string))
                    string++;
                loop=1;
                break;
            case '\n':
                command->line++;
                if(!END_OF_FILE(string))
                    string++;
                loop=1;
                break;
            default:
                command->cmd=0;
                fprintf(stderr,"Error, unknown command %c, line #%d\n",*string,command->line);
                return ERR_PARSE;
        };
    } while(loop);
    SKIP_SPACES(string);
    SKIP_COMMENT(string);
    if(!END_OF_LINE(string))
    {   fprintf(stderr,"Expected end of line not found, line %d: %s\n",command->line,string);
        return ERR_PARSE;
    };
    if(!END_OF_FILE(string))
        string++;
    *buffer=string;
    return 0;
}

void help()
{   printf(
        "Arguments are:\n" \
        "\t[-d|--dev <i2c_bus>] [-v|--verbose] [-h|--help] <script>\n" \
        "\t-d | --dev <i2c_bus>: Name of the I2C bus device to open.\n" \
        "\t-v | --verbose: Program will describe what it is doing.\n" \
        "\t-h | --help: Display this help.\n" \
        "\t<script>: Script to run.\n" \
        "Script syntax:\n" \
        "A script is describing a list of Read/Writes to perform on the I2C bus opened\n" \
        "The first command of a script must be 'A' to set device address on I2C bus\n" \
        "addresses are sub_addresses and values are 8-bit int values (0x means hex, O means octal, nothing means decimal\n" \
        "A string must be started and terminated by "" or '\n" \
        "If a script line contain # the remaining of the line is ignored\n" \
        "\tA addr:\n" \
        "\t  Set I2C device address on the bus, you can change the address any time in a script.\n" \
        "\tC subaddr value1 ... valueN [string]:\n" \
        "\t  Read I2C and compare result to <value1> ..., if not equal display <string> (if not NULL) and exit\n" \
        "\tD <n>\n" \
        "\t  Delay, if n < 10 then delay n seconds else delay n milliseconds\n" \
        "\tR subaddr size [string]:\n" \
        "\t  Read <size> values on I2C, display <string> and values read (if <string> is not NULL).\n" \
        "\tW subaddr value1 ... valueN:\n" \
        "\t  Write N values at sub address <addr> (note that address should be automatically incremented by device).\n" \
        "\tQ [string]:\n" \
        "\t  Quit and display <string>. Note that EOF will be interpreted as a Q command with string=NULL\n" \
        "Return codes:\n" \
        "\t0 means script has been run successfully\n" \
        "\t>0 means check command failed at the line indicated by the positive return value\n" \
        "\t%d means a system error happened (file I/O, memory allocation ...).\n" \
        "\t%d means an I2C lib error occured.\n" \
        "\t%d means a parse error occured.\n" \
        "\n",
        ERR_SYSTEM,
        ERR_I2C,
        ERR_PARSE
    );
}

int main(int argc,char *argv[])
{   unsigned char i2c_addr;
    i2c_handle_t fd;
    int ret=I2C_SUCCESS;
    char *buffer=NULL,*tmpbuf;
    unsigned int textsize;
    char *bus_dev=NULL;
    char *szfile=NULL;
    FILE *file;
    int verbose=0;
    cmd_t line;
    int count;
    
    /* First we extract args */
    for(count=1;count<argc;count++)
    {   if(*argv[count]=='-')
        {   if(strcmp(argv[count],"-d")==0)
            {   count++;
                bus_dev=argv[count];
            }
            else if(strcmp(argv[count],"--dev")==0)
            {   count++;
                bus_dev=argv[count];
            }
            else if(strcmp(argv[count],"-v")==0)
                verbose=1;
            else if(strcmp(argv[count],"--verbose")==0)
                verbose=1;
            else if(strcmp(argv[count],"-h")==0)
            {   help();
                return 0;
            }
            else if(strcmp(argv[count],"--help")==0)
            {   help();
                return 0;
            }
            else 
            {   fprintf(stderr,"Unknown option %s\n",argv[count]);
                help();
                return ERR_SYSTEM;
            };
        } 
        else if(szfile==NULL)
            szfile=argv[count];
        else
        {   fprintf(stderr,"What is %s ? Check the number of arguments\n",argv[count]);
            help();
            return ERR_SYSTEM;
        };
    }
    if(szfile==NULL)
    {   fprintf(stderr,"Missing script name that is mandatory\n");
        help();
        return ERR_SYSTEM;
    };
    
    /* Now we read the file */
    file=fopen(szfile,"r");
    if(file==NULL)
    {   fprintf(stderr,"Failed to open %s: %s\n",szfile,strerror(errno));
        return ERR_SYSTEM;
    };
    textsize=0;
    buffer=(char *)realloc(buffer,READ_BUFFER_ALLOC_SIZE);
    if(buffer==NULL)
    {   fprintf(stderr,"Not enough memory to allocate %d bytes\n",READ_BUFFER_ALLOC_SIZE);
        return ERR_SYSTEM;
    };
    while((count=fread(buffer+textsize,sizeof(char),READ_BUFFER_ALLOC_SIZE,file))==READ_BUFFER_ALLOC_SIZE)
    {   textsize+=READ_BUFFER_ALLOC_SIZE;
        buffer=(char *)realloc(buffer,textsize+READ_BUFFER_ALLOC_SIZE);
        if(buffer==NULL)
        {   fprintf(stderr,"Not enough contiguous memory to store %d bytes\n",textsize+READ_BUFFER_ALLOC_SIZE);
            return ERR_SYSTEM;
        };
    };
    if(!feof(file))
    {   fprintf(stderr,"Failed to read %s: %s\n",szfile,strerror(errno));
        free(buffer);
        return ERR_SYSTEM;
    };
    textsize+=count;
    /* Here is the real stuff: parse file and send I2C commands */
    /* First we get the device address */
    tmpbuf=buffer;
    line.line=0;
    if(parse_line(&tmpbuf,&line)<0)
    {   fprintf(stderr,"Failed to parse script\n");
        free(buffer);
        return ERR_PARSE;
    };
    if(line.cmd!='A')
    {   fprintf(stderr,"First line must be 'A <i2c addr>'\n");
        free(buffer);
        return ERR_SYSTEM;
    };
    i2c_addr=line.addr;
    VERBOSE("i2c address of device is 0x%0x, type is %d\n",i2c_addr,line.data[0]);
    fd=i2c_open(bus_dev,i2c_addr,line.data[0]);
    if(fd<0)
    {   fprintf(stderr,"Failed to open I2C device on bus %s: error %d\n",bus_dev,fd);
        free(buffer);
        return ERR_I2C;
    };
    for(count=0; line.cmd!='Q'; count++)
    {   if(parse_line(&tmpbuf,&line)<0)
        {   fprintf(stderr,"Failed to parse script\n");
            free(buffer);
            return ERR_PARSE;
        };
        switch(line.cmd)
        {   case 'A':   /* change dev addr */
                VERBOSE("Change i2c address from 0x%0x to 0x%0x (dev type is %d)\n",i2c_addr,line.addr,line.data[0]);
                i2c_close(fd);
                i2c_addr=line.addr;
                fd=i2c_open(bus_dev,i2c_addr,line.data[0]);
                if(fd<0)
                {   fprintf(stderr,"Failed to open I2C device on bus %s: error %d\n",bus_dev,fd);
                    free(buffer);
                    return ERR_I2C;
                };
                break;
            case 'R':   /* Read a value */
            {   int i;
            
                VERBOSE("Read address 0x%0x\n",line.addr);
                ret=i2c_read_block(fd,line.addr,line.data,line.size);
                if(ret<0)
                {   fprintf(stderr,"Failed to read I2C device at address 0x%02x on bus %s: error %d\n",i2c_addr,bus_dev,ret);
                    free(buffer);
                    return ERR_I2C;
                };
                if(line.string[0]!=0)
                    printf("%s: ",line.string);
                for(i=0;i<line.size;i++)
                    printf("0x%02x ",line.data[i]);
                printf("\n");
                break;
            }
            case 'C':   /* Check a register */
            {   int i;
                unsigned char *tmp=(char *)malloc(line.size);
                VERBOSE("Check %d bytes at address 0x%0x\n",line.size,line.addr);
                if(tmp==NULL)
                {   fprintf(stderr,"Failed to alloc memory\n");
                    free(buffer);
                    return ERR_SYSTEM;
                };
                ret=i2c_read_block(fd,line.addr,tmp,line.size);
                if(ret<0)
                {   fprintf(stderr,"Failed to read I2C device at address 0x%02x on bus %s: error %d\n",i2c_addr,bus_dev,ret);
                    free(tmp);
                    free(buffer);
                    return ERR_I2C;
                };
                for(i=0;i<line.size;i++)
                    if(tmp[i]!=line.data[i])
                    {   VERBOSE("Check error at addr 0x%x, read 0x%02x expected 0x%02x\n",line.addr+i,tmp[i],line.data[i]);
                        if(line.string[0]!=0)
                            printf("%s\n",line.string);
                        free(tmp);
                        free(buffer);
                        return line.line;
                    };
                free(tmp);
                break;
            }
            case 'D':   /* delay for the specified number of ms. */
            {
				int i, ret;
			    struct timespec sleeptime;
				for( i = 0; i < line.size; i++ )  {
					if( (line.u32[i] < 10) && (line.u32[i] > 0) )  {
						VERBOSE( "Delay %d seconds\n", line.u32[i] );
						sleeptime.tv_sec  = line.u32[i];
						sleeptime.tv_nsec = 0;
					}
					else if( (line.u32[i] < 1000) && (line.u32[i] > 0) )  {
						VERBOSE( "Delay %d milliseconds\n", line.u32[i] );
						sleeptime.tv_sec  = 0;
						sleeptime.tv_nsec = line.u32[i] * 1000000;
					}
					else  {
						printf( "i2cprog: delay value specified: %d, is out of range\n", line.u32[i] );
						continue;
					}
					ret = nanosleep( &sleeptime, NULL );
					if( ret )
						printf( "i2cprog: nanosleep returned error %d\n", ret );
				}
                break;
            }
            case 'W':   /* Write data */
                VERBOSE("Write %d bytes at address 0x%0x (first is 0x%0x)\n",line.size,line.addr,line.data[0]);
                ret=i2c_write_block(fd,line.addr,line.data,line.size);
                if(ret<0)
                {   fprintf(stderr,"Failed to write I2C device at address 0x%02x on bus %s: error %d\n",i2c_addr,bus_dev,ret);
                    free(buffer);
                    return ERR_I2C;
                };
                break;
            case 'Q':   /* Quit */
                VERBOSE("End of script\n");
                i2c_close(fd);
                if(line.string[0]!=0)
                    printf("%s\n",line.string);
                break;        
            default:
                fprintf(stderr,"Unknown command %c\n",line.cmd);
                free(buffer);
                return ERR_SYSTEM;
        };
    };
    VERBOSE("Executed %d commands from script %s\n",count,szfile);
    free(buffer);
    printf("\n");
    return 0;
}
