#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/proc_fs.h>
#include <asm/arch/platform.h>
#include <asm/uaccess.h>
#include <linux/kthread.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <linux/net.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <linux/mm.h>
#include <asm/fcntl.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include "kaud.h"
/*=======================================================================================================*/
static struct platform_device kaud_thread_device = {
        .name   = "kaudthread",
        .id     = 10,
        .dev    = {
                .platform_data  = NULL,
                .release = NULL,
        },
        .num_resources = 0,
};
/*-------------------------------------------------------------------------------------------------------*/
char QBOX_HEADER[] = {
	0x00, 0x00, 0x00, 0x00, 0x71, 0x62, 0x6F, 0x78, 0x01, 0x00, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01,
	0x00, 0x00, 0x02, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x71, 0x6D, 0x65, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x10
};

int QBOX_HEADER_SIZE = sizeof(QBOX_HEADER);
unsigned char   audbuf[200*1024];
gbuffer_table_t *kaud_gbuf = NULL;
kthread_t	kthd_params[2];
/*=======================================================================================================*/
#ifdef AUDIO_SRC_FROM_FILE
static void read_srcaud_from_file(void)
{
	int	len;
	struct file	*r_f = NULL;
	mm_segment_t	orig_fs;

	r_f = filp_open(AUDIO_FILE_NAME, O_RDWR|O_CREAT, 00);
	if(!r_f || !r_f->f_op || !r_f->f_op->read)
		DBG(INIT, "open %s error\n", AUDIO_FILE_NAME);

	r_f->f_pos = 0;

	orig_fs = get_fs();
	set_fs(KERNEL_DS);

	len = r_f->f_op->read(r_f, audbuf, 100*1024, &r_f->f_pos);
	DBG(INIT, "read out %s len: %d\n", AUDIO_FILE_NAME, len);

	set_fs(orig_fs);
	fput(r_f);
}
#else
static struct socket* set_up_udpserver_socket(int port_no)
{
	struct socket *sock;
	struct sockaddr_in sin;
	int error;

	/* First create a socket */
	error = sock_create(PF_INET, SOCK_DGRAM, IPPROTO_UDP, &sock) ;
	if (error < 0)
		DBG(RCVTHREAD, "Error during creation of socket; terminating\n");

	/* Now bind the socket */
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(port_no);

	error = sock->ops->bind(sock, (struct sockaddr*)&sin, sizeof(sin));
	if (error < 0)
	{
		DBG(RCVTHREAD, "Error binding socket\n");
		return 0;
	}

	return sock;
}

static struct socket* set_up_tcpserver_socket(int port_no)
{
	struct socket *sock;
	struct sockaddr_in sin;
	int error;

	/* First create a socket */
	error = sock_create(PF_INET, SOCK_STREAM, IPPROTO_TCP, &sock) ;
	if (error < 0)
		DBG(RCVTHREAD, "Error during creation of socket; terminating\n");

	/* Now bind the socket */
	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = INADDR_ANY;
	sin.sin_port = htons(port_no);

	error = sock->ops->bind(sock, (struct sockaddr*)&sin, sizeof(sin));
	if (error < 0)
	{
		DBG(RCVTHREAD, "Error binding socket\n");
		return 0;
	}

	/* Now, start listening on the socket */
	error=sock->ops->listen(sock, 5);
	if (error != 0)
		DBG(RCVTHREAD, "Error listening on socket\n");

	return sock;
}

static struct socket* server_accept_connection(struct socket *sock)
{
	struct socket * newsock;
	int error;

	/* Before accept: Clone the socket */
	error = sock_create(PF_INET, SOCK_STREAM,IPPROTO_TCP, &newsock);
	if (error < 0)
		DBG(RCVTHREAD, "Error during creation of the other socket; terminating\n");

	newsock->type = sock->type;
	newsock->ops = sock->ops;

	/* Do the actual accept */
	error = newsock->ops->accept(sock, newsock, 0);
	if (error < 0)
	{
		/* must add release the socket code here */
		DBG(RCVTHREAD, "Error accepting socket\n");
		return 0;
	}

	return newsock;
}

static int setup_server_daemon(net_def_t *np)
{
	int ret = 0;

	if(np->type == TCP)
	{
		np->sock_b = set_up_tcpserver_socket(np->port);
		np->sock_t = server_accept_connection(np->sock_b);
	}
	else
		np->sock_t = set_up_udpserver_socket(np->port);

	if(np->sock_t == NULL)
		ret = -1;

	return ret;
}

static size_t recvbuffer(struct socket *sock, unsigned char *buffer, size_t length)
{
	struct msghdr msg;
	struct iovec iov;
	int len, ret;
	mm_segment_t oldfs;

	/* Set the msghdr structure*/
	msg.msg_name = 0;
	msg.msg_namelen = 0;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags = 0;

	/* Set the iovec structure*/
	iov.iov_base = (void *) &buffer[0];
	iov.iov_len = (size_t)length;

	/* Recieve the message */
	oldfs = get_fs();
	set_fs(KERNEL_DS);

	len = 0;
	do {
		ret = sock_recvmsg(sock, &msg, length-len, 0/*MSG_DONTWAIT*/); // let it wait if there is no message
		//ret = sock_recvmsg(sock, &msg, length-len, MSG_DONTWAIT);
		if(ret==-EAGAIN || ret <0)
			break;

		len += ret;
		if(len < length)
		{
			iov.iov_base = (void *)(buffer+len);
			iov.iov_len = (size_t)(length-len);
		}
	} while(len < length);

	set_fs(oldfs);

	if((ret!=-EAGAIN) && (len!=0))
		DBG(RCVTHREAD, "RecvBuffer Recieved %i bytes\n", len);
	else
	{
		DBG(RCVTHREAD, "client terminate\n");
		len = -1; 
	}

	return len;
}
#endif
/*-------------------------------------------------------------------------------------------------------*/
#ifdef AUDIO_SRC_FROM_FILE
static int recv_gbuf_from_file(gbuffer_index_t *dest)
{
	int content_length = GBUFFER_UNIT_SIZE;
	static int count = 0;

	memcpy(dest->buf+QBOX_HEADER_SIZE, audbuf + count, content_length);
        QBOX_HEADER[0] = ((content_length + QBOX_HEADER_SIZE)  >> 24) & 0xff;
        QBOX_HEADER[1] = ((content_length + QBOX_HEADER_SIZE) >> 16) & 0xff;
        QBOX_HEADER[2] = ((content_length + QBOX_HEADER_SIZE) >> 8) & 0xff;
        QBOX_HEADER[3] = (content_length + QBOX_HEADER_SIZE) & 0xff;		
	memcpy (dest->buf, QBOX_HEADER, QBOX_HEADER_SIZE);
	
	if((count+=GBUFFER_UNIT_SIZE) >= 40*1024)	count = 0;

	return count;	
}
#else
static int recv_gbuf_from_net(struct socket *sock, gbuffer_index_t *dest)
{
	int content_length = GBUFFER_UNIT_SIZE, ret;

	ret = recvbuffer(sock, dest->buf+QBOX_HEADER_SIZE, content_length);
	if(ret < 0)
		return -1;

        QBOX_HEADER[0] = ((content_length + QBOX_HEADER_SIZE)  >> 24) & 0xff;
        QBOX_HEADER[1] = ((content_length + QBOX_HEADER_SIZE) >> 16) & 0xff;
        QBOX_HEADER[2] = ((content_length + QBOX_HEADER_SIZE) >> 8) & 0xff;
        QBOX_HEADER[3] = (content_length + QBOX_HEADER_SIZE) & 0xff;
	
	memcpy (dest->buf, QBOX_HEADER, QBOX_HEADER_SIZE);

	return 0;
}
#endif

static int copy_gbuf_to_aud_fifo(struct file *w_f, gbuffer_index_t *src)
{
	int len, content_length = GBUFFER_UNIT_SIZE, ret;
	//struct file *w_f = NULL;
	mm_segment_t orig_fs;

	//w_f = filp_open(ANTIAUDIO_FIFO_NAME, O_RDWR|O_CREAT, 00);
	if(!w_f || !w_f->f_op || !w_f->f_op->write)
	{
		DBG(SNDTHREAD, "SND write to fifo error\n");
		return 0;
	}

	//w_f->f_pos = 0;

	orig_fs = get_fs();
	set_fs(KERNEL_DS);

	len = 0;
	do {
		ret = w_f->f_op->write(w_f, src->buf, content_length+QBOX_HEADER_SIZE-len, &w_f->f_pos);	
		len += ret;
	} while(len < (content_length+QBOX_HEADER_SIZE));
	
	set_fs(orig_fs);
	//fput(w_f);

	return len;
}
/*-------------------------------------------------------------------------------------------------------*/
static int RCV_kthd_func(void *param)
{
	//gbuffer_index_t	*head, *tail;
	int	threshold, cli_out = 0;
	unsigned long	sig;
	siginfo_t	info;

	DBG(RCVTHREAD, "RCV kthread enter\n");

	//siginitsetinv(&current->blocked, sigmask(SIGKILL)|sigmask(SIGINT)|sigmask(SIGTERM));
	allow_signal(SIGUSR1);
	allow_signal(SIGUSR2);

#ifdef AUDIO_SRC_FROM_FILE
	read_srcaud_from_file();
#else
	if(setup_server_daemon(&(((kthread_t *)param)->snd_rcv_args.netarg)) < 0)
	{
		DBG(RCVTHREAD, "RCV net daemon fail\n");
		return 0;
	}
#endif

	while(1)
	{
		if(signal_pending(current))
		{
			sig = dequeue_signal(current, &current->blocked, &info);
			if(sig == SIGUSR1)
				((kthread_t *)param)->cmd = PLAY;
			else if(sig == SIGUSR2)
				((kthread_t *)param)->cmd = STOP;
		}

#ifndef AUDIO_SRC_FROM_FILE
		if(cli_out<0 && ((kthread_t *)param)->snd_rcv_args.netarg.type==TCP)
		{
			sock_release(((kthread_t *)param)->snd_rcv_args.netarg.sock_t);
			((kthread_t *)param)->snd_rcv_args.netarg.sock_t = server_accept_connection(((kthread_t *)param)->snd_rcv_args.netarg.sock_b);
			if(((kthread_t *)param)->snd_rcv_args.netarg.sock_t == NULL)
			{
				DBG(RCVTHREAD, "RCV net daemon accept fail\n");
				return 0;
			}
		}
#endif

		if(kthread_should_stop())
		{
			((kthread_t *)param)->terminate = 1;
#ifndef AUDIO_SRC_FROM_FILE
			if(((kthread_t *)param)->snd_rcv_args.netarg.type == TCP)
				sock_release(((kthread_t *)param)->snd_rcv_args.netarg.sock_b);
			sock_release(((kthread_t *)param)->snd_rcv_args.netarg.sock_t);
#endif
			break;
		}
		else if(((kthread_t *)param)->cmd == STOP)
		{
			down_interruptible(&kaud_gbuf->_gbufMutex);

#ifdef AUDIO_SRC_FROM_FILE
			recv_gbuf_from_file(kaud_gbuf->head);
#else
			cli_out = recv_gbuf_from_net(((kthread_t *)param)->snd_rcv_args.netarg.sock_t, kaud_gbuf->head);
			if(cli_out < 0)
			{
				up(&kaud_gbuf->_gbufMutex);
				SCH_TMOUT(2)
				continue;
			}
#endif
			kaud_gbuf->head->new = 1;
			kaud_gbuf->head = kaud_gbuf->head->next;

			up(&kaud_gbuf->_gbufMutex);
			
			SCH_TMOUT(1)
		}
		else if(((kthread_t *)param)->cmd == PLAY)
		{
			down_interruptible(&kaud_gbuf->_gbufMutex);

#ifdef AUDIO_SRC_FROM_FILE
			recv_gbuf_from_file(kaud_gbuf->head);
#else
			cli_out = recv_gbuf_from_net(((kthread_t *)param)->snd_rcv_args.netarg.sock_t, kaud_gbuf->head);
			if(cli_out < 0)
			{
				up(&kaud_gbuf->_gbufMutex);
				SCH_TMOUT(2)
				continue;
			}
#endif
			kaud_gbuf->head->new = 1;
			kaud_gbuf->head = kaud_gbuf->head->next;
			if(kaud_gbuf->head->idx == kaud_gbuf->tail->idx)
				kaud_gbuf->tail = kaud_gbuf->tail->next;

			up(&kaud_gbuf->_gbufMutex);

			threshold = ((kaud_gbuf->head->idx-kaud_gbuf->tail->idx) < 0) ? (GBUFFER_IDX_MAX+kaud_gbuf->head->idx-kaud_gbuf->tail->idx) : (kaud_gbuf->head->idx-kaud_gbuf->tail->idx);
			if(threshold >= RCV_THREAD_F_THRESHOLD)
				SCH(TMOUT,5)
			else
				SCH(TMOUT,1)
		}
	}
	
	DBG(RCVTHREAD, "RCV kthread exit\n");

	return 0;
}

static int SND_kthd_func(void *param)
{
	int 	len, threshold;
	unsigned long	sig;
	siginfo_t	info;
	//gbuffer_index_t	*head, *tail;

	DBG(SNDTHREAD, "SND kthread enter\n");

	//siginitsetinv(&current->blocked, sigmask(SIGKILL)|sigmask(SIGINT)|sigmask(SIGTERM));
	allow_signal(SIGUSR1);
	allow_signal(SIGUSR2);

	while(1)
	{
		if(signal_pending(current))
		{
			sig = dequeue_signal(current, &current->blocked, &info);
			if(sig == SIGUSR1)
				((kthread_t *)param)->cmd = PLAY;
			else if(sig == SIGUSR2)
				((kthread_t *)param)->cmd = STOP;
		}

		if(kthread_should_stop())
		{
			((kthread_t *)param)->terminate = 1;
			fput(((kthread_t *)param)->snd_rcv_args.fifoarg.wfp);
			break;
		}
		else if(((kthread_t *)param)->cmd == STOP)
			interruptible_sleep_on(&(((kthread_t *)param)->queue));
		else if(((kthread_t *)param)->cmd == PLAY)
		{
			down_interruptible(&kaud_gbuf->_gbufMutex);

			if (kaud_gbuf->head->idx == kaud_gbuf->tail->idx)
			{
				up(&kaud_gbuf->_gbufMutex);
				interruptible_sleep_on_timeout(&(((kthread_t *)param)->queue), SND_THREAD_TIMEOUT);
				continue;
			}
			else if(kaud_gbuf->tail->new == 1)
			{
				len = copy_gbuf_to_aud_fifo(((kthread_t *)param)->snd_rcv_args.fifoarg.wfp, kaud_gbuf->tail);	//PLAY but no LUA_PlayVoice( )## that will stop here after few time
				kaud_gbuf->tail->new = 0;
				kaud_gbuf->tail = kaud_gbuf->tail->next;
			}

			up(&kaud_gbuf->_gbufMutex);

			threshold = ((kaud_gbuf->head->idx-kaud_gbuf->tail->idx) < 0) ? (GBUFFER_IDX_MAX+kaud_gbuf->head->idx-kaud_gbuf->tail->idx) : (kaud_gbuf->head->idx-kaud_gbuf->tail->idx);
			if(threshold <= SND_THREAD_E_THRESHOLD)
				SCH(TMOUT,25)
			else
				SCH(TMOUT,2)
		}
	}
	
	DBG(SNDTHREAD, "SND kthread exit\n");

	return 0;
}

static int proc_read(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int	len = 0;

	if(!strcmp((char *)data, CFG_PROC_GBUF))
	{
		len = sprintf(page, "head.idx:%d tail.idx:%d head.new:%d tail.new:%d\n", kaud_gbuf->head->idx, kaud_gbuf->tail->idx, kaud_gbuf->head->new, kaud_gbuf->tail->new);
	}
	else
	{
		len = sprintf(page, "SND: cmd:%d terminate:%d\n", kthd_params[SND].cmd, kthd_params[SND].terminate);
		len += sprintf(page+len, "RCV: cmd:%d terminate:%d net:%d %d\n", kthd_params[RCV].cmd, kthd_params[RCV].terminate, kthd_params[RCV].snd_rcv_args.netarg.type, kthd_params[RCV].snd_rcv_args.netarg.port);
	}

	return len;
}
/*-------------------------------------------------------------------------------------------------------*/
static void proc_alloc(void)
{
	static int	first = 0;
	char	path[128] = {CFG_PROC_ENTRY "/"};
	int	p;

	if (!first)
	{
		if(!proc_mkdir(CFG_PROC_ENTRY, 0))	return;
		first = 1;
	}

	p = strlen(path);
	strcpy(path+p, CFG_PROC_GBUF);
	create_proc_read_entry(path, 0, 0, proc_read, CFG_PROC_GBUF);
	strcpy(path+p, CFG_PROC_KTHD);
	create_proc_read_entry(path, 0, 0, proc_read, CFG_PROC_KTHD);
}

static void proc_free(void)
{
	char path[128] = {CFG_PROC_ENTRY "/"};
	int 	p;

	p = strlen(path);
	strcpy(path+p, CFG_PROC_GBUF);
	remove_proc_entry(path, NULL);
	strcpy(path+p, CFG_PROC_KTHD);
	remove_proc_entry(path, NULL);
}

static void setup_kthd_area(int type)
{
	kthd_params[type].cmd = STOP;
	kthd_params[type].terminate = 0;
	kthd_params[type].arg = NULL;
	init_waitqueue_head(&(kthd_params[type].queue));
	init_MUTEX(&(kthd_params[type]._semMutex));

	if(type == SND)
	{
		kthd_params[type].thread = kthread_create(SND_kthd_func, (void *)&kthd_params[type], "SND_kthread");
		kthd_params[type].snd_rcv_args.fifoarg.wfp = filp_open(ANTIAUDIO_FIFO_NAME, O_RDWR|O_CREAT, 00);
		kthd_params[type].snd_rcv_args.fifoarg.wfp->f_pos = 0;
	}
	else
	{
		kthd_params[type].thread = kthread_create(RCV_kthd_func, (void *)&kthd_params[type], "RCV_kthread");
		kthd_params[type].snd_rcv_args.netarg.type = UDP;
		kthd_params[type].snd_rcv_args.netarg.port = PORT1;
		kthd_params[type].snd_rcv_args.netarg.sock_b = NULL;
		kthd_params[type].snd_rcv_args.netarg.sock_t = NULL;
	}

	wake_up_process(kthd_params[type].thread);
}

static gbuffer_table_t* gbuffer_init(int max, int size)
{
	int	space, i;
	u32	offset;
	gbuffer_table_t *gbuf;
	gbuffer_index_t *index;

	space = sizeof(gbuffer_table_t) + sizeof(gbuffer_index_t)*max;
	gbuf = (gbuffer_table_t *)KALLOC(space, GFP_ATOMIC);
	if(!gbuf)	return 0;
	memset((void *)gbuf, 0, space);

	//gbuf->va = (u32)consistent_alloc(GFP_DMA|GFP_ATOMIC, max*size, (dma_addr_t *)&gbuf->pa);
	gbuf->va = (u32)dma_alloc_coherent(0, max*size, (dma_addr_t *)&gbuf->pa, GFP_DMA | GFP_ATOMIC);

	if(!gbuf->va)
	{
		kfree(gbuf);
		return 0;
	}

	gbuf->max = max;
	gbuf->unit_size = size;
	init_MUTEX(&gbuf->_gbufMutex);
	DBG(INIT, "gbuf:%08X va:%08X pa:%08X max:%d unitsize:%d\n", (u32)gbuf, gbuf->va, gbuf->pa, gbuf->max, gbuf->unit_size);

	for(i=0, offset=0, index=gbuf->index; i<gbuf->max; i++, offset+=gbuf->unit_size)
	{
		index[i].buf = (u8 *)(gbuf->va+offset);
		index[i].phy = gbuf->pa+offset;
		index[i].idx = i;
		index[i].new = 0;
		index[i].pre = &index[(i+gbuf->max-1)%gbuf->max];
		index[i].next = &index[(i+1)%gbuf->max];
		DBG(INIT, "(%d):%08X %08X %d %08X %08X\n", i, (u32)(index[i].buf), index[i].phy, index[i].idx, (u32)(index[i].pre), (u32)(index[i].next));
	}

	gbuf->head = &index[0];
	gbuf->tail = &index[0];

	return gbuf;
}

static int create_aud_fifo_file(void)
{
	return kaud_mknod(AT_FDCWD, ANTIAUDIO_FIFO_NAME, S_IFIFO|0777, 0, sizeof(ANTIAUDIO_FIFO_NAME));
}

static void release_aud_fifo_file(void)
{
	kaud_unlink(AT_FDCWD, ANTIAUDIO_FIFO_NAME, sizeof(ANTIAUDIO_FIFO_NAME));
}

static void gbuffer_release(void)
{
	if(!kaud_gbuf)
	{
		DBG(INIT, "gbuf is null pointer\n");
		return;
	}

	//consistent_alloc(kaud_gbuf->va, kaud_gbuf->max*kaud_gbuf->unit_size, (dma_addr_t *)&kaud_gbuf->pa);
	dma_free_coherent(0, kaud_gbuf->max*kaud_gbuf->unit_size, (void *)kaud_gbuf->va, kaud_gbuf->pa);
	KFREE(kaud_gbuf, sizeof(gbuffer_table_t) + sizeof(gbuffer_index_t)*kaud_gbuf->max);
}

static void release_kthd_area(int type)
{
	if(kthd_params[type].thread != NULL)
		kthread_stop(kthd_params[type].thread);
}
/*=======================================================================================================*/
static int __init kaud_init(void)
{
	if(create_aud_fifo_file() == 0)
		DBG(INIT, "create %s ok\n", ANTIAUDIO_FIFO_NAME);
	else
		return 0;

	kaud_gbuf = gbuffer_init(GBUFFER_IDX_MAX, QBOX_HEADER_SIZE+GBUFFER_UNIT_SIZE);
	if(kaud_gbuf != NULL)
		DBG(INIT, "create gitter buffer ok\n");
	else
		return 0;

	setup_kthd_area(SND);
	setup_kthd_area(RCV);

	proc_alloc();

	//platform_device_register(&kaud_thread_device);
	DBG(INIT, "kernel aud init\n");

	return 0;
}

static void __exit kaud_exit(void) 
{
	release_kthd_area(SND);
	release_kthd_area(RCV);
	release_aud_fifo_file();
	gbuffer_release();
	proc_free();

	//platform_device_unregister(&kaud_thread_device);
	DBG(INIT, "kernel aud exit\n");
}

module_init(kaud_init);
module_exit(kaud_exit);

MODULE_AUTHOR("C.Y.Chen");
MODULE_DESCRIPTION("Driver for Pixord voice play");
MODULE_LICENSE("GPL");
