/**
 * setAK4633.c
 * Utility to configure the AK4633 audio ADC on mobicam2
 */

/*******************************************************************************
*
* The content of this file or document is CONFIDENTIAL and PROPRIETARY
* to Mobilygen Corporation.  It is subject to the terms of a
* License Agreement between Licensee and Mobilygen Corporation.
* restricting among other things, the use, reproduction, distribution
* and transfer.  Each of the embodiments, including this information and
* any derivative work shall retain this copyright notice.
*
* Copyright 2005-2007 Mobilygen Corporation.
* All rights reserved.
*
* QuArc is a registered trademark of Mobilygen Corporation.
*
*******************************************************************************/

#include <linux/gpio.h>
#include <linux/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>

#define GPIO_CSN_NAME  "/dev/gpio/native/gpio071"			// AUD_CSN uses gpio_2_31
#define GPIO_CCLK_NAME "/dev/gpio/native/gpio070"			// AUD_CCLK uses gpio_2_30
#define GPIO_CDTI_NAME "/dev/gpio/native/gpio069"			// AUD_CDTI uses gpio_2_29

#define VERBOSE(arg...) do { if(verbose) { printf(arg); printf("\n"); }; } while(0)
#define ERROR(arg...) do { fprintf(stderr,arg); fprintf(stderr,"\n"); } while(0)

static int samplingFrequency = 48000;

/******************************
 *
 * help
 *
 * Display program usage
 *
 *****************************/

void help(
    char *prog
)
{
    static int once = 0;
    if( once++ == 0 )  {
        printf( "setAK4633  Configure AK4633 Audio ADC on Mobicam2  v1.0.1\n" );
        printf( "Syntax is: %s [ --verbose | --help | -sf <samplingfrequency> --file <filename> ]\n", prog );
        printf( "\t--verbose or -v: Verbose output\n" );
        printf( "\t-sf: Use the specified sampling frequency in samples/sec - default is 48000\n" );
        printf( "\t--file or -f: Take register programming from the specified text file\n" );
        printf( "\t--help or -h: Display this help\n" );
        printf( "\n" );
    }
}


/******************************
 *
 * wrAK4633
 *
 * Write date to AK4633.  For details on the protocol used see the AK4633 data sheet.
 *
 * in       csn: file descriptor for chip select GPIO
 *			clk: file descriptor for data clock GPIO
 *			dti: file descriptor for data GPIO
 *      address: AK4633 register to write
 *         data: date to write
 *
 *****************************/

int wrAK4633(
    int csn,
    int clk,
    int dti,
    int verbose,
    int address,
    char data
)
{
    unsigned int i;
    int res;

    VERBOSE( "wrAK4633: writing 0x%02X to register 0x%02X", data, address );

    // Chip Select
    res  = ioctl( csn, GPIO_CLEAR );					// Chip select active

    // C1
    res |= ioctl( clk, GPIO_CLEAR );					// CS1 = 1
    res |= ioctl( dti, GPIO_SET );
    res |= ioctl( clk, GPIO_SET );

    // C0
    res |= ioctl( clk, GPIO_CLEAR );					// CS0 = 0
    res |= ioctl( dti, GPIO_CLEAR );
    res |= ioctl( clk, GPIO_SET );

    // R/W
    res |= ioctl( clk, GPIO_CLEAR );					// 1 = write
    res |= ioctl( dti, GPIO_SET );
    res |= ioctl( clk, GPIO_SET );

    // Address
    for( i = 0x10; i > 0; i >>= 1 )  {					// write 5 bit register address
        res |= ioctl( clk, GPIO_CLEAR );
        if( address & i )
            res |= ioctl( dti, GPIO_SET );
        else
            res |= ioctl( dti, GPIO_CLEAR );
        res |= ioctl( clk, GPIO_SET );
    }

    // Data
    for( i = 0x80; i > 0; i >>= 1 )  {					// write 8 bits of data
        res |= ioctl( clk, GPIO_CLEAR );
        if( data & i )
            res |= ioctl( dti, GPIO_SET );
        else
            res |= ioctl( dti, GPIO_CLEAR );
        res |= ioctl( clk, GPIO_SET );
    }

    // Chip Select
    res |= ioctl( csn, GPIO_SET );						// Chip select inactive

    return res;
}


/******************************
 *
 * main
 *
 *****************************/

int main(
    int argc,
    char *argv[]
)
{
    FILE * fs = NULL;
    int verbose = 0;
    int cclk_fd;
    int cdti_fd;
    int csn_fd;
    int count;
    int res;

    // First process the command line arguments
    for( count = 1; count < argc; count++ )  {
        if( (strcmp( argv[count], "-v" ) == 0) || (strcmp( argv[count], "--verbose" ) == 0) )
            verbose = 1;
        else if( (strcmp( argv[count], "-f") == 0 ) || (strcmp( argv[count], "--file" ) == 0) )  {
            fs = fopen( argv[++count], "r" );
            if( fs < 0 )  {
                ERROR( "setAK4633: unable to open '%s' for input\n", argv[count] );
                return -1;
            }
        }
        else if( strcmp( argv[count], "-sf") == 0 ) {
            samplingFrequency = atoi(argv[++count]);
        }

        else if( (strcmp( argv[count], "-h" ) == 0 ) || (strcmp( argv[count], "--help" ) == 0) )
        {
            help( argv[0] );
            return 1;
        }
        else
        {
            ERROR( "Unknown option %s", argv[count] );
            help( argv[0] );
            return -1;
        };
    }

    VERBOSE( "%s: compiled on %s at %s",  argv[0], __DATE__, __TIME__ );
    VERBOSE( "Using GPIO %s for AUD_CSN",  GPIO_CSN_NAME );
    VERBOSE( "Using GPIO %s for AUD_CCLK", GPIO_CCLK_NAME );
    VERBOSE( "Using GPIO %s for AUD_CDTI", GPIO_CDTI_NAME );

    // Open GPIO drivers that are used to control the GPIO lines connected to the AK4633
    csn_fd = open( GPIO_CSN_NAME, O_WRONLY );
    if( csn_fd < 0 )  {
        ERROR( "Failed to open GPIO_CSN_NAME gpio %s, open returned %d", GPIO_CSN_NAME, csn_fd );
        return -1;
    };
    cclk_fd = open( GPIO_CCLK_NAME, O_WRONLY );
    if( cclk_fd < 0 )  {
        ERROR( "Failed to open GPIO_CCLK_NAME gpio %s, open returned %d", GPIO_CCLK_NAME, cclk_fd );
        close( csn_fd );
        return -1;
    };
    cdti_fd = open( GPIO_CDTI_NAME, O_WRONLY );
    if( cdti_fd < 0 )  {
        ERROR( "Failed to open GPIO_CDTI_NAME gpio %s, open returned %d", GPIO_CDTI_NAME, cdti_fd );
        close( csn_fd );
        close( cclk_fd );
        return -1;
    };

    // If fs is non-null then we take the programming data from a text file
    if( fs )  {
        while( !feof( fs ))  {
            unsigned int op;
            unsigned int addr;
            unsigned int data;
            fscanf( fs, "%c %x %x\r", &op, &addr, &data );
            if( toupper( (unsigned char) op == 'W' ))
                if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (unsigned int) addr, (unsigned char) data )) < 0 )
                    printf( "setAK4633: error %d writing device\n", res );
        }
        fclose( fs );
    }
    // Otherwise we use this hard coded data
    else  {
        char value;

        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x00, (char) 0x40 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        //See Table 5 for selecting FS3-0 values to be programmed in register 5 
        switch(samplingFrequency)
        {
        case 8000:      //Not supported by MG3500
            value = 0x80;
            break;
        case 12000:     //Not supported by MG3500 
            value = 0x81;
            break;
        case 16000:
            value = 0x82;
            break;
        case 24000:
            value = 0x83;
            break;
        case 7350:      //Not supported by MG3500
            value = 0x84;
            break;
        case 11025:      //Not supported by MG3500
            value = 0x85;
            break;
        case 14700:      //Not supported by MG3500
            value = 0x86;
            break;
        case 22050:      //Not supported by MG3500
            value = 0x87;
            break;
        case 32000:
            value = 0xA2;
            break;
        case 48000:
            value = 0xA3;
            break;
        case 29400:      //Not supported by MG3500
            value = 0xA6;
            break;
        case 44100:
            value = 0xA7;
            break;
        }
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x05, value )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x02, (char) 0xA5 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x06, (char) 0x00 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x07, (char) 0x21 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x03, (char) 0x99 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x04, (char) 0xD3 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x1C, (char) 0x16 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x1D, (char) 0x1F )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x1E, (char) 0x2B )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x1F, (char) 0x1E )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x0E, (char) 0x13 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x01, (char) 0x03 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
        if( (res = wrAK4633( csn_fd, cclk_fd, cdti_fd, verbose, (int) 0x00, (char) 0xD5 )) < 0 )
            ERROR( "setAK4633: error %d writing device", res );
    }

    close( csn_fd );
    close( cclk_fd );
    close( cdti_fd );

    return 0;
}
