#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h> /* for put_user */

//include "qhal_dma.h"
#include <asm/arch/mobi_dma.h>
#include <linux/dma-mapping.h>

/* Prototypes . this would normally go in a .h file */

int init_module(void);
void cleanup_module(void);

static int device_open(struct inode *, struct file *);
static int device_release(struct inode *, struct file *);

static ssize_t device_read(struct file *, char *, size_t, loff_t *);
static ssize_t device_write(struct file *, const char *, size_t, loff_t *);
static int device_ioctl(struct inode *node, struct file *filep, unsigned int cmd, unsigned long arg);

#define SUCCESS 0
#define DEVICE_NAME "chardev" /* Dev name as it appears in /proc/devices */
#define BUF_LEN 80 /* Max length of the message from the device */

#define DMA_START_XFER	0x1700
#define DMA_STOP_XFER	0x1701

enum {
	START = 0x1,
	STOP = 0x2,
	INIT = 0x3,
};
int dma_status = INIT;
/* Global variables are declared as static, so are global within the file. */

static int Major; /* Major number assigned to our device driver */
static int Device_Open = 0; /* Is device open? Used to prevent multiple access to
			       the device */

static char msg[BUF_LEN]; /* The msg the device will give when asked */
static char *msg_Ptr;

static struct file_operations fops = {
	.read = device_read,
	.write = device_write,
	.open = device_open,
	.ioctl = device_ioctl,
	.release = device_release
};

struct qhaldma_data_t {
	struct semaphore mutex;
	wait_queue_head_t waitq;
	atomic_t txdone;
	mobi_dma_handle handle;
	unsigned char *data_src, *data_dst;
	unsigned int data_src_phy, data_dst_phy;
	uint32_t xfer_flags, burstflag;
} qhaldma_data;

static void qhaldma_handler(mobi_dma_handle dmah, mobi_dma_event event, void *ptr)
{
	atomic_set(&qhaldma_data.txdone, event);
	wake_up_interruptible(&qhaldma_data.waitq);
}

int qhaldma_open(void)
{
	init_MUTEX (&qhaldma_data.mutex);
	if (down_interruptible(&qhaldma_data.mutex))
		return -1;

	init_waitqueue_head(&qhaldma_data.waitq);
	qhaldma_data.handle=mobi_dma_request("mydrv", MOBI_DMA_O_WAITFOR_BUS);
	
	if (qhaldma_data.handle < 0) {
	     	up(&qhaldma_data.mutex);
		return -1;
	}

	if (mobi_dma_setup_handler(qhaldma_data.handle, qhaldma_handler, NULL) < 0) {   
		mobi_dma_free(qhaldma_data.handle);
		up(&qhaldma_data.mutex);
		return -1;
	}
	
	up(&qhaldma_data.mutex);
	return 0;
}

void qhaldma_close()
{
	down_interruptible(&qhaldma_data.mutex);
	mobi_dma_free(qhaldma_data.handle);
	
	dma_free_coherent(NULL,
			(4*PAGE_SIZE),
			qhaldma_data.data_src,
			qhaldma_data.data_src_phy);

	dma_free_coherent(NULL,
			(4*PAGE_SIZE),
			qhaldma_data.data_dst,
			qhaldma_data.data_dst_phy);

	up(&qhaldma_data.mutex);
	return;
}

void setup_my_dma (void)
{
	if (qhaldma_open() < 0) {
		printk("Error while setting up dma\n");
		return;
	}
	qhaldma_data.data_src = dma_alloc_coherent(NULL,
			(4*PAGE_SIZE),
			&qhaldma_data.data_src_phy,
			GFP_KERNEL);

	qhaldma_data.data_dst = dma_alloc_coherent(NULL,
			(4*PAGE_SIZE),
			&qhaldma_data.data_dst_phy,
			GFP_KERNEL);

	if (qhaldma_data.data_src == NULL ||
			qhaldma_data.data_dst == NULL) {
		printk("Error while allocating coherent mem\n");
		return;
	}

	memset(qhaldma_data.data_src, 0xa5, (4*PAGE_SIZE));
	memset(qhaldma_data.data_dst, 0x0, (4*PAGE_SIZE));
	
	qhaldma_data.xfer_flags |= MOBI_DMA_CONFIG_DATA_WIDTH_8;
	qhaldma_data.burstflag = MOBI_DMA_CONFIG_BURST_SIZE_1;

	if(mobi_dma_config(qhaldma_data.handle, DMA_CONFIG_XFER,
				qhaldma_data.xfer_flags, NULL) < 0) {
	 	printk("Error while configuring xfer flags\n");
	    	return;
	} /* xfer flags */

	/* Src Memory side */
	if(mobi_dma_config(qhaldma_data.handle,
				DMA_CONFIG_SRC,
				MOBI_DMA_CONFIG_TRANSFER_WIDTH_32 | qhaldma_data.burstflag
				| MOBI_DMA_CONFIG_ENDIAN_LE | MOBI_DMA_CONFIG_ADDRADJ_INC,
				NULL
			  ) < 0) {
		printk("Error while configuring mem_src flags\n");
		return;
	};

	/* Dst Memory side */
	if(mobi_dma_config(qhaldma_data.handle,
				DMA_CONFIG_DST,
				MOBI_DMA_CONFIG_TRANSFER_WIDTH_32 | qhaldma_data.burstflag
				| MOBI_DMA_CONFIG_ENDIAN_LE | MOBI_DMA_CONFIG_ADDRADJ_INC,
				NULL
			  ) < 0) {
		printk("Error while configuring mem_src flags\n");
		return;
	};

}
/* Functions */
int init_module(void)
{
	Major = register_chrdev(0, DEVICE_NAME, &fops);
	if (Major < 0)
	{
		printk ("Registering the character device failed with %d\n", Major);
		return Major;
	}

	printk("<1>I was assigned major number %d. To talk to\n", Major);
	printk("<1>the driver, create a dev file with\n");
	printk("'mknod /dev/hello c %d 0'.\n", Major);
	printk("<1>Try various minor numbers. Try to cat and echo to\n");
	printk("the device file.\n");
	printk("<1>Remove the device file and module when done.\n");
	return 0;
}

void cleanup_module(void)
{
	/* Unregister the device */
	int ret = unregister_chrdev(Major, DEVICE_NAME);
	if (ret < 0) 
		printk("Error in unregister_chrdev: %d\n", ret);
	else
		printk("\n\Sucess in remove module\n");
}

/* Called when a process tries to open the device file, like "cat /dev/mycharfile" */

static int device_open(struct inode *inode, struct file *file)
{
	static int counter = 0;
	if (Device_Open) 
		return -EBUSY;
	Device_Open++;

	sprintf(msg,"I already told you %d times Hello world!\n", counter++);
	msg_Ptr = msg;
	return SUCCESS;
}

/* Called when a process closes the device file.*/

static int device_release(struct inode *inode, struct file *file)
{
	Device_Open --; /* We're now ready for our next caller */
	return 0;
}

/* Called when a process, which already opened the dev file, attempts to read from
 * it.*/

static ssize_t device_read(struct file *filp,char *buffer,/*buffer to fill with data
*/size_t length, /*length of the buffer */loff_t *offset) /* Our offset in the file
*/
{
	/* Number of bytes actually written to the buffer */
	int bytes_read = 0;
	/* If we're at the end of the message, return 0 signifying end of file */
	if (*msg_Ptr == 0) 
		return 0;

	/* Actually put the data into the buffer */
	while (length && *msg_Ptr) 
	{
		/* The buffer is in the user data segment, not the kernel segment;  assignment
		 * won't work. We have to use put_user which copies data from  the kernel data segment
		 * to the user data segment. */
		put_user(*(msg_Ptr++), buffer++);
		length--;
		bytes_read++;
	}

	/* Most read functions return the number of bytes put into the buffer */

	return bytes_read;
}

/* Called when a process writes to dev file: echo "hi" > /dev/hello */

static ssize_t device_write(struct file *filp,const char *buff,size_t len,loff_t *off)
{

	struct page **page_list;
	int offset;
	int npages;
	unsigned long cur_base = 0;
	unsigned long offset = 0;
	int nents = 0, i;
	struct timeval tv_start = {0, 0}, tv_end = {0, 0};
	
	memset(&tv_start, 0, sizeof(struct timeval));
	memset(&tv_end, 0, sizeof(struct timeval));

	offset = (unsigned long) (buff & ~PAGE_MASK);
	npages = ALIGN(len + offset, PAGE_SIZE) >>  PAGE_SHIFT;
	cur_base = (unsigned long) buff  & PAGE_MASK;


	do_gettimeofday(&tv_start);
	nents = get_user_pages(current, current->mm, cur_base,
			npages,
			1, 1, page_list, NULL);

	do_gettimeofday(&tv_end);
	if (nents != npages)
		printk("Got less than asked.. got:%d asked:%d\n",
				nents, npages);


	for (i = 0; i < nents; i++) 
		put_page(page_list[i]);

	printk("Start: sec:%x usec:%x\n", tv_start.tv_sec, tv_start.tv_usec);
	printk("End  : sec:%x usec:%x\n", tv_end.tv_sec, tv_end.tv_usec);

	if (len == 0) {
		if (dma_status == START) {
			dma_status = STOP;
			msleep (1000);	
		}
	}
	//	printk ("<1>Sorry, this operation isn't supported.\n");
	return len;
}
static __inline__ void print_pkt(unsigned char *buf, int len)
{
	int j;
	printk("len = %d byte, buf addr: 0x%p", len, buf);
	for (j = 0; j < len; j++) {
		if ((j % 16) == 0)
			printk("\n %03x:", j);
		printk(" %02x", buf[j]);
	}
	printk("\n");
	return;
}
int glb_ctr = 0;
void do_dma_transfer(void)
{

	/* Data type done in qhaldma_open */
	if (mobi_dma_setup_single(qhaldma_data.handle,
				qhaldma_data.data_src_phy,
				qhaldma_data.data_dst_phy,
				(4*PAGE_SIZE)) < 0) {
		printk("Error while doing dma xfer\n");
		dma_status = STOP;
	}

	atomic_set(&qhaldma_data.txdone, 0);

	if(mobi_dma_enable(qhaldma_data.handle) < 0) {
		printk("Error while enabling dma\n");
		return;
	}

	if(wait_event_interruptible(qhaldma_data.waitq,
				atomic_read(&qhaldma_data.txdone)
				) < 0) {
		mobi_dma_disable(qhaldma_data.handle);
		printk("Error while waiting for dma xfer completion\n");
		return;
	}

	if(mobi_dma_disable(qhaldma_data.handle) < 0) {
		printk("Error while disabling dma\n");
		return;
	}

	if(atomic_read(&qhaldma_data.txdone)!=MOBI_DMA_EVENT_TRANSFER_COMPLETE)
		printk("Error in DMA Transfer..\n");
	if (glb_ctr < 10) {
		print_pkt(qhaldma_data.data_dst, 128);
		glb_ctr++;
		memset(qhaldma_data.data_dst, 0x0, (4*PAGE_SIZE));
	}

}
static int device_ioctl(struct inode *inode, struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
		case DMA_START_XFER:
			dma_status = START;
			printk("Dma xfer start\n");
			setup_my_dma();
			while (dma_status == START) {

				do_dma_transfer();
				//msleep(10);
				if (signal_pending(current)) {
					printk("Signal arrived from userspace\n");
					dma_status = STOP;
				}
			}
			break;
		case DMA_STOP_XFER:
			printk("Dma xfer stop\n");
			qhaldma_close();
			break;
		default:
			printk("Bad cmd\n");
			break;
	}
	return 0;
}

