/*
 * Copyright (c) 1993 by David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Stand-alone shell for system maintainance for Linux.
 * This program should NOT be built using shared libraries.
 *
 * 1.1.1, 	hacked to re-allow cmd line invocation of script file
 *		Pat Adamo, padamo@unix.asb.com
 */

#include "sash.h"

#undef INTERNAL_PATH_EXPANSION
#define FAVOUR_EXTERNAL_COMMANDS

#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>

/* #ifndef EMBED */
#define __signal(x,y,z) do {} while (0)
/* #endif */

/* static char version[] = "1.1.1"; */

extern int intflag;

extern void do_test();

typedef struct {
	char	name[10];
	char	usage[30];
	void	(*func)();
	int	minargs;
	int	maxargs;
} CMDTAB;


CMDTAB	cmdtab[] = {
	{"cd",		"[dirname]",		do_cd,	    1, 	2},
	{"echo",	"[args] ...",		do_echo,    1, 	MAXARGS},
	{"exec",	"filename [args]",	do_exec,    2, 	MAXARGS},
	{"exit",	"",			do_exit,    1, 	1},
	{"hexdump",	"[-s] filename",	do_hexdump, 1, 	3},
	{"pid",		"",			do_pid,     1, 	1},
	{"quit",	"",			do_exit,    1, 	1},
	{"setenv",	"name value",		do_setenv,  3, 	3},
	{"sleep",	"seconds",		do_sleep,   1, 	2},
	{"source",	"filename",		do_source,  2, 	2},
	{"umask",	"[mask]",		do_umask,   1, 	2},
	{{0},		{0},			0,          0, 	0},
};

static	FILE	*sourcefiles[MAXSOURCE];
static	int	sourcecount;

volatile static	BOOL	intcrlf = TRUE;


static	void	catchint();
static	void	catchquit();
/* static	void	catchchild(); */
static	void	readfile();
static	void	command();
static	void	runcmd();
static	void	showprompt();
static	BOOL	trybuiltin();
static	BOOL	command_in_path();

extern char ** environ;

int main(argc, argv, env)
	char	**argv;
	char	*env[];
{
/* 	char	*cp; */
	int dofile = 0;
	
	if ((argc > 1) && !strcmp(argv[1], "-c")) {
		/* We aren't that fancy a shell */
		exit(0);
	}

	if ((argc > 1) && strcmp(argv[1], "-t"))
		{
		dofile++;
		printf("Shell invoked to run file: %s\n",argv[1]);
		}
/* 	else */
/* 		printf("\nSash command shell (version %s)\n", version); */
	fflush(stdout);

	signal(SIGINT, catchint);
	signal(SIGQUIT, catchquit);
/* 	__signal(SIGCHLD, catchchild, SA_RESTART); */

	if (getenv("PATH") == NULL)
		putenv("PATH=/bin:/usr/bin:/etc");

	if (dofile)
		{
		/* open the file for reading! */
		readfile(argv[1]);
		}
	   else
		{
		readfile(NULL); /* no arguments! */
		} /* end if arguments supplied */
	exit(0);
}

char	buf[CMDLEN];


/*
 * Read commands from the specified file.
 * A null name pointer indicates to read from stdin.
 */
static void
readfile(name)
	char	*name;
{
	FILE	*fp;
	int	cc;
	BOOL	ttyflag;
	char	*ptr;

	if (sourcecount >= MAXSOURCE) {
		fprintf(stderr, "Too many source files\n");
		fflush(stderr);
		return;
	}

	fp = stdin;
	if (name) {
		fp = fopen(name, "r");
		if (fp == NULL) {
			perror(name);
			return;
		}
	}
	sourcefiles[sourcecount++] = fp;

	ttyflag = isatty(fileno(fp));

	while (TRUE) {
		fflush(stdout);
		if (ttyflag && fp == stdin) /* using terminal, so show prompt */
			showprompt();

		if (intflag && !ttyflag && (fp != stdin)) {
			fclose(fp);
			sourcecount--;
			return;
		}

		if (fgets(buf, CMDLEN - 1, fp) == NULL) {
			if (ferror(fp) && (errno == EINTR)) {
				clearerr(fp);
				continue;
			}
			break;
		}

		cc = strlen(buf);
		if (buf[cc - 1] == '\n')
			cc--;

		while ((cc > 0) && isblank(buf[cc - 1]))
			cc--;
		buf[cc] = '\0';
/* 		if (fp != stdin) */
/* 			{ */
			/* taking commands from file - echo */
/* 			printf("Command: %s\n",buf); */
/* 			} */ /* end if (fp != stdin) */

                /* remove leading spaces and look for a '#' */
                ptr = &buf[0];
                while (*ptr == ' ') {
                        ptr++;
                }
                if (*ptr != '#')
                        command(buf);
	}



	if (ferror(fp)) {
		perror("Reading command line");
		if (fp == stdin)
			exit(1);
	}

	clearerr(fp);
	if (fp != stdin)
		{
		fclose(fp);
/* 		printf("Execution Finished, Exiting\n"); */
		}

	sourcecount--;
}


/*
 * Parse and execute one null-terminated command line string.
 * This breaks the command line up into words, and expands wildcards.
 */
static void
command(cmd)
	char	*cmd;
{
	char	**argv;
	int	argc;
	int 	bg;
	char   *c;

	intflag = FALSE;

	freechunks();

	while (isblank(*cmd))
		cmd++;

	if ((c = strchr(cmd, '&')) != NULL) {
		*c = '\0';
		bg = 1;
	} else
		bg = 0;

	if ((cmd = expandenvvar(cmd)) == NULL)
		return;

	if ((*cmd == '\0') || !makeargs(cmd, &argc, &argv))
		return;

	/*
	 * Now look for the command in the builtin table, and execute
	 * the command if found.
	 */
#ifdef FAVOUR_EXTERNAL_COMMANDS
	if (!command_in_path(argv[0]))
#endif
	if (trybuiltin(argc, argv))
		return;

	/*
	 * Not found, run the program along the PATH list.
	 */
	runcmd(cmd, bg, argc, argv);
}


#ifdef FAVOUR_EXTERNAL_COMMANDS
/*
 * return true if we find this command in our
 * path.
 */
static BOOL
command_in_path(char *cmd)
{
	struct stat	stat_buf;

	if (strchr(cmd, '/') == 0) {
		char	* path;
		static char	path_copy[PATHLEN];
		
		/* Search path for binary */
		for (path = getenv("PATH"); path && *path; ) {
			char * p2;

			strcpy(path_copy, path);
			if ((p2 = strchr(path_copy, ':')) != NULL) {
				*p2 = '\0';
			}
		
			if (strlen(path_copy))
				strcat(path_copy, "/");
			strcat(path_copy, cmd);
			
			if (!stat(path_copy, &stat_buf) && (stat_buf.st_mode & 0111))
				return(TRUE);
			
			p2 = strchr(path, ':');
			if (p2)
				path = p2 + 1;
			else
				path = 0;
		}
	} else if (!stat(cmd, &stat_buf) && (stat_buf.st_mode & 0111))
		return(TRUE);
	return(FALSE);
}
#endif /* FAVOUR_EXTERNAL_COMMANDS */


/*
 * Try to execute a built-in command.
 * Returns TRUE if the command is a built in, whether or not the
 * command succeeds.  Returns FALSE if this is not a built-in command.
 */
static BOOL
trybuiltin(argc, argv)
	char	**argv;
{
	CMDTAB	*cmdptr;
	int	oac;
	int	newargc;
	int	matches;
	int	i;
	char	*newargv[MAXARGS];
	char	*nametable[MAXARGS];

	cmdptr = cmdtab - 1;
	do {
		cmdptr++;
		if (cmdptr->name[0] == 0)
			return FALSE;

	} while (strcmp(argv[0], cmdptr->name));
	
	/*
	 * Give a usage string if the number of arguments is too large
	 * or too small.
	 */
	if ((argc < cmdptr->minargs) || (argc > cmdptr->maxargs)) {
		fprintf(stderr, "usage: %s %s\n",
			cmdptr->name, cmdptr->usage);
		fflush(stderr);

		return TRUE;
	}

	/*
	 * Now for each command argument, see if it is a wildcard, and if
	 * so, replace the argument with the list of matching filenames.
	 */
	newargv[0] = argv[0];
	newargc = 1;
	oac = 0;

	while (++oac < argc) {
		matches = expandwildcards(argv[oac], MAXARGS, nametable);
		if (matches < 0)
			return TRUE;

		if ((newargc + matches) >= MAXARGS) {
			fprintf(stderr, "Too many arguments\n");
			fflush(stderr);
			return TRUE;
		}

		if (matches == 0)
			newargv[newargc++] = argv[oac];

		for (i = 0; i < matches; i++)
			newargv[newargc++] = nametable[i];
	}

	(*cmdptr->func)(newargc, newargv);

	return TRUE;
}


/*
 * Execute the specified command.
 */
static void
runcmd(cmd, bg, argc, argv)
	char	*cmd;
	int	bg;
	int	argc;
	char	**argv;
{
/* 	register char *	cp; */
	BOOL		magic;
	int		pid;
	int		status;
	int oac;
	int newargc;
	int matches;
	int i;
	char	*newargv[MAXARGS];
	char	*nametable[MAXARGS];
	
	newargv[0] = argv[0];
	
#ifdef INTERNAL_PATH_EXPANSION
	if (strchr(argv[0], '/') == 0) {
		char	* path;
		struct stat	stat_buf;
		static char	path_copy[PATHLEN];
		
		/* Search path for binary */
		for (path = getenv("PATH"); path && *path; ) {
			char * p2;
			strncpy(path_copy, path, sizeof(path_copy - 1));
			if (p2 = strchr(path_copy, ':')) {
				*p2 = '\0';
			}
		
			if (strlen(path_copy))
				strncat(path_copy, "/", sizeof(path_copy));
			strncat(path_copy, argv[0], sizeof(path_copy));
			
			if (!stat(path_copy, &stat_buf) && (stat_buf.st_mode & 0111)) {
				newargv[0] = path_copy;
				break;
			}
			
			p2 = strchr(path, ':');
			if (p2)
				path = p2 + 1;
			else
				path = 0;
		}
	}
#endif

	/*
	 * Now for each command argument, see if it is a wildcard, and if
	 * so, replace the argument with the list of matching filenames.
	 */
	newargc = 1;
	oac = 0;

	while (++oac < argc) {
		matches = expandwildcards(argv[oac], MAXARGS, nametable);
		if (matches < 0)
			return;

		if ((newargc + matches) >= MAXARGS) {
			fprintf(stderr, "Too many arguments\n");
			fflush(stderr);
			return;
		}

		if (matches == 0)
			newargv[newargc++] = argv[oac];

		for (i = 0; i < matches; i++)
			newargv[newargc++] = nametable[i];
	}
	
	newargv[newargc] = 0;

	magic = FALSE;
	
	/*
	for (cp = cmd; *cp; cp++) {
		if ((*cp >= 'a') && (*cp <= 'z'))
			continue;
		if ((*cp >= 'A') && (*cp <= 'Z'))
			continue;	
		if (isdecimal(*cp))
			continue;
		if (isblank(*cp))
			continue;

		if ((*cp == '.') || (*cp == '/') || (*cp == '-') ||
			(*cp == '+') || (*cp == '=') || (*cp == '_') ||
			(*cp == ':') || (*cp == ','))
				continue;

		magic = TRUE;
	}
	*/

	if (magic) {
		printf("%s: no such file or directory\n", cmd);
		fflush(stdout);
		system(cmd);
		return;
	}
	
	if (!bg)
		signal(SIGCHLD, SIG_DFL);

	/*
	 * No magic characters in the expanded command, so do the fork and
	 * exec ourself.  If this fails with ENOEXEC, then run the
	 * shell anyway since it might be a shell script.
	 */
	if (!(pid = vfork())) {
		int	ci;

		/*
		 * We are the child, so run the program.
		 * First close any extra file descriptors we have opened.
		 * be sure not to modify any globals after the vfork !
		 */	
		
		for (ci = 0; ci < sourcecount; ci++)
			if (sourcefiles[ci] != stdin)
				close(fileno(sourcefiles[ci]));
		
		signal(SIGINT, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
		signal(SIGCHLD, SIG_DFL);
		
		execvp(newargv[0], newargv);

		printf("%s: %s\n", newargv[0], (errno == ENOENT) ? "Bad command or file name" : strerror(errno));
		fflush(stdout);
		
		_exit(0);
	}
	
	if (pid < 0) {
/* 		__signal(SIGCHLD, catchchild, SA_RESTART); */
		perror("vfork failed");
		return;
	}
	
	if (bg) {
		printf("[%d]\n", pid);
		fflush(stdout);
		return;
	}

	if (pid) {
		int cpid;
		status = 0;
		intcrlf = FALSE;

		for (;;) {
			cpid = wait4(pid, &status, 0, 0);
			if ((cpid < 0) && (errno == EINTR))
				continue;
			if (cpid < 0)
				break;
			if (cpid != pid) {
/* 				fprintf(stderr, "sh %d: %d died (wanted %d)\n", getpid(), cpid, pid); */
				continue;
			}
		}

/* 		__signal(SIGCHLD, catchchild, SA_RESTART); */
		
		intcrlf = TRUE;
		if ((status & 0xff) == 0)
			return;

		fprintf(stderr, "pid %d: %s (signal %d)\n", pid,
			(status & 0x80) ? "core dumped" : "killed",
			status & 0x7f);
		fflush(stderr);

		return;
	}
	
	perror(argv[0]);
	exit(1);
}

void
do_source(argc, argv)
	char	**argv;
{
	readfile(argv[1]);
}

/*void
do_cd(argc, argv)
	char	**argv;
{
	char	*name;

	name = argv[1];
	
	if (chdir(name))
		perror("Unable to chdir to %s");
	
}*/

void
do_pid(argc, argv)
{
	printf("%d\n", getpid());
	fflush(stdout);
}

void
do_exec(argc, argv)
	char	**argv;
{
	char	*name;

	name = argv[1];

	if (access(name, 4)) {
		perror(name);
		return;
	}

	while (--sourcecount >= 0) {
		if (sourcefiles[sourcecount] != stdin)
			fclose(sourcefiles[sourcecount]);
	}

	argv[0] = name;
	argv[1] = NULL;

	execv(name, argv);

	perror(name);
	exit(1);
}

/*void
do_exit(argc, argv)
	char	**argv;
{
	if (argc>1)
		exit(atoi(argv[1]));
	else
		exit(0);
}*/


/*
 * Display the prompt string.
 */
static void
showprompt()
{
/* 	char	*cp; */
	char Sbuf[60];
	
#if 0
	sprintf(Sbuf, "%d", getpid());
	write(STDOUT, Sbuf, strlen(Sbuf));
#endif
	
	*Sbuf = '\0';
	getcwd(Sbuf, sizeof(Sbuf) - 1);
	strncat(Sbuf, "> ", sizeof(Sbuf) - 1);
	Sbuf[sizeof(Sbuf) - 1] = '\0';
	write(STDOUT, Sbuf, strlen(Sbuf));
}	


static void
catchint()
{
	signal(SIGINT, catchint);

	intflag = TRUE;

	if (intcrlf)
		write(STDOUT, "\n", 1);
}


static void
catchquit()
{
	signal(SIGQUIT, catchquit);

	intflag = TRUE;

	if (intcrlf)
		write(STDOUT, "\n", 1);
}

#if 0
static void
catchchild()
{
	char buf[40];
	pid_t pid;
	int status;
	
	/*signal(SIGCHLD, catchchild);*/ /* Unneeded */

	pid = wait4(-1, &status, WUNTRACED, 0);
	if (WIFSTOPPED(status))
		sprintf(buf, "sh %d: Child %d stopped\n", getpid(), pid);
	else
		sprintf(buf, "sh %d: Child %d died\n", getpid(), pid);
	
	intflag = TRUE;
	
	if (intcrlf)
		write(STDOUT, "\n", 1);
	
	write(STDOUT, buf, strlen(buf));
}
#endif	/* 0 */

/* END CODE */
