#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <errno.h>
#include <sys/types.h>
#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "log.h"
#include "libaes/aes_cbc.h"
#include "alg_info.h"
#include "ike_alg.h"

#ifndef NO_BRECIS_SEC
#include <brecis/msp_secv2.h>
#endif

#define  AES_CBC_BLOCK_SIZE	(128/BITS_PER_BYTE)
#define  AES_KEY_MIN_LEN	128
#define  AES_KEY_DEF_LEN	128
#define  AES_KEY_MAX_LEN	256

static void
do_aes(u_int8_t *buf, size_t buf_len, u_int8_t *key, size_t key_size, u_int8_t *iv, bool enc)
{
#ifdef NO_BRECIS_SEC
    aes_context aes_ctx;
    char iv_bak[AES_CBC_BLOCK_SIZE];
    char *new_iv = NULL;	/* logic will avoid copy to NULL */

    aes_set_key(&aes_ctx, key, key_size, 0);

    /*	
     *	my AES cbc does not touch passed IV (optimization for
     *	ESP handling), so I must "emulate" des-like IV
     *	crunching
     */
    if (!enc)
	    memcpy(new_iv=iv_bak, 
			    (char*) buf + buf_len-AES_CBC_BLOCK_SIZE,
			    AES_CBC_BLOCK_SIZE);

    AES_cbc_encrypt(&aes_ctx, buf, buf, buf_len, iv, enc);

    if (enc)
	    new_iv = (char*) buf + buf_len-AES_CBC_BLOCK_SIZE;

    memcpy(iv, new_iv, AES_CBC_BLOCK_SIZE);
#else
    MSP_SEC2_SA sa ;
    MSP_SEC2_WQE wqe ;
    int key_bytes ;
    int status ;

    /* I guess keysize can be specified in either bits or bytes. */
    switch (key_size)
    {
    case 32:			/* bytes */
    case 256:			/* bits */
	    key_bytes = 32 ;
	    sa.flags = SAFLG_MODE_CRYPT | SAFLG_AES_256 ;
	    break ;
    case 24:
    case 192:
	    key_bytes = 24 ;
	    sa.flags = SAFLG_MODE_CRYPT | SAFLG_AES_192 ;
	    break ;
    case 16:
    case 128:
    default:
	    key_bytes = 16 ;
	    sa.flags = SAFLG_MODE_CRYPT | SAFLG_AES_128 ;
	    break ;
    }
    memcpy(sa.crypt_keys, key, key_bytes) ;
    memcpy(sa.crypt_iv, iv, AES_CBC_BLOCK_SIZE) ;
    if (enc)
    {
	    sa.flags |= SAFLG_CBC_ENCRYPT ; 
    }
    else
    {
	    sa.flags |= SAFLG_AES_DECRYPT | SAFLG_CBC_DECRYPT ;
	    MSP_SEC2_SET_AES_DECRYPT_KEY(&sa, 0, 0) ;

	    /* we've already got our copy of the IV, so we can overwrite
	    ** the caller...
	    */
	    memcpy(iv, (char*) buf + buf_len-AES_CBC_BLOCK_SIZE,
			    AES_CBC_BLOCK_SIZE);
    }
#if 0    
    if (!enc)
    {
	    sa.flags |= SAFLG_AES_DECRYPT ;
	    MSP_SEC2_SET_AES_DECRYPT_KEY(&sa, 0, 0) ;

	    /* we've already got our copy of the IV, so we can overwrite
	    ** the caller...
	    */
	    memcpy(iv, (char*) buf + buf_len-AES_CBC_BLOCK_SIZE,
			    AES_CBC_BLOCK_SIZE);
    }
#endif
    MSP_SEC2_NEW_REQUEST(&wqe, 0, &sa, 0) ;
    MSP_SEC2_ADD_SG(&wqe, SG_GATHER, buf, buf_len) ;
    MSP_SEC2_ADD_SG(&wqe, SG_SCATTER, buf, buf_len) ;
    status = MSP_SEC2_END_REQUEST(&wqe) ;
    if (status)
	    fprintf(stderr,"Warning: bad return status from encryption 0x%x\n",
		    status) ;

    if (enc)
    {
	    memcpy(iv, (char*) buf + buf_len-AES_CBC_BLOCK_SIZE,
		   AES_CBC_BLOCK_SIZE);
    }
#endif
}

struct encrypt_desc algo_aes =
{
	algo_type: 	IKE_ALG_ENCRYPT,
	algo_id:   	OAKLEY_AES_CBC,
	algo_next: 	NULL, 
	enc_ctxsize: 	sizeof(aes_context),
	enc_blocksize: 	AES_CBC_BLOCK_SIZE,
	keyminlen: 	AES_KEY_MIN_LEN,
	keydeflen: 	AES_KEY_DEF_LEN,
	keymaxlen: 	AES_KEY_MAX_LEN,
	do_crypt: 	do_aes,
};
int ike_alg_aes_init(void);
int
ike_alg_aes_init(void)
{
#ifndef NO_BRECIS_SEC
	int info[2] ;

	if ( ( MSP_SEC_ENQUIRE(info) == -1 )
	     || ( (info[1] & SECENQ_API2_AES) == 0 ) )
	{
		return -EINVAL ;
	}
#endif

	int ret = ike_alg_register_enc(&algo_aes);
	return ret;
}
/*
IKE_ALG_INIT_NAME: ike_alg_aes_init
*/
