/*
 * receive code
 * Copyright (C) 1996, 1997  John Ioannidis.
 * Copyright (C) 1998, 1999, 2000, 2001  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

char ipsec_rcv_c_version[] = "RCSID $Id: ipsec_rcv.c,v 1.10 2003/05/23 21:30:10 swahl Exp $";

#include <linux/config.h>
#include <linux/version.h>

#define __NO_VERSION__
#include <linux/module.h>
#include <linux/kernel.h> /* printk() */

#define IPSEC_KLIPS1_COMPAT 1
#include "ipsec_param.h"

#ifdef MALLOC_SLAB
# include <linux/slab.h> /* kmalloc() */
#else /* MALLOC_SLAB */
# include <linux/malloc.h> /* kmalloc() */
#endif /* MALLOC_SLAB */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <freeswan.h>
#ifdef SPINLOCK
# ifdef SPINLOCK_23
#  include <linux/spinlock.h> /* *lock* */
# else /* SPINLOCK_23 */
#  include <asm/spinlock.h> /* *lock* */
# endif /* SPINLOCK_23 */
#endif /* SPINLOCK */
#ifdef NET_21
# include <asm/uaccess.h>
# include <linux/in6.h>
# define proto_priv cb
#endif /* NET21 */
#include <asm/checksum.h>
#include <net/ip.h>

#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_sa.h"

#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_tunnel.h"
#include "ipsec_rcv.h"
#if defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH)
# include "ipsec_ah.h"
#endif /* defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH) */
#ifdef CONFIG_IPSEC_ESP
# include "ipsec_esp.h"
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_IPCOMP
# include "ipcomp.h"
#endif /* CONFIG_IPSEC_COMP */

#include <pfkeyv2.h>
#include <pfkey.h>

#include "ipsec_proto.h"

#ifndef NO_BRECIS_HW
#include <brecis/msp_sec.h>
#include <brecis/msp_sec_kern.h>
static msp_sec_context msp_sec_ctx ;
#include <brecis/msp_secv2.h>
#endif /* NO_BRECIS_HW */

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
#include <linux/udp.h>
#endif
#if 0
#ifdef KLIPS_PRINT
#undef KLIPS_PRINT
int debug_rcv =1;
#define KLIPS_PRINT(flag, format, args...) \
		((flag) ? printk(KERN_INFO format , ## args) : 0)
#endif
#endif
#ifdef CONFIG_IPSEC_DEBUG
int debug_ah = 0;
int debug_esp = 0;
int debug_rcv = 0;
#endif /* CONFIG_IPSEC_DEBUG */
int sysctl_ipsec_inbound_policy_check = 1;

#if defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH)
__u32 zeroes[AH_AMAX];
#endif /* defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH) */
/*
 * Check-replay-window routine, adapted from the original 
 * by J. Hughes, from draft-ietf-ipsec-esp-des-md5-03.txt
 *
 *  This is a routine that implements a 64 packet window. This is intend-
 *  ed on being an implementation sample.
 */
DEBUG_NO_STATIC int
ipsec_checkreplaywindow(struct ipsec_sa*tdbp, __u32 seq)
{
	__u32 diff;
	
	if (tdbp->tdb_replaywin == 0)	/* replay shut off */
		return 1;
	if (seq == 0) 
		return 0;		/* first == 0 or wrapped */

	/* new larger sequence number */
	if (seq > tdbp->tdb_replaywin_lastseq) {
		return 1;		/* larger is good */
	}
	diff = tdbp->tdb_replaywin_lastseq - seq;

	/* too old or wrapped */ /* if wrapped, kill off SA? */
	if (diff >= tdbp->tdb_replaywin) {
		return 0;
	}
	/* this packet already seen */
	if (tdbp->tdb_replaywin_bitmap & (1 << diff))
		return 0;
	return 1;			/* out of order but good */
}

DEBUG_NO_STATIC int
ipsec_updatereplaywindow(struct ipsec_sa*tdbp, __u32 seq)
{
	__u32 diff;
	
	if (tdbp->tdb_replaywin == 0)	/* replay shut off */
		return 1;
	if (seq == 0) 
		return 0;		/* first == 0 or wrapped */

	/* new larger sequence number */
	if (seq > tdbp->tdb_replaywin_lastseq) {
		diff = seq - tdbp->tdb_replaywin_lastseq;

		/* In win, set bit for this pkt */
		if (diff < tdbp->tdb_replaywin)
			tdbp->tdb_replaywin_bitmap =
				(tdbp->tdb_replaywin_bitmap << diff) | 1;
		else
			/* This packet has way larger seq num */
			tdbp->tdb_replaywin_bitmap = 1;

		if(seq - tdbp->tdb_replaywin_lastseq - 1 > tdbp->tdb_replaywin_maxdiff) {
			tdbp->tdb_replaywin_maxdiff = seq - tdbp->tdb_replaywin_lastseq - 1;
		}
		tdbp->tdb_replaywin_lastseq = seq;
		return 1;		/* larger is good */
	}
	diff = tdbp->tdb_replaywin_lastseq - seq;

	/* too old or wrapped */ /* if wrapped, kill off SA? */
	if (diff >= tdbp->tdb_replaywin) {
/*
		if(seq < 0.25*max && tdbp->tdb_replaywin_lastseq > 0.75*max) {
			deltdbchain(tdbp);
		}
*/	
		return 0;
	}
	/* this packet already seen */
	if (tdbp->tdb_replaywin_bitmap & (1 << diff))
		return 0;
	tdbp->tdb_replaywin_bitmap |= (1 << diff);	/* mark as seen */
	return 1;			/* out of order but good */
}

int
#ifdef PROTO_HANDLER_SINGLE_PARM
ipsec_rcv(struct sk_buff *skb)
#else /* PROTO_HANDLER_SINGLE_PARM */
#ifdef NET_21
ipsec_rcv(struct sk_buff *skb, unsigned short xlen)
#else /* NET_21 */
ipsec_rcv(struct sk_buff *skb, struct device *dev, struct options *opt, 
		__u32 daddr_unused, unsigned short xlen, __u32 saddr,
                                   int redo, struct inet_protocol *protocol)
#endif /* NET_21 */
#endif /* PROTO_HANDLER_SINGLE_PARM */
{
#ifdef NET_21
#ifdef CONFIG_IPSEC_DEBUG
	struct device *dev = skb->dev;
#endif /* CONFIG_IPSEC_DEBUG */
#endif /* NET_21 */
	unsigned char protoc;
	struct iphdr *ipp;
	int authlen = 0;
#ifdef CONFIG_IPSEC_ESP
	struct esp *espp = NULL;
	int esphlen = 0;
#ifdef NO_BRECIS_HW
	__u32 iv[2];
#endif /* NO_BRECIS_HW */
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
	struct ah *ahp = NULL;
	int ahhlen = 0;
	struct iphdr ipo;
#endif /* CONFIG_IPSEC_AH */
	unsigned char *authenticator = NULL;
#ifdef NO_BRECIS_HW
	union {
		MD5_CTX		md5;
		SHA1_CTX	sha1;
	} tctx;
#endif /* NO_BRECIS_HW */
	__u8 hash[AH_AMAX];
#if defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH)
#endif /* defined(CONFIG_IPSEC_ESP) || defined(CONFIG_IPSEC_AH) */
#ifdef CONFIG_IPSEC_IPCOMP
	struct ipcomphdr*compp = NULL;
#endif /* CONFIG_IPSEC_IPCOMP */

	int hard_header_len;
	int iphlen;
	unsigned char *dat;
	struct ipsec_sa *tdbp = NULL;
	struct sa_id said;
	struct net_device_stats *stats = NULL;		/* This device's statistics */
	struct device *ipsecdev = NULL, *prvdev;
	struct ipsecpriv *prv;
	char name[9];
	char sa[SATOA_BUF];
	size_t sa_len;
	char ipaddr_txt[ADDRTOA_BUF];
	int i;
	struct in_addr ipaddr;
	__u8 next_header = 0;
	__u8 proto;
	
#ifdef CONFIG_IPSEC_ESP
	int pad = 0, padlen;
#endif /* CONFIG_IPSEC_ESP */
	int ilen;	/* content to be decrypted/authenticated */
	int len;	/* packet length */
	int replay = 0;	/* replay value in AH or ESP packet */
	__u8 *idat;	/* pointer to content to be decrypted/authenticated */
	struct ipsec_sa* tdbprev = NULL;	/* previous SA from outside of packet */
	struct ipsec_sa* tdbnext = NULL;	/* next SA towards inside of packet */
#ifdef INBOUND_POLICY_CHECK_eroute
	struct sockaddr_encap matcher;	/* eroute search key */
	struct eroute *er;
	struct ipsec_sa* policy_tdb = NULL;
	struct sa_id policy_said;
	struct sockaddr_encap policy_eaddr;
	struct sockaddr_encap policy_emask;
#endif /* INBOUND_POLICY_CHECK_eroute */

#if !defined(NO_BRECIS_HW) && defined (CONFIG_BRECIS_SEC_V2)
//	MSP_SEC2_SA brecis_sa ;
	unsigned int status ;
#endif

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
	__u16 natt_len = 0, natt_sport = 0, natt_dport = 0;
	__u8 natt_type = 0;
#endif

	/* Don't unlink in the middle of a turnaround */
	MOD_INC_USE_COUNT;
	
	if (skb == NULL) {
		KLIPS_PRINT(debug_rcv, 
			    "klips_debug:ipsec_rcv: "
			    "NULL skb passed in.\n");
		goto rcvleave;
	}
		
	if (skb->data == NULL) {
		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: "
			    "NULL skb->data passed in, packet is bogus, dropping.\n");
		goto rcvleave;
	}

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
	if (skb->sk && skb->nh.iph && skb->nh.iph->protocol==IPPROTO_UDP) {
		/**
		 * Packet comes from udp_queue_rcv_skb so it is already defrag,
		 * checksum verified, ... (ie safe to use)
		 *
		 * If the packet is not for us, return -1 and udp_queue_rcv_skb
		 * will continue to handle it (do not kfree skb !!).
		 */
		struct udp_opt *tp =  &(skb->sk->tp_pinfo.af_udp);
		struct iphdr *ip = (struct iphdr *)skb->nh.iph;
		struct udphdr *udp = (struct udphdr *)((__u32 *)ip+ip->ihl);
		__u8 *udpdata = (__u8 *)udp + sizeof(struct udphdr);
		__u32 *udpdata32 = (__u32 *)udpdata;

		natt_sport = ntohs(udp->source);
		natt_dport = ntohs(udp->dest);

		KLIPS_PRINT(debug_rcv,
		    "klips_debug:ipsec_rcv: "
		    "suspected ESPinUDP packet (NAT-Traversal) [%d].\n",
			tp->esp_in_udp);
		KLIPS_IP_PRINT(debug_rcv, ip);

		if (udpdata < skb->tail) {
			unsigned int len = skb->tail - udpdata;
			if ((len==1) && (udpdata[0]==0xff)) {
				KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
					/* not IPv6 compliant message */
				    "NAT-keepalive from %d.%d.%d.%d.\n", NIPQUAD(ip->saddr));
				goto rcvleave;
			}
			else if ( (tp->esp_in_udp == ESPINUDP_WITH_NON_IKE) &&
				(len > (2*sizeof(__u32) + sizeof(struct esp))) &&
				(udpdata32[0]==0) && (udpdata32[1]==0) ) {
				/* ESP Packet with Non-IKE header */
				KLIPS_PRINT(debug_rcv, 
					"klips_debug:ipsec_rcv: "
					"ESPinUDP pkt with Non-IKE - spi=0x%x\n",
					udpdata32[2]);
				natt_type = ESPINUDP_WITH_NON_IKE;
				natt_len = sizeof(struct udphdr)+(2*sizeof(__u32));
			}
			else if ( (tp->esp_in_udp == ESPINUDP_WITH_NON_ESP) &&
				(len > sizeof(struct esp)) &&
				(udpdata32[0]!=0) ) {
				/* ESP Packet without Non-ESP header */
				natt_type = ESPINUDP_WITH_NON_ESP;
				natt_len = sizeof(struct udphdr);
				KLIPS_PRINT(debug_rcv, 
					"klips_debug:ipsec_rcv: "
					"ESPinUDP pkt without Non-ESP - spi=0x%x\n",
					udpdata32[0]);
			}
			else {
				KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
					"IKE packet - not handled here\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
		}
		else {
			MOD_DEC_USE_COUNT;
			return -1;
		}
	}
#endif
		
#ifdef IPH_is_SKB_PULLED
	/* In Linux 2.4.4, the IP header has been skb_pull()ed before the
	   packet is passed to us. So we'll skb_push() to get back to it. */
	if (skb->data == skb->h.raw) {
		skb_push(skb, skb->h.raw - skb->nh.raw);
	}
#endif /* IPH_is_SKB_PULLED */

	ipp = (struct iphdr *)skb->data;
	iphlen = ipp->ihl << 2;
	/* dev->hard_header_len is unreliable and should not be used */
	hard_header_len = skb->mac.raw ? (skb->data - skb->mac.raw) : 0;
	if((hard_header_len < 0) || (hard_header_len > skb_headroom(skb)))
		hard_header_len = 0;

#ifdef NET_21
	/* if skb was cloned (most likely due to a packet sniffer such as
	   tcpdump being momentarily attached to the interface), make
	   a copy of our own to modify */
	if(skb_cloned(skb)) {
		/* include any mac header while copying.. */
		if(skb_headroom(skb) < hard_header_len) {
			printk(KERN_WARNING "klips_error:ipsec_rcv: "
			       "tried to skb_push hhlen=%d, %d available.  This should never happen, please report.\n",
			       hard_header_len,
			       skb_headroom(skb));
			goto rcvleave;
		}
		skb_push(skb, hard_header_len);
		if
#ifdef SKB_COW_NEW
               (skb_cow(skb, skb_headroom(skb)) != 0)
#else /* SKB_COW_NEW */
               ((skb = skb_cow(skb, skb_headroom(skb))) == NULL)
#endif /* SKB_COW_NEW */
		{
			goto rcvleave;
		}
		if(skb->len < hard_header_len) {
			printk(KERN_WARNING "klips_error:ipsec_rcv: "
			       "tried to skb_pull hhlen=%d, %d available.  This should never happen, please report.\n",
			       hard_header_len,
			       skb->len);
			goto rcvleave;
		}
		skb_pull(skb, hard_header_len);
	}
	
#endif /* NET_21 */

#if IP_FRAGMENT_LINEARIZE
	/* In Linux 2.4.4, we may have to reassemble fragments. They are
	   not assembled automatically to save TCP from having to copy
	   twice.
	*/
      if (skb_is_nonlinear(skb)) {
	if (skb_linearize(skb, GFP_ATOMIC) != 0) {
	  goto rcvleave;
	}
      }
      ipp = (struct iphdr *)skb->nh.iph;
      iphlen = ipp->ihl << 2;
#endif
	
#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
	if (natt_len) {
		/**
		 * Now, we are sure packet is ESPinUDP. Remove natt_len bytes from
		 * packet and modify protocol to ESP.
		 */
		if (((unsigned char *)skb->data > (unsigned char *)skb->nh.iph) &&
			((unsigned char *)skb->nh.iph > (unsigned char *)skb->head)) {
			unsigned int _len = (unsigned char *)skb->data -
				(unsigned char *)skb->nh.iph;
			KLIPS_PRINT(debug_rcv,
				"klips_debug:ipsec_rcv: adjusting skb: skb_push(%u)\n",
				_len);
			skb_push(skb, _len);
		}
		KLIPS_PRINT(debug_rcv,
		    "klips_debug:ipsec_rcv: "
			"removing %d bytes from ESPinUDP packet\n", natt_len);
		ipp = (struct iphdr *)skb->data;
		iphlen = ipp->ihl << 2;
		ipp->tot_len = htons(ntohs(ipp->tot_len) - natt_len);
		if (skb->len < iphlen + natt_len) {
			printk(KERN_WARNING
		       "klips_error:ipsec_rcv: "
		       "ESPinUDP packet is too small (%d < %d+%d). "
			   "This should never happen, please report.\n",
		       (int)(skb->len), iphlen, natt_len);
			goto rcvleave;
		}
		memmove(skb->data + natt_len, skb->data, iphlen);
		skb_pull(skb, natt_len);

		/* update nh.iph */
		ipp = skb->nh.iph = (struct iphdr *)skb->data;

		/* modify protocol */
		ipp->protocol = IPPROTO_ESP;

		skb->sk = NULL;

		KLIPS_IP_PRINT(debug_rcv, skb->nh.iph);
	}
#endif

	KLIPS_PRINT(debug_rcv, 
		    "klips_debug:ipsec_rcv: "
		    "<<< Info -- ");
	KLIPS_PRINTMORE(debug_rcv && skb->dev, "skb->dev=%s ",
		    skb->dev->name ? skb->dev->name : "NULL");
	KLIPS_PRINTMORE(debug_rcv && dev, "dev=%s ",
		    dev->name ? dev->name : "NULL");
	KLIPS_PRINTMORE(debug_rcv, "\n");
#if 0
	KLIPS_PRINT(debug_rcv && !(skb->dev && dev && (skb->dev == dev)),
		    "klips_debug:ipsec_rcv: "
		    "Informational -- **if this happens, find out why** skb->dev:%s is not equal to dev:%s\n",
		    skb->dev ? (skb->dev->name ? skb->dev->name : "NULL") : "NULL",
		    dev ? (dev->name ? dev->name : "NULL") : "NULL");
#endif
	protoc = ipp->protocol;
#ifndef NET_21
	if((!protocol) || (protocol->protocol != protoc)) {
		KLIPS_PRINT(debug_rcv & DB_RX_TDB,
			    "klips_debug:ipsec_rcv: "
			    "protocol arg is NULL or unequal to the packet contents, this is odd, using value in packet.\n");
	}
#endif /* !NET_21 */

	if( (protoc != IPPROTO_AH) &&
#ifdef CONFIG_IPSEC_IPCOMP_disabled_until_we_register_IPCOMP_HANDLER
	    (protoc != IPPROTO_COMP) &&
#endif /* CONFIG_IPSEC_IPCOMP */
	    (protoc != IPPROTO_ESP) ) {
		KLIPS_PRINT(debug_rcv & DB_RX_TDB,
			    "klips_debug:ipsec_rcv: Why the hell is someone "
			    "passing me a non-ipsec protocol = %d packet? -- dropped.\n",
			    protoc);
		goto rcvleave;
	}

	if(skb->dev) {
		for(i = 0; i < IPSEC_NUM_IF; i++) {
			sprintf(name, "ipsec%d", i);
			if(!strcmp(name, skb->dev->name)) {
				prv = (struct ipsecpriv *)(skb->dev->priv);
				if(prv) {
					stats = (struct net_device_stats *) &(prv->mystats);
				}
				ipsecdev = skb->dev;
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "Info -- pkt already proc'ed a group of ipsec headers, processing next group of ipsec headers.\n");
				break;
			}
			if((ipsecdev = ipsec_dev_get(name)) == NULL) {
				KLIPS_PRINT(debug_rcv,
					    "klips_error:ipsec_rcv: "
					    "device %s does not exist\n",
					    name);
			}
			prv = ipsecdev ? (struct ipsecpriv *)(ipsecdev->priv) : NULL;
			prvdev = prv ? (struct device *)(prv->dev) : NULL;
			
#if 0
			KLIPS_PRINT(debug_rcv && prvdev, 
				    "klips_debug:ipsec_rcv: "
				    "physical device for device %s is %s\n",
				    name,
				    prvdev->name);
#endif
			if(prvdev && skb->dev &&
			   !strcmp(prvdev->name, skb->dev->name)) {
				stats = prv ? ((struct net_device_stats *) &(prv->mystats)) : NULL;
				skb->dev = ipsecdev;
				KLIPS_PRINT(debug_rcv && prvdev, 
					    "klips_debug:ipsec_rcv: "
					    "assigning packet ownership to virtual device %s from physical device %s.\n",
					    name, prvdev->name);
				if(stats) {
					stats->rx_packets++;
				}
				break;
			}
		}
	} else {
		KLIPS_PRINT(debug_rcv, 
			    "klips_debug:ipsec_rcv: "
			    "device supplied with skb is NULL\n");
	}
			
	if(!stats) {
		ipsecdev = NULL;
	}
	KLIPS_PRINT((debug_rcv && !stats),
		    "klips_error:ipsec_rcv: "
		    "packet received from physical I/F (%s) not connected to ipsec I/F.  Cannot record stats.  May not have SA for decoding.  Is IPSEC traffic expected on this I/F?  Check routing.\n",
		    skb->dev ? (skb->dev->name ? skb->dev->name : "NULL") : "NULL");

	KLIPS_IP_PRINT(debug_rcv, ipp);

	/* begin decapsulating loop here */
	do {
		authlen = 0;
#ifdef CONFIG_IPSEC_ESP
		espp = NULL;
		esphlen = 0;
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
		ahp = NULL;
		ahhlen = 0;
#endif /* CONFIG_IPSEC_AH */
#ifdef CONFIG_IPSEC_IPCOMP
		compp = NULL;
#endif /* CONFIG_IPSEC_IPCOMP */

		len = skb->len;
		dat = skb->data;
		ipp = (struct iphdr *)skb->data;
		proto = ipp->protocol;
		ipaddr.s_addr = ipp->saddr;
		addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
		
		iphlen = ipp->ihl << 2;
		ipp->check = 0;			/* we know the sum is good */
		
#ifdef CONFIG_IPSEC_ESP
		/* XXX this will need to be 8 for IPv6 */
		if ((proto == IPPROTO_ESP) && ((len - iphlen) % 4)) {
			printk("klips_error:ipsec_rcv: "
			       "got packet with content length = %d from %s -- should be on 4 octet boundary, packet dropped\n",
			       len - iphlen,
			       ipaddr_txt);
			if(stats) {
				stats->rx_errors++;
			}
			goto rcvleave;
		}
#endif /* !CONFIG_IPSEC_ESP */
		
		/*
		 * Find tunnel control block and (indirectly) call the
		 * appropriate tranform routine. The resulting sk_buf
		 * is a valid IP packet ready to go through input processing.
		 */
		
		said.dst.s_addr = ipp->daddr;
		switch(proto) {
#ifdef CONFIG_IPSEC_ESP
		case IPPROTO_ESP:
			if (skb->len < ( hard_header_len
					 + sizeof(struct iphdr)
					 + sizeof(struct esp)) )
			{
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "runt esp packet of skb->len=%d received from %s, dropped.\n",
					    skb->len, ipaddr_txt) ;
				if (stats)
					stats->rx_errors++ ;
				goto rcvleave ;
			}
				
			espp = (struct esp *)(skb->data + iphlen);
			said.spi = espp->esp_spi;
			break;
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
		case IPPROTO_AH:
			ahp = (struct ah *)(skb->data + iphlen);
			if ( (skb->len <
			      ( hard_header_len
				+ sizeof(struct iphdr)
				+ sizeof(struct ah) ))
			     || (skb->len <
				 ( hard_header_len
				   + sizeof(struct iphdr)
				   + (ahp->ah_hl << 2)))
			     )
			{
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "runt ah packet of skb->len=%d received from %s, dropped.\n",
					    skb->len, ipaddr_txt) ;
				if (stats)
					stats->rx_errors++ ;
				goto rcvleave ;
			}

			said.spi = ahp->ah_spi;
			break;
#endif /* CONFIG_IPSEC_AH */
#ifdef CONFIG_IPSEC_IPCOMP
		case IPPROTO_COMP:
			if (skb->len <
			    ( hard_header_len
			      + sizeof(struct iphdr)
			      + sizeof(struct ipcomphdr)))
			{
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "runt comp packet of skb->len=%d received from %s, dropped.\n",
					    skb->len, ipaddr_txt) ;
				if (stats)
					stats->rx_errors++ ;
				goto rcvleave ;
			}
				
			compp = (struct ipcomphdr *)(skb->data + iphlen);
			said.spi = htonl((__u32)ntohs(compp->ipcomp_cpi));
			break;
#endif /* CONFIG_IPSEC_IPCOMP */
		default:
			if(stats) {
				stats->rx_errors++;
			}
			goto rcvleave;
		}
		said.proto = proto;
		sa_len = satoa(said, 0, sa, SATOA_BUF);
		if(sa_len == 0) {
		  strcpy(sa, "(error)");
		}
		
#ifdef CONFIG_IPSEC_AH
		if(proto == IPPROTO_AH) {
			ahhlen = (ahp->ah_hl << 2) +
				((caddr_t)&(ahp->ah_rpl) - (caddr_t)ahp);
			next_header = ahp->ah_nh;
			if (ahhlen != sizeof(struct ah)) {
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "bad authenticator length %d, expected %d from %s.\n",
					    ahhlen - ((caddr_t)(ahp->ah_data) - (caddr_t)ahp),
					    AHHMAC_HASHLEN,
					    ipaddr_txt);
				if(stats) {
					stats->rx_errors++;
				}
				goto rcvleave;
			}
			
		}
#endif /* CONFIG_IPSEC_AH */
		
		/*
		  The spinlock is to prevent any other process from
		  accessing or deleting the TDB hash table or any of the
		  TDBs while we are using and updating them.
		  
		  This is not optimal, but was relatively straightforward
		  at the time.  A better way to do it has been planned for
		  more than a year, to lock the hash table and put reference
		  counts on each TDB instead.  This is not likely to happen
		  in KLIPS1 unless a volunteer contributes it, but will be
		  designed into KLIPS2.
		*/
		if(tdbprev == NULL) {
			spin_lock(&tdb_lock);
		}
		
#ifdef CONFIG_IPSEC_IPCOMP
		if (proto == IPPROTO_COMP) {
			unsigned int flags = 0;
			if (tdbp == NULL) {
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "Incoming packet with outer IPCOMP header SA:%s: not yet supported by KLIPS, dropped\n",
					    sa_len ? sa : " (error)");
				if(stats) {
					stats->rx_dropped++;
				}

				goto rcvleave;
			}

			tdbprev = tdbp;
			tdbp = tdbnext;

			if(sysctl_ipsec_inbound_policy_check
			   && ((tdbp == NULL)
			       || (((ntohl(tdbp->tdb_said.spi) & 0x0000ffff)
				    != ntohl(said.spi))
				/* next line is a workaround for peer
				   non-compliance with rfc2393 */
				   && (tdbp->tdb_encalg != ntohl(said.spi)) 
				       ))) {
				char sa2[SATOA_BUF];
				size_t sa_len2 = 0;

				if(tdbp) {
					sa_len2 = satoa(tdbp->tdb_said, 0, sa2, SATOA_BUF);
				}
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "Incoming packet with SA(IPCA):%s does not match policy SA(IPCA):%s cpi=%04x cpi->spi=%08x spi=%08x, spi->cpi=%04x for SA grouping, dropped.\n",
					    sa_len ? sa : " (error)",
					    tdbp ? (sa_len2 ? sa2 : " (error)") : "NULL",
					    ntohs(compp->ipcomp_cpi),
					    (__u32)ntohl(said.spi),
					    tdbp ? (__u32)ntohl((tdbp->tdb_said.spi)) : 0,
					    tdbp ? (__u16)(ntohl(tdbp->tdb_said.spi) & 0x0000ffff) : 0);
				spin_unlock(&tdb_lock);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}

			if (tdbp) {
				tdbp->tdb_comp_ratio_cbytes += ntohs(ipp->tot_len);
				tdbnext = tdbp->tdb_inext;
			}
			next_header = compp->ipcomp_nh;

			skb = skb_decompress(skb, tdbp, &flags);
			if (!skb || flags) {
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "skb_decompress() returned error flags=%x, dropped.\n",
					    flags);
				if (stats) {
				    if (flags)
					stats->rx_errors++;
				    else
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
#ifdef NET_21
			ipp = skb->nh.iph;
#else /* NET_21 */
			ipp = skb->ip_hdr;
#endif /* NET_21 */

			if (tdbp) {
				tdbp->tdb_comp_ratio_dbytes += ntohs(ipp->tot_len);
			}

			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "packet decompressed SA(IPCA):%s cpi->spi=%08x spi=%08x, spi->cpi=%04x, nh=%d.\n",
				    sa_len ? sa : " (error)",
				    (__u32)ntohl(said.spi),
				    tdbp ? (__u32)ntohl((tdbp->tdb_said.spi)) : 0,
				    tdbp ? (__u16)(ntohl(tdbp->tdb_said.spi) & 0x0000ffff) : 0,
				    next_header);
			KLIPS_IP_PRINT(debug_rcv & DB_RX_PKTRX, ipp);

			continue;
			/* Skip rest of stuff and decapsulate next inner
			   packet, if any */
		}
#endif /* CONFIG_IPSEC_IPCOMP */
		
		tdbp = ipsec_sa_getbyid(&said);
		if (tdbp == NULL) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "no Tunnel Descriptor Block for SA:%s: incoming packet with no SA dropped\n",
				    sa_len ? sa : " (error)");
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
		if ((natt_type) &&
			( (ipp->saddr != (((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr.s_addr)) ||
			  (natt_sport != tdbp->ips_natt_sport)
			)) {
			struct sockaddr sipaddr;
			/** Advertise NAT-T addr change to pluto **/
			sipaddr.sa_family = AF_INET;
			((struct sockaddr_in*)&sipaddr)->sin_addr.s_addr = ipp->saddr;
			((struct sockaddr_in*)&sipaddr)->sin_port = htons(natt_sport);
			pfkey_nat_t_new_mapping(tdbp, &sipaddr, natt_sport);
			/**
			 * Then allow or block packet depending on
			 * sysctl_ipsec_inbound_policy_check.
			 *
			 * In all cases, pluto will update SA if new mapping is
			 * accepted.
			 */
			if (sysctl_ipsec_inbound_policy_check) {
				spin_unlock(&tdb_lock);
				ipaddr.s_addr = ipp->saddr;
				addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
				KLIPS_PRINT(debug_rcv,
					"klips_debug:ipsec_rcv: "
					"SA:%s, src=%s:%u of pkt does not agree with expected "
					"SA source address policy (pluto has been informed).\n",
					sa_len ? sa : " (error)",
					ipaddr_txt, natt_sport);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
		}
#endif

		if(sysctl_ipsec_inbound_policy_check) {
			if(ipp->saddr != ((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr.s_addr) {
				spin_unlock(&tdb_lock);
				ipaddr.s_addr = ipp->saddr;
				addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s, src=%s of pkt does not agree with expected SA source address policy.\n",
					    sa_len ? sa : " (error)",
					    ipaddr_txt);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
			ipaddr.s_addr = ipp->saddr;
			addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "SA:%s, src=%s of pkt agrees with expected SA source address policy.\n",
				    sa_len ? sa : " (error)",
				    ipaddr_txt);
			if(tdbnext) {
				if(tdbnext != tdbp) {
					spin_unlock(&tdb_lock);
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "unexpected SA:%s: does not agree with tdb->inext policy, dropped\n",
						    sa_len ? sa : " (error)");
					if(stats) {
						stats->rx_dropped++;
					}
					goto rcvleave;
				}
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s grouping from previous SA is OK.\n",
					    sa_len ? sa : " (error)");
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s First SA in group.\n",
					    sa_len ? sa : " (error)");
			}
			
			if(tdbp->tdb_onext) {
				if(tdbprev != tdbp->tdb_onext) {
					spin_unlock(&tdb_lock);
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "unexpected SA:%s: does not agree with tdb->onext policy, dropped.\n",
						    sa_len ? sa : " (error)");
					if(stats) {
						stats->rx_dropped++;
					}
					goto rcvleave;
				} else {
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "SA:%s grouping to previous SA is OK.\n",
						    sa_len ? sa : " (error)");
				}
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s No previous backlink in group.\n",
					    sa_len ? sa : " (error)");
			}
#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
			KLIPS_PRINT(debug_rcv,
				"klips_debug:ipsec_rcv: "
				"natt_type=%u tdbp->ips_natt_type=%u : %s\n",
				natt_type, tdbp->ips_natt_type,
				(natt_type==tdbp->ips_natt_type)?"ok":"bad");
			if (natt_type != tdbp->ips_natt_type) {
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s does not agree with expected NAT-T policy.\n",
					    sa_len ? sa : " (error)");
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
#endif
		}
		
		/* If it is in larval state, drop the packet, we cannot process yet. */
		if(tdbp->tdb_state == SADB_SASTATE_LARVAL) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "TDB in larval state, cannot be used yet, dropping packet.\n");
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
		
		if(tdbp->tdb_state == SADB_SASTATE_DEAD) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "TDB in dead state, cannot be used any more, dropping packet.\n");
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
		
		if(ipsec_lifetime_check(&tdbp->ips_life.ipl_bytes,   "bytes", sa,
					ipsec_life_countbased, ipsec_incoming, tdbp) == ipsec_life_harddied ||
		   ipsec_lifetime_check(&tdbp->ips_life.ipl_addtime, "addtime",sa,
					ipsec_life_timebased,  ipsec_incoming, tdbp) == ipsec_life_harddied ||
		   ipsec_lifetime_check(&tdbp->ips_life.ipl_addtime, "usetime",sa,
					ipsec_life_timebased,  ipsec_incoming, tdbp) == ipsec_life_harddied ||
		   ipsec_lifetime_check(&tdbp->ips_life.ipl_packets, "packets",sa, 
					ipsec_life_countbased, ipsec_incoming, tdbp) == ipsec_life_harddied) {
			ipsec_sa_delchain(tdbp);
			spin_unlock(&tdb_lock);
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}

		/* authenticate, if required */
		idat = dat + iphlen;
		switch(tdbp->tdb_authalg) {
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
		case AH_MD5:
			authlen = AHHMAC_HASHLEN;
			break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
		case AH_SHA:
			authlen = AHHMAC_HASHLEN;
			break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
		case AH_NONE:
			authlen = 0;
			break;
		default:
			tdbp->tdb_alg_errs += 1;
			spin_unlock(&tdb_lock);
			if(stats) {
				stats->rx_errors++;
			}
			goto rcvleave;
		}
		ilen = len - iphlen - authlen;

		if (ilen <= 0)
		{
			spin_unlock(&tdb_lock) ;
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "runt AH packet with no data, dropping.\n");
			if(stats)
				stats->rx_dropped++ ;
			goto rcvleave ;
		}
		
#ifdef CONFIG_IPSEC_ESP
		KLIPS_PRINT(proto == IPPROTO_ESP && debug_rcv, 
			    "klips_debug:ipsec_rcv: "
			    "packet from %s received with seq=%d (iv)=0x%08x%08x iplen=%d esplen=%d sa=%s\n",
			    ipaddr_txt,
			    (__u32)ntohl(espp->esp_rpl),
			    (__u32)ntohl(*((__u32 *)(espp->esp_iv)    )),
			    (__u32)ntohl(*((__u32 *)(espp->esp_iv) + 1)),
			    len,
			    ilen,
			    sa_len ? sa : " (error)");
#endif /* !CONFIG_IPSEC_ESP */
		
		switch(proto) {
#ifdef CONFIG_IPSEC_ESP
		case IPPROTO_ESP:
			replay = ntohl(espp->esp_rpl);
			authenticator = &(dat[len - authlen]);
			break;
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
		case IPPROTO_AH:
			replay = ntohl(ahp->ah_rpl);
			authenticator = ahp->ah_data;
			break;
#endif /* CONFIG_IPSEC_AH */
		}

		if (!ipsec_checkreplaywindow(tdbp, replay)) {
			tdbp->tdb_replaywin_errs += 1;
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv & DB_RX_REPLAY,
				    "klips_debug:ipsec_rcv: "
				    "duplicate frame from %s, packet dropped\n",
				    ipaddr_txt);
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
		
#ifdef NO_BRECIS_HW
		/*
		 * verify authenticator
		 */
		
		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: "
			    "encalg = %d, authalg = %d.\n",
			    tdbp->tdb_encalg,
			    tdbp->tdb_authalg);
		
		if(tdbp->tdb_authalg) {
			switch(tdbp->tdb_authalg) {
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
			case AH_MD5:
				tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->ictx;
				if(proto == IPPROTO_ESP) {
					MD5Update(&tctx.md5, (caddr_t)espp, ilen);
#ifdef CONFIG_IPSEC_AH
				} else {
					ipo = *ipp;
					ipo.tos = 0;	/* mutable RFC 2402 3.3.3.1.1.1 */
					ipo.frag_off = 0;
					ipo.ttl = 0;
					ipo.check = 0;
					
					MD5Update(&tctx.md5, (caddr_t)&ipo,
						  sizeof(struct iphdr));
					MD5Update(&tctx.md5, (caddr_t)ahp,
						  ahhlen - AHHMAC_HASHLEN);
					MD5Update(&tctx.md5, (caddr_t)zeroes,
						  AHHMAC_HASHLEN);
					MD5Update(&tctx.md5,
						  (caddr_t)dat + iphlen + ahhlen,
						  len - iphlen - ahhlen);
#endif /* CONFIG_IPSEC_AH */
				}
				MD5Final(hash, &tctx.md5);
				tctx.md5 = ((struct md5_ctx*)(tdbp->tdb_key_a))->octx;
				MD5Update(&tctx.md5, hash, AHMD596_ALEN);
				MD5Final(hash, &tctx.md5);
				break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
			case AH_SHA:
				tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx;
				if(proto == IPPROTO_ESP) {
					SHA1Update(&tctx.sha1, (caddr_t)espp, ilen);
#ifdef CONFIG_IPSEC_AH
				} else {
					ipo = *ipp;
					ipo.tos = 0;
					ipo.frag_off = 0;
					ipo.ttl = 0;
					ipo.check = 0;
					
					SHA1Update(&tctx.sha1, (caddr_t)&ipo,
						   sizeof(struct iphdr));
					SHA1Update(&tctx.sha1, (caddr_t)ahp,
						   ahhlen - AHHMAC_HASHLEN);
					SHA1Update(&tctx.sha1, (caddr_t)zeroes,
						   AHHMAC_HASHLEN);
					SHA1Update(&tctx.sha1,
						   (caddr_t)dat + iphlen + ahhlen,
						   len - iphlen - ahhlen);
#endif /* CONFIG_IPSEC_AH */
				}
				SHA1Final(hash, &tctx.sha1);
				tctx.sha1 = ((struct sha1_ctx*)(tdbp->tdb_key_a))->octx;
				SHA1Update(&tctx.sha1, hash, AHSHA196_ALEN);
				SHA1Final(hash, &tctx.sha1);
				break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
			case AH_NONE:
				break;
			}
		
			if(!authenticator) {
				tdbp->tdb_auth_errs += 1;
				spin_unlock(&tdb_lock);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}

			if (memcmp(hash, authenticator, authlen)) {
				tdbp->tdb_auth_errs += 1;
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "auth failed on incoming packet from %s: hash=%08x%08x%08x auth=%08x%08x%08x, dropped\n",
					    ipaddr_txt,
					    *(__u32*)&hash[0],
					    *(__u32*)&hash[4],
					    *(__u32*)&hash[8],
					    *(__u32*)authenticator,
					    *((__u32*)authenticator + 1),
					    *((__u32*)authenticator + 2));
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "authentication successful.\n");
			}
			
			memset((caddr_t)&tctx, 0, sizeof(tctx));
			memset(hash, 0, sizeof(hash));
		}

		/* If the sequence number == 0, expire SA, it had rolled */
		if(tdbp->tdb_replaywin && !replay /* !tdbp->tdb_replaywin_lastseq */) {
			ipsec_sa_delchain(tdbp);
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "replay window counter rolled, expiring SA.\n");
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}

		if (!ipsec_updatereplaywindow(tdbp, replay)) {
			tdbp->tdb_replaywin_errs += 1;
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv & DB_RX_REPLAY,
				    "klips_debug:ipsec_rcv: "
				    "duplicate frame from %s, packet dropped\n",
				    ipaddr_txt);
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
		
		switch(proto) {
#ifdef CONFIG_IPSEC_ESP
		case IPPROTO_ESP:
			switch(tdbp->tdb_encalg) {
#ifdef USE_SINGLE_DES
			case ESP_DES:
#endif
			case ESP_3DES:
				iv[0] = *((__u32 *)(espp->esp_iv)    );
				iv[1] = *((__u32 *)(espp->esp_iv) + 1);
				esphlen = sizeof(struct esp);
				break;
			default:
				tdbp->tdb_alg_errs += 1;
				spin_unlock(&tdb_lock);
				if(stats) {
					stats->rx_errors++;
				}
				goto rcvleave;
			}
			idat += esphlen;
			ilen -= esphlen;
			
			switch(tdbp->tdb_encalg) {
#ifdef USE_SINGLE_DES
			case ESP_DES:
            if ((ilen) % 8) {
					printk("klips_error:ipsec_rcv: "
					       "got packet with esplen = %d from %s "
					       "-- should be on 8 octet boundary, packet dropped\n",
					       ilen, ipaddr_txt);
               if(stats) {
                  stats->rx_errors++;
               }
               tdbp->tdb_encsize_errs += 1;
					spin_unlock(&tdb_lock);
               goto rcvleave;
            }
/*            des_cbc_encrypt(idat, idat, ilen,
                  tdbp->tdb_key_e,
                  (caddr_t)iv, 0);
*/
            des_ncbc_encrypt((des_cblock *)idat, (des_cblock *)idat, ilen,
                  ((struct des_eks *)(tdbp->tdb_key_e))[0].ks,
                  (des_cblock *)iv, 0);
            break;
#endif
			case ESP_3DES:
				if ((ilen) % 8) {
					tdbp->tdb_encsize_errs += 1;
					spin_unlock(&tdb_lock);
					printk("klips_error:ipsec_rcv: "
					       "got packet with esplen = %d from %s -- should be on 8 octet boundary, packet dropped\n",
					       ilen,
					       ipaddr_txt);
					if(stats) {
						stats->rx_errors++;
					}
					goto rcvleave;
				}
				des_ede3_cbc_encrypt((des_cblock *)idat,
						     (des_cblock *)idat,
						     ilen,
						     ((struct des_eks *)(tdbp->tdb_key_e))[0].ks,
						     ((struct des_eks *)(tdbp->tdb_key_e))[1].ks,
						     ((struct des_eks *)(tdbp->tdb_key_e))[2].ks,
						     (des_cblock *)iv, 0);
				break;
			default:
				break ;
			}
			next_header = idat[ilen - 1];
			padlen = idat[ilen - 2];
			pad = padlen + 2 + authlen;
			{
			        int badpad = 0;
				
				KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
					    "klips_debug:ipsec_rcv: "
					    "padlen=%d, contents: 0x<offset>: 0x<value> 0x<value> ...\n",
					    padlen);
				
				for (i = 1; i <= padlen; i++) {
					if((i % 16) == 1) {
						KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
							    "klips_debug:           %02x:",
							    i - 1);
					}
					KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
						    " %02x",
						    idat[ilen - 2 - padlen + i - 1]);
					if(i != idat[ilen - 2 - padlen + i - 1]) {
					        badpad = 1;
					} 
					if((i % 16) == 0) {
						KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
							    "\n");
					}
				}
				if((i % 16) != 1) {
					KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
							"\n");
				}
				if(badpad) {
					KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
						    "klips_debug:ipsec_rcv: "
						    "warning, decrypted packet from %s has bad padding\n",
						    ipaddr_txt);
					KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
						    "klips_debug:ipsec_rcv: "
						    "...may be bad decryption -- not dropped\n");
					tdbp->tdb_encpad_errs += 1;
				}
				
				KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
					    "klips_debug:ipsec_rcv: "
					    "packet decrypted from %s: next_header = %d, padding = %d\n",
					    ipaddr_txt,
					    next_header,
					    pad - 2 - authlen);
			}
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
		case IPPROTO_AH:
			break;
#endif /* CONFIG_IPSEC_AH */
		}
#else /* ! NO_BRECIS_HW */
		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: "
			    "encalg = %d, authalg = %d.\n",
			    tdbp->tdb_encalg,
			    tdbp->tdb_authalg);
		

		switch(proto)
		{
#ifdef CONFIG_IPSEC_ESP
		case IPPROTO_ESP:
			esphlen = sizeof(struct esp);

			if (tdbp->tdb_encalg == ESP_3DES)
			{
				if ((ilen - esphlen) % 8) {
					tdbp->tdb_encsize_errs += 1;
					spin_unlock(&tdb_lock);
					printk("klips_error:ipsec_rcv: "
					       "got packet with esplen = %d from %s -- should be on 8 octet boundary, packet dropped\n",
					       ilen - esphlen,
					       ipaddr_txt);
					if(stats) {
						stats->rx_errors++;
					}
					goto rcvleave;
				}

			}
			else if (tdbp->tdb_encalg == ESP_DES)
			{
				if ((ilen - esphlen) % 8) {
					tdbp->tdb_encsize_errs += 1;
					spin_unlock(&tdb_lock);
					printk("klips_error:ipsec_rcv: "
					       "got packet with esplen = %d from %s -- should be on 8 octet boundary, packet dropped\n",
					       ilen - esphlen,
					       ipaddr_txt);
					if(stats) {
						stats->rx_errors++;
					}
					goto rcvleave;
				}

			}
			else if (tdbp->tdb_encalg == ESP_AES)
			{
				esphlen += 8; /* iv is 16 bytes, not 8 */
				if ((ilen - esphlen) % 16) {
					tdbp->tdb_encsize_errs += 1;
					spin_unlock(&tdb_lock);
					printk("klips_error:ipsec_rcv: "
					       "got packet with esplen = %d from %s -- should be on 16 octet boundary, packet dropped\n",
					       ilen - esphlen,
					       ipaddr_txt);
					if(stats) {
						stats->rx_errors++;
					}
					goto rcvleave;
				}

			}
			else
			{
				tdbp->tdb_alg_errs += 1;
				spin_unlock(&tdb_lock);
				if(stats) {
					stats->rx_errors++;
				}
				goto rcvleave;
			}

#ifdef CONFIG_BRECIS_SEC_V2
			if (tdbp->brecis_sa.flags == 0)
			{
#endif /* CONFIG_BRECIS_SEC_V2 */
				/* use v1 hardware */
				/* assume 3des */

				/* first do auth */
				if (tdbp->auth_desc.ioctl.type != 0)
				{
					tdbp->auth_desc2.ioctl.srcaddr
						= (void*) espp ;
					tdbp->auth_desc2.ioctl.dstaddr = hash ;
					tdbp->auth_desc2.ioctl.buflen  = ilen ;

					msp_sec_ioctl_kernel(&tdbp->sec_context,
							     MSP_SEC_CTL,
							     &tdbp->auth_desc) ;
					if(!authenticator) {
						tdbp->tdb_auth_errs += 1;
						spin_unlock(&tdb_lock);
						if(stats) {
							stats->rx_dropped++;
						}
						goto rcvleave;
					}

					if (memcmp(hash, authenticator, authlen)) {
						tdbp->tdb_auth_errs += 1;
						spin_unlock(&tdb_lock);
						KLIPS_PRINT(debug_rcv & DB_RX_INAU,
							    "klips_debug:ipsec_rcv: "
							    "auth failed on incoming packet from %s: hash=%08x%08x%08x auth=%08x%08x%08x, dropped\n",
							    ipaddr_txt,
							    *(__u32*)&hash[0],
							    *(__u32*)&hash[4],
							    *(__u32*)&hash[8],
							    *(__u32*)authenticator,
							    *((__u32*)authenticator + 1),
							    *((__u32*)authenticator + 2));
						if(stats) {
							stats->rx_dropped++;
						}
						goto rcvleave;
					} else {
						KLIPS_PRINT(debug_rcv,
							    "klips_debug:ipsec_rcv: "
							    "authentication successful.\n");
					}

					memset(hash, 0, sizeof(hash));
				}

				/* now do decrypt */
				tdbp->crypt_desc.ioctl.srcaddr = idat ;
				tdbp->crypt_desc.ioctl.dstaddr = idat ;
				tdbp->crypt_desc.ioctl.buflen  = ilen ;
				memcpy((char *)&tdbp->crypt_desc.ioctl.ivhigh,
				       espp->esp_iv, 8) ;
				msp_sec_ioctl_kernel(&tdbp->sec_context,
						     MSP_SEC_CTL,
						     &tdbp->crypt_desc) ;

#ifdef CONFIG_BRECIS_SEC_V2
			}
			else
			{
			

				msp_sec2_new_request(0,	/* work_q */
						     4,	/* n_sg_entries */
						     &tdbp->brecis_sa,
						     0, /* control */
						     CBK_POLL,
						     &status,
						     BLK_POLL /* block */
						     ) ;
				msp_sec2_add_sg(0, /* work_q */
						SG_GATHER, /* gather */
						(void *)espp, /* addr. */
						ilen /* size */
						) ;
				msp_sec2_add_sg(0, /* work_q */
						SG_SCATTER, /* scatter */
						(void*)(idat + esphlen), /* addr */
						ilen - esphlen /* size */
						) ;

				if (authlen)
				{
					msp_sec2_add_sg(0, /* work_q */
							SG_GATHER, /* gather */
							((char *)espp) + ilen, /* addr */
							authlen	/* size */
							) ;

				}

				msp_sec2_end_request(0 /* work_q */ ) ;

				/* XXX fixme -- check right bits in status */
				if (status)
				{
					tdbp->tdb_auth_errs += 1;
					spin_unlock(&tdb_lock);
					KLIPS_PRINT(debug_rcv & DB_RX_INAU,
						    "klips_debug:ipsec_rcv: "
						    "auth failed on incoming packet from %s; dropped\n",
						    ipaddr_txt ) ;
					if(stats) {
						stats->rx_dropped++;
					}
					goto rcvleave;
				}
				else
				{
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "authentication successful.\n");
				}
			}
#endif /* CONFIG_BRECIS_SEC_V2 */

			/* If the sequence number == 0, expire SA, it had rolled */
			if(tdbp->tdb_replaywin && !replay /* !tdbp->tdb_replaywin_lastseq */) {
				ipsec_sa_delchain(tdbp);
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "replay window counter rolled, expiring SA.\n");
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}

			if (!ipsec_updatereplaywindow(tdbp, replay)) {
				tdbp->tdb_replaywin_errs += 1;
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv & DB_RX_REPLAY,
					    "klips_debug:ipsec_rcv: "
					    "duplicate frame from %s, packet dropped\n",
					    ipaddr_txt);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
		
			next_header = idat[ilen - 1];
			padlen = idat[ilen - 2];
			pad = padlen + 2 + authlen;
			{
			        int badpad = 0;
				
				KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
					    "klips_debug:ipsec_rcv: "
					    "padlen=%d, contents: 0x<offset>: 0x<value> 0x<value> ...\n",
					    padlen);
				
				for (i = 1; i <= padlen; i++) {
					if((i % 16) == 1) {
						KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
							    "klips_debug:           %02x:",
							    i - 1);
					}
					KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
						    " %02x",
						    idat[ilen - 2 - padlen + i - 1]);
					if(i != idat[ilen - 2 - padlen + i - 1]) {
					        badpad = 1;
					} 
					if((i % 16) == 0) {
						KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
							    "\n");
					}
				}
				if((i % 16) != 1) {
					KLIPS_PRINTMORE(debug_rcv & DB_RX_IPAD,
							"\n");
				}
				if(badpad) {
					KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
						    "klips_debug:ipsec_rcv: "
						    "warning, decrypted packet from %s has bad padding\n",
						    ipaddr_txt);
					KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
						    "klips_debug:ipsec_rcv: "
						    "...may be bad decryption -- not dropped\n");
					tdbp->tdb_encpad_errs += 1;
				}
				
				KLIPS_PRINT(debug_rcv & DB_RX_IPAD,
					    "klips_debug:ipsec_rcv: "
					    "packet decrypted from %s: next_header = %d, padding = %d\n",
					    ipaddr_txt,
					    next_header,
					    pad - 2 - authlen);
			}

		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: "
			    "encalg = %d, authalg = %d.\n",
			    tdbp->tdb_encalg,
			    tdbp->tdb_authalg);
		
				
			break ;
#endif
#ifdef CONFIG_IPSEC_AH
		case IPPROTO_AH:
				{
					sec_desc hd[5] ;

					hd[0].ioctl.status    = 0 ;
					hd[0].ioctl.type      = MSP_SEC_HMAC ;
					if (tdbp->tdb_authalg == AH_MD5)
						hd[0].ioctl.mode      = HMC_MD5 ;
					else
						hd[0].ioctl.mode      = HMC_SHA1 ;
			
					hd[0].ioctl.options   = MSP_SEC_DO_PADDING;
					hd[0].ioctl.buflen = tdbp->tdb_key_a_size ;
					hd[0].ioctl.srcaddr = tdbp->tdb_key_a ;
					hd[0].ioctl.dstaddr = hash ;
					hd[0].next = &hd[1] ;
					
					ipo = *ipp;
					ipo.tos = 0;	/* mutable RFC 2402 3.3.3.1.1.1 */
					ipo.frag_off = 0;
					ipo.ttl = 0;
					ipo.check = 0;

					hd[1].ioctl.buflen = sizeof(struct iphdr) ;
					hd[1].ioctl.srcaddr = (caddr_t)&ipo;
					hd[1].ioctl.dstaddr = hash ;
					hd[1].next = &hd[2] ;
						
					hd[2].ioctl.buflen = ahhlen - AHHMAC_HASHLEN ;
					hd[2].ioctl.srcaddr = (caddr_t)ahp;
					hd[2].ioctl.dstaddr = hash ;
					hd[2].next = &hd[3] ;
					
					hd[3].ioctl.buflen = AHHMAC_HASHLEN ;
					hd[3].ioctl.srcaddr =(caddr_t)zeroes;
					hd[3].ioctl.dstaddr = hash ;
					hd[3].next = &hd[4] ;
						
					hd[4].ioctl.buflen = len - iphlen - ahhlen ;
					hd[4].ioctl.srcaddr = (caddr_t)
						dat + iphlen + ahhlen ;
					hd[4].ioctl.dstaddr = hash ;
					hd[4].next = 0 ;
					
					msp_sec_ioctl_kernel(&msp_sec_ctx,
						     MSP_SEC_CTL,
						     &hd[0]) ;
				}
			if(!authenticator) {
				tdbp->tdb_auth_errs += 1;
				spin_unlock(&tdb_lock);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}

			if (memcmp(hash, authenticator, authlen)) {
				tdbp->tdb_auth_errs += 1;
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv & DB_RX_INAU,
					    "klips_debug:ipsec_rcv: "
					    "auth failed on incoming packet from %s: hash=%08x%08x%08x auth=%08x%08x%08x, dropped\n",
					    ipaddr_txt,
					    *(__u32*)&hash[0],
					    *(__u32*)&hash[4],
					    *(__u32*)&hash[8],
					    *(__u32*)authenticator,
					    *((__u32*)authenticator + 1),
					    *((__u32*)authenticator + 2));
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "authentication successful.\n");
			}
			
			/* If the sequence number == 0, expire SA, it had rolled */
			if(tdbp->tdb_replaywin && !replay /* !tdbp->tdb_replaywin_lastseq */) {
				ipsec_sa_delchain(tdbp);
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "replay window counter rolled, expiring SA.\n");
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}

			if (!ipsec_updatereplaywindow(tdbp, replay)) {
				tdbp->tdb_replaywin_errs += 1;
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv & DB_RX_REPLAY,
					    "klips_debug:ipsec_rcv: "
					    "duplicate frame from %s, packet dropped\n",
					    ipaddr_txt);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
		
			break ;
#endif
		}

#endif /* ! NO_BRECIS_HW */	       
		/*
		 *	Discard the original ESP/AH header
		 */
		
		ipp->protocol = next_header;
		
		switch(proto) {
#ifdef CONFIG_IPSEC_ESP
		case IPPROTO_ESP:
			ipp->tot_len = htons(ntohs(ipp->tot_len) - (esphlen + pad));
			memmove((void *)(skb->data + esphlen),
				(void *)(skb->data), iphlen);
			if(skb->len < esphlen) {
				spin_unlock(&tdb_lock);
				printk(KERN_WARNING
				       "klips_error:ipsec_rcv: "
				       "tried to skb_pull esphlen=%d, %d available.  This should never happen, please report.\n",
				       esphlen, (int)(skb->len));
				goto rcvleave;
			}
			skb_pull(skb, esphlen);

			KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
				    "klips_debug:ipsec_rcv: "
				    "trimming to %d.\n",
				    len - esphlen - pad);
			if(pad + esphlen <= len) {
			        skb_trim(skb, len - esphlen - pad);
			} else {
				spin_unlock(&tdb_lock);
			        KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
				            "klips_debug:ipsec_rcv: "
					    "bogus packet, size is zero or negative, dropping.\n");
				goto rcvleave;
			}
		break;
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_AH
		case IPPROTO_AH:
			ipp->tot_len = htons(ntohs(ipp->tot_len) - ahhlen);
			memmove((void *)(skb->data + ahhlen),
				(void *)(skb->data), iphlen);
			if(skb->len < ahhlen) {
				spin_unlock(&tdb_lock);
				printk(KERN_WARNING
				       "klips_error:ipsec_rcv: "
				       "tried to skb_pull ahhlen=%d, %d available.  This should never happen, please report.\n",
				       ahhlen,
				       (int)(skb->len));
				goto rcvleave;
			}
			skb_pull(skb, ahhlen);
			break;
#endif /* CONFIG_IPSEC_AH */
		}


		/*
		 *	Adjust pointers
		 */
		
		len = skb->len;
		dat = skb->data;
		
#ifdef NET_21
/*		skb->h.ipiph=(struct iphdr *)skb->data; */
		skb->nh.raw = skb->data;
		skb->h.raw = skb->nh.raw + (skb->nh.iph->ihl << 2);
		
		memset(&(IPCB(skb)->opt), 0, sizeof(struct ip_options));
#else /* NET_21 */
		skb->h.iph=(struct iphdr *)skb->data;
		skb->ip_hdr=(struct iphdr *)skb->data;
		memset(skb->proto_priv, 0, sizeof(struct options));
#endif /* NET_21 */
		
		ipp = (struct iphdr *)dat;
		ipp->check = 0;
		ipp->check = ip_fast_csum((unsigned char *)dat, iphlen >> 2);
		
		KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
			    "klips_debug:ipsec_rcv: "
			    "after <%s%s%s>, SA:%s:\n",
			    IPS_XFORM_NAME(tdbp),
			    sa_len ? sa : " (error)");
		KLIPS_IP_PRINT(debug_rcv & DB_RX_PKTRX, ipp);
		
		skb->protocol = htons(ETH_P_IP);
		skb->ip_summed = 0;

		tdbprev = tdbp;
		tdbnext = tdbp->tdb_inext;
		if(sysctl_ipsec_inbound_policy_check) {
			if(tdbnext) {
				if(tdbnext->tdb_onext != tdbp) {
					spin_unlock(&tdb_lock);
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "SA:%s, backpolicy does not agree with fwdpolicy.\n",
						    sa_len ? sa : " (error)");
					if(stats) {
						stats->rx_dropped++;
					}
					goto rcvleave;
				}
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s, backpolicy agrees with fwdpolicy.\n",
					    sa_len ? sa : " (error)");
				if(
					ipp->protocol != IPPROTO_AH  
					&& ipp->protocol != IPPROTO_ESP 
#ifdef CONFIG_IPSEC_IPCOMP
					&& ipp->protocol != IPPROTO_COMP
					&& (tdbnext->tdb_said.proto != IPPROTO_COMP
					    || (tdbnext->tdb_said.proto == IPPROTO_COMP
						&& tdbnext->tdb_inext))
#endif /* CONFIG_IPSEC_IPCOMP */
					&& ipp->protocol != IPPROTO_IPIP
					) {
					spin_unlock(&tdb_lock);
					KLIPS_PRINT(debug_rcv,
						    "klips_debug:ipsec_rcv: "
						    "packet with incomplete policy dropped, last successful SA:%s.\n",
						    sa_len ? sa : " (error)");
					if(stats) {
						stats->rx_dropped++;
					}
					goto rcvleave;
				}
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s, Another IPSEC header to process.\n",
					    sa_len ? sa : " (error)");
			} else {
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "No tdb_inext from this SA:%s.\n",
					    sa_len ? sa : " (error)");
			}
		}

#ifdef CONFIG_IPSEC_IPCOMP
		/* update ipcomp ratio counters, even if no ipcomp packet is present */
		if (tdbnext
		  && tdbnext->tdb_said.proto == IPPROTO_COMP
		  && ipp->protocol != IPPROTO_COMP) {
			tdbnext->tdb_comp_ratio_cbytes += ntohs(ipp->tot_len);
			tdbnext->tdb_comp_ratio_dbytes += ntohs(ipp->tot_len);
		}
#endif /* CONFIG_IPSEC_IPCOMP */

		tdbp->ips_life.ipl_bytes.ipl_count += len;
		tdbp->ips_life.ipl_bytes.ipl_last   = len;

		if(!tdbp->ips_life.ipl_usetime.ipl_count) {
			tdbp->ips_life.ipl_usetime.ipl_count = jiffies / HZ;
		}
		tdbp->ips_life.ipl_usetime.ipl_last = jiffies / HZ;
		tdbp->ips_life.ipl_packets.ipl_count += 1;
		
	/* end decapsulation loop here */
	} while(   (ipp->protocol == IPPROTO_ESP )
		|| (ipp->protocol == IPPROTO_AH  )
#ifdef CONFIG_IPSEC_IPCOMP
		|| (ipp->protocol == IPPROTO_COMP)
#endif /* CONFIG_IPSEC_IPCOMP */
		);
	
#ifdef CONFIG_IPSEC_IPCOMP
	if(tdbnext && tdbnext->tdb_said.proto == IPPROTO_COMP) {
		tdbprev = tdbp;
		tdbp = tdbnext;
		tdbnext = tdbp->tdb_inext;
	}
#endif /* CONFIG_IPSEC_IPCOMP */

#ifdef CONFIG_IPSEC_NAT_TRAVERSAL
	if ((natt_type) && (ipp->protocol != IPPROTO_IPIP)) {
		/**
		 * NAT-Traversal and Transport Mode:
		 *   we need to correct TCP/UDP checksum
		 *
		 * If we've got NAT-OA, we can fix checksum without recalculation.
		 */
		__u32 natt_oa = tdbp->ips_natt_oa ?
			((struct sockaddr_in*)(tdbp->ips_natt_oa))->sin_addr.s_addr : 0;
		__u16 pkt_len = skb->tail - (unsigned char *)ipp;
		__u16 data_len = pkt_len - (ipp->ihl << 2);

		switch (ipp->protocol) {
			case IPPROTO_TCP:
				if (data_len >= sizeof(struct tcphdr)) {
					struct tcphdr *tcp = (struct tcphdr *)((__u32 *)ipp+ipp->ihl);
					if (natt_oa) {
						__u32 buff[2] = { ~natt_oa, ipp->saddr };
						KLIPS_PRINT(debug_rcv,
				    		"klips_debug:ipsec_rcv: "
							"NAT-T & TRANSPORT: "
							"fix TCP checksum using NAT-OA\n");
						tcp->check = csum_fold(
							csum_partial((unsigned char *)buff, sizeof(buff),
							tcp->check^0xffff));
					}
					else {
						KLIPS_PRINT(debug_rcv,
			    			"klips_debug:ipsec_rcv: "
							"NAT-T & TRANSPORT: recalc TCP checksum\n");
						if (pkt_len > (ntohs(ipp->tot_len)))
							data_len -= (pkt_len - ntohs(ipp->tot_len));
						tcp->check = 0;
						tcp->check = csum_tcpudp_magic(ipp->saddr, ipp->daddr,
							data_len, IPPROTO_TCP,
							csum_partial((unsigned char *)tcp, data_len, 0));
					}
				}
				else {
					KLIPS_PRINT(debug_rcv,
			    		"klips_debug:ipsec_rcv: "
						"NAT-T & TRANSPORT: can't fix TCP checksum\n");
				}
				break;
			case IPPROTO_UDP:
				if (data_len >= sizeof(struct udphdr)) {
					struct udphdr *udp = (struct udphdr *)((__u32 *)ipp+ipp->ihl);
					if (udp->check == 0) {
						KLIPS_PRINT(debug_rcv,
				    		"klips_debug:ipsec_rcv: "
							"NAT-T & TRANSPORT: UDP checksum already 0\n");
					}
					else if (natt_oa) {
						__u32 buff[2] = { ~natt_oa, ipp->saddr };
						KLIPS_PRINT(debug_rcv,
				    		"klips_debug:ipsec_rcv: "
							"NAT-T & TRANSPORT: "
							"fix UDP checksum using NAT-OA\n");
						udp->check = csum_fold(
							csum_partial((unsigned char *)buff, sizeof(buff),
							udp->check^0xffff));
					}
					else {
						KLIPS_PRINT(debug_rcv,
				    		"klips_debug:ipsec_rcv: "
							"NAT-T & TRANSPORT: zero UDP checksum\n");
						udp->check = 0;
					}
				}
				else {
					KLIPS_PRINT(debug_rcv,
			    		"klips_debug:ipsec_rcv: "
						"NAT-T & TRANSPORT: can't fix UDP checksum\n");
				}
				break;
			default:
				KLIPS_PRINT(debug_rcv,
			    	"klips_debug:ipsec_rcv: "
					"NAT-T & TRANSPORT: non TCP/UDP packet -- do nothing\n");
				break;
		}
	}
#endif

	/*
	 * XXX this needs to be locked from when it was first looked
	 * up in the decapsulation loop.  Perhaps it is better to put
	 * the IPIP decap inside the loop.
	 */
	if(tdbnext) {
		tdbp = tdbnext;
		sa_len = satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
		if(ipp->protocol != IPPROTO_IPIP) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "SA:%s, Hey!  How did this get through?  Dropped.\n",
				    sa_len ? sa : " (error)");
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
		if(sysctl_ipsec_inbound_policy_check) {
			if((tdbnext = tdbp->tdb_inext)) {
				char sa2[SATOA_BUF];
				size_t sa_len2;
				sa_len2 = satoa(tdbnext->tdb_said, 0, sa2, SATOA_BUF);
				spin_unlock(&tdb_lock);
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "unexpected SA:%s after IPIP SA:%s\n",
					    sa_len2 ? sa2 : " (error)",
					    sa_len ? sa : " (error)");
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
			if(ipp->saddr != ((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr.s_addr) {
				spin_unlock(&tdb_lock);
				ipaddr.s_addr = ipp->saddr;
				addrtoa(ipaddr, 0, ipaddr_txt, sizeof(ipaddr_txt));
				KLIPS_PRINT(debug_rcv,
					    "klips_debug:ipsec_rcv: "
					    "SA:%s, src=%s of pkt does not agree with expected SA source address policy.\n",
					    sa_len ? sa : " (error)",
					    ipaddr_txt);
				if(stats) {
					stats->rx_dropped++;
				}
				goto rcvleave;
			}
		}

		/*
		 * XXX this needs to be locked from when it was first looked
		 * up in the decapsulation loop.  Perhaps it is better to put
		 * the IPIP decap inside the loop.
		 */
		tdbp->ips_life.ipl_bytes.ipl_count += len;
		tdbp->ips_life.ipl_bytes.ipl_last   = len;

		if(!tdbp->ips_life.ipl_usetime.ipl_count) {
			tdbp->ips_life.ipl_usetime.ipl_count = jiffies / HZ;
		}
		tdbp->ips_life.ipl_usetime.ipl_last = jiffies / HZ;
		tdbp->ips_life.ipl_packets.ipl_count += 1;
		
		if(skb->len < iphlen) {
			spin_unlock(&tdb_lock);
			printk(KERN_WARNING "klips_debug:ipsec_rcv: "
			       "tried to skb_pull iphlen=%d, %d available.  This should never happen, please report.\n",
			       iphlen,
			       (int)(skb->len));

			goto rcvleave;
		}
		skb_pull(skb, iphlen);

#ifdef NET_21
		ipp = (struct iphdr *)skb->nh.raw = skb->data;
		skb->h.raw = skb->nh.raw + (skb->nh.iph->ihl << 2);
		
		memset(&(IPCB(skb)->opt), 0, sizeof(struct ip_options));
#else /* NET_21 */
		ipp = skb->ip_hdr = skb->h.iph = (struct iphdr *)skb->data;

		memset(skb->proto_priv, 0, sizeof(struct options));
#endif /* NET_21 */

		skb->protocol = htons(ETH_P_IP);
		skb->ip_summed = 0;
		KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
			    "klips_debug:ipsec_rcv: "
			    "IPIP tunnel stripped.\n");
		KLIPS_IP_PRINT(debug_rcv & DB_RX_PKTRX, ipp);

		if(sysctl_ipsec_inbound_policy_check
		   /* 
		      Note: "xor" (^) logically replaces "not equal"
		      (!=) and "bitwise or" (|) logically replaces
		      "boolean or" (||).  This is done to speed up
		      execution by doing only bitwise operations and
		      no branch operations
		   */
		   && (((ipp->saddr & tdbp->tdb_mask_s.u.v4.sin_addr.s_addr)
			            ^ tdbp->tdb_flow_s.u.v4.sin_addr.s_addr)
		       | ((ipp->daddr & tdbp->tdb_mask_d.u.v4.sin_addr.s_addr)
			              ^ tdbp->tdb_flow_d.u.v4.sin_addr.s_addr)) )
		{
			struct in_addr daddr, saddr;
			char saddr_txt[ADDRTOA_BUF], daddr_txt[ADDRTOA_BUF];
			char sflow_txt[SUBNETTOA_BUF], dflow_txt[SUBNETTOA_BUF];
			
			subnettoa(tdbp->tdb_flow_s.u.v4.sin_addr,
				tdbp->tdb_mask_s.u.v4.sin_addr,
				0, sflow_txt, sizeof(sflow_txt));
			subnettoa(tdbp->tdb_flow_d.u.v4.sin_addr,
				tdbp->tdb_mask_d.u.v4.sin_addr,
				0, dflow_txt, sizeof(dflow_txt));
			spin_unlock(&tdb_lock);
			saddr.s_addr = ipp->saddr;
			daddr.s_addr = ipp->daddr;
			addrtoa(saddr, 0, saddr_txt, sizeof(saddr_txt));
			addrtoa(daddr, 0, daddr_txt, sizeof(daddr_txt));
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "SA:%s, inner tunnel policy [%s -> %s] does not agree with pkt contents [%s -> %s].\n",
				    sa_len ? sa : " (error)",
				    sflow_txt,
				    dflow_txt,
				    saddr_txt,
				    daddr_txt);
			if(stats) {
				stats->rx_dropped++;
			}
			goto rcvleave;
		}
	}

#ifdef INBOUND_POLICY_CHECK_eroute
	/*
	  Do *not* enable this without thoroughly checking spinlock issues
	  first.  In particular, nesting an eroute spinlock within a tdb
	  spinlock could result in a deadlock.  (Well, only on a SMP machine
	  under 2.4?)
	*/

	/*
	 * First things first -- look us up in the erouting tables.
	 */
	matcher.sen_len = sizeof (struct sockaddr_encap);
	matcher.sen_family = AF_ENCAP;
	matcher.sen_type = SENT_IP4;
	if(ipp->protocol == IPPROTO_IPIP) {
		struct iphdr *ipp2;

		ipp2 = (struct iphdr*) (((char*)ipp) + (ipp->ihl << 2));
		matcher.sen_ip_src.s_addr = ipp2->saddr;
		matcher.sen_ip_dst.s_addr = ipp2->daddr;
	} else {
		matcher.sen_ip_src.s_addr = ipp->saddr;
		matcher.sen_ip_dst.s_addr = ipp->daddr;
	}
	
	/*
	 * The spinlock is to prevent any other process from accessing or
	 * deleting the eroute while we are using and updating it.
	 */
	spin_lock(&eroute_lock);
	
	er = ipsec_findroute(&matcher);
	if(er) {
		policy_said = er->er_said;
		policy_eaddr = er->er_eaddr;
		policy_emask = er->er_emask;
		er->er_count++;
		er->er_lasttime = jiffies/HZ;
	}
	
	spin_unlock(&eroute_lock);

	if(er) {
		/*
		 * The spinlock is to prevent any other process from
		 * accessing or deleting the tdb while we are using and
		 * updating it.
		 */
		spin_lock(&tdb_lock);

		policy_tdb = gettdb(&policy_said);
		if (policy_tdb == NULL) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "no Tunnel Descriptor Block for SA%s: incoming packet with no policy SA, dropped.\n",
				    sa_len ? sa : " (error)");
			goto rcvleave;
		}
		
		sa_len = satoa(policy_said, 0, sa, SATOA_BUF);

		KLIPS_PRINT(debug_rcv,
			    "klips_debug:ipsec_rcv: "
			    "found policy Tunnel Descriptor Block -- SA:%s\n",
			    sa_len ? sa : " (error)");
		while(1) {
			if(policy_tdb->tdb_inext) {
				policy_tdb = policy_tdb->tdb_inext;
			} else {
				break;
			}
		}
		if(policy_tdb != tdbp) {
			spin_unlock(&tdb_lock);
			KLIPS_PRINT(debug_rcv,
				    "klips_debug:ipsec_rcv: "
				    "Tunnel Descriptor Block for SA%s: incoming packet with different policy SA, dropped.\n",
				    sa_len ? sa : " (error)");
			goto rcvleave;
		}

		/* spin_unlock(&tdb_lock); */
	}
#endif /* INBOUND_POLICY_CHECK_eroute */

	spin_unlock(&tdb_lock);

#ifdef NET_21
	if(stats) {
		stats->rx_bytes += skb->len;
	}
	if(skb->dst) {
		dst_release(skb->dst);
		skb->dst = NULL;
	}
	skb->pkt_type = PACKET_HOST;
	if(hard_header_len &&
	   (skb->mac.raw != (skb->data - hard_header_len)) &&
	   (hard_header_len <= skb_headroom(skb))) {
		/* copy back original MAC header */
		memmove(skb->data - hard_header_len, skb->mac.raw, hard_header_len);
		skb->mac.raw = skb->data - hard_header_len;
	}
#endif /* NET_21 */

#ifdef CONFIG_IPSEC_IPCOMP
	if(ipp->protocol == IPPROTO_COMP) {
		unsigned int flags = 0;

		if(sysctl_ipsec_inbound_policy_check) {
			KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
				"klips_debug:ipsec_rcv: "
				"inbound policy checking enabled, IPCOMP follows IPIP, dropped.\n");
			if (stats) {
				stats->rx_errors++;
			}
			goto rcvleave;
		}
		/*
		  XXX need a TDB for updating ratio counters but it is not
		  following policy anyways so it is not a priority
		*/
		skb = skb_decompress(skb, NULL, &flags);
		if (!skb || flags) {
			KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
				"klips_debug:ipsec_rcv: "
				"skb_decompress() returned error flags: %d, dropped.\n",
			       flags);
			if (stats) {
				stats->rx_errors++;
			}
			goto rcvleave;
		}
	}
#endif /* CONFIG_IPSEC_IPCOMP */

#ifdef SKB_RESET_NFCT
        nf_conntrack_put(skb->nfct);
        skb->nfct = NULL;
#ifdef CONFIG_NETFILTER_DEBUG
	skb->nf_debug = 0;
#endif /* CONFIG_NETFILTER_DEBUG */
#endif /* SKB_RESET_NFCT */
	KLIPS_PRINT(debug_rcv & DB_RX_PKTRX,
		    "klips_debug:ipsec_rcv: "
		    "netif_rx() called.\n");
	netif_rx(skb);

	MOD_DEC_USE_COUNT;
	return(0);
	
 rcvleave:
 	if(skb) {
#ifdef NET_21
                kfree_skb(skb);
#else /* NET_21 */
                kfree_skb(skb, FREE_WRITE);
#endif /* NET_21 */
	}

	MOD_DEC_USE_COUNT;
	return(0);
}

struct inet_protocol ah_protocol =
{
	ipsec_rcv,				/* AH handler */
	NULL,				/* TUNNEL error control */
	0,				/* next */
	IPPROTO_AH,			/* protocol ID */
	0,				/* copy */
	NULL,				/* data */
	"AH"				/* name */
};

struct inet_protocol esp_protocol = 
{
	ipsec_rcv,			/* ESP handler          */
	NULL,				/* TUNNEL error control */
	0,				/* next */
	IPPROTO_ESP,			/* protocol ID */
	0,				/* copy */
	NULL,				/* data */
	"ESP"				/* name */
};

#if 0
/* We probably don't want to install a pure IPCOMP protocol handler, but
   only want to handle IPCOMP if it is encapsulated inside an ESP payload
   (which is already handled) */
#ifdef CONFIG_IPSEC_IPCOMP
struct inet_protocol comp_protocol =
{
	ipsec_rcv,			/* COMP handler		*/
	NULL,				/* COMP error control	*/
	0,				/* next */
	IPPROTO_COMP,			/* protocol ID */
	0,				/* copy */
	NULL,				/* data */
	"COMP"				/* name */
};
#endif /* CONFIG_IPSEC_IPCOMP */
#endif


